/* ------------------------------------------------------------------------ */
/* LHa for UNIX                                                             */
/*              crcio.c -- crc input / output                               */
/*                                                                          */
/*      Modified                Nobutaka Watazaki                           */
/*                                                                          */
/*  Ver. 1.14   Source All chagned              1995.01.14  N.Watazaki      */
/* ------------------------------------------------------------------------ */
#include "lha.h"

/* ------------------------------------------------------------------------ */
#ifdef EUC
static int      putc_euc_cache;
#endif
static int      getc_euc_cache;

/* ------------------------------------------------------------------------ */
void
make_crctable( /* void */ )
{
    unsigned int    i, j, r;

    for (i = 0; i <= UCHAR_MAX; i++) {
        r = i;
        for (j = 0; j < CHAR_BIT; j++)
            if (r & 1)
                r = (r >> 1) ^ CRCPOLY;
            else
                r >>= 1;
        crctable[i] = r;
    }
}

/* ------------------------------------------------------------------------ */
unsigned int
calccrc(crc, p, n)
    unsigned int crc;
    unsigned char  *p;
    unsigned int    n;
{
    while (n-- > 0)
        crc = UPDATE_CRC(crc, *p++);
    return crc;
}

/* ------------------------------------------------------------------------ */
int
fread_crc(crcp, p, n, fp)
    unsigned int *crcp;
    unsigned char  *p;
    int             n;
    FILE           *fp;
{
    if (text_mode)
        n = fread_txt(p, n, fp);
    else
        n = fread(p, 1, n, fp);

    *crcp = calccrc(*crcp, p, n);
#ifdef NEED_INCREMENTAL_INDICATOR
    put_indicator(n);
#endif
    return n;
}

/* ------------------------------------------------------------------------ */
void
fwrite_crc(crcp, p, n, fp)
    unsigned int *crcp;
    unsigned char  *p;
    int             n;
    FILE           *fp;
{
    *crcp = calccrc(*crcp, p, n);
#ifdef NEED_INCREMENTAL_INDICATOR
    put_indicator(n);
#endif
    if (fp) {
        if (text_mode) {
            if (fwrite_txt(p, n, fp))
                fatal_error("File write error");
        }
        else {
            if (fwrite(p, 1, n, fp) < n)
                fatal_error("File write error");
        }
    }
}

/* ------------------------------------------------------------------------ */
void
init_code_cache( /* void */ )
{               /* called from copyfile() in util.c */
#ifdef EUC
    putc_euc_cache = EOF;
#endif
    getc_euc_cache = EOF;
}

/* ------------------------------------------------------------------------ */
#ifdef EUC
int
putc_euc(c, fd)
    int             c;
    FILE           *fd;
{
    int             d;

    if (putc_euc_cache == EOF) {
        if (!euc_mode || c < 0x81 || c > 0xFC) {
            return putc(c, fd);
        }
        if (c >= 0xA0 && c < 0xE0) {
            if (putc(0x8E, fd) == EOF) return EOF;  /* single shift */
            return putc(c, fd);
        }
        putc_euc_cache = c; /* save first byte */
        return c;
    }
    d = putc_euc_cache;
    putc_euc_cache = EOF;
    if (d >= 0xA0)
        d -= 0xE0 - 0xA0;
    if (c > 0x9E) {
        c = c - 0x9F + 0x21;
        d = (d - 0x81) * 2 + 0x22;
    }
    else {
        if (c > 0x7E)
            c--;
        c -= 0x1F;
        d = (d - 0x81) * 2 + 0x21;
    }
    if (putc(0x80 | d, fd) == EOF) return EOF;
    return putc(0x80 | c, fd);
}
#endif

/* ------------------------------------------------------------------------ */
int
fwrite_txt(p, n, fp)
    unsigned char  *p;
    int             n;
    FILE           *fp;
{
    while (--n >= 0) {
        if (*p != '\015' && *p != '\032') {
#ifdef EUC
            if (putc_euc(*p, fp) == EOF)
                break;
#else
            if (putc(*p, fp) == EOF)
                break;
#endif
        }
        p++;
    }
    return (ferror(fp));
}

/* ------------------------------------------------------------------------ */
int
fread_txt(p, n, fp)
    unsigned char  *p;
    int             n;
    FILE           *fp;
{
    int             c;
    int             cnt = 0;

    while (cnt < n) {
        if (getc_euc_cache != EOF) {
            c = getc_euc_cache;
            getc_euc_cache = EOF;
        }
        else {
            if ((c = fgetc(fp)) == EOF)
                break;
            if (c == '\n') {
                getc_euc_cache = c;
                ++origsize;
                c = '\r';
            }
#ifdef EUC
            else if (euc_mode && (c == 0x8E || (0xA0 < c && c < 0xFF))) {
                int             d = fgetc(fp);
                if (d == EOF) {
                    *p++ = c;
                    cnt++;
                    break;
                }
                if (c == 0x8E) {    /* single shift (KANA) */
                    if ((0x20 < d && d < 0x7F) || (0xA0 < d && d < 0xFF))
                        c = d | 0x80;
                    else
                        getc_euc_cache = d;
                }
                else {
                    if (0xA0 < d && d < 0xFF) { /* if GR */
                        c &= 0x7F;  /* convert to MS-kanji */
                        d &= 0x7F;
                        if (!(c & 1)) {
                            c--;
                            d += 0x7F - 0x21;
                        }
                        if ((d += 0x40 - 0x21) > 0x7E)
                            d++;
                        if ((c = (c >> 1) + 0x71) >= 0xA0)
                            c += 0xE0 - 0xA0;
                    }
                    getc_euc_cache = d;
                }
            }
#endif
        }
        *p++ = c;
        cnt++;
    }
    return cnt;
}
