﻿using System;
using System.Collections.Generic;

namespace FooEditEngine
{
    /// <summary>
    /// マーカーを表す
    /// </summary>
    public interface IMarker
    {
        /// <summary>
        /// マーカーの開始位置。-1を設定した場合、そのマーカーはレタリングされません
        /// </summary>
        int start { get;set;}
        /// <summary>
        /// マーカーの長さ。0を設定した場合、そのマーカーはレタリングされません
        /// </summary>
        int length { get; set; }
        /// <summary>
        /// マーカーのタイプ
        /// </summary>
        HilightType hilight { get; set; }
    }

    /// <summary>
    /// マーカーのタイプを表す列挙体
    /// </summary>
    [Flags]
    public enum HilightType
    {
        /// <summary>
        /// マーカーとして表示しないことを表す
        /// </summary>
        None = 0,
        /// <summary>
        /// 選択状態を表す
        /// </summary>
        Select = 1,
        /// <summary>
        /// URLを表す
        /// </summary>
        Url = 2,
        /// <summary>
        /// 実線を表す
        /// </summary>
        Sold = 4,
        /// <summary>
        /// 太目の実践を表す
        /// </summary>
        BoldSold = 5,
    }

    /// <summary>
    /// マーカー自身を表します
    /// </summary>
    public struct Marker : IMarker, IEqualityComparer<Marker>
    {
        #region IMarker メンバー

        /// <summary>
        /// 開始位置
        /// </summary>
        public int start
        {
            get;
            set;
        }

        /// <summary>
        /// 長さ
        /// </summary>
        public int length
        {
            get;
            set;
        }

        /// <summary>
        /// タイプ
        /// </summary>
        public HilightType hilight
        {
            get;
            set;
        }

        #endregion

        /// <summary>
        /// マーカーを作成します
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <param name="hilight">タイプ</param>
        /// <returns>マーカー</returns>
        public static Marker Create(int start, int length, HilightType hilight)
        {
            return new Marker { start = start, length = length, hilight = hilight };
        }

        /// <summary>
        /// 等しいかどうかを調べます
        /// </summary>
        /// <param name="x">比較されるマーカー</param>
        /// <param name="y">比較するマーカー</param>
        /// <returns>等しいなら真。そうでなければ偽</returns>
        public bool Equals(Marker x, Marker y)
        {
            return x.hilight == y.hilight && x.length == y.length && x.start == y.start;
        }

        /// <summary>
        /// ハッシュを得ます
        /// </summary>
        /// <param name="obj">マーカー</param>
        /// <returns>ハッシュ</returns>
        public int GetHashCode(Marker obj)
        {
            return this.start ^ this.length ^ (int)this.hilight;
        }
    }

    /// <summary>
    /// マーカークラスのコレクションを表します
    /// </summary>
    public class MarkerCollection : IEnumerable<Marker>
    {
        List<Marker> collection = new List<Marker>();

        internal MarkerCollection(Document doc)
        {
            doc.Update += new DocumentUpdateEventHandler(doc_Update);
            this.Updated +=new EventHandler((s,e)=>{});
        }

        /// <summary>
        /// 更新されたことを通知します
        /// </summary>
        public event EventHandler Updated;

        /// <summary>
        /// インデクサー
        /// </summary>
        /// <param name="i">0から始まるインデックス</param>
        /// <returns>マーカー</returns>
        public Marker this[int i]
        {
            get
            {                
                return this.collection[i];
            }
            internal set
            {
                this.collection[i] = value;
                this.Updated(this, null);
            }
        }

        internal void Add(Marker m)
        {
            this.collection.Add(m);
            this.Updated(this, null);
        }

        internal void RemoveAll(Predicate<Marker> func)
        {
            for (int i = 0; i < this.collection.Count; i++)
            {
                if (func(this.collection[i]))
                    this.collection.RemoveAt(i);
            }
            this.Updated(this, null);
        }

        /// <summary>
        /// マーカーをすべて削除します
        /// </summary>
        public void Clear()
        {
            this.collection.Clear();
            this.Updated(this, null);
        }

        /// <summary>
        /// 格納されているマーカーの数を表します
        /// </summary>
        public int Count
        {
            get
            {
                return this.collection.Count;
            }
        }

        #region IEnumerable<Marker> メンバー

        /// <summary>
        /// 列挙子を返します
        /// </summary>
        /// <returns>IEnumeratorオブジェクトを返します</returns>
        public IEnumerator<Marker> GetEnumerator()
        {
            foreach (Marker marker in this.collection)
                yield return marker;
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            throw new NotImplementedException();
        }

        #endregion

        void doc_Update(object sender, DocumentUpdateEventArgs e)
        {
            int deltaLength = 0;
            switch (e.type)
            {
                case UpdateType.Replace:
                    deltaLength = e.insertLength - e.removeLength;
                    for (int i = 0; i < this.Count; i++)
                    {
                        Marker m = this[i];
                        if (m.start < e.startIndex)
                            continue;
                        m.start += deltaLength;
                        this[i] = m;
                    }
                    break;
                default:
                    return;
            }
        }

    }
}
