﻿using System;
using System.Drawing;
using System.Drawing.Printing;

namespace FooEditEngine.Windows
{
    /// <summary>
    /// イベントデータ
    /// </summary>
    public class ParseCommandEventArgs
    {
        /// <summary>
        /// ページ番号
        /// </summary>
        public int PageNumber;
        /// <summary>
        /// プリンターの設定
        /// </summary>
        public PrinterSettings PrinterSetting;
        /// <summary>
        /// 処理前の文字列
        /// </summary>
        public string Original;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="nowPage">印刷中のページ番号</param>
        /// <param name="setting">プリンターの設定</param>
        /// <param name="org">処理前の文字列</param>
        public ParseCommandEventArgs(int nowPage,PrinterSettings setting,string org)
        {
            this.PageNumber = nowPage;
            this.PrinterSetting = setting;
            this.Original = org;
        }
    }

    /// <summary>
    /// コマンド処理用デリゲート
    /// </summary>
    /// <param name="sender">送信元のクラス</param>
    /// <param name="e">イベントデータ</param>
    /// <returns>処理後の文字列</returns>
    public delegate string ParseCommandHandler(object sender,ParseCommandEventArgs e);

    /// <summary>
    /// 印刷用のクラス
    /// </summary>
    public class FooPrintText
    {
        PrintableView view;
        PrintableTextRender render;
        int PageNumber;

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="doc">Documentオブジェクト</param>
        public FooPrintText()
        {
            this.PrintDocument = new PrintDocument();
            this.PrintDocument.PrintPage += new PrintPageEventHandler(PrintDocument_PrintPage);
            this.PrintDocument.EndPrint += new PrintEventHandler(PrintDocument_EndPrint);
            this.ParseHF = new ParseCommandHandler((s, e) => { return e.Original; });
        }

        /// <summary>
        /// 対象となるドキュメント
        /// </summary>
        public Document Document
        {
            get;
            set;
        }

        /// <summary>
        /// プリントドキュメント
        /// </summary>
        public PrintDocument PrintDocument
        {
            get;
            private set;
        }

        /// <summary>
        /// 行番号を表示するかどうか
        /// </summary>
        public bool DrawLineNumber
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷に使用するフォント
        /// </summary>
        public Font Font
        {
            get;
            set;
        }

        /// <summary>
        /// 用紙の端で折り返すなら真
        /// </summary>
        public bool isWordRap
        {
            get;
            set;
        }

        /// <summary>
        /// ヘッダー
        /// </summary>
        public string Header
        {
            get;
            set;
        }

        /// <summary>
        /// フッター
        /// </summary>
        public string Footer
        {
            get;
            set;
        }

        /// <summary>
        /// ヘッダーやフッターを処理する
        /// </summary>
        public ParseCommandHandler ParseHF;

        void PrintDocument_PrintPage(object sender, PrintPageEventArgs e)
        {
            if (this.Font == null || this.Document == null)
                throw new InvalidOperationException();

            if (view == null)
            {
                this.render = new PrintableTextRender(this.Font, e.Graphics);
                this.view = new PrintableView(this.Document, this.render);
                this.view.DrawLineNumber = this.DrawLineNumber;
                this.view.TabStops = 4;
                this.view.PageBound = e.MarginBounds;
                this.view.isLineBreak = this.isWordRap;
                this.PageNumber = 1;
            }

            if (e.PageSettings.PrinterSettings.PrintRange == PrintRange.SomePages)
            {
                for (; this.PageNumber < e.PageSettings.PrinterSettings.FromPage; this.PageNumber++)
                    if(this.view.TryPageDown())
                        return;
            }

            this.view.Header = this.ParseHF(this, new ParseCommandEventArgs(this.PageNumber, e.PageSettings.PrinterSettings,this.Header));
            this.view.Footer = this.ParseHF(this, new ParseCommandEventArgs(this.PageNumber, e.PageSettings.PrinterSettings, this.Footer));

            this.render.BeginDraw(e.Graphics);

            this.view.Draw(e.MarginBounds);

            e.HasMorePages = !this.view.TryPageDown();
            this.PageNumber++;

            this.render.EndDraw();

            if (e.HasMorePages && e.PageSettings.PrinterSettings.PrintRange == PrintRange.SomePages && this.PageNumber > e.PageSettings.PrinterSettings.ToPage)
                e.HasMorePages = false;
        }

        void PrintDocument_EndPrint(object sender, PrintEventArgs e)
        {
            this.view.Dispose();
            this.view = null;
            this.render = null;
        }

    }
}
