﻿using System;
using System.Text;
using System.Linq;
using System.Diagnostics;
using System.Text.RegularExpressions;
#if WINFORM
using System.Drawing;
#endif

namespace FooEditEngine
{
    /// <summary>
    /// ユーザー側からの処理を担当するクラス。一部を除き、こちらで行われた操作はアンドゥの対象になります
    /// </summary>
    internal class Controller
    {
        View View;
        Document Document;
        int AnchorIndex;
        
        public Controller(Document doc, View view)
        {
            this.Document = doc;
            this.Document.Update += new DocumentUpdateEventHandler(Document_Update);
            this.View = view;
            this.Document.Clear();
            this.CaretMoved += new EventHandler((s, e) => { });
        }
        
        /// <summary>
        /// キャレット移動時に通知される
        /// </summary>
        public event EventHandler CaretMoved;


        /// <summary>
        /// 矩形選択モードなら真を返し、そうでない場合は偽を返す
        /// </summary>
        public bool RectSelection
        {
            get;
            set;
        }

        /// <summary>
        /// 選択範囲の開始位置
        /// </summary>
        /// <remarks>SelectionLengthが0の場合、キャレット位置を表します</remarks>
        public int SelectionStart
        {
            get
            {
                if (this.Document.Selections.Count == 0)
                    return this.AnchorIndex;
                else
                    return this.Document.Selections.First().start;
            }
        }

        /// <summary>
        /// 選択範囲の長さ
        /// </summary>
        /// <remarks>矩形選択モードの場合、選択範囲の文字数ではなく、開始位置から終了位置までの長さとなります</remarks>
        public int SelectionLength
        {
            get
            {
                if (this.Document.Selections.Count == 0)
                    return 0;
                Selection last = this.Document.Selections.Last();
                return last.start + last.length - this.SelectionStart;
            }
        }

        /// <summary>
        /// 選択範囲内の文字列を返す
        /// </summary>
        /// <remarks>
        /// 未選択状態で代入したときは追加され、そうでない場合は選択範囲の文字列と置き換えられます。
        /// </remarks>
        public string SelectedText
        {
            get
            {
                if (this.View.LayoutLines.Count == 0 || this.Document.Selections.Count == 0)
                    return null;
                if (this.RectSelection)
                    return GetTextFromRectangleSelectArea(this.Document.Selections);
                else
                    return GetTextFromLineSelectArea(this.Document.Selections);
            }
            set
            {
                if (this.Document.FireUpdateEvent == false)
                    throw new InvalidOperationException("");
                this.RepleaceSelectionArea(this.Document.Selections, value);
            }
        }

        /// <summary>
        /// 指定された範囲を選択する
        /// </summary>
        /// <param name="start"></param>
        /// <param name="length"></param>
        /// <remarks>RectSelectionの値によって動作が変わります。真の場合は矩形選択モードに、そうでない場合は行ごとに選択されます</remarks>
        public void Select(int start, int length)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            if (start < 0 || start + length < 0 || start + length > this.Document.Length)
                throw new ArgumentOutOfRangeException("startかendが指定できる範囲を超えてます");
            this.Document.Selections.Clear();
            if (this.RectSelection && length != 0)
            {
                Rectangle2 rect = GetSelectRectangle(start, length);
                this.SelectByRectangle(rect);
            }
            else
            {
                if (length < 0)
                {
                    int oldStart = start;
                    start += length;
                    length = oldStart - start;
                }
                this.Document.Selections.Add(Selection.Create(start, length));
            }
        }

        private Rectangle2 GetSelectRectangle(int start, int length)
        {
            TextPoint startTextPoint = this.View.GetLayoutLineFromIndex(start);
            TextPoint endTextPoint = this.View.GetLayoutLineFromIndex(start + length);
            Point2 sp = new Point2(
                this.View.GetXFromIndex(this.View.LayoutLines[startTextPoint.row],startTextPoint.col),
                startTextPoint.row);
            Point2 ep = new Point2(
                this.View.GetXFromIndex(this.View.LayoutLines[endTextPoint.row], endTextPoint.col),
                endTextPoint.row);
            if (sp.Row > ep.Row)
            {
                if (sp.X > ep.X)
                    return new Rectangle2(ep.X, ep.Row,sp.X - ep.X, sp.Row - ep.Row + 1);
                else
                    return new Rectangle2(sp.X, ep.Row, ep.X - sp.X, sp.Row - ep.Row + 1);
            }
            else
            {
                if (sp.X > ep.X)
                    return new Rectangle2(ep.X, sp.Row, sp.X - ep.X, ep.Row - sp.Row + 1);
                else
                    return new Rectangle2(sp.X, sp.Row, ep.X - sp.X, ep.Row - sp.Row + 1);
            }
       }

        /// <summary>
        /// 単語単位で選択する
        /// </summary>
        /// <param name="index">探索を開始するインデックス</param>
        public void SelectWord(int index)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            Document str = this.Document;

            int start = index;
            while (start > 0 && str[start] != Document.NewLine && !Char.IsSeparator(str[start]) && !Char.IsPunctuation(str[start]))
                start--;

            if (str[start] == Document.NewLine || Char.IsSeparator(str[start]) || Char.IsPunctuation(str[start]))
                start++;

            int end = index;
            while (end < this.Document.Length - 1 && str[end] != Document.NewLine && !Char.IsSeparator(str[end]) && !Char.IsPunctuation(str[end]))
                end++;

            this.Select(start, end - start);
        }

        /// <summary>
        /// 選択を解除する
        /// </summary>
        public void DeSelectAll()
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            this.Document.Selections.Clear();
        }

        /// <summary>
        /// 任意のマーカーかどうか
        /// </summary>
        /// <param name="tp"></param>
        /// <param name="type"></param>
        /// <returns>真ならマーカーがある</returns>
        public bool IsMarker(TextPoint tp,HilightType type)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            int index = this.View.LayoutLines.GetIndexFromTextPoint(tp);
            return this.IsMarker(index, type);
        }

        /// <summary>
        /// 任意のマーカーかどうか判定する
        /// </summary>
        /// <param name="index"></param>
        /// <param name="type"></param>
        /// <returns>真ならマーカーがある</returns>
        public bool IsMarker(int index, HilightType type)
        {
            foreach (Marker m in this.Document.GetMarkers(index))
            {
                if (m.hilight == type)
                    return true;
            }
            return false;
        }

        /// <summary>
        /// キャレットを指定した位置に移動させる
        /// </summary>
        /// <param name="index"></param>
        public void JumpCaret(int index)
        {
            if (index < 0 || index > this.Document.Length)
                throw new ArgumentOutOfRangeException("indexが設定できる範囲を超えています");
            TextPoint tp = this.View.GetLayoutLineFromIndex(index);

            this.JumpCaret(tp.row, tp.col);
         }

        /// <summary>
        /// キャレットを指定した位置に移動させる
        /// </summary>
        /// <param name="row"></param>
        public void JumpCaret(int row,int col)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            this.View.JumpCaret(row, col);

            this.View.AdjustCaretAndSrc();

            this.SelectWithMoveCaret(false);

            this.CaretMoved(this, null);
        }

        /// <summary>
        /// スクロールすると同時にキャレットを移動させる
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        public void ScrollWithCaretMove(double x, int row,bool isSelected)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            if (this.View.TryScroll(x, row))
                return;
            TextPoint p = this.View.GetTextPointFromPostion(this.View.render.ClipRect.Location);
            this.View.JumpCaret(p.row, p.col);
            this.View.SetCaretPostion(0, 0);
            this.SelectWithMoveCaret(isSelected);
            this.CaretMoved(this, null);
        }

        /// <summary>
        /// キャレットを桁方向に移動させる
        /// </summary>
        /// <returns>再描写する必要があるなら真を返す</returns>
        /// <param name="realLength">負の値なら左側へ、そうでないなら右側へ移動する</param>
        public void MoveCaretHorizontical(int realLength, bool isSelected,bool alignWord = false)
        {
            for (int i = Math.Abs(realLength); i > 0; i--)
            {
                bool MoveFlow = realLength > 0;
                this.MoveCaretHorizontical(MoveFlow);

                if (alignWord)
                    this.AlignNearestWord(MoveFlow);
            }
            this.View.AdjustCaretAndSrc(AdjustFlow.Col);
            this.SelectWithMoveCaret(isSelected);
            this.CaretMoved(this, null);
        }

        void AlignNearestWord(bool MoveFlow)
        {
            string str = this.View.LayoutLines[this.View.CaretPostion.row];
            while (this.View.CaretPostion.col > 0 &&
                str[this.View.CaretPostion.col] != Document.NewLine &&
                str[this.View.CaretPostion.col] != Document.EndOfFile)
            {
                if (!char.IsPunctuation(str[this.View.CaretPostion.col]) &&
                    !char.IsSeparator(str[this.View.CaretPostion.col]))
                {
                    this.MoveCaretHorizontical(MoveFlow);
                }
                else
                {
                    if(MoveFlow)
                        this.MoveCaretHorizontical(MoveFlow);
                    break;
                }
            }
        }

        /// <summary>
        /// キャレットを行方向に移動させる
        /// </summary>
        /// <returns>再描写する必要があるなら真を返す</returns>
        /// <param name="deltarow">移動量</param>
        public void MoveCaretVertical(int deltarow,bool isSelected)
        {
            for (int i = Math.Abs(deltarow); i > 0; i--)
                this.MoveCaretVertical(deltarow > 0);
            this.View.AdjustCaretAndSrc(AdjustFlow.Both);
            this.SelectWithMoveCaret(isSelected);
            this.CaretMoved(this, null);
        }

        /// <summary>
        /// キャレット位置の文字を一文字削除する
        /// </summary>
        public void DoDeleteAction()
        {
            if (this.SelectionLength != 0)
            {
                this.SelectedText = "";
                return;
            }
            
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            TextPoint CaretPostion = this.View.CaretPostion;
            int index = this.View.GetIndexFromLayoutLine(CaretPostion);
            if (index == this.Document.Length - 1)
                return;            
            else
            {
                int next = this.View.render.AlignIndexToNearestCluster(this.View.LayoutLines[CaretPostion.row],CaretPostion.col,AlignDirection.Forward);
                
                int lineHeadIndex = this.View.LayoutLines.GetData(CaretPostion.row).index;

                this.Document.Replace(index, lineHeadIndex + next - index, "");
            }
        }

        /// <summary>
        /// キャレット位置の文字を一文字削除し、キャレット位置を後ろにずらす
        /// </summary>
        public void DoBackSpaceAction()
        {
            if (this.SelectionLength != 0 && this.RectSelection)
            {
                this.RemoveBeforeSelectionArea(this.Document.Selections, 1);
                return;
            }
            if (this.SelectionLength != 0)
            {
                this.SelectedText = "";
                return;
            }

            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            TextPoint CurrentPostion = this.View.CaretPostion;

            if (CurrentPostion.row == 0 && CurrentPostion.col == 0)
                return;

            int oldIndex = this.View.GetIndexFromLayoutLine(CurrentPostion);

            int newCol, newIndex;
            if (CurrentPostion.col > 0)
            {
                newCol = this.View.render.AlignIndexToNearestCluster(this.View.LayoutLines[CurrentPostion.row], CurrentPostion.col - 1, AlignDirection.Back);
                newIndex = this.View.GetIndexFromLayoutLine(new TextPoint(CurrentPostion.row, newCol));
            }
            else
            {
                newIndex = this.View.GetIndexFromLayoutLine(CurrentPostion);
                newIndex--;
            }

            this.Document.Replace(newIndex, oldIndex - newIndex, "");
        }

        /// <summary>
        /// キャレット位置で行を分割する
        /// </summary>
        public void DoEnterAction()
        {
            this.DoInputChar('\n');
        }

        /// <summary>
        /// キャレット位置に文字を入力し、その分だけキャレットを進める。isInsertModeの値により動作が変わります
        /// </summary>
        /// <param name="ch"></param>
        public void DoInputChar(char ch)
        {
            this.DoInputString(ch.ToString());
        }

        /// <summary>
        /// キャレット位置に文字列を挿入し、その分だけキャレットを進める。isInsertModeの値により動作が変わります
        /// </summary>
        /// <param name="str"></param>
        public void DoInputString(string str)
        {
            if (this.SelectionLength != 0)
            {
                if (str.Length != 0 && this.RectSelection)
                    this.InsertBeforeSelectionArea(this.Document.Selections, str);
                else
                    this.SelectedText = str;
                return;
            }

            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            int index = this.View.GetIndexFromLayoutLine(this.View.CaretPostion);
            int length = 0;
            if (this.View.InsertMode == false && this.Document[index] != Document.EndOfFile && this.Document[index] != Document.NewLine)
            {
                TextPoint CaretPos = this.View.CaretPostion;
                string lineString = this.View.LayoutLines[CaretPos.row];
                int end = this.View.render.AlignIndexToNearestCluster(lineString, CaretPos.col + str.Length - 1, AlignDirection.Forward);
                if (end > lineString.Length - 1)
                    end = lineString.Length - 1;
                end += this.View.LayoutLines.GetData(CaretPos.row).index;
                length = end - index;
            }
            this.Document.Replace(index, length, str);
        }

        /// <summary>
        /// キャレットの移動に合わせて選択する
        /// </summary>
        /// <param name="isSelected">選択状態にするかどうか</param>
        /// <remarks>
        /// キャレットを移動後、このメソッドを呼び出さない場合、Select()メソッドは正常に機能しません
        /// </remarks>
        public void SelectWithMoveCaret(bool isSelected)
        {
            if (this.View.CaretPostion.col < 0 || this.View.CaretPostion.row < 0)
                return;

            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            int CaretPostion = this.View.GetIndexFromLayoutLine(this.View.CaretPostion);
            
            SelectCollection Selections = this.Document.Selections;
            if (isSelected)
                this.Select(this.AnchorIndex, CaretPostion - this.AnchorIndex);
            else
            {
                this.Document.Selections.Clear();
                this.AnchorIndex = CaretPostion;
            }
        }

        /// <summary>
        /// JumpCaretで移動した位置からキャレットを移動し、選択状態にする
        /// </summary>
        /// <param name="tp"></param>
        public void MoveCaretAndSelect(TextPoint tp)
        {
            if (tp.row != this.View.CaretPostion.row)
                this.MoveCaretVertical(tp.row - this.View.CaretPostion.row, true);
            if (tp.col != this.View.CaretPostion.col)
                this.MoveCaretHorizontical(tp.col - this.View.CaretPostion.col, true);
        }

        /// <summary>
        /// キャレット位置を既定の位置に戻す
        /// </summary>
        public void ResetCaretPostion()
        {
            this.JumpCaret(0);
        }

        /// <summary>
        /// キャレットを一文字移動させる
        /// </summary>
        /// <param name="isMoveNext">真なら１文字すすめ、そうでなければ戻す</param>
        /// <remarks>このメソッドを呼び出した後でScrollToCaretメソッドとSelectWithMoveCaretメソッドを呼び出す必要があります</remarks>
        void MoveCaretHorizontical(bool isMoveNext)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            int delta = isMoveNext ? 0 : -1;
            int prevcol = this.View.CaretPostion.col;
            int col = this.View.CaretPostion.col + delta;
            string lineString = this.View.LayoutLines[this.View.CaretPostion.row];
            if (col < 0 || this.View.CaretPostion.row >= this.View.LayoutLines.Count)
            {
                if (this.View.CaretPostion.row == 0)
                {
                    col = 0;
                    return;
                }
                this.MoveCaretVertical(false);
                this.View.AdjustCaretAndSrc(AdjustFlow.Row);  //この段階で調整しないとスクロールされない
                col = this.View.LayoutLines.GetData(this.View.CaretPostion.row).length - 1;    //最終行以外はすべて改行コードが付くはず
            }
            else if (col >= lineString.Length || lineString[col] == Document.NewLine)
            {
                this.MoveCaretVertical(true);
                this.View.AdjustCaretAndSrc(AdjustFlow.Row);  //この段階で調整しないとスクロールされない
                col = 0;
            }
            else if (lineString[col] == Document.EndOfFile)
                return;
            else
            {
                AlignDirection direction = isMoveNext ? AlignDirection.Forward : AlignDirection.Back;
                col = this.View.render.AlignIndexToNearestCluster(lineString, col, direction);
            }

            this.View.JumpCaret(this.View.CaretPostion.row, col);
        }

        /// <summary>
        /// キャレットを行方向に移動させる
        /// </summary>
        /// <param name="isMoveNext">プラス方向に移動するなら真</param>
        /// <remarks>このメソッドを呼び出した後でScrollToCaretメソッドとSelectWithMoveCaretメソッドを呼び出す必要があります</remarks>
        void MoveCaretVertical(bool isMoveNext)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            int row = this.View.CaretPostion.row + (isMoveNext ? 1 : -1);
            if (row < 0)
                row = 0;
            else if (row >= this.View.LayoutLines.Count)
                row = this.View.LayoutLines.Count - 1;

            double x = this.View.CaretLocation.X -this.View.PageBound.X;
            string line = this.View.LayoutLines[row];
            int col = this.View.GetIndexFromX(line,x);
            
            this.View.JumpCaret(row, col);
        }

        private void SelectByRectangle(Rectangle2 rect)
        {
            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");
            if (rect.Top <= rect.Bottom)
            {
                for (int i = rect.Top; i <= rect.Bottom; i++)
                {
                    int lineHeadIndex = this.View.LayoutLines.GetIndexFromLineNumber(i);
                    int StartIndex = lineHeadIndex + this.View.GetIndexFromX(this.View.LayoutLines[i], rect.X);
                    int EndIndex = lineHeadIndex + this.View.GetIndexFromX(this.View.LayoutLines[i], rect.Right) - 1; //ユーザが期待しているのより１文字多いので削る

                    Debug.Assert(StartIndex <= EndIndex, "StartIndex <= EndIndexでない");

                    Selection sel;
                    sel = Selection.Create(StartIndex, EndIndex - StartIndex + 1);

                    this.Document.Selections.Add(sel);
                }
            }
        }

        private double SelectOneLine(Point sp, Point ep)
        {
            TextPoint left = this.View.GetTextPointFromPostion(sp);
            TextPoint right = this.View.GetTextPointFromPostion(ep);

            Debug.Assert(left.row == right.row, "left.rowとright.rowの値が同じでない");

            int StartIndex = this.View.GetIndexFromLayoutLine(left);
            int EndIndex = this.View.GetIndexFromLayoutLine(right);
            int LineEndIndex = this.View.LayoutLines.GetIndexFromLineNumber(left.row) + this.View.LayoutLines.GetLengthFromLineNumber(left.row);
            if (EndIndex > LineEndIndex)
                EndIndex = LineEndIndex;
            else if (EndIndex < 0)
                EndIndex = 0;

            Selection sel;
            sel = Selection.Create(StartIndex, EndIndex - StartIndex + 1);

            this.Document.Selections.Add(sel);

            return this.View.render.GetHeight(this.View.LayoutLines[left.row]);
        }

        private void RemoveBeforeSelectionArea(SelectCollection Selections, int length)
        {
            if (this.RectSelection == false || this.SelectionLength == 0 || this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException();

            //コピーしないとReplaceCommandを呼び出した段階で書き換えられてしまう
            int selectStart = this.SelectionStart;
            int selectEnd = this.SelectionStart + this.SelectionLength - 1;
            SelectCollection temp = new SelectCollection(this.Document.Selections);
            bool reverse = temp.First().start > temp.Last().start;

            int row = this.View.LayoutLines.GetLineNumberFromIndex(selectStart);
            int lineHeadIndex = this.View.LayoutLines.GetIndexFromLineNumber(row);
            if (selectStart - length < lineHeadIndex)
                return;

            this.Document.UndoManager.BeginUndoGroup();

            if (reverse)
            {
                for (int i = 0; i < temp.Count; i++)
                {
                    Selection sel = Util.NormalizeIMaker<Selection>(temp[i]);

                    this.Document.Replace(sel.start - length, length, "");
                }
            }
            else
            {
                for (int i = temp.Count - 1; i >= 0; i--)
                {
                    Selection sel = Util.NormalizeIMaker<Selection>(temp[i]);

                    this.Document.Replace(sel.start - length, length, "");
                }
            }

            this.Document.UndoManager.EndUndoGroup();

            if (reverse)
            {
                selectEnd -= length;
                selectStart -= length * (temp.Count - 1);
            }
            else
            {
                selectStart -= length;
                selectEnd -= length * (temp.Count - 1);
            }

            this.JumpCaret(selectEnd);

            this.Select(selectStart, selectEnd - selectStart);

            return;
        }

        private void InsertBeforeSelectionArea(SelectCollection Selections, string value)
        {
            if (value.Contains(Environment.NewLine))
                return;
            if (this.RectSelection == false || this.SelectionLength == 0 || value.Length == 0 || this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException();

            //コピーしないとReplaceCommandを呼び出した段階で書き換えられてしまう
            int selectStart = this.SelectionStart;
            int selectEnd = this.SelectionStart + this.SelectionLength - 1;
            SelectCollection temp = new SelectCollection(this.Document.Selections);
            bool reverse = temp.First().start > temp.Last().start;

            this.Document.UndoManager.BeginUndoGroup();

            if (reverse)
            {
                for (int i = 0; i < temp.Count; i++)
                {
                    Selection sel = Util.NormalizeIMaker<Selection>(temp[i]);

                    this.Document.Replace(sel.start, 0, value);
                }
            }
            else
            {
                for (int i = temp.Count - 1; i >= 0; i--)
                {
                    Selection sel = Util.NormalizeIMaker<Selection>(temp[i]);

                    this.Document.Replace(sel.start, 0, value);
                }
            }

            this.Document.UndoManager.EndUndoGroup();

            if (reverse)
            {
                selectEnd += value.Length + 1;
                selectStart += value.Length * temp.Count;
            }
            else
            {
                selectStart += value.Length;
                selectEnd += value.Length * temp.Count + 1;
            }

            this.JumpCaret(selectEnd);

            this.Select(selectStart, selectEnd - selectStart);

            return;
        }

        private void RepleaceSelectionArea(SelectCollection Selections, string value)
        {
            if (value == null)
                return;

            if (this.RectSelection == false)
            {
                string str = value.Replace(Environment.NewLine, Document.NewLine.ToString());

                Selection sel = Selection.Create(this.AnchorIndex, 0);
                if (Selections.Count > 0)
                    sel = Util.NormalizeIMaker<Selection>(this.Document.Selections.First());

                this.Document.Replace(sel.start, sel.length, str);
                return;
            }

            if (this.Document.FireUpdateEvent == false)
                throw new InvalidOperationException("");

            int StartIndex = this.SelectionStart;

            if (this.SelectionLength == 0)
            {
                int i;

                this.Document.UndoManager.BeginUndoGroup();

                string[] line = value.Split(new string[] { Environment.NewLine }, StringSplitOptions.RemoveEmptyEntries);

                TextPoint Current = this.View.GetLayoutLineFromIndex(this.SelectionStart);

                for (i = 0; i < line.Length && Current.row < this.View.LayoutLines.Count; i++, Current.row++)
                {
                    if (Current.col >= this.View.LayoutLines[Current.row].Length)
                        Current.col = this.View.LayoutLines[Current.row].Length - 1;
                    StartIndex = this.View.GetIndexFromLayoutLine(Current);
                    this.Document.Replace(StartIndex, 0, line[i]);
                    StartIndex += line[i].Length;
                }

                for (; i < line.Length; i++)
                {
                    StartIndex = this.Document.Length - 1;
                    string str = Document.NewLine + line[i];
                    this.Document.Replace(StartIndex, 0, str);
                    StartIndex += str.Length;
                }

                this.Document.UndoManager.EndUndoGroup();
            }
            else
            {
                if (value.Contains(Environment.NewLine))
                    throw new InvalidOperationException();

                SelectCollection temp = new SelectCollection(this.Document.Selections); //コピーしないとReplaceCommandを呼び出した段階で書き換えられてしまう

                this.Document.UndoManager.BeginUndoGroup();

                if (temp.First().start < temp.Last().start)
                {
                    for (int i = temp.Count - 1; i >= 0; i--)
                    {
                        Selection sel = Util.NormalizeIMaker<Selection>(temp[i]);

                        StartIndex = sel.start;

                        this.Document.Replace(sel.start, sel.length, value);
                    }
                }
                else
                {
                    for (int i = 0; i < temp.Count; i++)
                    {
                        Selection sel = Util.NormalizeIMaker<Selection>(temp[i]);

                        StartIndex = sel.start;

                        this.Document.Replace(sel.start, sel.length, value);
                    }
                }

                this.Document.UndoManager.EndUndoGroup();
            }
            this.JumpCaret(StartIndex);
        }

        private string GetTextFromLineSelectArea(SelectCollection Selections)
        {
            Selection sel = Util.NormalizeIMaker<Selection>(Selections.First());

            string str = this.Document.ToString(sel.start, sel.length).Replace(Document.NewLine.ToString(), Environment.NewLine);

            return str;
        }

        string GetTextFromRectangleSelectArea(SelectCollection Selections)
        {
            StringBuilder temp = new StringBuilder();
            if (Selections.First().start < Selections.Last().start)
            {
                for (int i = 0; i < this.Document.Selections.Count; i++)
                {
                    Selection sel = Util.NormalizeIMaker<Selection>(Selections[i]);

                    string str = this.Document.ToString(sel.start, sel.length).Replace(Document.NewLine.ToString(), Environment.NewLine);
                    if (str.IndexOf(Environment.NewLine) == -1)
                        temp.AppendLine(str);
                    else
                        temp.Append(str);
                }
            }
            else
            {
                for (int i = this.Document.Selections.Count - 1; i >= 0; i--)
                {
                    Selection sel = Util.NormalizeIMaker<Selection>(Selections[i]);

                    string str = this.Document.ToString(sel.start, sel.length).Replace(Document.NewLine.ToString(), Environment.NewLine);
                    if (str.IndexOf(Environment.NewLine) == -1)
                        temp.AppendLine(str);
                    else
                        temp.Append(str);
                }
            }
            return temp.ToString();
        }

        void Document_Update(object sender, DocumentUpdateEventArgs e)
        {
            switch (e.type)
            {
                case UpdateType.Replace:
                    if(this.Document.UndoManager.Grouping == false)
                        this.JumpCaret(e.startIndex + e.insertLength);
                    break;
                case UpdateType.Clear:
                    this.View.TryScroll(0, 0);
                    break;
            }
        }
    }
}
