﻿using System;
using System.Threading.Tasks;
using System.Printing;
using System.Windows;
using System.Windows.Xps;
using Shapes = System.Windows.Shapes;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Documents.Serialization;
using System.Windows.Media;

namespace FooEditEngine.WPF
{
    /// <summary>
    /// イベントデータ
    /// </summary>
    public class ParseCommandEventArgs
    {
        /// <summary>
        /// 印刷中のページ番号
        /// </summary>
        public int PageNumber;
        /// <summary>
        /// ページ範囲内で許容されている最大の番号
        /// </summary>
        public int MaxPageNumber;
        /// <summary>
        /// 処理前の文字列
        /// </summary>
        public string Original;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="nowPage">印刷中のページ番号</param>
        /// <param name="maxPage">印刷すべき最大のページ番号</param>
        /// <param name="org">処理前の文字列</param>
        public ParseCommandEventArgs(int nowPage,int maxPage,string org)
        {
            this.PageNumber = nowPage;
            this.MaxPageNumber = maxPage;
            this.Original = org;
        }
    }

    /// <summary>
    /// コマンド処理用デリゲート
    /// </summary>
    /// <param name="sender">送信元のクラス</param>
    /// <param name="e">イベントデータ</param>
    /// <returns>処理後の文字列</returns>
    public delegate string ParseCommandHandler(object sender,ParseCommandEventArgs e);

    /// <summary>
    /// 印刷用のクラス
    /// </summary>
    public class FooPrintText
    {
        /// <summary>
        /// コンストラクター
        /// </summary>
        public FooPrintText()
        {
            this.ParseHF = new ParseCommandHandler((s, e) => { return e.Original; });
        }

        /// <summary>
        /// 印刷する最小のページ番号
        /// </summary>
        public int StartPage
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷する最大のページ番号
        /// </summary>
        public int EndPage
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷する領域の大きさ
        /// </summary>
        public System.Windows.Rect PageRect
        {
            get;
            set;
        }

        /// <summary>
        /// 対象となるドキュメント
        /// </summary>
        public Document Document
        {
            get;
            set;
        }

        /// <summary>
        /// 行番号を表示するかどうか
        /// </summary>
        public bool DrawLineNumber
        {
            get;
            set;
        }

        /// <summary>
        /// ハイパーリンクに下線を引くなら真
        /// </summary>
        public bool MarkURL
        {
            get;
            set;
        }

        /// <summary>
        /// デフォルトの文字ブラシ
        /// </summary>
        public Brush Foreground
        {
            get;
            set;
        }

        /// <summary>
        /// URLを表すブラシ
        /// </summary>
        public Brush URL
        {
            get;
            set;
        }

        /// <summary>
        /// キーワード１を表すブラシ
        /// </summary>
        public Brush Keyword1
        {
            get;
            set;
        }

        /// <summary>
        /// キーワード２を表すブラシ
        /// </summary>
        public Brush Keyword2
        {
            get;
            set;
        }

        /// <summary>
        /// コメントを表すブラシ
        /// </summary>
        public Brush Comment
        {
            get;
            set;
        }

        /// <summary>
        /// 文字リテラルを表すブラシ
        /// </summary>
        public Brush Litral
        {
            get;
            set;
        }

        /// <summary>
        /// 印刷に使用するフォント
        /// </summary>
        public FontFamily Font
        {
            get;
            set;
        }

        /// <summary>
        /// フォントサイズ
        /// </summary>
        public double FontSize
        {
            get;
            set;
        }

        /// <summary>
        /// 用紙の端で折り返すなら真
        /// </summary>
        public bool isWordRap
        {
            get;
            set;
        }

        /// <summary>
        /// ヘッダー
        /// </summary>
        public string Header
        {
            get;
            set;
        }

        /// <summary>
        /// フッター
        /// </summary>
        public string Footer
        {
            get;
            set;
        }

        /// <summary>
        /// シンタックスハイライター
        /// </summary>
        public IHilighter Hilighter
        {
            get;
            set;
        }

        /// <summary>
        /// ヘッダーやフッターを処理する
        /// </summary>
        public ParseCommandHandler ParseHF;

        /// <summary>
        /// 印刷する
        /// </summary>
        /// <param name="queue">対象となるPrintQueue</param>
        /// <param name="ticket">対象となるPrintTicket</param>
        public void Print(PrintQueue queue,PrintTicket ticket)
        {
            if (this.Font == null || this.Document == null)
                throw new InvalidOperationException();

            PrintableTextRender render = new PrintableTextRender(this.Font, this.FontSize);
            render.Foreground = this.Foreground;
            render.Comment = this.Comment;
            render.Keyword1 = this.Keyword1;
            render.Keyword2 = this.Keyword2;
            render.Litral = this.Litral;
            render.URL = this.URL;
            PrintableView view = new PrintableView(this.Document, render);
            view.isLineBreak = this.isWordRap;
            view.DrawLineNumber = this.DrawLineNumber;
            view.Header = this.Header;
            view.Footer = this.Footer;
            view.UrlMark = this.MarkURL;
            view.PageBound = this.PageRect;
            view.Hilighter = this.Hilighter;
            view.PerfomLayouts();

            XpsDocumentWriter xpswriter = PrintQueue.CreateXpsDocumentWriter(queue);
            SerializerWriterCollator collator = xpswriter.CreateVisualsCollator(ticket, ticket);

            try
            {
                int currentPage = 0;

                for (int i = 0; i < view.LayoutLines.Count; i += view.LineCountOnScreen)
                {
                    if (this.EndPage != -1 && currentPage > this.EndPage)
                        break;

                    if (this.StartPage == -1 || currentPage >= this.StartPage)
                    {
                        view.Header = this.ParseHF(this, new ParseCommandEventArgs(currentPage, this.EndPage, this.Header));
                        view.Footer = this.ParseHF(this, new ParseCommandEventArgs(currentPage, this.EndPage, this.Footer));

                        DrawingVisual dv = new DrawingVisual();
                        using (DrawingContext dc = dv.RenderOpen())
                        {
                            render.SetDrawingContext(dc);
                            view.Draw(view.PageBound);
                        }

                        collator.Write(dv);
                    }

                    view.TryPageDown();

                    currentPage++;
                }

                collator.EndBatchWrite();
            }
            catch (PrintingCanceledException)
            {
            }
            finally
            {
                view.Dispose();
            }
        }
    }
}
