/**
 *  Impulser2 GUI
 *
 *  Copyright (C) 2006-2017 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "FreeverbEditor.hpp"

FreeverbEditor::FreeverbEditor(AudioEffect *effect) :
  AEffGUIEditor(effect), freeverb((Freeverb *)effect)
{
  CBitmap backBitmap(kBack);
  rect.left = 0; rect.top = 0;
  rect.right = backBitmap.getWidth();
  rect.bottom = backBitmap.getHeight();
  this->fileSelector = NULL;
  cLabel = NULL;
  vuL = vuR = 0;
  zoomFactor = 10.0;
  returnPath1 = new char[_MAX_PATH*2];
  initialPath1 = new char[_MAX_PATH*2];
}

FreeverbEditor::~FreeverbEditor()
{
  delete[] returnPath1;
  delete[] initialPath1;
}

bool FreeverbEditor::open(void *ptr)
{
  int slot = freeverb->getCurrentSlot();
  AEffGUIEditor::open(ptr);
  CRect size;
  CPoint point, offset;
  
  CBitmap *backBitmap = new CBitmap(kBack);
  size(0, 0, backBitmap->getWidth(), backBitmap->getHeight());
  CFrame * frame = new CFrame(size, ptr, this);
  frame->setBackground(backBitmap);
  backBitmap->forget();

  // load button
  CBitmap *onOffButton = new CBitmap(kOnOff);
  size (0, 0, onOffButton->getWidth(), onOffButton->getHeight () / 2);
  size.offset (10, 10);
  point (0, 0);
  cKickButton = new CKickButton(size, this, kKickTag, onOffButton->getHeight()/2, onOffButton, point);
  frame->addView(cKickButton);
  
  // filename
  size (0, 0, 715, 15);
  size.offset (40, 10);
  cLabel = new CTextLabel(size, (char*)freeverb->getSlotFileNameA(slot), 0);
  if(cLabel)
    {
      cLabel->setFont(kNormalFontBig);
      cLabel->setFontColor(kBlackCColor);
      cLabel->setBackColor(kWhiteCColor);
      frame->addView(cLabel);
    }
  
  // reset button
  CBitmap *resetButton = new CBitmap(kResetBitmap);
  size(0, 0, resetButton->getWidth(), resetButton->getHeight()/2);
  size.offset (715+40, 10);
  point(0, 0);
  cResetButton = new CKickButton(size, this, kResetTag, resetButton->getHeight()/2, resetButton, point);
  frame->addView(cResetButton);

  // unload button
  CBitmap *unloadButton = new CBitmap(kUnloadBitmap);
  size(0, 0, unloadButton->getWidth(), unloadButton->getHeight()/2);
  size.offset(715+40+resetButton->getWidth(), 10);
  point(0, 0);
  cUnloadButton = new CKickButton(size, this, kUnloadTag, unloadButton->getHeight()/2, unloadButton, point);
  frame->addView(cUnloadButton);
  
  // fader bitmaps
  CBitmap* hFaderBody   = new CBitmap(kFaderBodyId);
  CBitmap* hFaderHandle = new CBitmap(kFaderHandleId);
  int minPos = kFaderY;
  int maxPos = kFaderY + hFaderBody->getHeight() - hFaderHandle->getHeight();
  
  // CVuMeter
  CBitmap* vuOnBitmap  = new CBitmap (kVuOnBitmap);
  CBitmap* vuOffBitmap = new CBitmap (kVuOffBitmap);
  // VuMeter
  point(0, 0);
  offset(0, 0);
  size(kFaderX, kFaderY, kFaderX + hFaderBody->getWidth(), kFaderY + hFaderBody->getHeight());
  //size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  cVuMeterL = new CVuMeter(size, vuOnBitmap, vuOffBitmap, hFaderBody->getHeight());
  cVuMeterL->setDecreaseStepValue (0.1f);
  cVuMeterL->setValue(0.0f);
  frame->addView(cVuMeterL);
  size.offset(0 + hFaderBody->getWidth(), 0);
  cVuMeterR = new CVuMeter(size, vuOnBitmap, vuOffBitmap, hFaderBody->getHeight());
  cVuMeterR->setDecreaseStepValue (0.1f);
  cVuMeterR->setValue(0.0f);
  frame->addView(cVuMeterR);
  
  // Fader(overAll)
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  thrFader = new CVerticalSlider(size, this, KOATag+KOAThreshold, minPos, maxPos, hFaderHandle, hFaderBody, point);
  thrFader->setOffsetHandle (offset);
  thrFader->setValue(limitFader(freeverb->getParameter(KATParam(KOAThreshold))));
  thrFader->setFreeClick(false);
  thrFader->setZoomFactor(zoomFactor);
  frame->addView(thrFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  ceiFader = new CVerticalSlider(size, this, KOATag+KOACeiling, minPos, maxPos, hFaderHandle, hFaderBody, point);
  ceiFader->setOffsetHandle(offset);
  ceiFader->setValue(limitFader(freeverb->getParameter(KATParam(KOACeiling))));
  ceiFader->setFreeClick(false);
  ceiFader->setZoomFactor(zoomFactor);
  frame->addView(ceiFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  relFader = new CVerticalSlider(size, this, KOATag+KOARelease, minPos, maxPos, hFaderHandle, hFaderBody, point);
  relFader->setOffsetHandle(offset);
  relFader->setValue(limitFader(freeverb->getParameter(KATParam(KOARelease))));
  relFader->setFreeClick(false);
  relFader->setZoomFactor(zoomFactor);
  frame->addView(relFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  dryFader = new CVerticalSlider(size, this, KOATag+KOADry, minPos, maxPos, hFaderHandle, hFaderBody, point);
  dryFader->setOffsetHandle(offset);
  dryFader->setValue(limitFader(freeverb->getParameter(KATParam(KOADry))));
  dryFader->setFreeClick(false);
  dryFader->setZoomFactor(zoomFactor);
  frame->addView(dryFader);

  //
  // slot
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  wetFader = new CVerticalSlider(size, this, KWet, minPos, maxPos, hFaderHandle, hFaderBody, point);
  wetFader->setOffsetHandle(offset);
  wetFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,KWet))));
  wetFader->setFreeClick(false);
  wetFader->setZoomFactor(zoomFactor);
  frame->addView (wetFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  lpfFader = new CVerticalSlider(size, this, KLPF, minPos, maxPos, hFaderHandle, hFaderBody, point);
  lpfFader->setOffsetHandle(offset);
  lpfFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,KLPF))));
  lpfFader->setFreeClick(false);
  lpfFader->setZoomFactor(zoomFactor);
  frame->addView(lpfFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  hpfFader = new CVerticalSlider(size, this, KHPF, minPos, maxPos, hFaderHandle, hFaderBody, point);
  hpfFader->setOffsetHandle(offset);
  hpfFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,KHPF))));
  hpfFader->setFreeClick(false);
  hpfFader->setZoomFactor(zoomFactor);
  frame->addView(hpfFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  idlFader = new CVerticalSlider(size, this, KDelay, minPos, maxPos, hFaderHandle, hFaderBody, point);
  idlFader->setOffsetHandle(offset);
  idlFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,KDelay))));
  idlFader->setFreeClick(false);
  idlFader->setZoomFactor(zoomFactor);
  frame->addView(idlFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  widFader = new CVerticalSlider(size, this, KWidth, minPos, maxPos, hFaderHandle, hFaderBody, point);
  widFader->setOffsetHandle(offset);
  widFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,KWidth))));
  widFader->setFreeClick(false);
  widFader->setZoomFactor(zoomFactor);
  frame->addView(widFader);
  //
  size.offset (kFaderInc + hFaderBody->getWidth(), 0);
  strFader = new CVerticalSliderE(size, this, KIRStretch, minPos, maxPos, hFaderHandle, hFaderBody, point);
  strFader->setOffsetHandle(offset);
  strFader->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,KIRStretch))));
  strFader->setFreeClick(false);
  strFader->setZoomFactor(zoomFactor);
  frame->addView(strFader);
  
  // VuTextLabels L/R
  size(kFaderX - kFaderInc/2, kFaderY + hFaderBody->getHeight(),
       kFaderX - kFaderInc/2 + kDisplayXWidth, kFaderY + hFaderBody->getHeight() + kDisplayHeight);
  size.offset(hFaderBody->getWidth(), 0);

  // ParameterTextEdit
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  thrTextEdit = new CTextEdit(size, this, KRealTag+KOATag+KOAThreshold, NULL, NULL, kCenterText);
  thrTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  thrTextEdit->setFont(kNormalFont);
  thrTextEdit->setFontColor(kBlackCColor);
  thrTextEdit->setBackColor(kWhiteCColor);
  thrTextEdit->setFrameColor(kWhiteCColor);
  thrTextEdit->setValue(PARAM2DBA(freeverb->getParameter(KATParam(KOAThreshold))));
  frame->addView(thrTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  ceiTextEdit = new CTextEdit(size, this, KRealTag+KOATag+KOACeiling, NULL, NULL, kCenterText);
  ceiTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  ceiTextEdit->setFont(kNormalFont);
  ceiTextEdit->setFontColor(kBlackCColor);
  ceiTextEdit->setBackColor(kWhiteCColor);
  ceiTextEdit->setFrameColor(kWhiteCColor);
  ceiTextEdit->setValue(PARAM2DBA(freeverb->getParameter(KATParam(KOACeiling))));
  frame->addView(ceiTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  relTextEdit = new CTextEdit(size, this, KRealTag+KOATag+KOARelease, "", NULL, kCenterText);
  relTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  relTextEdit->setFont(kNormalFont);
  relTextEdit->setFontColor(kBlackCColor);
  relTextEdit->setBackColor(kWhiteCColor);
  relTextEdit->setFrameColor(kWhiteCColor);
  relTextEdit->setValue(PARAM2RELEASE(freeverb->getParameter(KATParam(KOARelease))));
  frame->addView(relTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  dryTextEdit = new CTextEdit(size, this, KRealTag+KOATag+KOADry, "", NULL, kCenterText);
  dryTextEdit->setStringConvert(FreeverbEditor::stringConvertDB);
  dryTextEdit->setFont(kNormalFont);
  dryTextEdit->setFontColor(kBlackCColor);
  dryTextEdit->setBackColor(kWhiteCColor);
  dryTextEdit->setFrameColor(kGreenCColor);
  dryTextEdit->setValue(PARAM2DB(freeverb->getParameter(KATParam(KOADry))));
  frame->addView(dryTextEdit);

  // slots
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  wetTextEdit = new CTextEdit(size, this, KRealTag+KWet, "", NULL, kCenterText);
  wetTextEdit->setStringConvert(FreeverbEditor::stringConvertDB);
  wetTextEdit->setFont(kNormalFont);
  wetTextEdit->setFontColor(kBlackCColor);
  wetTextEdit->setBackColor(kWhiteCColor);
  wetTextEdit->setFrameColor(kBlueCColor);
  wetTextEdit->setValue(PARAM2DB(freeverb->getParameter(KRTParam(slot,KWet))));
  frame->addView(wetTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  lpfTextEdit = new CTextEdit(size, this, KRealTag+KLPF,
			     "", NULL, kCenterText);
  lpfTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  lpfTextEdit->setFont(kNormalFont);
  lpfTextEdit->setFontColor(kBlackCColor);
  lpfTextEdit->setBackColor(kWhiteCColor);
  lpfTextEdit->setFrameColor(kWhiteCColor);
  lpfTextEdit->setValue(freeverb->getParameter(KRTParam(slot,KLPF)));
  frame->addView(lpfTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  hpfTextEdit = new CTextEdit(size, this, KRealTag+KHPF, "", NULL, kCenterText);
  hpfTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  hpfTextEdit->setFont(kNormalFont);
  hpfTextEdit->setFontColor(kBlackCColor);
  hpfTextEdit->setBackColor(kWhiteCColor);
  hpfTextEdit->setFrameColor(kWhiteCColor);
  hpfTextEdit->setValue(freeverb->getParameter(KRTParam(slot,KHPF)));
  frame->addView(hpfTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  idlTextEdit = new CTextEdit(size, this, KRealTag+KDelay, "", NULL, kCenterText);
  idlTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  idlTextEdit->setFont(kNormalFont);
  idlTextEdit->setFontColor(kBlackCColor);
  idlTextEdit->setBackColor(kWhiteCColor);
  idlTextEdit->setFrameColor(kWhiteCColor);
  idlTextEdit->setValue(PARAM2DELAYTIME(freeverb->getParameter(KRTParam(slot,KDelay))));
  frame->addView(idlTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  widTextEdit = new CTextEdit(size, this, KRealTag+KWidth, "", NULL, kCenterText);
  widTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  widTextEdit->setFont(kNormalFont);
  widTextEdit->setFontColor(kBlackCColor);
  widTextEdit->setBackColor(kWhiteCColor);
  widTextEdit->setFrameColor(kWhiteCColor);
  widTextEdit->setValue(100*freeverb->getParameter(KRTParam(slot,KWidth)));
  frame->addView(widTextEdit);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  strTextEdit = new CTextEdit(size, this, KRealTag+KIRStretch, NULL, NULL, kCenterText);
  strTextEdit->setStringConvert(FreeverbEditor::stringConvert2);
  strTextEdit->setFont(kNormalFont);
  strTextEdit->setFontColor(kBlackCColor);
  strTextEdit->setBackColor(kWhiteCColor);
  strTextEdit->setFrameColor(kWhiteCColor);
  strTextEdit->setValue(PARAM2STRETCH(freeverb->getNRTParameter(KNRTParam(slot, KIRStretch))));
  frame->addView(strTextEdit);

  // Button
  size(kFaderX - kFaderInc/2, kFaderY + hFaderBody->getHeight() + kDisplayHeight*2,
       kFaderX - kFaderInc/2 + kDisplayXWidth, kFaderY + hFaderBody->getHeight() + kDisplayHeight*3);
  size.offset(4*(kFaderInc + hFaderBody->getWidth())-kFaderInc, 0);
  // Skip Limiter OnOff Button
  CBitmap *onOffBmp = new CBitmap(kOnOffBitmap);
  skipLimiterB = new COnOffButton(size, this, KOATag + KOASkipLimiter, onOffBmp);
  skipLimiterB->setValue(effect->getParameter(KOASkipLimiter));
  frame->addView(skipLimiterB);

  // Button
  size(kFaderX - kFaderInc/2, kFaderY + hFaderBody->getHeight() + kDisplayHeight*2,
       kFaderX - kFaderInc/2 + kDisplayXWidth, kFaderY + hFaderBody->getHeight() + kDisplayHeight*3);
  size.offset(6*(kFaderInc + hFaderBody->getWidth())-kFaderInc, 0);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CBitmap *muteWetBitmap = new CBitmap (kMuteWetBitmap);
  muteWetButton = new COnOffButton (size, this, KMuteWet, muteWetBitmap);
  muteWetButton->setValue(freeverb->getParameter(KRTParam(slot,KMuteWet)));
  frame->addView(muteWetButton);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CBitmap *m2sBitmap = new CBitmap (kM2SBitmap);
  m2sButton = new COnOffButton (size, this, KM2S, m2sBitmap);
  m2sButton->setValue(freeverb->getParameter(KRTParam(slot,KM2S)));
  frame->addView(m2sButton);
  //
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CBitmap *swapBitmap = new CBitmap (kSWAPBitmap);
  lrSwapButton = new COnOffButton (size, this, KSwap, swapBitmap);
  lrSwapButton->setValue(freeverb->getParameter(KRTParam(slot,KSwap)));
  frame->addView(lrSwapButton);

  size(kFaderX - kFaderInc/2, kFaderY + hFaderBody->getHeight() + kDisplayHeight,
       kFaderX - kFaderInc/2 + kDisplayXWidth, kFaderY + hFaderBody->getHeight() + kDisplayHeight*2);

  // Labels
  size.offset(2*(kFaderInc + hFaderBody->getWidth())-kFaderInc, 0);
  CTextLabel * uLabela1 = new CTextLabel(size, "THR", 0);
  uLabela1->setBackColor(kBlueCColor);
  frame->addView(uLabela1);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CTextLabel * uLabela2 = new CTextLabel(size, "CEI", 0);
  uLabela2->setBackColor(kBlueCColor);
  frame->addView(uLabela2);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CTextLabel * uLabela3 = new CTextLabel(size, "REL", 0);
  uLabela3->setBackColor(kBlueCColor);
  frame->addView(uLabela3);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CTextLabel * uLabel1 = new CTextLabel(size, "DRY", 0);
  uLabel1->setBackColor(kBlueCColor);
  frame->addView(uLabel1);
  
  size.offset(kFaderInc + hFaderBody->getWidth(), 0); CTextLabel * uLabel2 = new CTextLabel(size, "WET", 0); frame->addView(uLabel2);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0); CTextLabel * uLabel3 = new CTextLabel(size, "LPF", 0); frame->addView(uLabel3);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0); CTextLabel * uLabel4 = new CTextLabel(size, "HPF", 0); frame->addView(uLabel4);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0); CTextLabel * uLabel5 = new CTextLabel(size, "IDL", 0); frame->addView(uLabel5);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0); CTextLabel * uLabel6 = new CTextLabel(size, "WID", 0); frame->addView(uLabel6);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0); CTextLabel * uLabel7 = new CTextLabel(size, "STR", 0); frame->addView(uLabel7);
  size.offset(kFaderInc + hFaderBody->getWidth(), 0);

  CBitmap *ckKnob1 = new CBitmap(kKnob1Bitmap);
  
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+8,                     kFaderY,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+8+ckKnob1->getWidth(), kFaderY+ckKnob1->getHeight()/200);
  lrbAKnob = new CAnimKnob(size, this, KLRBalance, 200, ckKnob1->getHeight()/200, ckKnob1, CPoint(0,0));
  lrbAKnob->setValue(limitFader(freeverb->getParameter(KRTParam(slot,KLRBalance))));
  frame->addView(lrbAKnob);
  
  size.offset(kFaderInc+hFaderBody->getWidth()+2,0);
  irattAKnob = new CAnimKnobE(size, this, KIRAttack, 200, ckKnob1->getHeight()/200, ckKnob1, CPoint(0,0));
  irattAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,KIRAttack))));
  frame->addView(irattAKnob);
  
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+8,                    kFaderY+kDisplayHeight*2+ckKnob1->getHeight()/200,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+8+ckKnob1->getWidth(),kFaderY+kDisplayHeight*2+ckKnob1->getHeight()/200*2);
  irholAKnob = new CAnimKnobE(size, this, KIRHold, 200, ckKnob1->getHeight()/200, ckKnob1, CPoint(0,0));
  irholAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,KIRHold))));
  frame->addView(irholAKnob);
  
  size.offset(kFaderInc+hFaderBody->getWidth()+2,0);
  irdecAKnob = new CAnimKnobE(size, this, KIRDecay, 200, ckKnob1->getHeight()/200, ckKnob1, CPoint(0,0));
  irdecAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,KIRDecay))));
  frame->addView(irdecAKnob);
  
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+8,                    kFaderY+kDisplayHeight*4+ckKnob1->getHeight()/200*2,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+8+ckKnob1->getWidth(),kFaderY+kDisplayHeight*4+ckKnob1->getHeight()/200*3);
  irsusAKnob = new CAnimKnobE(size, this, KIRSustain, 200, ckKnob1->getHeight()/200, ckKnob1, CPoint(0,0));
  irsusAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,KIRSustain))));
  frame->addView(irsusAKnob);
  
  size.offset(kFaderInc+hFaderBody->getWidth()+2,0);
  irrelAKnob = new CAnimKnobE(size, this, KIRRelease, 200, ckKnob1->getHeight()/200, ckKnob1, CPoint(0,0));
  irrelAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,KIRRelease))));
  frame->addView(irrelAKnob);

  //
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5,               kFaderY+ckKnob1->getHeight()/200,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5+kDisplayXWidth,kFaderY+ckKnob1->getHeight()/200+kDisplayHeight);
  lrbTextEdit = new CTextEdit(size, this, KRealTag+KLRBalance, "", NULL, kCenterText);
  lrbTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  lrbTextEdit->setFont(kNormalFont);
  lrbTextEdit->setFontColor(kBlackCColor); lrbTextEdit->setBackColor(kWhiteCColor); lrbTextEdit->setFrameColor(kWhiteCColor);
  lrbTextEdit->setValue(PARAM2LRB(freeverb->getParameter(KRTParam(slot,KLRBalance))));
  frame->addView(lrbTextEdit);
  
  size.offset(kFaderInc+hFaderBody->getWidth()+2,0);
  irattTextEdit = new CTextEdit(size, this, KRealTag+KIRAttack, "", NULL, kCenterText);
  irattTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  irattTextEdit->setFont(kNormalFont);
  irattTextEdit->setFontColor(kBlackCColor); irattTextEdit->setBackColor(kWhiteCColor); irattTextEdit->setFrameColor(kWhiteCColor);
  irattTextEdit->setValue(PARAM2IRATTACK(freeverb->getNRTParameter(KNRTParam(slot,KIRAttack))));
  frame->addView(irattTextEdit);

  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5,               kFaderY+ckKnob1->getHeight()/200*2+kDisplayHeight*2,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5+kDisplayXWidth,kFaderY+ckKnob1->getHeight()/200*2+kDisplayHeight*3);
  irholTextEdit = new CTextEdit(size, this, KRealTag+KIRHold, "", NULL, kCenterText);
  irholTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  irholTextEdit->setFont(kNormalFont);
  irholTextEdit->setFontColor(kBlackCColor); irholTextEdit->setBackColor(kWhiteCColor); irholTextEdit->setFrameColor(kWhiteCColor);
  irholTextEdit->setValue(PARAM2IRHOLD(freeverb->getNRTParameter(KNRTParam(slot,KIRHold))));
  frame->addView(irholTextEdit);
  
  size.offset(kFaderInc+hFaderBody->getWidth()+2,0);
  irdecTextEdit = new CTextEdit(size, this, KRealTag+KIRDecay, "", NULL, kCenterText);
  irdecTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  irdecTextEdit->setFont(kNormalFont);
  irdecTextEdit->setFontColor(kBlackCColor); irdecTextEdit->setBackColor(kWhiteCColor); irdecTextEdit->setFrameColor(kWhiteCColor);
  irdecTextEdit->setValue(PARAM2IRDECAY(freeverb->getNRTParameter(KNRTParam(slot,KIRDecay))));
  frame->addView(irdecTextEdit);

  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5,               kFaderY+ckKnob1->getHeight()/200*3+kDisplayHeight*4,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5+kDisplayXWidth,kFaderY+ckKnob1->getHeight()/200*3+kDisplayHeight*5);
  irsusTextEdit = new CTextEdit(size, this, KRealTag+KIRSustain, "", NULL, kCenterText);
  irsusTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  irsusTextEdit->setFont(kNormalFont);
  irsusTextEdit->setFontColor(kBlackCColor); irsusTextEdit->setBackColor(kWhiteCColor); irsusTextEdit->setFrameColor(kWhiteCColor);
  irsusTextEdit->setValue(PARAM2IRSUSTAIN(freeverb->getNRTParameter(KNRTParam(slot,KIRSustain))));
  frame->addView(irsusTextEdit);
  
  size.offset(kFaderInc+hFaderBody->getWidth()+2,0);
  irrelTextEdit = new CTextEdit(size, this, KRealTag+KIRRelease, "", NULL, kCenterText);
  irrelTextEdit->setStringConvert(FreeverbEditor::stringConvert);
  irrelTextEdit->setFont(kNormalFont);
  irrelTextEdit->setFontColor(kBlackCColor); irrelTextEdit->setBackColor(kWhiteCColor); irrelTextEdit->setFrameColor(kWhiteCColor);
  irrelTextEdit->setValue(PARAM2IRRELEASE(freeverb->getNRTParameter(KNRTParam(slot,KIRRelease))));
  frame->addView(irrelTextEdit);
  
  // Label
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5,                kFaderY + ckKnob1->getHeight()/200+kDisplayHeight,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5+kDisplayXWidth, kFaderY + ckKnob1->getHeight()/200+kDisplayHeight*2);
  CTextLabel * uLabel8 = new CTextLabel(size, "LRB", 0);  frame->addView(uLabel8);size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CTextLabel * uLabel9 = new CTextLabel(size, "IATT", 0); frame->addView(uLabel9);size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5,                kFaderY + ckKnob1->getHeight()/200*2+kDisplayHeight*3,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5+kDisplayXWidth, kFaderY + ckKnob1->getHeight()/200*2+kDisplayHeight*4);
  CTextLabel * uLabela = new CTextLabel(size, "IHOL", 0); frame->addView(uLabela);size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CTextLabel * uLabelb = new CTextLabel(size, "IDEC", 0); frame->addView(uLabelb);size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  size(kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5,                kFaderY + ckKnob1->getHeight()/200*3+kDisplayHeight*5,
       kFaderX+(kFaderInc+hFaderBody->getWidth())*11+kSlotIncY+5+kDisplayXWidth, kFaderY + ckKnob1->getHeight()/200*3+kDisplayHeight*6);
  CTextLabel * uLabelc = new CTextLabel(size, "ISUS", 0); frame->addView(uLabelc);size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  CTextLabel * uLabeld = new CTextLabel(size, "IREL", 0); frame->addView(uLabeld);size.offset(kFaderInc + hFaderBody->getWidth(), 0);
  
  // Slot Selector
  CBitmap *ckButton = new CBitmap(kSelectBitmap);
  size(kFaderX + (kFaderInc+hFaderBody->getWidth())*12 + kSlotIncY +60,                        kFaderY,
       kFaderX + (kFaderInc+hFaderBody->getWidth())*12 + kSlotIncY +60 + ckButton->getWidth(), kFaderY + ckButton->getHeight()/2);
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      cSlotSelector[si] = new COnOffButton(size, this, KSlotHead+si, ckButton);
      if(freeverb->getCurrentSlot() == si)
	{ cSlotSelector[si]->setValue(1); }
      else
	{ cSlotSelector[si]->setValue(0); }
      frame->addView(cSlotSelector[si]);
      size.offset(0, ckButton->getHeight()/2 + kSlotIncY);
    }
  // slot number
  size(kFaderX + (kFaderInc+hFaderBody->getWidth())*12 + kSlotIncY + ckButton->getWidth()  +60, kFaderY,
       kFaderX + (kFaderInc+hFaderBody->getWidth())*12 + kSlotIncY + ckButton->getWidth()*2+60,
       kFaderY + ckButton->getHeight()/2);
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      char nstr[16];
      snprintf(nstr, 16, "%d", si);
      CTextLabel * uCL = new CTextLabel(size, nstr, 0);
      uCL->setFontColor(kWhiteCColor);
      uCL->setBackColor(kBlueCColor);
      uCL->setFrameColor(kBlueCColor);
      frame->addView(uCL);
      size.offset(0, ckButton->getHeight()/2 + kSlotIncY);
    }
  // slot filename
  size(kFaderX + (kFaderInc+hFaderBody->getWidth())*12 + kSlotIncY + ckButton->getWidth()*2 +60, kFaderY,
       kFaderX + (kFaderInc+hFaderBody->getWidth())*12 + kSlotIncY + ckButton->getWidth()*2 + kSlotLabelW+60,
       kFaderY + ckButton->getHeight()/2);
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    {
      cSlotLabel[si] = new CTextLabel(size, "", 0);
      cSlotLabel[si]->setFont(kNormalFont);
      cSlotLabel[si]->setFontColor(kBlueCColor);
      cSlotLabel[si]->setBackColor(kWhiteCColor);
      frame->addView(cSlotLabel[si]);
      size.offset(0, ckButton->getHeight()/2 + kSlotIncY);
    }

  // Version Label
  size(80, backBitmap->getHeight() - 21, 80 + 70, backBitmap->getHeight() - 6);
  CTextLabel * vLabel = new CTextLabel(size, "VST "VERSION, 0);
  vLabel->setFontColor(kWhiteCColor);
  vLabel->setBackColor(kBlueCColor);
  vLabel->setFrameColor(kBlueCColor);
  frame->addView(vLabel);

  // Splash
  CBitmap *splashBitmap = new CBitmap(kSplashBitmap);
  point(0, 0);
  CRect toDisplay (0, 0, splashBitmap->getWidth(), splashBitmap->getHeight());
  toDisplay.offset (50, 50);
  cSplashScreen = new CSplashScreen(size, this, kAbout, splashBitmap, toDisplay, point);
  frame->addView(cSplashScreen);
  splashBitmap->forget();  

  // Info Label
  size(310, backBitmap->getHeight() - 51, 310 + 70, backBitmap->getHeight() - 36);
  cInfoLabel = new CTextLabel(size, "", 0);
  cInfoLabel->setFontColor(kWhiteCColor);
  cInfoLabel->setBackColor(kBlueCColor);
  cInfoLabel->setFrameColor(kBlueCColor);
  //frame->addView(cInfoLabel);

  // Fs Label
  size(310, backBitmap->getHeight() - 21, 310 + 70, backBitmap->getHeight() - 6);
  cFsInfoLabel = new CTextLabel(size, "", 0);
  cFsInfoLabel->setFontColor(kWhiteCColor);
  cFsInfoLabel->setBackColor(kBlueCColor);
  cFsInfoLabel->setFrameColor(kBlueCColor);
  frame->addView(cFsInfoLabel);

  // IR Graph
  size(0, 0, kcGraphWidth, kcGraphHeight);
  size.offset(kpGraphHX, kpGraphHY);
  ccGraph = new CAHDSRGraph(size, kBlueCColor);
  frame->addView(ccGraph);

  updateGraph();

  frame->setDirty();
  this->frame = frame;

  // setSlotLabeA runs under frame != NULL
  for(int si = 0;si < MODEL_SLOT_SIZE;si ++)
    setSlotLabelA(si, (char*)freeverb->getSlotFileNameA(si));
  
  return true;
}

void FreeverbEditor::updateGraph()
{
  int slot = freeverb->getCurrentSlot();
  ccGraph->updateValue(freeverb->getNRTParameter(KNRTParam(slot,KIRAttack)), freeverb->getNRTParameter(KNRTParam(slot,KIRHold)),
		       freeverb->getNRTParameter(KNRTParam(slot,KIRDecay)), freeverb->getNRTParameter(KNRTParam(slot,KIRSustain)),
		       freeverb->getNRTParameter(KNRTParam(slot,KIRRelease)));
}

void FreeverbEditor::stringConvert(float value, char * string)
{
  snprintf(string, 256, "%1.1f", value);
}

void FreeverbEditor::stringConvert2(float value, char * string)
{
  snprintf(string, 256, "%1.3f", value);
}

void FreeverbEditor::stringConvertDB(float value, char * string)
{
  if(value <= -100)
    snprintf(string, 256, "mute");
  else
    snprintf(string, 256, "%1.1f", value);
}

void FreeverbEditor::close()
{
  if (frame != NULL) delete frame;
  frame = NULL; cLabel = NULL;
}

void FreeverbEditor::idle()
{
  if(!frame) return;
  if(cVuMeterL) cVuMeterL->setValue(vuL);
  if(cVuMeterR) cVuMeterR->setValue(vuR);
  char vs[16];
  sprintf(vs, "%1.0f", freeverb->getSampleRate());
  if(cFsInfoLabel) cFsInfoLabel->setText(vs);
  if(cInfoLabel) cInfoLabel->setText(infoText.c_str());
  AEffGUIEditor::idle();
}

void FreeverbEditor::setInfo(const char * str){ infoText = str; }

// called from non-GUI process
void FreeverbEditor::setParameter(VstInt32 index, float value)
{  
  if(!frame) return;
  
  // overAll parameters
  if(index >= KNumParams*MODEL_SLOT_SIZE&&index < KNumParams*MODEL_SLOT_SIZE+KOANumParams)
    {
      int l = index-KNumParams*MODEL_SLOT_SIZE;
      switch(l)
        {
	case KOASkipLimiter:
	  skipLimiterB->setValue(freeverb->getParameter(index));
	  break;
        case KOARelease:
	  if(relFader)
	    relFader->setValue(limitFader(freeverb->getParameter(index)));
	  if(relTextEdit)
	    relTextEdit->setValue(PARAM2RELEASE(freeverb->getParameter(index)));
          break;
        case KOACeiling:
	  if(ceiFader)
	    ceiFader->setValue(limitFader(freeverb->getParameter(index)));
	  if(ceiTextEdit)
	    ceiTextEdit->setValue(PARAM2DBA(freeverb->getParameter(index)));
          break;
        case KOAThreshold:
	  if(thrFader)
	    thrFader->setValue(limitFader(freeverb->getParameter(index)));
	  if(thrTextEdit)
	    thrTextEdit->setValue(PARAM2DBA(freeverb->getParameter(index)));
          break;
	case KOADry:
	  if(dryFader)
	    dryFader->setValue(limitFader(freeverb->getParameter(index)));
	  if(dryTextEdit)
	    dryTextEdit->setValue(PARAM2DB(freeverb->getParameter(index)));	  
        default:
          break;
        }
      // will not process slot parameters
      return;
    }

  // Set Slot Parameters  
  VstInt32 slot = 0;
  // VST
  if(index >= 0&&index < KNumParams*MODEL_SLOT_SIZE)
    {
      slot = index/KNumParams; index = index%KNumParams;
    }
  // Internal Parameters (non VST-RT-Parameters)
  if(index >= KInternal&&index < KInternal+KNumNRTParams*MODEL_SLOT_SIZE)
    {
      slot = (index - KInternal)/KNumNRTParams;
      index = index - KNumNRTParams*slot;
    }
  
  switch (index)
    {
    case KWet:
      if(wetFader) wetFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,index))));
      if(wetTextEdit) wetTextEdit->setValue(PARAM2DB(freeverb->getParameter(KRTParam(slot,index))));
      break;
    case KLPF:
      if(lpfFader) lpfFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,index))));
      if(lpfTextEdit) lpfTextEdit->setValue(freeverb->getParameter(KRTParam(slot,index)));
      break;
    case KHPF:
      if(hpfFader) hpfFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,index))));
      if(hpfTextEdit) hpfTextEdit->setValue(freeverb->getParameter(KRTParam(slot,index)));
      break;
    case KDelay:
      if(idlFader) idlFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,index))));
      if(idlTextEdit) idlTextEdit->setValue(PARAM2DELAYTIME(freeverb->getParameter(KRTParam(slot,index))));
      break;
    case KWidth:
      if(widFader) widFader->setValue(limitFader(freeverb->getParameter(KRTParam(slot,index))));
      if(widTextEdit) widTextEdit->setValue(100*freeverb->getParameter(KRTParam(slot,index)));
      break;
    case KLRBalance:
      if(lrbAKnob) lrbAKnob->setValue(limitFader(freeverb->getParameter(KRTParam(slot,index))));
      if(lrbTextEdit) lrbTextEdit->setValue(PARAM2LRB(freeverb->getParameter(KRTParam(slot,index))));
      break;
      
      // Non-realtime
    case KIRStretch:
      if(strFader) strFader->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,index))));
      if(strTextEdit) strTextEdit->setValue(PARAM2STRETCH(freeverb->getNRTParameter(KNRTParam(slot,index))));
      break;
    case KIRAttack:
      if(irattAKnob) irattAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,index))));
      if(irattTextEdit) irattTextEdit->setValue(PARAM2IRATTACK(freeverb->getNRTParameter(KNRTParam(slot,index))));
      break;      
    case KIRHold:
      if(irholAKnob) irholAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,index))));
      if(irholTextEdit) irholTextEdit->setValue(PARAM2IRHOLD(freeverb->getNRTParameter(KNRTParam(slot,index))));
      break;
    case KIRDecay:
      if(irdecAKnob) irdecAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,index))));
      if(irdecTextEdit) irdecTextEdit->setValue(PARAM2IRDECAY(freeverb->getNRTParameter(KNRTParam(slot,index))));
      break;
    case KIRSustain:
      if(irsusAKnob) irsusAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,index))));
      if(irsusTextEdit) irsusTextEdit->setValue(PARAM2IRSUSTAIN(freeverb->getNRTParameter(KNRTParam(slot,index))));
      break;
    case KIRRelease:
      if(irrelAKnob) irrelAKnob->setValue(limitFader(freeverb->getNRTParameter(KNRTParam(slot,index))));
      if(irrelTextEdit) irrelTextEdit->setValue(PARAM2IRRELEASE(freeverb->getNRTParameter(KNRTParam(slot,index))));
      break;
      
    case KM2S:
      m2sButton->setValue(value);
      break;
    case KSwap:
      lrSwapButton->setValue(value);
      break;
    case KMuteWet:
      muteWetButton->setValue(value);
      break;
    default:
      break;
    }
  
  updateGraph();
}

void FreeverbEditor::setVuMeter(float L, float R)
{
  vuL = L; vuR = R;
}

void FreeverbEditor::valueChanged(CControl * pControl)
{
  // any GUI change will be process by this method.
  // the control Tag has two additional value to mark real-value and Overall/slot parameters.
  VstInt32 OAP = KNumParams*MODEL_SLOT_SIZE;
  VstInt32 Tag = pControl->getTag();
  float Value = pControl->getValue();
  // freeverb->writeLogA("FreeverbEditor.cpp: vC: %d %f\n", Tag, Value);
  // realTag: convert real parameter to 0~1 value
  if(Tag >= KRealTag)
    {
      char str[16];
      ((CTextEdit*)pControl)->getText(str);
      Value = atof(str);
      Tag -= KRealTag;
      switch(Tag)
	{
	case KOATag+KOARelease:
	  Value = RELEASE2PARAM(Value); break;
	case KOATag+KOACeiling:
	  Value = DBA2PARAM(Value); break;
	case KOATag+KOAThreshold:
	  Value = DBA2PARAM(Value); break;
	case KOATag+KOADry:
	  Value = DB2PARAM(Value); break;
	case KWet:
	  Value = DB2PARAM(Value); break;
	case KDelay:
	  Value = DELAYTIME2PARAM(Value); break;
	case KWidth:
	  Value = Value/100; break;
	case KIRStretch:
	  Value = STRETCH2PARAM(Value); break;
	case KLRBalance:
	  Value = LRB2PARAM(Value); break;
	case KMuteWet:
	case KLPF:
	case KHPF:
	case KIRAttack:
	case KIRHold:
	case KIRDecay:
	case KIRSustain:
	case KIRRelease:
	default:
	  break;
	}
      // Value Range Check
      if(Value < 0) Value = 0; if(Value > 1) Value = 1;
    }
  // overAll parameters
  if(Tag >= KOATag)
    {
      int oaTag = Tag - KOATag;
      // Value Range Check
      if(Value < 0) Value = 0; if(Value > 1) Value = 1;
      switch(oaTag)
	{
	case KOASkipLimiter:
	case KOARelease:
	case KOACeiling:
	case KOAThreshold:
	case KOADry:
	  freeverb->setParameterAutomated(KNumParams*MODEL_SLOT_SIZE+oaTag, Value);
	  pControl->setDirty();
	default:
	  break;
	}
      return;
    }
  // slot parameters
  int slot = freeverb->getCurrentSlot();
  switch (Tag)
    {
    case KWet:
    case KMuteWet:
    case KLPF:
    case KHPF:
    case KM2S:
    case KSwap:
    case KDelay:
    case KWidth:
    case KLRBalance:
      freeverb->setParameterAutomated(KRTParam(slot, (VstInt32)Tag), Value);
      pControl->setDirty();
      break;
    case KIRStretch:
    case KIRAttack:
    case KIRHold:
    case KIRDecay:
    case KIRSustain:
    case KIRRelease:
      freeverb->setNRTParameter(KNRTParam(slot, (VstInt32)Tag), Value);
      setParameter(KNRTParam(slot,(VstInt32)Tag), Value);
      // reload slot if mouse click was released.
      if(strFader->isMouseDown() == false&&irattAKnob->isMouseDown() == false
	 &&irholAKnob->isMouseDown() == false&&irdecAKnob->isMouseDown() == false
	 &&irsusAKnob->isMouseDown() == false&&irrelAKnob->isMouseDown() == false) freeverb->pushReloadCue(slot);
      pControl->setDirty();
      break;
    case kKickTag:
      if(Value > 0.5f)
	{
	  if(this->fileSelector != NULL) break;
#if (__LP64__ & MAC)
	  this->fileSelector = CNewFileSelector::create(this->frame, CNewFileSelector::kSelectFile);
	  if (this->fileSelector)
	    {
	      this->fileSelector->setInitialDirectory(freeverb->getSlotFileNameA(freeverb->getCurrentSlot()));
	      this->fileSelector->setAllowMultiFileSelection(false);
	      // (const char *description, const char *extension, const char *mimeType=0, int macType=0)
	      this->fileSelector->addFileExtension(   CFileExtension("FLAC File", "flac", "audio/x-flac"));
	      this->fileSelector->addFileExtension(   CFileExtension("AIFF File", "aif",  "audio/aiff"));
	      this->fileSelector->addFileExtension(   CFileExtension("AIFC File", "aif",  "audio/x-aifc"));
	      this->fileSelector->addFileExtension(   CFileExtension("OggVorbis", "ogg",  "audio/x-ogg"));
	      this->fileSelector->setDefaultExtension(CFileExtension("Wave File", "wav",  "audio/wav"));
	      this->fileSelector->setTitle("Please select a impulse response file to load.");
	      this->fileSelector->run((VSTGUI::CBaseObject*)this);
	      this->fileSelector->forget();
	      this->fileSelector = NULL;
	    }
#else
	  this->fileSelector = new CFileSelector(NULL);
	  // VstFileType (char* _name, char *_macType, char *_dosType, char *_unixType = 0, char *_mimeType1 = 0, char *_mimeType2 = 0)
	  VstFileType waveType("Wave File", "WAVE", "wav", "wave", "audio/wav", "audio/x-wav");
	  VstFileType flacType("FLAC File", "FLAC", "flac", "flac", "audio/x-flac", "audio/x-flac");
	  VstFileType aiffType("AIFF File", "AIFF", "aif", "aiff", "audio/aiff", "audio/x-aiff");
	  VstFileType aifcType("AIFC File", "AIFC", "aif", "aifc", "audio/x-aifc", "audio/x-aifc");
	  VstFileType vorbisType("Vorbis File", "OGG", "ogg", "ogg", "audio/x-ogg", "audio/x-ogg");
	  VstFileType types[] = {waveType, aiffType, aifcType, flacType, vorbisType,};
	  VstFileSelect vstFileSelect;
	  memset(&vstFileSelect, 0, sizeof(VstFileSelect));
	  vstFileSelect.command     = kVstFileLoad;
	  vstFileSelect.type        = kVstFileType;
	  strcpy(vstFileSelect.title, "Please select a impulse response file to load.");
	  vstFileSelect.nbFileTypes = 5;
	  vstFileSelect.fileTypes   = (VstFileType*)&types;
	  vstFileSelect.returnPath  = this->returnPath1;
	  vstFileSelect.initialPath = this->initialPath1;
	  strncpy(this->initialPath1, freeverb->getSlotFileNameA(freeverb->getCurrentSlot()), _MAX_PATH*2);
	  vstFileSelect.future[0] = 1; // utf-8 path on macosx'
	  if(this->fileSelector->run(&vstFileSelect))
	    {
	      freeverb->pushLoadCueA(freeverb->getCurrentSlot(), vstFileSelect.returnPath);
	    }
	  delete this->fileSelector;
	  this->fileSelector = NULL;
#endif
	}
      break;
    case kResetTag:
      freeverb->setParameter(KRTParam(slot,KWet),DB2PARAM(-25.0f));
      freeverb->setParameter(KRTParam(slot,KMuteWet),0);
      freeverb->setParameter(KRTParam(slot,KLPF),0);
      freeverb->setParameter(KRTParam(slot,KHPF),0);
      freeverb->setParameter(KRTParam(slot,KWidth),1);
      freeverb->setParameter(KRTParam(slot,KM2S),0);
      freeverb->setParameter(KRTParam(slot,KSwap),0);
      freeverb->setParameter(KRTParam(slot,KDelay),0.5f);
      freeverb->setParameter(KRTParam(slot,KLRBalance),0.5f);
      freeverb->setNRTParameter(KNRTParam(slot,KIRStretch),0.5f);
      freeverb->setNRTParameter(KNRTParam(slot,KIRAttack), 0.0f);
      freeverb->setNRTParameter(KNRTParam(slot,KIRHold),   1.0f);
      freeverb->setNRTParameter(KNRTParam(slot,KIRDecay),  0.0f);
      freeverb->setNRTParameter(KNRTParam(slot,KIRSustain),0.8f);
      freeverb->setNRTParameter(KNRTParam(slot,KIRRelease),0.0f);
      // overall
      freeverb->setParameter(OAP+KOARelease,RELEASE2PARAM(100));
      freeverb->setParameter(OAP+KOACeiling,DBA2PARAM(0));
      freeverb->setParameter(OAP+KOAThreshold,DBA2PARAM(-2));
      freeverb->setParameter(OAP+KOADry,DB2PARAM(-5.0f));
      setViewSlot(slot);
      freeverb->pushReloadCue(slot);
      break;
    case kUnloadTag:
      setFileNameLabelA("");
      setSlotLabelA(slot, " ");
      freeverb->setSlotFileNameA(freeverb->getCurrentSlot(), "");
      freeverb->unloadSlot(freeverb->getCurrentSlot(), 0);
      break; 
    default:
      if(Tag >= KSlotHead&&Tag < KSlotHead + MODEL_SLOT_SIZE)
	{
	  int selectedSlot = Tag - KSlotHead;
	  cSlotSelector[freeverb->getCurrentSlot()]->setValue(0);
	  cSlotSelector[freeverb->getCurrentSlot()]->setDirty();
	  freeverb->setCurrentSlot(selectedSlot);
	  cSlotSelector[freeverb->getCurrentSlot()]->setValue(1);
	  cSlotSelector[freeverb->getCurrentSlot()]->setDirty();
	  setViewSlot(freeverb->getCurrentSlot());
	}
      break;
    }
}

#if (__LP64__ & MAC)
CMessageResult FreeverbEditor::notify (CBaseObject * sender, const char * message)
{
  if (message == CNewFileSelector::kSelectEndMessage)
    {
      CNewFileSelector * sel = dynamic_cast<CNewFileSelector*>(sender);
      if(sel)
	{
	  if(strlen(sel->getSelectedFile(0)) > 0)
	    {
	      freeverb->pushLoadCueA(freeverb->getCurrentSlot(), sel->getSelectedFile(0));
	    }
	  return kMessageNotified;
	}
    }
  return kMessageNotified;
}
#endif

void FreeverbEditor::setFileNameLabelA(const char * filename)
{
  if(frame == NULL) return;
  cLabel->setText(filename);
}

void FreeverbEditor::setViewSlot(int slot)
{
  if(!frame) return;
  // filenamelabel
  setSlotLabelA(slot, freeverb->getSlotFileNameA(slot));
  setFileNameLabelA(freeverb->getSlotFileNameA(slot));
  // fader
  wetFader->setValue(freeverb->getParameter(KRTParam(slot,KWet)));       wetFader->setDirty();
  lpfFader->setValue(freeverb->getParameter(KRTParam(slot,KLPF)));       lpfFader->setDirty();
  hpfFader->setValue(freeverb->getParameter(KRTParam(slot,KHPF)));       hpfFader->setDirty();
  idlFader->setValue(freeverb->getParameter(KRTParam(slot,KDelay)));     idlFader->setDirty();
  widFader->setValue(freeverb->getParameter(KRTParam(slot,KWidth)));     widFader->setDirty();
  lrbAKnob->setValue(freeverb->getParameter(KRTParam(slot,KLRBalance))); lrbAKnob->setDirty();
  // fader non-realtime
  strFader->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRStretch)));   strFader->setDirty();
  irattAKnob->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRAttack)));  irattAKnob->setDirty();
  irholAKnob->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRHold)));    irholAKnob->setDirty();
  irdecAKnob->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRDecay)));   irdecAKnob->setDirty();
  irsusAKnob->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRSustain))); irsusAKnob->setDirty();
  irrelAKnob->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRRelease))); irrelAKnob->setDirty();
  
  // display
  wetTextEdit->setValue(PARAM2DB(freeverb->getParameter(KRTParam(slot,KWet)))); wetTextEdit->setDirty();
  lpfTextEdit->setValue(freeverb->getParameter(KRTParam(slot,KLPF))); lpfTextEdit->setDirty();
  hpfTextEdit->setValue(freeverb->getParameter(KRTParam(slot,KHPF))); hpfTextEdit->setDirty();
  idlTextEdit->setValue(PARAM2DELAYTIME(freeverb->getParameter(KRTParam(slot,KDelay)))); idlTextEdit->setDirty();
  widTextEdit->setValue(100*freeverb->getParameter(KRTParam(slot,KWidth))); widTextEdit->setDirty();
  lrbTextEdit->setValue(PARAM2LRB(freeverb->getParameter(KRTParam(slot,KLRBalance)))); lrbTextEdit->setDirty();
  // display non-realtime
  strTextEdit->setValue(PARAM2STRETCH(freeverb->getNRTParameter(KNRTParam(slot,KIRStretch)))); strTextEdit->setDirty();
  irattTextEdit->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRAttack)));  irattTextEdit->setDirty();
  irholTextEdit->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRHold)));    irholTextEdit->setDirty();
  irdecTextEdit->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRDecay)));   irdecTextEdit->setDirty();
  irsusTextEdit->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRSustain))); irsusTextEdit->setDirty();
  irrelTextEdit->setValue(freeverb->getNRTParameter(KNRTParam(slot,KIRRelease))); irrelTextEdit->setDirty();

  // onoffbutton
  m2sButton->setValue(freeverb->getParameter(KRTParam(slot,KM2S))); m2sButton->setDirty();
  lrSwapButton->setValue(freeverb->getParameter(KRTParam(slot,KSwap))); lrSwapButton->setDirty();
  muteWetButton->setValue(freeverb->getParameter(KRTParam(slot,KMuteWet))); muteWetButton->setDirty();
  
  updateGraph();
}

void FreeverbEditor::setSlotLabelA(int slot, const char * filename)
{
  if(frame == NULL) return;
  if(filename == NULL||strlen(filename) == 0) return;
  if(filename[0] != ' ')
    {
#ifdef WIN32
      char szPath[_MAX_PATH*2], szDrive[_MAX_DRIVE], szDir[_MAX_DIR];
      char szFileName[_MAX_FNAME], szExt[_MAX_EXT];
      snprintf(szPath, _MAX_PATH*2, "%s", filename);
      _splitpath(szPath, szDrive, szDir, szFileName, szExt);
      cSlotLabel[slot]->setText(szFileName);
#endif
#ifdef MAC
      char path[PATH_MAX];
      memcpy(path, filename, PATH_MAX);
      cSlotLabel[slot]->setText(basename(path));      
#endif
    }
  else
    cSlotLabel[slot]->setText(filename);
}

void FreeverbEditor::setGUIZoomFactor(float factor)
{
  if(factor > 0) zoomFactor = factor;
}

float FreeverbEditor::limitFader(float value)
{
  if(value < 0) return 0;
  if(value > 1) return 1;
  return value;
}
