/**
 *  MBCompressor GUI
 *
 *  Copyright (C) 2006-2017 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "MBCompressorEditor.hpp"

MBCompressorEditor::MBCompressorEditor(AudioEffect *effect) :
  AEffGUIEditor(effect),
  compressor((MBCompressor *) effect)
{
  CBitmap backBitmap(kBackBMP);
  rect.left = 0; rect.top = 0;
  rect.right = backBitmap.getWidth(); rect.bottom = backBitmap.getHeight();
}

// GlobalFaderConfig

// parameterIndex, x, y,
static VstInt32 FaderSet[][3] = {
  {KCompLowT,  100, 236,}, {KCompMidT,  232, 236,}, {KCompHighT,  364, 236,},
  {KCompLowRa, 100, 257,}, {KCompMidRa, 232, 257,}, {KCompHighRa, 364, 257,},
  {KCompLowSK, 100, 278,}, {KCompMidSK, 232, 278,}, {KCompHighSK, 364, 278,},
  {KCompLowA,  100, 299,}, {KCompMidA,  232, 299,}, {KCompHighA,  364, 299,},
  {KCompLowR,  100, 320,}, {KCompMidR,  232, 320,}, {KCompHighR,  364, 320,},
  {KCompLowG,  100, 341,}, {KCompMidG,  232, 341,}, {KCompHighG,  364, 341,},
  {KCompRMS,   100, 362,}, {KCompLookahead, 100, 383,},
  // KAutoGain,
  {KLimitC, 502, 187,},
  {KLimitT, 502, 218,},
  {KLimitA, 502, 249,},
  {KLimitR, 502, 280,},
  {KLimitRMS, 502, 311,}, {KLimitLookahead, 502, 342,},
};
#define KFaderNum sizeof(FaderSet)/(sizeof(VstInt32)*3)
// GUI change trigger for textedit input
#define KTextEditPrefix 1000
CHorizontalSlider * HSliderSet[KFaderNum];
CTextEdit * TEditSet[KFaderNum];

void MBCompressorEditor::registerFader(CFrame * frame, VstInt32 data[3], CBitmap * body, CBitmap * handle,
				       float initFaderValue, float initTextValue, float zoomFactor, VstInt32 wide, int setIndex)
{
  if(frame == NULL||data == NULL||body == NULL||handle == NULL) return;
  // fader
  CPoint point(0,0);
  CRect size(0, 0, body->getWidth(), handle->getHeight());
  size.offset(data[1], data[2]);
  int fMinPos = data[1];
  int fMaxPos = data[1] + body->getWidth() - handle->getWidth();
  CHorizontalSlider * slider =
    new CHorizontalSlider(size, this, data[0], fMinPos, fMaxPos, handle, body, point);
  slider->setValue(param2gui(FaderSet[setIndex][0], initFaderValue));
  slider->setFreeClick(false);
  slider->setZoomFactor(zoomFactor);
  frame->addView(slider);
  HSliderSet[setIndex] = slider;
  // inputbox
  size(0, 0, wide-body->getWidth(), handle->getHeight());
  size.offset(data[1]+body->getWidth(), data[2]);
  CTextEdit * editor = new CTextEdit(size, this, KTextEditPrefix + data[0], "", NULL, kCenterText);
  editor->setStringConvert(MBCompressorEditor::stringConvert);
  editor->setFont(kNormalFontSmall);
  editor->setFontColor(kBlackCColor);
  editor->setBackColor(kWhiteCColor);
  editor->setFrameColor(kWhiteCColor);
  editor->setValue(initTextValue);
  frame->addView(editor);
  TEditSet[setIndex] = editor;
}

void MBCompressorEditor::stringConvert(float value, char * string)
{
  snprintf(string, 256, "%1.1f", value);
}

bool MBCompressorEditor::open(void *ptr)
{
  AEffGUIEditor::open(ptr);
  CRect size;

  // Background
  CBitmap *backBitmap = new CBitmap(kBackBMP);
  size(0, 0, backBitmap->getWidth(), backBitmap->getHeight());
  CFrame * frame = new CFrame(size, ptr, this);
  frame->setBackground(backBitmap);
  backBitmap->forget();
  
  // VuMeter
  CBitmap* vuOnBMP  = new CBitmap(kVuOnBMP);
  CBitmap* vuOffBMP = new CBitmap(kVuOffBMP);

  setVuMeter(0,0,0,0);
  // in
  size(0, 0, vuOnBMP->getWidth(), vuOnBMP->getHeight());
  size.offset(kpVuIX, kpVuIY);
  cVuMeterIL = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight());
  cVuMeterIL->setDecreaseStepValue (0.1f);
  cVuMeterIL->setValue(0.0f);
  frame->addView(cVuMeterIL);

  size.offset(vuOnBMP->getWidth(), 0);
  cVuMeterIR = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight());
  cVuMeterIR->setDecreaseStepValue (0.1f);
  cVuMeterIR->setValue(0.0f);
  frame->addView(cVuMeterIR);

  // out
  size(0, 0, vuOnBMP->getWidth(), vuOnBMP->getHeight());
  size.offset(kpVuOX, kpVuOY);
  cVuMeterOL = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight());
  cVuMeterOL->setDecreaseStepValue (0.1f);
  cVuMeterOL->setValue(0.0f);
  frame->addView(cVuMeterOL);

  size.offset(vuOnBMP->getWidth(), 0);
  cVuMeterOR = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight());
  cVuMeterOR->setDecreaseStepValue (0.1f);
  cVuMeterOR->setValue(0.0f);
  frame->addView(cVuMeterOR);

  // ReductionMeter
  vuOnBMP  = new CBitmap(kRedOnBMP);
  vuOffBMP = new CBitmap(kRedOffBMP);
  
  size(0, 0, vuOnBMP->getWidth(), vuOnBMP->getHeight());
  size.offset(kpRedX, kpRedY);
  cRMeter1
    = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight(), kHorizontal);
  cRMeter1->setDecreaseStepValue (0.1f);
  cRMeter1->setValue(0.0f);
  frame->addView(cRMeter1);

  size.offset(vuOnBMP->getWidth()+kpRedOffset, 0);
  cRMeter2
    = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight(), kHorizontal);
  cRMeter2->setDecreaseStepValue (0.1f);
  cRMeter2->setValue(0.0f);
  frame->addView(cRMeter2);
  size.offset(vuOnBMP->getWidth()+kpRedOffset, 0);

  cRMeter3
    = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight(), kHorizontal);
  cRMeter3->setDecreaseStepValue (0.1f);
  cRMeter3->setValue(0.0f);
  frame->addView(cRMeter3);

  // limiter
  size(0, 0, vuOnBMP->getWidth(), vuOnBMP->getHeight());
  size.offset(kpLRedX, kpLRedY);
  lRMeter
    = new CVuMeter(size, vuOnBMP, vuOffBMP, vuOnBMP->getHeight(), kHorizontal);
  lRMeter->setDecreaseStepValue (0.1f);
  lRMeter->setValue(0.0f);
  frame->addView(lRMeter);

  // ----------
  // FreqSlider
  // ----------
  CBitmap* hSBody   = new CBitmap(kFreqBarBMP);
  CBitmap* hSHandle = new CBitmap(kSliderTBMP);
  int fMinPos = kpFreqX;
  int fMaxPos = kpFreqX + hSBody->getWidth() - hSHandle->getWidth();
  CPoint point;

  point(0,0);
  size(0, 0, hSBody->getWidth(), hSHandle->getHeight());
  size.offset(kpFreqX, kpFreqY);
  cFreqHS1 = new CHorizontalSliderE(size, this, KFreqLowDivider, fMinPos, fMaxPos, hSHandle, hSBody, point);
  cFreqHS1->setValue(param2gui(KFreqLowDivider, effect->getParameter(KFreqLowDivider)));
  cFreqHS1->setFreeClick(false);
  //cFreqHS1->setZoomFactor(zoomFactor);
  frame->addView(cFreqHS1);
  
  size.offset(0, vuOnBMP->getHeight()+kpFreqOffset);
  cFreqHS2 = new CHorizontalSliderE(size, this, KFreqHighDivider, fMinPos, fMaxPos, hSHandle, hSBody, point);
  cFreqHS2->setValue(param2gui(KFreqHighDivider, effect->getParameter(KFreqHighDivider)));
  cFreqHS2->setFreeClick(false);
  //cFreqHS2->setZoomFactor(zoomFactor);
  frame->addView(cFreqHS2);

  float fc1 = compressor->getModelParameter(KFreqLowDivider);
  float fc2 = compressor->getModelParameter(KFreqHighDivider);
  if(fc1 > fc2)
    {
      fc2 = compressor->getModelParameter(KFreqLowDivider);
      fc1 = compressor->getModelParameter(KFreqHighDivider);      
    }
  
  // -----------
  // FreqDisplay
  // -----------
  size(0, 0, kpFreqTEWidth, kpFreqTEHeight);
  size.offset(kpFreqTE1X, kpFreqTE1Y);
  cFreqTE1 = new CTextEdit(size, this, KTextEditPrefix + KFreqLowDivider, "", NULL, kCenterText);
  cFreqTE1->setStringConvert(MBCompressorEditor::stringConvert);
  cFreqTE1->setFont(kNormalFontSmall);
  cFreqTE1->setFontColor(kBlackCColor);
  cFreqTE1->setBackColor(kWhiteCColor);
  cFreqTE1->setFrameColor(kWhiteCColor);
  cFreqTE1->setValue(fc1);
  frame->addView(cFreqTE1);
  
  size(0, 0, kpFreqTEWidth, kpFreqTEHeight);
  size.offset(kpFreqTE2X, kpFreqTE2Y);
  cFreqTE2 = new CTextEdit(size, this, KTextEditPrefix + KFreqHighDivider, "", NULL, kCenterText);
  cFreqTE2->setStringConvert(MBCompressorEditor::stringConvert);
  cFreqTE2->setFont(kNormalFontSmall);
  cFreqTE2->setFontColor(kBlackCColor);
  cFreqTE2->setBackColor(kWhiteCColor);
  cFreqTE2->setFrameColor(kWhiteCColor);
  cFreqTE2->setValue(fc2);
  frame->addView(cFreqTE2);
  
  // register Faders
  CBitmap* hFBody = new CBitmap(kHSliderBMP);
  for(unsigned int fi = 0;fi < KFaderNum;fi ++)
    {
      registerFader(frame, FaderSet[fi], hFBody, hSHandle, effect->getParameter(FaderSet[fi][0]), compressor->getModelParameter(FaderSet[fi][0]),
		    10, 128, fi);
    }

  // OnOff Button
  CBitmap *onOffBmp = new CBitmap(kOnOffBMP);
  size(0, 0, onOffBmp->getWidth(), onOffBmp->getHeight()/2);
  size.offset(kpAGX, kpAGY);
  autoGainB = new COnOffButton(size, this, KAutoGain, onOffBmp);
  autoGainB->setValue(effect->getParameter(KAutoGain));
  frame->addView(autoGainB);

  // graph
  size(0, 0, kpGraphWidth, kpGraphHeight);
  size.offset(kpGraphLX, kpGraphLY);
  graphL = new CCGraph(size, kBlueCColor);
  frame->addView(graphL);

  size(0, 0, kpGraphWidth, kpGraphHeight);
  size.offset(kpGraphMX, kpGraphMY);
  graphM = new CCGraph(size, kGreenCColor);
  frame->addView(graphM);

  size(0, 0, kpGraphWidth, kpGraphHeight);
  size.offset(kpGraphHX, kpGraphHY);
  graphH = new CCGraph(size, kYellowCColor);
  frame->addView(graphH);

  size(0, 0, kpGraphWidth, kpGraphHeight);
  size.offset(kpGraphTX, kpGraphTY);
  lgraph = new CLGraph(size, kRedCColor);
  frame->addView(lgraph);

  updateGraph();
  
  frame->setDirty();
  this->frame = frame;
  return true;
}

void MBCompressorEditor::close()
{
  if (frame != NULL) delete frame;
  frame = NULL;
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

float MBCompressorEditor::R2VudB(float value)
{
  float dB = UTILS::R2dB(value);
  if(dB <= -60) dB = -60;
  if(dB >= 0) dB = 0;
  return LINCV(-60,0,0,1,dB);
}

float MBCompressorEditor::gui2param(int index, pfloat_t value)
{
  switch(index)
    {
    case KCompLowRa:
    case KCompMidRa:
    case KCompHighRa:
    case KFreqLowDivider:
    case KFreqHighDivider:
      return 1-sqrt(sqrt(value));

    default:
      return 1-value;
    }
}

pfloat_t MBCompressorEditor::param2gui(int index, float value)
{
  switch(index)
    {
    case KCompLowRa:
    case KCompMidRa:
    case KCompHighRa:
    case KFreqLowDivider:
    case KFreqHighDivider:
      return limitFader(pow(1-value,4));
    default:
      return limitFader(1-value);
    }
}

void MBCompressorEditor::idle()
{
  if(!frame) return;
  if(cVuMeterIL) cVuMeterIL->setValue(R2VudB(vuIL));
  if(cVuMeterIR) cVuMeterIR->setValue(R2VudB(vuIR));
  if(cVuMeterOL) cVuMeterOL->setValue(R2VudB(vuOL));
  if(cVuMeterOR) cVuMeterOR->setValue(R2VudB(vuOR));
  if(cRMeter1) cRMeter1->setValue(re1);
  if(cRMeter1) cRMeter2->setValue(re2);
  if(cRMeter1) cRMeter3->setValue(re3);
  if(lRMeter) lRMeter->setValue(le);
  AEffGUIEditor::idle();
}

void MBCompressorEditor::updateGraph()
{
  graphL->updateValue(compressor->getModelParameter(KCompLowT),
		      compressor->getModelParameter(KCompLowRa),
		      compressor->getModelParameter(KCompLowSK),
		      compressor->getModelParameter(KCompLowG),
		      compressor->getModelParameter(KAutoGain));
  graphM->updateValue(compressor->getModelParameter(KCompMidT),
		      compressor->getModelParameter(KCompMidRa),
		      compressor->getModelParameter(KCompMidSK),
		      compressor->getModelParameter(KCompMidG),
		      compressor->getModelParameter(KAutoGain));
  graphH->updateValue(compressor->getModelParameter(KCompHighT),
		      compressor->getModelParameter(KCompHighRa),
		      compressor->getModelParameter(KCompHighSK),
		      compressor->getModelParameter(KCompHighG),
		      compressor->getModelParameter(KAutoGain));
  lgraph->updateValue(compressor->getModelParameter(KLimitC),
		      compressor->getModelParameter(KLimitT));
}

void MBCompressorEditor::setParameter(VstInt32 index, float value)
{
  if(!frame) return;
  updateGraph();
  // fader/input set
  for(unsigned int fi = 0;fi < KFaderNum;fi ++)
    {
      if(index == FaderSet[fi][0])
	{
	  HSliderSet[fi]->setValue(param2gui(index, value));
	  TEditSet[fi]->setValue(compressor->getModelParameter(index));
	}
    }
  float fc1, fc2;
  switch(index)
    {
    case KFreqLowDivider:
    case KFreqHighDivider:
      fc1 = compressor->getModelParameter(KFreqLowDivider);
      fc2 = compressor->getModelParameter(KFreqHighDivider);
      if(fc1 > fc2)
	{
	  fc2 = compressor->getModelParameter(KFreqLowDivider);
	  fc1 = compressor->getModelParameter(KFreqHighDivider);      
	}
      if(cFreqTE1)
	cFreqTE1->setValue(fc1);
      if(cFreqTE2)
	cFreqTE2->setValue(fc2);
      break;
    default:
      break;
    }
  switch(index)
    {
    case KFreqLowDivider:
      if(cFreqHS1)
	cFreqHS1->setValue(param2gui(index, value));
      break;
    case KFreqHighDivider:
      if(cFreqHS2)
	cFreqHS2->setValue(param2gui(index, value));
      break;
    case KAutoGain:
      if(autoGainB)
	autoGainB->setValue(value);
    default:
      break;
    }
}

// callback from GUI
void MBCompressorEditor::valueChanged(CControl* control)
{
  VstInt32 Tag = control->getTag();
  float Value = control->getValue();
  char str[16];
  // check textedit
  for(unsigned int fi = 0;fi < KFaderNum;fi ++)
    {
      if(Tag == FaderSet[fi][0])
	{
	  effect->setParameterAutomated(Tag, gui2param(Tag, Value));
	}
      if(Tag == FaderSet[fi][0]+KTextEditPrefix)
	{
	  Tag -= KTextEditPrefix;
	  ((CTextEdit*)control)->getText(str);
	  Value = compressor->model2param(Tag, atof(str));
	  effect->setParameterAutomated(Tag, Value);
	}
    }
  switch(Tag)
    {
    case KFreqLowDivider:
    case KFreqHighDivider:
      if(cFreqHS1->isMouseDown() == false&&cFreqHS2->isMouseDown() == false)
	{
	  effect->setParameterAutomated(Tag, gui2param(Tag, Value));
	}
      else
	// update GUI only
	{
	  setParameter(Tag, gui2param(Tag, Value));
	  float fc1, fc2;
	  fc1 = compressor->param2model(Tag, gui2param(Tag, cFreqHS1->getValue()));
	  fc2 = compressor->param2model(Tag, gui2param(Tag, cFreqHS2->getValue()));
	  if(fc1 > fc2)
	    {
	      fc2 = compressor->param2model(Tag, gui2param(Tag, cFreqHS1->getValue()));
	      fc1 = compressor->param2model(Tag, gui2param(Tag, cFreqHS2->getValue()));
	    }
	  if(cFreqTE1) cFreqTE1->setValue(fc1);
	  if(cFreqTE2) cFreqTE2->setValue(fc2);
	}
      break;
    case KFreqLowDivider+KTextEditPrefix:      
    case KFreqHighDivider+KTextEditPrefix:
      Tag -= KTextEditPrefix;
      ((CTextEdit*)control)->getText(str);
      Value = compressor->model2param(Tag, atof(str));
      effect->setParameterAutomated(Tag, Value);
      break;
    case KAutoGain:
      effect->setParameterAutomated(Tag, Value);
    default:
      break;
    }
  control->setDirty();
}

void MBCompressorEditor::setVuMeter(float iL, float iR, float oL, float oR)
{
  vuIL = iL; vuIR = iR; vuOL = oL; vuOR = oR;
}


void MBCompressorEditor::setRMeter(float r1, float r2, float r3, float l)
{
  re1 = r1, re2 = r2, re3 = r3, le = l;
}

float MBCompressorEditor::limitFader(float value)
{
  if(value < 0) return 0;
  if(value > 1) return 1;
  return value;
}
