/**
 *  Impulser2 DLL
 *
 *  Copyright (C) 2006-2017 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "CheckCPU.hpp"
#include "Freeverb.hpp"

#if WIN32
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif


#define FV3_CONF_XML_W    L"freeverb3cfg.xml"
#define FV3_CONF_XML       "freeverb3cfg.xml"
#define FV3_IR2_IRP_XML_W L"freeverb3prg.xml"
#define FV3_IR2_IRP_XML    "freeverb3prg.xml"
#define FV3_IR2_LOG_W     L"freeverb3ir2.log"
#define FV3_IR2_LOG        "freeverb3ir2.log"
#define FV3_IR2_CONF_VERSION 103
#define FV3_IR2_CONF_KEY_VERSION         "/Freeverb3VST_Impulser2/Version"
#define FV3_IR2_CONF_KEY_FRAGMENTSIZE    "/Freeverb3VST_Impulser2/FragmentSize"
#define FV3_IR2_CONF_KEY_ZEROLATENCY     "/Freeverb3VST_Impulser2/ZeroLatency"
#define FV3_IR2_CONF_KEY_OUTPUT_LOG      "/Freeverb3VST_Impulser2/OutputLog"
#define FV3_IR2_CONF_KEY_CONVERTER_TYPE  "/Freeverb3VST_Impulser2/ConverterType"
#define FV3_IR2_CONF_KEY_FACTOR          "/Freeverb3VST_Impulser2/Factor"
#define FV3_IR2_CONF_KEY_MT              "/Freeverb3VST_Impulser2/MultiThread"
#define FV3_IR2_CONF_KEY_GUI_ZOOM_FACTOR "/Freeverb3VST_Impulser2/GuiZoomFactor"
#define FV3_IR2_CONF_KEY_DISABLE_MIDI    "/Freeverb3VST_Impulser2/DisableMidiEvents"

#ifdef WIN32
wchar_t ModuleFileNameW[MAX_PATH];
#else
#define MAX_PATH PATH_MAX
#endif
char XMLConfigPath[MAX_PATH], XMLProgramPath[MAX_PATH], LogFilePath[MAX_PATH];

void initXMLConfigPath()
{
  XMLConfigPath[0] = '\0';
#ifdef WIN32
  wchar_t szDrive[_MAX_DRIVE], szDir[_MAX_DIR], szFileName[_MAX_FNAME], szExt[_MAX_EXT], szConfigFile[MAX_PATH];
  _wsplitpath(ModuleFileNameW, szDrive, szDir, szFileName, szExt);
  wsprintfW(szConfigFile, L"%s%s"FV3_CONF_XML_W, szDrive, szDir);
  WideCharToMultiByte(CP_ACP, WC_NO_BEST_FIT_CHARS, szConfigFile, -1, XMLConfigPath, MAX_PATH, NULL, NULL);
  wsprintfW(szConfigFile, L"%s%s"FV3_IR2_IRP_XML_W, szDrive, szDir);
  WideCharToMultiByte(CP_ACP, WC_NO_BEST_FIT_CHARS, szConfigFile, -1, XMLProgramPath, MAX_PATH, NULL, NULL);
  wsprintfW(szConfigFile, L"%s%s"FV3_IR2_LOG_W, szDrive, szDir);
  WideCharToMultiByte(CP_ACP, WC_NO_BEST_FIT_CHARS, szConfigFile, -1, LogFilePath, MAX_PATH, NULL, NULL);
#endif
#ifdef MAC
  char hDIR[MAX_PATH];
  getHomeDirectoryName(hDIR);
  snprintf(XMLConfigPath, MAX_PATH, "%s/Library/Audio/Plug-Ins/VST/%s", hDIR, FV3_CONF_XML);
  snprintf(XMLProgramPath, MAX_PATH, "%s/Library/Audio/Plug-Ins/VST/%s", hDIR, FV3_IR2_IRP_XML);
  snprintf(LogFilePath, MAX_PATH, "%s/Library/Audio/Plug-Ins/VST/%s", hDIR, FV3_IR2_LOG);
#endif
}

void checkHostCanDo(Freeverb * plug, const char * canDo)
{
  plug->writeLogA("canHostDo %s: %d\n", canDo, plug->canHostDo((char*)canDo));
}

#if WIN32
extern "C" __declspec(dllexport) AEffect *VSTPluginMain(audioMasterCallback audioMaster);
#elif MAC
extern "C" AEffect *VSTPluginMain(audioMasterCallback audioMaster);
extern "C" AEffect *main_macho(audioMasterCallback audioMaster){return VSTPluginMain(audioMaster);}
extern "C" AEffect *main_plugin(audioMasterCallback audioMaster){return VSTPluginMain(audioMaster);}
#endif

extern "C" AEffect *VSTPluginMain(audioMasterCallback audioMaster)
{
  initXMLConfigPath();
  XMLConfig xmlConfig;
  xmlConfig.registerXML(XMLConfigPath);
  if(!audioMaster(NULL, audioMasterVersion, 0, 0, NULL, 0)) return NULL;
  if(detectCPU() == 0) return NULL;
  Freeverb *plug;
  try
    {
      plug = new Freeverb(audioMaster);
      if(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_VERSION) < FV3_IR2_CONF_VERSION)
	{
	  MacMessageBox(XMLConfigPath, "Freeverb3 Impulser2: You should update the configuration XML file, skipping...");
	  plug->setLogFileName("");
	  plug->setFragmentSize(512);
	  plug->setFactor(16);
	  plug->setZeroLatency(1);
	  plug->setMT(1);
	  plug->setConverterType(0);
	  plug->setGUIZoomFactor(20);
	  plug->setIRProgsFileName("");
	}
      else
	{
	  if(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_OUTPUT_LOG) > 0) plug->setLogFileName(LogFilePath);
	  if(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_DISABLE_MIDI) > 0) plug->disableMidiEvents();
	  plug->writeLogA("===========================\n");
	  plug->writeLogA("Freeverb3_vst Impulser2 Log\n");
	  plug->writeLogA("Version: %s\n", VERSION);
	  plug->writeLogA("===========================\n");
	  plug->writeLogA("SIMDFlag: %08x MXCSR: %08x\n", UTILS::getSIMDFlag(), UTILS::getMXCSR());
	  plug->setFragmentSize(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_FRAGMENTSIZE));
	  plug->setFactor(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_FACTOR));
	  plug->setZeroLatency(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_ZEROLATENCY));
	  plug->setMT(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_MT));
	  plug->setConverterType(xmlConfig.getKeyAsInt(FV3_IR2_CONF_KEY_CONVERTER_TYPE));
	  plug->setGUIZoomFactor(xmlConfig.getKeyAsFloat(FV3_IR2_CONF_KEY_GUI_ZOOM_FACTOR));
	  plug->setIRProgsFileName(XMLProgramPath);
	  plug->writeLogA("IRPrograms file = %s\n", XMLProgramPath);
	  if(plug->registerIRPrograms() == 0) plug->writeLogA("IRPrograms were loaded.\n");
	  else plug->writeLogA("IRPrograms were not loaded.\n");
	}
    }
  catch(std::bad_alloc)
    {
      return NULL;
    }
  plug->writeLogA("Main.cpp: VSTPluginMain: VstVersion=%d\n", plug->getVstVersion());
  char hostVendorString[kVstMaxVendorStrLen];
  char hostProductString[kVstMaxProductStrLen];
  if(plug->getHostVendorString(hostVendorString)) plug->writeLogA("Host Vendor: %s (%d)\n", hostVendorString, plug->getVendorVersion());
  if(plug->getHostProductString(hostProductString)) plug->writeLogA("Host Product: %s\n", hostProductString);
  plug->writeLogA("Host Language: %d\n", plug->getHostLanguage());
  checkHostCanDo(plug, "sendVstEvents");
  checkHostCanDo(plug, "sendVstMidiEvent");
  checkHostCanDo(plug, "sendVstTimeInfo");
  checkHostCanDo(plug, "receiveVstEvents");
  checkHostCanDo(plug, "receiveVstMidiEvent");
  checkHostCanDo(plug, "reportConnectionChanges");
  checkHostCanDo(plug, "acceptIOChanges");
  return plug->getAeffect();
}

#if WIN32
void *hInstance;
extern "C" BOOL WINAPI
DllMain(HINSTANCE hInst, DWORD dwReason, LPVOID lpvReserved)
{
  DWORD dwRet;      
  hInstance = hInst;
  switch(dwReason)
    {
    case DLL_PROCESS_ATTACH:
      dwRet = GetModuleFileNameW((HMODULE)hInstance, ModuleFileNameW, MAX_PATH);
      if(dwRet == 0) return FALSE;
      break;
    case DLL_PROCESS_DETACH:
      break;
    case DLL_THREAD_ATTACH:
      break;
    case DLL_THREAD_DETACH:
      break;
    }
  return TRUE;
}
#endif
