/**
 *  NRev
 *
 *  Copyright (C) 2006-2017 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "NReverb.hpp"
#include "GeneralEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "SRCFact", "Wet", "Dry", "FeedBack", "Damping", "OutLPF", "InHPF", "RT60", "Width", "PreDelay",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "X", "dB", "dB", "%", "%", "%", "%", "s", "%", "ms",
};

#define KNumPrograms 24
#define KPresetPrograms 13

static char presetProgramName[KPresetPrograms][kVstMaxProgNameLen] = {
  "Default Reverb",

  "Hall (Light) 1X",
  "Hall (Light) 3X",
  "Hall (Dark) 1X",
  "Hall (Dark) 3X",

  "Room (Light) 1X",
  "Room (Light) 3X",
  "Room (Dark) 1X",
  "Room (Dark) 3X",

  "Stadium 1X",
  "Stadium 3X",

  "Drum Chamber 1X",
  "Drum Chamber 3X",
};
static float presetProgram[KPresetPrograms][KNumParams] = {
  // Factor, Wet, Dry, Feedback, Damp1, Damp2, Damp3, ReverbTime, Width, Delay,
  {3.6, -20.0,  0.0,  0.6,  0.71, 0.85, 0.10, 5.05, 1.0, 15,},

  {1.6, -10.0, -2.0,  0.65, 0.3,  0.7,  0.43, 2.05, 0.9, 10,},
  {3.6, -10.0, -2.0,  0.65, 0.3,  0.7,  0.43, 2.05, 0.9, 10,},
  {1.6, -15.0, -2.0,  0.6,  0.85, 0.85, 0.10, 3.05, 1.0, 15,},
  {3.6, -15.0, -2.0,  0.6,  0.85, 0.85, 0.10, 3.05, 1.0, 15,},

  {1.6, -10.0, -2.0,  0.7,  0.4,  0.2,  0.25, 1.0,  0.9, 2,},
  {3.6, -10.0, -2.0,  0.7,  0.4,  0.2,  0.25, 1.0,  0.9, 2,},
  {1.6, -10.0, -2.0,  0.6,  0.9,  0.9,  0.10, 1.0,  1.0, 0,},
  {3.6, -10.0, -2.0,  0.6,  0.9,  0.9,  0.10, 1.0,  1.0, 0,},

  {1.6, -15.0, -2.0,  0.6,  0.21, 0.6,  0.05, 2.98, 1.0, 20,},
  {3.6, -15.0, -2.0,  0.6,  0.21, 0.6,  0.05, 2.98, 1.0, 20,},

  {1.6, -10.0, -2.0,  0.56, 0.9,  0.9,  0.2,  1.07, 1.0,  2,},
  {3.6, -10.0, -2.0,  0.56, 0.9,  0.9,  0.2,  1.07, 1.0,  2,},
};

static const char * grpName[6] = { "LEV", "RVB", "FLT", "", "", "", };
static const VstInt32 grpMtx[6][6] = {
  {KDry, KWet, -1, -1, -1, -1},
  {KOFactor, KWidth, KFeedback, KDelay, KReverbTime, -1,},
  {KDamp3, KDamp1, KDamp2, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
};
static const char * nameMtx[6][6] = {
  {"DRY", "WET", "", "", "", "",},
  {"FAC", "WID", "FBA", "IDEL", "RT60", "",},
  {"IHPF", "DAMP", "OLPF", "", "", "",},
  {"", "", "", "", "", "",},
  {"", "", "", "", "", "",},
  {"", "", "", "", "", "",},
};

static float ParamConverter(int index, float value){ return NRev::model2param(index, value); }

NRev::NRev(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'N', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif 
  converter_type = FV3_SRC_LPF_IIR_2;
  nrev_ = new NREV();
  nrev_->setMuteOnChange(true);
  nrev_->setSampleRate(FV3_REVBASE_DEFAULT_FS);
  nrevb_ = new NREVB();
  nrevb_->setMuteOnChange(true);
  nrevb_->setSampleRate(FV3_REVBASE_DEFAULT_FS);
  NREVB * nrevb_dcast = dynamic_cast<NREVB*>(nrevb_);
  if(nrevb_dcast != NULL) nrevb_dcast->setapfeedback(0.3);
  model = nrev_;
  programs = new NRevProgram[numPrograms];
  for(int pc = 0;pc < KPresetPrograms;pc ++) programs[pc].setProgram(presetProgramName[pc], presetProgram[pc]);
  setProgram(0);
  GeneralEditor * _editor = new GeneralEditor(this);
  _editor->registerConst(EFFECT_NAME, grpName, grpMtx, nameMtx);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;
}

NRev::~NRev()
{
  freeProcessBlock();
  delete[] programs;
  delete nrev_;
  delete nrevb_;
}

bool NRev::getEffectName (char* name)
{
  strcpy (name, EFFECT_NAME);
  return true;
}

bool NRev::getVendorString (char* text)
{
  strcpy (text, VENDOR_STRING);
  return true;
}

bool NRev::getProductString (char* text)
{
  strcpy (text, "Freeverb3");
  return true;
}

VstPlugCategory NRev::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 NRev::canDo (char* text)
{
  if (!strcmp (text, "1in1out")) return 1;
  if (!strcmp (text, "2in2out")) return 1;
  if (!strcmp (text, "1in2out")) return 1;
  return -1;
}

bool NRev::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool NRev::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

VstInt32 NRev::getProgram()
{
  return curProgram;
}

bool NRev::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

void NRev::setProgram(VstInt32 program)
{
  if(program < numPrograms)
    {
      NRevProgram * p = &programs[program];
      curProgram = program;
      setParameterM(KOFactor, p->fOFactor);
      setParameterM(KWet, p->fWet);
      setParameterM(KDry, p->fDry);
      setParameterM(KFeedback, p->fFeedback);
      setParameterM(KDamp1, p->fDamp1);
      setParameterM(KDamp2, p->fDamp2);
      setParameterM(KDamp3, p->fDamp3);
      setParameterM(KReverbTime, p->fReverbTime);
      setParameterM(KWidth, p->fWidth);
      setParameterM(KDelay, p->fDelay);
      lock();
      model->mute();
      ProcessBlock::mute();
      unlock();
    }
}

void NRev::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

bool NRev::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void NRev::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void NRev::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void NRev::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void NRev::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  NRevProgram * p = &programs[curProgram];
  long factorp = 1;
  switch (index)
    {
    case KOFactor:
      p->fOFactor = value;
      factorp = std::ceil(value); if(factorp == 0) factorp = 1;
      if((value-std::floor(value)) > 0.5||value == 5)
	{
	  if(typeid(*model) != typeid(NREVB))
	    {
	      lock();
	      model = nrevb_;
	      model->setSampleRate(getSampleRate());
	      unlock();
	      setProgram(curProgram);
	    }
	}
      else
	{
	  if(typeid(*model) != typeid(NREV))
	    {
	      lock();
	      model = nrev_;
	      model->setSampleRate(getSampleRate());
	      unlock();
	      setProgram(curProgram);
	    }
	}
      if(factorp != model->getOSFactor())
	{
	  lock();
	  model->setOSFactor(factorp, converter_type);
	  unlock();
	  setProgram(curProgram);
	}
      break;
    case KWet:
      model->setwet(p->fWet = value);
      break;
    case KDry:
      model->setdry(p->fDry = value);
      break;
    case KFeedback:
      model->setfeedback(p->fFeedback = value);
      break;
    case KDamp1:
      model->setdamp(p->fDamp1 = value);
      break;
    case KDamp2:
      model->setdamp2(p->fDamp2 = value);
      break;
    case KDamp3:
      model->setdamp3(p->fDamp3 = value);
      break;
    case KReverbTime:
      model->setrt60(p->fReverbTime = value);
      break;
    case KWidth:
      model->setwidth(p->fWidth = value);
      break;
    case KDelay:
      lock();
      model->setPreDelay(p->fDelay = value);
      unlock();
      break;
    default:
      break;
    }
}

float NRev::getParameter(VstInt32 index)
{
  float ret = 0.0f;
  NRevProgram * p = &programs[curProgram];
  
  switch (index)
    {
    case KOFactor:
      ret = p->fOFactor;
      break;
    case KWet:
      ret = p->fWet;
      break;
    case KDry:
      ret = p->fDry;
      break;
    case KFeedback:
      ret = p->fFeedback;
      break;
    case KDamp1:
      ret = p->fDamp1;
      break;
    case KDamp2:
      ret = p->fDamp2;
      break;
    case KDamp3:
      ret = p->fDamp3;
      break;
    case KReverbTime:
      ret = p->fReverbTime;
      break;
    case KWidth:
      ret = p->fWidth;
      break;
    case KDelay:
      ret = p->fDelay;
      break;
    default:
      break;
    }
  return model2param(index, ret);
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t NRev::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KOFactor:
      if(p2m) return LINCV(0,1,0,5,value);
      else return LINCV(0,5,0,1,value);
    case KWet:
    case KDry:
      if(p2m) return LINCV(0,1,-100,20,value);
      else return LINCV(-100,20,0,1,value);
    case KReverbTime:
      if(p2m) return LINCV(0,1,0,16,value);
      else return LINCV(0,16,0,1,value);
    case KDelay:
      if(p2m) return LINCV(0,1,-500,500,value);
      else return LINCV(-500,500,0,1,value);
    case KFeedback:
    case KDamp1:
    case KDamp2:
    case KDamp3:
    case KWidth:
    default:
      return value;
    }
}

pfloat_t NRev::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float NRev::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void NRev::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void NRev::getParameterDisplay(VstInt32 index, char *text)
{
  NRevProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOFactor:
      if(typeid(*model) == typeid(NREVB))
	snprintf(text, kVstMaxParamStrLen, "%ld-B", model->getOSFactor());
      else
	snprintf(text, kVstMaxParamStrLen, "%ld-A", model->getOSFactor());
      break;
    case KWet:
      float2string(p->fWet, text, kVstMaxParamStrLen);
      break;
    case KDry:
      float2string(p->fDry, text, kVstMaxParamStrLen);
      break;
    case KFeedback:
      float2string(p->fFeedback*100, text, kVstMaxParamStrLen);
      break;
    case KDamp1:
      float2string(p->fDamp1*100, text, kVstMaxParamStrLen);
      break;
    case KDamp2:
      float2string(p->fDamp2*100, text, kVstMaxParamStrLen);
      break;
    case KDamp3:
      float2string(p->fDamp3*100, text, kVstMaxParamStrLen);
      break;
    case KReverbTime:
      float2string(p->fReverbTime, text, kVstMaxParamStrLen);
      break;
    case KWidth:
      float2string(p->fWidth*100, text, kVstMaxParamStrLen);
      break;
    case KDelay:
      float2string(p->fDelay, text, kVstMaxParamStrLen);
      break;
    default:
      break;
    }
}

void NRev::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float NRev::getSampleRate()
{
  return currentFs;
}

void NRev::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||model->getSampleRate() != sampleRate)
    {
      currentFs = (double)sampleRate;
      lock();
      model->setSampleRate(sampleRate);
      unlock();
    }
}

void NRev::suspend()
{
  ;
}

void NRev::resume()
{
  lock();
  if(updateBlockSize() > 0) allocProcessBlock(updateBlockSize());
  model->mute();
  ProcessBlock::mute();
  unlock();
  setInitialDelay(model->getLatency()+ProcessBlock::getLatency());
}

void NRev::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void NRev::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void NRev::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void NRev::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  if(tryLock() == true)
    {
      model->processreplace(inL,inR,outL,outR,(long int)sampleFrames);
      unlock();
    }
  else
    {
      memcpy(outL, inL, sizeof(pfloat_t)*sampleFrames);
      memcpy(outR, inR, sizeof(pfloat_t)*sampleFrames);
    }
}

void NRev::setConverterType(int type)
{
  converter_type = type;
}

void NRev::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
