/**
 *  VSTGUI Extension
 *
 *  Copyright (C) 2004 Steinberg Media Technologies
 *  Copyright (C) 2006-2017 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _VSTGUIEV_HPP
#define _VSTGUIEV_HPP

#include <cmath>
#include <vstgui.h>
#include <vstcontrols.h>

#define LINCV(imin,imax,omin,omax,val) \
((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

class CVerticalSliderE : public CVerticalSlider
{
public:
  CVerticalSliderE(const CRect& size, CControlListener* listener, long tag,
		   long iMinPos, long iMaxPos, CBitmap* handle, CBitmap* background,
		   const CPoint& offset = CPoint (0, 0), const long style = kBottom)
    : CVerticalSlider(size, listener, tag, iMinPos, iMaxPos, handle, background, offset, style)
  {isMouseD = false;}
  virtual ~CVerticalSliderE(){}
  virtual CMouseEventResult onMouseDown(CPoint& where, const long& buttons)
  {isMouseD = true; return CVerticalSlider::onMouseDown(where, buttons);}
  virtual CMouseEventResult onMouseUp(CPoint& where, const long& buttons)
  {isMouseD = false; if(listener) listener->valueChanged (this);
    return CVerticalSlider::onMouseUp(where, buttons);}
  virtual CMouseEventResult onMouseMoved (CPoint& where, const long& buttons)
  {return CVerticalSlider::onMouseMoved(where, buttons);}
  bool isMouseDown(){return isMouseD;}
protected:
  bool isMouseD;
};

class CHorizontalSliderE : public CHorizontalSlider
{
public:
  CHorizontalSliderE(const CRect& size, CControlListener* listener, long tag,
		   long iMinPos, long iMaxPos, CBitmap* handle, CBitmap* background,
		   const CPoint& offset = CPoint (0, 0), const long style = kBottom)
    : CHorizontalSlider(size, listener, tag, iMinPos, iMaxPos, handle, background, offset, style)
  {isMouseD = false;}
  virtual ~CHorizontalSliderE(){}
  virtual CMouseEventResult onMouseDown(CPoint& where, const long& buttons)
  {isMouseD = true; return CHorizontalSlider::onMouseDown(where, buttons);}
  virtual CMouseEventResult onMouseUp(CPoint& where, const long& buttons)
  {isMouseD = false; if(listener) listener->valueChanged (this);
    return CHorizontalSlider::onMouseUp(where, buttons);}
  virtual CMouseEventResult onMouseMoved (CPoint& where, const long& buttons)
  {return CHorizontalSlider::onMouseMoved(where, buttons);}
  bool isMouseDown(){return isMouseD;}
protected:
  bool isMouseD;
};

class CAnimKnobE : public CAnimKnob
{
public:
  CAnimKnobE(const CRect& size, CControlListener* listener, long tag, long subPixmaps,
	     CCoord heightOfOneImage, CBitmap* background, const CPoint& offset = CPoint (0, 0))
    : CAnimKnob(size, listener, tag, subPixmaps, heightOfOneImage, background, offset)
  {isMouseD = false;}
  virtual ~CAnimKnobE(){}
  virtual CMouseEventResult onMouseDown(CPoint& where, const long& buttons)
  {isMouseD = true; return CAnimKnob::onMouseDown(where, buttons);}
  virtual CMouseEventResult onMouseUp(CPoint& where, const long& buttons)
  {isMouseD = false; if(listener) listener->valueChanged (this);
    return CAnimKnob::onMouseUp(where, buttons);}
  virtual CMouseEventResult onMouseMoved (CPoint& where, const long& buttons)
  {return CAnimKnob::onMouseMoved(where, buttons);}
  bool isMouseDown(){return isMouseD;}
protected:
  bool isMouseD;
};

class CTextLabelB : public CTextLabel
{
public:
  CTextLabelB(const CRect& size, const char* txt = 0, CBitmap* background = 0, const long style = 0, CControlListener* listener = 0, long tag = 0)
    : CTextLabel(size, txt, background, style)
  { CControl::listener = listener, CControl::tag = tag; }
  virtual ~CTextLabelB(){}
  virtual CMouseEventResult onMouseDown(CPoint& where, const long& buttons)
  {if(listener) listener->valueChanged (this); return kMouseEventHandled;}
};

// Example:
// size(0, 0, 220, 220);
// size.offset(0, 0);
// nMeter = new CNeedleMeter(size, kBlueCColor, CPoint(110, 200), 45, 135, 20, 180);
// frame->addView(nMeter);

class CNeedleMeter : public CView
{
public:
  CNeedleMeter(const CRect &size, const CColor &line1, const CColor &line2, const CPoint &center,
	       const float &arc1, const float &arc2, const float &len1, const float &len2, const float &len3) :
  CView(size)
  {
    setTransparency(true);
    setMouseEnabled(false);
    offset(size.left, size.top);
    needleCenter = center;
    narc1 = arc1, narc2 = arc2;
    nlen1 = len1, nlen2 = len2, nlen3 = len3;
    linecolor1 = line1; linecolor2 = line2;
    current1 = 0; current2 = 0;
    secondNeedle = true;
  }

  virtual ~CNeedleMeter(){};

  virtual void draw(CDrawContext *pContext)
  {
    pContext->setLineWidth(1);
    float currentDegree = 0;
    if(secondNeedle)
      {
	if(current2 < 0) current2 = 0; if(current2 > 1) current2 = 1;
	currentDegree = narc1 + (narc2 - narc1) * current2;
	pContext->setFrameColor(linecolor2);
	pContext->moveTo(CPoint(offset.x + needleCenter.x - nlen1*std::cos(M_PI*currentDegree/180.),
				offset.y + needleCenter.y - nlen1*std::sin(M_PI*currentDegree/180.)));
	pContext->lineTo(CPoint(offset.x + needleCenter.x - nlen3*std::cos(M_PI*currentDegree/180.),
				offset.y + needleCenter.y - nlen3*std::sin(M_PI*currentDegree/180.)));
      }
    if(current1 < 0) current1 = 0; if(current1 > 1) current1 = 1;
    currentDegree = narc1 + (narc2 - narc1) * current1;
    pContext->setFrameColor(linecolor1);
    pContext->moveTo(CPoint(offset.x + needleCenter.x - nlen1*std::cos(M_PI*currentDegree/180.),
			    offset.y + needleCenter.y - nlen1*std::sin(M_PI*currentDegree/180.)));
    pContext->lineTo(CPoint(offset.x + needleCenter.x - nlen2*std::cos(M_PI*currentDegree/180.),
			    offset.y + needleCenter.y - nlen2*std::sin(M_PI*currentDegree/180.)));
    setDirty(false);
  }

  virtual void doIdleStuff() { if(pParentFrame) pParentFrame->doIdleStuff(); }
  virtual bool isDirty() const { if(dirty||CView::isDirty()) return true; else return false; }
  virtual void setDirty(const bool val = true){ dirty = val; }  
  void updateValue(float t1)           { current1 = t1; current2 = 0; setDirty(); }
  void updateValue(float t1, float t2) { current1 = t1; current2 = t2; setDirty(); }
  void setSecondNeedle(bool t) { secondNeedle = t; }

private:
  CNeedleMeter(const CNeedleMeter&);
  bool secondNeedle, dirty;
  float current1, current2, narc1, narc2, nlen1, nlen2, nlen3;
  CPoint offset, needleCenter;
  CColor linecolor1, linecolor2;
};

#endif
