/**
 *  Progenitor Reverb
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ProG_Reverb.hpp"
#include "GeneralEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "SRCFact", "Dry", "ERefWet", "ERefWid", "ERefFac", "ERefSend", "Wet", "Width", "PreDelay", "RT60",
  "InLPF", "Damping", "BassLPF", "BassFact", "OutLPF", "OutLPBW", "Spin", "Wander",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "x", "dB", "dB", "-", "x", "-", "dB", "-", "ms", "s",
  "Hz", "Hz", "Hz", "-", "Hz", "-", "Hz", "-",
};

#define KNumPrograms 32
#define KPresetPrograms 19

static char presetProgramName[KPresetPrograms][kVstMaxProgNameLen] = {
  "Default Reverb",

  "Smal Hall (Light)",
  "Small Hall (Dark)",
  "Medium Hall (Light)",
  "Medium Hall (Dark)",
  "Large Hall (Light)",
  "Large Hall (Dark)",

  "Small Room (Tight)",
  "Small Room (Bright)",
  "Medium Room (Tight)",
  "Medium Room (Bright)",
  "Large Room (Tight)",
  "Large Room (Bright)",
  "Medium Room ER (Tight)",
  "Medium Room ER (Bright)",

  "Plate High (Late Only)",
  "Plate Low (Late Only)",

  "Long Reverb (12s)",
  "Long Reverb (30s)",
};

static float presetProgram[KPresetPrograms][KNumParams] = {
  // Factor, Dry, ERefWet, ERefWidth, ERefFactor, ERefSend, Wet, Width, Delay, RT60,
  // InputLPF, DampLPF, BassLPF, BassFactor, OutputLPF, OutputBW, Spin, Wander
  { 0.1, -10,  -9.0,  0.7, 1.6, 0.4, -0, 1.0, 20,  3.2, 17000,  7000,  500, 0.15, 10000, 1, 0.7, 0.35, },

  { 1.1,  -8,  -9.0,  0.7, 1.0, 0.3, -8, 1.0, 10,  2.1, 18000,  9000,  600, 0.25, 17000, 1, 0.7, 0.45, },
  { 0.1,  -8,  -9.0,  0.7, 1.0, 0.3, -8, 1.0, 10,  2.3, 18000,  7000,  600, 0.2,   9000, 1, 0.5, 0.35, },
  { 1.1,  -8,  -9.0,  0.7, 1.2, 0.3, -8, 1.0, 10,  2.8, 18000,  8000,  500, 0.2,  16000, 1, 0.7, 0.35, },
  { 0.1,  -8,  -9.0,  0.7, 1.2, 0.3, -8, 1.0, 10,  2.9, 18000,  6000,  500, 0.15,  8000, 1, 0.5, 0.30, },
  { 1.1,  -8,  -9.0,  0.7, 1.4, 0.2, -8, 1.0, 18,  3.8, 18000,  9000,  400, 0.2,  14000, 1, 1.0, 0.15, },
  { 1.1,  -8,  -9.0,  0.7, 1.5, 0.2, -8, 1.0, 18,  4.2, 18000,  5000,  400, 0.2,   7000, 1, 0.5, 0.20, },

  { 1.1,  -8,  -8.0, -0.4, 0.7, 0.7, -8, 0.8,  5,  0.5, 18000, 18000, 1000, 0.3,  18000, 1, 1.6, 0.20, },
  { 2.1,  -8,  -8.0,  0.6, 0.8, 0.7, -8, 0.9,  5,  0.5, 18000, 10000,  300, 0.3,  18000, 1, 0.4, 0.40, },
  { 1.1,  -8,  -8.0, -0.4, 1.2, 0.5, -8, 0.8,  8,  0.8, 18000, 18000, 1000, 0.1,  18000, 1, 1.6, 0.20, },
  { 1.1,  -8,  -8.0,  0.6, 1.2, 0.5, -8, 0.9, 16,  1.2, 18000, 10000,  300, 0.1,  18000, 1, 0.4, 0.40, },
  { 1.1,  -8,  -8.0, -0.4, 2.2, 0.2, -8, 0.9, 10,  1.8, 18000, 16000, 1000, 0.1,  18000, 1, 1.6, 0.20, },
  { 1.1,  -8,  -8.0,  0.6, 2.2, 0.2, -8, 0.9, 20,  1.9, 18000,  9000,  500, 0.1,  18000, 1, 0.4, 0.40, },
  { 1.1,  -7,  -7.0, -0.4, 1.2, 0.5,-70, 0.8,  8,  0.8, 18000, 18000, 1000, 0.1,  18000, 1, 1.6, 0.20, },
  { 1.1,  -7,  -7.0,  0.6, 1.2, 0.5,-70, 0.9, 16,  1.2, 18000, 10000,  300, 0.1,  18000, 1, 0.4, 0.40, },

  { 1.1, -20, -70.0,  1.0, 1.0, 0.0, -8, 1.0,  0,  1.8, 18000, 16000, 1000, 0.1,  18000, 1, 1.6, 0.20, },
  { 1.1, -20, -70.0,  1.0, 1.0, 0.0, -8, 1.0,  0,  1.9, 18000,  9000,  500, 0.2,  18000, 1, 0.4, 0.40, },

  { 1.1, -15, -16.0,  0.1, 1.0, 0.1, -5, 1.0,  0, 12.0, 18000, 10000,  100, 0.05, 18000, 1, 1.0, 0.50, },
  { 1.1, -15, -16.0,  0.1, 1.0, 0.1, -5, 1.0,  0, 30.0, 18000,  9000,  100, 0.05, 18000, 1, 1.0, 0.60, },
};

static const char * grpName[6] = { "LEV", "ER", "RVB", "FLT", "RT60", "MOD", };
static const VstInt32 grpMtx[6][6] = {
  {KDry, KERefWet, KERtoLate, KWet, -1, -1},
  {KERefWet, KERefWidth, KERefFactor, KERtoLate, -1, -1,},
  {KOFactor, KWidth, KDelay, KBassLPF, KBassFactor, KRT60,},
  {KInputLPF, KDampLPF, KBassLPF, KBassFactor, KOutputLPF, KOutputBW,},
  {KRT60, KRT60, KRT60, KRT60, KRT60, KRT60,},
  {-1, -1, -1, -1, KSpin, KWander,},
};
static const char * nameMtx[6][6] = {
  {"DRY", "EWET", "ESEN", "WET", "", "",},
  {"EWET", "EWID", "EFAC", "ESEN", "", "",},
  {"FAC", "WID", "IDEL", "BASL", "BASF", "RT60",},
  {"ILPF", "DAMP", "BASL", "BASF", "OLPF", "OLBW",},
  {"RT60", "RT60", "RT60", "RT60", "RT60", "RT60",},
  {"", "", "", "", "SPN", "WAN",},
};

static float ParamConverter(int index, float value){ return ProGRev::model2param(index, value); }

ProGRev::ProGRev(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'P', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  converter_type = FV3_SRC_LPF_IIR_2;
  vERtoLate = dryDB = erDB = 0;

  dsp_eref = new EARLYREF();
  dsp_eref->setMuteOnChange(true);
  dsp_eref->setdryr(0);
  dsp_eref->setwet(0); // 0dB
  dsp_eref->setLRDelay(0.3);
  dsp_eref->setLRCrossApFreq(750, 4);
  dsp_eref->setDiffusionApFreq(150, 4);
  dsp_eref->setSampleRate(FV3_REVBASE_DEFAULT_FS);

  dsp_prev = new PROGREV();
  dsp_prev->setMuteOnChange(true);
  dsp_prev->setdryr(0); // mute dry signal
  dsp_prev->setSampleRate(FV3_REVBASE_DEFAULT_FS);

  programs = new ProGRevProgram[numPrograms];
  for(int pc = 0;pc < KPresetPrograms;pc ++) programs[pc].setProgram(presetProgramName[pc], presetProgram[pc]);
  setProgram(0);
  GeneralEditor * _editor = new GeneralEditor(this);
  _editor->registerConst(EFFECT_NAME, grpName, grpMtx, nameMtx);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;

  ModeProG2 = true;
}

ProGRev::~ProGRev()
{
  freeProcessBlock();
  delete[] programs;
  delete dsp_eref;
  delete dsp_prev;
}

bool ProGRev::getEffectName (char* name)
{
  strcpy (name, EFFECT_NAME);
  return true;
}

bool ProGRev::getVendorString (char* text)
{
  strcpy (text, VENDOR_STRING);
  return true;
}

bool ProGRev::getProductString (char* text)
{
  strcpy (text, "Freeverb3");
  return true;
}

VstPlugCategory ProGRev::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 ProGRev::canDo (char* text)
{
  if (!strcmp (text, "1in1out")) return 1;
  if (!strcmp (text, "2in2out")) return 1;
  if (!strcmp (text, "1in2out")) return 1;
  return -1;
}

bool ProGRev::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool ProGRev::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

VstInt32 ProGRev::getProgram()
{
  return curProgram;
}

bool ProGRev::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

void ProGRev::setProgram(VstInt32 program)
{
  if(programs == NULL||program >= numPrograms||program < 0) return;
  ProGRevProgram * p = &programs[program];
  curProgram = program;
  setParameterM(KOFactor,   p->fOFactor);
  setParameterM(KDry,       p->fDry);
  setParameterM(KERefWet,   p->fERefWet);
  setParameterM(KERefWidth, p->fERefWidth);
  setParameterM(KERefFactor,p->fERefFactor);
  setParameterM(KERtoLate,  p->fERtoLate);
  setParameterM(KWet,       p->fWet);
  setParameterM(KWidth,     p->fWidth);
  setParameterM(KDelay,     p->fDelay);
  setParameterM(KRT60,      p->fRT60);
  setParameterM(KInputLPF,  p->fInputLPF);
  setParameterM(KDampLPF,   p->fDampLPF);
  setParameterM(KBassLPF,   p->fBassLPF);
  setParameterM(KBassFactor,p->fBassFactor);
  setParameterM(KOutputLPF, p->fOutputLPF);
  setParameterM(KSpin,      p->fSpin);
  setParameterM(KWander,    p->fWander);
  lock();
  dsp_eref->mute();
  dsp_prev->mute();
  ProcessBlock::mute();
  unlock();
}

void ProGRev::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

bool ProGRev::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void ProGRev::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void ProGRev::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void ProGRev::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void ProGRev::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  ProGRevProgram * p = &programs[curProgram];
  long factorp = 1;
  switch (index)
    {
    case KOFactor:
      p->fOFactor = value;
      factorp = std::ceil(value); if(factorp == 0) factorp = 1;
      if((value-std::floor(value)) > 0.5||value == 4)
	{
	  ModeProG2 = false;
	  dsp_prev->setReverbType(FV3_REVTYPE_PROG);
	}
      else
	{
	  ModeProG2 = true;
	  dsp_prev->setReverbType(FV3_REVTYPE_PROG2);
	}
      
      if(factorp != dsp_prev->getOSFactor())
	{
	  lock();
	  dsp_prev->setOSFactor((int)factorp, converter_type);
	  unlock();
	}
      break;
    case KDry:
      dryDB = (p->fDry = value);
      break;
    case KERefWet:
      erDB = (p->fERefWet = value);
      break;
    case KERefWidth:
      dsp_eref->setwidth(p->fERefWidth = value);
      break;
    case KERefFactor:
      lock();
      dsp_eref->setRSFactor(p->fERefFactor = value);
      unlock();
      break;
    case KERtoLate:
      vERtoLate = (p->fERtoLate = value);
      break;

    case KWet:
      dsp_prev->setwet(p->fWet = value);
      break;
    case KWidth:
      dsp_prev->setwidth(p->fWidth = value);
      break;
    case KDelay:
      lock();
      dsp_prev->setPreDelay(p->fDelay = value);
      unlock();
      break;
    case KRT60:
      p->fRT60 = value;
      // for exception
      if(value >= 0.1) dsp_prev->setrt60(value);
      else dsp_prev->setrt60(0.1);
      break;
    case KInputLPF:
      dsp_prev->setinputdamp(p->fInputLPF = value);
      break;
    case KDampLPF:
      dsp_prev->setdamp(p->fDampLPF = value);
      break;
    case KBassLPF:
      dsp_prev->setdamp2(p->fBassLPF = value);
      break;
    case KBassFactor:
      dsp_prev->setbassboost(p->fBassFactor = value);
      break;
    case KOutputLPF:
      dsp_prev->setoutputdamp(p->fOutputLPF = value);
      break;
    case KOutputBW:
      dsp_prev->setoutputdampbw(p->fOutputBW = value);
      break;
    case KSpin:
      dsp_prev->setspin(p->fSpin = value);
      dsp_prev->setspin2(std::sqrt(100.-(10.-value)*(10.-value))/2.);
      break;
    case KWander:
      // The values are limited to reduce buggy signal clips.
      dsp_prev->setwander(0.1 + (p->fWander = value)/2.);
      dsp_prev->setwander2(0.1 + value/2.);
      break;
    default:
      break;
    }
}

float ProGRev::getParameter(VstInt32 index)
{
  float ret = 0.0f;
  ProGRevProgram * p = &programs[curProgram];
  ret = p->getParameterValue(index);
  return model2param(index, ret);
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t ProGRev::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KOFactor:
      if(p2m) return LINCV(0,1,0,4,value);
      else return LINCV(0,4,0,1,value);
    case KOutputBW:
      if(p2m) return LINCV(0,1,0.5,5.5,value);
      else return LINCV(0.5,5.5,0,1,value);
    case KERefWidth:
      if(p2m) return LINCV(0,1,-1,1,value);
      else return LINCV(-1,1,0,1,value);
    case KERefFactor:
      if(p2m) return LINCV(0,1,0.5,2.5,value);
      else return LINCV(0.5,2.5,0,1,value);
    case KDry:
    case KERefWet:
    case KWet:
      if(p2m) return LINCV(0,1,-70,10,value);
      else return LINCV(-70,10,0,1,value);
    case KDelay:
      if(p2m) return LINCV(0,1,-500,500,value);
      else return LINCV(-500,500,0,1,value);
    case KRT60:
      if(p2m) return (value*value*value*value)*30.;
      else return std::sqrt(std::sqrt(value/30.));
    case KBassLPF:
      if(p2m) return LINCV(0,1,50,1050,value);
      else return LINCV(50,1050,0,1,value);
    case KInputLPF:
    case KDampLPF:
    case KOutputLPF:
      if(p2m) return LINCV(0,1,200,18000,value);
      else return LINCV(200,18000,0,1,value);
    case KSpin:
      if(p2m) return LINCV(0,1,0,10,value);
      else return LINCV(0,10,0,1,value);
    case KWander:
      if(p2m) return (value*value);
      else return std::sqrt(value);
    case KBassFactor:
      if(p2m) return LINCV(0,1,0,0.5,value);
      else return LINCV(0,0.5,0,1,value);
    case KWidth:
    case KERtoLate:
    default:
      return value;
    }
}

pfloat_t ProGRev::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float ProGRev::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void ProGRev::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void ProGRev::getParameterDisplay(VstInt32 index, char *text)
{
  ProGRevProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOFactor:
      if(ModeProG2)
        snprintf(text, kVstMaxParamStrLen, "%ld-A", dsp_prev->getOSFactor());
      else
        snprintf(text, kVstMaxParamStrLen, "%ld-B", dsp_prev->getOSFactor());
      break;
    default:
      float2string(p->getParameterValue(index), text, kVstMaxParamStrLen);
      break;
    }
}

void ProGRev::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float ProGRev::getSampleRate()
{
  return currentFs;
}

void ProGRev::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||dsp_prev->getSampleRate() != sampleRate)
    {
      currentFs = (double)sampleRate;
      lock();
      dsp_eref->setSampleRate(sampleRate);
      dsp_prev->setSampleRate(sampleRate);
      unlock();
    }
}

void ProGRev::suspend()
{
  ;
}

void ProGRev::resume()
{
  lock();
  if(updateBlockSize() > 0)
    {
      allocProcessBlock(updateBlockSize());
      tmp1Block.alloc(updateBlockSize(), 2);
      tmp2Block.alloc(updateBlockSize(), 2);
    }
  dsp_eref->mute();
  dsp_prev->mute();
  ProcessBlock::mute();
  unlock();
  setInitialDelay(dsp_eref->getLatency()+dsp_prev->getLatency()+ProcessBlock::getLatency());
}

void ProGRev::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void ProGRev::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void ProGRev::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void ProGRev::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  UTILS::mute(outL, sampleFrames);
  UTILS::mute(outR, sampleFrames);
  
  if(tmp1Block.getsize() < sampleFrames)
    {
      lock();
      try
	{
	  tmp1Block.alloc(sampleFrames, 2);
	  tmp2Block.alloc(sampleFrames, 2);
	}
      catch(std::bad_alloc){ unlock(); return; }
      unlock();
    }

  if(tryLock() == true)
    {
      dsp_eref->processreplace(inL,inR,tmp1Block.L,tmp1Block.R,(long int)sampleFrames);
      for(long i = 0;i < sampleFrames;i ++)
	{
	  tmp2Block.L[i] = tmp1Block.L[i]*vERtoLate + inL[i];
	  tmp2Block.R[i] = tmp1Block.R[i]*vERtoLate + inR[i];
	}
      dsp_prev->processreplace(tmp2Block.L,tmp2Block.R,outL,outR,(long int)sampleFrames);
      pfloat_t dryR = UTILS::dB2R(dryDB), erR = UTILS::dB2R(erDB);
      for(long i = 0;i < sampleFrames;i ++)
	{
	  outL[i] += tmp1Block.L[i]*erR + inL[i]*dryR;
	  outR[i] += tmp1Block.R[i]*erR + inR[i]*dryR;
	}
      unlock();
    }
}

void ProGRev::setConverterType(int type)
{
  converter_type = type;
}

void ProGRev::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
