/**
 *  Freeverb
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "FReverb.hpp"
#include "GeneralEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "SRCFacto", "Wet", "Dry", "Damp", "RoomSize", "Width", "Delay",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "X", "dB", "dB", "%", "u", "%", "ms",
};

#define KNumPrograms 24

static const char * grpName[6] = { "LEV", "RVB", "", "", "", "", };
static const VstInt32 grpMtx[6][6] = {
  {KDry, KWet, -1, -1, -1, -1},
  {KOFactor, KWidth, KDelay, KDamp, KRoomsize, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
};
static const char * nameMtx[6][6] = {
  {"DRY", "WET", "", "", "", "",},
  {"FAC", "WID", "IDEL", "DAMP", "SIZE", "",},
  {"", "", "", "", "", "",},
  {"", "", "", "", "", "",},
  {"", "", "", "", "", "",},
  {"", "", "", "", "", "",},
};

static float ParamConverter(int index, float value){ return FReverb::model2param(index, value); }

FReverb::FReverb(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'F', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  converter_type = FV3_SRC_LPF_IIR_2;
  model = new REVMODEL;
  model->setMuteOnChange(true);
  model->setSampleRate(FV3_REVBASE_DEFAULT_FS);
  programs = new FReverbProgram[numPrograms];
  setProgram(0);
  GeneralEditor * _editor = new GeneralEditor(this);
  _editor->registerConst(EFFECT_NAME, grpName, grpMtx, nameMtx);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;
}

FReverb::~FReverb()
{
  freeProcessBlock();
  delete[] programs;
  delete model;
}

bool FReverb::getEffectName (char* name)
{
  strcpy(name, EFFECT_NAME);
  return true;
}

bool FReverb::getVendorString (char* text)
{
  strcpy(text, VENDOR_STRING);
  return true;
}

bool FReverb::getProductString (char* text)
{
  strcpy(text, "Freeverb3");
  return true;
}

VstPlugCategory FReverb::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 FReverb::canDo (char* text)
{
  if (!strcmp(text, "1in1out")) return 1;
  if (!strcmp(text, "2in2out")) return 1;
  if (!strcmp(text, "1in2out")) return 1;
  return -1;
}

bool FReverb::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

VstInt32 FReverb::getProgram()
{
  return curProgram;
}

void FReverb::setProgram(VstInt32 program)
{
  if(program < numPrograms)
    {
      FReverbProgram * p = &programs[program];
      curProgram = program;
      setParameterM(KOFactor, p->fOFactor);
      setParameterM(KWet, p->fWet);
      setParameterM(KDry, p->fDry);
      setParameterM(KDamp, p->fDamp);
      setParameterM(KRoomsize, p->fRoomsize);
      setParameterM(KWidth, p->fWidth);
      setParameterM(KDelay, p->fDelay);
      lock();
      model->mute();
      ProcessBlock::mute();
      unlock();
    }
}

bool FReverb::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool FReverb::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool FReverb::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void FReverb::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

void FReverb::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void FReverb::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void FReverb::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void FReverb::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  FReverbProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOFactor:
      lock();
      p->fOFactor = value;
      model->setOSFactor((int)value, converter_type);
      unlock();
      break;
    case KWet:
      model->setwet(p->fWet = value);
      break;
    case KDry:
      model->setdry(p->fDry = value);
      break;
    case KDamp:
      model->setdamp(p->fDamp = value);
      break;
    case KRoomsize:
      model->setroomsize(p->fRoomsize = value);
      break;
    case KWidth:
      model->setwidth(p->fWidth = value);
      break;
    case KDelay:
      lock();
      model->setPreDelay(p->fDelay = value);
      unlock();
      break;
    default:
      break;
    }
}

float FReverb::getParameter(VstInt32 index)
{
  float ret = 0;
  FReverbProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOFactor:
      ret = p->fOFactor;
      break;
    case KWet:
      ret = p->fWet;
      break;
    case KDry:
      ret = p->fDry;
      break;
    case KDamp:
      ret = p->fDamp;
      break;
    case KRoomsize:
      ret = p->fRoomsize;
      break;
    case KWidth:
      ret = p->fWidth;
      break;
    case KDelay:
      ret = p->fDelay;
      break;
    default:
      break;
    }
  return model2param(index, ret);
}

void FReverb::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void FReverb::getParameterDisplay(VstInt32 index, char *text)
{
  FReverbProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOFactor:
      int2string((int)p->fOFactor, text, kVstMaxParamStrLen);
      break;
    case KWet:
      dB2string(p->fWet*FV3_FREV_SCALE_WET, text, kVstMaxParamStrLen);
      break;
    case KDry:
      dB2string(p->fDry*FV3_FREV_SCALE_DRY, text, kVstMaxParamStrLen);
      break;
    case KDamp:
      float2string(p->fDamp*100, text, kVstMaxParamStrLen);
      break;
    case KRoomsize:
      float2string(p->fRoomsize*FV3_FREV_SCALE_ROOM+FV3_FREV_OFFSET_ROOM, text, kVstMaxParamStrLen);
      break;
    case KWidth:
      float2string(p->fWidth*100, text, kVstMaxParamStrLen);
      break;
    case KDelay:
      float2string(p->fDelay, text, kVstMaxParamStrLen);
      break;
    default:
      break;
    }
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t FReverb::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KOFactor:
      if(p2m) return LINCV(0,1,1,5,value);
      else return LINCV(1,5,0,1,value);
    case KDelay:
      if(p2m) return LINCV(0,1,-500,500,value);
      else return LINCV(-500,500,0,1,value);
    case KRoomsize:
    case KWet:
    case KDry:
    case KWidth:
    case KDamp:
    default:
      return value;
    }
}

pfloat_t FReverb::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float FReverb::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void FReverb::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float FReverb::getSampleRate()
{
  return currentFs;
}

void FReverb::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||model->getSampleRate() != sampleRate)
    {
      currentFs = (double)sampleRate;
      lock();
      model->setSampleRate(sampleRate);
      unlock();
    }
}

void FReverb::suspend()
{
  ;
}

void FReverb::resume()
{
  lock();
  if(updateBlockSize() > 0) allocProcessBlock(updateBlockSize());
  model->mute();
  ProcessBlock::mute();
  unlock();
  setInitialDelay(model->getLatency()+ProcessBlock::getLatency());
}

void FReverb::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void FReverb::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void FReverb::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void FReverb::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  if(tryLock() == true)
    {
      model->processreplace(inL,inR,outL,outR,(long int)sampleFrames);
      unlock();
    }
  else
    {
      memcpy(outL, inL, sizeof(pfloat_t)*sampleFrames);
      memcpy(outR, inR, sizeof(pfloat_t)*sampleFrames);
    }
}

void FReverb::setConverterType(int type)
{
  converter_type = type;
}

void FReverb::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
