/**
 *  WindCompressor
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "MBCompressor.hpp"
#include "MBCompressorEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "LoFreqDv", "HiFreqDv", "OACRMS  ", "OACLooka",
  "L Attack", "L Releas", "L Thresh", "L SoftKn", "L Ratio ", "L Gain  ",
  "M Attack", "M Releas", "M Thresh", "M SoftKn", "M Ratio ", "M Gain  ",
  "H Attack", "H Releas", "H Thresh", "H SoftKn", "H Ratio ", "H Gain  ",
  "LimitRMS", "LiLookah", "LiAttack", "LiReleas", "LThresho", "LCeiling",
  "AutoGain",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "Hz", "Hz", "ms", "ms",
  "ms", "ms", "dB", "dB", "x", "dB", 
  "ms", "ms", "dB", "dB", "x", "dB", 
  "ms", "ms", "dB", "dB", "x", "dB", 
  "ms", "ms", "ms", "ms", "dB", "dB",
  "-",
};

#define KNumPrograms 32
#define KPresetPrograms 16
static char presetProgramName[KPresetPrograms][kVstMaxProgNameLen+1] = {
  "Default Compressor",
  "CD Master 1",
  "CD Master 2",
  "MaxLevel SoftClip",
  "MaxLevel HardClip",
  "Vocal",
  "De Esser",
  "Punchy Mix",
  "Dance 1",
  "Dance 2",
  "Dance 3",
  "ClubMix",
  "Lo-Fi 1",
  "Lo-Fi 2",
  "AM Radio",
  "Slap Bass",
};

static float presetProgram[KPresetPrograms][KNumParams] = {
  { 140, 6300, 30, 10,
    0, 50, -27, 10, 5, -4,
    0, 80, -17, 10, 3, -4,
    0, 80, -20, 10, 7, -4,
    0, 1, 0, 100, -2, 0, 1, },
  { 140, 6300, 30, 10,
    0, 50, -27, 10, 5, 0,
    0, 80, -17, 10, 3, 0,
    0, 80, -20, 10, 7, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 140, 3200, 30, 10,
    0, 50, -27, 10, 5, 0,
    0, 80, -17, 10, 3, 0,
    0, 80, -20, 10, 7, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 315, 3200, 30, 10,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 315, 3200, 30, 10,
    6, 25, -17, 0, 3.5, 0,
    6, 25, -17, 0, 3.5, 0,
    6, 25, -17, 0, 3.5, 0,
    0, 1, 0, 100, 0, 0, 1, },
  { 560, 5000, 30, 10,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -48, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    0, 1, 0, 100, -1, 0, 1, },
  { 160, 7000, 30, 10,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -34, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 0, },
  { 200, 2000, 30, 10,
    6, 25, -27, 10, 3.5, 0,
    6, 25, 0,   10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 70, 11000, 30, 10,
    6, 25, -27, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -34, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 70, 6300, 30, 10,
    6, 25, -27, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -34, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 70, 3200, 30, 10,
    6, 25, -27, 10, 3.5, 0,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -34, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 160, 3200, 30, 10,
    40, 25, -30, 10, 3.5, 0,
    6,  25, -17, 10, 3.5, 0,
    6,  25, -17, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
  { 180, 2600, 30, 10,
    26, 25, -5,  10, 5,   0,
    6,  25, -17, 10, 3.5, 0,
    6,  25, 0,   10, 1,   0,
    0, 1, 0, 100, -2, 0, 1, },
  { 560, 7000, 30, 10,
    6,  25, -17, 10, 3.5, -60,
    18, 25, -48, 10, 5,   -10,
    6,  25, -17, 10, 3.5, -60,
    0, 1, 0, 100, -2, 0, 1, },
  { 100, 3200, 30, 10,
    6, 25, -36, 10, 20,  0,
    6, 25, -17, 10, 3.5, 0,
    6, 25, -48, 10, 4.5, 0,
    0, 1, 0, 100, -2, 0, 0, },
  { 180, 4000, 30, 10,
    30, 25, -30, 10, 3.5, 0,
    6, 25,  -17, 10, 3.5, 0,
    1, 10,  -21, 10, 3.5, 0,
    0, 1, 0, 100, -2, 0, 1, },
};

MBCompressor::MBCompressor(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker()
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'C', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  editor = new MBCompressorEditor(this);  
  sampleSize = 0;
  fragmentSize = 8192;
  sfSize = 1024;
  gainL = gainM = gainH = 0; // dB
  againL = againM = againH = 1;
  autoGain = 1;
  programs = new MBCProgram[numPrograms];
  for(int pc = 0;pc < KPresetPrograms;pc ++) programs[pc].setProgram(presetProgramName[pc], presetProgram[pc]);
  setSampleRate(FV3_REVBASE_DEFAULT_FS);
  setProgram(0);
}

MBCompressor::~MBCompressor()
{
  freeBlock();
  delete[] programs;
}

bool MBCompressor::getEffectName (char* name)
{
  strcpy(name, EFFECT_NAME);
  return true;
}

bool MBCompressor::getVendorString(char* text)
{
  strcpy(text, VENDOR_STRING);
  return true;
}

bool MBCompressor::getProductString(char* text)
{
  strcpy(text, "Freeverb3");
  return true;
}

VstPlugCategory MBCompressor::getPlugCategory()
{
  return (kPlugCategMastering);
}

VstInt32 MBCompressor::canDo(char* text)
{
  if (!strcmp (text, "1in1out")) return 1;
  if (!strcmp (text, "2in2out")) return 1;
  if (!strcmp (text, "1in2out")) return 1;
  return -1;
}


bool MBCompressor::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool MBCompressor::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}


bool MBCompressor::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

VstInt32 MBCompressor::getProgram()
{
  return curProgram;
}

void MBCompressor::setProgram(VstInt32 program)
{
  if(program < numPrograms)
    {
      MBCProgram * p = &programs[program];
      curProgram = program;

      setParameterM(KFreqLowDivider, p->fFreqLowDivider);
      setParameterM(KFreqHighDivider, p->fFreqHighDivider);
      setParameterM(KCompRMS, p->fCompRMS);
      setParameterM(KCompLookahead, p->fCompLookahead);

      setParameterM(KCompLowA, p->fCompLowA);
      setParameterM(KCompLowR, p->fCompLowR);
      setParameterM(KCompLowT, p->fCompLowT);
      setParameterM(KCompLowSK, p->fCompLowSK);
      setParameterM(KCompLowRa, p->fCompLowRa);
      setParameterM(KCompLowG, p->fCompLowG);

      setParameterM(KCompMidA, p->fCompMidA);
      setParameterM(KCompMidR, p->fCompMidR);
      setParameterM(KCompMidT, p->fCompMidT);
      setParameterM(KCompMidSK, p->fCompMidSK);
      setParameterM(KCompMidRa, p->fCompMidRa);
      setParameterM(KCompMidG, p->fCompMidG);

      setParameterM(KCompHighA, p->fCompHighA);
      setParameterM(KCompHighR, p->fCompHighR);
      setParameterM(KCompHighT, p->fCompHighT);
      setParameterM(KCompHighSK, p->fCompHighSK);
      setParameterM(KCompHighRa, p->fCompHighRa);
      setParameterM(KCompHighG, p->fCompHighG);

      setParameterM(KLimitRMS, p->fLimitRMS);
      setParameterM(KLimitLookahead, p->fLimitLookahead);
      setParameterM(KLimitA, p->fLimitA);
      setParameterM(KLimitR, p->fLimitR);
      setParameterM(KLimitT, p->fLimitT);
      setParameterM(KLimitC, p->fLimitC);
      setParameterM(KAutoGain, p->fAutoGain);

      lock();
      muteAll();
      unlock();
    }
}

void MBCompressor::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

bool MBCompressor::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void MBCompressor::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void MBCompressor::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((MBCompressorEditor *)editor)->setParameter(index, value);
}

void MBCompressor::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((MBCompressorEditor *)editor)->setParameter(index, model2param(index, value));
}

void MBCompressor::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  MBCProgram * p = &programs[curProgram];
  switch (index)
    {
    case KFreqLowDivider:
      lock();
      split.setLowFreqDivider(p->fFreqLowDivider = value);
      muteAll();
      unlock();
      break;
    case KFreqHighDivider:
      lock();
      split.setHighFreqDivider(p->fFreqHighDivider = value);
      muteAll();
      unlock();
      break;
    case KCompRMS:
      lock();
      compL.setRMS(p->fCompRMS = value);
      compM.setRMS(value);
      compH.setRMS(value);
      muteAll();
      unlock();
      break;
    case KCompLookahead:
      lock();
      compL.setLookahead(p->fCompLookahead = value);
      compM.setLookahead(value);
      compH.setLookahead(value);
      muteAll();
      unlock();
      break;
      
      // lowband
    case KCompLowA:
      compL.setAttack(p->fCompLowA = value);
      break;
    case KCompLowR:
      compL.setRelease(p->fCompLowR = value);
      break;
    case KCompLowT:
      compL.setThreshold(p->fCompLowT = value);
      againL = AGC(compL.getThreshold(), compL.getRatio());
      break;
    case KCompLowSK:
      compL.setSoftKnee(p->fCompLowSK = value);
      break;
    case KCompLowRa:
      compL.setRatio(p->fCompLowRa = value);
      againL = AGC(compL.getThreshold(), compL.getRatio());
      break;
    case KCompLowG:
      gainL = p->fCompLowG = value;
      break;

      // midband
    case KCompMidA:
      compM.setAttack(p->fCompMidA = value);
      break;
    case KCompMidR:
      compM.setRelease(p->fCompMidR = value);
      break;
    case KCompMidT:
      compM.setThreshold(p->fCompMidT = value);
      againM = AGC(compM.getThreshold(), compM.getRatio());
      break;
    case KCompMidSK:
      compM.setSoftKnee(p->fCompMidSK = value);
      break;
    case KCompMidRa:
      compM.setRatio(p->fCompMidRa = value);
      againM = AGC(compM.getThreshold(), compM.getRatio());
      break;
    case KCompMidG:
      gainM = p->fCompMidG = value;
      break;

      // highband
    case KCompHighA:
      compH.setAttack(p->fCompHighA = value);
      break;
    case KCompHighR:
      compH.setRelease(p->fCompHighR = value);
      break;
    case KCompHighT:
      compH.setThreshold(p->fCompHighT = value);
      againH = AGC(compH.getThreshold(), compH.getRatio());
      break;
    case KCompHighSK:
      compH.setSoftKnee(p->fCompHighSK = value);
      break;
    case KCompHighRa:
      compH.setRatio(p->fCompHighRa = value);
      againH = AGC(compH.getThreshold(), compH.getRatio());
      break;
    case KCompHighG:
      gainH = p->fCompHighG = value;
      break;
      
    case KLimitRMS:
      lock();
      limit.setRMS(p->fLimitRMS = value);
      muteAll();
      unlock();
      break;
    case KLimitLookahead:
      lock();
      limit.setLookahead(p->fLimitLookahead = value);
      limit.setLookaheadRatio(1);
      if(currentFs != 0)
	{
	  limiterDelay = (int)(currentFs*value/1000.0);
	  lLookaheadDelayL.setsize(totalDelay - limiterDelay);
	  lLookaheadDelayR.setsize(totalDelay - limiterDelay);
	}
      muteAll();
      unlock();
      break;
    case KLimitA:
      limit.setAttack(p->fLimitA = value);
      break;
    case KLimitR:
      limit.setRelease(p->fLimitR = value);
      break;
    case KLimitT:
      limit.setThreshold(p->fLimitT = value);
      if(p->fLimitT > p->fLimitC)
	limit.setThreshold(p->fLimitC);
      break;
    case KLimitC:
      limit.setCeiling(p->fLimitC = value);
      if(p->fLimitT > p->fLimitC)
	limit.setCeiling(p->fLimitT);
      break;
      
    case KAutoGain:
      p->fAutoGain = autoGain = value;
      break;

    default:
      break;
    }
}

float MBCompressor::getParameter(VstInt32 index)
{
  return getParameter(index, true);
}

float MBCompressor::getModelParameter(VstInt32 index)
{
  return getParameter(index, false);
}

float MBCompressor::getParameter(VstInt32 index, bool vstp)
{
  float ret = 0.0f;
  switch (index)
    {
    case KFreqLowDivider:
      ret = split.getLowFreqDivider();
      break;
    case KFreqHighDivider:
      ret = split.getHighFreqDivider();
      break;
    case KCompRMS:
      ret = compL.getRMS();
      break;
    case KCompLookahead:
      ret = compL.getLookahead();
      break;
      
      // lowband
    case KCompLowA:
      ret = compL.getAttack();
      break;
    case KCompLowR:
      ret = compL.getRelease();
      break;
    case KCompLowT:
      ret = compL.getThreshold();
      break;
    case KCompLowSK:
      ret = compL.getSoftKnee();
      break;
    case KCompLowRa:
      ret = compL.getRatio();
      break;
    case KCompLowG:
      ret = gainL;
      break;
      
      // midband
    case KCompMidA:
      ret = compM.getAttack();
      break;
    case KCompMidR:
      ret = compM.getRelease();
      break;
    case KCompMidT:
      ret = compM.getThreshold();
      break;
    case KCompMidSK:
      ret = compM.getSoftKnee();
      break;
    case KCompMidRa:
      ret = compM.getRatio();
      break;
    case KCompMidG:
      ret = gainM;
      break;
      
      // highband
    case KCompHighA:
      ret = compH.getAttack();
      break;
    case KCompHighR:
      ret = compH.getRelease();
      break;
    case KCompHighT:
      ret = compH.getThreshold();
      break;
    case KCompHighSK:
      ret = compH.getSoftKnee();
      break;
    case KCompHighRa:
      ret = compH.getRatio();
      break;
    case KCompHighG:
      ret = gainH;
      break;
      
    case KLimitRMS:
      ret = limit.getRMS();
      break;
    case KLimitLookahead:
      ret = limit.getLookahead();
      break;
    case KLimitA:
      ret = limit.getAttack();
      break;
    case KLimitR:
      ret = limit.getRelease();
      break;
    case KLimitT:
      ret = limit.getThreshold();
      break;
    case KLimitC:
      ret = limit.getCeiling();
      break;
    case KAutoGain:
      ret = autoGain;
      break;

    default:
      break;
    }
  if(vstp) return model2param(index, ret);
  else return ret;
}

void MBCompressor::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams)
    strcpy(label, parameterName[index]);
}

void MBCompressor::getParameterDisplay(VstInt32 index, char *text)
{
  int len = kVstMaxParamStrLen;
  switch (index)
    {
    case KFreqLowDivider:
      float2string(split.getLowFreqDivider(), text, len);
      break;
    case KFreqHighDivider:
      float2string(split.getHighFreqDivider(), text, len);
      break;
    case KCompRMS:
      float2string(compL.getRMS(), text, len);
      break;
    case KCompLookahead:
      float2string(compL.getLookahead(), text, len);
      break;
      
      // lowband
    case KCompLowA:
      float2string(compL.getAttack(), text, len);
      break;
    case KCompLowR:
      float2string(compL.getRelease(), text, len);
      break;
    case KCompLowT:
      float2string(compL.getThreshold(), text, len);
      break;
    case KCompLowSK:
      float2string(compL.getSoftKnee(), text, len);
      break;
    case KCompLowRa:
      float2string(compL.getRatio(), text, len);
      break;
    case KCompLowG:
      float2string(gainL, text, len);
      break;
      
      // midband
    case KCompMidA:
      float2string(compM.getAttack(), text, len);
      break;
    case KCompMidR:
      float2string(compM.getRelease(), text, len);
      break;
    case KCompMidT:
      float2string(compM.getThreshold(), text, len);
      break;
    case KCompMidSK:
      float2string(compM.getSoftKnee(), text, len);
      break;
    case KCompMidRa:
      float2string(compM.getRatio(), text, len);
      break;
    case KCompMidG:
      float2string(gainM, text, len);
      break;
      
      // highband
    case KCompHighA:
      float2string(compH.getAttack(), text, len);
      break;
    case KCompHighR:
      float2string(compH.getRelease(), text, len);
      break;
    case KCompHighT:
      float2string(compH.getThreshold(), text, len);
      break;
    case KCompHighSK:
      float2string(compH.getSoftKnee(), text, len);
      break;
    case KCompHighRa:
      float2string(compH.getRatio(), text, len);
      break;
    case KCompHighG:
      float2string(gainH, text, len);
      break;
      
    case KLimitRMS:
      float2string(limit.getRMS(), text, len);
      break;
    case KLimitLookahead:
      float2string(limit.getLookahead(), text, len);
      break;
    case KLimitA:
      float2string(limit.getAttack(), text, len);
      break;
    case KLimitR:
      float2string(limit.getRelease(), text, len);
      break;
    case KLimitT:
      float2string(limit.getThreshold(), text, len);
      break;
    case KLimitC:
      float2string(limit.getCeiling(), text, len);
      break;
    case KAutoGain:
      if(autoGain != 0)
	strncpy(text, "ON", len);
      else
	strncpy(text, "OFF", len);
      break;
      
    default:
      break;
    }
}

pfloat_t MBCompressor::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
      // Hz 50-10000
    case KFreqLowDivider:
    case KFreqHighDivider:
      if(p2m) return LINCV(0,1,25,20000,value);
      else return LINCV(25,20000,0,1,value);

      // ms 0-100
    case KCompRMS:
    case KCompLookahead:
    case KLimitRMS:
      if(p2m) return LINCV(0,1,0,100,value);
      else return LINCV(0,100,0,1,value);

      // ms 0-10
    case KLimitLookahead:
      if(p2m) return LINCV(0,1,0,10,value);
      else return LINCV(0,10,0,1,value);

      // ms 0-5000
    case KCompLowA:
    case KCompLowR:
    case KCompMidA:
    case KCompMidR:
    case KCompHighA:
    case KCompHighR:
    case KLimitA:
    case KLimitR:
      if(p2m) return LINCV(0,1,0,5000,value);
      else return LINCV(0,5000,0,1,value);

      // dB -60-20
    case KCompLowT:
    case KCompLowG:
    case KCompMidT:
    case KCompMidG:
    case KCompHighT:
    case KCompHighG:
    case KLimitT:
    case KLimitC:
      if(p2m) return LINCV(0,1,-60,20,value);
      else return LINCV(-60,20,0,1,value);

      // dB 0-20
    case KCompLowSK:
    case KCompMidSK:
    case KCompHighSK:
      if(p2m) return LINCV(0,1,0,20,value);
      else return LINCV(0,20,0,1,value);

      // Ratio 0.1-20
    case KCompLowRa:
    case KCompMidRa:
    case KCompHighRa:
      if(p2m) return LINCV(0,1,0.1,20,value);
      else return LINCV(0.1,20,0,1,value);

      // no conversion
    case KAutoGain:
    default:
      return value;
    }
}

pfloat_t MBCompressor::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float MBCompressor::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void MBCompressor::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams)
    strcpy(label, parameterLabel[index]);
}

void MBCompressor::suspend()
{
  if(editor != NULL)
    {
      ((MBCompressorEditor*)editor)->setRMeter(1,1,1,1);
      ((MBCompressorEditor*)editor)->setVuMeter(0,0,0,0);
    }
}

void MBCompressor::resume()
{
  lock();
  if(updateBlockSize() > 0&&updateBlockSize() > sampleSize) allocBlock(updateBlockSize());
  if(0 < updateBlockSize()&&sampleSize != updateBlockSize())
    {
      // reload with new blockSize
      sfSize = UTILS::checkPow2(updateBlockSize());
      split.setFragmentSize(sfSize);
      if(sfSize > fragmentSize)
	split.setFactor(1);
      else
	split.setFactor(fragmentSize/sfSize);
    }
  muteAll();
  unlock();
  
  totalDelay = (int)(currentFs*0.01); // 10ms
  // provide lookahead delay
  // setInitialDelay(split.getLatency()+totalDelay);
  setInitialDelay(split.getLatency());
  
  if(editor != NULL)
    {
      ((MBCompressorEditor*)editor)->setRMeter(1,1,1,1);
      ((MBCompressorEditor*)editor)->setVuMeter(0,0,0,0);
    }
}

void MBCompressor::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void MBCompressor::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void MBCompressor::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void MBCompressor::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  float iL = 0, iR = 0, oL = 0, oR = 0;
  
  for(VstInt32 i = 0;i < sampleFrames;i ++)
    {
      float al = fabs(inL[i]);
      float ar = fabs(inR[i]);
      if(iL < al) iL = al;
      if(iR < ar) iR = ar;      
    }

  if(sampleSize < sampleFrames)
    if(allocBlock(sampleFrames) != sampleFrames) return;
  
  try
    {
      if(tryLock() == true)
	{
	  pfloat_t fgainL = UTILS::dB2R(gainL);
	  pfloat_t fgainM = UTILS::dB2R(gainM);
	  pfloat_t fgainH = UTILS::dB2R(gainH);
	  split.splitR(inL, inR, lowBandInput.L, lowBandInput.R, midBandInput.L, midBandInput.R, highBandInput.L, highBandInput.R, sampleFrames);
	  compL.processreplace(lowBandInput.L, lowBandInput.R, lowBandOutput.L, lowBandOutput.R, sampleFrames);
	  compM.processreplace(midBandInput.L, midBandInput.R, midBandOutput.L, midBandOutput.R, sampleFrames);
	  compH.processreplace(highBandInput.L, highBandInput.R, highBandOutput.L, highBandOutput.R, sampleFrames);
	  if(autoGain != 0)
	    split.mergeR(lowBandOutput.L, lowBandOutput.R, midBandOutput.L, midBandOutput.R, highBandOutput.L, highBandOutput.R, allOutput.L, allOutput.R,
			  againL*fgainL, againM*fgainM, againH*fgainH, sampleFrames);
	  else
	    split.mergeR(lowBandOutput.L, lowBandOutput.R, midBandOutput.L, midBandOutput.R, highBandOutput.L, highBandOutput.R, allOutput.L, allOutput.R,
			  fgainL, fgainM, fgainH, sampleFrames);
	  limit.processreplace(allOutput.L, allOutput.R, outL, outR, sampleFrames);
	  for(VstInt32 i = 0;i < sampleFrames;i ++)
	    {
	      outL[i] = lLookaheadDelayL.process(outL[i]);
	      outR[i] = lLookaheadDelayR.process(outR[i]);
	    }
	  unlock();
	}
      else
	{
	  memcpy(inL, outL, sizeof(pfloat_t)*sampleFrames);
	  memcpy(inR, outR, sizeof(pfloat_t)*sampleFrames);
	}
    }
  catch(...)
    {
      ;
    }

  for(VstInt32 i = 0;i < sampleFrames;i ++)
    {
      float al = fabs(outL[i]);
      float ar = fabs(outR[i]);
      if(oL < al) oL = al;
      if(oR < ar) oR = ar;      
    }
  
  if(editor != NULL)
    {
      ((MBCompressorEditor*)editor)->setRMeter(compL.getCGain(), compM.getCGain(), compH.getCGain(), limit.getCGain());
      ((MBCompressorEditor*)editor)->setVuMeter(iL, iR, oL, oR);
    }
}

int MBCompressor::allocBlock(int size)
{
  freeBlock();
  if(size <= 0) return 0;
  try
    {
      lowBandInput.alloc(size, 2);
      midBandInput.alloc(size, 2);
      highBandInput.alloc(size, 2);
      lowBandOutput.alloc(size, 2);
      midBandOutput.alloc(size, 2);
      highBandOutput.alloc(size, 2);
      allOutput.alloc(size, 2);
    }
  catch(std::bad_alloc)
    {
      freeBlock();
      return 0;
    }
  return (sampleSize = size);
}

void MBCompressor::freeBlock()
{
  if(sampleSize != 0)
    {
      lowBandInput.free();
      midBandInput.free();
      highBandInput.free();
      lowBandOutput.free();
      midBandOutput.free();
      highBandOutput.free();
      allOutput.free();
    }
  sampleSize = 0;
}

pfloat_t MBCompressor::AGC(pfloat_t Threshold, pfloat_t Ratio)
{
  pfloat_t Gain = 1.0/UTILS::dB2R((1-1/Ratio)*Threshold);
  return Gain;
}

float MBCompressor::getSampleRate()
{
  return currentFs;
}

void MBCompressor::setSampleRate (float sampleRate)
{
  if(currentFs != (int)sampleRate)
    {
      currentFs = (int)sampleRate;
      totalDelay = (int)(currentFs*0.01); // 10ms
      limiterDelay = 0;
      lock();
      split.setSampleRate(currentFs);
      compL.setSampleRate(currentFs);
      compM.setSampleRate(currentFs);
      compH.setSampleRate(currentFs);
      limit.setSampleRate(currentFs);
      muteAll();
      unlock();
    }
}

void MBCompressor::setFilterType(int type)
{
  if(type > 0&&type <= 6)
    {
      split.setWindow(type);
      filterType = type;
    }
}

void MBCompressor::setFilterParameter(int value)
{
  if(value >= 0)
    {
      split.setParameter(value);
      filterParameter = value;
    }
}

void MBCompressor::setFragmentSize(int size)
{
  if(size > 0)
    {
      fragmentSize = size;
    }
}

void MBCompressor::setTransitionBandwidth(int value)
{
  if(value > 0)
    {
      lock();
      split.setTransitionBand(value);
      unlock();
    }
}

void MBCompressor::muteAll()
{
  split.mute();
  compL.mute(), compM.mute(), compH.mute();
  limit.mute();
  lLookaheadDelayL.mute();
  lLookaheadDelayR.mute();
  ProcessBlock::mute();
}
