/**
 *  STRev
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "STRev.hpp"
#include "GeneralEditor.hpp"

static char parameterName[KNumParams][kVstMaxParamStrLen+1] = {
  "SRCFact", "Dry", "Wet", "Width", "PreDelay", "RT60", "InLPF", "Damping", "OutLPF", "Mod Fq", "Mod Str",
};

static char parameterLabel[KNumParams][kVstMaxParamStrLen+1] = {
  "X", "dB", "dB", "_", "ms", "s", "Hz", "Hz", "Hz", "Hz", "_",
};

#define KNumPrograms 24
#define KPresetPrograms 13

static char presetProgramName[KPresetPrograms][kVstMaxProgNameLen] = {
  "Default Reverb",

  "XP Hall x1",
  "XP Hall x2",

  "Light Hall x1",
  "Light Hall x2",

  "Dark Hall x1",
  "Dark Hall x2",

  "Light Room x1",
  "Light Room x2",

  "Drum Chamber x1",
  "Drum Chamber x2",
};
static float presetProgram[KPresetPrograms][KNumParams] = {
  // Factor, Dry, Wet, Width, Delay, RT60, InputLPF, DampLPF, OutputLPF, Spin, Wander
  {1, -6.0,  -11.0,  1.0,  10, 2.0, 14000,  3800,  9000, 0.8, 0.4,},
  
  {1, -6.0,  -11.0,  1.0,  10, 2.2, 18000,  4600, 10000, 0.5, 0.4,},
  {2, -6.0,  -11.0,  1.0,  10, 2.4, 14000,  3800,  8000, 0.8, 0.4,},
  
  {1, -4.0,  -14.0,  1.0,  10, 2.3, 19000,  4000, 12000, 0.8, 0.6,},
  {2, -4.0,  -14.0,  1.0,  10, 2.3, 19000,  4000, 12000, 0.8, 0.6,},

  {1, -6.0,  -12.0,  1.0,  10, 2.5,  7500,  3800,  8000, 0.9, 0.4,},
  {2, -6.0,  -12.0,  1.0,  10, 2.5,  7500,  3800,  8000, 0.9, 0.4,},

  {1, -6.0,  -13.0,  1.0,  10, 1.2, 20000, 20000, 20000, 1.0, 0.6,},
  {2, -6.0,  -13.0,  1.0,  10, 1.2, 20000, 20000, 20000, 1.0, 0.6,},

  {1, -6.0,  -13.0,  1.0,  10, 1.2,  5000,  5000,  5000, 1.0, 0.6,},
  {2, -6.0,  -13.0,  1.0,  10, 1.2,  5000,  5000,  5000, 1.0, 0.6,},
};

static const char * grpName[6] = { "LEV", "RVB", "FLT", "", "RT60", "MOD", };
static const VstInt32 grpMtx[6][6] = {
  {KDry, KWet, -1, -1, -1, -1},
  {KOFactor,  KWidth, KDelay, KRT60, -1, -1,},
  {KInputLPF, KDampLPF, KOutputLPF, -1, -1, -1,},
  {-1, -1, -1, -1, -1, -1,},
  {KRT60, KRT60, KRT60, KRT60, KRT60, KRT60,},
  {-1, -1, -1, -1, KSpin, KWander,},
};
static const char * nameMtx[6][6] = {
  {"DRY", "WET", "", "", "", "",},
  {"FAC", "WID", "IDEL", "RT60", "", "",},
  {"ILPF", "DAMP", "OLPF", "", "", "",},
  {"", "", "", "", "", "",},
  {"RT60", "RT60", "RT60", "RT60", "RT60", "RT60",},
  {"", "", "", "", "SPN", "WAN",},
};

static float ParamConverter(int index, float value){ return STRev::model2param(index, value); }

STRev::STRev(audioMasterCallback audioMaster)
  : AudioEffectX(audioMaster, KNumPrograms, KNumParams), ProcessBlock(), Locker(),
    currentFs(FV3_REVBASE_DEFAULT_FS)
{
  setNumInputs(2);
  setNumOutputs(2);
  setUniqueID(CCONST('W', 'n', 'T', '1'));
  canProcessReplacing();
#ifdef PLUGDOUBLE
  canDoubleReplacing();
#endif
  converter_type = FV3_SRC_LPF_IIR_2;
  model = new STREV();
  model->setMuteOnChange(true);
  model->setdccutfreq(6);
  model->setspinlimit(12);
  model->setspindiff(0.15);
  model->setSampleRate(FV3_REVBASE_DEFAULT_FS);

  programs = new STRevProgram[numPrograms];
  for(int pc = 0;pc < KPresetPrograms;pc ++) programs[pc].setProgram(presetProgramName[pc], presetProgram[pc]);
  setProgram(0);
  GeneralEditor * _editor = new GeneralEditor(this);
  _editor->registerConst(EFFECT_NAME, grpName, grpMtx, nameMtx);
  _editor->registerParamConverter(ParamConverter);
  editor = _editor;
}

STRev::~STRev()
{
  freeProcessBlock();
  delete[] programs;
  delete model;
}

bool STRev::getEffectName (char* name)
{
  strcpy (name, EFFECT_NAME);
  return true;
}

bool STRev::getVendorString (char* text)
{
  strcpy (text, VENDOR_STRING);
  return true;
}

bool STRev::getProductString (char* text)
{
  strcpy (text, "Freeverb3");
  return true;
}

VstPlugCategory STRev::getPlugCategory()
{
  return (kPlugCategRoomFx);
}

VstInt32 STRev::canDo (char* text)
{
  if (!strcmp (text, "1in1out")) return 1;
  if (!strcmp (text, "2in2out")) return 1;
  if (!strcmp (text, "1in2out")) return 1;
  return -1;
}

bool STRev::getInputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Input", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

bool STRev::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
  bool returnCode = false;
  if(index == 0)
    {
      sprintf(properties->label, "%s Left Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  else if(index == 1)
    {
      sprintf(properties->label, "%s Right Output", EFFECT_NAME);
      properties->flags = kVstPinIsStereo|kVstPinIsActive;
      returnCode = true;
    }
  return returnCode;
}

VstInt32 STRev::getProgram()
{
  return curProgram;
}

bool STRev::setBypass(bool onOff)
{
  byPass = onOff;
  return onOff;
}

void STRev::setProgram(VstInt32 program)
{
  if(program < numPrograms)
    {
      STRevProgram * p = &programs[program];
      curProgram = program;
      setParameterM(KOFactor,  p->fOFactor);
      setParameterM(KDry,      p->fDry);
      setParameterM(KWet,      p->fWet);
      setParameterM(KWidth,    p->fWidth);
      setParameterM(KDelay,    p->fDelay);
      setParameterM(KRT60,     p->fRT60);
      setParameterM(KInputLPF, p->fInputLPF);
      setParameterM(KDampLPF,  p->fDampLPF);
      setParameterM(KOutputLPF,p->fOutputLPF);
      setParameterM(KSpin,     p->fSpin);
      setParameterM(KWander,   p->fWander);
      lock();
      model->mute();
      ProcessBlock::mute();
      unlock();
    }
}

void STRev::setProgramName(char *name)
{
  strcpy(programs[curProgram].name, name);
}

bool STRev::getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text)
{
  if(index >= numPrograms) return false;
  strcpy(text, programs[index].name);
  return true;
}

void STRev::getProgramName(char *name)
{
  strcpy(name, programs[curProgram].name);
}

void STRev::setParameter(VstInt32 index, float value)
{
  setParameter(index, value, true);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, value);
}

void STRev::setParameterM(VstInt32 index, float value)
{
  setParameter(index, value, false);
  if(editor != NULL) ((AEffGUIEditor*)editor)->setParameter(index, model2param(index, value));
}

void STRev::setParameter(VstInt32 index, float value, bool vstp)
{
  // The VST host stores parameters in float(0.0-1.0).
  if(vstp) value = param2model(index, value);
  // stores parameters to current Program.
  STRevProgram * p = &programs[curProgram];
  long factorp = 1;
  switch (index)
    {
    case KOFactor:
      p->fOFactor = value;
      factorp = (long)value;
      if(factorp != model->getOSFactor())
	{
	  lock();
	  model->setOSFactor((int)factorp, converter_type);
	  unlock();
	}
      break;
    case KDry:
      model->setdry(p->fDry = value);
      break;
    case KWet:
      model->setwet(p->fWet = value);
      break;
    case KWidth:
      model->setwidth(p->fWidth = value);
      break;
    case KDelay:
      lock();
      model->setPreDelay(p->fDelay = value);
      unlock();
      break;
    case KRT60:
      model->setrt60(p->fRT60 = value);
      break;
    case KInputLPF:
      model->setinputdamp(p->fInputLPF = value);
      break;
    case KDampLPF:
      model->setdamp(p->fDampLPF = value);
      break;
    case KOutputLPF:
      model->setoutputdamp(p->fOutputLPF = value);
      break;
    case KSpin:
      model->setspin(p->fSpin = value);
      break;
    case KWander:
      model->setwander(p->fWander = value);
      break;
    default:
      break;
    }
}

float STRev::getParameter(VstInt32 index)
{
  float ret = 0.0f;
  STRevProgram * p = &programs[curProgram];
  ret = p->getParameterValue(index);
  return model2param(index, ret);
}

#define LINCV(imin,imax,omin,omax,val) ((omin)+((val)-(imin))*((omax)-(omin))/((imax)-(imin)))

pfloat_t STRev::pconv(int index, pfloat_t value, bool p2m)
{
  switch (index)
    {
    case KOFactor:
      if(p2m) return LINCV(0,1,1,6,value);
      else return LINCV(1,6,0,1,value);
    case KDry:
    case KWet:
      if(p2m) return LINCV(0,1,-90,20,value);
      else return LINCV(-90,20,0,1,value);
    case KDelay:
      if(p2m) return LINCV(0,1,-500,500,value);
      else return LINCV(-500,500,0,1,value);
    case KRT60:
      if(p2m) return LINCV(0,1,0,16,value);
      else return LINCV(0,16,0,1,value);
    case KInputLPF:
    case KDampLPF:
    case KOutputLPF:
      if(p2m) return LINCV(0,1,0,24000,value);
      else return LINCV(0,24000,0,1,value);
    case KSpin:
      if(p2m) return LINCV(0,1,0,10,value);
      else return LINCV(0,10,0,1,value);
    case KWander:
    case KWidth:
    default:
      return value;
    }
}

pfloat_t STRev::param2model(int index, float value)
{
  return pconv(index, (pfloat_t)value, true);
}

float STRev::model2param(int index, pfloat_t value)
{
  return (float)pconv(index, value, false);
}

void STRev::getParameterName(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterName[index]);
}

void STRev::getParameterDisplay(VstInt32 index, char *text)
{
  STRevProgram * p = &programs[curProgram];
  switch (index)
    {
    case KOFactor:
      int2string((VstInt32)p->getParameterValue(index), text, kVstMaxParamStrLen);
      break;
    default:
      float2string(p->getParameterValue(index), text, kVstMaxParamStrLen);
      break;
    }
}

void STRev::getParameterLabel(VstInt32 index, char *label)
{
  if(index < KNumParams) strcpy(label, parameterLabel[index]);
}

float STRev::getSampleRate()
{
  return currentFs;
}

void STRev::setSampleRate(float sampleRate)
{
  if(currentFs != sampleRate||model->getSampleRate() != sampleRate)
    {
      currentFs = (double)sampleRate;
      lock();
      model->setSampleRate(sampleRate);
      unlock();
    }
}

void STRev::suspend()
{
  ;
}

void STRev::resume()
{
  lock();
  if(updateBlockSize() > 0) allocProcessBlock(updateBlockSize());
  model->mute();
  ProcessBlock::mute();
  unlock();
  setInitialDelay(model->getLatency()+ProcessBlock::getLatency());
}

void STRev::process(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_process(inputs, outputs, sampleFrames);
}

void STRev::processReplacing(float **inputs, float **outputs, VstInt32 sampleFrames)
{
  p_processReplacing(inputs, outputs, sampleFrames);
}

#ifdef PLUGDOUBLE
void STRev::processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames)
{
  p_processDoubleReplacing(inputs, outputs, sampleFrames);
}
#endif

void STRev::processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *outL, pfloat_t *outR, VstInt32 sampleFrames)
{
  if(tryLock() == true)
    {
      model->processreplace(inL,inR,outL,outR,(long int)sampleFrames);
      unlock();
    }
  else
    {
      memcpy(outL, inL, sizeof(pfloat_t)*sampleFrames);
      memcpy(outR, inR, sizeof(pfloat_t)*sampleFrames);
    }
}

void STRev::setConverterType(int type)
{
  converter_type = type;
}

void STRev::setLatency(int size)
{
  ProcessBlock::setLatency(size);
}
