/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef BUFFER_BUILDER_HXX
#define BUFFER_BUILDER_HXX

#include <glibmm/ustring.h>
#include <boost/range.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/utility.hpp>
#include "text_view_layoutable.hxx"
#include "text_line.hxx"
#include "text_element_plain.hxx"
#include "text_element_anchor.hxx"
#include "text_element_id.hxx"
#include "text_element_res_num_map.hxx"
#include "text_element_id_map.hxx"


namespace dialektos {


/*! @brief build text lines and text elements. */
class BufferBuilder : boost::noncopyable {
public:
  /*! @brief constructor. */
  BufferBuilder(text_view::Layoutable& view, int res_num);

  ~BufferBuilder();

  /*! @brief receive a text and output it to the thread view */
  template <typename RangeT, typename RangeT2>
  void add_text(const RangeT& range, bool bold, const RangeT2& href);

  /*! @brief receive an id and output to it the thread view. */
  template <typename RangeT, typename RangeT2>
  void add_id(const RangeT& range, bool bold, const RangeT2& id);

  /*! @brief receive the number of res and output it to the thread view. */
  void add_res_num(int res_num, bool bold);

  /*! @brief append current text line to the thread view and prepare new line. */
  void new_line(int left_margin);

  /*! @brief flush the cache. */
  void flush();

private:

  /*! @brief add respondent numbers to the res num map. */
  template <typename RangeT>
  void add_respondents(const RangeT& range);

  text_view::Layoutable& text_view_;
  int res_num_;
  TextLine* current_line_;
};

template <typename RangeT, typename RangeT2>
void BufferBuilder::add_text(const RangeT& range, bool bold,
    const RangeT2& href) {
  assert( current_line_ != 0);

  typedef typename boost::range_const_iterator<RangeT>::type Iterator;
  Iterator start = boost::begin(range);
  Iterator end = boost::end(range);
  if (current_line_->empty()) {
    // this is the first element. skip white space.
    for (; start != end; ++start) if (*start != ' ') break;
  }

  if (boost::empty(href)){
    current_line_->add_element(
        new text_element::Plain(std::make_pair(start, end), bold));
  } else {
    if (boost::starts_with(std::make_pair(start, end), ">>")) {
      using boost::next;
      add_respondents(std::make_pair(next(next(start)), end));
    }
    current_line_->add_element(
        new text_element::Anchor(std::make_pair(start, end), bold, href));
  }
}

template <typename RangeT, typename RangeT2>
void BufferBuilder::add_id(const RangeT& range, bool bold,
    const RangeT2& id) {
  assert( current_line_ != 0);

  Glib::ustring id_str = Glib::ustring(boost::begin(id), boost::end(id));
  current_line_->add_element(new text_element::ID(range, bold, id_str,
      text_view_.get_id_map()));
  text_view_.get_id_map()->add(id_str, res_num_);
}

template <typename RangeT>
void BufferBuilder::add_respondents(const RangeT& range) {
  typedef typename boost::range_const_iterator<RangeT>::type Iterator;
  using boost::lexical_cast;
  using boost::begin;
  using boost::end;

  boost::iterator_range<Iterator> hiphen;
  hiphen = boost::algorithm::find_first(range, "-");

  if (hiphen.empty()) {
    // consider one res
    int num = 0;
    try {
      num = lexical_cast<int>(std::string(begin(range), end(range)));
    } catch (const std::exception&) {}
    if (num > 0) text_view_.get_res_num_map()->add(num, res_num_);
  } else {
    // consider a range of multiple res
    int left = 0;
    int right = 0;
    try {
      left = lexical_cast<int>(std::string(begin(range), begin(hiphen)));
      right = lexical_cast<int>(std::string(end(hiphen), end(range)));
    } catch (const std::exception&) {}
    if (left > right) std::swap(left, right);
    if (left != 0 && right != 0 && right - left < 10) {
      // TODO the time of dictionary reference should be only once.
      for (int i = left; i <= right; ++i)
        text_view_.get_res_num_map()->add(i, res_num_);
    }
  }
}


}  // namespace dialektos

#endif
