/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef DAT_PARSER_HXX
#define DAT_PARSER_HXX

#include <algorithm>
#include <map>
#include <string>
#include <boost/range.hpp>
#include <boost/algorithm/string.hpp>


namespace dialektos {


/*! @brief DAT parser.

Supply text representing one 'res'.
Once parsing the 'res', the object must not be re-used.
You should prepare new object for next 'res'.
Template argument 'BufferBuilderT' must have interfaces
@code
class BufferBuilderT {
public:
  void add_text(range, bold, href);
  void add_id(range, bold, id);
  void add_res_num(res_num, bold);
  void new_line(left_margin);
};
@endcode
here 'range' is Boost.Range concepts.
DriverT is HTML parser driver template and
FunctionT is HTML parser functions template.
*/
template <typename BufferBuilderT,
template <typename> class HtmlParserDriverT,
template <typename> class HtmlParserFunctionT>
class DatParser:
  public HtmlParserDriverT<HtmlParserFunctionT<
    DatParser<BufferBuilderT, HtmlParserDriverT, HtmlParserFunctionT> > > {
private:
  typedef DatParser<BufferBuilderT,
    HtmlParserDriverT, HtmlParserFunctionT> ThisClass;
  typedef HtmlParserDriverT<HtmlParserFunctionT<ThisClass> > SuperClass;
  using SuperClass::parse_html;

  BufferBuilderT& buffer_builder_;
  const int res_num_;
  int left_margin_;
  bool bold_;
  std::string href_;
  std::string buffer_;
public:

  /*! @brief constructor. */
  explicit DatParser(BufferBuilderT& view, int res_num):
    buffer_builder_(view), res_num_(res_num), left_margin_(1), bold_(false),
    href_() {
  }

  /*! @brief Supply text representing one 'res'. */
  template<typename RangeT>
  void operator()(const RangeT& range) {
    drive_res_line(range);
  }

public:
  /*! @brief implementing FunctionT::on_data */
  template <typename RangeT>
  void on_data(const RangeT& range) {
    buffer_ += std::string(boost::begin(range), boost::end(range));
  }

  /*! @brief implementing FunctionT::on_start_tag */
  template <typename RangeT, typename MapT>
  void on_start_tag(const RangeT& range,
      const MapT& attrs) {
    using boost::algorithm::equals;

    flush();

    if (equals(range, "br")) {
      do_new_line();
    } else if (equals(range, "b")) {
      bold_ = true;
    } else if (equals(range, "a")) {
      typedef typename MapT::key_type KeyType;
      typename MapT::const_iterator h = attrs.find(KeyType("href"));
      if (h != attrs.end()) href_ = (*h).second;
    }
  }

  /*! @brief implementing FunctionT::on_end_tag */
  template <typename RangeT>
  void on_end_tag(const RangeT& range) {
    using boost::algorithm::equals;

    flush();

    if (equals(range, "b")) {
      bold_ = false;
    } else if (equals(range, "a")) {
      href_.erase();
    }
  }

private:
  /*! @brief BufferBuilderT must implement add_text(range, bold, href)*/
  template <typename RangeT, typename RangeT2>
  void add_text(const RangeT& range, bool bold, const RangeT2& href) {
    buffer_builder_.add_text(range, bold, href);
  }

  /*! @brief BufferBuilderT must implement add_id(range, bold, id) */
  template <typename RangeT, typename RangeT2>
  void add_id(const RangeT& range, bool bold, const RangeT2& id) {
    buffer_builder_.add_id(range, bold, id);
  }

  /*! @brief BufferBuilderT must implement add_res_num(res_num, bold) */
  void add_res_num(int res_num, bool bold) {
    buffer_builder_.add_res_num(res_num, bold);
  }

  /*! @brief BufferBuilderT must implement new_line(left_margin) */
  void new_line(int left_margin) {
    buffer_builder_.new_line(left_margin);
  }

private:
  void on_res_num() {

    flush();

    add_res_num(res_num_, bold_);
  }

  template <typename RangeT, typename RangeT2>
  void on_id(const RangeT& range, const RangeT2& id) {

    flush();

    //buffer_builder_.add_text(range, bold_, id);
    add_id(range, bold_, id);
  }

  void do_new_line() {
    flush();
    new_line(left_margin_);
  }

  void flush() {
    if (!buffer_.empty()) {
      add_text(buffer_, bold_, href_);
      buffer_.clear();
    }
  }

  template <typename RangeT>
  void drive_res_line(const RangeT& range) {
    typedef typename boost::range_const_iterator<RangeT>::type Iterator;
    typedef typename boost::iterator_range<Iterator> IterRange;

    if (boost::empty(range)) return;

    boost::iterator_range<Iterator> name;
    boost::iterator_range<Iterator> mail;
    boost::iterator_range<Iterator> date;
    boost::iterator_range<Iterator> msg;

    enum Hogehoge {
      HogeName, HogeMail, HogeDate, HogeMsg, HogeEnd
    };
    Hogehoge state = HogeName;

    for (Iterator it = boost::begin(range); it != boost::end(range);) {

      boost::iterator_range<Iterator> rdlm(it, boost::end(range));
      rdlm = boost::find_first(rdlm, "<>");
      if (!rdlm) break;

      Iterator dlm = boost::begin(rdlm);

      it = skip_white_space(std::make_pair(it, dlm));

      switch (state) {
      case HogeName:
        // Name
        state = HogeMail;
        name = std::make_pair(it, dlm);
        break;
      case HogeMail:
        state = HogeDate;
        mail = std::make_pair(it, dlm);
        break;
      case HogeDate:
        state = HogeMsg;
        date = std::make_pair(it, dlm);
        break;
      case HogeMsg:
        state = HogeEnd;
        msg = std::make_pair(it, dlm);
        break;
      case HogeEnd:
        break;
      }
      it = boost::end(rdlm);
    }

    do_new_line();

    on_res_num();
    on_data(std::string(" "));

    if (state != HogeEnd) {
      // failure parsing the res line.
      on_data(range);
    } else {
      parse_name(name);
      parse_mail(mail);
      parse_date(date);
      left_margin_ = 20;
      do_new_line();
      parse_msg(msg);
    }

    flush();
  }

  template <typename RangeT>
  void parse_name(const RangeT& range) {
    parse_html(std::string("<b>"));
    parse_html(range);
    parse_html(std::string("</b>"));
  }

  template <typename RangeT>
  void parse_mail(const RangeT& range) {
    on_data(std::string("["));
    parse_html(range);
    on_data(std::string("]"));
  }

  template <typename RangeT>
  void parse_date(const RangeT& range) {
    typedef typename boost::range_const_iterator<RangeT>::type Iterator;
    typedef typename boost::iterator_range<Iterator> IterRange;

    IterRange id = range;
    id = boost::find_first(range, "ID:");
    if (id) {
      on_data(std::make_pair(boost::begin(range), boost::begin(id)));
      Iterator id_end = std::find(boost::end(id), boost::end(range), ' ');

      on_id(id, std::make_pair(boost::end(id), id_end));
      on_data(std::make_pair(boost::end(id), boost::end(range)));
    } else {
      on_data(range);
    }
  }

  template <typename RangeT>
  void parse_msg(const RangeT& range) {
    parse_html(range);
  }

};


/*! @brief helper function for running DatParser. */
template <
  template <typename> class HtmlParserDriverT,
  template <typename> class HtmlParserFunctionT,
  typename BufferBuilderT,
  typename RangeT>
void run_parser(BufferBuilderT& view, const RangeT& range, int res_num) {
  DatParser<BufferBuilderT, HtmlParserDriverT, HtmlParserFunctionT>
  parser(view, res_num);
  parser(range);
}

} // namespace dialektos

#endif
