/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "history_data.hxx"

#include <boost/archive/xml_iarchive.hpp>
#include <boost/archive/xml_oarchive.hpp>
#include <boost/serialization/nvp.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <algorithm>
#include <functional>
#include <fstream>
#include <string>
#include <iostream>
#include <vector>
#include <map>
#include "history_item.hxx"
#include "misc.hxx"


namespace dialektos {

namespace history {


namespace {
struct CompareTime :
  public std::binary_function<const Item&, const Item&, bool> {
  bool operator()(const Item& lhs, const Item& rhs) const {
    return lhs.time_ < rhs.time_;
  }
};
} // anonymous namespace


Data::Data() : map_(), writable_(false) {
}

Data::~Data() {
}

void Data::from_xml(const boost::filesystem::path& xml) {
  if (!boost::filesystem::exists(xml) ||
      !boost::filesystem::is_regular_file(xml))
    return;

  std::vector<Item> vec;
  std::ifstream ifs(xml.file_string().c_str());
  try {
    boost::archive::xml_iarchive ia(ifs);
    ia >> boost::serialization::make_nvp("History", vec);
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << e.what() << " " << xml.file_string() << std::endl;
    return;
  }

  map_.clear();
  BOOST_FOREACH(const Item& item, vec) map_[item.uri_] = item;
}

void Data::to_xml(const boost::filesystem::path& xml) const {
  if (!writable_) return;
  if (!misc::create_directories(xml.parent_path())) return;

  std::vector<Item> _vec;
  _vec.reserve(map_.size());
  BOOST_FOREACH(const MapType::value_type& pair, map_)
    _vec.push_back(pair.second);
  std::sort(_vec.begin(), _vec.end(), std::not2(CompareTime()));
  std::vector<Item> vec(_vec.begin(),
      _vec.size() > 100 ? _vec.begin()+100 : _vec.end());

  std::ofstream ofs(xml.file_string().c_str());
  try {
    boost::archive::xml_oarchive oa(ofs);
    oa << boost::serialization::make_nvp("History", vec);
    boost::filesystem::last_write_time(xml.parent_path(),
        boost::filesystem::last_write_time(xml));
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << e.what() << " " << xml.file_string() << std::endl;
  }
}

void Data::push(const Item::UriType& uri, const Item::NameType& name) {
  if (!writable_) return;
  if (uri.empty() || name.empty()) return;

  Item item;
  item.uri_ = uri;
  item.name_ = name;
  item.time_ = boost::posix_time::second_clock::universal_time();
  map_[uri] = item;
}

void Data::erase(const Item::UriType& uri) {
  map_.erase(uri);
}

std::vector<Item> Data::get_recent() const {
  std::vector<Item> order;
  order.reserve(map_.size());
  BOOST_FOREACH(const MapType::value_type& pair, map_)
    order.push_back(pair.second);

  std::sort(order.begin(), order.end(), std::not2(CompareTime()));
  return order;
}


} // namespace history

} // namespace dialektos
