/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TEXT_ELEMENT_PLAIN_HXX
#define TEXT_ELEMENT_PLAIN_HXX

#include <glibmm/refptr.h>
#include <glibmm/ustring.h>
#include <gdkmm/cursor.h>
#include <gdkmm/rectangle.h>
#include <pangomm/context.h>
#include <pangomm/item.h>
#include <pangomm/fontmetrics.h>
#include <boost/range.hpp>
#include "text_element_char_size_cache.hxx"


namespace dialektos {


namespace text_view {
  class DrawingSet;
  class LayoutSet;
  class GetSelectedSet;
}


/*! @brief namespace for text element */
namespace text_element {


//class Rectangle {
//public:
//  Rectangle(double x, double y, double width, double height):
//    x_(x), y_(y), width_(width), height_(height) {}
//
//  double get_x() const { return x_; }
//  double get_y() const { return y_; }
//  double get_width() const { return width_; }
//  double get_height() const { return height_; }
//  double get_x_right() const { return x_ + width_; }
//  double get_y_bottom() const { return y_ + height_; }
//
//private:
//  double x_;
//  double y_;
//  double width_;
//  double height_;
//};


/*! @brief Text element class for Plain text */
class Plain {
public:

  /*! @brief Constructor
   *
   * @param range is string represented by Boost.Range concepts
   * @param bold is whether the text is bold or not
   */
  template <typename RangeT>
  Plain(const RangeT& range, bool bold) :
    str_(boost::begin(range), boost::end(range)),
    bold_(bold), x_(0), y_(0), x_start_(0), x_end_(0), items_(), metrics_() {}

  virtual ~Plain() {}

  /*! @brief For clone */
  Plain* clone() const { return do_clone(); }

  /*! @brief Remove right side white spaces */
  void trim_right();

  /*! @brief return string
   *
   * @return text
   */
  Glib::ustring get_text() const { return str_; }

  /*! @brief return selected string
   *
   * @return selected text
   */
  Glib::ustring get_selected_text(const text_view::GetSelectedSet& set) const;

  /*! @brief return left-top position of this element.
   *
   * @return pair of x and y, y position is on the adjustment.
   */
  std::pair<gdouble, gdouble> get_xy() const { return std::make_pair(x_, y_); }

  /*! @brief Calculate the position of characters.
   *
   * @param set is information sets for layout.
   */
  void layout(text_view::LayoutSet& set);

  /*! @brief Draw the element.
   *
   * @param set is tool sets for drawing.
   */
  void draw(text_view::DrawingSet& set) const;

  /*! @brief predicate whether (x, y) is on the element.
   *
   * @param x position on the widget.
   * @param y position on the adjustment.
   * @return true if (x, y) on the element, false if not.
   */
  bool is_xy_on_this(gdouble x, gdouble y) const;

  /*! @brief predicate whether (x, y) is near to the element.
   *
   * @param x position on the widget.
   * @param y position on the adjustment.
   * @return true if (x, y) near to the element, false if not.
   *
   * Also returns true if (x, y) is on the spaces caused by character wrapping.
   */
  bool is_xy_near_to_this(gdouble x, gdouble y) const;

  /*! @brief Gets a rectangle of a character under (x, y).
   *
   * @param x on the widget.
   * @param y on the adjustment.
   * @return the rectangle. y is on the adjustment.
   */
  Gdk::Rectangle xy_to_rectangle(gdouble x, gdouble y) const;

  /*! @brief convert a character index from the x,y position.

   * @param x position on the widget.
   * @param y position on the adjustment.
   */
  int xy_to_index(gdouble x, gdouble y) const;

  /*! @brief return a cursor type appropriated to the plain text element.
   *
   * @return the cursor type
   */
  virtual Gdk::CursorType get_cursor_type() const;

  void itemize(const Glib::RefPtr<Pango::Context>&, const Pango::FontMetrics&);

  /*! @brief return an approximate character height.
   *
   * @param layout for calculating the character height.
   * @return the approximate height.
   */
  static double get_approximate_char_height(const Pango::FontMetrics&);

protected:

  virtual void do_draw_glyphs(text_view::DrawingSet&, const Pango::Item&,
      const Pango::GlyphString&, double x, double y,
      bool in_selection) const;

  Pango::FontMetrics get_metrics() const;

private:
  virtual Plain* do_clone() const { return new Plain(*this); }

  void do_draw(text_view::DrawingSet&, const Pango::Item&,
      const Pango::GlyphString&, double x, double y,
      bool in_selection) const;

  void get_selection_start_end_index(const Plain*, size_t, const Plain*, size_t,
      size_t&, size_t&) const;

//  static void initialize_char_width_cache();

  Glib::ustring str_;
  bool bold_;
  gdouble x_;  // top-left x
  gdouble y_;  // top-left y on the adjustment
  gdouble x_start_;
  gdouble x_end_;

  std::vector<Pango::Item> items_;

  Pango::FontMetrics metrics_;

  static CharSizeCache char_size_cache;
};

/*! @brief clone the text element type. This function is for Boost.PtrContainer.
 *
 * @param rhs is copied object.
 * @return new object.
 */
inline Plain* new_clone(const Plain& rhs) { return rhs.clone(); }


} // namespace text_element

} // namespace dialektos

#endif
