/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "text_line.hxx"

#include <boost/foreach.hpp>
#include <boost/lambda/lambda.hpp>
#include <algorithm>
#include "text_view_drawing_set.hxx"
#include "text_element_plain.hxx"


namespace dialektos {


TextLine::TextLine(int res_num, int l_mgn) :
  res_num_(res_num), element_list_(), height_(12), y_(0), width_(0),
  left_margin_(l_mgn), right_margin_(1), metrics_() {
}

TextLine::TextLine(const TextLine& rhs) :
  res_num_(rhs.res_num_), element_list_(rhs.element_list_),
  height_(rhs.height_), y_(rhs.y_), width_(rhs.width_),
  left_margin_(rhs.left_margin_), right_margin_(rhs.right_margin_),
  metrics_(rhs.metrics_) {
}

TextLine::~TextLine(){}

void TextLine::add_element(text_element::Plain* element) {
  element_list_.push_back(element);
}

void TextLine::trim_right() {
  if (element_list_.empty()) return;

  element_list_.back().trim_right();
}

void TextLine::itemize(const Glib::RefPtr<Pango::Context>& context,
    const Pango::FontMetrics& metrics) {
  BOOST_FOREACH(text_element::Plain& element, element_list_)
    element.itemize(context, metrics);
  metrics_ = metrics;
}


bool TextLine::empty() const {
  return element_list_.empty();
}

bool TextLine::includes(const text_element::Plain& right) const {
  using namespace boost::lambda;
  return std::find_if(element_list_.begin(), element_list_.end(),
      &_1 == &right) != element_list_.end();
}

Glib::ustring TextLine::get_selected_text(
    const text_view::GetSelectedSet& set) const {
  Glib::ustring selected;
  BOOST_FOREACH(const text_element::Plain& element, element_list_) {
    selected += element.get_selected_text(set);
  }
  return selected;
}

void TextLine::layout(text_view::LayoutSet& set) {
  width_ = set.x_end - set.x_start;
  y_ = set.y;

  set.x_start += left_margin_;
  set.x_end -= right_margin_;
  set.x = set.x_start;

  BOOST_FOREACH(text_element::Plain& elem, element_list_) {
    elem.layout(set);
  }
  set.y += text_element::Plain::get_approximate_char_height(metrics_);

  height_ = set.y - y_;
}

void TextLine::draw(text_view::DrawingSet& set) const {
  BOOST_FOREACH(const text_element::Plain& elem, element_list_) {
    elem.draw(set);
  }
}

const text_element::Plain* TextLine::get_text_element(
    double x, double y_adj) const {

  BOOST_FOREACH(const text_element::Plain& element, element_list_) {
    if (element.is_xy_on_this(x, y_adj)) return &element;
  }
  return 0;
}

const text_element::Plain* TextLine::get_nearest_text_element(
    double& x, double& y_adj) const {

  if (empty()) return 0;

  x = std::min(double(width_-right_margin_),
      std::max(x, double(left_margin_)));
  y_adj = std::min(y_ + height_, std::max(y_adj, y_));

  BOOST_FOREACH(const text_element::Plain& element, element_list_) {
    if (element.is_xy_near_to_this(x, y_adj)) return &element;
  }

  // the last element is the nearest element if not found.
  return &element_list_.back();
}


} // namespace dialektos
