/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "text_view_layoutable.hxx"

#include <boost/foreach.hpp>
#include "text_view_drawing_set.hxx"
#include "text_line.hxx"
#include "text_element_res_num_map.hxx"
#include "text_element_id_map.hxx"


namespace dialektos {


namespace text_view {


Layoutable::Layoutable():
  Scrollable(), pango_layout_(create_pango_layout("")), line_list_(),
      id_map_(new text_element::IDMap),
      res_num_map_(new text_element::ResNumMap),
      res_num_(0),
      metrics_(get_pango_context()->get_metrics(
          get_pango_context()->get_font_description())),
      width_(-1) {
}

Layoutable::~Layoutable() {
}

void Layoutable::add_line(TextLine* line) {
  line->itemize(get_pango_context(), metrics_);
  line_list_.push_back(line);
}

void Layoutable::relayout() {
  const int width = get_width();
  LayoutSet set;
  set.y = 0;

  BOOST_FOREACH(TextLine& line, line_list_) {
    set.x = 0;
    set.x_start = 0;
    set.x_end = width;

    line.layout(set);
  }

  width_ = width;
  adjustment_.set_upper(set.y);
}

void Layoutable::safe_layout() {
  if (width_ == -1)
    relayout();
  else if (width_ != get_width()) {
    const gdouble value = adjustment_.get_value();
    TextLine* first_line = 0;
    gdouble y = 0;
    BOOST_FOREACH(TextLine& line, line_list_) {
      if (y + line.get_height() > value) {
        first_line = &line;
        break;
      }
      y += line.get_height();
    }
    gdouble delta = y - value;

    relayout();

    y = 0;
    BOOST_FOREACH(TextLine& line, line_list_) {
      if (&line == first_line) {
        break;
      }
      y += line.get_height();
    }
    adjustment_.set_value(y - delta);
  }
}

bool Layoutable::on_configure_event(GdkEventConfigure* event) {
  safe_layout();
  return Scrollable::on_configure_event(event);
}

void Layoutable::on_realize() {
  Scrollable::on_realize();
  metrics_ = get_pango_context()->get_metrics(
      get_pango_context()->get_font_description());
}

void Layoutable::on_style_changed(const Glib::RefPtr<Gtk::Style>& previous) {
  Scrollable::on_style_changed(previous);
  const Pango::FontMetrics metrics = get_pango_context()->get_metrics(
      get_pango_context()->get_font_description());
  BOOST_FOREACH(TextLine& line, line_list_)
    line.itemize(get_pango_context(), metrics);
  safe_layout();
}

const TextLine* Layoutable::get_text_line(const gdouble y_adj) const {

  gdouble sum = 0;
  BOOST_FOREACH(const TextLine& line, line_list_) {
    const gdouble height = line.get_height();
    if (sum + height > y_adj)
      return &line;
    sum += height;
  }
  return 0;
}

const TextLine* Layoutable::get_nearest_text_line(gdouble& y_adj) const {
  if (line_list_.empty()) return 0;

  if (y_adj < 0) {
    y_adj = 0;
    return &line_list_[0];
  }
  if (y_adj > adjustment_.get_upper()) {
    y_adj = adjustment_.get_upper();
    return &line_list_.back();
  }

  return get_text_line(y_adj);
}

const text_element::Plain* Layoutable::get_text_element(const gdouble x,
    const gdouble y_adj) const {

  const TextLine* line = get_text_line(y_adj);
  if (!line) return 0;
  return line->get_text_element(x, y_adj);
}

const text_element::Plain* Layoutable::get_nearest_text_element(
    gdouble& x, gdouble& y_adj) const {

  const TextLine* line = get_nearest_text_line(y_adj);
  if (!line) return 0;
  return line->get_nearest_text_element(x, y_adj);
}


} // namespace text_view

} // namespace dialektos
