/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "text_view_popupable.hxx"

#include <glibmm/ustring.h>
#include <gdkmm/general.h>
#include <gdkmm/rectangle.h>
#include <boost/foreach.hpp>
#include <boost/unordered_set.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/lexical_cast.hpp>
#include "text_line.hxx"
#include "text_element_anchor.hxx"
#include "text_element_id.hxx"
#include "text_element_anchor.hxx"
#include "text_element_res_num.hxx"
#include "text_element_id_map.hxx"
#include "text_element_res_num_map.hxx"
#include "text_view_popup.hxx"


namespace dialektos {

namespace text_view {


Popupable::Popupable(): Drawable(), popup_() {}

Popupable::~Popupable() {}

bool Popupable::on_timeout() {
  if (!popup_)
    return false;

  if (is_pointer_on_the_element()) return true;

  if (is_pointer_on_the_popup()) return true;

  popup_.reset(0);
  return false;
}

bool Popupable::is_pointer_on_the_element() const {
  if (!popup_) return false;

  using text_element::Plain;
  using text_element::Anchor;

  int x_ptr, y_ptr;
  get_pointer(x_ptr, y_ptr);
  y_ptr += adjustment_.get_value();

  const Plain* element = get_text_element(x_ptr, y_ptr);
  if (const Anchor* anchor = dynamic_cast<const Anchor*>(element))
    if (popup_->is_same_origin(*anchor)) return true;

  return false;
}

bool Popupable::is_pointer_on_the_popup() const {
  if (!popup_) return false;

  int x_ptr, y_ptr;
  popup_->get_pointer(x_ptr, y_ptr);

  if (x_ptr >= 0 && x_ptr < popup_->get_width() &&
      y_ptr >= 0 && y_ptr < popup_->get_height())
    return true;

  if (popup_->on_the_child()) return true;

  return false;
}

void Popupable::do_id_hovered_event(const text_element::ID& elem) {
  if (popup_) return;

  Glib::ustring id = elem.get_href();
  const std::vector<int> res_num_list = id_map_->get_res_num_list(id);
  const boost::unordered_set<int>
  res_num_set(res_num_list.begin(), res_num_list.end());

  if (res_num_set.size() < 2) return;

  LineListType new_line;
  build_line_list_for_popup(res_num_set, new_line);

  do_popup(elem, new_line);
}

void Popupable::do_anchor_hovered_event(const text_element::Anchor& elem) {
  if (popup_) return;

  using namespace boost::lambda;

  Glib::ustring text = elem.get_text();
  if (!boost::algorithm::starts_with(text, ">>")) return;

  boost::trim_left_if(text, _1 == '>');

  boost::iterator_range<Glib::ustring::iterator> ret =
    boost::algorithm::find_first(text, "-");

  boost::unordered_set<int> set;

  if (!ret.empty()) {
    // consider like '123-456'
    int left_num = 0;
    int right_num = 0;
    try {
      left_num = boost::lexical_cast<int>(
          Glib::ustring(text.begin(), ret.begin()));
      right_num = boost::lexical_cast<int>(
          Glib::ustring(ret.end(), text.end()));
    } catch (const std::exception&) { return; }
    if (left_num > right_num) std::swap(left_num, right_num);
    for (int i = left_num; i <= right_num; ++i) set.insert(i);
  } else {
    // consider only digit string.
    int res_num = 0;
    try {
      res_num = boost::lexical_cast<int>(text);
    } catch (const std::exception&) { return; }
    set.insert(res_num);
  }

  LineListType new_line;
  build_line_list_for_popup(set, new_line);

  do_popup(elem, new_line);
}

void Popupable::do_res_num_hovered_event(const text_element::ResNum& elem) {
  if (popup_) return;

  int res_num = elem.get_res_num();
  const std::vector<int> res_num_list =
    res_num_map_->get_res_num_list(res_num);
  const boost::unordered_set<int>
  res_num_set(res_num_list.begin(), res_num_list.end());

  if (res_num_set.empty()) return;

  LineListType new_line;
  build_line_list_for_popup(res_num_set, new_line);

  do_popup(elem, new_line);
}

void Popupable::do_popup(const text_element::Anchor& elem,
    const LineListType& line) {

  if (line.empty()) return;

  int _x, _y;
  get_pointer(_x, _y);
  double x = _x;
  double y = _y;
  y += adjustment_.get_value();
  Gdk::Rectangle rect = elem.xy_to_rectangle(x, y);

  //if (popup_) delete popup_;
  popup_.reset(new Popup(line, get_original_line_list(),
      elem, id_map_, res_num_map_));

  int x_org, y_org;
  get_window()->get_origin(x_org, y_org);

  int width, height;
  popup_->get_size(width, height);
  height -= 2;

  x_org += rect.get_x();
  y_org += rect.get_y() - adjustment_.get_value() - height;

  if (x_org + width > Gdk::screen_width()) {
    x_org = Gdk::screen_width() - width;
  }
  if (y_org < 0) {
    y_org += height;
    y_org += rect.get_height();
  }

  popup_->move(x_org, y_org);
  popup_->show_all();

  Glib::signal_timeout().connect(sigc::mem_fun(
      *this, &Popupable::on_timeout), 100);
}

void Popupable::on_anchor_hovered_event(const text_element::Anchor& elem) {
  if (popup_ && popup_->is_same_origin(elem)) return;

  using text_element::ID;
  using text_element::ResNum;

//  if (typeid(elem) == typeid(ID)) {
  if (const ID* id = dynamic_cast<const ID*>(&elem)) {
    do_id_hovered_event(*id);
    return;
  }

//  if (typeid(elem) == typeid(ResNum)) {
  if (const ResNum* res_num = dynamic_cast<const ResNum*>(&elem)) {
    do_res_num_hovered_event(*res_num);
    return;
  }

  do_anchor_hovered_event(elem);
}

const Popupable::LineListType& Popupable::get_original_line_list() const {
  return line_list_;
}

void Popupable::build_line_list_for_popup(
    const boost::unordered_set<int>& res_nums, LineListType& target) const {

  BOOST_FOREACH(const TextLine& line, get_original_line_list()) {
    const int res_num = line.get_res_num();
    boost::unordered_set<int>::const_iterator it = res_nums.find(res_num);
    if (it != res_nums.end()) target.push_back(new TextLine(line));
  }
}


} // namespace text_view

} // namespace diakeltos
