/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */


#include "thread_idx_cache.hxx"

#include <boost/archive/xml_iarchive.hpp>
#include <boost/archive/xml_oarchive.hpp>
#include <boost/archive/binary_iarchive.hpp>
#include <boost/archive/binary_oarchive.hpp>
#include <boost/serialization/vector.hpp>
#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>
#include <boost/xpressive/xpressive.hpp>
#include <boost/unordered_map.hpp>
#include <boost/unordered_set.hpp>
#include <boost/lexical_cast.hpp>
#include <vector>
#include <fstream>
#include <iostream>
#include "thread_idx.hxx"
#include "misc.hxx"


namespace dialektos {


std::vector<ThreadIdxCache> ThreadIdxCache::from_xml(
    const boost::filesystem::path& cache_xml) {
  std::vector<ThreadIdxCache> cache_vector;

  if (boost::filesystem::exists(cache_xml) &&
      boost::filesystem::is_regular_file(cache_xml)) {
    std::ifstream ifs(cache_xml.file_string().c_str());
    try {
      boost::archive::binary_iarchive ia(ifs);
      ia >> cache_vector;
    } catch (const boost::archive::archive_exception& e) {
      // TODO thread safety.
      std::cerr << "ThreadIdxCache::from_xml(): " << e.what() << std::endl;
    }
    ifs.close();
  }

  return cache_vector;
}

void ThreadIdxCache::to_xml(const boost::filesystem::path& cache_xml,
    const std::vector<ThreadIdxCache>& cache) {
  if (cache.empty()) return;

  if (!misc::create_directories(cache_xml.parent_path())) return;
  std::ofstream ofs(cache_xml.file_string().c_str());
  try {
    boost::archive::binary_oarchive oa(ofs);
    oa << cache;
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << "ThreadIdxCache::to_xml(): " << e.what() << std::endl;
  }
  ofs.close();
}

std::vector<ThreadIdxCache> ThreadIdxCache::from_directory(
    const boost::filesystem::path& idx_dir) {

  const boost::filesystem::path cache_xml = idx_dir / ".cache";
  std::vector<ThreadIdxCache> cache_vector = ThreadIdxCache::from_xml(cache_xml);

  boost::unordered_map<ThreadID, ThreadIdxCache> caches;
  BOOST_FOREACH(const ThreadIdxCache& cache, cache_vector) {
    caches.insert(std::make_pair(cache.id_, cache));
  }
  idx_dir_scan(idx_dir, caches);

  cache_vector.clear();
  typedef std::pair<ThreadID, ThreadIdxCache> PairType;
  BOOST_FOREACH(const PairType& cache, caches) {
    cache_vector.push_back(cache.second);
  }

  to_xml(cache_xml, cache_vector);

  return cache_vector;
}

boost::unordered_set<ThreadIdxCache::ThreadID> ThreadIdxCache::get_exist_ids(
    const boost::filesystem::path& idx_dir) {
  using namespace boost::xpressive;
  const sregex regex = (s1=-repeat<9, 10>(_d)) >> ".xml";

  boost::unordered_set<ThreadID> exist_ids;

  if (!boost::filesystem::exists(idx_dir) ||
      !boost::filesystem::is_directory(idx_dir))
    return exist_ids;

  try {
    const boost::filesystem::directory_iterator it_end;
    for (boost::filesystem::directory_iterator it(idx_dir); it != it_end; ++it) {

      const boost::filesystem::path leaf = it->path();
      if (!boost::filesystem::is_regular_file(leaf)) continue;

      const std::string filename = leaf.filename();
      smatch what;
      if (!regex_match(filename, what, regex)) continue;
      const ThreadID id = what[1];
      exist_ids.insert(id);
    }
  } catch(const boost::filesystem::filesystem_error& e) {
    std::cerr << e.what() << std::endl;
  }
  return exist_ids;
}

std::vector<ThreadIdxCache::ThreadID> ThreadIdxCache::get_deleted_ids(
    const boost::unordered_map<ThreadID, ThreadIdxCache>& caches,
    const boost::unordered_set<ThreadID>& exist_ids,
    const std::string& prefix) {
  typedef std::pair<ThreadID, ThreadIdxCache> PairType;
  std::vector<ThreadID> deleted_ids;
  BOOST_FOREACH(const PairType& pair, caches) {
    const std::string& cached = pair.first;
    if (prefix != cached.substr(0, 3)) continue;
    if (exist_ids.find(cached) == exist_ids.end())
      deleted_ids.push_back(cached);
  }
  return deleted_ids;
}

void ThreadIdxCache::remove_deleted_ids(
    boost::unordered_map<ThreadID, ThreadIdxCache>& caches,
    const std::vector<ThreadID>& deleted_ids) {
  BOOST_FOREACH(const ThreadID& id, deleted_ids) {
    std::cout << "removed " << id << std::endl;
    caches.erase(id);
  }
}

void ThreadIdxCache::merge_idx(const boost::filesystem::path& idx_dir,
    const boost::unordered_set<ThreadID>& exist_ids,
    boost::unordered_map<ThreadID, ThreadIdxCache>& caches) {

  BOOST_FOREACH(const ThreadID& id, exist_ids) {

    const boost::filesystem::path leaf = idx_dir / (id + ".xml");
    const std::time_t last_modified = boost::filesystem::last_write_time(leaf);
    boost::unordered_map<ThreadID, ThreadIdxCache>::iterator it =  caches.find(id);
    if (it != caches.end()) {
      const ThreadIdxCache& cache = it->second;
      const std::time_t cache_last_modified =
        boost::lexical_cast<std::time_t>(cache.idx_last_modified_);
      if (last_modified <= cache_last_modified) continue;
    }

//    std::cout << "modified:" << id << std::endl;
    ThreadIdx idx = ThreadIdx::from_xml(leaf);
    if (idx.last_modified_.empty()) continue;
    ThreadIdxCache& cache = caches[id];
    cache.id_ = id;
    cache.idx_last_modified_ = boost::lexical_cast<std::string>(last_modified);
    cache.line_count_ = idx.line_count_;
    cache.title_ = idx.title_;
  }
}

void ThreadIdxCache::idx_dir_scan(const boost::filesystem::path& idx_dir,
    boost::unordered_map<ThreadID, ThreadIdxCache>& caches) {

  std::vector<DirectoryTimeStamp> _directories =
    directory_timestamp_from_xml(idx_dir / "dirs.xml");
  boost::unordered_map<std::string, std::time_t> directories;
  BOOST_FOREACH(const DirectoryTimeStamp& dir, _directories) {
    directories[dir.filename] = dir.last_modified;
  }

  try {
    const boost::filesystem::directory_iterator it_end;
    for (boost::filesystem::directory_iterator it(idx_dir);
        it != it_end; ++it) {

      const boost::filesystem::path sub_dir = it->path();
      if (!boost::filesystem::is_directory(sub_dir)) continue;

      const std::string filename = sub_dir.filename();
      const std::time_t last_modified =
        boost::filesystem::last_write_time(sub_dir);
      if (directories.find(filename) != directories.end() &&
          directories[filename] == last_modified) {
        continue;
      }

      directories[filename] = last_modified;

      const boost::unordered_set<ThreadID> exist_ids = get_exist_ids(sub_dir);
      const std::vector<ThreadID> deleted_ids =
        get_deleted_ids(caches, exist_ids, sub_dir.filename());
      remove_deleted_ids(caches, deleted_ids);
      merge_idx(sub_dir, exist_ids, caches);

    }
  } catch(const boost::filesystem::filesystem_error& e) {
    std::cerr << e.what() << std::endl;
  }

  _directories.clear();
  typedef boost::unordered_map<std::string, std::time_t>::value_type PairType;
  BOOST_FOREACH(const PairType& pair, directories) {
    DirectoryTimeStamp dir;
    dir.filename = pair.first;
    dir.last_modified = pair.second;
    _directories.push_back(dir);
  }
  directory_timestamp_to_xml(idx_dir / "dirs.xml", _directories);
}

std::vector<DirectoryTimeStamp> ThreadIdxCache::directory_timestamp_from_xml(
    const boost::filesystem::path& xml) {
  std::vector<DirectoryTimeStamp> cache_vector;

  if (boost::filesystem::exists(xml) && boost::filesystem::is_regular_file(xml)) {
    std::ifstream ifs(xml.file_string().c_str());
    try {
      boost::archive::xml_iarchive ia(ifs);
      ia >> boost::serialization::make_nvp("DirectoryTimeStamp", cache_vector);
    } catch (const boost::archive::archive_exception& e) {
      std::cerr << "directory_timestamp_from_xml(): " << e.what() << std::endl;
    }
    ifs.close();
  }

  return cache_vector;
}

void ThreadIdxCache::directory_timestamp_to_xml(
    const boost::filesystem::path& xml,
    const std::vector<DirectoryTimeStamp>& cache) {
  if (cache.empty()) return;

  if (!misc::create_directories(xml.parent_path())) return;
  std::ofstream ofs(xml.file_string().c_str());
  try {
    boost::archive::xml_oarchive oa(ofs);
    oa << boost::serialization::make_nvp("DirectoryTimeStamp", cache);
  } catch (const boost::archive::archive_exception& e) {
    std::cerr << "directory_timestamp_to_xml(): " << e.what() << std::endl;
  }
}


} // namespace dialektos
