/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef THREAD_LIST_MODEL_HXX
#define THREAD_LIST_MODEL_HXX

#include <glibmm/object.h>
#include <gtkmm/treemodel.h>
#include <boost/unordered_map.hpp>
#include <boost/mpl/vector.hpp>
#include <boost/mpl/find.hpp>
#include <boost/mpl/size.hpp>
#include <boost/mpl/inherit.hpp>
#include <boost/mpl/inherit_linearly.hpp>
#include <boost/array.hpp>
#include <boost/integer_traits.hpp>
#include <boost/function.hpp>
#include <string>
#include <vector>


namespace dialektos {

namespace model_column {

template <typename T, int ID = 0>
struct TypeHolder {
  typedef T type;
  enum { Identifyer = ID };
};

template <typename TypeHolderT>
struct FieldHolder {
  FieldHolder() : data_() {}
  typedef typename TypeHolderT::type ValueType;
  ValueType data_;
};

template <typename TypeHolderT>
inline typename TypeHolderT::type& field(FieldHolder<TypeHolderT>& t) {
  return t.data_;
}
template <typename TypeHolderT>
inline const typename TypeHolderT::type& field(const FieldHolder<TypeHolderT>& t) {
  return t.data_;
}

typedef TypeHolder<int> Number;
typedef TypeHolder<int, 1> ResNum;
typedef TypeHolder<int, 2> LineCount;
typedef TypeHolder<double> Average;
typedef TypeHolder<std::string> Title;
typedef TypeHolder<std::string, 1> ID;
typedef boost::mpl::vector<ID, Number, Title, ResNum, LineCount, Average> List;

} // namespace model_column


struct ModelColumns : public boost::mpl::inherit_linearly<
  model_column::List,
  boost::mpl::inherit<boost::mpl::_1, model_column::FieldHolder<boost::mpl::_2> >
  >::type {
  typedef model_column::List type;
};

namespace {
template <typename TypeHolderT, typename ValueType> struct DoCompare;
template <typename TypeHolderT, typename ValueType> struct DoCompareReverse;

template <typename TypeHolderT>
struct DoCompare<TypeHolderT, std::string> :
  public std::binary_function<ModelColumns, ModelColumns, bool> {
  bool operator()(const ModelColumns& lhs, const ModelColumns& rhs) const {
    typedef typename TypeHolderT::type ValueType;
    const ValueType& left = model_column::field<TypeHolderT>(lhs);
    const ValueType& right = model_column::field<TypeHolderT>(rhs);

    if (left.empty() && right.empty()) return false;
    if (left.empty()) return false;
    if (right.empty()) return true;

    return left < right;
  }
};

template <typename TypeHolderT>
struct DoCompareReverse<TypeHolderT, std::string> :
  public std::binary_function<ModelColumns, ModelColumns, bool> {
  bool operator()(const ModelColumns& lhs, const ModelColumns& rhs) const {
    typedef typename TypeHolderT::type ValueType;
    const ValueType& left = model_column::field<TypeHolderT>(lhs);
    const ValueType& right = model_column::field<TypeHolderT>(rhs);

    if (left.empty() && right.empty()) return false;
    if (left.empty()) return false;
    if (right.empty()) return true;

    return left > right;
  }
};

template <typename ColumnType>
struct DoCompare<ColumnType, int> {
  bool operator()(const ModelColumns& lhs, const ModelColumns& rhs) const {
    int left_num = model_column::field<ColumnType>(lhs);
    int right_num = model_column::field<ColumnType>(rhs);

    if (left_num == 0) left_num = boost::integer_traits<int>::const_max;
    if (right_num == 0) right_num = boost::integer_traits<int>::const_max;

    return left_num < right_num;
  }
};
template <typename ColumnType>
struct DoCompareReverse<ColumnType, int> {
  bool operator()(const ModelColumns& lhs, const ModelColumns& rhs) const {
    int left_num = model_column::field<ColumnType>(lhs);
    int right_num = model_column::field<ColumnType>(rhs);

    if (left_num == 0) left_num = boost::integer_traits<int>::const_min;
    if (right_num == 0) right_num = boost::integer_traits<int>::const_min;

    return left_num > right_num;
  }
};

template <typename ColumnType>
struct DoCompare<ColumnType, double> {
  bool operator()(const ModelColumns& lhs, const ModelColumns& rhs) const {
    double left_num = model_column::field<ColumnType>(lhs);
    double right_num = model_column::field<ColumnType>(rhs);

    if (left_num == 0) left_num = boost::integer_traits<int>::const_max;
    if (right_num == 0) right_num = boost::integer_traits<int>::const_max;

    return left_num < right_num;
  }
};
template <typename ColumnType>
struct DoCompareReverse<ColumnType, double> {
  bool operator()(const ModelColumns& lhs, const ModelColumns& rhs) const {
    double left_num = model_column::field<ColumnType>(lhs);
    double right_num = model_column::field<ColumnType>(rhs);

    if (left_num == 0) left_num = boost::integer_traits<int>::const_min;
    if (right_num == 0) right_num = boost::integer_traits<int>::const_min;

    return left_num > right_num;
  }
};

template <typename TypeHolderT>
struct Compare : public DoCompare<TypeHolderT, typename TypeHolderT::type> {};

template <typename TypeHolderT>
struct CompareReverse :
  public DoCompareReverse<TypeHolderT, typename TypeHolderT::type> {};

}


class ThreadListModel : public Glib::Object, public Gtk::TreeModel {
public:
  typedef Gtk::TreeModel::iterator iterator;

  static Glib::RefPtr<ThreadListModel> create();

  virtual ~ThreadListModel();
  void append(const ModelColumns& record);
  void set_buffer(const boost::unordered_map<model_column::ID::type, ModelColumns>& buffer);
  void get_buffer(boost::unordered_map<model_column::ID::type, ModelColumns>& buffer) const;
  const ModelColumns& get_model_columns(size_t row_index) const;
  void update_row(const ModelColumns& record);
  template <typename TypeHolderT> void set_sort_function(bool ascendant);
  void sort();
protected:
  ThreadListModel();

  virtual GType get_column_type_vfunc(int index) const;
  virtual Gtk::TreeModelFlags get_flags_vfunc() const;
  virtual int get_n_columns_vfunc() const;
  virtual bool get_iter_vfunc(const Gtk::TreeModel::Path& path, iterator& iter) const;
  virtual Gtk::TreeModel::Path get_path_vfunc(const iterator& iter) const;
  virtual void get_value_vfunc(const iterator& iter, int column,
      Glib::ValueBase& value) const;

  virtual bool iter_children_vfunc(const iterator& parent, iterator& iter) const;
  virtual bool iter_has_child_vfunc(const iterator& iter) const;
  virtual bool iter_is_valid(const iterator& iter) const;
  virtual int iter_n_children_vfunc(const iterator& iter) const;
  virtual int iter_n_root_children_vfunc() const;
  virtual bool iter_next_vfunc(const iterator& iter, iterator& iter_next) const;
  virtual bool iter_nth_child_vfunc(const iterator& parent, int n,
      iterator& iter) const;
  virtual bool iter_nth_root_child_vfunc(int n, iterator& iter) const;
  virtual bool iter_parent_vfunc(const iterator& child, iterator& iter) const;

private:
  void do_after_sort();

  typedef std::vector<ModelColumns> StoreType;
  typedef boost::unordered_map<model_column::ID::type, size_t> OrderType;

  StoreType list_;
  OrderType order_;

  boost::function<bool (const ModelColumns&, const ModelColumns&)>
    sort_function_;
};

template <typename TypeHolderT>
void ThreadListModel::set_sort_function(bool ascendant) {
  if (ascendant) sort_function_ = Compare<TypeHolderT>();
  else sort_function_ = CompareReverse<TypeHolderT>();
}


} // namespace dialektos

#endif
