/*
 * Copyright (C) 2009 by Aiwota Programmer
 * aiwotaprog@tetteke.tk
 *
 * This file is part of Dialektos.
 *
 * Dialektos is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dialektos is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dialektos.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "thread_window.hxx"

#include <glibmm/convert.h>
#include <gtkmm/stock.h>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>
#include <boost/format.hpp>
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include "bbs_detail_base.hxx"
#include "thread_idx.hxx"
#include "http_get.hxx"
#include "misc.hxx"
#include "uri_opener.hxx"
#include "thread_window_state.hxx"
#include "board_window.hxx"


namespace dialektos {


void ThreadWindow::create(std::auto_ptr<bbs_detail::Base> bbs) {
  regist(new ThreadWindow(bbs));
}

ThreadWindow::ThreadWindow(std::auto_ptr<bbs_detail::Base> bbs) :
  SuperClass(),
  text_view_(new text_view::TextView),
  scrolled_(text_view_->get_adjustment()),
  bbs_(bbs), http_getter_(), idx_() {

  // additional menuitems for board window
  action_group_->add(Gtk::Action::create("FileDelete", Gtk::Stock::DELETE),
      sigc::mem_fun(*this, &ThreadWindow::on_action_file_delete));
  action_group_->add(
      Gtk::Action::create("FileBoard", Gtk::Stock::GO_UP, "Show _Board"),
      sigc::mem_fun(*this, &ThreadWindow::on_action_file_board));
  action_group_->add(
      Gtk::Action::create(
          "FileCloseRemoveHistory", Gtk::Stock::CLEAR, "CloseH",
          "Close and Remove from History"),
      sigc::mem_fun(*this,
          &ThreadWindow::on_action_file_close_remove_history));

  Glib::ustring ui =
    "<ui>"
    "  <menubar name='MenuBar'>"
    "    <menu action='MenuFile'>"
    "      <menuitem action='FileDelete'/>"
    "      <menuitem action='FileCloseRemoveHistory'/>"
    "      <separator/>"
    "      <menuitem action='FileBoard'/>"
    "    </menu>"
    "  </menubar>"
    "  <toolbar name='ToolBar'>"
    "    <toolitem action='FileCloseRemoveHistory'/>"
    "  </toolbar>"
    "</ui>";

  ui_manager_->add_ui_from_string(ui);

  add(scrolled_);
  initialize_text_view();

  ThreadWindowState state;
  state.from_xml(boost::filesystem::path(bbs_->get_thread_state_path()));
  set_default_size(state.width, state.height);
  if (state.menubar) menubar_->show();
  if (state.toolbar) toolbar_->show();
  if (state.statusbar) statusbar_.show();

  show_all();

  load();
  text_view_->jump_to_res_num(state.displayed_res_num);
}

void ThreadWindow::initialize_text_view() {
  scrolled_.add(*text_view_);

  text_view_->signal_button_press_event().connect_notify(
      sigc::mem_fun(*this, &ThreadWindow::on_child_button_press));
  text_view_->signal_uri_clicked().connect(
      sigc::mem_fun(*this, &ThreadWindow::on_uri_clicked));

  text_view_->show();
}

bool ThreadWindow::load() {
  using namespace boost::posix_time;
  ptime start = microsec_clock::local_time();

  if (bbs_) {

    bbs_->load_thread(*text_view_);
    text_view_->relayout();
    text_view_->queue_draw();

    std::cout <<
    to_simple_string(microsec_clock::local_time() - start) << std::endl;

    idx_ = ThreadIdx::from_xml(bbs_->get_thread_idx_path());
    set_title(idx_.title_);
    histories.push(bbs_->get_thread_uri(), idx_.title_);

    if (text_view_->get_res_num() == 0) on_action_view_refresh();
  }

  return false;
}

bool ThreadWindow::is_same(const bbs_detail::Base& bbs) const {
  const bbs_detail::Base& lhs = *bbs_;
  return lhs == bbs;
}

std::string ThreadWindow::get_uri() const {
  return bbs_->get_thread_uri();
}


void ThreadWindow::on_action_view_refresh() {
  if (http_getter_) return;

  statusbar_.push("HTTP/1.0 GET...");

  const std::string uri = bbs_->get_thread_dat_uri();
  http::Header request_header = bbs_->get_thread_dat_request_header();
  if (!request_header.get_range().empty()) {
    request_header.set_if_modified_since(idx_.last_modified_);
    request_header.set_if_none_match(idx_.etag_);
  }

  http_getter_.reset(new http::GetInThread(uri, request_header));
  http_getter_->signal_end().connect(
      sigc::mem_fun(*this, &ThreadWindow::on_http_get_end));
  http_getter_->run();
}

void ThreadWindow::on_action_view_stop() {
  if (http_getter_) http_getter_->cancel();
}

void ThreadWindow::on_action_edit_copy() {
  const Glib::ustring selected = text_view_->get_selected_text();
  Gtk::Clipboard::get()->set_text(selected);
}

void ThreadWindow::on_action_file_delete() {
  const boost::filesystem::path dat(bbs_->get_thread_file_path());
  try {
    if (boost::filesystem::exists(dat)) boost::filesystem::remove(dat);
  } catch (const boost::filesystem::filesystem_error& e) {
    std::cerr << e.what() << std::endl;
  }

  const boost::filesystem::path xml(bbs_->get_thread_idx_path());
  try {
    if (boost::filesystem::exists(xml)) boost::filesystem::remove(xml);
  } catch (const boost::filesystem::filesystem_error& e) {
    std::cerr << e.what() << std::endl;
  }
}

void ThreadWindow::on_action_file_board() {
  uri_opener::open(bbs_->get_board_uri());
}

void ThreadWindow::on_action_file_close_remove_history() {
  histories.erase(bbs_->get_thread_uri());

  std::vector<ApplicationWindow*> closes;
  closes.push_back(this);
  close_windows(closes);
}

void ThreadWindow::on_http_get_end(bool success) {
//  const std::string uri = http_getter_->get_uri();
//  const http::Header request_header = http_getter_->get_request_header();
  const http::Response response = http_getter_->get_response();
  const boost::system::error_code err = http_getter_->get_error();
  http_getter_.reset(0);
  if (err) {
    statusbar_.push(err.message());
    return;
  }
  if (!success) {
    statusbar_.push("Canceled.");
    return;
  }

  on_refresh_end(response.get_status_line(), response.get_header());

  const int code = response.get_status_line().get_code();
  if (code != 200 && code != 206) return;

  if (code == 200) {
    // initialize text_view_
    scrolled_.remove(*text_view_);
    text_view_.reset(new text_view::TextView);
    scrolled_.set_adjustment(text_view_->get_adjustment());

    initialize_text_view();
  }

  bbs_->load_thread_from_string(response.get_content(), *text_view_);
  text_view_->relayout();
  text_view_->queue_draw();

  save_content(response);
  inform_to_board(idx_);
}

void ThreadWindow::on_refresh_end(const http::StatusLine& status,
    const http::Header& header) {
  std::string message = status.get_line();
  const std::string last_modified = header.get_last_modified();
  if (!last_modified.empty()) {
    message += " ";
    message += last_modified;
  }
  statusbar_.push(message);
}

void ThreadWindow::save_content(const http::Response& response) {
  const int code = response.get_status_line().get_code();
  if (code != 200 && code != 206) return;

  if (!misc::create_directories(boost::filesystem::path(
      bbs_->get_thread_file_path()).parent_path())) return;
  std::ofstream ofs(bbs_->get_thread_file_path().c_str(),
      code == 200 ?
          std::ios::out | std::ios::trunc : std::ios::out | std::ios::app);
  ofs << response.get_content();
  ofs.close();

  if (code == 200) {
    idx_.title_ = bbs_->get_title_from_string(response.get_content());
    set_title(idx_.title_);
    histories.push(bbs_->get_thread_uri(), idx_.title_);
  }

  idx_.last_modified_ = response.get_header().get_last_modified();
  idx_.etag_ = response.get_header().get_etag();

  idx_.line_count_ = text_view_->get_res_num();

  idx_.to_xml(boost::filesystem::path(bbs_->get_thread_idx_path()));
}

void ThreadWindow::save_state() const {
  ThreadWindowState state;
  state.width = get_width();
  state.height = get_height();
  state.menubar = menubar_->is_visible();
  state.toolbar = toolbar_->is_visible();
  state.statusbar = statusbar_.is_visible();
  state.displayed_res_num = text_view_->get_displayed_res_num();
  state.to_xml(boost::filesystem::path(bbs_->get_thread_state_path()));
}

void ThreadWindow::on_informed_from_board(const int res_num) {
  if (res_num == 0) {
    statusbar_.push("Dropped.");
  } else if (res_num > idx_.line_count_) {
    std::stringstream ss;
    ss << boost::format("Update. %1% -> %2%") % idx_.line_count_ % res_num;
    statusbar_.push(ss.str());
  }
}


void ThreadWindow::inform_to_board(const ThreadIdx& idx) const {
  BOOST_FOREACH(ApplicationWindow& window, windows) {
    if (BoardWindow* board_window = dynamic_cast<BoardWindow*>(&window)) {
      board_window->on_informed_from(*bbs_, idx);
    }
  }
}

void ThreadWindow::on_uri_clicked(const std::string& uri) const {
  if (uri_opener::open(uri)) return;
  std::stringstream ss;
  ss << boost::format("firefox %1%") % uri;
  std::system(ss.str().c_str());
}

} // namespace dialektos
