#include "foldermodel.h"
#include "folderview_private.h"

#include <QDebug>
#include <QDrag>
#include <QDragEnterEvent>
#include <QEvent>
#include <QMenu>
#include <QMimeData>

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::FolderView_Private
/// \param parent   親オブジェクト
///
/// コンストラクタ
///
FolderView_Private::FolderView_Private(QAbstractItemView *parent) :
    QObject(parent),
    m_view(parent),
    m_dragStartPos(),
    m_dragging(false),
    m_saveRow(0),
    m_savePath(),
    m_saveName()
{
    m_view->setDragEnabled(true);
    m_view->setAcceptDrops(true);
    m_view->setDropIndicatorShown(true);
    m_view->setContextMenuPolicy(Qt::DefaultContextMenu);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::onDragEnter
/// \param obj  イベントが発生したオブジェクト
/// \param e    ドラッグエンターイベントオブジェクト
/// \return 処理した場合はtrueを返します。
///
bool FolderView_Private::onDragEnter(QObject *obj, QDragEnterEvent *e)
{
    Q_UNUSED(obj);

    if (e->mimeData()->hasUrls()) {
        e->acceptProposedAction();
        return true;
    }
    return false;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::onDrop
/// \param obj  イベントが発生したオブジェクト
/// \param e    ドロップイベントオブジェクト
/// \return 処理した場合はtrueを返します。
///
bool FolderView_Private::onDrop(QObject *obj, QDropEvent *e)
{
    Q_UNUSED(obj);

    if (m_dragging) {
        // 自分自身へのドロップなら何もしない
        return true;
    }

    QStringList list;
    foreach (const QUrl &url, e->mimeData()->urls()) {
        QString path = QFileInfo(url.toLocalFile()).canonicalFilePath();
        if (!path.isEmpty()) {
            list << path;
        }
        else {
            qDebug() << "drop path is empty." << url;
        }
    }

    if (list.isEmpty()) {
        // ローカルファイルのドロップでなければ何もしない
        return true;
    }

    e->acceptProposedAction();

    QMenu menu(m_view);
    QAction *actCopy = menu.addAction(tr("コピー"));
    QAction *actMove = menu.addAction(tr("移動"));
    menu.addSeparator();
    menu.addAction(tr("キャンセル"));

    QAction *selected = menu.exec(m_view->mapToGlobal(e->pos()));
    FolderModel *model = static_cast<FolderModel*>(m_view->model());
    if (selected == actMove) {
        emit moveItems(list, model->rootPath());
    }
    else if (selected == actCopy) {
        emit copyItems(list, model->rootPath());
    }
    return true;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::onMouseMove
/// \param obj  イベントが発生したオブジェクト
/// \param e    マウスイベントオブジェクト
/// \return 処理した場合はtrueを返します。
///
bool FolderView_Private::onMouseMove(QObject *obj, QMouseEvent *e)
{
    Q_UNUSED(obj);

    if (!(e->buttons() & Qt::LeftButton) && !(e->buttons() & Qt::RightButton))
    {
        return false;
    }

    if (m_dragStartPos.isNull() ||
        (e->pos() - m_dragStartPos).manhattanLength() < 5)
    {
        return false;
    }

    FolderModel *model = static_cast<FolderModel*>(m_view->model());
    QFileInfoList list = model->markedItems();
    if (list.isEmpty()) {
        list << model->fileInfo(m_view->currentIndex());
    }
    if (list.size() == 1 && list[0].fileName() == "..") {
        return false;
    }

    QList<QUrl> urls;
    foreach (const QFileInfo &info, list) {
        urls << QUrl::fromLocalFile(info.absoluteFilePath());
    }

    QDrag *drag = new QDrag(this);
    QMimeData *mimeData = new QMimeData;
    mimeData->setUrls(urls);
    drag->setMimeData(mimeData);

    m_dragging = true;
    drag->exec(Qt::CopyAction);
    m_dragging = false;
    m_dragStartPos = QPoint();

    return true;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::onMousePress
/// \param obj  イベントが発生したオブジェクト
/// \param e    マウスイベントオブジェクト
/// \return 処理した場合はtrueを返します。
///
bool FolderView_Private::onMousePress(QObject *obj, QMouseEvent *e)
{
    Q_UNUSED(obj);

    m_dragStartPos = QPoint();
    if ((e->buttons() & Qt::LeftButton) || (e->buttons() & Qt::RightButton))
    {
        FolderModel *model = static_cast<FolderModel*>(m_view->model());
        QModelIndex index = m_view->indexAt(e->pos());
        if (index.isValid() && model->fileName(index) != "..") {
            m_dragStartPos = e->pos();
        }
    }

    // 本来のイベント処理も行う
    return false;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::model_PostReset
///
/// モデルリセット後の処理
///
void FolderView_Private::model_PostReset()
{
    FolderModel *model = static_cast<FolderModel*>(sender());

    // フォルダが変わった場合は先頭行を選択する
    if (m_savePath != model->rootPath()) {
        m_view->setCurrentIndex(model->index(0, 0));
        return;
    }

    // 保存した名前と同名アイテムが見つかれば、再選択する
    for (int row = 0; row < model->rowCount(); row++) {
        QModelIndex index = model->index(row, 0);
        if (model->fileName(index) == m_saveName) {
            m_view->setCurrentIndex(index);
            return;
        }
    }

    // 同名アイテムが見つからなければ、行を維持する
    if (m_saveRow >= model->rowCount()) {
        m_saveRow = model->rowCount() - 1;
    }
    m_view->setCurrentIndex(model->index(m_saveRow, 1));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::model_PreReset
///
/// モデルリセット前の処理
///
void FolderView_Private::model_PreReset()
{
    // 現在行と名前を保存しておく
    if (m_view->model() && m_view->currentIndex().isValid()) {
        FolderModel *model = static_cast<FolderModel*>(m_view->model());
        m_saveRow = m_view->currentIndex().row();
        m_saveName = model->fileName(m_view->currentIndex());
        m_savePath = model->rootPath();
    }
    else {
        m_saveRow = 0;
        m_saveName = QString::null;
        m_savePath = QString::null;
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderView_Private::eventFilter
/// \param obj  イベントが発生したオブジェクト
/// \param e    イベントオブジェクト
/// \return 処理した場合はtrueを返します。
///
bool FolderView_Private::eventFilter(QObject *obj, QEvent *e)
{
    if (e->type() == QEvent::MouseButtonPress) {
        return onMousePress(obj, static_cast<QMouseEvent*>(e));
    }
    else if (e->type() == QEvent::MouseMove) {
        return onMouseMove(obj, static_cast<QMouseEvent*>(e));
    }
    else if (e->type() == QEvent::DragEnter) {
        return onDragEnter(obj, static_cast<QDragEnterEvent*>(e));
    }
    else if (e->type() == QEvent::Drop) {
        return onDrop(obj, static_cast<QDropEvent*>(e));
    }

    return false;
}
