#include "global.h"

///////////////////////////////////////////////////////////////////////////////
/// \brief appendActionShortcut
/// \param action   アクションオブジェクト
/// \param ks       キーシーケンス文字列
///
/// アクションにショートカットキーを追加します。
///
void appendActionShortcut(QAction *action, const QString &ks)
{
    QList<QKeySequence> shortcuts = action->shortcuts();
    shortcuts << QKeySequence(ks);
    action->setShortcuts(shortcuts);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief detectCode
/// \param bytes    バイト列
/// \return エンコード文字列を返します。
///
std::string detectCode(const QByteArray &bytes)
{
    const quint8 bEscape = 0x1B;
    const quint8 bAt = 0x40;
    const quint8 bDollar = 0x24;
    const quint8 bAnd = 0x26;
    const quint8 bOpen = 0x28;    //'('
    const quint8 bB = 0x42;
    const quint8 bD = 0x44;
    const quint8 bJ = 0x4A;
    const quint8 bI = 0x49;

    int len = bytes.size();
    quint8 b1, b2, b3, b4;

    for (int i = 0; i < len; i++) {
        b1 = bytes[i];
        if (b1 <= 0x06 || b1 == 0x7F || b1 == 0xFF) {
            //'binary'
            return "Binary";
        }
    }

    bool notJapanese = true;
    for (int i = 0; i < len; i++) {
        b1 = bytes[i];
        if (b1 == bEscape || 0x80 <= b1) {
            notJapanese = false;
            break;
        }
    }
    if (notJapanese) {
        return "UTF-8";
    }

    for (int i = 0; i < len - 2; i++) {
        b1 = bytes[i];
        b2 = bytes[i + 1];
        b3 = bytes[i + 2];

        if (b1 == bEscape){
            if ((b2 == bDollar && b3 == bAt) ||
                (b2 == bDollar && b3 == bB) ||
                (b2 == bOpen && (b3 == bB || b3 == bJ)) ||
                (b2 == bOpen && b3 == bI))
            {
                return "ISO 2022-JP";
            }
            if (i < len - 3) {
                b4 = bytes[i + 3];
                if (b2 == bDollar && b3 == bOpen && b4 == bD) {
                    return "ISO 2022-JP";
                }
                if (i < len - 5 &&
                    b2 == bAnd && b3 == bAt && b4 == bEscape &&
                    bytes[i + 4] == bDollar && bytes[i + 5] == bB)
                {
                    return "ISO 2022-JP";
                }
            }
        }
    }

    int sjis = 0;
    int euc = 0;
    int utf8 = 0;
    for (int i = 0; i < len - 1; i++) {
        b1 = bytes[i];
        b2 = bytes[i + 1];
        if (((0x81 <= b1 && b1 <= 0x9F) || (0xE0 <= b1 && b1 <= 0xFC)) &&
            ((0x40 <= b2 && b2 <= 0x7E) || (0x80 <= b2 && b2 <= 0xFC)))
        {
            sjis += 2;
            i++;
        }
    }
    for (int i = 0; i < len - 1; i++) {
        b1 = bytes[i];
        b2 = bytes[i + 1];
        if (((0xA1 <= b1 && b1 <= 0xFE) && (0xA1 <= b2 && b2 <= 0xFE)) ||
            (b1 == 0x8E && (0xA1 <= b2 && b2 <= 0xDF)))
        {
            euc += 2;
            i++;
        }
        else if (i < len - 2) {
            b3 = bytes[i + 2];
            if (b1 == 0x8F && (0xA1 <= b2 && b2 <= 0xFE) &&
                (0xA1 <= b3 && b3 <= 0xFE))
            {
                euc += 3;
                i += 2;
            }
        }
    }
    for (int i = 0; i < len - 1; i++) {
        b1 = bytes[i];
        b2 = bytes[i + 1];
        if ((0xC0 <= b1 && b1 <= 0xDF) && (0x80 <= b2 && b2 <= 0xBF)) {
            utf8 += 2;
            i++;
        }
        else if (i < len - 2) {
            b3 = bytes[i + 2];
            if ((0xE0 <= b1 && b1 <= 0xEF) && (0x80 <= b2 && b2 <= 0xBF) &&
                (0x80 <= b3 && b3 <= 0xBF))
            {
                utf8 += 3;
                i += 2;
            }
        }
    }

    if (euc > sjis && euc > utf8) {
        return "EUC-JP";
    }
    else if (sjis > euc && sjis > utf8) {
        return "Shift-JIS";
    }
    else if (utf8 > euc && utf8 > sjis) {
        return "UTF-8";
    }

#ifdef Q_OS_WIN
    return "Shift-JIS";
#else
    return "UTF-8";
#endif

}

///////////////////////////////////////////////////////////////////////////////
/// \brief fileSizeToString
/// \param size サイズ
/// \return サイズの文字列表記を返します。
///
QString fileSizeToString(qint64 size)
{
    if (size >= 1024 * 1024 * 1024) {
        return QString("%1GB").arg(int(10 * size / (1024 * 1024 * 1024)) / 10);
    }
    if (size >= 1024 * 1024) {
        return QString("%1MB").arg(int(10 * size / (1024 * 1024)) / 10);
    }
    if (size >= 1024) {
        return QString("%1KB").arg(int(10 * size / (1024)) / 10);
    }
    return QString("%1B").arg(size);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief reconnectAction
/// \param sender   送信元アクション
/// \param signal   シグナル
/// \param reciever 送信先オブジェクト
/// \param slot     スロット
///
/// シグナルにスロットを再接続します。
///
void reconnectAction(QAction *sender, const char *signal, QObject *reciever, const char *slot)
{
    sender->setEnabled(true);
    sender->disconnect();
    QObject::connect(sender, signal, reciever, slot);
}
