#include "operationworker.h"
#include "operationdialog.h"
#include "ui_operationdialog.h"

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::OperationDialog
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
OperationDialog::OperationDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::OperationDialog),
    m_worker(NULL),
    m_Error(false)
{
    ui->setupUi(this);
    ui->textEdit->setVisible(false);
    resize(parent->width() * 0.8, height());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::~OperationDialog
///
/// デストラクタ
///
OperationDialog::~OperationDialog()
{
    delete ui;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::setAutoClose
/// \param yes  処理完了時に自動で閉じる場合はtrue
///
void OperationDialog::setAutoClose(bool yes)
{
    ui->chkAutoClose->setChecked(yes);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::autoClose
/// \return 処理完了時に自動で閉じる場合はtrue
///
bool OperationDialog::autoClose() const
{
    return ui->chkAutoClose->isChecked();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::onProgress
/// \param msg  表示するメッセージ
///
/// 処理経過メッセージを表示します。
///
void OperationDialog::onProgress(const QString &msg)
{
    ui->textEdit->append(msg + " ... ");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::onSuccess
/// \param msg  表示するメッセージ
///
/// 処理成功メッセージを表示します。
///
void OperationDialog::onSuccess(const QString &msg)
{
    ui->textEdit->append("<font color='green'>" + msg + "</font>");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::onError
/// \param msg  表示するメッセージ
///
/// エラーメッセージを表示します。
///
void OperationDialog::onError(const QString &msg)
{
    m_Error = true;
    ui->textEdit->append("<font color='red'><strong>" + msg + "</strong></font>");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::onFinished
/// \param abort    中断した場合はtrue
///
/// 処理終了時の処理を行います。
///
void OperationDialog::onFinished(bool abort)
{
    ui->progressBar->setMaximum(1);
    ui->progressBar->setValue(1);
    ui->btnCloseCancel->setText(tr("閉じる"));

    ui->textEdit->append("");
    if (abort) {
        ui->textEdit->append(tr("操作は途中でキャンセルされました。"));
    }
    else {
        ui->textEdit->append(tr("完了"));
        if (!m_Error && ui->chkAutoClose->checkState() == Qt::Checked) {
            QDialog::accept();
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::on_btnCloseCancel_clicked
///
/// 「閉じる」または「キャンセル」ボタンクリック時の処理を行います。
///
void OperationDialog::on_btnCloseCancel_clicked()
{
    if (ui->btnCloseCancel->text() == tr("閉じる")) {
        QDialog::accept();
    }
    else {
        m_worker->abort();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::on_btnShowDetail_clicked
///
/// 詳細表示／非表示の処理を行います。
///
void OperationDialog::on_btnShowDetail_clicked()
{
    if (ui->textEdit->isVisible()) {
        ui->textEdit->setVisible(false);
        ui->btnShowDetail->setText(tr("詳細を表示"));
    }
    else {
        ui->textEdit->setVisible(true);
        ui->btnShowDetail->setText(tr("詳細を隠す"));
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief OperationDialog::exec
/// \return ダイアログ処理の結果を返します。
///
/// ダイアログを表示します。
///
int OperationDialog::exec()
{
    connect(m_worker, SIGNAL(finished(bool)), this, SLOT(onFinished(bool)));
    connect(m_worker, SIGNAL(error(QString)), this, SLOT(onError(QString)));
    connect(m_worker, SIGNAL(progress(QString)), this, SLOT(onProgress(QString)));
    connect(m_worker, SIGNAL(success(QString)), this, SLOT(onSuccess(QString)));
    connect(m_worker, SIGNAL(information(QString)), ui->label, SLOT(setText(QString)));

    ui->label->setText(m_worker->initialText());
    m_worker->start();
    return QDialog::exec();
}
