#include "thumbnailworker.h"

#include <QDebug>
#include <QTimer>
#include <QThread>

const int SLEEP_TIME = 100;

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailWorker::ThumbnailWorker
///
/// コンストラクタ
///
ThumbnailWorker::ThumbnailWorker() :
    AbstractWorker(),
    m_pathList()
{
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailWorker::addPath
/// \param path パス
///
/// パスを追加します。
///
void ThumbnailWorker::addPath(const QString &path)
{
    QMutexLocker locker(&m_mutex);
    m_pathList << path;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailWorker::clearPath
///
/// パスをクリアします。
///
void ThumbnailWorker::clearPath()
{
    QMutexLocker locker(&m_mutex);
    m_pathList.clear();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailWorker::getPath
/// \return リストからパスを取得します。
///
QString ThumbnailWorker::getPath()
{
    QMutexLocker locker(&m_mutex);
    if (m_pathList.isEmpty()) {
        return QString();
    }
    return m_pathList.takeFirst();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ThumbnailWorker::run
///
/// サムネイルを生成します。
///
void ThumbnailWorker::run()
{
    qDebug() << "ThumbnailWorker::run() enter.";

    while (!isAborted()) {
        QString path = getPath();
        if (!path.isEmpty()) {
            QPixmap pixmap(path);
            if (!pixmap.isNull()) {
                QSize size(256, 256);
                double scaleX = 1.0 * size.width() / pixmap.width();
                double scaleY = 1.0 * size.height() / pixmap.height();
                double scaleFactor = qMin(scaleX, scaleY);
                if (scaleFactor < 1) {
                    pixmap = pixmap.scaled(pixmap.size() * scaleFactor,
                                           Qt::IgnoreAspectRatio,
                                           Qt::SmoothTransformation);
                }
                emit resultReady(path, pixmap);
            }
        }
        this->thread()->msleep(SLEEP_TIME);
    }

    emit finished();
    qDebug() << "ThumbnailWorker::run() finished.";
}
