/*
 * Decompiled with CFR 0.152.
 */
package org.h2.store;

import java.io.IOException;
import java.io.OutputStream;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import java.sql.SQLException;
import java.util.Properties;
import org.h2.constant.SysProperties;
import org.h2.message.Message;
import org.h2.message.Trace;
import org.h2.message.TraceSystem;
import org.h2.store.fs.FileSystem;
import org.h2.util.ByteUtils;
import org.h2.util.FileUtils;
import org.h2.util.NetUtils;
import org.h2.util.RandomUtils;
import org.h2.util.SortedProperties;

public class FileLock {
    public static final int LOCK_NO = 0;
    public static final int LOCK_FILE = 1;
    public static final int LOCK_SOCKET = 2;
    private static final String MAGIC = "FileLock";
    private static final String FILE = "file";
    private static final String SOCKET = "socket";
    private static final int RANDOM_BYTES = 16;
    private static final int SLEEP_GAP = 25;
    private static final int TIME_GRANULARITY = 2000;
    private String method;
    private String ipAddress;
    private int sleep;
    private long lastWrite;
    private Properties properties;
    private FileSystem fs;
    private volatile String fileName;
    private volatile ServerSocket socket;
    private boolean locked;
    private Trace trace;

    public FileLock(TraceSystem traceSystem, int sleep) {
        this.trace = traceSystem.getTrace("fileLock");
        this.sleep = sleep;
    }

    public synchronized void lock(String fileName, boolean allowSocket) throws SQLException {
        this.fs = FileSystem.getInstance(fileName);
        this.fileName = fileName;
        if (this.locked) {
            throw Message.getInternalError("already locked");
        }
        if (allowSocket) {
            this.lockSocket();
        } else {
            this.lockFile();
        }
        this.locked = true;
    }

    protected void finalize() {
        if (!SysProperties.runFinalize) {
            return;
        }
        if (this.locked) {
            this.unlock();
        }
    }

    public synchronized void unlock() {
        if (!this.locked) {
            return;
        }
        try {
            if (this.fileName != null && this.load().equals(this.properties)) {
                this.fs.delete(this.fileName);
            }
            if (this.socket != null) {
                this.socket.close();
            }
        }
        catch (Exception e) {
            this.trace.debug("unlock", e);
        }
        this.fileName = null;
        this.socket = null;
        this.locked = false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void save() throws SQLException {
        try {
            OutputStream out = this.fs.openFileOutputStream(this.fileName, false);
            try {
                this.properties.setProperty("method", String.valueOf(this.method));
                this.properties.store(out, MAGIC);
            }
            finally {
                out.close();
            }
            this.lastWrite = this.fs.getLastModified(this.fileName);
            this.trace.debug("save " + this.properties);
        }
        catch (IOException e) {
            throw this.getException(e);
        }
    }

    private Properties load() throws SQLException {
        try {
            Properties p2 = FileUtils.loadProperties(this.fileName);
            this.trace.debug("load " + p2);
            return p2;
        }
        catch (IOException e) {
            throw this.getException(e);
        }
    }

    private void waitUntilOld() throws SQLException {
        for (int i = 0; i < 10; ++i) {
            long last = this.fs.getLastModified(this.fileName);
            long dist = System.currentTimeMillis() - last;
            if (dist < -2000L) {
                throw this.error("Lock file modified in the future: dist=" + dist);
            }
            if (dist < 25L) {
                try {
                    Thread.sleep(dist + 1L);
                }
                catch (Exception e) {
                    this.trace.debug("sleep", e);
                }
                continue;
            }
            return;
        }
        throw this.error("Lock file recently modified");
    }

    private void lockFile() throws SQLException {
        this.method = FILE;
        this.properties = new SortedProperties();
        byte[] bytes = RandomUtils.getSecureBytes(16);
        String random = ByteUtils.convertBytesToString(bytes);
        this.properties.setProperty("id", Long.toHexString(System.currentTimeMillis()) + random);
        if (!this.fs.createNewFile(this.fileName)) {
            this.waitUntilOld();
            String m2 = this.load().getProperty("method", FILE);
            if (!m2.equals(FILE)) {
                throw this.error("Unsupported lock method " + m2);
            }
            this.save();
            this.sleep(2 * this.sleep);
            if (!this.load().equals(this.properties)) {
                throw this.error("Locked by another process");
            }
            this.fs.delete(this.fileName);
            if (!this.fs.createNewFile(this.fileName)) {
                throw this.error("Another process was faster");
            }
        }
        this.save();
        this.sleep(25);
        if (!this.load().equals(this.properties)) {
            this.fileName = null;
            throw this.error("Concurrent update");
        }
        Thread watchdog = new Thread(new Runnable(){

            public void run() {
                try {
                    while (FileLock.this.fileName != null) {
                        try {
                            if (!FileLock.this.fs.exists(FileLock.this.fileName) || FileLock.this.fs.getLastModified(FileLock.this.fileName) != FileLock.this.lastWrite) {
                                FileLock.this.save();
                            }
                            Thread.sleep(FileLock.this.sleep);
                        }
                        catch (Exception e) {
                            FileLock.this.trace.debug("watchdog", e);
                        }
                    }
                }
                catch (Exception e) {
                    FileLock.this.trace.debug("watchdog", e);
                }
                FileLock.this.trace.debug("watchdog end");
            }
        });
        watchdog.setName("H2 File Lock Watchdog " + this.fileName);
        watchdog.setDaemon(true);
        watchdog.setPriority(9);
        watchdog.start();
    }

    private void lockSocket() throws SQLException {
        this.method = SOCKET;
        this.properties = new SortedProperties();
        try {
            this.ipAddress = InetAddress.getLocalHost().getHostAddress();
        }
        catch (UnknownHostException e) {
            throw this.getException(e);
        }
        if (!this.fs.createNewFile(this.fileName)) {
            InetAddress address;
            this.waitUntilOld();
            long read = this.fs.getLastModified(this.fileName);
            Properties p2 = this.load();
            String m2 = p2.getProperty("method", SOCKET);
            if (m2.equals(FILE)) {
                this.lockFile();
                return;
            }
            if (!m2.equals(SOCKET)) {
                throw this.error("Unsupported lock method " + m2);
            }
            String ip = p2.getProperty("ipAddress", this.ipAddress);
            if (!this.ipAddress.equals(ip)) {
                throw this.error("Locked by another computer: " + ip);
            }
            String port = p2.getProperty("port", "0");
            int portId = Integer.parseInt(port);
            try {
                address = InetAddress.getByName(ip);
            }
            catch (UnknownHostException e) {
                throw this.getException(e);
            }
            for (int i = 0; i < 3; ++i) {
                try {
                    Socket s = new Socket(address, portId);
                    s.close();
                    throw this.error("Locked by another process");
                }
                catch (BindException e) {
                    throw this.error("Bind Exception");
                }
                catch (ConnectException e) {
                    this.trace.debug("lockSocket not connected " + port, e);
                    continue;
                }
                catch (IOException e) {
                    throw this.error("IOException");
                }
            }
            if (read != this.fs.getLastModified(this.fileName)) {
                throw this.error("Concurrent update");
            }
            this.fs.delete(this.fileName);
            if (!this.fs.createNewFile(this.fileName)) {
                throw this.error("Another process was faster");
            }
        }
        try {
            this.socket = NetUtils.createServerSocket(0, false);
            int port = this.socket.getLocalPort();
            this.properties.setProperty("ipAddress", this.ipAddress);
            this.properties.setProperty("port", String.valueOf(port));
        }
        catch (Exception e) {
            this.trace.debug("lock", e);
            this.socket = null;
            this.lockFile();
            return;
        }
        this.save();
        Thread watchdog = new Thread(new Runnable(){

            public void run() {
                while (FileLock.this.socket != null) {
                    try {
                        FileLock.this.trace.debug("watchdog accept");
                        Socket s = FileLock.this.socket.accept();
                        s.close();
                    }
                    catch (Exception e) {
                        FileLock.this.trace.debug("watchdog", e);
                    }
                }
                FileLock.this.trace.debug("watchdog end");
            }
        });
        watchdog.setDaemon(true);
        watchdog.setName("H2 File Lock Watchdog (Socket) " + this.fileName);
        watchdog.start();
    }

    private void sleep(int time) throws SQLException {
        try {
            Thread.sleep(time);
        }
        catch (InterruptedException e) {
            throw this.getException(e);
        }
    }

    private SQLException getException(Throwable t) {
        return Message.getSQLException(8000, null, t);
    }

    private SQLException error(String reason) {
        return Message.getSQLException(90020, reason);
    }

    public static int getFileLockMethod(String method) throws SQLException {
        if (method == null || method.equalsIgnoreCase("FILE")) {
            return 1;
        }
        if (method.equalsIgnoreCase("NO")) {
            return 0;
        }
        if (method.equalsIgnoreCase("SOCKET")) {
            return 2;
        }
        throw Message.getSQLException(90060, method);
    }
}

