﻿Option Explicit On
Option Strict On

Imports System.Windows.Media.Animation
Imports System.Runtime.Remoting.Channels.Ipc
Imports System.Runtime.Remoting.Channels
Imports System.Runtime.Remoting
Imports System.Windows.Threading
Imports System.IO
Imports System.ComponentModel

''' <summary>メインウィンドウクラス。</summary>
''' <remarks>
''' アイコンを表示するメインウィンドウ。
''' </remarks>
Class MainWindow

    ' 自動で隠す用タイマー
    Private mDispatcherTimer As DispatcherTimer

    ' データコレクション
    Private mDatas As Collection

    ' アイコン情報テーブル
    Private mCellDic As Dictionary(Of IconCell, IconInfo)

    ' アイコン画像テーブル
    Private mImageDic As Dictionary(Of String, ImageSource)

    ' 選択中アイコン格納コントロールを記憶
    Private mSelectIconCell As IconCell

    ' 削除モード（下部パネルから「削除」を選択されたとき真）
    Private mRemoving As Boolean

    ''' <summary>コンストラクタ。</summary>
    Public Sub New()
        InitializeComponent()
        Me.DataContext = Me.settingPanel

        ' 自動で隠すためのタイマー生成
        Me.mDispatcherTimer = New DispatcherTimer(DispatcherPriority.Normal)
        AddHandler Me.mDispatcherTimer.Tick, AddressOf dispatcherTimer_Tick

        ' データ格納領域を生成
        ' 1. データコレクション
        ' 2. アイコン情報テーブル
        ' 3. アイコン画像格納テーブル
        Me.mDatas = New Collection()
        Me.mCellDic = New Dictionary(Of IconCell, IconInfo)
        Me.mImageDic = New Dictionary(Of String, ImageSource)
    End Sub

    ''' <summary>ウィンドウロードイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub MainWindow_Loaded(sender As Object, e As RoutedEventArgs) Handles Me.Loaded
        ' XMLファイルを読み込む
        Me.mDatas.LoadXMLFile()

        ' 上部タイトルを消去
        Me.headerBar.Title = String.Empty

        ' 設定パネルにデータを流す
        With Me.mDatas.Settings
            Me.settingPanel.SetColor(.BackColor)
            Me.settingPanel.SetHideClickAfter(.HideClickAfter)
            Me.settingPanel.SetAutoHideMode(.AutoHideMode)

            Me.settingPanel.SetShowTaskbar(.ShowTaskbar)
            Me.ShowInTaskbar = .ShowTaskbar
        End With

        ' 起動時の引数でページを指定されていたら変更
        Dim args() As String = System.Environment.GetCommandLineArgs()
        Dim page As Integer = 0
        If args.Length >= 2 AndAlso Integer.TryParse(args(1), page) Then
            Me.mDatas.PageIcons.ChangePage(page)
        End If

        ' メッセージ受け取り用の口を開く
        Dim svCannel As New IpcServerChannel("hachi_launcher")
        ChannelServices.RegisterChannel(svCannel, True)
        Dim msg As New RemoteMsg()
        AddHandler msg.CalllHandler, AddressOf msg_CallHandler
        RemotingServices.Marshal(msg, "message", GetType(RemoteMsg))

        ' アイコン画像を取り込み／表示位置調整
        Me.LoadCurrentIcons()
        Me.FrameLocationSetting()

        Me.OnDoWork()
    End Sub

    Private Sub OnDoWork()
        For Each icon As IconInfo In Me.mDatas.PageIcons.AllIconInfo
            Dim imag As ImageSource
            If Not Me.mImageDic.ContainsKey(icon.Path) Then
                imag = SellApis.GetIcon(icon.Path, SellApis.SHIL_EXTRALARGE)
                Me.mImageDic.Add(icon.Path, imag)
            End If
        Next
    End Sub

#Region "多重起動処理、およびスタート位置設定"

    Delegate Sub RelocationWindowHandler(sendMsg As Integer)

    ' 多重起動時のメッセージを受け取るイベントハンドラ
    Private Sub msg_CallHandler(sendMsg As Integer)
        Me.Dispatcher.Invoke(New RelocationWindowHandler(AddressOf RelocationWindow), New Object() {sendMsg})
    End Sub

    ' 多重起動のメッセージを受けたのでウィンドウの表示を調整
    Private Sub RelocationWindow(sendMsg As Integer)
        Me.mDatas.PageIcons.ChangePage(sendMsg)

        'Me.WindowState = Windows.WindowState.Normal
        Me.LoadCurrentIcons()
        Me.FrameLocationSetting()
        Me.Show()
    End Sub

    ' ウィンドウの表示位置をタスクバーに近い位置へ調整
    Private Sub FrameLocationSetting()
        Me.WindowStartupLocation = Windows.WindowStartupLocation.Manual

        Dim hwnd As IntPtr = LocationApis.FindWindow("Shell_TrayWnd", String.Empty)
        Dim rec = New LocationApis.Win32RECT
        GetWindowRect(hwnd, rec)

        Dim p As Win32POINT
        LocationApis.GetCursorPos(p)

        Dim w As Integer, h As Integer
        If rec.Left = 0 AndAlso rec.Right = SystemParameters.PrimaryScreenWidth Then
            w = p.X - CInt(Me.Width / 2)

            If w < 0 Then w = 0
            If w + Me.Width > SystemParameters.PrimaryScreenWidth Then
                w = CInt(SystemParameters.PrimaryScreenWidth - Me.Width)
            End If

            If rec.Top = 0 Then
                h = rec.Bottom + 5
            Else
                h = CInt(rec.Top - Me.Height - 5)
            End If
        Else
            h = p.Y - CInt(Me.Height / 2)

            If h < 0 Then h = 0
            If h + Me.Height > SystemParameters.PrimaryScreenHeight Then
                h = CInt(SystemParameters.PrimaryScreenHeight - Me.Height)
            End If

            If rec.Left = 0 Then
                w = rec.Right + 5
            Else
                w = CInt(rec.Left - Me.Width - 5)
            End If
        End If

        Me.Left = w
        Me.Top = h
    End Sub

#End Region

    ''' <summary>アイコンを画面に配置する。</summary>
    ''' <remarks>
    ''' 管理領域からアイコン情報を取得する。
    ''' </remarks>
    Private Sub LoadCurrentIcons()
        Me.mCellDic.Clear()
        Me.mainPanel.ClearIconImage()

        For Each icon As IconInfo In Me.mDatas.PageIcons.CurrentIcons
            ' 1. 配置コントロール
            Dim cell As IconCell = Me.mainPanel(icon.Row, icon.Column)
            cell.Hovering = False
            Me.mCellDic.Add(cell, icon)

            ' 2. 画像情報を取り込む
            '    2-1. 読み込み済みならばテーブルから取得
            '    2-2. 読み込んでいなければ画像を取得
            Dim imag As ImageSource
            If Me.mImageDic.ContainsKey(icon.Path) Then
                imag = Me.mImageDic(icon.Path)
            Else
                imag = SellApis.GetIcon(icon.Path, SellApis.SHIL_EXTRALARGE)
                Me.mImageDic.Add(icon.Path, imag)
            End If
            cell.IconImage = imag
        Next

        ' ページ名前を表示しているバーを設定
        Me.pageTitleBar.SetNames(Me.mDatas.PageIcons.PageTitles)
    End Sub

    ''' <summary>ウィンドウクローズイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub Window_Closed(sender As Object, e As EventArgs)
        Me.mDatas.SaveXMLFile()
    End Sub

#Region "ヘッダコントロールイベント処理"

    ' 「閉じる」ボタン押下イベントハンドラ
    Private Sub headerBar_NotifyCloseButtonClick(sender As Object, senderbutton As Object) Handles headerBar.NotifyCloseButtonClick
        Me.SlowCloseAnimeRun()
    End Sub

    ' ヘッダコントロールマウスダウンイベントハンドラ
    Private Sub headerBar_NotifyMouseDown(sender As Object, e As MouseButtonEventArgs) Handles headerBar.NotifyMouseDown
        If e.LeftButton = MouseButtonState.Pressed Then
            Me.DragMove()
        End If
    End Sub

    ' 「設定」ボタン押下イベントハンドラ
    Private Sub headerBar_NotifySettingButtonClick(sender As Object, senderbutton As Object) Handles headerBar.NotifySettingButtonClick
        With Me.settingPanelPopup
            .PlacementTarget = Me
            .Placement = Primitives.PlacementMode.Relative
            .Width = Me.ActualWidth
            .Height = 310
            .PlacementRectangle = New Rect(0, Me.ActualHeight - 310, Me.ActualWidth, 310)
            .IsOpen = True
        End With
    End Sub

#End Region

#Region "設定パネルコントロールイベント処理"

    ' 自動で隠すモード変更イベントハンドラ
    Private Sub settingPanel_ChangeAutoHideMode(sender As Object, newMode As Integer) Handles settingPanel.ChangeAutoHideMode
        Me.mDatas.Settings.AutoHideMode = newMode
    End Sub

    ' 背景色変更イベントハンドラ
    Private Sub settingPanel_ChangedColor(sender As Object, newColor As Color) Handles settingPanel.ChangedColor
        Me.mDatas.Settings.BackColor = newColor
    End Sub

    ' アイコンクリックで隠す変更イベントハンドラ
    Private Sub settingPanel_ChangeHideClickAfter(sender As Object, newValue As Boolean) Handles settingPanel.ChangeHideClickAfter
        Me.mDatas.Settings.HideClickAfter = newValue
    End Sub


    Private Sub settingPanel_ChangeShowTaskbar(sender As Object, newValue As Boolean) Handles settingPanel.ChangeShowTaskbar
        Me.mDatas.Settings.ShowTaskbar = newValue
        Me.ShowInTaskbar = newValue
    End Sub

#End Region

#Region "ホバー中のアイコンの名前を表示する"

    ' ホバー中の IconCell の通知イベントハンドラ
    Private Sub mainPanel_NotifyHovering(sender As Object, hoverIconCell As IconCell) Handles mainPanel.NotifyHovering
        Me.headerBar.Title = Me.mCellDic(hoverIconCell).Name
    End Sub

    ' アイコンを表示しているパネルからマウスが離脱したときの通知イベントハンドラ
    Private Sub mainPanel_NotifyMouseLeave(sender As Object) Handles mainPanel.NotifyMouseLeave
        Me.headerBar.Title = String.Empty
    End Sub

#End Region

    ''' <summary>指定アイコン格納コントロールに紐付くアイコン情報を取得する。</summary>
    ''' <param name="icell">アイコン格納コントロール。</param>
    ''' <returns>アイコン情報。</returns>
    Private Function SearchIconInfo(icell As IconCell) As IconInfo
        If Me.mCellDic.ContainsKey(icell) Then
            Return Me.mCellDic(icell)
        Else
            Return Nothing
        End If
    End Function

    ''' <summary>ゆっくりウィンドウを隠すアニメーションを実行する。</summary>
    Private Sub SlowCloseAnimeRun()
        If Me.mDatas.Settings.HideClickAfter Then
            DirectCast(Me.Resources("slowMiniAnime"), Storyboard).Begin()
        End If
    End Sub

    ''' <summary>隠すアニメーションの終了後、ウィンドウを最小化する。</summary>
    Private Sub slowAnime_Completed(sender As Object, e As EventArgs)
        'Me.WindowState = Windows.WindowState.Minimized
        Me.Hide()
    End Sub

#Region "アイコンクリック／右クリック／削除処理"

    ' アイコン左クリック処理
    Private Sub mainPanel_Clicked(sender As Object, hoverIconCell As IconCell) Handles mainPanel.Clicked
        Try
            Dim proc = New Process()
            With proc.StartInfo
                .FileName = Me.mCellDic(hoverIconCell).Path
            End With
            proc.Start()                ' 起動
            Me.SlowCloseAnimeRun()      ' ゆっくり隠す

        Catch ex As Exception
            MessageBox.Show(ex.Message)
        End Try
    End Sub

    ' アイコン右クリック処理
    Private Sub mainPanel_ClickedByRight(sender As Object, hoverIconCell As IconCell) Handles mainPanel.ClickedByRight
        ' 右クリックで選択されたアイコン格納コントロールを記憶
        Me.mSelectIconCell = hoverIconCell
        Dim info As IconInfo = Me.SearchIconInfo(hoverIconCell)
        If info Is Nothing Then Return

        ' 下部パネルの表示
        With Me.underPanelPopup
            .PlacementTarget = Me
            .Placement = Primitives.PlacementMode.Relative
            .Width = Me.ActualWidth
            .Height = 68
            .PlacementRectangle = New Rect(0, Me.ActualHeight - 68, Me.ActualWidth, 68)
            .IsOpen = True
        End With
        Me.underPanel.IconNameText = Me.mCellDic(hoverIconCell).Name

        ' 削除モードを False へ
        Me.mRemoving = False

        ' アイコンをホバーモードにしておく
        Me.mainPanel.IsHovering = True
    End Sub

    ' アイコン削除処理イベントハンドラ
    Private Sub mainPanel_Removed(sender As Object, hoverIconCell As IconCell) Handles mainPanel.Removed
        Dim revInfo As IconInfo = Me.SearchIconInfo(hoverIconCell)
        If revInfo IsNot Nothing Then
            Me.mDatas.PageIcons.CurrentIcons.Remove(revInfo)
        End If
        Me.LoadCurrentIcons()
        Me.mDatas.SaveXMLFile()
    End Sub

#End Region

#Region "下部パネル操作イベントハンドラ"

    ' 管理者としてアプリケーションの実行を行う
    Private Sub underPanel_NotifyAdministratorRun(sender As Object) Handles underPanel.NotifyAdministratorRun
        Try
            ' 対象アイコン情報を取得
            Dim info As IconInfo = Me.SearchIconInfo(Me.mSelectIconCell)

            Dim proc = New Process()
            With proc.StartInfo
                .FileName = info.Path
                .Verb = "runas"
            End With
            proc.Start()                ' 管理者として実行
            Me.SlowCloseAnimeRun()      ' ゆっくり隠す

        Catch ex As Exception
            MessageBox.Show(ex.Message)
        End Try
    End Sub

    ' Enterキーでアイコン名称の編集が終了したら実行
    '   ポップアップを閉じる
    Private Sub underPanel_NotifyEditFinish(sender As Object) Handles underPanel.NotifyEditFinish
        Me.underPanelPopup.IsOpen = False
    End Sub

    ' ポップアップが閉じたときに実行
    '   削除処理中でなければ、アイコン名を更新
    Private Sub underPanelPopup_Closed(sender As Object, e As EventArgs) Handles underPanelPopup.Closed
        If Not Me.mRemoving Then
            Dim info As IconInfo = Me.SearchIconInfo(Me.mSelectIconCell)
            info.Name = Me.underPanel.IconNameText
            Me.LoadCurrentIcons()
            Me.mDatas.SaveXMLFile()
        End If
        Me.mainPanel.IsHovering = False
    End Sub

    ' アイコン削除処理を実行
    Private Sub underPanel_NotifyRemoveIcon(sender As Object) Handles underPanel.NotifyRemoveIcon
        Me.mRemoving = True
        Me.underPanelPopup.IsOpen = False
        Me.mSelectIconCell.StartRemoveAnime()
    End Sub

#End Region

#Region "ドラッグ・ドロップイベントハンドラ"

    ' アイコンドロップイベントハンドラ
    Private Sub dragPanel_DropOfIcon(sender As Object, fromCell As IconCell, toCell As IconCell) Handles mainPanel.DropOfIcon
        ' 移動元／先のアイコン格納領域を取得
        Dim fmInfo As IconInfo = Me.SearchIconInfo(fromCell)
        Dim toInfo As IconInfo = Me.SearchIconInfo(toCell)

        ' 移動先があれば入れ替え／なければ移動
        If toInfo IsNot Nothing Then
            fmInfo.Swap(toInfo)
        Else
            fmInfo.Move(toCell.Column, toCell.Row)
        End If

        Me.LoadCurrentIcons()
    End Sub

    ' ファイルドロップイベントハンドラ
    Private Sub dragPanel_DropOfFile(sender As Object, dropPath As String, dropPos As Point) Handles mainPanel.DropOfFile
        ' マウス位置のアイコン格納領域を取得
        Dim icell As IconCell = Me.mainPanel(dropPos)
        If icell Is Nothing Then Return

        ' アイコン情報が紐付くか調査する
        ' 1. 情報が紐付く場合、パスを更新
        ' 2. 紐付かない場合、追加する
        Dim info As IconInfo = SearchIconInfo(icell)
        If info IsNot Nothing Then
            info.UpdatePath(dropPath, Application.APPDATA_PATH) ' 1
        Else
            Dim icons As List(Of IconInfo) = Me.mDatas.PageIcons.CurrentIcons ' 2
            icons.Add(IconInfo.CreateNewIcon(icell.Column, icell.Row, dropPath, Application.APPDATA_PATH))
        End If

        Me.LoadCurrentIcons()
        Me.mDatas.SaveXMLFile()
    End Sub

#End Region

#Region "ページタイトルバー操作イベントハンドラ"

    ' ページスライド開始のとき、アイコンを隠す
    Private Sub pageTitleBar_SlideAnimeRun(sender As Object) Handles pageTitleBar.SlideAnimeRun
        Me.mainPanel.IsHovering = True
        DirectCast(Me.Resources("hideAnime"), Storyboard).Begin()
    End Sub

    ' アイコンを隠し終わったのち、カレントページを変更して再表示
    Private Sub pageTitleBar_SlideAnimeFinish(sender As Object, slideCount As Integer) Handles pageTitleBar.SlideAnimeFinish
        DirectCast(Me.Resources("showAnime"), Storyboard).Begin()
        Me.mDatas.PageIcons.IncrementPage(slideCount)
        Me.LoadCurrentIcons()
        Me.mainPanel.IsHovering = False
        Me.mDatas.SaveXMLFile()
    End Sub

    ' ページ追加処理
    Private Sub pageTitleBar_NotifyAddPage(sender As Object, e As RoutedEventArgs) Handles pageTitleBar.NotifyAddPage
        Me.mDatas.PageIcons.InsertPage()
        Me.LoadCurrentIcons()
        Me.mDatas.SaveXMLFile()
    End Sub

    ' ページ削除処理
    Private Sub pageTitleBar_NotifyRemovePage(sender As Object, e As RoutedEventArgs) Handles pageTitleBar.NotifyRemovePage
        Me.mDatas.PageIcons.RemovePage()
        Me.LoadCurrentIcons()
        Me.mDatas.SaveXMLFile()
    End Sub

    ' ページ名変更処理
    Private Sub pageTitleBar_ChangePageName(sender As Object, selectIndex As Integer, newName As String) Handles pageTitleBar.ChangePageName
        Me.mDatas.PageIcons.UpdatePageTitle(newName, selectIndex)
        Me.LoadCurrentIcons()
        Me.mDatas.SaveXMLFile()
    End Sub

#End Region

#Region "自動で隠すためのイベントハンドラ、処理"

    ' マウスが離脱したとき、自動隠しモードにしたがってタイマーを設定
    '   タイマーが経過したのちウィンドウを隠す
    Private Sub window_MouseLeave(sender As Object, e As MouseEventArgs) Handles window.MouseLeave
        ' 各種ポップアップが表示されていない、かつドラッグ中でなければタイマー起動
        If Not Me.settingPanelPopup.IsOpen AndAlso
           Not Me.underPanelPopup.IsOpen AndAlso
           Not Me.pageTitleBar.IsEditOpen AndAlso
           Not Me.mainPanel.IsDragging Then

            Select Case Me.mDatas.Settings.AutoHideMode
                Case 1  ' 30秒後、隠す
                    Me.mDispatcherTimer.Interval = New TimeSpan(0, 0, 30)
                    Me.mDispatcherTimer.Start()
                Case 2  ' 3秒後、隠す
                    Me.mDispatcherTimer.Interval = New TimeSpan(0, 0, 3)
                    Me.mDispatcherTimer.Start()
                Case 3  ' 0.5秒後、隠す
                    Me.mDispatcherTimer.Interval = New TimeSpan(5000000)
                    Me.mDispatcherTimer.Start()
            End Select
        End If
    End Sub

    ' タイマーイベントハンドラ（ウィンドウを隠す）
    Private Sub dispatcherTimer_Tick(sender As Object, e As EventArgs)
        Me.mDispatcherTimer.Stop()
        SlowCloseAnimeRun()
    End Sub

    ' マウスが侵入したとき、タイマーを停止する
    Private Sub window_MouseEnter(sender As Object, e As MouseEventArgs) Handles window.MouseEnter
        Me.mDispatcherTimer.Stop()
    End Sub

#End Region

End Class
