﻿#pragma once

/*!
 *  @file defines.h
 *  @brief 主要なマクロ定義ヘッダ / Purpose: global constants and macro definitions
 *  @date 2014/01/02
 *  @author
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke\n
 *\n
 * This software may be copied and distributed for educational, research,\n
 * and not for profit purposes provided that this copyright and statement\n
 * are included in all such copies.  Other copyrights may also apply.\n
 *  @details
 * Do not edit this file unless you know *exactly* what you are doing.\n
 *\n
 * Some of the values in this file were chosen to preserve game balance,\n
 * while others are hard-coded based on the format of old save-files, the\n
 * definition of arrays in various places, mathematical properties, fast\n
 * computation, storage limits, or the format of external text files.\n
 *\n
 * Changing some of these values will induce crashes or memory errors or\n
 * savefile mis-reads.  Most of the comments in this file are meant as\n
 * reminders, not complete descriptions, and even a complete knowledge\n
 * of the source may not be sufficient to fully understand the effects\n
 * of changing certain definitions.\n
 *\n
 * Lastly, note that the code does not always use the symbolic constants\n
 * below, and sometimes uses various hard-coded values that may not even\n
 * be defined in this file, but which may be related to definitions here.\n
 * This is of course bad programming practice, but nobody is perfect...\n
 *\n
 * For example, there are MANY things that depend on the screen being\n
 * 80x24, with the top line used for messages, the bottom line being\n
 * used for status, and exactly 22 lines used to show the dungeon.\n
 * Just because your screen can hold 46 lines does not mean that the\n
 * game will work if you try to use 44 lines to show the dungeon.\n
 *\n
 * You have been warned.\n
 */

#define VERSION_NAME "Hengband" /*!< バリアント名称 / Name of the version/variant */

/*!
 * @brief ゲームのバージョン番号定義 / "Program Version Number" of the game
 * @details
 * 本FAKE_VERSIONそのものは未使用である。Zangと整合性を合わせるための疑似的処理のためFAKE_VER_MAJORは実値-10が該当のバージョン番号となる。
 * <pre>
 * FAKE_VER_MAJOR=1,2 were reserved for ZAngband version 1.x.x/2.x.x .
 * Program Version of Hengband version is
 *   "(FAKE_VER_MAJOR-10).(FAKE_VER_MINOR).(FAKE_VER_PATCH)".
 * </pre>
 */
#define FAKE_VERSION 0

#define FAKE_VER_MAJOR 12 /*!< ゲームのバージョン番号定義(メジャー番号 + 10) */
#define FAKE_VER_MINOR 2 /*!< ゲームのバージョン番号定義(マイナー番号) */
#define FAKE_VER_PATCH 1 /*!< ゲームのバージョン番号定義(パッチ番号) */
#define FAKE_VER_EXTRA 7 /*!< ゲームのバージョン番号定義(エクストラ番号) */


 /*!
  * @brief バージョンが開発版が安定版かを返す
  */
#define	IS_STABLE_VERSION (FAKE_VER_MINOR % 2 == 0 && FAKE_VER_EXTRA == 0)

/*!
 * @brief セーブファイル上のバージョン定義(メジャー番号) / "Savefile Version Number" for Hengband 1.1.1 and later
 * @details
 * 当面FAKE_VER_*を参照しておく。
 * <pre>
 * First three digits may be same as the Program Version.  But not
 * always same.  It means that newer version may preserves lower
 * compatibility with the older version.
 * For example, newer Hengband 1.4.4 creates savefiles marked with
 * Savefile Version 1.4.0.0 .  It means that Hengband 1.4.0 can load a
 * savefile of Hengband 1.4.4 (lower compatibility!).
 * Upper compatibility is always guaranteed.
 * </pre>
 */
#define H_VER_MAJOR (FAKE_VER_MAJOR-10) /*!< セーブファイル上のバージョン定義(メジャー番号) */
#define H_VER_MINOR FAKE_VER_MINOR /*!< セーブファイル上のバージョン定義(マイナー番号) */
#define H_VER_PATCH FAKE_VER_PATCH /*!< セーブファイル上のバージョン定義(パッチ番号) */
#define H_VER_EXTRA FAKE_VER_EXTRA /*!< セーブファイル上のバージョン定義(エクストラ番号) */

#define ANGBAND_2_8_1 /*!< Angband 2.8.1以降から有効な処理分岐を定義 */
#define ZANGBAND /*!< Zangband 以降から有効な処理分岐を定義 */

/*!
 * @brief generate.cで用いられる基本的なブロック数単位(垂直方向)
 * Number of grids in each block (vertically) Probably hard-coded to 11, see "generate.c"
 */
#define BLOCK_HGT 11

/*!
 * @brief generate.cで用いられる基本的なブロック数単位(水平方向)
 * Number of grids in each block (horizontally) Probably hard-coded to 11, see "generate.c"
 */
#define BLOCK_WID 11

/*!
 * @brief 表示上の基本的なパネル単位(垂直方向、BLOCK_HGTの倍数で設定すること)
 * Number of grids in each panel (vertically) Must be a multiple of BLOCK_HGT
 */
#define PANEL_HGT 11

/*!
 * @brief 表示上の基本的なパネル単位(水平方向、BLOCK_WIDの倍数で設定すること)
 * Number of grids in each panel (horizontally) Must be a multiple of BLOCK_WID
 */
#define PANEL_WID 33

/*!
 * @brief 表示上の基本的なブロック単位(垂直方向、PANEL_HGTの倍数で設定すること)
 * Number of grids used to display the dungeon (vertically). Must be a multiple of 11, probably hard-coded to 22.
 */
#define SCREEN_HGT 22

/*!
 * @brief 表示上の基本的なブロック単位(水平方向、PANEL_WIDの倍数で設定すること)
 * Number of grids used to display the dungeon (horizontally). Must be a multiple of 33, probably hard-coded to 66.
 */
#define SCREEN_WID 66

/*!
 * @brief 表示上のダンジョンの最大垂直サイズ(SCREEN_HGTの3倍が望ましい)
 * Maximum dungeon height in grids, must be a multiple of SCREEN_HGT, probably hard-coded to SCREEN_HGT * 3.
 */
#define MAX_HGT 66

/*!
 * @brief 表示上のダンジョンの最大水平サイズ(SCREEN_WIDの3倍が望ましい)
 * Maximum dungeon width in grids, must be a multiple of SCREEN_WID, probably hard-coded to SCREEN_WID * 3.
 */
#define MAX_WID 198

/*
 * Arena constants
 */
#define MAX_ARENA_MONS 41 /*<! 闘技場のイベント件数 -KMW- */
#define ARENA_DEFEATED_OLD_VER (-(MAX_SHORT)) /*<! 旧バージョンの闘技場敗北定義 */


#define MAX_BLDG 32 /*!< 施設の種類最大数 / Number of buildings */


#define BUILDING_NON_MEMBER 0 /*!< 不明(現在未使用) */
#define BUILDING_MEMBER     1 /*!< 不明(現在未使用) */
#define BUILDING_OWNER      2 /*!< 施設の種族/職業条件が一致している状態値 */

#define MAX_SEXES        2 /*!< 性別の定義最大数 / Maximum number of player "sex" types (see "table.c", etc) */
#define MAX_CLASS       28 /*!< 職業の最大定義数 Maximum number of player "class" types (see "table.c", etc) */
#define MAX_SEIKAKU     13 /*!< 性格の最大定義数 */

/* ELDRITCH_HORRORによるsanity blast処理に関するメッセージの最大数 / Number of entries in the sanity-blast descriptions */
#define MAX_SAN_HORROR 20 /*!< 恐ろしい対象の形容数(正常時) */
#define MAX_SAN_FUNNY 22  /*!< 恐ろしい対象の形容数(幻覚時) */
#define MAX_SAN_COMMENT 5 /*!< 恐ろしい対象を見たときの絶叫メッセージ数(幻覚時) */


/* カオスの変異 / Chaos mutations */


#define MAX_MA 17 /*!< 修行僧マーシャルアーツの技数 / Monk martial arts... */
#define MA_KNEE 1 /*!< 金的効果ID */
#define MA_SLOW 2 /*!< 膝蹴り効果ID */

#define MAX_MIND_POWERS  21 /*!< 超能力の数 / Mindcraft */


#ifdef JP
#define MAX_SILLY_ATTACK 41 /*!< 幻覚時のジョークメッセージ最大数 */
#else
#define MAX_SILLY_ATTACK 29 /*!< Hallucination stuff */
#endif

#define MIND_MINDCRAFTER    0 /*!< 特殊能力: 超能力 */
#define MIND_KI             1 /*!< 特殊能力: 練気 */
#define MIND_BERSERKER      2 /*!< 特殊能力: 怒り */
#define MIND_MIRROR_MASTER  3 /*!< 特殊能力: 鏡魔法 */
#define MIND_NINJUTSU       4 /*!< 特殊能力: 忍術 */


/*
 * Size of memory reserved for initialization of some arrays
 */
#define FAKE_NAME_SIZE  40 * 1024L /*!< ゲーム情報の種別毎に用意される名前用バッファの容量 */
#define FAKE_TEXT_SIZE 150 * 1024L /*!< ゲーム情報の種別毎に用意されるテキスト用バッファの容量 */
#define FAKE_TAG_SIZE   10 * 1024L /*!< ゲーム情報の種別毎に用意されるタグ用バッファの容量 */

#define MAX_HISCORES    999 /*!< スコア情報保存の最大数 / Maximum number of high scores in the high score file */



/*!
 * @brief ダンジョンの最深層 / Maximum dungeon level. 
 * @details
 * The player can never reach this level
 * in the dungeon, and this value is used for various calculations
 * involving object and monster creation.  It must be at least 100.
 * Setting it below 128 may prevent the creation of some objects.
 */
#define MAX_DEPTH       128 

#define MAX_SAVED_FLOORS       20 /*!< 保存フロアの最大数 / Maximum number of saved floors. */

/*!
 * @brief プレイヤー用光源処理配列サイズ / Maximum size of the "lite" array (see "current_floor_ptr->grid_array.c")
 * @details Note that the "lite radius" will NEVER exceed 14, and we would
 * never require more than 581 entries in the array for circular "lite".
 */
#define LITE_MAX 600

/*!
 * @brief モンスター用光源処理配列サイズ / Maximum size of the "mon_lite" array (see "current_floor_ptr->grid_array.c")
 * @details Note that the "view radius" will NEVER exceed 20, monster illumination
 * flags are dependent on CAVE_VIEW, and even if the "view" was octagonal,
 * we would never require more than 1520 entries in the array.
 */
#define MON_LITE_MAX 1536

/*!
 * @brief 視界処理配列サイズ / Maximum size of the "view" array (see "current_floor_ptr->grid_array.c")
 * @details Note that the "view radius" will NEVER exceed 20, and even if the "view"
 * was octagonal, we would never require more than 1520 entries in the array.
 */
#define VIEW_MAX 1536

/*!
 * @brief 視界及び光源の過渡処理配列サイズ / Maximum size of the "temp" array (see "current_floor_ptr->grid_array.c")
 * @details We must be as large as "VIEW_MAX" and "LITE_MAX" for proper functioning
 * of "update_view()" and "update_lite()".  We must also be as large as the
 * largest illuminatable room, but no room is larger than 800 grids.  We
 * must also be large enough to allow "good enough" use as a circular queue,
 * to calculate monster flow, but note that the flow code is "paranoid".
 */
#define TEMP_MAX 2298

/*!
 * @brief 再描画処理用配列サイズ / Maximum size of the "redraw" array (see "current_floor_ptr->grid_array.c")
 * @details We must be large for proper functioning of delayed redrawing.
 * We must also be as large as two times of the largest view area.
 * Note that maximum view grids are 1149 entries.
 */
#define REDRAW_MAX 2298

#define KEYMAP_MODE_ORIG	0 /*!< オリジナルキー配置 / Mode for original keyset commands */
#define KEYMAP_MODE_ROGUE	1 /*!< ローグライクキー配置 / Mode for roguelike keyset commands */
#define KEYMAP_MODES		2 /*!< キー配置の数 / Number of keymap modes */





/*!
 * @brief マクロ登録の最大数 / Maximum number of macros (see "io.c")
 * @note Default: assume at most 256 macros are used
 */
#define MACRO_MAX       256

/*!
 * @brief 銘情報の最大数 / Maximum number of "quarks" (see "io.c")
 * @note 
 * Default: assume at most 512 different inscriptions are used<br>
 * Was 512... 256 quarks added for random artifacts<br>
 */
#define QUARK_MAX       768

/*
 * OPTION: Maximum number of messages to remember (see "io.c")
 * Default: assume maximal memorization of 2048 total messages
 */
#define MESSAGE_MAX  81920

/*
 * OPTION: Maximum space for the message text buffer (see "io.c")
 * Default: assume that each of the 2048 messages is repeated an
 * average of three times, and has an average length of 48
 */
#define MESSAGE_BUF 655360


/*
 * Maximum value storable in a "byte" (hard-coded)
 */
#define MAX_UCHAR       255

/*
 * Maximum value storable in a "s16b" (hard-coded)
 */
#define MAX_SHORT       32767

/*
 * Maximum length of object's name
 */
#define MAX_NLEN        160

/*
 * Special internal key
 */
#define SPECIAL_KEY_QUEST    255
#define SPECIAL_KEY_BUILDING 254
#define SPECIAL_KEY_STORE    253
#define SPECIAL_KEY_QUIT     252


/*
 * Misc constants
 */
#define TOWN_DAWN         10000   /*!< 1日分のターン / Number of ticks from dawn to dawn XXX */
#define TURNS_PER_TICK    10L     /*!< 時間経過処理を行うターン数の刻み / Number of energy-gain-turns per ticks */
#define INN_DUNGEON_TURN_ADJ 10   /*!< 宿屋で時間をつぶした場合に増えるcurrent_world_ptr->dungeon_turnの倍率 */
#define MAX_DAYS          20000   /*!< 内部処理中で保持される最大日数 / Maximum days */
#define BREAK_GLYPH       550     /*!< 守りのルーンの強靭度 / Rune of protection resistance */
#define BREAK_MINOR_GLYPH 299     /*!< 爆発のルーンの発動しやすさ / For explosive runes */
#define BTH_PLUS_ADJ      3       /*!< 武器経験値及びプレイヤーの打撃/射撃能力に応じた修正値倍率 / Adjust BTH per plus-to-hit */
#define MON_MULT_ADJ      8       /*!< モンスターの増殖しにくさの基本倍率 / High value slows multiplication */
#define MON_SUMMON_ADJ    2       /*!< 現在未使用 Adjust level of summoned creatures */
#define MON_DRAIN_LIFE    2       /*!< モンスターの打撃によるプレイヤーの経験値吸収基本倍率(%) / Percent of player exp drained per hit */
#define USE_DEVICE        3       /*!< 魔道具の最低失敗基準値 x> Harder devices x< Easier devices     */



#define CAN_TWO_HANDS_WIELDING() (!p_ptr->riding || (p_ptr->pet_extra_flags & PF_RYOUTE))


/*
 * Player constants
 */
#define PY_MAX_EXP      99999999L       /*!< プレイヤー経験値の最大値 / Maximum exp */
#define PY_MAX_GOLD     999999999L      /*!< プレイヤー所持金の最大値 / Maximum gold */
#define PY_MAX_LEVEL    50              /*!< プレイヤーレベルの最大値 / Maximum level */

/*
 * Player "food" crucial values
 */
#define PY_FOOD_MAX     15000   /*!< 食べ過ぎ～満腹の閾値 / Food value (Bloated) */
#define PY_FOOD_FULL    10000   /*!< 満腹～平常の閾値 / Food value (Normal) */
#define PY_FOOD_ALERT   2000    /*!< 平常～空腹の閾値 / Food value (Hungry) */
#define PY_FOOD_WEAK    1000    /*!< 空腹～衰弱の閾値 / Food value (Weak) */
#define PY_FOOD_FAINT   500     /*!< 衰弱～衰弱(赤表示/麻痺)の閾値 / Food value (Fainting) */
#define PY_FOOD_STARVE  100     /*!< 衰弱(赤表示/麻痺)～飢餓ダメージの閾値 / Food value (Starving) */

/*
 * Player regeneration constants
 */
#define PY_REGEN_NORMAL         197     /* Regen factor*2^16 when full */
#define PY_REGEN_WEAK           98      /* Regen factor*2^16 when weak */
#define PY_REGEN_FAINT          33      /* Regen factor*2^16 when fainting */
#define PY_REGEN_HPBASE         1442    /* Min amount hp regen*2^16 */
#define PY_REGEN_MNBASE         524     /* Min amount mana regen*2^16 */

/*
 * 職業ごとの選択可能な魔法領域現在の所 bitrh.cとtables.cでのみ使用。
 * Possible realms that can be chosen currently used only by birth.c and tables.c
 */
#define CH_NONE         0x00
#define CH_LIFE         0x01
#define CH_SORCERY      0x02
#define CH_NATURE       0x04
#define CH_CHAOS        0x08
#define CH_DEATH        0x10
#define CH_TRUMP        0x20
#define CH_ARCANE       0x40
#define CH_ENCHANT      0x80
#define CH_DAEMON       0x100
#define CH_CRUSADE      0x200

#define CH_MUSIC        0x08000	/* This is 16th bit */
#define CH_HISSATSU     0x10000
#define CH_HEX          0x20000


/*
 * Magic realms
 */
#define REALM_NONE         0
#define REALM_LIFE         1
#define REALM_SORCERY      2
#define REALM_NATURE       3
#define REALM_CHAOS        4
#define REALM_DEATH        5
#define REALM_TRUMP        6
#define REALM_ARCANE       7
#define REALM_CRAFT        8
#define REALM_DAEMON       9
#define REALM_CRUSADE      10
#define MAX_MAGIC          10
#define MIN_TECHNIC        16
#define REALM_MUSIC        16
#define REALM_HISSATSU     17
#define REALM_HEX          18
#define MAX_REALM          18

#define VALID_REALM        (MAX_REALM + MAX_MAGIC - MIN_TECHNIC + 1)
#define NUM_TECHNIC        (MAX_REALM - MIN_TECHNIC + 1)

#define is_magic(A) ((((A) > REALM_NONE) && ((A) < MAX_MAGIC + 1)) ? TRUE : FALSE)
#define tval2realm(A) ((A) - TV_LIFE_BOOK + 1)
#define technic2magic(A)      (is_magic(A) ? (A) : (A) - MIN_TECHNIC + 1 + MAX_MAGIC)
#define is_good_realm(REALM)   ((REALM) == REALM_LIFE || (REALM) == REALM_CRUSADE)

/*
 * Magic-books for the realms
 */
#define REALM1_BOOK     (p_ptr->realm1 + TV_LIFE_BOOK - 1)
#define REALM2_BOOK     (p_ptr->realm2 + TV_LIFE_BOOK - 1)


/*
 * Mode constant for do_spell()
 */
#define SPELL_NAME   0
#define SPELL_DESC   1
#define SPELL_INFO   2
#define SPELL_CAST   3
#define SPELL_FAIL   4
#define SPELL_STOP   5
#define SPELL_CONT   6


/*
 * Maximum number of "normal" pack slots, and the index of the "overflow"
 * slot, which can hold an item, but only temporarily, since it causes the
 * pack to "overflow", dropping the "last" item onto the ground.  Since this
 * value is used as an actual slot, it must be less than "INVEN_RARM" (below).
 * Note that "INVEN_PACK" is probably hard-coded by its use in savefiles, and
 * by the fact that the screen can only show 23 items plus a one-line prompt.
 */
#define INVEN_PACK      23 /*!< アイテムスロット…所持品(0～) */

/*
 * Indexes used for various "equipment" slots (hard-coded by savefiles, etc).
 */
#define INVEN_RARM      24 /*!< アイテムスロット…右手 */
#define INVEN_LARM      25 /*!< アイテムスロット…左手 */
#define INVEN_BOW       26 /*!< アイテムスロット…射撃 */
#define INVEN_RIGHT     27 /*!< アイテムスロット…右手指 */
#define INVEN_LEFT      28 /*!< アイテムスロット…左手指 */
#define INVEN_NECK      29 /*!< アイテムスロット…首 */
#define INVEN_LITE      30 /*!< アイテムスロット…光源 */
#define INVEN_BODY      31 /*!< アイテムスロット…体 */
#define INVEN_OUTER     32 /*!< アイテムスロット…体の上 */
#define INVEN_HEAD      33 /*!< アイテムスロット…頭部 */
#define INVEN_HANDS     34 /*!< アイテムスロット…腕部 */
#define INVEN_FEET      35 /*!< アイテムスロット…脚部 */

/*
 * used for get_random_ego()
 */
#define INVEN_AMMO     23

/*
 * Total number of p_ptr->inventory_list slots (hard-coded).
 */
#define INVEN_TOTAL     36

/*
 * Fake p_ptr->inventory_list slot for selecting force (hard-coded).
 */
#define INVEN_FORCE     1111


/*
 * Indexes of the various "stats" (hard-coded by savefiles, etc).
 */
#define A_STR 0
#define A_INT 1
#define A_WIS 2
#define A_DEX 3
#define A_CON 4
#define A_CHR 5
#define A_MAX 6

/*
 * Player sex constants (hard-coded by save-files, arrays, etc)
 */
#define SEX_FEMALE              0
#define SEX_MALE                1


/*
 * Player race constants (hard-coded by save-files, arrays, etc)
 */
#define RACE_HUMAN               0
#define RACE_HALF_ELF            1
#define RACE_ELF                 2
#define RACE_HOBBIT              3
#define RACE_GNOME               4
#define RACE_DWARF               5
#define RACE_HALF_ORC            6
#define RACE_HALF_TROLL          7
#define RACE_AMBERITE            8
#define RACE_HIGH_ELF            9
#define RACE_BARBARIAN          10
#define RACE_HALF_OGRE          11
#define RACE_HALF_GIANT         12
#define RACE_HALF_TITAN         13
#define RACE_CYCLOPS            14
#define RACE_YEEK               15
#define RACE_KLACKON            16
#define RACE_KOBOLD             17
#define RACE_NIBELUNG           18
#define RACE_DARK_ELF           19
#define RACE_DRACONIAN          20
#define RACE_MIND_FLAYER        21
#define RACE_IMP                22
#define RACE_GOLEM              23
#define RACE_SKELETON           24
#define RACE_ZOMBIE             25
#define RACE_VAMPIRE            26
#define RACE_SPECTRE            27
#define RACE_SPRITE             28
#define RACE_BEASTMAN           29
#define RACE_ENT                30
#define RACE_ANGEL              31
#define RACE_DEMON              32
#define RACE_DUNADAN            33
#define RACE_S_FAIRY            34
#define RACE_KUTAR              35
#define RACE_ANDROID            36
#define RACE_MERFOLK            37

/*
 * Maximum number of player "race" types (see "table.c", etc)
 */
#define MAX_RACES               38


/*
 * Player class constants (hard-coded by save-files, arrays, etc)
 */
#define CLASS_WARRIOR            0
#define CLASS_MAGE               1
#define CLASS_PRIEST             2
#define CLASS_ROGUE              3
#define CLASS_RANGER             4
#define CLASS_PALADIN            5
#define CLASS_WARRIOR_MAGE       6
#define CLASS_CHAOS_WARRIOR      7
#define CLASS_MONK               8
#define CLASS_MINDCRAFTER        9
#define CLASS_HIGH_MAGE         10
#define CLASS_TOURIST           11
#define CLASS_IMITATOR          12
#define CLASS_BEASTMASTER       13
#define CLASS_SORCERER          14
#define CLASS_ARCHER            15
#define CLASS_MAGIC_EATER       16
#define CLASS_BARD              17
#define CLASS_RED_MAGE          18
#define CLASS_SAMURAI           19
#define CLASS_FORCETRAINER      20
#define CLASS_BLUE_MAGE         21
#define CLASS_CAVALRY           22
#define CLASS_BERSERKER         23
#define CLASS_SMITH             24
#define CLASS_MIRROR_MASTER     25
#define CLASS_NINJA             26
#define CLASS_SNIPER            27

#define SEIKAKU_FUTUU	 0
#define SEIKAKU_CHIKARA	 1
#define SEIKAKU_KIREMONO 2
#define SEIKAKU_SHIAWASE 3
#define SEIKAKU_SUBASI	 4
#define SEIKAKU_INOCHI	 5
#define SEIKAKU_COMBAT	 6
#define SEIKAKU_NAMAKE	 7
#define SEIKAKU_SEXY	 8
#define SEIKAKU_LUCKY	 9
#define SEIKAKU_GAMAN	 10
#define SEIKAKU_MUNCHKIN 11
#define SEIKAKU_CHARGEMAN 12

/*** Screen Locations ***/

#define VER_INFO_ROW 3   //!< タイトル表記(行)


#define ROW_MAP                 0
#define COL_MAP                 12

/*
 * Number of feats we change to (Excluding default). Used in f_info.txt.
 */
#define MAX_FEAT_STATES	 8


/*
 * Wilderness terrains
 */
#define TERRAIN_EDGE             0 /* Edge of the World */
#define TERRAIN_TOWN             1 /* Town */
#define TERRAIN_DEEP_WATER       2 /* Deep water */
#define TERRAIN_SHALLOW_WATER    3 /* Shallow water */
#define TERRAIN_SWAMP            4 /* Swamp */
#define TERRAIN_DIRT             5 /* Dirt */
#define TERRAIN_GRASS            6 /* Grass */
#define TERRAIN_TREES            7 /* Trees */
#define TERRAIN_DESERT           8 /* Desert */
#define TERRAIN_SHALLOW_LAVA     9 /* Shallow lava */
#define TERRAIN_DEEP_LAVA       10 /* Deep lava */
#define TERRAIN_MOUNTAIN        11 /* Mountain */

#define MAX_WILDERNESS          12 /* Maximum wilderness index */


/*
 * Feature flags - should be used instead of feature indexes unless generating.
 * Originally from UnAngband, and modified into TR-like style in Hengband
 */

#define FF_LOS           0              /*!< 視界が通る地形である */
#define FF_PROJECT       1              /*!< 飛び道具が通過できる地形である */
#define FF_MOVE          2              /*!< 移動可能な地形である */
#define FF_PLACE         3              /*!< モンスター配置をしても良い地形である(cave_empty_bold/cave_empty_gridで利用) */
#define FF_DROP          4              /*!< アイテムを落としてよい地形である */
#define FF_SECRET        5              /*!< 隠し扉やトラップが潜んでいる地形である */
#define FF_NOTICE        6              /*!< 何か興味を引くものがある地形である(シフトキー＋方向で走行中の時に止まる基準) */
#define FF_REMEMBER      7              /*!< 常に記憶対象となる地形である(記憶喪失時に忘れたりしなくなる) */
#define FF_OPEN          8              /*!< 開けるコマンドの対象となる地形である */
#define FF_CLOSE         9              /*!< 閉じるコマンドの対象となる地形である */
#define FF_BASH          10             /*!< 体当たりコマンドの対象となる地形である */
#define FF_SPIKE         11             /*!< くさびを打つコマンドの対象となる地形である */
#define FF_DISARM        12             /*!< 解除コマンドの対象となる地形である */
#define FF_STORE         13             /*!< 店舗の入口となる地形である */
#define FF_TUNNEL        14             /*!< 魔王変化などで掘り進められる地形である */
#define FF_MAY_HAVE_GOLD 15             /*!< 何か財宝を隠した可能性のある地形である？(f_infoに使用している地形なし) */
#define FF_HAS_GOLD      16             /*!< 財宝を含んだ地形である */
#define FF_HAS_ITEM      17             /*!< アイテムを含んだ地形である */
#define FF_DOOR          18             /*!< ドアのある地形である */
#define FF_TRAP          19             /*!< トラップのある地形である */
#define FF_STAIRS        20             /*!< 階段のある地形である */
#define FF_GLYPH         21             /*!< 守りのルーンが張られた地形である */
#define FF_LESS          22             /*!< 階上に通じる地形である */
#define FF_MORE          23             /*!< 階下に通じる地形である */
#define FF_AVOID_RUN     24             /*!< 自動移動機能時に障害として迂回すべき地形である */
#define FF_FLOOR         25             /*!< 床のある地形である */
#define FF_WALL          26             /*!< 壁のある地形である */
#define FF_PERMANENT     27             /*!< 絶対に破壊できない永久地形である */
/* #define FF_XXX00         28  未定義 */
/* #define FF_XXX01         29  未定義 */
/* #define FF_XXX02         30  未定義 */
#define FF_HIT_TRAP      31             /*!< トラップのある地形である(TRAPと常に重複している？) */
/* #define FF_BRIDGE        32  未使用 */
/* #define FF_RIVER         33  未使用 */
/* #define FF_LAKE          34  未使用 */
/* #define FF_BRIDGED       35  未使用 */
/* #define FF_COVERED       36  未使用 */
#define FF_GLOW          37             /*!< 常に光っている地形である */
#define FF_ENSECRET      38             /*!< 不明(f_info.txt上で利用している地形がない) */
#define FF_WATER         39             /*!< 水のある地形である */
#define FF_LAVA          40             /*!< 溶岩のある地形である */
#define FF_SHALLOW       41             /*!< 浅い地形である */
#define FF_DEEP          42             /*!< 深い地形である */
#define FF_POISON_PUDDLE 43             /*!< 毒溜まりがある */
#define FF_HURT_ROCK     44             /*!< 岩石溶解の対象となる地形である */
/* #define FF_HURT_FIRE     45 */       /*!< 未使用 */
/* #define FF_HURT_COLD     46 */       /*!< 未使用 */
/* #define FF_HURT_ACID     47 */       /*!< 未使用 */
#define FF_COLD_PUDDLE   48             /*!< 冷気溜まりがある */
#define FF_ACID_PUDDLE   49             /*!< 酸溜まりがある */
/* #define FF_OIL           50 */       /*!< 未使用 */
#define FF_ELEC_PUDDLE   51             /*!< 接地部が帯電している */
/* #define FF_CAN_CLIMB     52 */       /*!< 未使用 */
#define FF_CAN_FLY       53             /*!< 飛行可能な地形である */
#define FF_CAN_SWIM      54             /*!< 泳ぐことが可能な地形である */
#define FF_CAN_PASS      55             /*!< 通過可能な地形である */
/* #define FF_CAN_OOZE      56 */       /*!< 未使用 */
#define FF_CAN_DIG       57             /*!< 掘削コマンドの対象となる地形である */
/* #define FF_HIDE_ITEM     58  未使用 */
/* #define FF_HIDE_SNEAK    59  未使用 */
/* #define FF_HIDE_SWIM     60  未使用 */
/* #define FF_HIDE_DIG      61  未使用 */
/* #define FF_KILL_HUGE     62  未使用 */
/* #define FF_KILL_MOVE     63  未使用 */
/* #define FF_PICK_TRAP     64  未使用 */
/* #define FF_PICK_DOOR     65  未使用 */
/* #define FF_ALLOC         66  未使用 */
/* #define FF_CHEST         67  未使用 */
/* #define FF_DROP_1D2      68  未使用 */
/* #define FF_DROP_2D2      69  未使用 */
/* #define FF_DROP_GOOD     70  未使用 */
/* #define FF_DROP_GREAT    71  未使用 */
/* #define FF_HURT_POIS     72  未使用 */
/* #define FF_HURT_ELEC     73  未使用 */
/* #define FF_HURT_WATER    74  未使用 */
/* #define FF_HURT_BWATER   75  未使用 */
/* #define FF_USE_FEAT      76  未使用 */
/* #define FF_GET_FEAT      77  未使用 */
/* #define FF_GROUND        78  未使用 */
/* #define FF_OUTSIDE       79  未使用 */
/* #define FF_EASY_HIDE     80  未使用 */
/* #define FF_EASY_CLIMB    81  未使用 */
/* #define FF_MUST_CLIMB    82  未使用 */
#define FF_TREE          83             /*!< 木の生えた地形である */
/* #define FF_NEED_TREE     84  未使用 */
/* #define FF_BLOOD         85  未使用 */
/* #define FF_DUST          86  未使用 */
/* #define FF_SLIME         87  未使用 */
#define FF_PLANT         88             /*!< 植物の生えた地形である */
/* #define FF_XXX2          89  未定義 */
/* #define FF_INSTANT       90  未使用 */
/* #define FF_EXPLODE       91  未使用 */
/* #define FF_TIMED         92  未使用 */
/* #define FF_ERUPT         93  未使用 */
/* #define FF_STRIKE        94  未使用 */
/* #define FF_SPREAD        95  未使用 */
#define FF_SPECIAL       96             /*!< クエストやダンジョンに関わる特別な地形である */
#define FF_HURT_DISI     97             /*!< 分解属性の対象となる地形である */
#define FF_QUEST_ENTER   98             /*!< クエストの入り口である */
#define FF_QUEST_EXIT    99             /*!< クエストの出口である */
#define FF_QUEST         100            /*!< クエストに関する地形である */
#define FF_SHAFT         101            /*!< 坑道である。(2階層移動する階段である) */
#define FF_MOUNTAIN      102            /*!< ダンジョンの山地形である */
#define FF_BLDG          103            /*!< 施設の入り口である */
#define FF_MINOR_GLYPH   104            /*!< 爆発のルーンのある地形である */
#define FF_PATTERN       105            /*!< パターンのある地形である */
#define FF_TOWN          106            /*!< 広域マップ用の街がある地形である */
#define FF_ENTRANCE      107            /*!< 広域マップ用のダンジョンがある地形である */
#define FF_MIRROR        108            /*!< 鏡使いの鏡が張られた地形である */
#define FF_UNPERM        109            /*!< 破壊不能な地形である(K:フラグ向け？) */
#define FF_TELEPORTABLE  110            /*!< テレポート先の対象となる地形である */
#define FF_CONVERT       111            /*!< 地形生成処理中の疑似フラグ */
#define FF_GLASS         112            /*!< ガラス製の地形である */


#define FF_FLAG_MAX      113
#define FF_FLAG_SIZE     (1 + ((FF_FLAG_MAX - 1) / 32))

/* Which features are dynamic */
#define have_dynamic_flags(ARRAY) \
	(!!((ARRAY)[(FF_INSTANT / 32)] & \
	    ((1UL << (FF_INSTANT % 32)) | \
	     (1UL << (FF_EXPLODE % 32)) | \
	     (1UL << (FF_TIMED % 32)) | \
	     (1UL << (FF_ERUPT % 32)) | \
	     (1UL << (FF_STRIKE % 32)) | \
	     (1UL << (FF_SPREAD % 32)))))


/*
 * Feature action flags
 */
#define FAF_DESTROY     0x01
#define FAF_NO_DROP     0x02
#define FAF_CRASH_GLASS 0x04


/*
 * Bit flags for teleportation
 */
#define TELEPORT_NONMAGICAL 0x00000001
#define TELEPORT_PASSIVE    0x00000002
#define TELEPORT_DEC_VALOUR 0x00000004


/* Types of doors */
#define DOOR_DEFAULT    -1
#define DOOR_DOOR        0
#define DOOR_GLASS_DOOR  1
#define DOOR_CURTAIN     2

#define MAX_DOOR_TYPES   3

#define feat_locked_door_random(DOOR_TYPE) \
	(feat_door[(DOOR_TYPE)].num_locked ? \
	 feat_door[(DOOR_TYPE)].locked[randint0(feat_door[(DOOR_TYPE)].num_locked)] : feat_none)

#define feat_jammed_door_random(DOOR_TYPE) \
	(feat_door[(DOOR_TYPE)].num_jammed ? \
	 feat_door[(DOOR_TYPE)].jammed[randint0(feat_door[(DOOR_TYPE)].num_jammed)] : feat_none)




/* Types of pattern tiles */
#define NOT_PATTERN_TILE      -1
#define PATTERN_TILE_START    0
#define PATTERN_TILE_1        1
#define PATTERN_TILE_2        2
#define PATTERN_TILE_3        3
#define PATTERN_TILE_4        4
#define PATTERN_TILE_END      5
#define PATTERN_TILE_OLD      6
#define PATTERN_TILE_TELEPORT 7
#define PATTERN_TILE_WRECKED  8


/* Types of conversions */
#define CONVERT_TYPE_FLOOR   0
#define CONVERT_TYPE_WALL    1
#define CONVERT_TYPE_INNER   2
#define CONVERT_TYPE_OUTER   3
#define CONVERT_TYPE_SOLID   4
#define CONVERT_TYPE_STREAM1 5
#define CONVERT_TYPE_STREAM2 6


/*
 * Bit flags for the *_can_enter() and monster_can_cross_terrain()
 */
#define CEM_RIDING              0x0001
#define CEM_P_CAN_ENTER_PATTERN 0x0002


/* Lighting levels of features' attr and char */

#define F_LIT_STANDARD 0 /* Standard */
#define F_LIT_LITE     1 /* Brightly lit */
#define F_LIT_DARK     2 /* Darkened */

#define F_LIT_NS_BEGIN 1 /* Nonstandard */
#define F_LIT_MAX      3



/*** Ego-Item indexes (see "lib/edit/e_info.txt") ***/


/* Nothing */
/* xxx */
/* xxx */
/* xxx */

/* Body Armor */
#define EGO_A_MORGUL            4
#define EGO_A_DEMON             5
#define EGO_DRUID               6
#define EGO_OLOG                7
#define EGO_RESISTANCE          8
#define EGO_ELVENKIND           9
#define EGO_DWARVEN             10
#define EGO_PERMANENCE          11
#define EGO_YOIYAMI             12
#define EGO_URUKISH             13
/* xxx */
/* xxx */

/* Shields */
#define EGO_ENDURE_ACID         16
#define EGO_ENDURE_ELEC         17
#define EGO_ENDURE_FIRE         18
#define EGO_ENDURE_COLD         19
#define EGO_ENDURANCE           20
#define EGO_REFLECTION          21
#define EGO_NIGHT_DAY           22
#define EGO_S_PROTECTION        238
#define EGO_S_DWARVEN           239

/* Crowns and Helms */
#define EGO_DARK                23
#define EGO_BRILLIANCE          24
#define EGO_H_PROTECTION        25
#define EGO_H_DEMON             26
#define EGO_MAGI                27
#define EGO_MIGHT               28
#define EGO_LORDLINESS          29
#define EGO_SEEING              30
#define EGO_INFRAVISION         31
#define EGO_LITE                32
#define EGO_TELEPATHY           33
#define EGO_REGENERATION        34
#define EGO_TELEPORTATION       35
#define EGO_FOOL                36
#define EGO_BASILISK            37
#define EGO_ANCIENT_CURSE       38
#define EGO_SICKLINESS          39

/* Cloaks */
#define EGO_PROTECTION          40
#define EGO_STEALTH             41
#define EGO_AMAN                42
#define EGO_AURA_FIRE           43
#define EGO_ENVELOPING          44
#define EGO_VULNERABILITY       45
#define EGO_IRRITATION          46
#define EGO_AURA_ELEC           47
#define EGO_AURA_COLD          128
#define EGO_BAT                129
#define EGO_NAZGUL             240

/* Gloves */
#define EGO_FREE_ACTION         48
#define EGO_SLAYING             49
#define EGO_AGILITY             50
#define EGO_POWER               51
#define EGO_2WEAPON             52
#define EGO_MAGIC_MASTERY       53
#define EGO_WEAKNESS            54
#define EGO_CLUMSINESS          55

/* Boots */
#define EGO_SLOW_DESCENT        56
#define EGO_QUIET               57
#define EGO_MOTION              58
#define EGO_SPEED               59
#define EGO_JUMP                60
#define EGO_NOISE               61
#define EGO_SLOWNESS            62
#define EGO_ANNOYANCE           63

/* Weapons */
#define EGO_HA                  64
#define EGO_DF                  65
#define EGO_BLESS_BLADE         66
/* xxx */
#define EGO_WEST                68
#define EGO_ATTACKS             69
#define EGO_SLAYING_WEAPON      70
#define EGO_FORCE_WEAPON        71
#define EGO_BRAND_ACID          72
#define EGO_BRAND_ELEC          73
#define EGO_BRAND_FIRE          74
#define EGO_BRAND_COLD          75
#define EGO_BRAND_POIS          76
#define EGO_CHAOTIC             77
#define EGO_SHARPNESS           78
#define EGO_EARTHQUAKES         79
#define EGO_XXX_1               80
#define EGO_XXX_2               81
#define EGO_XXX_3               82
#define EGO_XXX_4               83
#define EGO_XXX_5               84
#define EGO_XXX_6               85
#define EGO_XXX_7               86
#define EGO_WEIRD               87
#define EGO_KILL_ANIMAL         88
#define EGO_KILL_EVIL           89
#define EGO_KILL_UNDEAD         90
#define EGO_KILL_DEMON          91
#define EGO_KILL_ORC            92
#define EGO_KILL_TROLL          93
#define EGO_KILL_GIANT          94
#define EGO_KILL_DRAGON         95
#define EGO_VAMPIRIC            96
#define EGO_PRISM               97
#define EGO_TRUMP               98
#define EGO_PATTERN             99
#define EGO_DIGGING             100
#define EGO_DEMON               101
#define EGO_MORGUL              102
#define EGO_KILL_HUMAN          103

/* Bows */
#define EGO_ACCURACY            104
#define EGO_VELOCITY            105
/* xxx */
/* xxx */
#define EGO_EXTRA_MIGHT         108
#define EGO_EXTRA_SHOTS         109
/* xxx */
/* xxx */

/* Ammo */
#define EGO_HURT_ANIMAL         112
#define EGO_HURT_EVIL           113
/* xxx */
/* xxx */
/* xxx */
/* xxx */
/* xxx */
#define EGO_HURT_DRAGON         119
#define EGO_SLAYING_BOLT        120
#define EGO_LIGHTNING_BOLT      121
#define EGO_FLAME               122
#define EGO_FROST               123
#define EGO_WOUNDING            124
#define EGO_BACKBITING          125
#define EGO_SHATTERED           126
#define EGO_BLASTED             127

#define EGO_LITE_SHINE          140
#define EGO_LITE_ILLUMINATION   141
#define EGO_LITE_AURA_FIRE      142
#define EGO_LITE_INFRA          143
#define EGO_LITE_LONG           144
#define EGO_LITE_DARKNESS       145
#define EGO_LITE_EYE            146

#define EGO_RING_HERO           150
#define EGO_RING_SLAY           151
#define EGO_RING_SUPER_AC       152
#define EGO_RING_MAGIC_MIS      153
#define EGO_RING_FIRE_BOLT      154
#define EGO_RING_COLD_BOLT      155
#define EGO_RING_ELEC_BOLT      156
#define EGO_RING_ACID_BOLT      157
#define EGO_RING_MANA_BOLT      158
#define EGO_RING_FIRE_BALL      159
#define EGO_RING_COLD_BALL      160
#define EGO_RING_ELEC_BALL      161
#define EGO_RING_ACID_BALL      162
#define EGO_RING_MANA_BALL      163
#define EGO_RING_DRAGON_F       164
#define EGO_RING_DRAGON_C       165
#define EGO_RING_D_SPEED        166
#define EGO_RING_BERSERKER      167
#define EGO_RING_HUNTER         168
#define EGO_RING_THROW          169
#define EGO_RING_REGEN          170
#define EGO_RING_LITE           171
#define EGO_RING_M_DETECT       172
#define EGO_RING_STEALTH        173
#define EGO_RING_TELE_AWAY      174
#define EGO_RING_TO_H           175
#define EGO_RING_TO_D           176
#define EGO_RING_RES_LITE       177
#define EGO_RING_RES_DARK       178
#define EGO_RING_WIZARD         179
#define EGO_RING_TRUE           180
#define EGO_RING_DRAIN_EXP      181
#define EGO_RING_NO_MELEE       182
#define EGO_RING_AGGRAVATE      183
#define EGO_RING_TY_CURSE       184
#define EGO_RING_RES_TIME       185
#define EGO_RING_TELEPORT       186
#define EGO_RING_ALBINO         187

#define EGO_AMU_SLOW_D          210
#define EGO_AMU_INFRA           211
#define EGO_AMU_SEE_INVIS       212
#define EGO_AMU_HOLD_EXP        213
#define EGO_AMU_DRAIN_EXP       214
#define EGO_AMU_FOOL            215
#define EGO_AMU_AGGRAVATE       216
#define EGO_AMU_TY_CURSE        217
#define EGO_AMU_AC              218
#define EGO_AMU_IDENT           219
#define EGO_AMU_CHARM           220
#define EGO_AMU_STEALTH         221
#define EGO_AMU_JUMP            222
#define EGO_AMU_TELEPORT        223
#define EGO_AMU_D_DOOR          224
#define EGO_AMU_DEFENDER        225
#define EGO_AMU_RES_FIRE        226
#define EGO_AMU_RES_FIRE_       227
#define EGO_AMU_RES_COLD        228
#define EGO_AMU_RES_COLD_       229
#define EGO_AMU_RES_ELEC        230
#define EGO_AMU_RES_ELEC_       231
#define EGO_AMU_RES_ACID        232
#define EGO_AMU_RES_ACID_       233
#define EGO_AMU_LEVITATION      234
#define EGO_AMU_GREAT           235
#define EGO_AMU_DETECTION       236
#define EGO_AMU_NAIVETY         237
// MAX 240


/*** Object "tval" and "sval" codes ***/


/*
 * The values for the "tval" field of various objects.
 *
 * This value is the primary means by which items are sorted in the
 * player p_ptr->inventory_list, followed by "sval" and "cost".
 *
 * Note that a "BOW" with tval = 19 and sval S = 10*N+P takes a missile
 * weapon with tval = 16+N, and does (xP) damage when so combined.  This
 * fact is not actually used in the source, but it kind of interesting.
 *
 * Note that as of 2.7.8, the "item flags" apply to all items, though
 * only armor and weapons and a few other items use any of these flags.
 */

#define TV_SKELETON      1      /* Skeletons ('s'), not specified */
#define TV_BOTTLE        2      /* Empty bottles ('!') */
#define TV_JUNK          3      /* Sticks, Pottery, etc ('~') */
#define TV_WHISTLE       4      /* Whistle ('~') */
#define TV_SPIKE         5      /* Spikes ('~') */
#define TV_CHEST         7      /* Chests ('&') */
#define TV_FIGURINE      8      /* Magical figurines */
#define TV_STATUE        9      /* Statue, what a silly object... */
#define TV_CORPSE       10      /* Corpses and Skeletons, specific */
#define TV_CAPTURE      11      /* Monster ball */
#define TV_NO_AMMO      15      /* Ammo for crimson */
#define TV_SHOT         16      /* Ammo for slings */
#define TV_ARROW        17      /* Ammo for bows */
#define TV_BOLT         18      /* Ammo for x-bows */
#define TV_BOW          19      /* Slings/Bows/Xbows */
#define TV_DIGGING      20      /* Shovels/Picks */
#define TV_HAFTED       21      /* Priest Weapons */
#define TV_POLEARM      22      /* Axes and Pikes */
#define TV_SWORD        23      /* Edged Weapons */
#define TV_BOOTS        30      /* Boots */
#define TV_GLOVES       31      /* Gloves */
#define TV_HELM         32      /* Helms */
#define TV_CROWN        33      /* Crowns */
#define TV_SHIELD       34      /* Shields */
#define TV_CLOAK        35      /* Cloaks */
#define TV_SOFT_ARMOR   36      /* Soft Armor */
#define TV_HARD_ARMOR   37      /* Hard Armor */
#define TV_DRAG_ARMOR   38      /* Dragon Scale Mail */
#define TV_LITE         39      /* Lites (including Specials) */
#define TV_AMULET       40      /* Amulets (including Specials) */
#define TV_RING         45      /* Rings (including Specials) */
#define TV_CARD         50
#define TV_STAFF        55
#define TV_WAND         65
#define TV_ROD          66
#define TV_PARCHMENT    69
#define TV_SCROLL       70
#define TV_POTION       75
#define TV_FLASK        77
#define TV_FOOD         80
#define TV_LIFE_BOOK    90
#define TV_SORCERY_BOOK 91
#define TV_NATURE_BOOK  92
#define TV_CHAOS_BOOK   93
#define TV_DEATH_BOOK   94
#define TV_TRUMP_BOOK   95
#define TV_ARCANE_BOOK  96
#define TV_CRAFT_BOOK 97
#define TV_DAEMON_BOOK  98
#define TV_CRUSADE_BOOK 99
#define TV_MUSIC_BOOK   105
#define TV_HISSATSU_BOOK 106
#define TV_HEX_BOOK     107
#define TV_GOLD         127     /* Gold can only be picked up by players */

#define TV_EQUIP_BEGIN    TV_SHOT
#define TV_EQUIP_END      TV_CARD
#define TV_MISSILE_BEGIN  TV_SHOT
#define TV_MISSILE_END    TV_BOLT
#define TV_WEARABLE_BEGIN TV_BOW
#define TV_WEARABLE_END   TV_CARD
#define TV_WEAPON_BEGIN   TV_BOW
#define TV_WEAPON_END     TV_SWORD
#define TV_ARMOR_BEGIN    TV_BOOTS
#define TV_ARMOR_END      TV_DRAG_ARMOR

/* Any subvalue */
#define SV_ANY 					255

/* The "sval" codes for TV_FIGURINE */
#define SV_FIGURINE_NORMAL		0

#define SV_CAPTURE_NONE		0

/* The "sval" codes for TV_STATUE */
#define SV_WOODEN_STATUE		0
#define SV_CLAY_STATUE			1
#define SV_STONE_STATUE			2
#define SV_IRON_STATUE			3
#define SV_COPPER_STATUE		4
#define SV_SILVER_STATUE		5
#define SV_GOLDEN_STATUE		6
#define SV_IVORY_STATUE			7
#define SV_MITHRIL_STATUE		8
#define SV_ORNATE_STATUE		9
#define SV_PHOTO			50

/* The "sval" codes for TV_CORPSE */
#define SV_SKELETON 			0
#define SV_CORPSE			1

/* The "sval" codes for TV_SHOT/TV_ARROW/TV_BOLT */
#define SV_AMMO_LIGHT                    0	/* pebbles */
#define SV_AMMO_NORMAL                   1	/* shots, arrows, bolts */
#define SV_AMMO_HEAVY                    2	/* seeker arrows and bolts, mithril shots */

/* The "sval" codes for TV_BOW (note information in "sval") */
#define SV_SLING                         2	/* (x2) */
#define SV_SHORT_BOW                    12	/* (x2) */
#define SV_LONG_BOW                     13	/* (x3) */
#define SV_LIGHT_XBOW                   23	/* (x3) */
#define SV_HEAVY_XBOW                   24	/* (x4) */
#define SV_CRIMSON                      50	/* (x0) */
#define SV_HARP                         51	/* (x0) */
#define SV_NAMAKE_BOW                   63	/* (x3) */

/* The "sval" codes for TV_DIGGING */
#define SV_SHOVEL                        1
#define SV_GNOMISH_SHOVEL                2
#define SV_DWARVEN_SHOVEL                3
#define SV_PICK                          4
#define SV_ORCISH_PICK                   5
#define SV_DWARVEN_PICK                  6
#define SV_MATTOCK                       7

/* The "sval" values for TV_HAFTED */
#define SV_CLUB                          1	/* 1d4  */
#define SV_WHIP                          2	/* 1d6  */
#define SV_QUARTERSTAFF                  3	/* 1d9  */
#define SV_NUNCHAKU                      4	/* 2d3  */
#define SV_MACE                          5	/* 2d4  */
#define SV_BALL_AND_CHAIN                6	/* 2d4  */
#define SV_JO_STAFF                      7	/* 1d7  */
#define SV_WAR_HAMMER                    8	/* 3d3  */
#define SV_THREE_PIECE_ROD              11	/* 3d3  */
#define SV_MORNING_STAR                 12	/* 2d6  */
#define SV_FLAIL                        13	/* 2d6  */
#define SV_BO_STAFF                     14	/* 1d11 */
#define SV_LEAD_FILLED_MACE             15	/* 3d4  */
#define SV_TETSUBO                      16	/* 2d7  */
#define SV_TWO_HANDED_FLAIL             18	/* 3d6  */
#define SV_GREAT_HAMMER                 19	/* 4d6  */
#define SV_MACE_OF_DISRUPTION           20	/* 5d8  */
#define SV_WIZSTAFF                     21	/* 1d2  */
#define SV_GROND                        50	/* 3d9  */
#define SV_NAMAKE_HAMMER                63	/* 1d77 */

/* The "sval" values for TV_POLEARM */
#define SV_HATCHET                       1	/* 1d5 */
#define SV_SPEAR                         2	/* 1d6 */
#define SV_SICKLE                        3	/* 2d3 */
#define SV_AWL_PIKE                      4	/* 1d8 */
#define SV_TRIDENT                       5	/* 1d9 */
#define SV_FAUCHARD                      6  /* 1d10 */
#define SV_BROAD_SPEAR                   7	/* 1d9 */
#define SV_PIKE                          8	/* 2d5 */
#define SV_NAGINATA                      9  /* 2d6 */
#define SV_BEAKED_AXE                   10	/* 2d6 */
#define SV_BROAD_AXE                    11	/* 2d6 */
#define SV_LUCERNE_HAMMER               12	/* 2d5  */
#define SV_GLAIVE                       13	/* 2d6 */
#define SV_LAJATANG                     14	/* 2d7 */
#define SV_HALBERD                      15	/* 3d4 */
#define SV_GUISARME                     16  /* 2d5 */
#define SV_SCYTHE                       17	/* 5d3 */
#define SV_LANCE                        20	/* 2d8 */
#define SV_BATTLE_AXE                   22	/* 2d8 */
#define SV_GREAT_AXE                    25	/* 4d4 */
#define SV_TRIFURCATE_SPEAR             26	/* 2d9 */
#define SV_LOCHABER_AXE                 28	/* 3d8 */
#define SV_HEAVY_LANCE                  29  /* 4d8 */
#define SV_SCYTHE_OF_SLICING            30	/* 8d4 */
#define SV_TSURIZAO                     40	/* 1d1 */
#define SV_DEATH_SCYTHE                 50	/* 10d10 */

/* The "sval" codes for TV_SWORD */
#define SV_BROKEN_DAGGER                 1  /* 1d1 */
#define SV_BROKEN_SWORD                  2  /* 1d2 */
#define SV_DAGGER                        4  /* 1d4 */
#define SV_MAIN_GAUCHE                   5  /* 1d5 */
#define SV_TANTO                         6  /* 1d5 */
#define SV_RAPIER                        7  /* 1d6 */
#define SV_SMALL_SWORD                   8  /* 1d6 */
#define SV_BASILLARD                     9  /* 1d8 */
#define SV_SHORT_SWORD                  10  /* 1d7 */
#define SV_SABRE                        11  /* 1d7 */
#define SV_CUTLASS                      12  /* 1d7 */
#define SV_WAKIZASHI                    13  /* 2d4 */
#define SV_KHOPESH                      14  /* 2d4 */
#define SV_TULWAR                       15  /* 2d4 */
#define SV_BROAD_SWORD                  16  /* 2d5 */
#define SV_LONG_SWORD                   17  /* 2d5 */
#define SV_SCIMITAR                     18  /* 2d5 */
#define SV_NINJATO                      19  /* 1d9 */
#define SV_KATANA                       20  /* 3d4 */
#define SV_BASTARD_SWORD                21  /* 3d4 */
#define SV_GREAT_SCIMITAR               22  /* 4d5 */
#define SV_CLAYMORE                     23  /* 2d8 */
#define SV_ESPADON                      24  /* 2d9 */
#define SV_TWO_HANDED_SWORD             25  /* 3d6 */
#define SV_FLAMBERGE                    26  /* 3d7 */
#define SV_NO_DACHI                     27  /* 5d4 */
#define SV_EXECUTIONERS_SWORD           28  /* 4d5 */
#define SV_ZWEIHANDER                   29  /* 4d6 */
#define SV_BLADE_OF_CHAOS               30  /* 6d5 */
#define SV_DIAMOND_EDGE                 31  /* 7d5 */
#define SV_DOKUBARI                     32  /* 1d1 */
#define SV_HAYABUSA                     33  /* 1d6 */

/* The "sval" codes for TV_SHIELD */
#define SV_SMALL_LEATHER_SHIELD          2
#define SV_SMALL_METAL_SHIELD            3
#define SV_LARGE_LEATHER_SHIELD          4
#define SV_LARGE_METAL_SHIELD            5
#define SV_DRAGON_SHIELD                 6
#define SV_KNIGHT_SHIELD                 7
#define SV_MIRROR_SHIELD                10
#define SV_YATA_MIRROR                  50

/* The "sval" codes for TV_HELM */
#define SV_HARD_LEATHER_CAP              2
#define SV_METAL_CAP                     3
#define SV_JINGASA                       4  /* 4 */
#define SV_IRON_HELM                     5
#define SV_STEEL_HELM                    6
#define SV_DRAGON_HELM                   7
#define SV_KABUTO                        8  /* 7 */

/* The "sval" codes for TV_CROWN */
#define SV_IRON_CROWN                   10
#define SV_GOLDEN_CROWN                 11
#define SV_JEWELED_CROWN                12
#define SV_CHAOS                        50

/* The "sval" codes for TV_BOOTS */
#define SV_PAIR_OF_SOFT_LEATHER_BOOTS    2
#define SV_PAIR_OF_HARD_LEATHER_BOOTS    3
#define SV_PAIR_OF_DRAGON_GREAVE         4
#define SV_PAIR_OF_METAL_SHOD_BOOTS      6

/* The "sval" codes for TV_CLOAK */
#define SV_CLOAK                         1
#define SV_ELVEN_CLOAK                   2
#define SV_FUR_CLOAK                     3
#define SV_ETHEREAL_CLOAK                5
#define SV_SHADOW_CLOAK                  6

/* The "sval" codes for TV_GLOVES */
#define SV_SET_OF_LEATHER_GLOVES         1
#define SV_SET_OF_GAUNTLETS              2
#define SV_SET_OF_DRAGON_GLOVES          3
#define SV_SET_OF_CESTI                  5

/* The "sval" codes for TV_SOFT_ARMOR */
#define SV_T_SHIRT                       0
#define SV_FILTHY_RAG                    1
#define SV_ROBE                          2
#define SV_PAPER_ARMOR                   3  /* 4 */
#define SV_SOFT_LEATHER_ARMOR            4
#define SV_SOFT_STUDDED_LEATHER          5
#define SV_HARD_LEATHER_ARMOR            6
#define SV_HARD_STUDDED_LEATHER          7
#define SV_RHINO_HIDE_ARMOR              8
#define SV_CORD_ARMOR                    9  /*  6 */
#define SV_PADDED_ARMOR                 10  /*  4 */
#define SV_LEATHER_SCALE_MAIL           11
#define SV_LEATHER_JACK                 12
#define SV_KUROSHOUZOKU                 13
#define SV_STONE_AND_HIDE_ARMOR         15  /* 15 */
#define SV_ABUNAI_MIZUGI                50
#define SV_YOIYAMI_ROBE                 60
#define SV_NAMAKE_ARMOR                 63

/* The "sval" codes for TV_HARD_ARMOR */
#define SV_RUSTY_CHAIN_MAIL              1  /* 14- */
#define SV_RING_MAIL                     2  /* 12  */
#define SV_METAL_SCALE_MAIL              3  /* 13  */
#define SV_CHAIN_MAIL                    4  /* 14  */
#define SV_DOUBLE_RING_MAIL              5  /* 15  */
#define SV_AUGMENTED_CHAIN_MAIL          6  /* 16  */
#define SV_DOUBLE_CHAIN_MAIL             7  /* 16  */
#define SV_BAR_CHAIN_MAIL                8  /* 18  */
#define SV_METAL_BRIGANDINE_ARMOUR       9  /* 19  */
#define SV_SPLINT_MAIL                  10  /* 19  */
#define SV_DO_MARU                      11  /* 20  */
#define SV_PARTIAL_PLATE_ARMOUR         12  /* 22  */
#define SV_METAL_LAMELLAR_ARMOUR        13  /* 23  */
#define SV_HARAMAKIDO                   14  /* 17  */
#define SV_FULL_PLATE_ARMOUR            15  /* 25  */
#define SV_O_YOROI                      16  /* 24  */
#define SV_RIBBED_PLATE_ARMOUR          18  /* 28  */
#define SV_MITHRIL_CHAIN_MAIL           20  /* 28+ */
#define SV_MITHRIL_PLATE_MAIL           25  /* 35+ */
#define SV_ADAMANTITE_PLATE_MAIL        30  /* 40+ */

/* The "sval" codes for TV_DRAG_ARMOR */
#define SV_DRAGON_BLACK                  1
#define SV_DRAGON_BLUE                   2
#define SV_DRAGON_WHITE                  3
#define SV_DRAGON_RED                    4
#define SV_DRAGON_GREEN                  5
#define SV_DRAGON_MULTIHUED              6
#define SV_DRAGON_SHINING               10
#define SV_DRAGON_LAW                   12
#define SV_DRAGON_BRONZE                14
#define SV_DRAGON_GOLD                  16
#define SV_DRAGON_CHAOS                 18
#define SV_DRAGON_BALANCE               20
#define SV_DRAGON_POWER                 30

/* The sval codes for TV_LITE */
#define SV_LITE_TORCH                    0
#define SV_LITE_LANTERN                  1
#define SV_LITE_FEANOR                   2
#define SV_LITE_EDISON                   3
#define SV_LITE_GALADRIEL                4
#define SV_LITE_ELENDIL                  5
#define SV_LITE_JUDGE                    6
#define SV_LITE_LORE                     7
#define SV_LITE_PALANTIR                 8
#define SV_LITE_FLY_STONE                9

/* The "sval" codes for TV_AMULET */
#define SV_AMULET_DOOM                   0
#define SV_AMULET_TELEPORT               1
#define SV_AMULET_ADORNMENT              2
#define SV_AMULET_SLOW_DIGEST            3
#define SV_AMULET_RESIST_ACID            4
#define SV_AMULET_SEARCHING              5
#define SV_AMULET_BRILLIANCE             6
#define SV_AMULET_CHARISMA               7
#define SV_AMULET_THE_MAGI               8
#define SV_AMULET_REFLECTION             9
#define SV_AMULET_CARLAMMAS             10
#define SV_AMULET_INGWE                 11
#define SV_AMULET_DWARVES               12
#define SV_AMULET_NO_MAGIC              13
#define SV_AMULET_NO_TELE               14
#define SV_AMULET_RESISTANCE            15
#define SV_AMULET_TELEPATHY             16
#define SV_AMULET_FARAMIR               17
#define SV_AMULET_BOROMIR               18
#define SV_AMULET_MAGATAMA              19
#define SV_AMULET_INROU                 20
#define SV_AMULET_INTELLIGENCE          21
#define SV_AMULET_WISDOM                22
#define SV_AMULET_MAGIC_MASTERY         23
#define SV_AMULET_NIGHT                 24

/* The sval codes for TV_RING */
#define SV_RING_WOE                      0
#define SV_RING_AGGRAVATION              1
#define SV_RING_WEAKNESS                 2
#define SV_RING_STUPIDITY                3
#define SV_RING_TELEPORTATION            4
#define SV_RING_SLOW_DIGESTION           6
#define SV_RING_LEVITATION_FALL             7
#define SV_RING_RESIST_FIRE              8
#define SV_RING_RESIST_COLD              9
#define SV_RING_SUSTAIN_STR             10
#define SV_RING_SUSTAIN_INT             11
#define SV_RING_SUSTAIN_WIS             12
#define SV_RING_SUSTAIN_CON             13
#define SV_RING_SUSTAIN_DEX             14
#define SV_RING_SUSTAIN_CHR             15
#define SV_RING_PROTECTION              16
#define SV_RING_ACID                    17
#define SV_RING_FLAMES                  18
#define SV_RING_ICE                     19
#define SV_RING_RESIST_POIS             20
#define SV_RING_FREE_ACTION             21
#define SV_RING_SEE_INVIS               22
#define SV_RING_SEARCHING               23
#define SV_RING_STR                     24
#define SV_RING_ELEC                    25
#define SV_RING_DEX                     26
#define SV_RING_CON                     27
#define SV_RING_ACCURACY                28
#define SV_RING_DAMAGE                  29
#define SV_RING_SLAYING                 30
#define SV_RING_SPEED                   31
#define SV_RING_FRAKIR                  32
#define SV_RING_TULKAS                  33
#define SV_RING_NARYA                   34
#define SV_RING_NENYA                   35
#define SV_RING_VILYA                   36
#define SV_RING_POWER                   37
#define SV_RING_RES_FEAR                38
#define SV_RING_RES_LD                  39
#define SV_RING_RES_NETHER              40
#define SV_RING_RES_NEXUS               41
#define SV_RING_RES_SOUND               42
#define SV_RING_RES_CONFUSION           43
#define SV_RING_RES_SHARDS              44
#define SV_RING_RES_DISENCHANT          45
#define SV_RING_RES_CHAOS               46
#define SV_RING_RES_BLINDNESS           47
#define SV_RING_LORDLY                  48
#define SV_RING_ATTACKS                 49
#define SV_RING_AHO                     50
#define SV_RING_SHOTS                   51
#define SV_RING_SUSTAIN                 52
#define SV_RING_DEC_MANA                53
#define SV_RING_WARNING                 54
#define SV_RING_MUSCLE                  55

#define SV_EXPRESS_CARD                  0

/* The "sval" codes for TV_STAFF */
#define SV_STAFF_DARKNESS                0
#define SV_STAFF_SLOWNESS                1
#define SV_STAFF_HASTE_MONSTERS          2
#define SV_STAFF_SUMMONING               3
#define SV_STAFF_TELEPORTATION           4
#define SV_STAFF_IDENTIFY                5
#define SV_STAFF_REMOVE_CURSE            6
#define SV_STAFF_STARLITE                7
#define SV_STAFF_LITE                    8
#define SV_STAFF_MAPPING                 9
#define SV_STAFF_DETECT_GOLD            10
#define SV_STAFF_DETECT_ITEM            11
#define SV_STAFF_DETECT_TRAP            12
#define SV_STAFF_DETECT_DOOR            13
#define SV_STAFF_DETECT_INVIS           14
#define SV_STAFF_DETECT_EVIL            15
#define SV_STAFF_CURE_LIGHT             16
#define SV_STAFF_CURING                 17
#define SV_STAFF_HEALING                18
#define SV_STAFF_THE_MAGI               19
#define SV_STAFF_SLEEP_MONSTERS         20
#define SV_STAFF_SLOW_MONSTERS          21
#define SV_STAFF_SPEED                  22
#define SV_STAFF_PROBING                23
#define SV_STAFF_DISPEL_EVIL            24
#define SV_STAFF_POWER                  25
#define SV_STAFF_HOLINESS               26
#define SV_STAFF_GENOCIDE               27
#define SV_STAFF_EARTHQUAKES            28
#define SV_STAFF_DESTRUCTION            29
#define SV_STAFF_ANIMATE_DEAD           30
#define SV_STAFF_MSTORM                 31
#define SV_STAFF_NOTHING                32


/* The "sval" codes for TV_WAND */
#define SV_WAND_HEAL_MONSTER             0
#define SV_WAND_HASTE_MONSTER            1
#define SV_WAND_CLONE_MONSTER            2
#define SV_WAND_TELEPORT_AWAY            3
#define SV_WAND_DISARMING                4
#define SV_WAND_TRAP_DOOR_DEST           5
#define SV_WAND_STONE_TO_MUD             6
#define SV_WAND_LITE                     7
#define SV_WAND_SLEEP_MONSTER            8
#define SV_WAND_SLOW_MONSTER             9
#define SV_WAND_CONFUSE_MONSTER         10
#define SV_WAND_FEAR_MONSTER            11
#define SV_WAND_HYPODYNAMIA              12
#define SV_WAND_POLYMORPH               13
#define SV_WAND_STINKING_CLOUD          14
#define SV_WAND_MAGIC_MISSILE           15
#define SV_WAND_ACID_BOLT               16
#define SV_WAND_CHARM_MONSTER           17
#define SV_WAND_FIRE_BOLT               18
#define SV_WAND_COLD_BOLT               19
#define SV_WAND_ACID_BALL               20
#define SV_WAND_ELEC_BALL               21
#define SV_WAND_FIRE_BALL               22
#define SV_WAND_COLD_BALL               23
#define SV_WAND_WONDER                  24
#define SV_WAND_DISINTEGRATE            25
#define SV_WAND_DRAGON_FIRE             26
#define SV_WAND_DRAGON_COLD             27
#define SV_WAND_DRAGON_BREATH           28
#define SV_WAND_ROCKETS                 29
#define SV_WAND_STRIKING                30
#define SV_WAND_GENOCIDE                31

/* The "sval" codes for TV_ROD */
#define SV_ROD_DETECT_TRAP               0
#define SV_ROD_DETECT_DOOR               1
#define SV_ROD_IDENTIFY                  2
#define SV_ROD_RECALL                    3
#define SV_ROD_ILLUMINATION              4
#define SV_ROD_MAPPING                   5
#define SV_ROD_DETECTION                 6
#define SV_ROD_PROBING                   7
#define SV_ROD_CURING                    8
#define SV_ROD_HEALING                   9
#define SV_ROD_RESTORATION              10
#define SV_ROD_SPEED                    11
#define SV_ROD_PESTICIDE                12
#define SV_ROD_TELEPORT_AWAY            13
#define SV_ROD_DISARMING                14
#define SV_ROD_LITE                     15
#define SV_ROD_SLEEP_MONSTER            16
#define SV_ROD_SLOW_MONSTER             17
#define SV_ROD_HYPODYNAMIA               18
#define SV_ROD_POLYMORPH                19
#define SV_ROD_ACID_BOLT                20
#define SV_ROD_ELEC_BOLT                21
#define SV_ROD_FIRE_BOLT                22
#define SV_ROD_COLD_BOLT                23
#define SV_ROD_ACID_BALL                24
#define SV_ROD_ELEC_BALL                25
#define SV_ROD_FIRE_BALL                26
#define SV_ROD_COLD_BALL                27
#define SV_ROD_HAVOC                    28
#define SV_ROD_STONE_TO_MUD             29
#define SV_ROD_AGGRAVATE                30


/* The "sval" codes for TV_SCROLL */

#define SV_SCROLL_DARKNESS               0
#define SV_SCROLL_AGGRAVATE_MONSTER      1
#define SV_SCROLL_CURSE_ARMOR            2
#define SV_SCROLL_CURSE_WEAPON           3
#define SV_SCROLL_SUMMON_MONSTER         4
#define SV_SCROLL_SUMMON_UNDEAD          5
#define SV_SCROLL_SUMMON_PET             6
#define SV_SCROLL_TRAP_CREATION          7
#define SV_SCROLL_PHASE_DOOR             8
#define SV_SCROLL_TELEPORT               9
#define SV_SCROLL_TELEPORT_LEVEL        10
#define SV_SCROLL_WORD_OF_RECALL        11
#define SV_SCROLL_IDENTIFY              12
#define SV_SCROLL_STAR_IDENTIFY         13
#define SV_SCROLL_REMOVE_CURSE          14
#define SV_SCROLL_STAR_REMOVE_CURSE     15
#define SV_SCROLL_ENCHANT_ARMOR         16
#define SV_SCROLL_ENCHANT_WEAPON_TO_HIT 17
#define SV_SCROLL_ENCHANT_WEAPON_TO_DAM 18
/* xxx enchant missile? */
#define SV_SCROLL_STAR_ENCHANT_ARMOR    20
#define SV_SCROLL_STAR_ENCHANT_WEAPON   21
#define SV_SCROLL_RECHARGING            22
#define SV_SCROLL_MUNDANITY             23
#define SV_SCROLL_LIGHT                 24
#define SV_SCROLL_MAPPING               25
#define SV_SCROLL_DETECT_GOLD           26
#define SV_SCROLL_DETECT_ITEM           27
#define SV_SCROLL_DETECT_TRAP           28
#define SV_SCROLL_DETECT_DOOR           29
#define SV_SCROLL_DETECT_INVIS          30
/* xxx (detect evil?) */
#define SV_SCROLL_SATISFY_HUNGER        32
#define SV_SCROLL_BLESSING              33
#define SV_SCROLL_HOLY_CHANT            34
#define SV_SCROLL_HOLY_PRAYER           35
#define SV_SCROLL_MONSTER_CONFUSION     36
#define SV_SCROLL_PROTECTION_FROM_EVIL  37
#define SV_SCROLL_RUNE_OF_PROTECTION    38
#define SV_SCROLL_TRAP_DOOR_DESTRUCTION 39
/* xxx */
#define SV_SCROLL_STAR_DESTRUCTION      41
#define SV_SCROLL_DISPEL_UNDEAD         42
#define SV_SCROLL_SPELL                 43
#define SV_SCROLL_GENOCIDE              44
#define SV_SCROLL_MASS_GENOCIDE         45
#define SV_SCROLL_ACQUIREMENT           46
#define SV_SCROLL_STAR_ACQUIREMENT      47
#define SV_SCROLL_FIRE                  48
#define SV_SCROLL_ICE                   49
#define SV_SCROLL_CHAOS                 50
#define SV_SCROLL_RUMOR                 51
#define SV_SCROLL_ARTIFACT              52
#define SV_SCROLL_RESET_RECALL          53
#define SV_SCROLL_SUMMON_KIN            54
#define SV_SCROLL_AMUSEMENT             55
#define SV_SCROLL_STAR_AMUSEMENT        56

/* The "sval" codes for TV_POTION */
#define SV_POTION_WATER                  0
#define SV_POTION_APPLE_JUICE            1
#define SV_POTION_SLIME_MOLD             2
/* xxx (fixed color) */
#define SV_POTION_SLOWNESS               4
#define SV_POTION_SALT_WATER             5
#define SV_POTION_POISON                 6
#define SV_POTION_BLINDNESS              7
/* xxx */
#define SV_POTION_BOOZE              9
/* xxx */
#define SV_POTION_SLEEP                 11
/* xxx */
#define SV_POTION_LOSE_MEMORIES         13
/* xxx */
#define SV_POTION_RUINATION             15
#define SV_POTION_DEC_STR               16
#define SV_POTION_DEC_INT               17
#define SV_POTION_DEC_WIS               18
#define SV_POTION_DEC_DEX               19
#define SV_POTION_DEC_CON               20
#define SV_POTION_DEC_CHR               21
#define SV_POTION_DETONATIONS           22
#define SV_POTION_DEATH                 23
#define SV_POTION_INFRAVISION           24
#define SV_POTION_DETECT_INVIS          25
#define SV_POTION_SLOW_POISON           26
#define SV_POTION_CURE_POISON           27
#define SV_POTION_BOLDNESS              28
#define SV_POTION_SPEED                 29
#define SV_POTION_RESIST_HEAT           30
#define SV_POTION_RESIST_COLD           31
#define SV_POTION_HEROISM               32
#define SV_POTION_BESERK_STRENGTH       33
#define SV_POTION_CURE_LIGHT            34
#define SV_POTION_CURE_SERIOUS          35
#define SV_POTION_CURE_CRITICAL         36
#define SV_POTION_HEALING               37
#define SV_POTION_STAR_HEALING          38
#define SV_POTION_LIFE                  39
#define SV_POTION_RESTORE_MANA          40
#define SV_POTION_RESTORE_EXP           41
#define SV_POTION_RES_STR               42
#define SV_POTION_RES_INT               43
#define SV_POTION_RES_WIS               44
#define SV_POTION_RES_DEX               45
#define SV_POTION_RES_CON               46
#define SV_POTION_RES_CHR               47
#define SV_POTION_INC_STR               48
#define SV_POTION_INC_INT               49
#define SV_POTION_INC_WIS               50
#define SV_POTION_INC_DEX               51
#define SV_POTION_INC_CON               52
#define SV_POTION_INC_CHR               53
/* xxx */
#define SV_POTION_AUGMENTATION          55
#define SV_POTION_ENLIGHTENMENT         56
#define SV_POTION_STAR_ENLIGHTENMENT    57
#define SV_POTION_SELF_KNOWLEDGE        58
#define SV_POTION_EXPERIENCE            59
#define SV_POTION_RESISTANCE            60
#define SV_POTION_CURING                61
#define SV_POTION_INVULNERABILITY       62
#define SV_POTION_NEW_LIFE              63
#define SV_POTION_NEO_TSUYOSHI          64
#define SV_POTION_TSUYOSHI              65
#define SV_POTION_POLYMORPH             66

/* The "sval" codes for TV_FLASK */
#define SV_FLASK_OIL                   0

/* The "sval" codes for TV_FOOD */
#define SV_FOOD_POISON                   0
#define SV_FOOD_BLINDNESS                1
#define SV_FOOD_PARANOIA                 2
#define SV_FOOD_CONFUSION                3
#define SV_FOOD_HALLUCINATION            4
#define SV_FOOD_PARALYSIS                5
#define SV_FOOD_WEAKNESS                 6
#define SV_FOOD_SICKNESS                 7
#define SV_FOOD_STUPIDITY                8
#define SV_FOOD_NAIVETY                  9
#define SV_FOOD_UNHEALTH                10
#define SV_FOOD_DISEASE                 11
#define SV_FOOD_CURE_POISON             12
#define SV_FOOD_CURE_BLINDNESS          13
#define SV_FOOD_CURE_PARANOIA           14
#define SV_FOOD_CURE_CONFUSION          15
#define SV_FOOD_CURE_SERIOUS            16
#define SV_FOOD_RESTORE_STR             17
#define SV_FOOD_RESTORE_CON             18
#define SV_FOOD_RESTORING               19
/* many missing mushrooms */
#define SV_FOOD_BISCUIT                 32
#define SV_FOOD_JERKY                   33
#define SV_FOOD_RATION                  35
#define SV_FOOD_SLIME_MOLD              36
#define SV_FOOD_WAYBREAD                37
#define SV_FOOD_PINT_OF_ALE             38
#define SV_FOOD_PINT_OF_WINE            39


#define SV_ROD_MIN_DIRECTION    12 /*!< この値以降の小項目IDを持ったロッドは使用時にターゲットを要求する / Special "sval" limit -- first "aimed" rod */

#define SV_CHEST_MIN_LARGE      4  /*!< この値以降の小項目IDを持った箱は大型の箱としてドロップ数を増やす / Special "sval" limit -- first "large" chest */
#define SV_CHEST_KANDUME        50 /*!< 箱アイテムの小項目ID: おもちゃのカンヅメ */

/*
 * Special "sval" limit -- first "good" magic/prayer book
 */
#define SV_BOOK_MIN_GOOD    2


#define OBJ_GOLD_LIST   480     /* First "gold" entry */
#define MAX_GOLD        18      /* Number of "gold" entries */

/*** General flag values ***/

/*
 * Special caster ID for project()
 */
#define PROJECT_WHO_UNCTRL_POWER -1 /*!< 魔法効果の自然発生要因: 名状し難い力の解放 */
#define PROJECT_WHO_GLASS_SHARDS -2 /*!< 魔法効果の自然発生要因: 破壊されたガラス地形の破片 */


/*
 * Bit flags for the "enchant()" function
 */
#define ENCH_TOHIT      0x01 /*!< 装備強化処理: 命中強化 / Enchant to hit */
#define ENCH_TODAM      0x02 /*!< 装備強化処理: ダメージ強化 / Enchant to damage */
#define ENCH_TOAC       0x04 /*!< 装備強化処理: AC強化 / Enchant to AC */
#define ENCH_FORCE      0x08 /*!< 装備強化処理: 無条件に成功させる / Force enchantment */

/*
 * target_set用関数の利用用途フラグ / Bit flags for the "target_set" function
 */
#define TARGET_KILL     0x01 /*!< モンスターへの狙いをつける(視界内モンスターのみクエリ対象) / Target monsters */
#define TARGET_LOOK     0x02 /*!< "L"ookコマンド向けの既存情報確認向け(全ての有為な情報をクエリ対象) / Describe grid fully */
#define TARGET_XTRA     0x04 /*!< 現在未使用 / Currently unused flag */
#define TARGET_GRID     0x08 /*!< 全てのマス対象にする(現在未使用) / Select from all grids */


/*
 * Bit flags for control of get_check_strict()
 */
#define CHECK_OKAY_CANCEL 0x01
#define CHECK_NO_ESCAPE   0x02
#define CHECK_NO_HISTORY  0x04
#define CHECK_DEFAULT_Y   0x08




/*
 * get_item()関数でアイテムの選択を行うフラグ / Bit flags for the "get_item" function
 */
#define USE_EQUIP 0x01 /*!< アイテム表示/選択範囲: 装備品からの選択を許可する / Allow equip items */
#define USE_INVEN 0x02 /*!< アイテム表示/選択範囲: 所持品からの選択を許可する /  Allow inven items */
#define USE_FLOOR 0x04 /*!< アイテム表示/選択範囲: 床下のアイテムからの選択を許可する /  Allow floor items */
#define USE_FORCE 0x08 /*!< 特殊: wキーで錬気術への切り替えを許可する */
#define IGNORE_BOTHHAND_SLOT 0x10 /*!< アイテム表示/選択範囲: 両手持ちスロットを選択に含めない */
#define USE_FULL  0x20 /*!< アイテム表示/選択範囲: 空欄まですべて表示する*/




/*
 * p_ptr->updateに充てるプレイヤーステータス更新処理要求 / Bit flags for the "p_ptr->update" variable
 */


/*
 * Bit flags for the "p_ptr->redraw" variable
 */
#define PR_MISC         0x00000001L     /*!< 再描画フラグ: 種族と職業 / Display Race/Class */
#define PR_TITLE        0x00000002L     /*!< 再描画フラグ: 称号 / Display Title */
#define PR_LEV          0x00000004L     /*!< 再描画フラグ: レベル / Display Level */
#define PR_EXP          0x00000008L     /*!< 再描画フラグ: 経験値 / Display Experience */
#define PR_STATS        0x00000010L     /*!< 再描画フラグ: ステータス /  Display Stats */
#define PR_ARMOR        0x00000020L     /*!< 再描画フラグ: AC / Display Armor */
#define PR_HP           0x00000040L     /*!< 再描画フラグ: HP / Display Hitpoints */
#define PR_MANA         0x00000080L     /*!< 再描画フラグ: MP / Display Mana */
#define PR_GOLD         0x00000100L     /*!< 再描画フラグ: 所持金 / Display Gold */
#define PR_DEPTH        0x00000200L     /*!< 再描画フラグ: ダンジョンの階 / Display Depth */
#define PR_EQUIPPY      0x00000400L     /*!< 再描画フラグ: 装備シンボル / Display equippy chars */
#define PR_HEALTH       0x00000800L     /*!< 再描画フラグ: モンスターのステータス / Display Health Bar */
#define PR_CUT          0x00001000L     /*!< 再描画フラグ: 負傷度 / Display Extra (Cut) */
#define PR_STUN         0x00002000L     /*!< 再描画フラグ: 朦朧度 / Display Extra (Stun) */
#define PR_HUNGER       0x00004000L     /*!< 再描画フラグ: 空腹度 / Display Extra (Hunger) */
#define PR_STATUS       0x00008000L     /*!< 再描画フラグ: プレイヤーの付与状態 /  Display Status Bar */
#define PR_XXX0         0x00010000L     /*!< (unused) */
#define PR_UHEALTH      0x00020000L     /*!< 再描画フラグ: ペットのステータス / Display Uma Health Bar */
#define PR_XXX1         0x00040000L     /*!< (unused) */
#define PR_XXX2         0x00080000L     /*!< (unused) */
#define PR_STATE        0x00100000L     /*!< 再描画フラグ: プレイヤーの行動状態 / Display Extra (State) */
#define PR_SPEED        0x00200000L     /*!< 再描画フラグ: 加速 / Display Extra (Speed) */
#define PR_STUDY        0x00400000L     /*!< 再描画フラグ: 学習 / Display Extra (Study) */
#define PR_IMITATION    0x00800000L     /*!< 再描画フラグ: ものまね / Display Extra (Imitation) */
#define PR_EXTRA        0x01000000L     /*!< 再描画フラグ: 拡張ステータス全体 / Display Extra Info */
#define PR_BASIC        0x02000000L     /*!< 再描画フラグ: 基本ステータス全体 / Display Basic Info */
#define PR_MAP          0x04000000L     /*!< 再描画フラグ: ゲームマップ / Display Map */
#define PR_WIPE         0x08000000L     /*!< 再描画フラグ: 画面消去 / Hack -- Total Redraw */
/* xxx */
/* xxx */
/* xxx */
/* xxx */

/*
 * Bit flags for the "p_ptr->window" variable (etc)
 */
#define PW_INVEN        0x00000001L     /*!<サブウィンドウ描画フラグ: 所持品-装備品 / Display inven/equip */
#define PW_EQUIP        0x00000002L     /*!<サブウィンドウ描画フラグ: 装備品-所持品 / Display equip/inven */
#define PW_SPELL        0x00000004L     /*!<サブウィンドウ描画フラグ: 魔法一覧 / Display spell list */
#define PW_PLAYER       0x00000008L     /*!<サブウィンドウ描画フラグ: プレイヤーのステータス / Display character */
#define PW_MONSTER_LIST 0x00000010L     /*!<サブウィンドウ描画フラグ: 視界内モンスターの一覧 / Display monster list */
/* xxx */
/* xxx */
#define PW_MESSAGE      0x00000040L     /*!<サブウィンドウ描画フラグ: メッセージログ / Display messages */
#define PW_OVERHEAD     0x00000080L     /*!<サブウィンドウ描画フラグ: 周辺の光景 / Display overhead view */
#define PW_MONSTER      0x00000100L     /*!<サブウィンドウ描画フラグ: モンスターの思い出 / Display monster recall */
#define PW_OBJECT       0x00000200L     /*!<サブウィンドウ描画フラグ: アイテムの知識 / Display object recall */
#define PW_DUNGEON      0x00000400L     /*!<サブウィンドウ描画フラグ: ダンジョンの地形 / Display dungeon view */
#define PW_SNAPSHOT     0x00000800L     /*!<サブウィンドウ描画フラグ: 記念写真 / Display snap-shot */
/* xxx */
/* xxx */
#define PW_BORG_1       0x00004000L     /*!<サブウィンドウ描画フラグ: ボーグメッセージ / Display borg messages */
#define PW_BORG_2       0x00008000L     /*!<サブウィンドウ描画フラグ: ボーグステータス / Display borg status */

/* Empty hand status */
#define EMPTY_HAND_NONE 0x0000 /* Both hands are used */
#define EMPTY_HAND_LARM 0x0001 /* Left hand is empty */
#define EMPTY_HAND_RARM 0x0002 /* Right hand is empty */

/*** General index values ***/


/*
 * Some things which induce learning
 */
#define DRS_ACID         1
#define DRS_ELEC         2
#define DRS_FIRE         3
#define DRS_COLD         4
#define DRS_POIS         5
#define DRS_NETH         6
#define DRS_LITE         7
#define DRS_DARK         8
#define DRS_FEAR         9
#define DRS_CONF        10
#define DRS_CHAOS       11
#define DRS_DISEN       12
#define DRS_BLIND       13
#define DRS_NEXUS       14
#define DRS_SOUND       15
#define DRS_SHARD       16
#define DRS_FREE        30
#define DRS_MANA        31
#define DRS_REFLECT     32



#define DAMAGE_FORCE    1
#define DAMAGE_GENO     2
#define DAMAGE_LOSELIFE 3
#define DAMAGE_ATTACK   4
#define DAMAGE_NOESCAPE 5
#define DAMAGE_USELIFE  6


/*
 * アイテムの簡易鑑定定義 / Game generated inscription indices. These are stored in the object,
 * and are used to index the string array from tables.c.
 */
#define FEEL_NONE              0 /*!< 簡易鑑定: 未鑑定 */
#define FEEL_BROKEN            1 /*!< 簡易鑑定: 壊れている */
#define FEEL_TERRIBLE          2 /*!< 簡易鑑定: 恐ろしい */
#define FEEL_WORTHLESS         3 /*!< 簡易鑑定: 無価値 */
#define FEEL_CURSED            4 /*!< 簡易鑑定: 呪われている */
#define FEEL_UNCURSED          5 /*!< 簡易鑑定: 呪われていない */
#define FEEL_AVERAGE           6 /*!< 簡易鑑定: 並 */
#define FEEL_GOOD              7 /*!< 簡易鑑定: 上質 */
#define FEEL_EXCELLENT         8 /*!< 簡易鑑定: 高級 */
#define FEEL_SPECIAL           9 /*!< 簡易鑑定: 特別 */

#define FEEL_MAX               9 /*!< 簡易鑑定の種別数 */

/*
 * 変愚ver1.5.0以前に使われていたアイテムの追加特性フラグ / Hack -- special "xtra" object powers
 */
#define EGO_XTRA_SUSTAIN        1 /*!< 旧版アイテムフラグ(非推奨): 追加維持能力 / Sustain one stat */
#define EGO_XTRA_POWER          2 /*!< 旧版アイテムフラグ(非推奨): 追加上級耐性 / High resist */
#define EGO_XTRA_ABILITY        3 /*!< 旧版アイテムフラグ(非推奨): 追加能力 / Special ability */


/*** Object flag values ***/


/*!
 * 箱のトラップ定義 Chest trap flags (see "tables.c")
 */
#define CHEST_LOSE_STR          0x0001 /*!< 箱のトラップ: STR減少の毒針 */
#define CHEST_LOSE_CON          0x0002 /*!< 箱のトラップ: CON減少の毒針 */
#define CHEST_POISON            0x0004 /*!< 箱のトラップ: 毒針 */
#define CHEST_PARALYZE          0x0008 /*!< 箱のトラップ: 麻痺ガス */
#define CHEST_EXPLODE           0x0010 /*!< 箱のトラップ: 爆発 */
#define CHEST_SUMMON            0x0020 /*!< 箱のトラップ: 召喚のルーン(モンスター) */
#define CHEST_SCATTER           0x0040 /*!< 箱のトラップ: アイテム散乱 */
#define CHEST_E_SUMMON          0x0080 /*!< 箱のトラップ: 召喚のルーン(エレメンタル) */
#define CHEST_BIRD_STORM        0x0100 /*!< 箱のトラップ: 召喚のルーン(鳥) */
#define CHEST_H_SUMMON          0x0200 /*!< 箱のトラップ: 召喚のルーン(強敵)  */
#define CHEST_RUNES_OF_EVIL     0x0400 /*!< 箱のトラップ: 邪悪なルーン */
#define CHEST_ALARM             0x0800 /*!< 箱のトラップ: 警報装置 */


/*
 * Special Object Flags
 */
#define IDENT_SENSE     0x01    /* Item has been "sensed" */
#define IDENT_FIXED     0x02    /* Item has been "haggled" */
#define IDENT_EMPTY     0x04    /* Item charges are known */
#define IDENT_KNOWN     0x08    /* Item abilities are known */
#define IDENT_STORE     0x10    /* Item is storebought !!!! */
#define IDENT_MENTAL    0x20    /* Item information is known */
#if 0
#define IDENT_CURSED    0x40    /* Item is temporarily cursed */
#endif
#define IDENT_BROKEN    0x80    /* Item is permanently worthless */


/* 
 * How object is marked (flags in object_type.mark)
 * OM_FOUND --- original boolean flag
 * OM_NOMSG --- temporary flag to suppress messages which were
 *              already printed in autopick_pickup_items().
 */
#define OM_FOUND        0x01    /*!< アイテムを一度でも視界に収めたことがあるか */
#define OM_NOMSG        0x02    /* temporary flag to suppress messages */
#define OM_NO_QUERY     0x04    /* Query for auto-pick was already answered as 'No' */
#define OM_AUTODESTROY  0x08    /* Destroy later to avoid illegal inventry shift */
#define OM_TOUCHED      0x10    /* Object was touched by player */


/*
 * Object flags
 *
 * Old variables for object flags such as flags1, flags2, and flags3
 * are obsolated.  Now single array flgs[TR_FLAG_SIZE] contains all
 * object flags.  And each flag is refered by single index number
 * instead of a bit mask.
 *
 * Therefore it's very easy to add a lot of new flags; no one need to
 * worry about in which variable a new flag should be put, nor to
 * modify a huge number of files all over the source directory at once
 * to add new flag variables such as flags4, a_ability_flags1, etc...
 *
 * All management of flags is now treated using a set of macros
 * instead of bit operations.
 * Note: These macros are using division, modulo, and bit shift
 * operations, and it seems that these operations are rather slower
 * than original bit operation.  But since index numbers are almost
 * always given as constant, such slow operations are performed in the
 * compile time.  So there is no problem on the speed.
 *
 * Exceptions of new flag management is a set of flags to control
 * object generation and the curse flags.  These are not yet rewritten
 * in new index form; maybe these have no merit of rewriting.
 */

#define have_flag(ARRAY, INDEX) !!((ARRAY)[(INDEX)/32] & (1L << ((INDEX)%32)))
#define add_flag(ARRAY, INDEX) ((ARRAY)[(INDEX)/32] |= (1L << ((INDEX)%32)))
#define remove_flag(ARRAY, INDEX) ((ARRAY)[(INDEX)/32] &= ~(1L << ((INDEX)%32)))
#define is_pval_flag(INDEX) ((TR_STR <= (INDEX) && (INDEX) <= TR_MAGIC_MASTERY) || (TR_STEALTH <= (INDEX) && (INDEX) <= TR_BLOWS))
#define have_pval_flags(ARRAY) !!((ARRAY)[0] & (0x00003f7f))


#define TR_STR                 0      /* STR += "pval" */
#define TR_INT                 1      /* INT += "pval" */
#define TR_WIS                 2      /* WIS += "pval" */
#define TR_DEX                 3      /* DEX += "pval" */
#define TR_CON                 4      /* CON += "pval" */
#define TR_CHR                 5      /* CHR += "pval" */
#define TR_MAGIC_MASTERY       6      /* Later */
#define TR_FORCE_WEAPON        7      /* Later */
#define TR_STEALTH             8      /* Stealth += "pval" */
#define TR_SEARCH              9      /* Search += "pval" */
#define TR_INFRA               10     /* Infra += "pval" */
#define TR_TUNNEL              11     /* Tunnel += "pval" */
#define TR_SPEED               12     /* Speed += "pval" */
#define TR_BLOWS               13     /* Blows += "pval" */
#define TR_CHAOTIC             14
#define TR_VAMPIRIC            15
#define TR_SLAY_ANIMAL         16
#define TR_SLAY_EVIL           17
#define TR_SLAY_UNDEAD         18
#define TR_SLAY_DEMON          19
#define TR_SLAY_ORC            20
#define TR_SLAY_TROLL          21
#define TR_SLAY_GIANT          22
#define TR_SLAY_DRAGON         23
#define TR_KILL_DRAGON         24     /* Execute Dragon */
#define TR_VORPAL              25     /* Later */
#define TR_IMPACT              26     /* Cause Earthquakes */
#define TR_BRAND_POIS          27
#define TR_BRAND_ACID          28
#define TR_BRAND_ELEC          29
#define TR_BRAND_FIRE          30
#define TR_BRAND_COLD          31

#define TR_SUST_STR            32
#define TR_SUST_INT            33
#define TR_SUST_WIS            34
#define TR_SUST_DEX            35
#define TR_SUST_CON            36
#define TR_SUST_CHR            37
#define TR_RIDING              38
#define TR_EASY_SPELL          39
#define TR_IM_ACID             40
#define TR_IM_ELEC             41
#define TR_IM_FIRE             42
#define TR_IM_COLD             43
#define TR_THROW               44     /* Later */
#define TR_REFLECT             45     /* Reflect 'bolts' */
#define TR_FREE_ACT            46     /* Free Action */
#define TR_HOLD_EXP            47     /* Hold EXP */
#define TR_RES_ACID            48
#define TR_RES_ELEC            49
#define TR_RES_FIRE            50
#define TR_RES_COLD            51
#define TR_RES_POIS            52
#define TR_RES_FEAR            53     /* Added for Zangband */
#define TR_RES_LITE            54
#define TR_RES_DARK            55
#define TR_RES_BLIND           56
#define TR_RES_CONF            57
#define TR_RES_SOUND           58
#define TR_RES_SHARDS          59
#define TR_RES_NETHER          60
#define TR_RES_NEXUS           61
#define TR_RES_CHAOS           62
#define TR_RES_DISEN           63

#define TR_SH_FIRE             64     /* Immolation (Fire) */
#define TR_SH_ELEC             65     /* Electric Sheath */
#define TR_SLAY_HUMAN          66     /* Slay human */
#define TR_SH_COLD             67     /* cold aura */
#define TR_NO_TELE             68     /* Anti-teleportation */
#define TR_NO_MAGIC            69     /* Anti-magic */
#define TR_DEC_MANA            70     /* ??? */
#define TR_TY_CURSE            71     /* The Ancient Curse */
#define TR_WARNING             72     /* Warning */
#define TR_HIDE_TYPE           73     /* Hide "pval" description */
#define TR_SHOW_MODS           74     /* Always show Tohit/Todam */
#define TR_SLAY_GOOD           75
#define TR_LEVITATION          76     /* Feather Falling */
#define TR_LITE_1              77     /* Light Radius 1*/
#define TR_SEE_INVIS           78     /* See Invisible */
#define TR_TELEPATHY           79     /* Telepathy */
#define TR_SLOW_DIGEST         80     /* Item slows down digestion */
#define TR_REGEN               81     /* Item induces regeneration */
#define TR_XTRA_MIGHT          82     /* Bows get extra multiplier */
#define TR_XTRA_SHOTS          83     /* Bows get extra shots */
#define TR_IGNORE_ACID         84     /* Item ignores Acid Damage */
#define TR_IGNORE_ELEC         85     /* Item ignores Elec Damage */
#define TR_IGNORE_FIRE         86     /* Item ignores Fire Damage */
#define TR_IGNORE_COLD         87     /* Item ignores Cold Damage */
#define TR_ACTIVATE            88     /* Item can be activated */
#define TR_DRAIN_EXP           89     /* Item drains Experience */
#define TR_TELEPORT            90     /* Item teleports player */
#define TR_AGGRAVATE           91     /* Item aggravates monsters */
#define TR_BLESSED             92     /* Item is Blessed */
#define TR_ES_ATTACK           93     /* Fake flag for Smith */
#define TR_ES_AC               94     /* Fake flag for Smith */
#define TR_KILL_GOOD           95

#define TR_KILL_ANIMAL         96
#define TR_KILL_EVIL           97
#define TR_KILL_UNDEAD         98
#define TR_KILL_DEMON          99
#define TR_KILL_ORC            100
#define TR_KILL_TROLL          101
#define TR_KILL_GIANT          102
#define TR_KILL_HUMAN          103
#define TR_ESP_ANIMAL          104
#define TR_ESP_UNDEAD          105
#define TR_ESP_DEMON           106
#define TR_ESP_ORC             107
#define TR_ESP_TROLL           108
#define TR_ESP_GIANT           109
#define TR_ESP_DRAGON          110
#define TR_ESP_HUMAN           111
#define TR_ESP_EVIL            112
#define TR_ESP_GOOD            113
#define TR_ESP_NONLIVING       114
#define TR_ESP_UNIQUE          115
#define TR_FULL_NAME           116
#define TR_FIXED_FLAVOR        117
#define TR_ADD_L_CURSE         118
#define TR_ADD_H_CURSE         119
#define TR_DRAIN_HP            120
#define TR_DRAIN_MANA          121
#define TR_LITE_2			   122
#define TR_LITE_3			   123
#define TR_LITE_M1			   124    /* Permanent decrease Light Area (-1) */
#define TR_LITE_M2			   125    /* Permanent decrease Light Area (-1) */
#define TR_LITE_M3			   126    /* Permanent decrease Light Area (-1) */
#define TR_LITE_FUEL		   127	  /* Lights need Fuels */

#define TR_CALL_ANIMAL         128
#define TR_CALL_DEMON          129
#define TR_CALL_DRAGON         130
#define TR_CALL_UNDEAD         131
#define TR_COWARDICE           132
#define TR_LOW_MELEE           133
#define TR_LOW_AC              134
#define TR_LOW_MAGIC           135
#define TR_FAST_DIGEST         136
#define TR_SLOW_REGEN          137

#define TR_FLAG_MAX            138
#define TR_FLAG_SIZE           5


#define TRG_INSTA_ART           0x00000001L     /* Item must be an artifact */
#define TRG_QUESTITEM           0x00000002L     /* quest level item -KMW- */
#define TRG_XTRA_POWER          0x00000004L     /* Extra power */
#define TRG_ONE_SUSTAIN         0x00000008L     /* One sustain */
#define TRG_XTRA_RES_OR_POWER   0x00000010L     /* Extra resistance or power */
#define TRG_XTRA_H_RES          0x00000020L     /* Extra high resistance */
#define TRG_XTRA_E_RES          0x00000040L     /* Extra element resistance */
#define TRG_XTRA_L_RES          0x00000080L     /* Extra lordly resistance */
#define TRG_XTRA_D_RES          0x00000100L     /* Extra dragon resistance */
#define TRG_XTRA_RES            0x00000200L     /* Extra resistance */
#define TRG_CURSED              0x00000400L     /* Item is Cursed */
#define TRG_HEAVY_CURSE         0x00000800L     /* Item is Heavily Cursed */
#define TRG_PERMA_CURSE         0x00001000L     /* Item is Perma Cursed */
#define TRG_RANDOM_CURSE0       0x00002000L     /* Item is Random Cursed */
#define TRG_RANDOM_CURSE1       0x00004000L     /* Item is Random Cursed */
#define TRG_RANDOM_CURSE2       0x00008000L     /* Item is Random Cursed */
#define TRG_XTRA_DICE           0x00010000L     /* Extra dice */
#define TRG_POWERFUL            0x00020000L     /* Item has good value even if Cursed */


#define MAX_CURSE 18

#define TRC_CURSED              0x00000001L
#define TRC_HEAVY_CURSE         0x00000002L
#define TRC_PERMA_CURSE         0x00000004L
#define TRC_XXX1                0x00000008L
#define TRC_TY_CURSE            0x00000010L
#define TRC_AGGRAVATE           0x00000020L
#define TRC_DRAIN_EXP           0x00000040L
#define TRC_SLOW_REGEN          0x00000080L
#define TRC_ADD_L_CURSE         0x00000100L
#define TRC_ADD_H_CURSE         0x00000200L
#define TRC_CALL_ANIMAL         0x00000400L
#define TRC_CALL_DEMON          0x00000800L
#define TRC_CALL_DRAGON         0x00001000L
#define TRC_COWARDICE           0x00002000L
#define TRC_TELEPORT            0x00004000L
#define TRC_LOW_MELEE           0x00008000L
#define TRC_LOW_AC              0x00010000L
#define TRC_LOW_MAGIC           0x00020000L
#define TRC_FAST_DIGEST         0x00040000L
#define TRC_DRAIN_HP            0x00080000L
#define TRC_DRAIN_MANA          0x00100000L
#define TRC_CALL_UNDEAD         0x00200000L

#define TRC_TELEPORT_SELF       0x00000001L
#define TRC_CHAINSWORD          0x00000002L

#define TRC_SPECIAL_MASK \
	(TRC_TY_CURSE | TRC_AGGRAVATE)

#define TRC_HEAVY_MASK   \
	(TRC_TY_CURSE | TRC_AGGRAVATE | TRC_DRAIN_EXP | TRC_ADD_H_CURSE | \
	 TRC_CALL_DEMON | TRC_CALL_DRAGON | TRC_CALL_UNDEAD | TRC_TELEPORT)

#define TRC_P_FLAG_MASK  \
	(TRC_TELEPORT_SELF | TRC_CHAINSWORD | \
	 TRC_TY_CURSE | TRC_DRAIN_EXP | TRC_ADD_L_CURSE | TRC_ADD_H_CURSE | \
	 TRC_CALL_ANIMAL | TRC_CALL_DEMON | TRC_CALL_DRAGON | TRC_COWARDICE | \
	 TRC_TELEPORT | TRC_DRAIN_HP | TRC_DRAIN_MANA | TRC_CALL_UNDEAD)



/*** Monster blow constants ***/


/*!
 * @note モンスターの打撃方法 / New monster blow methods
 * 打撃の種別に応じて傷と朦朧が発生するかがコメントの通りに決まる
 */
#define RBM_HIT          1 /*!< モンスターの攻撃種別:殴る(傷/朦朧が半々) */
#define RBM_TOUCH        2 /*!< モンスターの攻撃種別:触る */
#define RBM_PUNCH        3 /*!< モンスターの攻撃種別:パンチする(朦朧) */
#define RBM_KICK         4 /*!< モンスターの攻撃種別:蹴る(朦朧) */
#define RBM_CLAW         5 /*!< モンスターの攻撃種別:ひっかく(傷) */
#define RBM_BITE         6 /*!< モンスターの攻撃種別:噛む(傷) */
#define RBM_STING        7 /*!< モンスターの攻撃種別:刺す */
#define RBM_SLASH        8 /*!< モンスターの攻撃種別:斬る(傷) */
#define RBM_BUTT         9 /*!< モンスターの攻撃種別:角で突く(朦朧) */
#define RBM_CRUSH       10 /*!< モンスターの攻撃種別:体当たりする(朦朧) */
#define RBM_ENGULF      11 /*!< モンスターの攻撃種別:飲み込む */
#define RBM_CHARGE      12 /*!< モンスターの攻撃種別:請求書を寄越す */
#define RBM_CRAWL       13 /*!< モンスターの攻撃種別:体の上を這い回る */
#define RBM_DROOL       14 /*!< モンスターの攻撃種別:よだれをたらす */
#define RBM_SPIT        15 /*!< モンスターの攻撃種別:つばを吐く */
#define RBM_EXPLODE     16 /*!< モンスターの攻撃種別:爆発する */
#define RBM_GAZE        17 /*!< モンスターの攻撃種別:にらむ */
#define RBM_WAIL        18 /*!< モンスターの攻撃種別:泣き叫ぶ */
#define RBM_SPORE       19 /*!< モンスターの攻撃種別:胞子を飛ばす */
#define RBM_XXX4        20 /*!< モンスターの攻撃種別:未定義 */
#define RBM_BEG         21 /*!< モンスターの攻撃種別:金をせがむ */
#define RBM_INSULT      22 /*!< モンスターの攻撃種別:侮辱する */
#define RBM_MOAN        23 /*!< モンスターの攻撃種別:うめく */
#define RBM_SHOW        24 /*!< モンスターの攻撃種別:歌う */
#define RBM_SHOOT       25 /*!< モンスターの攻撃種別:射撃(非打撃) */


/*!
 * @note モンスターの攻撃効果 / New monster blow effects
 */
#define RBE_HURT         1 /*!< モンスターの攻撃効果: 攻撃する*/
#define RBE_POISON       2 /*!< モンスターの攻撃効果: 毒をくらわす*/
#define RBE_UN_BONUS     3 /*!< モンスターの攻撃効果: 劣化させる*/
#define RBE_UN_POWER     4 /*!< モンスターの攻撃効果: 充填魔力を吸収する*/
#define RBE_EAT_GOLD     5 /*!< モンスターの攻撃効果: 金を盗む*/
#define RBE_EAT_ITEM     6 /*!< モンスターの攻撃効果: アイテムを盗む*/
#define RBE_EAT_FOOD     7 /*!< モンスターの攻撃効果: あなたの食糧を食べる*/
#define RBE_EAT_LITE     8 /*!< モンスターの攻撃効果: 明かりを吸収する*/
#define RBE_ACID         9 /*!< モンスターの攻撃効果: 酸を飛ばす*/
#define RBE_ELEC        10 /*!< モンスターの攻撃効果: 感電させる*/
#define RBE_FIRE        11 /*!< モンスターの攻撃効果: 燃やす*/
#define RBE_COLD        12 /*!< モンスターの攻撃効果: 凍らせる*/
#define RBE_BLIND       13 /*!< モンスターの攻撃効果: 盲目にする*/
#define RBE_CONFUSE     14 /*!< モンスターの攻撃効果: 混乱させる*/
#define RBE_TERRIFY     15 /*!< モンスターの攻撃効果: 恐怖させる*/
#define RBE_PARALYZE    16 /*!< モンスターの攻撃効果: 麻痺させる*/
#define RBE_LOSE_STR    17 /*!< モンスターの攻撃効果: 腕力を減少させる*/
#define RBE_LOSE_INT    18 /*!< モンスターの攻撃効果: 知能を減少させる*/
#define RBE_LOSE_WIS    19 /*!< モンスターの攻撃効果: 賢さを減少させる*/
#define RBE_LOSE_DEX    20 /*!< モンスターの攻撃効果: 器用さを減少させる*/
#define RBE_LOSE_CON    21 /*!< モンスターの攻撃効果: 耐久力を減少させる*/
#define RBE_LOSE_CHR    22 /*!< モンスターの攻撃効果: 魅力を減少させる*/
#define RBE_LOSE_ALL    23 /*!< モンスターの攻撃効果: 全ステータスを減少させる*/
#define RBE_SHATTER     24 /*!< モンスターの攻撃効果: 粉砕する*/
#define RBE_EXP_10      25 /*!< モンスターの攻撃効果: 経験値を減少(10d6+)させる*/
#define RBE_EXP_20      26 /*!< モンスターの攻撃効果: 経験値を減少(20d6+)させる*/
#define RBE_EXP_40      27 /*!< モンスターの攻撃効果: 経験値を減少(40d6+)させる*/
#define RBE_EXP_80      28 /*!< モンスターの攻撃効果: 経験値を減少(80d6+)させる*/
#define RBE_DISEASE     29 /*!< モンスターの攻撃効果: 病気にする*/
#define RBE_TIME        30 /*!< モンスターの攻撃効果: 時間を逆戻りさせる*/
#define RBE_DR_LIFE     31 /*!< モンスターの攻撃効果: 生命力を吸収する*/
#define RBE_DR_MANA     32 /*!< モンスターの攻撃効果: 魔力を奪う*/
#define RBE_SUPERHURT   33 /*!< モンスターの攻撃効果: 強力に攻撃する*/
#define RBE_INERTIA     34 /*!< モンスターの攻撃効果: 減速させる*/
#define RBE_STUN        35 /*!< モンスターの攻撃効果: 朦朧とさせる*/


 /*
  * Hack -- "torch" masks
  */
#define RF7_LITE_MASK \
	(RF7_HAS_LITE_1 | RF7_SELF_LITE_1 | RF7_HAS_LITE_2 | RF7_SELF_LITE_2)

#define RF7_DARK_MASK \
	(RF7_HAS_DARK_1 | RF7_SELF_DARK_1 | RF7_HAS_DARK_2 | RF7_SELF_DARK_2)

#define RF7_HAS_LD_MASK \
	(RF7_HAS_LITE_1 | RF7_HAS_LITE_2 | RF7_HAS_DARK_1 | RF7_HAS_DARK_2)

#define RF7_SELF_LD_MASK \
	(RF7_SELF_LITE_1 | RF7_SELF_LITE_2 | RF7_SELF_DARK_1 | RF7_SELF_DARK_2)

  /*
   * Hack -- effective elemental and poison immunity mask
   */
#define RFR_EFF_IM_ACID_MASK  (RFR_IM_ACID | RFR_RES_ALL)
#define RFR_EFF_IM_ELEC_MASK  (RFR_IM_ELEC | RFR_RES_ALL)
#define RFR_EFF_IM_FIRE_MASK  (RFR_IM_FIRE | RFR_RES_ALL)
#define RFR_EFF_IM_COLD_MASK  (RFR_IM_COLD | RFR_RES_ALL)
#define RFR_EFF_IM_POIS_MASK  (RFR_IM_POIS | RFR_RES_ALL)
#define RFR_EFF_RES_SHAR_MASK (RFR_RES_SHAR | RFR_RES_ALL)
#define RFR_EFF_RES_CHAO_MASK (RFR_RES_CHAO | RFR_RES_ALL)
#define RFR_EFF_RES_NEXU_MASK (RFR_RES_NEXU | RFR_RES_ALL)


#define MR1_SINKA 0x01


/* Hack -- Determine monster race appearance index is same as race index */
#define is_original_ap(A) \
	 (bool)(((A)->ap_r_idx == (A)->r_idx) ? TRUE : FALSE)

#define is_original_ap_and_seen(A) \
	 (bool)((A)->ml && !p_ptr->image && ((A)->ap_r_idx == (A)->r_idx))

/*
 * Is the monster seen by the player?
 */
#define is_seen(A) \
	((bool)((A)->ml && (!ignore_unview || p_ptr->inside_battle || \
	 (player_can_see_bold((A)->fy, (A)->fx) && projectable(p_ptr->y, p_ptr->x, (A)->fy, (A)->fx)))))


/*** Option Definitions ***/


#define OPT_PAGE_INPUT          1
#define OPT_PAGE_MAPSCREEN      2
#define OPT_PAGE_TEXT           3
#define OPT_PAGE_GAMEPLAY       4
#define OPT_PAGE_DISTURBANCE    5
#define OPT_PAGE_BIRTH          6
#define OPT_PAGE_AUTODESTROY    7
#define OPT_PAGE_PLAYRECORD    10

#define OPT_PAGE_JAPANESE_ONLY 99


/*** Macro Definitions ***/


/*
 * Hack -- The main "screen"
 */
#define term_screen     (angband_term[0])

/*
 * Return the "attr" for a given item.
 * Use "flavor" if available.
 * Default to user definitions.
 */
#define object_attr(T) \
	((k_info[(T)->k_idx].flavor) ? \
	 (k_info[k_info[(T)->k_idx].flavor].x_attr) : \
	 ((!(T)->k_idx || ((T)->tval != TV_CORPSE) || ((T)->sval != SV_CORPSE) || \
	   (k_info[(T)->k_idx].x_attr != TERM_DARK)) ? \
	  (k_info[(T)->k_idx].x_attr) : (r_info[(T)->pval].x_attr)))

/*
 * Return the "char" for a given item.
 * Use "flavor" if available.
 * Default to user definitions.
 */
#define object_char(T) \
	((k_info[(T)->k_idx].flavor) ? \
	 (k_info[k_info[(T)->k_idx].flavor].x_char) : \
	 (k_info[(T)->k_idx].x_char))

/*
 * Convert an "attr"/"char" pair into a "pict" (P)
 */
#define PICT(A,C) \
	((((u16b)(A)) << 8) | ((byte)(C)))

/*
 * Convert a "pict" (P) into an "attr" (A)
 */
#define PICT_A(P) \
	((byte)((P) >> 8))

/*
 * Convert a "pict" (P) into an "char" (C)
 */
#define PICT_C(P) \
	((char)((byte)(P)))


/*
 * Convert a "location" (Y,X) into a "grid" (G)
 */
#define GRID(Y,X) \
	(256 * (Y) + (X))

/*
 * Convert a "grid" (G) into a "location" (Y)
 */
#define GRID_Y(G) \
	((int)((G) / 256U))

/*
 * Convert a "grid" (G) into a "location" (X)
 */
#define GRID_X(G) \
	((int)((G) % 256U))

/*
 * Determines if a map location is currently "on screen" -RAK-
 * Note that "panel_contains(Y,X)" always implies "in_bounds2(Y,X)".
 */
#define panel_contains(Y,X) \
  (((Y) >= panel_row_min) && ((Y) <= panel_row_max) && \
   ((X) >= panel_col_min) && ((X) <= panel_col_max))


/*
 * Hack -- Prepare to use the "Secure" routines
 */
#if defined(SET_UID) && defined(SECURE)
extern int PlayerUID;
# define getuid() PlayerUID
# define geteuid() PlayerUID
#endif



/*** Color constants ***/


/*
 * Angband "attributes" (with symbols, and base (R,G,B) codes)
 *
 * The "(R,G,B)" codes are given in "fourths" of the "maximal" value,
 * and should "gamma corrected" on most (non-Macintosh) machines.
 */
#define TERM_DARK                0  /*!< 'd' - 黒 0,0,0 */
#define TERM_WHITE               1  /*!< 'w' - 白 4,4,4 */
#define TERM_SLATE               2  /*!< 's' - 灰 2,2,2 */
#define TERM_ORANGE              3  /*!< 'o' - 橙 4,2,0 */
#define TERM_RED                 4  /*!< 'r' - 赤 3,0,0 */
#define TERM_GREEN               5  /*!< 'g' - 緑 0,2,1 */
#define TERM_BLUE                6  /*!< 'b' - 青 0,0,4 */
#define TERM_UMBER               7  /*!< 'u' - 琥珀 2,1,0 */
#define TERM_L_DARK              8  /*!< 'D' - 暗い灰 1,1,1 */
#define TERM_L_WHITE             9  /*!< 'W' - 明るい灰 3,3,3 */
#define TERM_VIOLET             10  /*!< 'v' - 紫 4,0,4 */
#define TERM_YELLOW             11  /*!< 'y' - 黄 4,4,0 */
#define TERM_L_RED              12  /*!< 'R' - 明るい赤 4,0,0 */
#define TERM_L_GREEN            13  /*!< 'G' - 明るい緑 0,4,0 */
#define TERM_L_BLUE             14  /*!< 'B' - 明るい青 0,4,4 */
#define TERM_L_UMBER            15  /*!< 'U' - 明るい琥珀 3,2,1 */


/*
 * Not using graphical tiles for this feature?
 */
#define is_ascii_graphics(A) (!((A) & 0x80))

/*** Music constants ***/

#define MUSIC_BASIC_DEFAULT    0
#define MUSIC_BASIC_GAMEOVER   1
#define MUSIC_BASIC_EXIT       2
#define MUSIC_BASIC_TOWN       3
#define MUSIC_BASIC_FIELD1     4
#define MUSIC_BASIC_FIELD2     5
#define MUSIC_BASIC_FIELD3     6
#define MUSIC_BASIC_DUN_LOW    7
#define MUSIC_BASIC_DUN_MED    8
#define MUSIC_BASIC_DUN_HIGH   9
#define MUSIC_BASIC_DUN_FEEL1 10
#define MUSIC_BASIC_DUN_FEEL2 11
#define MUSIC_BASIC_WINNER    12
#define MUSIC_BASIC_BUILD     13
#define MUSIC_BASIC_WILD      14
#define MUSIC_BASIC_QUEST     15
#define MUSIC_BASIC_ARENA     16
#define MUSIC_BASIC_BATTLE    17
#define MUSIC_BASIC_QUEST_CLEAR 18
#define MUSIC_BASIC_FINAL_QUEST_CLEAR 19
#define MUSIC_BASIC_AMBUSH    20
#define MUSIC_BASIC_MAX       21 /*!< BGM定義の最大数 */

/*** Sound constants ***/

/*
 * Mega-Hack -- some primitive sound support (see "main-win.c")
 *
 * Some "sound" constants for "Term_xtra(TERM_XTRA_SOUND, val)"
 */
#define SOUND_HIT        1
#define SOUND_MISS       2
#define SOUND_FLEE       3
#define SOUND_DROP       4
#define SOUND_KILL       5
#define SOUND_LEVEL      6
#define SOUND_DEATH      7
#define SOUND_STUDY      8
#define SOUND_TELEPORT   9
#define SOUND_SHOOT     10
#define SOUND_QUAFF     11
#define SOUND_ZAP       12
#define SOUND_WALK      13
#define SOUND_TPOTHER   14
#define SOUND_HITWALL   15
#define SOUND_EAT       16
#define SOUND_STORE1    17
#define SOUND_STORE2    18
#define SOUND_STORE3    19
#define SOUND_STORE4    20
#define SOUND_DIG       21
#define SOUND_OPENDOOR  22
#define SOUND_SHUTDOOR  23
#define SOUND_TPLEVEL   24
#define SOUND_SCROLL	25
#define SOUND_BUY	    26
#define SOUND_SELL	    27
#define SOUND_WARN	    28
#define SOUND_ROCKET    29 /*!< Somebody's shooting rockets */
#define SOUND_N_KILL    30 /*!< The player kills a non-living/undead monster */
#define SOUND_U_KILL    31 /*!< The player kills a unique */
#define SOUND_QUEST     32 /*!< The player has just completed a quest */
#define SOUND_HEAL      33 /*!< The player was healed a little bit */
#define SOUND_X_HEAL    34 /*!< The player was healed full health */
#define SOUND_BITE      35 /*!< A monster bites you */
#define SOUND_CLAW      36 /*!< A monster claws you */
#define SOUND_M_SPELL   37 /*!< A monster casts a miscellaneous spell */
#define SOUND_SUMMON    38 /*!< A monster casts a summoning spell  */
#define SOUND_BREATH    39 /*!< A monster breathes */
#define SOUND_BALL      40 /*!< A monster casts a ball / bolt spell */
#define SOUND_M_HEAL    41 /*!< A monster heals itself somehow */
#define SOUND_ATK_SPELL 42 /*!< A monster casts a misc. offensive spell */
#define SOUND_EVIL      43 /*!< Something nasty has just happened! */
#define SOUND_TOUCH     44 /*!< A monster touches you */
#define SOUND_STING     45 /*!< A monster stings you */
#define SOUND_CRUSH     46 /*!< A monster crushes / envelopes you */
#define SOUND_SLIME     47 /*!< A monster drools/spits/etc on you */
#define SOUND_WAIL      48 /*!< A monster wails */
#define SOUND_WINNER    49 /*!< Just won the game! */
#define SOUND_FIRE      50 /*!< An item was burned  */
#define SOUND_ACID      51 /*!< An item was destroyed by acid */
#define SOUND_ELEC      52 /*!< An item was destroyed by electricity */
#define SOUND_COLD      53 /*!< An item was shattered */
#define SOUND_ILLEGAL   54 /*!< Illegal command attempted */
#define SOUND_FAIL      55 /*!< Fail to get a spell off / activate an item */
#define SOUND_WAKEUP    56 /*!< A monster wakes up */
#define SOUND_INVULN    57 /*!< Invulnerability! */
#define SOUND_FALL      58 /*!< Falling through a trapdoor... */
#define SOUND_PAIN      59 /*!< A monster is in pain! */
#define SOUND_DESTITEM  60 /*!< An item was destroyed by misc. means */
#define SOUND_MOAN      61 /*!< A monster makes a moan/beg/insult attack */
#define SOUND_SHOW      62 /*!< A monster makes a "show" attack */
#define SOUND_UNUSED    63 /*!< (no sound for gaze attacks) */
#define SOUND_EXPLODE   64 /*!< Something (or somebody) explodes */
#define SOUND_GLASS     65 /*!< A glass feature was crashed */
#define SOUND_REFLECT   66 /*!< A bolt was reflected */

/*
 * Mega-Hack -- maximum known sounds
 */
#define SOUND_MAX 67 /*!< 効果音定義の最大数 */


/*** Hack ***/


/*
 * Hack -- attempt to reduce various values
 */
#ifdef ANGBAND_LITE
# undef MACRO_MAX
# define MACRO_MAX      128
# undef QUARK_MAX
# define QUARK_MAX      128
# undef MESSAGE_MAX
# define MESSAGE_MAX    128
# undef MESSAGE_BUF
# define MESSAGE_BUF    4096
#endif





/*
 * Available graphic modes
 */
#define GRAPHICS_NONE       0
#define GRAPHICS_ORIGINAL   1
#define GRAPHICS_ADAM_BOLT  2
#define GRAPHICS_HENGBAND   3

/*
 * Modes for the random name generator
 */
#define NAME_DWARF  1
#define NAME_ELF    2
#define NAME_GNOME  3
#define NAME_HOBBIT 4
#define NAME_HUMAN  5
#define NAME_ORC    6

#ifdef JP
#define JVERB_AND 1
#define JVERB_TO  2
#define JVERB_OR  3
#endif

/*
 * Modes for the tokenizer
 */
#define TOKENIZE_CHECKQUOTE 0x01  /* Special handling of single quotes */

/*
 * Parse errors
 */
#define PARSE_ERROR_GENERIC                  1
#define PARSE_ERROR_ABSOLETE_FILE            2
#define PARSE_ERROR_MISSING_RECORD_HEADER    3
#define PARSE_ERROR_NON_SEQUENTIAL_RECORDS   4
#define PARSE_ERROR_INVALID_FLAG             5
#define PARSE_ERROR_UNDEFINED_DIRECTIVE      6
#define PARSE_ERROR_OUT_OF_MEMORY            7
#define PARSE_ERROR_OUT_OF_BOUNDS            8
#define PARSE_ERROR_TOO_FEW_ARGUMENTS        9
#define PARSE_ERROR_UNDEFINED_TERRAIN_TAG   10
#define PARSE_ERROR_MAX                     11

#define GINOU_MAX      10

#define NO_TOWN 6
#define SECRET_TOWN 5

#define NIKKI_HIGAWARI     0
#define NIKKI_BUNSHOU      1
#define NIKKI_ART          2
#define NIKKI_UNIQUE       3
#define NIKKI_FIX_QUEST_C  4
#define NIKKI_FIX_QUEST_F  5
#define NIKKI_RAND_QUEST_C 6
#define NIKKI_RAND_QUEST_F 7
#define NIKKI_MAXDEAPTH    8
#define NIKKI_TRUMP        9
#define NIKKI_STAIR       10
#define NIKKI_RECALL      11
#define NIKKI_TO_QUEST    12
#define NIKKI_TELE_LEV    13
#define NIKKI_BUY         14
#define NIKKI_SELL        15
#define NIKKI_ARENA       16
#define NIKKI_HANMEI      17
#define NIKKI_LEVELUP     18
#define NIKKI_GAMESTART   19
#define NIKKI_WIZ_TELE    20
#define NIKKI_NAMED_PET   21
#define NIKKI_PAT_TELE    22
#define NIKKI_ART_SCROLL  23

#define NIKKI_WIZARD_LOG  24


#define RECORD_NAMED_PET_NAME        0
#define RECORD_NAMED_PET_UNNAME      1
#define RECORD_NAMED_PET_DISMISS     2
#define RECORD_NAMED_PET_DEATH       3
#define RECORD_NAMED_PET_MOVED       4
#define RECORD_NAMED_PET_LOST_SIGHT  5
#define RECORD_NAMED_PET_DESTROY     6
#define RECORD_NAMED_PET_EARTHQUAKE  7
#define RECORD_NAMED_PET_GENOCIDE    8
#define RECORD_NAMED_PET_WIZ_ZAP     9
#define RECORD_NAMED_PET_TELE_LEVEL  10
#define RECORD_NAMED_PET_BLAST       11
#define RECORD_NAMED_PET_HEAL_LEPER  12
#define RECORD_NAMED_PET_COMPACT     13
#define RECORD_NAMED_PET_LOSE_PARENT 14

#define MAX_MANE 16
#define MAX_MONSPELLS 96
#define MONSPELL_TYPE_BOLT 1
#define MONSPELL_TYPE_BALL 2
#define MONSPELL_TYPE_BREATH 3
#define MONSPELL_TYPE_SUMMON 4
#define MONSPELL_TYPE_OTHER 5

#define EATER_EXT 36
#define EATER_CHARGE 0x10000L
#define EATER_ROD_CHARGE 0x10L

#define MAX_KUBI 20

#define DETECT_RAD_DEFAULT 30
#define DETECT_RAD_MAP     30
#define DETECT_RAD_ALL     255

/* Maximum "Nazguls" number */
#define MAX_NAZGUL_NUM 5

#define DO_AUTOPICK       0x01
#define DO_AUTODESTROY    0x02
#define DO_DISPLAY        0x04
#define DONT_AUTOPICK     0x08
#define ITEM_DISPLAY      0x10
#define DO_QUERY_AUTOPICK 0x20


#define MAGIC_GLOVE_REDUCE_MANA 0x0001
#define MAGIC_FAIL_5PERCENT     0x0002
#define MAGIC_GAIN_EXP          0x0004

#define VIRTUE_LARGE 1
#define VIRTUE_SMALL 2

#define SPELL_DD_S 27
#define SPELL_DD_T 13
#define SPELL_SW   22
#define SPELL_KABE 20

#define KNOW_STAT   0x01
#define KNOW_HPRATE 0x02

/*
 * Music songs
 */
#define MUSIC_NONE              0
#define MUSIC_SLOW              1
#define MUSIC_BLESS             2
#define MUSIC_STUN              3
#define MUSIC_L_LIFE            4
#define MUSIC_FEAR              5
#define MUSIC_HERO              6
#define MUSIC_MIND              7
#define MUSIC_STEALTH           8
#define MUSIC_ID                9
#define MUSIC_CONF              10
#define MUSIC_SOUND             11
#define MUSIC_CHARM             12
#define MUSIC_WALL              13
#define MUSIC_RESIST            14
#define MUSIC_SPEED             15
#define MUSIC_DISPEL            16
#define MUSIC_SARUMAN           17
#define MUSIC_QUAKE             18
#define MUSIC_STASIS            19
#define MUSIC_SHERO             20
#define MUSIC_H_LIFE            21
#define MUSIC_INVULN            22
#define MUSIC_PSI               23

#define MUSIC_DETECT            101

#define SINGING_SONG_EFFECT(P_PTR) ((P_PTR)->magic_num1[0])
#define INTERUPTING_SONG_EFFECT(P_PTR) ((P_PTR)->magic_num1[1])
#define SINGING_COUNT(P_PTR) ((P_PTR)->magic_num1[2])
#define SINGING_SONG_ID(P_PTR) ((P_PTR)->magic_num2[0])
#define music_singing(X) ((p_ptr->pclass == CLASS_BARD) && (p_ptr->magic_num1[0] == (X)))
#define music_singing_any() ((p_ptr->pclass == CLASS_BARD) && p_ptr->magic_num1[0])

/*
 *  Special essence id for Weapon smith
 */
#define MIN_SPECIAL_ESSENCE 200

#define ESSENCE_ATTACK        (MIN_SPECIAL_ESSENCE + 0)
#define ESSENCE_AC            (MIN_SPECIAL_ESSENCE + 1)
#define ESSENCE_TMP_RES_ACID  (MIN_SPECIAL_ESSENCE + 2)
#define ESSENCE_TMP_RES_ELEC  (MIN_SPECIAL_ESSENCE + 3)
#define ESSENCE_TMP_RES_FIRE  (MIN_SPECIAL_ESSENCE + 4)
#define ESSENCE_TMP_RES_COLD  (MIN_SPECIAL_ESSENCE + 5)
#define ESSENCE_SH_FIRE       (MIN_SPECIAL_ESSENCE + 6)
#define ESSENCE_SH_ELEC       (MIN_SPECIAL_ESSENCE + 7)
#define ESSENCE_SH_COLD       (MIN_SPECIAL_ESSENCE + 8)
#define ESSENCE_RESISTANCE    (MIN_SPECIAL_ESSENCE + 9)
#define ESSENCE_SUSTAIN       (MIN_SPECIAL_ESSENCE + 10)
#define ESSENCE_SLAY_GLOVE    (MIN_SPECIAL_ESSENCE + 11)


#define DUNGEON_FEAT_PROB_NUM 3

/*
 * Flags for save/load temporal saved floor file
 */
#define SLF_SECOND     	 0x0001  /* Called from another save/load function */
#define SLF_NO_KILL      0x0002  /* Don't kill temporal files */


/*
 * Flags for wr_item()/rd_item()
 */
#define SAVE_ITEM_PVAL         0x00000001
#define SAVE_ITEM_DISCOUNT     0x00000002
#define SAVE_ITEM_NUMBER       0x00000004
#define SAVE_ITEM_NAME1        0x00000008
#define SAVE_ITEM_NAME2        0x00000010
#define SAVE_ITEM_TIMEOUT      0x00000020
#define SAVE_ITEM_TO_H         0x00000040
#define SAVE_ITEM_TO_D         0x00000080
#define SAVE_ITEM_TO_A         0x00000100
#define SAVE_ITEM_AC           0x00000200
#define SAVE_ITEM_DD           0x00000400
#define SAVE_ITEM_DS           0x00000800
#define SAVE_ITEM_IDENT        0x00001000
#define SAVE_ITEM_MARKED       0x00002000
#define SAVE_ITEM_ART_FLAGS0   0x00004000
#define SAVE_ITEM_ART_FLAGS1   0x00008000
#define SAVE_ITEM_ART_FLAGS2   0x00010000
#define SAVE_ITEM_ART_FLAGS3   0x00020000
#define SAVE_ITEM_CURSE_FLAGS  0x00040000
#define SAVE_ITEM_HELD_M_IDX   0x00080000
#define SAVE_ITEM_XTRA1        0x00100000
#define SAVE_ITEM_XTRA2        0x00200000
#define SAVE_ITEM_XTRA3        0x00400000
#define SAVE_ITEM_XTRA4        0x00800000
#define SAVE_ITEM_XTRA5        0x01000000
#define SAVE_ITEM_FEELING      0x02000000
#define SAVE_ITEM_INSCRIPTION  0x04000000
#define SAVE_ITEM_ART_NAME     0x08000000
#define SAVE_ITEM_ART_FLAGS4   0x10000000


/*
 * Flags for wr_monster()/rd_monster()
 */
#define SAVE_MON_AP_R_IDX     0x00000001
#define SAVE_MON_SUB_ALIGN    0x00000002
#define SAVE_MON_CSLEEP       0x00000004
#define SAVE_MON_FAST         0x00000008
#define SAVE_MON_SLOW         0x00000010
#define SAVE_MON_STUNNED      0x00000020
#define SAVE_MON_CONFUSED     0x00000040
#define SAVE_MON_MONFEAR      0x00000080
#define SAVE_MON_TARGET_Y     0x00000100
#define SAVE_MON_TARGET_X     0x00000200
#define SAVE_MON_INVULNER     0x00000400
#define SAVE_MON_SMART        0x00000800
#define SAVE_MON_EXP          0x00001000
#define SAVE_MON_MFLAG2       0x00002000
#define SAVE_MON_NICKNAME     0x00004000
#define SAVE_MON_PARENT       0x00008000


/*
 * Constant for kinds of mimic
 */
#define MIMIC_NONE       0
#define MIMIC_DEMON      1
#define MIMIC_DEMON_LORD 2
#define MIMIC_VAMPIRE    3


#define MIMIC_FLAGS choice
#define MIMIC_IS_NONLIVING 0x00000001
#define MIMIC_IS_DEMON     0x00000002
#define MIMIC_IS_UNDEAD    0x00000004


#define prace_is_(A) (!p_ptr->mimic_form && (p_ptr->prace == A))

/* Sub-alignment flags for neutral monsters */
#define SUB_ALIGN_NEUTRAL 0x0000
#define SUB_ALIGN_EVIL    0x0001
#define SUB_ALIGN_GOOD    0x0002

#define IS_WIZARD_CLASS() \
	(p_ptr->pclass == CLASS_MAGE || p_ptr->pclass == CLASS_HIGH_MAGE || p_ptr->pclass == CLASS_SORCERER || p_ptr->pclass == CLASS_MAGIC_EATER || p_ptr->pclass == CLASS_BLUE_MAGE)

/* Multishadow effects is determined by current_world_ptr->game_turn */
#define CHECK_MULTISHADOW() (p_ptr->multishadow && (current_world_ptr->game_turn & 1))

/* Is "teleport level" ineffective to this target? */
#define TELE_LEVEL_IS_INEFF(TARGET) \
	(p_ptr->inside_arena || p_ptr->inside_battle || \
	 (p_ptr->inside_quest && !random_quest_number(current_floor_ptr->dun_level)) || \
	 (((TARGET) <= 0) && (quest_number(current_floor_ptr->dun_level) || (current_floor_ptr->dun_level >= d_info[p_ptr->dungeon_idx].maxdepth)) && \
	  (current_floor_ptr->dun_level >= 1) && ironman_downward))


/*
 * Max numbers of macro trigger names
 */
#define MAX_MACRO_MOD 12
#define MAX_MACRO_TRIG 200 /*!< 登録を許すマクロ（トリガー）の最大数 */

/* Max size of screen dump buffer */
#define SCREEN_BUF_MAX_SIZE (4 * 65536)


/*
 * Special key code used for inkey_special()
 */
#define SKEY_MOD_MASK     0x0f00
#define SKEY_MOD_SHIFT    0x0100
#define SKEY_MOD_CONTROL  0x0200

#define SKEY_MASK         0xf000
#define SKEY_DOWN   	  0xf001
#define SKEY_LEFT   	  0xf002
#define SKEY_RIGHT  	  0xf003
#define SKEY_UP     	  0xf004
#define SKEY_PGUP   	  0xf005
#define SKEY_PGDOWN 	  0xf006
#define SKEY_TOP    	  0xf007
#define SKEY_BOTTOM 	  0xf008

#define MTIMED_CSLEEP   0 /* Monster is sleeping */
#define MTIMED_FAST     1 /* Monster is temporarily fast */
#define MTIMED_SLOW     2 /* Monster is temporarily slow */
#define MTIMED_STUNNED  3 /* Monster is stunned */
#define MTIMED_CONFUSED 4 /* Monster is confused */
#define MTIMED_MONFEAR  5 /* Monster is afraid */
#define MTIMED_INVULNER 6 /* Monster is temporarily invulnerable */

#define MAX_MTIMED      7

#define MON_CSLEEP(M_PTR)   ((M_PTR)->mtimed[MTIMED_CSLEEP])
#define MON_FAST(M_PTR)     ((M_PTR)->mtimed[MTIMED_FAST])
#define MON_SLOW(M_PTR)     ((M_PTR)->mtimed[MTIMED_SLOW])
#define MON_STUNNED(M_PTR)  ((M_PTR)->mtimed[MTIMED_STUNNED])
#define MON_CONFUSED(M_PTR) ((M_PTR)->mtimed[MTIMED_CONFUSED])
#define MON_MONFEAR(M_PTR)  ((M_PTR)->mtimed[MTIMED_MONFEAR])
#define MON_INVULNER(M_PTR) ((M_PTR)->mtimed[MTIMED_INVULNER])

/*
 * Bit flags for screen_object()
 */
#define SCROBJ_FAKE_OBJECT  0x00000001
#define SCROBJ_FORCE_DETAIL 0x00000002

/*
 * For travel command (auto run)
 */
#define TRAVEL

/* Sniper */
#define SP_NONE          0
#define SP_LITE          1
#define SP_AWAY          2
#define SP_FIRE          3
#define SP_KILL_WALL     4
#define SP_COLD          5
#define SP_KILL_TRAP     6
#define SP_ELEC          7
#define SP_PIERCE        8
#define SP_RUSH          9
#define SP_DOUBLE        10
#define SP_EXPLODE       11
#define SP_EVILNESS      12
#define SP_HOLYNESS      13
#define SP_FINAL         14
#define SP_NEEDLE        15

#define CONCENT_RADAR_THRESHOLD 2
#define CONCENT_TELE_THRESHOLD  5

/* Hex */
#define hex_spelling_any() \
	((p_ptr->realm1 == REALM_HEX) && (p_ptr->magic_num1[0]))
#define hex_spelling(X) \
	((p_ptr->realm1 == REALM_HEX) && (p_ptr->magic_num1[0] & (1L << (X))))
#define CASTING_HEX_FLAGS(P_PTR) ((P_PTR)->magic_num1[0])
#define CASTING_HEX_NUM(P_PTR) ((P_PTR)->magic_num2[0])
#define HEX_REVENGE_POWER(P_PTR) ((P_PTR)->magic_num1[2])
#define HEX_REVENGE_TURN(P_PTR) ((P_PTR)->magic_num2[2])
#define HEX_REVENGE_TYPE(P_PTR) ((P_PTR)->magic_num2[1])

/*
  Language selection macro
*/
#ifdef JP
#define _(JAPANESE,ENGLISH) (JAPANESE)
#else
#define _(JAPANESE,ENGLISH) (ENGLISH)
#endif

/* Lite flag macro */
#define have_lite_flag(ARRAY) \
	(have_flag(ARRAY, TR_LITE_1) || have_flag(ARRAY, TR_LITE_2) || have_flag(ARRAY, TR_LITE_3))

#define have_dark_flag(ARRAY) \
	(have_flag(ARRAY, TR_LITE_M1) || have_flag(ARRAY, TR_LITE_M2) || have_flag(ARRAY, TR_LITE_M3))

/* Spell Type flag */
#define MONSTER_TO_PLAYER     0x01
#define MONSTER_TO_MONSTER    0x02

/* summoning number */
#define S_NUM_6     (easy_band ? 2 : 6)
#define S_NUM_4     (easy_band ? 1 : 4)

/* monster spell number */
#define RF4_SPELL_START 32 * 3
#define RF5_SPELL_START 32 * 4
#define RF6_SPELL_START 32 * 5

#define RF4_SPELL_SIZE 32
#define RF5_SPELL_SIZE 32
#define RF6_SPELL_SIZE 32

/* Spell Damage Calc Flag*/
#define DAM_ROLL 1
#define DAM_MAX 2
#define DAM_MIN 3
#define DICE_NUM 4
#define DICE_SIDE 5
#define DICE_MULT 6
#define DICE_DIV 7
#define BASE_DAM 8

/* Cheat Info Type */
#define CHEAT_OBJECT 0
#define CHEAT_MONSTER 1
#define CHEAT_DUNGEON 2
#define CHEAT_MISC 3

#define COMMAND_ARG_REST_UNTIL_DONE -2   /*!<休憩コマンド引数 … 必要な分だけ回復 */
#define COMMAND_ARG_REST_FULL_HEALING -1 /*!<休憩コマンド引数 … HPとMPが全回復するまで */

/*!
 * チートオプションの最大数 / Number of cheating options
 */
#define CHEAT_MAX 10
