﻿/*!
 *  @file defines.h
 *  @brief 主要なマクロ定義ヘッダ / Purpose: global constants and macro definitions
 *  @date 2014/01/02
 *  @author
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke\n
 *\n
 * This software may be copied and distributed for educational, research,\n
 * and not for profit purposes provided that this copyright and statement\n
 * are included in all such copies.  Other copyrights may also apply.\n
 *  @details
 * Do not edit this file unless you know *exactly* what you are doing.\n
 *\n
 * Some of the values in this file were chosen to preserve game balance,\n
 * while others are hard-coded based on the format of old save-files, the\n
 * definition of arrays in various places, mathematical properties, fast\n
 * computation, storage limits, or the format of external text files.\n
 *\n
 * Changing some of these values will induce crashes or memory errors or\n
 * savefile mis-reads.  Most of the comments in this file are meant as\n
 * reminders, not complete descriptions, and even a complete knowledge\n
 * of the source may not be sufficient to fully understand the effects\n
 * of changing certain definitions.\n
 *\n
 * Lastly, note that the code does not always use the symbolic constants\n
 * below, and sometimes uses various hard-coded values that may not even\n
 * be defined in this file, but which may be related to definitions here.\n
 * This is of course bad programming practice, but nobody is perfect...\n
 *\n
 * For example, there are MANY things that depend on the screen being\n
 * 80x24, with the top line used for messages, the bottom line being\n
 * used for status, and exactly 22 lines used to show the dungeon.\n
 * Just because your screen can hold 46 lines does not mean that the\n
 * game will work if you try to use 44 lines to show the dungeon.\n
 *\n
 * You have been warned.\n
 */

#define VERSION_NAME "Hengband" /*!< バリアント名称 / Name of the version/variant */

/*!
 * @brief ゲームのバージョン番号定義 / "Program Version Number" of the game
 * @details
 * 本FAKE_VERSIONそのものは未使用である。Zangと整合性を合わせるための疑似的処理のためFAKE_VER_MAJORは実値-10が該当のバージョン番号となる。
 * <pre>
 * FAKE_VER_MAJOR=1,2 were reserved for ZAngband version 1.x.x/2.x.x .
 * Program Version of Hengband version is
 *   "(FAKE_VER_MAJOR-10).(FAKE_VER_MINOR).(FAKE_VER_PATCH)".
 * </pre>
 */
#define FAKE_VERSION 0

#define FAKE_VER_MAJOR 12 /*!< ゲームのバージョン番号定義(メジャー番号 + 10) */
#define FAKE_VER_MINOR 2 /*!< ゲームのバージョン番号定義(マイナー番号) */
#define FAKE_VER_PATCH 1 /*!< ゲームのバージョン番号定義(パッチ番号) */
#define FAKE_VER_EXTRA 5 /*!< ゲームのバージョン番号定義(エクストラ番号) */


 /*!
  * @brief バージョンが開発版が安定版かを返す
  */
#define	IS_STABLE_VERSION (FAKE_VER_MINOR % 2 == 0 && FAKE_VER_EXTRA == 0)

/*!
 * @brief セーブファイル上のバージョン定義(メジャー番号) / "Savefile Version Number" for Hengband 1.1.1 and later
 * @details
 * 当面FAKE_VER_*を参照しておく。
 * <pre>
 * First three digits may be same as the Program Version.  But not
 * always same.  It means that newer version may preserves lower
 * compatibility with the older version.
 * For example, newer Hengband 1.4.4 creates savefiles marked with
 * Savefile Version 1.4.0.0 .  It means that Hengband 1.4.0 can load a
 * savefile of Hengband 1.4.4 (lower compatibility!).
 * Upper compatibility is always guaranteed.
 * </pre>
 */
#define H_VER_MAJOR (FAKE_VER_MAJOR-10) /*!< セーブファイル上のバージョン定義(メジャー番号) */
#define H_VER_MINOR FAKE_VER_MINOR /*!< セーブファイル上のバージョン定義(マイナー番号) */
#define H_VER_PATCH FAKE_VER_PATCH /*!< セーブファイル上のバージョン定義(パッチ番号) */
#define H_VER_EXTRA FAKE_VER_EXTRA /*!< セーブファイル上のバージョン定義(エクストラ番号) */

#define ANGBAND_2_8_1 /*!< Angband 2.8.1以降から有効な処理分岐を定義 */
#define ZANGBAND /*!< Zangband 以降から有効な処理分岐を定義 */

/*!
 * @brief generate.cで用いられる基本的なブロック数単位(垂直方向)
 * Number of grids in each block (vertically) Probably hard-coded to 11, see "generate.c"
 */
#define BLOCK_HGT 11

/*!
 * @brief generate.cで用いられる基本的なブロック数単位(水平方向)
 * Number of grids in each block (horizontally) Probably hard-coded to 11, see "generate.c"
 */
#define BLOCK_WID 11

/*!
 * @brief 表示上の基本的なパネル単位(垂直方向、BLOCK_HGTの倍数で設定すること)
 * Number of grids in each panel (vertically) Must be a multiple of BLOCK_HGT
 */
#define PANEL_HGT 11

/*!
 * @brief 表示上の基本的なパネル単位(水平方向、BLOCK_WIDの倍数で設定すること)
 * Number of grids in each panel (horizontally) Must be a multiple of BLOCK_WID
 */
#define PANEL_WID 33

/*!
 * @brief 表示上の基本的なブロック単位(垂直方向、PANEL_HGTの倍数で設定すること)
 * Number of grids used to display the dungeon (vertically). Must be a multiple of 11, probably hard-coded to 22.
 */
#define SCREEN_HGT 22

/*!
 * @brief 表示上の基本的なブロック単位(水平方向、PANEL_WIDの倍数で設定すること)
 * Number of grids used to display the dungeon (horizontally). Must be a multiple of 33, probably hard-coded to 66.
 */
#define SCREEN_WID 66

/*!
 * @brief 表示上のダンジョンの最大垂直サイズ(SCREEN_HGTの3倍が望ましい)
 * Maximum dungeon height in grids, must be a multiple of SCREEN_HGT, probably hard-coded to SCREEN_HGT * 3.
 */
#define MAX_HGT 66

/*!
 * @brief 表示上のダンジョンの最大水平サイズ(SCREEN_WIDの3倍が望ましい)
 * Maximum dungeon width in grids, must be a multiple of SCREEN_WID, probably hard-coded to SCREEN_WID * 3.
 */
#define MAX_WID 198

/*
 * Quest constants
 */
#define MIN_RANDOM_QUEST 40 /*<! ランダムクエストを割り当てるクエストIDの開始値 */
#define MAX_RANDOM_QUEST 49 /*<! ランダムクエストを割り当てるクエストIDの終了値 */

/*!
 * @brief 該当IDが固定クエストかどうかを判定する / Check is the quest index is "fixed"
 * @param Q_IDX クエストID
 * @return 固定クエストならばTRUEを返す
 */
#define is_fixed_quest_idx(Q_IDX) (((Q_IDX) < MIN_RANDOM_QUEST) || ((Q_IDX) > MAX_RANDOM_QUEST))

#define QUEST_TOWER1 5 /*<! 塔クエスト(第1階層)に割り振るクエストID */
#define QUEST_TOWER2 6 /*<! 塔クエスト(第2階層)に割り振るクエストID */
#define QUEST_TOWER3 7 /*<! 塔クエスト(第3階層)に割り振るクエストID */
#define QUEST_OBERON 8 /*<! オベロン打倒クエストに割り振るクエストID */
#define QUEST_SERPENT 9 /*<! サーペント打倒クエストに割り振るクエストID */

/*
 * Arena constants
 */
#define MAX_ARENA_MONS 41 /*<! 闘技場のイベント件数 -KMW- */
#define ARENA_DEFEATED_OLD_VER (-(MAX_SHORT)) /*<! 旧バージョンの闘技場敗北定義 */


#define MAX_STORES 10 /*!< store.c用の店舗の種類最大数 / Total number of stores (see "store.c", etc) */
#define MAX_BLDG 32 /*!< 施設の種類最大数 / Number of buildings */

/*
 * Store types
 */
#define STORE_GENERAL   0 /*!< 店舗の種類: 雑貨屋 */
#define STORE_ARMOURY   1 /*!< 店舗の種類: 防具屋 */
#define STORE_WEAPON    2 /*!< 店舗の種類: 武器屋 */
#define STORE_TEMPLE    3 /*!< 店舗の種類: 寺院 */
#define STORE_ALCHEMIST 4 /*!< 店舗の種類: 錬金術の店 */
#define STORE_MAGIC     5 /*!< 店舗の種類: 魔道具屋 */
#define STORE_BLACK     6 /*!< 店舗の種類: ブラック・マーケット */
#define STORE_HOME      7 /*!< 店舗の種類: 我が家 */
#define STORE_BOOK      8 /*!< 店舗の種類: 書店 */
#define STORE_MUSEUM    9 /*!< 店舗の種類: 博物館 */

#define BUILDING_NON_MEMBER 0 /*!< 不明(現在未使用) */
#define BUILDING_MEMBER     1 /*!< 不明(現在未使用) */
#define BUILDING_OWNER      2 /*!< 施設の種族/職業条件が一致している状態値 */

#define MAX_OWNERS      32 /*!< 各店舗毎の店主定義最大数 / Total number of owners per store (see "store.c", etc) */
#define MAX_SEXES        2 /*!< 性別の定義最大数 / Maximum number of player "sex" types (see "table.c", etc) */
#define MAX_CLASS       28 /*!< 職業の最大定義数 Maximum number of player "class" types (see "table.c", etc) */
#define MAX_SEIKAKU     12 /*!< 性格の最大定義数 */
#define MAX_PATRON      16 /*!< カオスパトロンの最大定義数 / The number of "patrons" available (for Chaos Warriors) */

/* ELDRITCH_HORRORによるsanity blast処理に関するメッセージの最大数 / Number of entries in the sanity-blast descriptions */
#define MAX_SAN_HORROR 20 /*!< 恐ろしい対象の形容数(正常時) */
#define MAX_SAN_FUNNY 22  /*!< 恐ろしい対象の形容数(幻覚時) */
#define MAX_SAN_COMMENT 5 /*!< 恐ろしい対象を見たときの絶叫メッセージ数(幻覚時) */

/* カオスパトロンからの報酬種別定義 / Chaos Warrior: Reward types: */
#define REW_POLY_SLF    1   /*!< カオスパトロンからの報酬: 自己変容 */
#define REW_GAIN_EXP    2   /*!< カオスパトロンからの報酬: 経験値増加 */
#define REW_LOSE_EXP    3   /*!< カオスパトロンからの報酬: 経験値減少 */
#define REW_GOOD_OBJ    4   /*!< カオスパトロンからの報酬: GOODなアイテム単体の下賜 */
#define REW_GREA_OBJ    5   /*!< カオスパトロンからの報酬: GREATなアイテム単体の下賜 */
#define REW_CHAOS_WP    6   /*!< カオスパトロンからの報酬: 混沌武器の下賜 */
#define REW_GOOD_OBS    7   /*!< カオスパトロンからの報酬: GOODなアイテム複数の下賜 */
#define REW_GREA_OBS    8   /*!< カオスパトロンからの報酬: GREATなアイテム複数の下賜 */
#define REW_TY_CURSE    9   /*!< カオスパトロンからの報酬: 太古の怨念 */
#define REW_SUMMON_M    10  /*!< カオスパトロンからの報酬: 敵対的なモンスターの召喚(通常) */
#define REW_H_SUMMON    11  /*!< カオスパトロンからの報酬: 敵対的なモンスターの召喚(hi-summon) */
#define REW_DO_HAVOC    12  /*!< カオスパトロンからの報酬: 混沌招来 */
#define REW_GAIN_ABL    13  /*!< カオスパトロンからの報酬: 増強 */
#define REW_LOSE_ABL    14  /*!< カオスパトロンからの報酬: 1能力低下 */
#define REW_RUIN_ABL    15  /*!< カオスパトロンからの報酬: 全能力低下 */
#define REW_AUGM_ABL    16  /*!< カオスパトロンからの報酬: 1能力上昇 */
#define REW_POLY_WND    17  /*!< カオスパトロンからの報酬: 傷の変化 */
#define REW_HEAL_FUL    18  /*!< カオスパトロンからの報酬: 完全回復 */
#define REW_HURT_LOT    19  /*!< カオスパトロンからの報酬: 分解の球によるダメージ */
#define REW_CURSE_WP    20  /*!< カオスパトロンからの報酬: 武器呪縛 */
#define REW_CURSE_AR    21  /*!< カオスパトロンからの報酬: 防具呪縛 */
#define REW_PISS_OFF    22  /*!< カオスパトロンからの報酬: 苛立ち */
#define REW_WRATH       23  /*!< カオスパトロンからの報酬: 怒り */
#define REW_DESTRUCT    24  /*!< カオスパトロンからの報酬: *破壊* */
#define REW_GENOCIDE    25  /*!< カオスパトロンからの報酬: シンボル抹殺 */
#define REW_MASS_GEN    26  /*!< カオスパトロンからの報酬: 周辺抹殺 */
#define REW_DISPEL_C    27  /*!< カオスパトロンからの報酬: モンスター退散 */
#define REW_UNUSED_1    28  /*!< カオスパトロンからの報酬: 未使用 */
#define REW_UNUSED_2    29  /*!< カオスパトロンからの報酬: 未使用 */
#define REW_UNUSED_3    30  /*!< カオスパトロンからの報酬: 未使用 */
#define REW_UNUSED_4    31  /*!< カオスパトロンからの報酬: 未使用 */
#define REW_UNUSED_5    32  /*!< カオスパトロンからの報酬: 未使用 */
#define REW_IGNORE      33  /*!< カオスパトロンからの報酬: 無視 */
#define REW_SER_UNDE    34  /*!< カオスパトロンからの報酬: アンデッドの下僕下賜 */
#define REW_SER_DEMO    35  /*!< カオスパトロンからの報酬: 悪魔の下僕下賜 */
#define REW_SER_MONS    36  /*!< カオスパトロンからの報酬: モンスターの下僕下賜 */

/* カオスの変異 / Chaos mutations */

/* レイシャル型の変異 / "Activatable" mutations must be in MUT1_* */
#define MUT1_SPIT_ACID                  0x00000001L /*!< 突然変異: 酸の唾 */
#define MUT1_BR_FIRE                    0x00000002L /*!< 突然変異: 炎のブレス */
#define MUT1_HYPN_GAZE                  0x00000004L /*!< 突然変異: 催眠睨み */
#define MUT1_TELEKINES                  0x00000008L /*!< 突然変異: 念動力 */
#define MUT1_VTELEPORT                  0x00000010L /*!< 突然変異: テレポート / Voluntary teleport */
#define MUT1_MIND_BLST                  0x00000020L /*!< 突然変異: 精神攻撃 */
#define MUT1_RADIATION                  0x00000040L /*!< 突然変異: 放射能 */
#define MUT1_VAMPIRISM                  0x00000080L /*!< 突然変異: 吸血 */
#define MUT1_SMELL_MET                  0x00000100L /*!< 突然変異: 金属嗅覚 */
#define MUT1_SMELL_MON                  0x00000200L /*!< 突然変異: 敵臭嗅覚 */
#define MUT1_BLINK                      0x00000400L /*!< 突然変異: ショート・テレポート */
#define MUT1_EAT_ROCK                   0x00000800L /*!< 突然変異: 岩喰い */
#define MUT1_SWAP_POS                   0x00001000L /*!< 突然変異: 位置交換 */
#define MUT1_SHRIEK                     0x00002000L /*!< 突然変異: 叫び */
#define MUT1_ILLUMINE                   0x00004000L /*!< 突然変異: 照明 */
#define MUT1_DET_CURSE                  0x00008000L /*!< 突然変異: 呪い感知 */
#define MUT1_BERSERK                    0x00010000L /*!< 突然変異: 狂戦士化 */
#define MUT1_POLYMORPH                  0x00020000L /*!< 突然変異: 変身 */
#define MUT1_MIDAS_TCH                  0x00040000L /*!< 突然変異: ミダスの手 */
#define MUT1_GROW_MOLD                  0x00080000L /*!< 突然変異: カビ発生 */
#define MUT1_RESIST                     0x00100000L /*!< 突然変異: エレメント耐性 */
#define MUT1_EARTHQUAKE                 0x00200000L /*!< 突然変異: 地震 */
#define MUT1_EAT_MAGIC                  0x00400000L /*!< 突然変異: 魔力喰い */
#define MUT1_WEIGH_MAG                  0x00800000L /*!< 突然変異: 魔力感知 */
#define MUT1_STERILITY                  0x01000000L /*!< 突然変異: 増殖阻止 */
#define MUT1_PANIC_HIT                  0x02000000L /*!< 突然変異: ヒットアンドアウェイ */
#define MUT1_DAZZLE                     0x04000000L /*!< 突然変異: 眩惑 */
#define MUT1_LASER_EYE                  0x08000000L /*!< 突然変異: レーザー・アイ */
#define MUT1_RECALL                     0x10000000L /*!< 突然変異: 帰還 */
#define MUT1_BANISH                     0x20000000L /*!< 突然変異: 邪悪消滅 */
#define MUT1_COLD_TOUCH                 0x40000000L /*!< 突然変異: 凍結の手 */
#define MUT1_LAUNCHER                   0x80000000L /*!< 突然変異: アイテム投げ */

/* 常時効果つきの変異1 / Randomly activating mutations must be MUT2_* */
#define MUT2_BERS_RAGE                  0x00000001L /*!< 突然変異: 狂戦士化の発作 */
#define MUT2_COWARDICE                  0x00000002L /*!< 突然変異: 臆病 */
#define MUT2_RTELEPORT                  0x00000004L /*!< 突然変異: ランダムテレポート / Random teleport, instability */
#define MUT2_ALCOHOL                    0x00000008L /*!< 突然変異: アルコール分泌 */
#define MUT2_HALLU                      0x00000010L /*!< 突然変異: 幻覚を引き起こす精神錯乱 */
#define MUT2_FLATULENT                  0x00000020L /*!< 突然変異: 猛烈な屁 */
#define MUT2_SCOR_TAIL                  0x00000040L /*!< 突然変異: サソリの尻尾 */
#define MUT2_HORNS                      0x00000080L /*!< 突然変異: ツノ */
#define MUT2_BEAK                       0x00000100L /*!< 突然変異: クチバシ */
#define MUT2_ATT_DEMON                  0x00000200L /*!< 突然変異: デーモンを引き付ける */
#define MUT2_PROD_MANA                  0x00000400L /*!< 突然変異: 制御できない魔力のエネルギー */
#define MUT2_SPEED_FLUX                 0x00000800L /*!< 突然変異: ランダムな加減速 */
#define MUT2_BANISH_ALL                 0x00001000L /*!< 突然変異: ランダムなモンスター消滅 */
#define MUT2_EAT_LIGHT                  0x00002000L /*!< 突然変異: 光源喰い */
#define MUT2_TRUNK                      0x00004000L /*!< 突然変異: 象の鼻 */
#define MUT2_ATT_ANIMAL                 0x00008000L /*!< 突然変異: 動物を引き寄せる */
#define MUT2_TENTACLES                  0x00010000L /*!< 突然変異: 邪悪な触手 */
#define MUT2_RAW_CHAOS                  0x00020000L /*!< 突然変異: 純カオス */
#define MUT2_NORMALITY                  0x00040000L /*!< 突然変異: ランダムな変異の消滅 */
#define MUT2_WRAITH                     0x00080000L /*!< 突然変異: ランダムな幽体化 */
#define MUT2_POLY_WOUND                 0x00100000L /*!< 突然変異: ランダムな傷の変化 */
#define MUT2_WASTING                    0x00200000L /*!< 突然変異: 衰弱 */
#define MUT2_ATT_DRAGON                 0x00400000L /*!< 突然変異: ドラゴンを引き寄せる */
#define MUT2_WEIRD_MIND                 0x00800000L /*!< 突然変異: ランダムなテレパシー */
#define MUT2_NAUSEA                     0x01000000L /*!< 突然変異: 落ち着きの無い胃 */
#define MUT2_CHAOS_GIFT                 0x02000000L /*!< 突然変異: カオスパトロン */
#define MUT2_WALK_SHAD                  0x04000000L /*!< 突然変異: ランダムな現実変容 */
#define MUT2_WARNING                    0x08000000L /*!< 突然変異: 警告 */
#define MUT2_INVULN                     0x10000000L /*!< 突然変異: ランダムな無敵化 */
#define MUT2_SP_TO_HP                   0x20000000L /*!< 突然変異: ランダムなMPからHPへの変換 */
#define MUT2_HP_TO_SP                   0x40000000L /*!< 突然変異: ランダムなHPからMPへの変換 */
#define MUT2_DISARM                     0x80000000L /*!< 突然変異: ランダムな武器落とし */


/* 常時効果つきの変異2 / Other mutations will be mainly in MUT3_* */
#define MUT3_HYPER_STR                  0x00000001L /*!< 突然変異: 超人的な力 */
#define MUT3_PUNY                       0x00000002L /*!< 突然変異: 虚弱 */
#define MUT3_HYPER_INT                  0x00000004L /*!< 突然変異: 生体コンピュータ */
#define MUT3_MORONIC                    0x00000008L /*!< 突然変異: 精神薄弱 */
#define MUT3_RESILIENT                  0x00000010L /*!< 突然変異: 弾力のある体 */
#define MUT3_XTRA_FAT                   0x00000020L /*!< 突然変異: 異常な肥満 */
#define MUT3_ALBINO                     0x00000040L /*!< 突然変異: アルビノ */
#define MUT3_FLESH_ROT                  0x00000080L /*!< 突然変異: 腐敗した肉体 */
#define MUT3_SILLY_VOI                  0x00000100L /*!< 突然変異: 間抜けなキーキー声 */
#define MUT3_BLANK_FAC                  0x00000200L /*!< 突然変異: のっぺらぼう */
#define MUT3_ILL_NORM                   0x00000400L /*!< 突然変異: 幻影に覆われた体 */
#define MUT3_XTRA_EYES                  0x00000800L /*!< 突然変異: 第三の目 */
#define MUT3_MAGIC_RES                  0x00001000L /*!< 突然変異: 魔法防御 */
#define MUT3_XTRA_NOIS                  0x00002000L /*!< 突然変異: 騒音 */
#define MUT3_INFRAVIS                   0x00004000L /*!< 突然変異: 赤外線視力 */
#define MUT3_XTRA_LEGS                  0x00008000L /*!< 突然変異: 追加の脚 */
#define MUT3_SHORT_LEG                  0x00010000L /*!< 突然変異: 短い脚 */
#define MUT3_ELEC_TOUC                  0x00020000L /*!< 突然変異: 電撃オーラ */
#define MUT3_FIRE_BODY                  0x00040000L /*!< 突然変異: 火炎オーラ */
#define MUT3_WART_SKIN                  0x00080000L /*!< 突然変異: イボ肌 */
#define MUT3_SCALES                     0x00100000L /*!< 突然変異: 鱗肌 */
#define MUT3_IRON_SKIN                  0x00200000L /*!< 突然変異: 鉄の肌 */
#define MUT3_WINGS                      0x00400000L /*!< 突然変異: 翼 */
#define MUT3_FEARLESS                   0x00800000L /*!< 突然変異: 恐れ知らず */
#define MUT3_REGEN                      0x01000000L /*!< 突然変異: 急回復 */
#define MUT3_ESP                        0x02000000L /*!< 突然変異: テレパシー */
#define MUT3_LIMBER                     0x04000000L /*!< 突然変異: しなやかな肉体 */
#define MUT3_ARTHRITIS                  0x08000000L /*!< 突然変異: 関節の痛み */
#define MUT3_BAD_LUCK                   0x10000000L /*!< 突然変異: 黒いオーラ(不運) */
#define MUT3_VULN_ELEM                  0x20000000L /*!< 突然変異: 元素攻撃弱点 */
#define MUT3_MOTION                     0x40000000L /*!< 突然変異: 正確で力強い動作 */
#define MUT3_GOOD_LUCK                  0x80000000L /*!< 突然変異: 白いオーラ(幸運) */



#define MAX_MA 17 /*!< 修行僧マーシャルアーツの技数 / Monk martial arts... */
#define MA_KNEE 1 /*!< 金的効果ID */
#define MA_SLOW 2 /*!< 膝蹴り効果ID */

#define MAX_MIND_POWERS  21 /*!< 超能力の数 / Mindcraft */


#ifdef JP
#define MAX_SILLY_ATTACK 41 /*!< 幻覚時のジョークメッセージ最大数 */
#else
#define MAX_SILLY_ATTACK 29 /*!< Hallucination stuff */
#endif

#define MIND_MINDCRAFTER    0 /*!< 特殊能力: 超能力 */
#define MIND_KI             1 /*!< 特殊能力: 練気 */
#define MIND_BERSERKER      2 /*!< 特殊能力: 怒り */
#define MIND_MIRROR_MASTER  3 /*!< 特殊能力: 鏡魔法 */
#define MIND_NINJUTSU       4 /*!< 特殊能力: 忍術 */


/*
 * Size of memory reserved for initialization of some arrays
 */
#define FAKE_NAME_SIZE  40 * 1024L /*!< ゲーム情報の種別毎に用意される名前用バッファの容量 */
#define FAKE_TEXT_SIZE 150 * 1024L /*!< ゲーム情報の種別毎に用意されるテキスト用バッファの容量 */
#define FAKE_TAG_SIZE   10 * 1024L /*!< ゲーム情報の種別毎に用意されるタグ用バッファの容量 */

#define MAX_HISCORES    999 /*!< スコア情報保存の最大数 / Maximum number of high scores in the high score file */



/*!
 * @brief ダンジョンの最深層 / Maximum dungeon level. 
 * @details
 * The player can never reach this level
 * in the dungeon, and this value is used for various calculations
 * involving object and monster creation.  It must be at least 100.
 * Setting it below 128 may prevent the creation of some objects.
 */
#define MAX_DEPTH       128 

#define MAX_SAVED_FLOORS       20 /*!< 保存フロアの最大数 / Maximum number of saved floors. */

/*!
 * @brief プレイヤー用光源処理配列サイズ / Maximum size of the "lite" array (see "cave.c")
 * @details Note that the "lite radius" will NEVER exceed 14, and we would
 * never require more than 581 entries in the array for circular "lite".
 */
#define LITE_MAX 600

/*!
 * @brief モンスター用光源処理配列サイズ / Maximum size of the "mon_lite" array (see "cave.c")
 * @details Note that the "view radius" will NEVER exceed 20, monster illumination
 * flags are dependent on CAVE_VIEW, and even if the "view" was octagonal,
 * we would never require more than 1520 entries in the array.
 */
#define MON_LITE_MAX 1536

/*!
 * @brief 視界処理配列サイズ / Maximum size of the "view" array (see "cave.c")
 * @details Note that the "view radius" will NEVER exceed 20, and even if the "view"
 * was octagonal, we would never require more than 1520 entries in the array.
 */
#define VIEW_MAX 1536

/*!
 * @brief 視界及び光源の過渡処理配列サイズ / Maximum size of the "temp" array (see "cave.c")
 * @details We must be as large as "VIEW_MAX" and "LITE_MAX" for proper functioning
 * of "update_view()" and "update_lite()".  We must also be as large as the
 * largest illuminatable room, but no room is larger than 800 grids.  We
 * must also be large enough to allow "good enough" use as a circular queue,
 * to calculate monster flow, but note that the flow code is "paranoid".
 */
#define TEMP_MAX 2298

/*!
 * @brief 再描画処理用配列サイズ / Maximum size of the "redraw" array (see "cave.c")
 * @details We must be large for proper functioning of delayed redrawing.
 * We must also be as large as two times of the largest view area.
 * Note that maximum view grids are 1149 entries.
 */
#define REDRAW_MAX 2298

#define KEYMAP_MODE_ORIG	0 /*!< オリジナルキー配置 / Mode for original keyset commands */
#define KEYMAP_MODE_ROGUE	1 /*!< ローグライクキー配置 / Mode for roguelike keyset commands */
#define KEYMAP_MODES		2 /*!< キー配置の数 / Number of keymap modes */





/*!
 * @brief マクロ登録の最大数 / Maximum number of macros (see "io.c")
 * @note Default: assume at most 256 macros are used
 */
#define MACRO_MAX       256

/*!
 * @brief 銘情報の最大数 / Maximum number of "quarks" (see "io.c")
 * @note 
 * Default: assume at most 512 different inscriptions are used<br>
 * Was 512... 256 quarks added for random artifacts<br>
 */
#define QUARK_MAX       768

/*
 * OPTION: Maximum number of messages to remember (see "io.c")
 * Default: assume maximal memorization of 2048 total messages
 */
#define MESSAGE_MAX  81920

/*
 * OPTION: Maximum space for the message text buffer (see "io.c")
 * Default: assume that each of the 2048 messages is repeated an
 * average of three times, and has an average length of 48
 */
#define MESSAGE_BUF 655360


/*
 * Maximum value storable in a "byte" (hard-coded)
 */
#define MAX_UCHAR       255

/*
 * Maximum value storable in a "s16b" (hard-coded)
 */
#define MAX_SHORT       32767

/*
 * Maximum length of object's name
 */
#define MAX_NLEN        160

/*
 * Special internal key
 */
#define SPECIAL_KEY_QUEST    255
#define SPECIAL_KEY_BUILDING 254
#define SPECIAL_KEY_STORE    253
#define SPECIAL_KEY_QUIT     252

/*
 * Store constants
 */
#define STORE_INVEN_MAX 24              /* Max number of discrete objs in inven */
#define STORE_CHOICES   48              /* Number of items to choose stock from */
#define STORE_OBJ_LEVEL 5               /* Magic Level for normal stores */
#define STORE_TURNOVER  9               /* Normal shop turnover, per day */
#define STORE_MIN_KEEP  6               /* Min slots to "always" keep full */
#define STORE_MAX_KEEP  18              /* Max slots to "always" keep full */
#define STORE_SHUFFLE   21              /* 1/Chance (per day) of an owner changing */
#define STORE_TICKS     1000            /* Number of ticks between turnovers */

/*! @brief 消費する行動エネルギー値を正規乱数で返す(中央100/分散25) / Random energy */
#define ENERGY_NEED() (randnor(100, 25))

/*! @brief 加速値に応じた基本行動エネルギー消費量を返す / Extract energy from speed (Assumes that SPEED is unsigned) */
#define SPEED_TO_ENERGY(SPEED) (((SPEED) > 199) ? 49 : extract_energy[(SPEED)])


/*
 * Misc constants
 */
#define TOWN_DAWN         10000   /*!< 1日分のターン / Number of ticks from dawn to dawn XXX */
#define TURNS_PER_TICK    10L     /*!< 時間経過処理を行うターン数の刻み / Number of energy-gain-turns per ticks */
#define INN_DUNGEON_TURN_ADJ 10   /*!< 宿屋で時間をつぶした場合に増えるdungeon_turnの倍率 */
#define MAX_DAYS          20000   /*!< 内部処理中で保持される最大日数 / Maximum days */
#define BREAK_GLYPH       550     /*!< 守りのルーンの強靭度 / Rune of protection resistance */
#define BREAK_MINOR_GLYPH 299     /*!< 爆発のルーンの発動しやすさ / For explosive runes */
#define BTH_PLUS_ADJ      3       /*!< 武器経験値及びプレイヤーの打撃/射撃能力に応じた修正値倍率 / Adjust BTH per plus-to-hit */
#define MON_MULT_ADJ      8       /*!< モンスターの増殖しにくさの基本倍率 / High value slows multiplication */
#define MON_SUMMON_ADJ    2       /*!< 現在未使用 Adjust level of summoned creatures */
#define MON_DRAIN_LIFE    2       /*!< モンスターの打撃によるプレイヤーの経験値吸収基本倍率(%) / Percent of player exp drained per hit */
#define USE_DEVICE        3       /*!< 魔道具の最低失敗基準値 x> Harder devices x< Easier devices     */


/* "Biases" for random artifact gen */

#define BIAS_ELEC            1 /*!< ランダムアーティファクトバイアス:電撃 */
#define BIAS_POIS            2 /*!< ランダムアーティファクトバイアス:毒 */
#define BIAS_FIRE            3 /*!< ランダムアーティファクトバイアス:火炎 */
#define BIAS_COLD            4 /*!< ランダムアーティファクトバイアス:冷気 */
#define BIAS_ACID            5 /*!< ランダムアーティファクトバイアス:酸 */
#define BIAS_STR             6 /*!< ランダムアーティファクトバイアス:腕力 */
#define BIAS_INT             7 /*!< ランダムアーティファクトバイアス:知力 */
#define BIAS_WIS             8 /*!< ランダムアーティファクトバイアス:賢さ */
#define BIAS_DEX             9 /*!< ランダムアーティファクトバイアス:器用さ */
#define BIAS_CON            10 /*!< ランダムアーティファクトバイアス:耐久 */
#define BIAS_CHR            11 /*!< ランダムアーティファクトバイアス:魅力 */
#define BIAS_CHAOS          12 /*!< ランダムアーティファクトバイアス:混沌 */
#define BIAS_PRIESTLY       13 /*!< ランダムアーティファクトバイアス:プリースト系 */
#define BIAS_NECROMANTIC    14 /*!< ランダムアーティファクトバイアス:死霊 */
#define BIAS_LAW            15 /*!< ランダムアーティファクトバイアス:法 */
#define BIAS_ROGUE          16 /*!< ランダムアーティファクトバイアス:盗賊系 */
#define BIAS_MAGE           17 /*!< ランダムアーティファクトバイアス:メイジ系 */
#define BIAS_WARRIOR        18 /*!< ランダムアーティファクトバイアス:戦士系 */
#define BIAS_RANGER         19 /*!< ランダムアーティファクトバイアス:レンジャー系 */
#define MAX_BIAS            20 /*!< ランダムアーティファクトバイアス:最大数 */

/*** Pet constants ***/


/*
 * ペットに関するコマンド群 / Pet Commands
 */
#define PET_DISMISS				1  /*!< ペットに関するコマンド: ペットを離す */
#define PET_TARGET				2  /*!< ペットに関するコマンド: ペットのターゲットを指定 */
#define PET_STAY_CLOSE			3  /*!< ペットに関するコマンド: 近くにいろ */
#define PET_FOLLOW_ME			4  /*!< ペットに関するコマンド: ついて来い */
#define PET_SEEK_AND_DESTROY	5  /*!< ペットに関するコマンド: 敵を見つけて倒せ */
#define PET_ALLOW_SPACE			6  /*!< ペットに関するコマンド: 少し離れていろ */
#define PET_STAY_AWAY			7  /*!< ペットに関するコマンド: 離れていろ */
#define PET_OPEN_DOORS			8  /*!< ペットに関するコマンド: ドア解放の許可 */
#define PET_TAKE_ITEMS			9  /*!< ペットに関するコマンド: アイテム取得の許可 */
#define PET_TELEPORT			10 /*!< ペットに関するコマンド: テレポートの許可 */
#define PET_ATTACK_SPELL		11 /*!< ペットに関するコマンド: 攻撃魔法の許可 */
#define PET_SUMMON_SPELL		12 /*!< ペットに関するコマンド: 召喚魔法の許可 */
#define PET_BALL_SPELL			13 /*!< ペットに関するコマンド: プレイヤーを魔法に巻き込む許可 */
#define PET_RIDING				14 /*!< ペットに関するコマンド: ペットに乗る */
#define PET_NAME				15 /*!< ペットに関するコマンド: ペットに名前をつける */
#define PET_RYOUTE				16 /*!< ペットに関するコマンド: 騎乗中に両手で武器を使うかどうか */

/*
 * Follow distances
 */
#define PET_CLOSE_DIST				1		/*!<ペットの行動範囲…近くにいろ */
#define PET_FOLLOW_DIST				6		/*!<ペットの行動範囲…ついて来い */
#define PET_SEEK_DIST				10		/*!<ペットの行動範囲…特になし? */
#define PET_DESTROY_DIST			255		/*!<ペットの行動範囲…敵を見つけて倒せ */
#define PET_SPACE_DIST				(-10)	/*!<ペットの行動範囲…少し離れていろ */
#define PET_AWAY_DIST				(-25)	/*!<ペットの行動範囲…離れていろ */

#define PF_OPEN_DOORS   0x0001 /*!< ペットの行動許可…ドアを開けてよい */
#define PF_PICKUP_ITEMS 0x0002 /*!< ペットの行動許可…アイテムを拾ってよい */
#define PF_TELEPORT     0x0004 /*!< ペットの行動許可…テレポート魔法を使ってよい */
#define PF_ATTACK_SPELL 0x0008 /*!< ペットの行動許可…攻撃魔法を使ってよい */
#define PF_SUMMON_SPELL 0x0010 /*!< ペットの行動許可…召喚魔法を使ってよい */
#define PF_BALL_SPELL   0x0020 /*!< ペットの行動許可…ボール魔法でプレイヤーを巻き込んでよい */
#define PF_RYOUTE       0x0040 /*!< プレイヤーの騎乗フラグ…武器を片手で持つ */


#define CAN_TWO_HANDS_WIELDING() (!p_ptr->riding || (p_ptr->pet_extra_flags & PF_RYOUTE))


/*!
 * @brief ベースアイテム生成階層が加算される確率
 * @details
 * There is a 1/10 (10%) chance of inflating the requested object_level
 * during the creation of an object (see "get_obj_num()" in "object.c").
 * Lower values yield better objects more often.
 */
#define GREAT_OBJ       10

/*!
 * @brief 深層モンスターが生成される(NASTY生成)の基本確率(1/x)
 * @details
 * There is a 1/25 (4%) chance of inflating the requested monster_level
 * during the creation of a monsters (see "get_mon_num()" in "monster.c").
 * Lower values yield harder monsters more often.
 */
#define NASTY_MON_BASE     25
#define NASTY_MON_MAX      3	/*!< 深層モンスターが1フロアに生成される最大数  */
#define NASTY_MON_PLUS_MAX 25	/*!< 深層モンスターの階層加算最大量 */

#define PENETRATE_INVULNERABILITY 13 /*!< 無敵化が破られる確率(1/x) / 1/x chance of hurting even if invulnerable! */



/*
 * Refueling constants
 */
#define FUEL_TORCH      5000    /*!< 松明の基本寿命値 / Maximum amount of fuel in a torch */
#define FUEL_LAMP       15000   /*!< ランタンの基本寿命値 / Maximum amount of fuel in a lantern */

/*
 * More maximum values
 */
#define MAX_SIGHT       20      /*!< プレイヤーの最大視界範囲(マス) / Maximum view distance */
#define MAX_RANGE       (p_ptr->inside_battle ? 36 : 18)      /*!< プレイヤーの攻撃射程(マス) / Maximum range (spells, etc) */
#define AAF_LIMIT       100     /*!< モンスターの限界感知範囲(マス) Limit of sensing radius */

#define MIN_M_ALLOC_TD          4 /*!< 街(昼間)の最低住人配置数 / The town starts out with 4 residents during the day */
#define MIN_M_ALLOC_TN          8 /*!< 街(夜間)の最低住人配置数 / The town starts out with 8 residents during the night */

/*!
 * @brief モンスター増殖の最大数
 * @details
 * A monster can only "multiply" (reproduce) if there are fewer than 100
 * monsters on the level capable of such spontaneous reproduction.  This
 * is a hack which prevents the "m_list[]" array from exploding due to
 * reproducing monsters.  Messy, but necessary.
 */
#define MAX_REPRO       100

/*
 * Player constants
 */
#define PY_MAX_EXP      99999999L       /*!< プレイヤー経験値の最大値 / Maximum exp */
#define PY_MAX_GOLD     999999999L      /*!< プレイヤー所持金の最大値 / Maximum gold */
#define PY_MAX_LEVEL    50              /*!< プレイヤーレベルの最大値 / Maximum level */

/*
 * Player "food" crucial values
 */
#define PY_FOOD_MAX     15000   /*!< 食べ過ぎ～満腹の閾値 / Food value (Bloated) */
#define PY_FOOD_FULL    10000   /*!< 満腹～平常の閾値 / Food value (Normal) */
#define PY_FOOD_ALERT   2000    /*!< 平常～空腹の閾値 / Food value (Hungry) */
#define PY_FOOD_WEAK    1000    /*!< 空腹～衰弱の閾値 / Food value (Weak) */
#define PY_FOOD_FAINT   500     /*!< 衰弱～衰弱(赤表示/麻痺)の閾値 / Food value (Fainting) */
#define PY_FOOD_STARVE  100     /*!< 衰弱(赤表示/麻痺)～飢餓ダメージの閾値 / Food value (Starving) */

/*
 * Player regeneration constants
 */
#define PY_REGEN_NORMAL         197     /* Regen factor*2^16 when full */
#define PY_REGEN_WEAK           98      /* Regen factor*2^16 when weak */
#define PY_REGEN_FAINT          33      /* Regen factor*2^16 when fainting */
#define PY_REGEN_HPBASE         1442    /* Min amount hp regen*2^16 */
#define PY_REGEN_MNBASE         524     /* Min amount mana regen*2^16 */

/*
 * 職業ごとの選択可能な魔法領域現在の所 bitrh.cとtables.cでのみ使用。
 * Possible realms that can be chosen currently used only by birth.c and tables.c
 */
#define CH_NONE         0x00
#define CH_LIFE         0x01
#define CH_SORCERY      0x02
#define CH_NATURE       0x04
#define CH_CHAOS        0x08
#define CH_DEATH        0x10
#define CH_TRUMP        0x20
#define CH_ARCANE       0x40
#define CH_ENCHANT      0x80
#define CH_DAEMON       0x100
#define CH_CRUSADE      0x200

#define CH_MUSIC        0x08000	/* This is 16th bit */
#define CH_HISSATSU     0x10000
#define CH_HEX          0x20000


/*
 * Magic realms
 */
#define REALM_NONE         0
#define REALM_LIFE         1
#define REALM_SORCERY      2
#define REALM_NATURE       3
#define REALM_CHAOS        4
#define REALM_DEATH        5
#define REALM_TRUMP        6
#define REALM_ARCANE       7
#define REALM_CRAFT      8
#define REALM_DAEMON       9
#define REALM_CRUSADE      10
#define MAX_MAGIC          10
#define MIN_TECHNIC        16
#define REALM_MUSIC        16
#define REALM_HISSATSU     17
#define REALM_HEX          18
#define MAX_REALM          18

#define VALID_REALM        (MAX_REALM + MAX_MAGIC - MIN_TECHNIC + 1)
#define NUM_TECHNIC        (MAX_REALM - MIN_TECHNIC + 1)

#define is_magic(A) ((((A) > REALM_NONE) && ((A) < MAX_MAGIC + 1)) ? TRUE : FALSE)
#define tval2realm(A) ((A) - TV_LIFE_BOOK + 1)
#define technic2magic(A)      (is_magic(A) ? (A) : (A) - MIN_TECHNIC + 1 + MAX_MAGIC)
#define is_good_realm(REALM)   ((REALM) == REALM_LIFE || (REALM) == REALM_CRUSADE)

/*
 * Magic-books for the realms
 */
#define REALM1_BOOK     (p_ptr->realm1 + TV_LIFE_BOOK - 1)
#define REALM2_BOOK     (p_ptr->realm2 + TV_LIFE_BOOK - 1)


/*
 * Mode constant for do_spell()
 */
#define SPELL_NAME   0
#define SPELL_DESC   1
#define SPELL_INFO   2
#define SPELL_CAST   3
#define SPELL_FAIL   4
#define SPELL_STOP   5
#define SPELL_CONT   6


/*
 * Maximum number of "normal" pack slots, and the index of the "overflow"
 * slot, which can hold an item, but only temporarily, since it causes the
 * pack to "overflow", dropping the "last" item onto the ground.  Since this
 * value is used as an actual slot, it must be less than "INVEN_RARM" (below).
 * Note that "INVEN_PACK" is probably hard-coded by its use in savefiles, and
 * by the fact that the screen can only show 23 items plus a one-line prompt.
 */
#define INVEN_PACK      23 /*!< アイテムスロット…所持品(0～) */

/*
 * Indexes used for various "equipment" slots (hard-coded by savefiles, etc).
 */
#define INVEN_RARM      24 /*!< アイテムスロット…右手 */
#define INVEN_LARM      25 /*!< アイテムスロット…左手 */
#define INVEN_BOW       26 /*!< アイテムスロット…射撃 */
#define INVEN_RIGHT     27 /*!< アイテムスロット…右手指 */
#define INVEN_LEFT      28 /*!< アイテムスロット…左手指 */
#define INVEN_NECK      29 /*!< アイテムスロット…首 */
#define INVEN_LITE      30 /*!< アイテムスロット…光源 */
#define INVEN_BODY      31 /*!< アイテムスロット…体 */
#define INVEN_OUTER     32 /*!< アイテムスロット…体の上 */
#define INVEN_HEAD      33 /*!< アイテムスロット…頭部 */
#define INVEN_HANDS     34 /*!< アイテムスロット…腕部 */
#define INVEN_FEET      35 /*!< アイテムスロット…脚部 */

/*
 * used for get_random_ego()
 */
#define INVEN_AMMO     23

/*
 * Total number of inventory slots (hard-coded).
 */
#define INVEN_TOTAL     36

/*
 * Fake inventory slot for selecting force (hard-coded).
 */
#define INVEN_FORCE     1111


/*
 * Indexes of the various "stats" (hard-coded by savefiles, etc).
 */
#define A_STR   0
#define A_INT   1
#define A_WIS   2
#define A_DEX   3
#define A_CON   4
#define A_CHR   5

/*
 * Player sex constants (hard-coded by save-files, arrays, etc)
 */
#define SEX_FEMALE              0
#define SEX_MALE                1


/*
 * Player race constants (hard-coded by save-files, arrays, etc)
 */
#define RACE_HUMAN               0
#define RACE_HALF_ELF            1
#define RACE_ELF                 2
#define RACE_HOBBIT              3
#define RACE_GNOME               4
#define RACE_DWARF               5
#define RACE_HALF_ORC            6
#define RACE_HALF_TROLL          7
#define RACE_AMBERITE            8
#define RACE_HIGH_ELF            9
#define RACE_BARBARIAN          10
#define RACE_HALF_OGRE          11
#define RACE_HALF_GIANT         12
#define RACE_HALF_TITAN         13
#define RACE_CYCLOPS            14
#define RACE_YEEK               15
#define RACE_KLACKON            16
#define RACE_KOBOLD             17
#define RACE_NIBELUNG           18
#define RACE_DARK_ELF           19
#define RACE_DRACONIAN          20
#define RACE_MIND_FLAYER        21
#define RACE_IMP                22
#define RACE_GOLEM              23
#define RACE_SKELETON           24
#define RACE_ZOMBIE             25
#define RACE_VAMPIRE            26
#define RACE_SPECTRE            27
#define RACE_SPRITE             28
#define RACE_BEASTMAN           29
#define RACE_ENT                30
#define RACE_ANGEL              31
#define RACE_DEMON              32
#define RACE_DUNADAN            33
#define RACE_S_FAIRY            34
#define RACE_KUTAR              35
#define RACE_ANDROID            36

/*
 * Maximum number of player "race" types (see "table.c", etc)
 */
#define MAX_RACES               37


/*
 * Player class constants (hard-coded by save-files, arrays, etc)
 */
#define CLASS_WARRIOR            0
#define CLASS_MAGE               1
#define CLASS_PRIEST             2
#define CLASS_ROGUE              3
#define CLASS_RANGER             4
#define CLASS_PALADIN            5
#define CLASS_WARRIOR_MAGE       6
#define CLASS_CHAOS_WARRIOR      7
#define CLASS_MONK               8
#define CLASS_MINDCRAFTER        9
#define CLASS_HIGH_MAGE         10
#define CLASS_TOURIST           11
#define CLASS_IMITATOR          12
#define CLASS_BEASTMASTER       13
#define CLASS_SORCERER          14
#define CLASS_ARCHER            15
#define CLASS_MAGIC_EATER       16
#define CLASS_BARD              17
#define CLASS_RED_MAGE          18
#define CLASS_SAMURAI           19
#define CLASS_FORCETRAINER      20
#define CLASS_BLUE_MAGE         21
#define CLASS_CAVALRY           22
#define CLASS_BERSERKER         23
#define CLASS_SMITH             24
#define CLASS_MIRROR_MASTER     25
#define CLASS_NINJA             26
#define CLASS_SNIPER            27

#define SEIKAKU_FUTUU	 0
#define SEIKAKU_CHIKARA	 1
#define SEIKAKU_KIREMONO 2
#define SEIKAKU_SHIAWASE 3
#define SEIKAKU_SUBASI	 4
#define SEIKAKU_INOCHI	 5
#define SEIKAKU_COMBAT	 6
#define SEIKAKU_NAMAKE	 7
#define SEIKAKU_SEXY	 8
#define SEIKAKU_LUCKY	 9
#define SEIKAKU_GAMAN	 10
#define SEIKAKU_MUNCHKIN 11

/*** Screen Locations ***/

#define VER_INFO_ROW 3   //!< タイトル表記(行)

/*
 * Some screen locations for various display routines
 * Currently, row 8 and 15 are the only "blank" rows.
 * That leaves a "border" around the "stat" values.
 */

#define ROW_RACE                1
#define COL_RACE                0       /* <race name> */

/*#define ROW_CLASS               2 */
/*#define COL_CLASS               0 */      /* <class name> */

#define ROW_TITLE               2
#define COL_TITLE               0       /* <title> or <mode> */

/*#define ROW_SEIKAKU		4 */
/*#define COL_SEIKAKU      	0*/	/* <seikaku> */

#define ROW_DAY                 21
#define COL_DAY                 0       /* day */

#define ROW_DUNGEON             22
#define COL_DUNGEON             0       /* dungeon */

#define ROW_LEVEL               3
#define COL_LEVEL               0       /* "LEVEL xxxxxx" */

#define ROW_EXP                 4
#define COL_EXP                 0       /* "EXP xxxxxxxx" */

#define ROW_GOLD                5
#define COL_GOLD                0       /* "AU xxxxxxxxx" */

#define ROW_EQUIPPY             6
#define COL_EQUIPPY             0       /* equippy chars */

#define ROW_STAT                7
#define COL_STAT                0       /* "xxx   xxxxxx" */

#define ROW_AC                  13
#define COL_AC                  0       /* "Cur AC xxxxx" */

#define ROW_HPMP                14
#define COL_HPMP                0

#define ROW_CURHP               14
#define COL_CURHP               0       /* "Cur HP xxxxx" */

#define ROW_CURSP               15
#define COL_CURSP               0       /* "Cur SP xxxxx" */

#define ROW_RIDING_INFO          16
#define COL_RIDING_INFO          0       /* "xxxxxxxxxxxx" */

#define ROW_INFO                17
#define COL_INFO                0       /* "xxxxxxxxxxxx" */

#define ROW_MAP						0
#define COL_MAP                  12

#define ROW_CUT                 18
#define COL_CUT                 0       /* <cut> */

#define ROW_STUN                19
#define COL_STUN                0       /* <stun> */

#define ROW_HUNGRY              20
#define COL_HUNGRY              0       /* "Weak" / "Hungry" / "Full" / "Gorged" */

#define ROW_STATE               20
#define COL_STATE                7      /* <state> */

#define ROW_SPEED               (-1)
#define COL_SPEED               (-24)      /* "Slow (-NN)" or "Fast (+NN)" */

#define ROW_STUDY               (-1)
#define COL_STUDY               (-13)      /* "Study" */

#define ROW_DEPTH               (-1)
#define COL_DEPTH               (-8)      /* "Lev NNN" / "NNNN ft" */

#define ROW_STATBAR             (-1)
#define COL_STATBAR              0
#define MAX_COL_STATBAR         (-26)


/*
 * Number of feats we change to (Excluding default). Used in f_info.txt.
 */
#define MAX_FEAT_STATES	 8


/*
 * Wilderness terrains
 */
#define TERRAIN_EDGE             0 /* Edge of the World */
#define TERRAIN_TOWN             1 /* Town */
#define TERRAIN_DEEP_WATER       2 /* Deep water */
#define TERRAIN_SHALLOW_WATER    3 /* Shallow water */
#define TERRAIN_SWAMP            4 /* Swamp */
#define TERRAIN_DIRT             5 /* Dirt */
#define TERRAIN_GRASS            6 /* Grass */
#define TERRAIN_TREES            7 /* Trees */
#define TERRAIN_DESERT           8 /* Desert */
#define TERRAIN_SHALLOW_LAVA     9 /* Shallow lava */
#define TERRAIN_DEEP_LAVA       10 /* Deep lava */
#define TERRAIN_MOUNTAIN        11 /* Mountain */

#define MAX_WILDERNESS          12 /* Maximum wilderness index */


/*
 * Feature flags - should be used instead of feature indexes unless generating.
 * Originally from UnAngband, and modified into TR-like style in Hengband
 */

#define FF_LOS           0              /*!< 視界が通る地形である */
#define FF_PROJECT       1              /*!< 飛び道具が通過できる地形である */
#define FF_MOVE          2              /*!< 移動可能な地形である */
#define FF_PLACE         3              /*!< モンスター配置をしても良い地形である(cave_empty_bold/cave_empty_gridで利用) */
#define FF_DROP          4              /*!< アイテムを落としてよい地形である */
#define FF_SECRET        5              /*!< 隠し扉やトラップが潜んでいる地形である */
#define FF_NOTICE        6              /*!< 何か興味を引くものがある地形である(シフトキー＋方向で走行中の時に止まる基準) */
#define FF_REMEMBER      7              /*!< 常に記憶対象となる地形である(記憶喪失時に忘れたりしなくなる) */
#define FF_OPEN          8              /*!< 開けるコマンドの対象となる地形である */
#define FF_CLOSE         9              /*!< 閉じるコマンドの対象となる地形である */
#define FF_BASH          10             /*!< 体当たりコマンドの対象となる地形である */
#define FF_SPIKE         11             /*!< くさびを打つコマンドの対象となる地形である */
#define FF_DISARM        12             /*!< 解除コマンドの対象となる地形である */
#define FF_STORE         13             /*!< 店舗の入口となる地形である */
#define FF_TUNNEL        14             /*!< 魔王変化などで掘り進められる地形である */
#define FF_MAY_HAVE_GOLD 15             /*!< 何か財宝を隠した可能性のある地形である？(f_infoに使用している地形なし) */
#define FF_HAS_GOLD      16             /*!< 財宝を含んだ地形である */
#define FF_HAS_ITEM      17             /*!< アイテムを含んだ地形である */
#define FF_DOOR          18             /*!< ドアのある地形である */
#define FF_TRAP          19             /*!< トラップのある地形である */
#define FF_STAIRS        20             /*!< 階段のある地形である */
#define FF_GLYPH         21             /*!< 守りのルーンが張られた地形である */
#define FF_LESS          22             /*!< 階上に通じる地形である */
#define FF_MORE          23             /*!< 階下に通じる地形である */
#define FF_AVOID_RUN     24             /*!< 自動移動機能時に障害として迂回すべき地形である */
#define FF_FLOOR         25             /*!< 床のある地形である */
#define FF_WALL          26             /*!< 壁のある地形である */
#define FF_PERMANENT     27             /*!< 絶対に破壊できない永久地形である */
/* #define FF_XXX00         28  未定義 */
/* #define FF_XXX01         29  未定義 */
/* #define FF_XXX02         30  未定義 */
#define FF_HIT_TRAP      31             /*!< トラップのある地形である(TRAPと常に重複している？) */
/* #define FF_BRIDGE        32  未使用 */
/* #define FF_RIVER         33  未使用 */
/* #define FF_LAKE          34  未使用 */
/* #define FF_BRIDGED       35  未使用 */
/* #define FF_COVERED       36  未使用 */
#define FF_GLOW          37             /*!< 常に光っている地形である */
#define FF_ENSECRET      38             /*!< 不明(f_info.txt上で利用している地形がない) */
#define FF_WATER         39             /*!< 水のある地形である */
#define FF_LAVA          40             /*!< 溶岩のある地形である */
#define FF_SHALLOW       41             /*!< 浅い地形である */
#define FF_DEEP          42             /*!< 深い地形である */
/* #define FF_FILLED        43 */       /*!< 未使用 */
#define FF_HURT_ROCK     44             /*!< 岩石溶解の対象となる地形である */
/* #define FF_HURT_FIRE     45 */       /*!< 未使用 */
/* #define FF_HURT_COLD     46 */       /*!< 未使用 */
/* #define FF_HURT_ACID     47 */       /*!< 未使用 */
/* #define FF_ICE           48 */       /*!< 未使用 */
/* #define FF_ACID          49 */       /*!< 未使用 */
/* #define FF_OIL           50 */       /*!< 未使用 */
/* #define FF_XXX04      51 */          /*!< 未使用 */
/* #define FF_CAN_CLIMB     52 */       /*!< 未使用 */
#define FF_CAN_FLY       53             /*!< 飛行可能な地形である */
#define FF_CAN_SWIM      54             /*!< 泳ぐことが可能な地形である */
#define FF_CAN_PASS      55             /*!< 通過可能な地形である */
/* #define FF_CAN_OOZE      56 */       /*!< 未使用 */
#define FF_CAN_DIG       57             /*!< 掘削コマンドの対象となる地形である */
/* #define FF_HIDE_ITEM     58  未使用 */
/* #define FF_HIDE_SNEAK    59  未使用 */
/* #define FF_HIDE_SWIM     60  未使用 */
/* #define FF_HIDE_DIG      61  未使用 */
/* #define FF_KILL_HUGE     62  未使用 */
/* #define FF_KILL_MOVE     63  未使用 */
/* #define FF_PICK_TRAP     64  未使用 */
/* #define FF_PICK_DOOR     65  未使用 */
/* #define FF_ALLOC         66  未使用 */
/* #define FF_CHEST         67  未使用 */
/* #define FF_DROP_1D2      68  未使用 */
/* #define FF_DROP_2D2      69  未使用 */
/* #define FF_DROP_GOOD     70  未使用 */
/* #define FF_DROP_GREAT    71  未使用 */
/* #define FF_HURT_POIS     72  未使用 */
/* #define FF_HURT_ELEC     73  未使用 */
/* #define FF_HURT_WATER    74  未使用 */
/* #define FF_HURT_BWATER   75  未使用 */
/* #define FF_USE_FEAT      76  未使用 */
/* #define FF_GET_FEAT      77  未使用 */
/* #define FF_GROUND        78  未使用 */
/* #define FF_OUTSIDE       79  未使用 */
/* #define FF_EASY_HIDE     80  未使用 */
/* #define FF_EASY_CLIMB    81  未使用 */
/* #define FF_MUST_CLIMB    82  未使用 */
#define FF_TREE          83             /*!< 木の生えた地形である */
/* #define FF_NEED_TREE     84  未使用 */
/* #define FF_BLOOD         85  未使用 */
/* #define FF_DUST          86  未使用 */
/* #define FF_SLIME         87  未使用 */
#define FF_PLANT         88             /*!< 植物の生えた地形である */
/* #define FF_XXX2          89  未定義 */
/* #define FF_INSTANT       90  未使用 */
/* #define FF_EXPLODE       91  未使用 */
/* #define FF_TIMED         92  未使用 */
/* #define FF_ERUPT         93  未使用 */
/* #define FF_STRIKE        94  未使用 */
/* #define FF_SPREAD        95  未使用 */
#define FF_SPECIAL       96             /*!< クエストやダンジョンに関わる特別な地形である */
#define FF_HURT_DISI     97             /*!< 分解属性の対象となる地形である */
#define FF_QUEST_ENTER   98             /*!< クエストの入り口である */
#define FF_QUEST_EXIT    99             /*!< クエストの出口である */
#define FF_QUEST         100            /*!< クエストに関する地形である */
#define FF_SHAFT         101            /*!< 坑道である。(2階層移動する階段である) */
#define FF_MOUNTAIN      102            /*!< ダンジョンの山地形である */
#define FF_BLDG          103            /*!< 施設の入り口である */
#define FF_MINOR_GLYPH   104            /*!< 爆発のルーンのある地形である */
#define FF_PATTERN       105            /*!< パターンのある地形である */
#define FF_TOWN          106            /*!< 広域マップ用の街がある地形である */
#define FF_ENTRANCE      107            /*!< 広域マップ用のダンジョンがある地形である */
#define FF_MIRROR        108            /*!< 鏡使いの鏡が張られた地形である */
#define FF_UNPERM        109            /*!< 破壊不能な地形である(K:フラグ向け？) */
#define FF_TELEPORTABLE  110            /*!< テレポート先の対象となる地形である */
#define FF_CONVERT       111            /*!< 地形生成処理中の疑似フラグ */
#define FF_GLASS         112            /*!< ガラス製の地形である */


#define FF_FLAG_MAX      113
#define FF_FLAG_SIZE     (1 + ((FF_FLAG_MAX - 1) / 32))

/* Which features are dynamic */
#define have_dynamic_flags(ARRAY) \
	(!!((ARRAY)[(FF_INSTANT / 32)] & \
	    ((1UL << (FF_INSTANT % 32)) | \
	     (1UL << (FF_EXPLODE % 32)) | \
	     (1UL << (FF_TIMED % 32)) | \
	     (1UL << (FF_ERUPT % 32)) | \
	     (1UL << (FF_STRIKE % 32)) | \
	     (1UL << (FF_SPREAD % 32)))))


/*
 * Feature action flags
 */
#define FAF_DESTROY     0x01
#define FAF_NO_DROP     0x02
#define FAF_CRASH_GLASS 0x04


/*
 * Bit flags for teleportation
 */
#define TELEPORT_NONMAGICAL 0x00000001
#define TELEPORT_PASSIVE    0x00000002
#define TELEPORT_DEC_VALOUR 0x00000004


/* Types of doors */
#define DOOR_DEFAULT    -1
#define DOOR_DOOR        0
#define DOOR_GLASS_DOOR  1
#define DOOR_CURTAIN     2

#define MAX_DOOR_TYPES   3

#define feat_locked_door_random(DOOR_TYPE) \
	(feat_door[(DOOR_TYPE)].num_locked ? \
	 feat_door[(DOOR_TYPE)].locked[randint0(feat_door[(DOOR_TYPE)].num_locked)] : feat_none)

#define feat_jammed_door_random(DOOR_TYPE) \
	(feat_door[(DOOR_TYPE)].num_jammed ? \
	 feat_door[(DOOR_TYPE)].jammed[randint0(feat_door[(DOOR_TYPE)].num_jammed)] : feat_none)


/* Types of normal traps */
#define NOT_TRAP        -1
#define TRAP_TRAPDOOR    0
#define TRAP_PIT         1
#define TRAP_SPIKED_PIT  2
#define TRAP_POISON_PIT  3
#define TRAP_TY_CURSE    4
#define TRAP_TELEPORT    5
#define TRAP_FIRE        6
#define TRAP_ACID        7
#define TRAP_SLOW        8
#define TRAP_LOSE_STR    9
#define TRAP_LOSE_DEX   10
#define TRAP_LOSE_CON   11
#define TRAP_BLIND      12
#define TRAP_CONFUSE    13
#define TRAP_POISON     14
#define TRAP_SLEEP      15
#define TRAP_TRAPS      16
#define TRAP_ALARM      17


/* Types of special traps */
#define TRAP_OPEN       18
#define TRAP_ARMAGEDDON 19
#define TRAP_PIRANHA    20


/* Maximum locked/jammed doors */
#define MAX_LJ_DOORS 8


/* Types of pattern tiles */
#define NOT_PATTERN_TILE      -1
#define PATTERN_TILE_START    0
#define PATTERN_TILE_1        1
#define PATTERN_TILE_2        2
#define PATTERN_TILE_3        3
#define PATTERN_TILE_4        4
#define PATTERN_TILE_END      5
#define PATTERN_TILE_OLD      6
#define PATTERN_TILE_TELEPORT 7
#define PATTERN_TILE_WRECKED  8


/* Types of conversions */
#define CONVERT_TYPE_FLOOR   0
#define CONVERT_TYPE_WALL    1
#define CONVERT_TYPE_INNER   2
#define CONVERT_TYPE_OUTER   3
#define CONVERT_TYPE_SOLID   4
#define CONVERT_TYPE_STREAM1 5
#define CONVERT_TYPE_STREAM2 6


/*
 * Bit flags for the *_can_enter() and monster_can_cross_terrain()
 */
#define CEM_RIDING              0x0001
#define CEM_P_CAN_ENTER_PATTERN 0x0002


/* Lighting levels of features' attr and char */

#define F_LIT_STANDARD 0 /* Standard */
#define F_LIT_LITE     1 /* Brightly lit */
#define F_LIT_DARK     2 /* Darkened */

#define F_LIT_NS_BEGIN 1 /* Nonstandard */
#define F_LIT_MAX      3


/*** Artifact indexes (see "lib/edit/a_info.txt") ***/

/* Lites */
#define ART_GALADRIEL            1
#define ART_ELENDIL              2
#define ART_JUDGE                3
#define ART_EDISON               7
#define ART_PALANTIR             15
#define ART_STONE_LORE           17
#define ART_FLY_STONE            147
#define ART_ORB_OF_FATE          245 
/* Amulets */
#define ART_CARLAMMAS            4
#define ART_INGWE                5
#define ART_DWARVES              6
#define ART_FARAMIR              18
#define ART_BOROMIR              143
#define ART_MAGATAMA             149
#define ART_INROU                166
#define ART_NIGHT                215
#define ART_SACRED_KNIGHTS       217
#define ART_HELL                 218
#define ART_CHARMED              219
#define ART_GOGO                 220

/* Rings */
#define ART_FRAKIR               8
#define ART_TULKAS               9
#define ART_NARYA               10
#define ART_NENYA               11
#define ART_VILYA               12
#define ART_POWER               13
#define ART_AHO                 14

/* Dragon Scale */
#define ART_RAZORBACK           129
#define ART_BLADETURNER         130
#define ART_SEIRYU              201

/* Hard Armour */
#define ART_SOULKEEPER          19
#define ART_ISILDUR             20
#define ART_ROHIRRIM            21
#define ART_LOHENGRIN           22
#define ART_JULIAN              23
#define ART_ARVEDUI             24
#define ART_CASPANION           25
#define ART_GILES               168
#define ART_MORLOK              203
#define ART_VETERAN             206

/* Soft Armour */
#define ART_SHIVA_JACKET        26
#define ART_HITHLOMIR           27
#define ART_THALKETTOTH         28
#define ART_HIMRING             127
#define ART_ICANUS              131
#define ART_NAMAKE_ARMOR        183
#define ART_GHB                 192
#define ART_DASAI               200
#define ART_KESHO               204
#define ART_MILIM               246

/* Shields */
#define ART_THORIN              30
#define ART_CELEGORM            31
#define ART_ANARION             32
#define ART_GIL_GALAD           138
#define ART_YENDOR              141
#define ART_YATA                151
#define ART_EARENDIL            186
#define ART_PERSEUS             197

/* Helms and Crowns */
#define ART_INDRA               33
#define ART_CHAOS               34
#define ART_BERUTHIEL           35
#define ART_THRANDUIL           36
#define ART_THENGEL             37
#define ART_HAMMERHAND          38
#define ART_DOR                 39
#define ART_HOLHENNETH          40
#define ART_TERROR              41
#define ART_AMBER               42
#define ART_NUMENOR             132
#define ART_STONEMASK           146

/* Cloaks */
#define ART_JACK                43
#define ART_COLLUIN             44
#define ART_HOLCOLLETH          45
#define ART_THINGOL             46
#define ART_THORONGIL           47
#define ART_COLANNON            48
#define ART_LUTHIEN             49
#define ART_TUOR                50
#define ART_MOOK                205
#define ART_HEAVENLY_MAIDEN     233

/* Gloves */
#define ART_CAMBELEG            52
#define ART_CAMMITHRIM          53
#define ART_PAURHACH            54
#define ART_CORWIN              55
#define ART_PAURAEGEN           56
#define ART_PAURNEN             57
#define ART_THANOS              58
#define ART_FINGOLFIN           59
#define ART_PAURNIMMEN          185

/* Boots */
#define ART_FEANOR              60
#define ART_FLORA               61
#define ART_THROR               62
#define ART_SHIVA_BOOTS         63
#define ART_GLASS               165
#define ART_GETA                210

/* Digging */
#define ART_NAIN                211

/* Swords */
#define ART_MAEDHROS            64
#define ART_CAINE               65
#define ART_NARTHANC            66
#define ART_NIMTHANC            67
#define ART_DETHANC             68
#define ART_RILIA               69
#define ART_FIONA               70
#define ART_CALRIS              71
#define ART_GRAYSWANDIR         72
#define ART_GLAMDRING           73
#define ART_NOTHUNG             74
#define ART_ORCRIST             75
#define ART_GURTHANG            76
#define ART_ZARCUTHRA           77
#define ART_TWILIGHT            78
#define ART_GONDRICAM           79
#define ART_CRISDURIAN          80
#define ART_AGLARANG            81
#define ART_RINGIL              82
#define ART_ANDURIL             83
#define ART_WEREWINDLE          84
#define ART_CHAINSWORD          85
#define ART_FORASGIL            86
#define ART_CARETH              87
#define ART_STING               88
#define ART_SOULSWORD           89
#define ART_MERLIN              90
#define ART_DOOMCALLER          91
#define ART_VORPAL_BLADE        92
#define ART_SLAYER              123
#define ART_KUSANAGI            128
#define ART_HURIN               133
#define ART_AZAGHAL             135
#define ART_NOVA                137
#define ART_CHARIOT             140
#define ART_WORPAL_BLADE        142
#define ART_MURAMASA            144
#define ART_ZANTETSU            150
#define ART_SOULCRUSH           154
#define ART_FALIS               155
#define ART_HRUNTING            156
#define ART_ANUBIS              158
#define ART_GURENKI             160
#define ART_TAILBITER           167
#define ART_MUSASI_KATANA       171
#define ART_MUSASI_WAKIZASI     172
#define ART_QUICKTHORN          174
#define ART_TINYTHORN           175
#define ART_EXCALIBUR           176
#define ART_EXCALIPUR           177
#define ART_EXCALIBUR_J         179
#define ART_ARUNRUTH            184
#define ART_HAKABUSA            189
#define ART_STORMBRINGER        190
#define ART_NARSIL              191
#define ART_KANNU               193
#define ART_GRIMTOOTH           196
#define ART_KAMUI               198
#define ART_GOURYU              207
#define ART_EOWYN               216
#define ART_NANACHO             248
#define ART_ROBINTON            251

/* Polearms */
#define ART_THEODEN             93
#define ART_PAIN                94
#define ART_OSONDIR             95
#define ART_TIL                 96
#define ART_RUNESPEAR           97
#define ART_DESTINY             98
#define ART_HAGEN               99
#define ART_EORLINGAS           100
#define ART_DURIN               101
#define ART_EONWE               102
#define ART_BALLI               103
#define ART_LOTHARANG           104
#define ART_DWARVES_AXE         105
#define ART_BARUKKHELED         106
#define ART_WRATH               107
#define ART_ULMO                108
#define ART_AVAVIR              109
#define ART_BENKEI              152
#define ART_TAIKOBO             159
#define ART_TONBO               161
#define ART_GAEBOLG             163
#define ART_ARRYU               164
#define ART_AEGLOS              187
#define ART_BLOOD               199
#define ART_NUMAHOKO            202

/* The sword of the Dawn */
#define ART_DAWN                110

/* Hafted */
#define ART_GROND               111
#define ART_TOTILA              112
#define ART_THUNDERFIST         113
#define ART_BLOODSPIKE          114
#define ART_FIRESTAR            115
#define ART_TARATOL             116
#define ART_AULE                117
#define ART_NAR                 118
#define ART_ERIRIL              119
#define ART_GANDALF             120
#define ART_DEATHWREAKER        121
#define ART_TURMIL              122
#define ART_MJOLLNIR            136
#define ART_WINBLOWS            139
#define ART_XIAOLONG            145
#define ART_NYOIBOU             157
#define ART_JONES               162
#define ART_HYOUSIGI            169
#define ART_MATOI               170
#define ART_IRON_BALL           173
#define ART_SAMSON              178
#define ART_NAMAKE_HAMMER       181
#define ART_BOLISHOI            188
#define ART_SHUTEN_DOJI         194
#define ART_G_HAMMER            195
#define ART_AEGISFANG           208
#define ART_HERMIT              209
#define ART_GOTHMOG             212
#define ART_JIZO                213
#define ART_FUNDIN              214
#define ART_AESCULAPIUS         225

/* Bows */
#define ART_BELTHRONDING        124
#define ART_BARD                125
#define ART_BRAND               126
#define ART_CRIMSON             16
#define ART_BUCKLAND            134
#define ART_YOICHI              148
#define ART_HARAD               180
#define ART_NAMAKE_BOW          182
#define ART_ROBIN_HOOD          221
#define ART_HELLFIRE            222

/* Arrows */
#define ART_BARD_ARROW          153

/*** Ego-Item indexes (see "lib/edit/e_info.txt") ***/


/* Nothing */
/* xxx */
/* xxx */
/* xxx */

/* Body Armor */
#define EGO_A_MORGUL            4
#define EGO_A_DEMON             5
#define EGO_DRUID               6
#define EGO_OLOG                7
#define EGO_RESISTANCE          8
#define EGO_ELVENKIND           9
#define EGO_DWARVEN             10
#define EGO_PERMANENCE          11
#define EGO_YOIYAMI             12
#define EGO_URUKISH             13
/* xxx */
/* xxx */

/* Shields */
#define EGO_ENDURE_ACID         16
#define EGO_ENDURE_ELEC         17
#define EGO_ENDURE_FIRE         18
#define EGO_ENDURE_COLD         19
#define EGO_ENDURANCE           20
#define EGO_REFLECTION          21
#define EGO_NIGHT_DAY           22
#define EGO_S_PROTECTION        238
#define EGO_S_DWARVEN           239

/* Crowns and Helms */
#define EGO_DARK                23
#define EGO_BRILLIANCE          24
#define EGO_H_PROTECTION        25
#define EGO_H_DEMON             26
#define EGO_MAGI                27
#define EGO_MIGHT               28
#define EGO_LORDLINESS          29
#define EGO_SEEING              30
#define EGO_INFRAVISION         31
#define EGO_LITE                32
#define EGO_TELEPATHY           33
#define EGO_REGENERATION        34
#define EGO_TELEPORTATION       35
#define EGO_FOOL                36
#define EGO_BASILISK            37
#define EGO_ANCIENT_CURSE       38
#define EGO_SICKLINESS          39

/* Cloaks */
#define EGO_PROTECTION          40
#define EGO_STEALTH             41
#define EGO_AMAN                42
#define EGO_AURA_FIRE           43
#define EGO_ENVELOPING          44
#define EGO_VULNERABILITY       45
#define EGO_IRRITATION          46
#define EGO_AURA_ELEC           47
#define EGO_AURA_COLD          128
#define EGO_BAT                129
#define EGO_NAZGUL             240

/* Gloves */
#define EGO_FREE_ACTION         48
#define EGO_SLAYING             49
#define EGO_AGILITY             50
#define EGO_POWER               51
#define EGO_2WEAPON             52
#define EGO_MAGIC_MASTERY       53
#define EGO_WEAKNESS            54
#define EGO_CLUMSINESS          55

/* Boots */
#define EGO_SLOW_DESCENT        56
#define EGO_QUIET               57
#define EGO_MOTION              58
#define EGO_SPEED               59
#define EGO_JUMP                60
#define EGO_NOISE               61
#define EGO_SLOWNESS            62
#define EGO_ANNOYANCE           63

/* Weapons */
#define EGO_HA                  64
#define EGO_DF                  65
#define EGO_BLESS_BLADE         66
/* xxx */
#define EGO_WEST                68
#define EGO_ATTACKS             69
#define EGO_SLAYING_WEAPON      70
#define EGO_FORCE_WEAPON        71
#define EGO_BRAND_ACID          72
#define EGO_BRAND_ELEC          73
#define EGO_BRAND_FIRE          74
#define EGO_BRAND_COLD          75
#define EGO_BRAND_POIS          76
#define EGO_CHAOTIC             77
#define EGO_SHARPNESS           78
#define EGO_EARTHQUAKES         79
#define EGO_XXX_1               80
#define EGO_XXX_2               81
#define EGO_XXX_3               82
#define EGO_XXX_4               83
#define EGO_XXX_5               84
#define EGO_XXX_6               85
#define EGO_XXX_7               86
#define EGO_WEIRD               87
#define EGO_KILL_ANIMAL         88
#define EGO_KILL_EVIL           89
#define EGO_KILL_UNDEAD         90
#define EGO_KILL_DEMON          91
#define EGO_KILL_ORC            92
#define EGO_KILL_TROLL          93
#define EGO_KILL_GIANT          94
#define EGO_KILL_DRAGON         95
#define EGO_VAMPIRIC            96
#define EGO_PRISM               97
#define EGO_TRUMP               98
#define EGO_PATTERN             99
#define EGO_DIGGING             100
#define EGO_DEMON               101
#define EGO_MORGUL              102
#define EGO_KILL_HUMAN          103

/* Bows */
#define EGO_ACCURACY            104
#define EGO_VELOCITY            105
/* xxx */
/* xxx */
#define EGO_EXTRA_MIGHT         108
#define EGO_EXTRA_SHOTS         109
/* xxx */
/* xxx */

/* Ammo */
#define EGO_HURT_ANIMAL         112
#define EGO_HURT_EVIL           113
/* xxx */
/* xxx */
/* xxx */
/* xxx */
/* xxx */
#define EGO_HURT_DRAGON         119
#define EGO_SLAYING_BOLT        120
#define EGO_LIGHTNING_BOLT      121
#define EGO_FLAME               122
#define EGO_FROST               123
#define EGO_WOUNDING            124
#define EGO_BACKBITING          125
#define EGO_SHATTERED           126
#define EGO_BLASTED             127

#define EGO_LITE_SHINE          140
#define EGO_LITE_ILLUMINATION   141
#define EGO_LITE_AURA_FIRE      142
#define EGO_LITE_INFRA          143
#define EGO_LITE_LONG           144
#define EGO_LITE_DARKNESS       145
#define EGO_LITE_EYE            146

#define EGO_RING_HERO           150
#define EGO_RING_SLAY           151
#define EGO_RING_SUPER_AC       152
#define EGO_RING_MAGIC_MIS      153
#define EGO_RING_FIRE_BOLT      154
#define EGO_RING_COLD_BOLT      155
#define EGO_RING_ELEC_BOLT      156
#define EGO_RING_ACID_BOLT      157
#define EGO_RING_MANA_BOLT      158
#define EGO_RING_FIRE_BALL      159
#define EGO_RING_COLD_BALL      160
#define EGO_RING_ELEC_BALL      161
#define EGO_RING_ACID_BALL      162
#define EGO_RING_MANA_BALL      163
#define EGO_RING_DRAGON_F       164
#define EGO_RING_DRAGON_C       165
#define EGO_RING_D_SPEED        166
#define EGO_RING_BERSERKER      167
#define EGO_RING_HUNTER         168
#define EGO_RING_THROW          169
#define EGO_RING_REGEN          170
#define EGO_RING_LITE           171
#define EGO_RING_M_DETECT       172
#define EGO_RING_STEALTH        173
#define EGO_RING_TELE_AWAY      174
#define EGO_RING_TO_H           175
#define EGO_RING_TO_D           176
#define EGO_RING_RES_LITE       177
#define EGO_RING_RES_DARK       178
#define EGO_RING_WIZARD         179
#define EGO_RING_TRUE           180
#define EGO_RING_DRAIN_EXP      181
#define EGO_RING_NO_MELEE       182
#define EGO_RING_AGGRAVATE      183
#define EGO_RING_TY_CURSE       184
#define EGO_RING_RES_TIME       185
#define EGO_RING_TELEPORT       186
#define EGO_RING_ALBINO         187

#define EGO_AMU_SLOW_D          210
#define EGO_AMU_INFRA           211
#define EGO_AMU_SEE_INVIS       212
#define EGO_AMU_HOLD_EXP        213
#define EGO_AMU_DRAIN_EXP       214
#define EGO_AMU_FOOL            215
#define EGO_AMU_AGGRAVATE       216
#define EGO_AMU_TY_CURSE        217
#define EGO_AMU_AC              218
#define EGO_AMU_IDENT           219
#define EGO_AMU_CHARM           220
#define EGO_AMU_STEALTH         221
#define EGO_AMU_JUMP            222
#define EGO_AMU_TELEPORT        223
#define EGO_AMU_D_DOOR          224
#define EGO_AMU_DEFENDER        225
#define EGO_AMU_RES_FIRE        226
#define EGO_AMU_RES_FIRE_       227
#define EGO_AMU_RES_COLD        228
#define EGO_AMU_RES_COLD_       229
#define EGO_AMU_RES_ELEC        230
#define EGO_AMU_RES_ELEC_       231
#define EGO_AMU_RES_ACID        232
#define EGO_AMU_RES_ACID_       233
#define EGO_AMU_LEVITATION      234
#define EGO_AMU_GREAT           235
#define EGO_AMU_DETECTION       236
#define EGO_AMU_NAIVETY         237
// MAX 240

/* Activation effects for random artifacts */
#define ACT_SUNLIGHT            1
#define ACT_BO_MISS_1           2
#define ACT_BA_POIS_1           3
#define ACT_BO_ELEC_1           4
#define ACT_BO_ACID_1           5
#define ACT_BO_COLD_1           6
#define ACT_BO_FIRE_1           7
#define ACT_BA_COLD_1           8
#define ACT_BA_FIRE_1           9
#define ACT_HYPODYNAMIA_1       10
#define ACT_BA_COLD_2           11
#define ACT_BA_ELEC_2           12
#define ACT_HYPODYNAMIA_2       13
#define ACT_DRAIN_1             14
#define ACT_BO_MISS_2           15
#define ACT_BA_FIRE_3           16
#define ACT_BA_COLD_3           17
#define ACT_BA_ELEC_3           18
#define ACT_WHIRLWIND           19
#define ACT_DRAIN_2             20
#define ACT_CALL_CHAOS          21
#define ACT_ROCKET              22
#define ACT_DISP_EVIL           23
#define ACT_BA_MISS_3           24
#define ACT_DISP_GOOD           25
#define ACT_BO_MANA             26
#define ACT_BA_FIRE_2           27
#define ACT_BA_WATER            28
#define ACT_BA_STAR             29
#define ACT_BA_DARK             30
#define ACT_BA_MANA             31
#define ACT_PESTICIDE           32
#define ACT_BLINDING_LIGHT      33
#define ACT_BIZARRE             34
#define ACT_CAST_BA_STAR        35
#define ACT_BLADETURNER         36
#define ACT_BA_ACID_1           37
#define ACT_BR_FIRE             38
#define ACT_BR_COLD             39
#define ACT_BR_DRAGON           40
#define ACT_BA_FIRE_4           41
#define ACT_BA_NUKE_1           42
/* 42 - 50 unused */
#define ACT_CONFUSE             51
#define ACT_SLEEP               52
#define ACT_QUAKE               53
#define ACT_TERROR              54
#define ACT_TELE_AWAY           55
#define ACT_BANISH_EVIL         56
#define ACT_GENOCIDE            57
#define ACT_MASS_GENO           58
#define ACT_SCARE_AREA          59
#define ACT_AGGRAVATE           60
/* 59 - 64 unused */
#define ACT_CHARM_ANIMAL        65
#define ACT_CHARM_UNDEAD        66
#define ACT_CHARM_OTHER         67
#define ACT_CHARM_ANIMALS       68
#define ACT_CHARM_OTHERS        69
#define ACT_SUMMON_ANIMAL       70
#define ACT_SUMMON_PHANTOM      71
#define ACT_SUMMON_ELEMENTAL    72
#define ACT_SUMMON_DEMON        73
#define ACT_SUMMON_UNDEAD       74
#define ACT_SUMMON_HOUND        75
#define ACT_SUMMON_DAWN         76
#define ACT_SUMMON_OCTOPUS      77
/* 76 - 80 unused */
#define ACT_CHOIR_SINGS         80
#define ACT_CURE_LW             81
#define ACT_CURE_MW             82
#define ACT_CURE_POISON         83
#define ACT_REST_EXP            84
#define ACT_REST_ALL            85
#define ACT_CURE_700            86
#define ACT_CURE_1000           87
#define ACT_CURING              88
#define ACT_CURE_MANA_FULL      89
/* 90 unused */
#define ACT_ESP                 91
#define ACT_BERSERK             92
#define ACT_PROT_EVIL           93
#define ACT_RESIST_ALL          94
#define ACT_SPEED               95
#define ACT_XTRA_SPEED          96
#define ACT_WRAITH              97
#define ACT_INVULN              98
#define ACT_HELO                99
#define ACT_HELO_SPEED          100
#define ACT_RESIST_ACID         101
#define ACT_RESIST_FIRE         102
#define ACT_RESIST_COLD         103
#define ACT_RESIST_ELEC         104
#define ACT_RESIST_POIS         105
/* 106 - 110 unused */
#define ACT_LIGHT               111
#define ACT_MAP_LIGHT           112
#define ACT_DETECT_ALL          113
#define ACT_DETECT_XTRA         114
#define ACT_ID_FULL             115
#define ACT_ID_PLAIN            116
#define ACT_RUNE_EXPLO          117
#define ACT_RUNE_PROT           118
#define ACT_SATIATE             119
#define ACT_DEST_DOOR           120
#define ACT_STONE_MUD           121
#define ACT_RECHARGE            122
#define ACT_ALCHEMY             123
#define ACT_DIM_DOOR            124
#define ACT_TELEPORT            125
#define ACT_RECALL              126
#define ACT_JUDGE               127
#define ACT_TELEKINESIS         128
#define ACT_DETECT_UNIQUE       129
#define ACT_ESCAPE              130
#define ACT_DISP_CURSE_XTRA     131
#define ACT_BRAND_FIRE_BOLTS    132
#define ACT_RECHARGE_XTRA       133
#define ACT_LORE                134
#define ACT_SHIKOFUMI           135
#define ACT_PHASE_DOOR          136
#define ACT_DETECT_ALL_MONS     137
#define ACT_ULTIMATE_RESIST     138
/* 127 -> unused */
#define ACT_FALLING_STAR        246
#define ACT_STRAIN_HASTE        247
#define ACT_TELEPORT_LEVEL      248
#define ACT_GRAND_CROSS         249
#define ACT_CAST_OFF            250
#define ACT_FISHING             251
#define ACT_INROU               252
#define ACT_MURAMASA            253
#define ACT_BLOODY_MOON         254
#define ACT_CRIMSON             255

/*** Object "tval" and "sval" codes ***/


/*
 * The values for the "tval" field of various objects.
 *
 * This value is the primary means by which items are sorted in the
 * player inventory, followed by "sval" and "cost".
 *
 * Note that a "BOW" with tval = 19 and sval S = 10*N+P takes a missile
 * weapon with tval = 16+N, and does (xP) damage when so combined.  This
 * fact is not actually used in the source, but it kind of interesting.
 *
 * Note that as of 2.7.8, the "item flags" apply to all items, though
 * only armor and weapons and a few other items use any of these flags.
 */

#define TV_SKELETON      1      /* Skeletons ('s'), not specified */
#define TV_BOTTLE        2      /* Empty bottles ('!') */
#define TV_JUNK          3      /* Sticks, Pottery, etc ('~') */
#define TV_WHISTLE       4      /* Whistle ('~') */
#define TV_SPIKE         5      /* Spikes ('~') */
#define TV_CHEST         7      /* Chests ('&') */
#define TV_FIGURINE      8      /* Magical figurines */
#define TV_STATUE        9      /* Statue, what a silly object... */
#define TV_CORPSE       10      /* Corpses and Skeletons, specific */
#define TV_CAPTURE      11      /* Monster ball */
#define TV_NO_AMMO      15      /* Ammo for crimson */
#define TV_SHOT         16      /* Ammo for slings */
#define TV_ARROW        17      /* Ammo for bows */
#define TV_BOLT         18      /* Ammo for x-bows */
#define TV_BOW          19      /* Slings/Bows/Xbows */
#define TV_DIGGING      20      /* Shovels/Picks */
#define TV_HAFTED       21      /* Priest Weapons */
#define TV_POLEARM      22      /* Axes and Pikes */
#define TV_SWORD        23      /* Edged Weapons */
#define TV_BOOTS        30      /* Boots */
#define TV_GLOVES       31      /* Gloves */
#define TV_HELM         32      /* Helms */
#define TV_CROWN        33      /* Crowns */
#define TV_SHIELD       34      /* Shields */
#define TV_CLOAK        35      /* Cloaks */
#define TV_SOFT_ARMOR   36      /* Soft Armor */
#define TV_HARD_ARMOR   37      /* Hard Armor */
#define TV_DRAG_ARMOR   38      /* Dragon Scale Mail */
#define TV_LITE         39      /* Lites (including Specials) */
#define TV_AMULET       40      /* Amulets (including Specials) */
#define TV_RING         45      /* Rings (including Specials) */
#define TV_CARD         50
#define TV_STAFF        55
#define TV_WAND         65
#define TV_ROD          66
#define TV_PARCHMENT    69
#define TV_SCROLL       70
#define TV_POTION       75
#define TV_FLASK        77
#define TV_FOOD         80
#define TV_LIFE_BOOK    90
#define TV_SORCERY_BOOK 91
#define TV_NATURE_BOOK  92
#define TV_CHAOS_BOOK   93
#define TV_DEATH_BOOK   94
#define TV_TRUMP_BOOK   95
#define TV_ARCANE_BOOK  96
#define TV_CRAFT_BOOK 97
#define TV_DAEMON_BOOK  98
#define TV_CRUSADE_BOOK 99
#define TV_MUSIC_BOOK   105
#define TV_HISSATSU_BOOK 106
#define TV_HEX_BOOK     107
#define TV_GOLD         127     /* Gold can only be picked up by players */

#define TV_EQUIP_BEGIN    TV_SHOT
#define TV_EQUIP_END      TV_CARD
#define TV_MISSILE_BEGIN  TV_SHOT
#define TV_MISSILE_END    TV_BOLT
#define TV_WEARABLE_BEGIN TV_BOW
#define TV_WEARABLE_END   TV_CARD
#define TV_WEAPON_BEGIN   TV_BOW
#define TV_WEAPON_END     TV_SWORD
#define TV_ARMOR_BEGIN    TV_BOOTS
#define TV_ARMOR_END      TV_DRAG_ARMOR

/* Any subvalue */
#define SV_ANY 					255

/* The "sval" codes for TV_FIGURINE */
#define SV_FIGURINE_NORMAL		0

#define SV_CAPTURE_NONE		0

/* The "sval" codes for TV_STATUE */
#define SV_WOODEN_STATUE		0
#define SV_CLAY_STATUE			1
#define SV_STONE_STATUE			2
#define SV_IRON_STATUE			3
#define SV_COPPER_STATUE		4
#define SV_SILVER_STATUE		5
#define SV_GOLDEN_STATUE		6
#define SV_IVORY_STATUE			7
#define SV_MITHRIL_STATUE		8
#define SV_ORNATE_STATUE		9
#define SV_PHOTO			50

/* The "sval" codes for TV_CORPSE */
#define SV_SKELETON 			0
#define SV_CORPSE			1

/* The "sval" codes for TV_SHOT/TV_ARROW/TV_BOLT */
#define SV_AMMO_LIGHT                    0	/* pebbles */
#define SV_AMMO_NORMAL                   1	/* shots, arrows, bolts */
#define SV_AMMO_HEAVY                    2	/* seeker arrows and bolts, mithril shots */

/* The "sval" codes for TV_BOW (note information in "sval") */
#define SV_SLING                         2	/* (x2) */
#define SV_SHORT_BOW                    12	/* (x2) */
#define SV_LONG_BOW                     13	/* (x3) */
#define SV_LIGHT_XBOW                   23	/* (x3) */
#define SV_HEAVY_XBOW                   24	/* (x4) */
#define SV_CRIMSON                      50	/* (x0) */
#define SV_HARP                         51	/* (x0) */
#define SV_NAMAKE_BOW                   63	/* (x3) */

/* The "sval" codes for TV_DIGGING */
#define SV_SHOVEL                        1
#define SV_GNOMISH_SHOVEL                2
#define SV_DWARVEN_SHOVEL                3
#define SV_PICK                          4
#define SV_ORCISH_PICK                   5
#define SV_DWARVEN_PICK                  6
#define SV_MATTOCK                       7

/* The "sval" values for TV_HAFTED */
#define SV_CLUB                          1	/* 1d4  */
#define SV_WHIP                          2	/* 1d6  */
#define SV_QUARTERSTAFF                  3	/* 1d9  */
#define SV_NUNCHAKU                      4	/* 2d3  */
#define SV_MACE                          5	/* 2d4  */
#define SV_BALL_AND_CHAIN                6	/* 2d4  */
#define SV_JO_STAFF                      7	/* 1d7  */
#define SV_WAR_HAMMER                    8	/* 3d3  */
#define SV_THREE_PIECE_ROD              11	/* 3d3  */
#define SV_MORNING_STAR                 12	/* 2d6  */
#define SV_FLAIL                        13	/* 2d6  */
#define SV_BO_STAFF                     14	/* 1d11 */
#define SV_LEAD_FILLED_MACE             15	/* 3d4  */
#define SV_TETSUBO                      16	/* 2d7  */
#define SV_TWO_HANDED_FLAIL             18	/* 3d6  */
#define SV_GREAT_HAMMER                 19	/* 4d6  */
#define SV_MACE_OF_DISRUPTION           20	/* 5d8  */
#define SV_WIZSTAFF                     21	/* 1d2  */
#define SV_GROND                        50	/* 3d9  */
#define SV_NAMAKE_HAMMER                63	/* 1d77 */

/* The "sval" values for TV_POLEARM */
#define SV_HATCHET                       1	/* 1d5 */
#define SV_SPEAR                         2	/* 1d6 */
#define SV_SICKLE                        3	/* 2d3 */
#define SV_AWL_PIKE                      4	/* 1d8 */
#define SV_TRIDENT                       5	/* 1d9 */
#define SV_FAUCHARD                      6  /* 1d10 */
#define SV_BROAD_SPEAR                   7	/* 1d9 */
#define SV_PIKE                          8	/* 2d5 */
#define SV_NAGINATA                      9  /* 2d6 */
#define SV_BEAKED_AXE                   10	/* 2d6 */
#define SV_BROAD_AXE                    11	/* 2d6 */
#define SV_LUCERNE_HAMMER               12	/* 2d5  */
#define SV_GLAIVE                       13	/* 2d6 */
#define SV_LAJATANG                     14	/* 2d7 */
#define SV_HALBERD                      15	/* 3d4 */
#define SV_GUISARME                     16  /* 2d5 */
#define SV_SCYTHE                       17	/* 5d3 */
#define SV_LANCE                        20	/* 2d8 */
#define SV_BATTLE_AXE                   22	/* 2d8 */
#define SV_GREAT_AXE                    25	/* 4d4 */
#define SV_TRIFURCATE_SPEAR             26	/* 2d9 */
#define SV_LOCHABER_AXE                 28	/* 3d8 */
#define SV_HEAVY_LANCE                  29  /* 4d8 */
#define SV_SCYTHE_OF_SLICING            30	/* 8d4 */
#define SV_TSURIZAO                     40	/* 1d1 */
#define SV_DEATH_SCYTHE                 50	/* 10d10 */

/* The "sval" codes for TV_SWORD */
#define SV_BROKEN_DAGGER                 1  /* 1d1 */
#define SV_BROKEN_SWORD                  2  /* 1d2 */
#define SV_DAGGER                        4  /* 1d4 */
#define SV_MAIN_GAUCHE                   5  /* 1d5 */
#define SV_TANTO                         6  /* 1d5 */
#define SV_RAPIER                        7  /* 1d6 */
#define SV_SMALL_SWORD                   8  /* 1d6 */
#define SV_BASILLARD                     9  /* 1d8 */
#define SV_SHORT_SWORD                  10  /* 1d7 */
#define SV_SABRE                        11  /* 1d7 */
#define SV_CUTLASS                      12  /* 1d7 */
#define SV_WAKIZASHI                    13  /* 2d4 */
#define SV_KHOPESH                      14  /* 2d4 */
#define SV_TULWAR                       15  /* 2d4 */
#define SV_BROAD_SWORD                  16  /* 2d5 */
#define SV_LONG_SWORD                   17  /* 2d5 */
#define SV_SCIMITAR                     18  /* 2d5 */
#define SV_NINJATO                      19  /* 1d9 */
#define SV_KATANA                       20  /* 3d4 */
#define SV_BASTARD_SWORD                21  /* 3d4 */
#define SV_GREAT_SCIMITAR               22  /* 4d5 */
#define SV_CLAYMORE                     23  /* 2d8 */
#define SV_ESPADON                      24  /* 2d9 */
#define SV_TWO_HANDED_SWORD             25  /* 3d6 */
#define SV_FLAMBERGE                    26  /* 3d7 */
#define SV_NO_DACHI                     27  /* 5d4 */
#define SV_EXECUTIONERS_SWORD           28  /* 4d5 */
#define SV_ZWEIHANDER                   29  /* 4d6 */
#define SV_BLADE_OF_CHAOS               30  /* 6d5 */
#define SV_DIAMOND_EDGE                 31  /* 7d5 */
#define SV_DOKUBARI                     32  /* 1d1 */
#define SV_HAYABUSA                     33  /* 1d6 */

/* The "sval" codes for TV_SHIELD */
#define SV_SMALL_LEATHER_SHIELD          2
#define SV_SMALL_METAL_SHIELD            3
#define SV_LARGE_LEATHER_SHIELD          4
#define SV_LARGE_METAL_SHIELD            5
#define SV_DRAGON_SHIELD                 6
#define SV_KNIGHT_SHIELD                 7
#define SV_MIRROR_SHIELD                10
#define SV_YATA_MIRROR                  50

/* The "sval" codes for TV_HELM */
#define SV_HARD_LEATHER_CAP              2
#define SV_METAL_CAP                     3
#define SV_JINGASA                       4  /* 4 */
#define SV_IRON_HELM                     5
#define SV_STEEL_HELM                    6
#define SV_DRAGON_HELM                   7
#define SV_KABUTO                        8  /* 7 */

/* The "sval" codes for TV_CROWN */
#define SV_IRON_CROWN                   10
#define SV_GOLDEN_CROWN                 11
#define SV_JEWELED_CROWN                12
#define SV_CHAOS                        50

/* The "sval" codes for TV_BOOTS */
#define SV_PAIR_OF_SOFT_LEATHER_BOOTS    2
#define SV_PAIR_OF_HARD_LEATHER_BOOTS    3
#define SV_PAIR_OF_DRAGON_GREAVE         4
#define SV_PAIR_OF_METAL_SHOD_BOOTS      6

/* The "sval" codes for TV_CLOAK */
#define SV_CLOAK                         1
#define SV_ELVEN_CLOAK                   2
#define SV_FUR_CLOAK                     3
#define SV_ETHEREAL_CLOAK                5
#define SV_SHADOW_CLOAK                  6

/* The "sval" codes for TV_GLOVES */
#define SV_SET_OF_LEATHER_GLOVES         1
#define SV_SET_OF_GAUNTLETS              2
#define SV_SET_OF_DRAGON_GLOVES          3
#define SV_SET_OF_CESTI                  5

/* The "sval" codes for TV_SOFT_ARMOR */
#define SV_T_SHIRT                       0
#define SV_FILTHY_RAG                    1
#define SV_ROBE                          2
#define SV_PAPER_ARMOR                   3  /* 4 */
#define SV_SOFT_LEATHER_ARMOR            4
#define SV_SOFT_STUDDED_LEATHER          5
#define SV_HARD_LEATHER_ARMOR            6
#define SV_HARD_STUDDED_LEATHER          7
#define SV_RHINO_HIDE_ARMOR              8
#define SV_CORD_ARMOR                    9  /*  6 */
#define SV_PADDED_ARMOR                 10  /*  4 */
#define SV_LEATHER_SCALE_MAIL           11
#define SV_LEATHER_JACK                 12
#define SV_KUROSHOUZOKU                 13
#define SV_STONE_AND_HIDE_ARMOR         15  /* 15 */
#define SV_ABUNAI_MIZUGI                50
#define SV_YOIYAMI_ROBE                 60
#define SV_NAMAKE_ARMOR                 63

/* The "sval" codes for TV_HARD_ARMOR */
#define SV_RUSTY_CHAIN_MAIL              1  /* 14- */
#define SV_RING_MAIL                     2  /* 12  */
#define SV_METAL_SCALE_MAIL              3  /* 13  */
#define SV_CHAIN_MAIL                    4  /* 14  */
#define SV_DOUBLE_RING_MAIL              5  /* 15  */
#define SV_AUGMENTED_CHAIN_MAIL          6  /* 16  */
#define SV_DOUBLE_CHAIN_MAIL             7  /* 16  */
#define SV_BAR_CHAIN_MAIL                8  /* 18  */
#define SV_METAL_BRIGANDINE_ARMOUR       9  /* 19  */
#define SV_SPLINT_MAIL                  10  /* 19  */
#define SV_DO_MARU                      11  /* 20  */
#define SV_PARTIAL_PLATE_ARMOUR         12  /* 22  */
#define SV_METAL_LAMELLAR_ARMOUR        13  /* 23  */
#define SV_HARAMAKIDO                   14  /* 17  */
#define SV_FULL_PLATE_ARMOUR            15  /* 25  */
#define SV_O_YOROI                      16  /* 24  */
#define SV_RIBBED_PLATE_ARMOUR          18  /* 28  */
#define SV_MITHRIL_CHAIN_MAIL           20  /* 28+ */
#define SV_MITHRIL_PLATE_MAIL           25  /* 35+ */
#define SV_ADAMANTITE_PLATE_MAIL        30  /* 40+ */

/* The "sval" codes for TV_DRAG_ARMOR */
#define SV_DRAGON_BLACK                  1
#define SV_DRAGON_BLUE                   2
#define SV_DRAGON_WHITE                  3
#define SV_DRAGON_RED                    4
#define SV_DRAGON_GREEN                  5
#define SV_DRAGON_MULTIHUED              6
#define SV_DRAGON_SHINING               10
#define SV_DRAGON_LAW                   12
#define SV_DRAGON_BRONZE                14
#define SV_DRAGON_GOLD                  16
#define SV_DRAGON_CHAOS                 18
#define SV_DRAGON_BALANCE               20
#define SV_DRAGON_POWER                 30

/* The sval codes for TV_LITE */
#define SV_LITE_TORCH                    0
#define SV_LITE_LANTERN                  1
#define SV_LITE_FEANOR                   2
#define SV_LITE_EDISON                   3
#define SV_LITE_GALADRIEL                4
#define SV_LITE_ELENDIL                  5
#define SV_LITE_JUDGE                    6
#define SV_LITE_LORE                     7
#define SV_LITE_PALANTIR                 8
#define SV_LITE_FLY_STONE                9

/* The "sval" codes for TV_AMULET */
#define SV_AMULET_DOOM                   0
#define SV_AMULET_TELEPORT               1
#define SV_AMULET_ADORNMENT              2
#define SV_AMULET_SLOW_DIGEST            3
#define SV_AMULET_RESIST_ACID            4
#define SV_AMULET_SEARCHING              5
#define SV_AMULET_BRILLIANCE             6
#define SV_AMULET_CHARISMA               7
#define SV_AMULET_THE_MAGI               8
#define SV_AMULET_REFLECTION             9
#define SV_AMULET_CARLAMMAS             10
#define SV_AMULET_INGWE                 11
#define SV_AMULET_DWARVES               12
#define SV_AMULET_NO_MAGIC              13
#define SV_AMULET_NO_TELE               14
#define SV_AMULET_RESISTANCE            15
#define SV_AMULET_TELEPATHY             16
#define SV_AMULET_FARAMIR               17
#define SV_AMULET_BOROMIR               18
#define SV_AMULET_MAGATAMA              19
#define SV_AMULET_INROU                 20
#define SV_AMULET_INTELLIGENCE          21
#define SV_AMULET_WISDOM                22
#define SV_AMULET_MAGIC_MASTERY         23
#define SV_AMULET_NIGHT                 24

/* The sval codes for TV_RING */
#define SV_RING_WOE                      0
#define SV_RING_AGGRAVATION              1
#define SV_RING_WEAKNESS                 2
#define SV_RING_STUPIDITY                3
#define SV_RING_TELEPORTATION            4
#define SV_RING_SLOW_DIGESTION           6
#define SV_RING_LEVITATION_FALL             7
#define SV_RING_RESIST_FIRE              8
#define SV_RING_RESIST_COLD              9
#define SV_RING_SUSTAIN_STR             10
#define SV_RING_SUSTAIN_INT             11
#define SV_RING_SUSTAIN_WIS             12
#define SV_RING_SUSTAIN_CON             13
#define SV_RING_SUSTAIN_DEX             14
#define SV_RING_SUSTAIN_CHR             15
#define SV_RING_PROTECTION              16
#define SV_RING_ACID                    17
#define SV_RING_FLAMES                  18
#define SV_RING_ICE                     19
#define SV_RING_RESIST_POIS             20
#define SV_RING_FREE_ACTION             21
#define SV_RING_SEE_INVIS               22
#define SV_RING_SEARCHING               23
#define SV_RING_STR                     24
#define SV_RING_ELEC                    25
#define SV_RING_DEX                     26
#define SV_RING_CON                     27
#define SV_RING_ACCURACY                28
#define SV_RING_DAMAGE                  29
#define SV_RING_SLAYING                 30
#define SV_RING_SPEED                   31
#define SV_RING_FRAKIR                  32
#define SV_RING_TULKAS                  33
#define SV_RING_NARYA                   34
#define SV_RING_NENYA                   35
#define SV_RING_VILYA                   36
#define SV_RING_POWER                   37
#define SV_RING_RES_FEAR                38
#define SV_RING_RES_LD                  39
#define SV_RING_RES_NETHER              40
#define SV_RING_RES_NEXUS               41
#define SV_RING_RES_SOUND               42
#define SV_RING_RES_CONFUSION           43
#define SV_RING_RES_SHARDS              44
#define SV_RING_RES_DISENCHANT          45
#define SV_RING_RES_CHAOS               46
#define SV_RING_RES_BLINDNESS           47
#define SV_RING_LORDLY                  48
#define SV_RING_ATTACKS                 49
#define SV_RING_AHO                     50
#define SV_RING_SHOTS                   51
#define SV_RING_SUSTAIN                 52
#define SV_RING_DEC_MANA                53
#define SV_RING_WARNING                 54
#define SV_RING_MUSCLE                  55

#define SV_EXPRESS_CARD                  0

/* The "sval" codes for TV_STAFF */
#define SV_STAFF_DARKNESS                0
#define SV_STAFF_SLOWNESS                1
#define SV_STAFF_HASTE_MONSTERS          2
#define SV_STAFF_SUMMONING               3
#define SV_STAFF_TELEPORTATION           4
#define SV_STAFF_IDENTIFY                5
#define SV_STAFF_REMOVE_CURSE            6
#define SV_STAFF_STARLITE                7
#define SV_STAFF_LITE                    8
#define SV_STAFF_MAPPING                 9
#define SV_STAFF_DETECT_GOLD            10
#define SV_STAFF_DETECT_ITEM            11
#define SV_STAFF_DETECT_TRAP            12
#define SV_STAFF_DETECT_DOOR            13
#define SV_STAFF_DETECT_INVIS           14
#define SV_STAFF_DETECT_EVIL            15
#define SV_STAFF_CURE_LIGHT             16
#define SV_STAFF_CURING                 17
#define SV_STAFF_HEALING                18
#define SV_STAFF_THE_MAGI               19
#define SV_STAFF_SLEEP_MONSTERS         20
#define SV_STAFF_SLOW_MONSTERS          21
#define SV_STAFF_SPEED                  22
#define SV_STAFF_PROBING                23
#define SV_STAFF_DISPEL_EVIL            24
#define SV_STAFF_POWER                  25
#define SV_STAFF_HOLINESS               26
#define SV_STAFF_GENOCIDE               27
#define SV_STAFF_EARTHQUAKES            28
#define SV_STAFF_DESTRUCTION            29
#define SV_STAFF_ANIMATE_DEAD           30
#define SV_STAFF_MSTORM                 31
#define SV_STAFF_NOTHING                32


/* The "sval" codes for TV_WAND */
#define SV_WAND_HEAL_MONSTER             0
#define SV_WAND_HASTE_MONSTER            1
#define SV_WAND_CLONE_MONSTER            2
#define SV_WAND_TELEPORT_AWAY            3
#define SV_WAND_DISARMING                4
#define SV_WAND_TRAP_DOOR_DEST           5
#define SV_WAND_STONE_TO_MUD             6
#define SV_WAND_LITE                     7
#define SV_WAND_SLEEP_MONSTER            8
#define SV_WAND_SLOW_MONSTER             9
#define SV_WAND_CONFUSE_MONSTER         10
#define SV_WAND_FEAR_MONSTER            11
#define SV_WAND_HYPODYNAMIA              12
#define SV_WAND_POLYMORPH               13
#define SV_WAND_STINKING_CLOUD          14
#define SV_WAND_MAGIC_MISSILE           15
#define SV_WAND_ACID_BOLT               16
#define SV_WAND_CHARM_MONSTER           17
#define SV_WAND_FIRE_BOLT               18
#define SV_WAND_COLD_BOLT               19
#define SV_WAND_ACID_BALL               20
#define SV_WAND_ELEC_BALL               21
#define SV_WAND_FIRE_BALL               22
#define SV_WAND_COLD_BALL               23
#define SV_WAND_WONDER                  24
#define SV_WAND_DISINTEGRATE            25
#define SV_WAND_DRAGON_FIRE             26
#define SV_WAND_DRAGON_COLD             27
#define SV_WAND_DRAGON_BREATH           28
#define SV_WAND_ROCKETS                 29
#define SV_WAND_STRIKING                30
#define SV_WAND_GENOCIDE                31

/* The "sval" codes for TV_ROD */
#define SV_ROD_DETECT_TRAP               0
#define SV_ROD_DETECT_DOOR               1
#define SV_ROD_IDENTIFY                  2
#define SV_ROD_RECALL                    3
#define SV_ROD_ILLUMINATION              4
#define SV_ROD_MAPPING                   5
#define SV_ROD_DETECTION                 6
#define SV_ROD_PROBING                   7
#define SV_ROD_CURING                    8
#define SV_ROD_HEALING                   9
#define SV_ROD_RESTORATION              10
#define SV_ROD_SPEED                    11
#define SV_ROD_PESTICIDE                12
#define SV_ROD_TELEPORT_AWAY            13
#define SV_ROD_DISARMING                14
#define SV_ROD_LITE                     15
#define SV_ROD_SLEEP_MONSTER            16
#define SV_ROD_SLOW_MONSTER             17
#define SV_ROD_HYPODYNAMIA               18
#define SV_ROD_POLYMORPH                19
#define SV_ROD_ACID_BOLT                20
#define SV_ROD_ELEC_BOLT                21
#define SV_ROD_FIRE_BOLT                22
#define SV_ROD_COLD_BOLT                23
#define SV_ROD_ACID_BALL                24
#define SV_ROD_ELEC_BALL                25
#define SV_ROD_FIRE_BALL                26
#define SV_ROD_COLD_BALL                27
#define SV_ROD_HAVOC                    28
#define SV_ROD_STONE_TO_MUD             29
#define SV_ROD_AGGRAVATE                30


/* The "sval" codes for TV_SCROLL */

#define SV_SCROLL_DARKNESS               0
#define SV_SCROLL_AGGRAVATE_MONSTER      1
#define SV_SCROLL_CURSE_ARMOR            2
#define SV_SCROLL_CURSE_WEAPON           3
#define SV_SCROLL_SUMMON_MONSTER         4
#define SV_SCROLL_SUMMON_UNDEAD          5
#define SV_SCROLL_SUMMON_PET             6
#define SV_SCROLL_TRAP_CREATION          7
#define SV_SCROLL_PHASE_DOOR             8
#define SV_SCROLL_TELEPORT               9
#define SV_SCROLL_TELEPORT_LEVEL        10
#define SV_SCROLL_WORD_OF_RECALL        11
#define SV_SCROLL_IDENTIFY              12
#define SV_SCROLL_STAR_IDENTIFY         13
#define SV_SCROLL_REMOVE_CURSE          14
#define SV_SCROLL_STAR_REMOVE_CURSE     15
#define SV_SCROLL_ENCHANT_ARMOR         16
#define SV_SCROLL_ENCHANT_WEAPON_TO_HIT 17
#define SV_SCROLL_ENCHANT_WEAPON_TO_DAM 18
/* xxx enchant missile? */
#define SV_SCROLL_STAR_ENCHANT_ARMOR    20
#define SV_SCROLL_STAR_ENCHANT_WEAPON   21
#define SV_SCROLL_RECHARGING            22
#define SV_SCROLL_MUNDANITY             23
#define SV_SCROLL_LIGHT                 24
#define SV_SCROLL_MAPPING               25
#define SV_SCROLL_DETECT_GOLD           26
#define SV_SCROLL_DETECT_ITEM           27
#define SV_SCROLL_DETECT_TRAP           28
#define SV_SCROLL_DETECT_DOOR           29
#define SV_SCROLL_DETECT_INVIS          30
/* xxx (detect evil?) */
#define SV_SCROLL_SATISFY_HUNGER        32
#define SV_SCROLL_BLESSING              33
#define SV_SCROLL_HOLY_CHANT            34
#define SV_SCROLL_HOLY_PRAYER           35
#define SV_SCROLL_MONSTER_CONFUSION     36
#define SV_SCROLL_PROTECTION_FROM_EVIL  37
#define SV_SCROLL_RUNE_OF_PROTECTION    38
#define SV_SCROLL_TRAP_DOOR_DESTRUCTION 39
/* xxx */
#define SV_SCROLL_STAR_DESTRUCTION      41
#define SV_SCROLL_DISPEL_UNDEAD         42
#define SV_SCROLL_SPELL                 43
#define SV_SCROLL_GENOCIDE              44
#define SV_SCROLL_MASS_GENOCIDE         45
#define SV_SCROLL_ACQUIREMENT           46
#define SV_SCROLL_STAR_ACQUIREMENT      47
#define SV_SCROLL_FIRE                  48
#define SV_SCROLL_ICE                   49
#define SV_SCROLL_CHAOS                 50
#define SV_SCROLL_RUMOR                 51
#define SV_SCROLL_ARTIFACT              52
#define SV_SCROLL_RESET_RECALL          53
#define SV_SCROLL_SUMMON_KIN            54
#define SV_SCROLL_AMUSEMENT             55
#define SV_SCROLL_STAR_AMUSEMENT        56

/* The "sval" codes for TV_POTION */
#define SV_POTION_WATER                  0
#define SV_POTION_APPLE_JUICE            1
#define SV_POTION_SLIME_MOLD             2
/* xxx (fixed color) */
#define SV_POTION_SLOWNESS               4
#define SV_POTION_SALT_WATER             5
#define SV_POTION_POISON                 6
#define SV_POTION_BLINDNESS              7
/* xxx */
#define SV_POTION_CONFUSION              9
/* xxx */
#define SV_POTION_SLEEP                 11
/* xxx */
#define SV_POTION_LOSE_MEMORIES         13
/* xxx */
#define SV_POTION_RUINATION             15
#define SV_POTION_DEC_STR               16
#define SV_POTION_DEC_INT               17
#define SV_POTION_DEC_WIS               18
#define SV_POTION_DEC_DEX               19
#define SV_POTION_DEC_CON               20
#define SV_POTION_DEC_CHR               21
#define SV_POTION_DETONATIONS           22
#define SV_POTION_DEATH                 23
#define SV_POTION_INFRAVISION           24
#define SV_POTION_DETECT_INVIS          25
#define SV_POTION_SLOW_POISON           26
#define SV_POTION_CURE_POISON           27
#define SV_POTION_BOLDNESS              28
#define SV_POTION_SPEED                 29
#define SV_POTION_RESIST_HEAT           30
#define SV_POTION_RESIST_COLD           31
#define SV_POTION_HEROISM               32
#define SV_POTION_BESERK_STRENGTH       33
#define SV_POTION_CURE_LIGHT            34
#define SV_POTION_CURE_SERIOUS          35
#define SV_POTION_CURE_CRITICAL         36
#define SV_POTION_HEALING               37
#define SV_POTION_STAR_HEALING          38
#define SV_POTION_LIFE                  39
#define SV_POTION_RESTORE_MANA          40
#define SV_POTION_RESTORE_EXP           41
#define SV_POTION_RES_STR               42
#define SV_POTION_RES_INT               43
#define SV_POTION_RES_WIS               44
#define SV_POTION_RES_DEX               45
#define SV_POTION_RES_CON               46
#define SV_POTION_RES_CHR               47
#define SV_POTION_INC_STR               48
#define SV_POTION_INC_INT               49
#define SV_POTION_INC_WIS               50
#define SV_POTION_INC_DEX               51
#define SV_POTION_INC_CON               52
#define SV_POTION_INC_CHR               53
/* xxx */
#define SV_POTION_AUGMENTATION          55
#define SV_POTION_ENLIGHTENMENT         56
#define SV_POTION_STAR_ENLIGHTENMENT    57
#define SV_POTION_SELF_KNOWLEDGE        58
#define SV_POTION_EXPERIENCE            59
#define SV_POTION_RESISTANCE            60
#define SV_POTION_CURING                61
#define SV_POTION_INVULNERABILITY       62
#define SV_POTION_NEW_LIFE              63
#define SV_POTION_NEO_TSUYOSHI          64
#define SV_POTION_TSUYOSHI              65
#define SV_POTION_POLYMORPH             66

/* The "sval" codes for TV_FLASK */
#define SV_FLASK_OIL                   0

/* The "sval" codes for TV_FOOD */
#define SV_FOOD_POISON                   0
#define SV_FOOD_BLINDNESS                1
#define SV_FOOD_PARANOIA                 2
#define SV_FOOD_CONFUSION                3
#define SV_FOOD_HALLUCINATION            4
#define SV_FOOD_PARALYSIS                5
#define SV_FOOD_WEAKNESS                 6
#define SV_FOOD_SICKNESS                 7
#define SV_FOOD_STUPIDITY                8
#define SV_FOOD_NAIVETY                  9
#define SV_FOOD_UNHEALTH                10
#define SV_FOOD_DISEASE                 11
#define SV_FOOD_CURE_POISON             12
#define SV_FOOD_CURE_BLINDNESS          13
#define SV_FOOD_CURE_PARANOIA           14
#define SV_FOOD_CURE_CONFUSION          15
#define SV_FOOD_CURE_SERIOUS            16
#define SV_FOOD_RESTORE_STR             17
#define SV_FOOD_RESTORE_CON             18
#define SV_FOOD_RESTORING               19
/* many missing mushrooms */
#define SV_FOOD_BISCUIT                 32
#define SV_FOOD_JERKY                   33
#define SV_FOOD_RATION                  35
#define SV_FOOD_SLIME_MOLD              36
#define SV_FOOD_WAYBREAD                37
#define SV_FOOD_PINT_OF_ALE             38
#define SV_FOOD_PINT_OF_WINE            39


#define SV_ROD_MIN_DIRECTION    12 /*!< この値以降の小項目IDを持ったロッドは使用時にターゲットを要求する / Special "sval" limit -- first "aimed" rod */

#define SV_CHEST_MIN_LARGE      4  /*!< この値以降の小項目IDを持った箱は大型の箱としてドロップ数を増やす / Special "sval" limit -- first "large" chest */
#define SV_CHEST_KANDUME        50 /*!< 箱アイテムの小項目ID: おもちゃのカンヅメ */

/*
 * Special "sval" limit -- first "good" magic/prayer book
 */
#define SV_BOOK_MIN_GOOD    2


#define OBJ_GOLD_LIST   480     /* First "gold" entry */
#define MAX_GOLD        18      /* Number of "gold" entries */


/*** General flag values ***/


/*
 * 特殊なマス状態フラグ / Special cave grid flags
 */
#define CAVE_MARK       0x0001    /*!< 現在プレイヤーの記憶に収まっている / memorized feature */
#define CAVE_GLOW       0x0002    /*!< マス自体が光源を持っている / self-illuminating */
#define CAVE_ICKY       0x0004    /*!< 生成されたVaultの一部である / part of a vault */
#define CAVE_ROOM       0x0008    /*!< 生成された部屋の一部である / part of a room */
#define CAVE_LITE       0x0010    /*!< 現在光に照らされている / lite flag  */
#define CAVE_VIEW       0x0020    /*!< 現在プレイヤーの視界に収まっている / view flag */
#define CAVE_TEMP       0x0040    /*!< 光源に関する処理のアルゴリズム用記録フラグ / temp flag */
#define CAVE_XTRA       0x0080    /*!< 視界に関する処理のアルゴリズム用記録フラグ(update_view()等参照) / misc flag */
#define CAVE_MNLT       0x0100    /*!< モンスターの光源によって照らされている / Illuminated by monster */
#define CAVE_MNDK       0x8000    /*!< モンスターの暗源によって暗闇になっている / Darken by monster */

/* Used only while cave generation */
#define CAVE_FLOOR      0x0200	/*!< フロア属性のあるマス */
#define CAVE_EXTRA      0x0400
#define CAVE_INNER      0x0800
#define CAVE_OUTER      0x1000
#define CAVE_SOLID      0x2000
#define CAVE_VAULT      0x4000
#define CAVE_MASK (CAVE_FLOOR | CAVE_EXTRA | CAVE_INNER | CAVE_OUTER | CAVE_SOLID | CAVE_VAULT)

/* Used only after cave generation */
#define CAVE_KNOWN      0x0200    /* Directly viewed or map detected flag */
#define CAVE_NOTE       0x0400    /* Flag for delayed visual update (needs note_spot()) */
#define CAVE_REDRAW     0x0800    /* Flag for delayed visual update (needs lite_spot()) */
#define CAVE_OBJECT     0x1000    /* Mirror, glyph, etc. */
#define CAVE_UNSAFE     0x2000    /* Might have trap */
#define CAVE_IN_DETECT  0x4000    /* trap detected area (inner circle only) */


/*
 * project()関数に用いられる、遠隔攻撃特性ビットフラグ / Bit flags for the "project()" function
 */
#define PROJECT_JUMP        0x0001 /*!< 遠隔攻撃特性: 発動者からの軌跡を持たず、指定地点に直接発生する(予め置いたトラップ、上空からの発生などのイメージ) / Jump directly to the target location (this is a hack) */
#define PROJECT_BEAM        0x0002 /*!< 遠隔攻撃特性: ビーム範囲を持つ。 / Work as a beam weapon (affect every grid passed through) */
#define PROJECT_THRU        0x0004 /*!< 遠隔攻撃特性: 目標地点に到達しても射程と遮蔽の限り引き延ばす。 / Continue "through" the target (used for "bolts"/"beams") */
#define PROJECT_STOP        0x0008 /*!< 遠隔攻撃特性: 道中にプレイヤーかモンスターがいた時点で到達地点を更新して停止する(壁や森はPROJECT_DISIがない限り最初から貫通しない) */
#define PROJECT_GRID        0x0010 /*!< 遠隔攻撃特性: 射程内の地形に影響を及ぼす / Affect each grid in the "blast area" in some way */
#define PROJECT_ITEM        0x0020 /*!< 遠隔攻撃特性: 射程内のアイテムに影響を及ぼす / Affect each object in the "blast area" in some way */
#define PROJECT_KILL        0x0040 /*!< 遠隔攻撃特性: 射程内のモンスターに影響を及ぼす / Affect each monster in the "blast area" in some way */
#define PROJECT_HIDE        0x0080 /*!< 遠隔攻撃特性: / Hack -- disable "visual" feedback from projection */
#define PROJECT_DISI        0x0100 /*!< 遠隔攻撃特性: / Disintegrate non-permanent features */
#define PROJECT_PLAYER      0x0200 /*!< 遠隔攻撃特性: / Main target is player (used for riding player) */
#define PROJECT_AIMED       0x0400 /*!< 遠隔攻撃特性: / Target is only player or monster, so don't affect another. Depend on PROJECT_PLAYER. (used for minimum (rad == 0) balls on riding player) */
#define PROJECT_REFLECTABLE 0x0800 /*!< 遠隔攻撃特性: 反射可能(ボルト系魔法に利用) / Refrectable spell attacks (used for "bolts") */
#define PROJECT_NO_HANGEKI  0x1000 /*!< 遠隔攻撃特性: / Avoid counter attacks of monsters */
#define PROJECT_PATH        0x2000 /*!< 遠隔攻撃特性: / Only used for printing project path */
#define PROJECT_FAST        0x4000 /*!< 遠隔攻撃特性: / Hide "visual" of flying bolts until blast */
#define PROJECT_LOS         0x8000 /*!< 遠隔攻撃特性: /  */


/*
 * Special caster ID for project()
 */
#define PROJECT_WHO_UNCTRL_POWER -1 /*!< 魔法効果の自然発生要因: 名状し難い力の解放 */
#define PROJECT_WHO_GLASS_SHARDS -2 /*!< 魔法効果の自然発生要因: 破壊されたガラス地形の破片 */


/*
 * Bit flags for the "enchant()" function
 */
#define ENCH_TOHIT      0x01 /*!< 装備強化処理: 命中強化 / Enchant to hit */
#define ENCH_TODAM      0x02 /*!< 装備強化処理: ダメージ強化 / Enchant to damage */
#define ENCH_TOAC       0x04 /*!< 装備強化処理: AC強化 / Enchant to AC */
#define ENCH_FORCE      0x08 /*!< 装備強化処理: 無条件に成功させる / Force enchantment */

/*
 * target_set用関数の利用用途フラグ / Bit flags for the "target_set" function
 */
#define TARGET_KILL     0x01 /*!< モンスターへの狙いをつける(視界内モンスターのみクエリ対象) / Target monsters */
#define TARGET_LOOK     0x02 /*!< "L"ookコマンド向けの既存情報確認向け(全ての有為な情報をクエリ対象) / Describe grid fully */
#define TARGET_XTRA     0x04 /*!< 現在未使用 / Currently unused flag */
#define TARGET_GRID     0x08 /*!< 全てのマス対象にする(現在未使用) / Select from all grids */


/*
 * Bit flags for control of get_check_strict()
 */
#define CHECK_OKAY_CANCEL 0x01
#define CHECK_NO_ESCAPE   0x02
#define CHECK_NO_HISTORY  0x04
#define CHECK_DEFAULT_Y   0x08


/*
 * Some bit-flags for the "smart" field
 */
#define SM_RES_ACID             0x00000001 /*!< モンスターの学習フラグ: プレイヤーに酸耐性あり */
#define SM_RES_ELEC             0x00000002 /*!< モンスターの学習フラグ: プレイヤーに電撃耐性あり */
#define SM_RES_FIRE             0x00000004 /*!< モンスターの学習フラグ: プレイヤーに火炎耐性あり */
#define SM_RES_COLD             0x00000008 /*!< モンスターの学習フラグ: プレイヤーに冷気耐性あり */
#define SM_RES_POIS             0x00000010 /*!< モンスターの学習フラグ: プレイヤーに毒耐性あり */
#define SM_RES_NETH             0x00000020 /*!< モンスターの学習フラグ: プレイヤーに地獄耐性あり */
#define SM_RES_LITE             0x00000040 /*!< モンスターの学習フラグ: プレイヤーに閃光耐性あり */
#define SM_RES_DARK             0x00000080 /*!< モンスターの学習フラグ: プレイヤーに暗黒耐性あり */
#define SM_RES_FEAR             0x00000100 /*!< モンスターの学習フラグ: プレイヤーに恐怖耐性あり */
#define SM_RES_CONF             0x00000200 /*!< モンスターの学習フラグ: プレイヤーに混乱耐性あり */
#define SM_RES_CHAOS            0x00000400 /*!< モンスターの学習フラグ: プレイヤーにカオス耐性あり */
#define SM_RES_DISEN            0x00000800 /*!< モンスターの学習フラグ: プレイヤーに劣化耐性あり */
#define SM_RES_BLIND            0x00001000 /*!< モンスターの学習フラグ: プレイヤーに盲目耐性あり */
#define SM_RES_NEXUS            0x00002000 /*!< モンスターの学習フラグ: プレイヤーに因果混乱耐性あり */
#define SM_RES_SOUND            0x00004000 /*!< モンスターの学習フラグ: プレイヤーに轟音耐性あり */
#define SM_RES_SHARD            0x00008000 /*!< モンスターの学習フラグ: プレイヤーに破片耐性あり */
#define SM_OPP_ACID             0x00010000 /*!< モンスターの学習フラグ: プレイヤーに二重酸耐性あり */
#define SM_OPP_ELEC             0x00020000 /*!< モンスターの学習フラグ: プレイヤーに二重電撃耐性あり */
#define SM_OPP_FIRE             0x00040000 /*!< モンスターの学習フラグ: プレイヤーに二重火炎耐性あり */
#define SM_OPP_COLD             0x00080000 /*!< モンスターの学習フラグ: プレイヤーに二重冷気耐性あり */
#define SM_OPP_POIS             0x00100000 /*!< モンスターの学習フラグ: プレイヤーに二重毒耐性あり */
#define SM_OPP_XXX1             0x00200000 /*!< 未使用 / (unused) */
#define SM_CLONED               0x00400000 /*!< クローンである / Cloned */
#define SM_PET                  0x00800000 /*!< ペットである / Pet */
#define SM_IMM_ACID             0x01000000 /*!< モンスターの学習フラグ: プレイヤーに酸免疫あり */
#define SM_IMM_ELEC             0x02000000 /*!< モンスターの学習フラグ: プレイヤーに電撃免疫あり */
#define SM_IMM_FIRE             0x04000000 /*!< モンスターの学習フラグ: プレイヤーに火炎免疫あり */
#define SM_IMM_COLD             0x08000000 /*!< モンスターの学習フラグ: プレイヤーに冷気免疫あり */
#define SM_FRIENDLY             0x10000000 /*!< 友好的である / Friendly */
#define SM_IMM_REFLECT          0x20000000 /*!< モンスターの学習フラグ: プレイヤーに反射あり */
#define SM_IMM_FREE             0x40000000 /*!< モンスターの学習フラグ: プレイヤーに麻痺耐性あり */
#define SM_IMM_MANA             0x80000000 /*!< モンスターの学習フラグ: プレイヤーにMPがない */


/*
 * get_item()関数でアイテムの選択を行うフラグ / Bit flags for the "get_item" function
 */
#define USE_EQUIP	0x01	/*!< アイテム選択範囲: 装備品からの選択を許可する / Allow equip items */
#define USE_INVEN	0x02	/*!< アイテム選択範囲: 所持品からの選択を許可する /  Allow inven items */
#define USE_FLOOR	0x04	/*!< アイテム選択範囲: 床下のアイテムからの選択を許可する /  Allow floor items */


/*
 * p_ptr->noticeに充てるアイテム更新処理要求 / Bit flags for the "p_ptr->notice" variable
 */
#define PN_COMBINE      0x00000001L     /*!< アイテム処理フラグ: アイテムの結合を要する / Combine the pack */
#define PN_REORDER      0x00000002L     /*!< アイテム処理フラグ: アイテムの並び替えを要する / Reorder the pack */
#define PN_AUTODESTROY  0x00000004L     /*!< アイテム処理フラグ: アイテムの自動破壊を要する / Auto-destroy marked item */
/* xxx (many) */


/*
 * p_ptr->updateに充てるプレイヤーステータス更新処理要求 / Bit flags for the "p_ptr->update" variable
 */
#define PU_BONUS        0x00000001L     /*!< ステータス更新フラグ: 能力値修正 / Calculate bonuses */
#define PU_TORCH        0x00000002L     /*!< ステータス更新フラグ: 光源半径 / Calculate torch radius */
/* xxx (many) */
#define PU_HP           0x00000010L     /*!< ステータス更新フラグ: HP / Calculate chp and mhp */
#define PU_MANA         0x00000020L     /*!< ステータス更新フラグ: MP / Calculate csp and msp */
#define PU_SPELLS       0x00000040L     /*!< ステータス更新フラグ: 魔法学習数 / Calculate spells */
/* xxx (many) */
/* xxx (many) */
#define PU_UN_VIEW      0x00010000L     /*!< ステータス更新フラグ: 地形の視界外化 / Forget view */
#define PU_UN_LITE      0x00020000L     /*!< ステータス更新フラグ: 明暗範囲の視界外化 / Forget lite */
/* xxx (many) */
#define PU_VIEW         0x00100000L     /*!< ステータス更新フラグ: 視界 / Update view */
#define PU_LITE         0x00200000L     /*!< ステータス更新フラグ: 明暗範囲 / Update lite */
#define PU_MON_LITE     0x00400000L     /*!< ステータス更新フラグ: モンスターの光源範囲 / Monster illumination */
#define PU_DELAY_VIS    0x00800000L     /*!< ステータス更新フラグ: 視界の追加更新 / Mega-Hack -- Delayed visual update */
#define PU_MONSTERS     0x01000000L     /*!< ステータス更新フラグ: モンスターのステータス / Update monsters */
#define PU_DISTANCE     0x02000000L     /*!< ステータス更新フラグ: プレイヤーとモンスターの距離 / Update distances */
/* xxx */
#define PU_FLOW         0x10000000L     /*!< ステータス更新フラグ: プレイヤーから各マスへの到達距離 / Update flow */
/* xxx (many) */


/*
 * Bit flags for the "p_ptr->redraw" variable
 */
#define PR_MISC         0x00000001L     /*!< 再描画フラグ: 種族と職業 / Display Race/Class */
#define PR_TITLE        0x00000002L     /*!< 再描画フラグ: 称号 / Display Title */
#define PR_LEV          0x00000004L     /*!< 再描画フラグ: レベル / Display Level */
#define PR_EXP          0x00000008L     /*!< 再描画フラグ: 経験値 / Display Experience */
#define PR_STATS        0x00000010L     /*!< 再描画フラグ: ステータス /  Display Stats */
#define PR_ARMOR        0x00000020L     /*!< 再描画フラグ: AC / Display Armor */
#define PR_HP           0x00000040L     /*!< 再描画フラグ: HP / Display Hitpoints */
#define PR_MANA         0x00000080L     /*!< 再描画フラグ: MP / Display Mana */
#define PR_GOLD         0x00000100L     /*!< 再描画フラグ: 所持金 / Display Gold */
#define PR_DEPTH        0x00000200L     /*!< 再描画フラグ: ダンジョンの階 / Display Depth */
#define PR_EQUIPPY      0x00000400L     /*!< 再描画フラグ: 装備シンボル / Display equippy chars */
#define PR_HEALTH       0x00000800L     /*!< 再描画フラグ: モンスターのステータス / Display Health Bar */
#define PR_CUT          0x00001000L     /*!< 再描画フラグ: 負傷度 / Display Extra (Cut) */
#define PR_STUN         0x00002000L     /*!< 再描画フラグ: 朦朧度 / Display Extra (Stun) */
#define PR_HUNGER       0x00004000L     /*!< 再描画フラグ: 空腹度 / Display Extra (Hunger) */
#define PR_STATUS       0x00008000L     /*!< 再描画フラグ: プレイヤーの付与状態 /  Display Status Bar */
#define PR_XXX0         0x00010000L     /*!< (unused) */
#define PR_UHEALTH      0x00020000L     /*!< 再描画フラグ: ペットのステータス / Display Uma Health Bar */
#define PR_XXX1         0x00040000L     /*!< (unused) */
#define PR_XXX2         0x00080000L     /*!< (unused) */
#define PR_STATE        0x00100000L     /*!< 再描画フラグ: プレイヤーの行動状態 / Display Extra (State) */
#define PR_SPEED        0x00200000L     /*!< 再描画フラグ: 加速 / Display Extra (Speed) */
#define PR_STUDY        0x00400000L     /*!< 再描画フラグ: 学習 / Display Extra (Study) */
#define PR_IMITATION    0x00800000L     /*!< 再描画フラグ: ものまね / Display Extra (Imitation) */
#define PR_EXTRA        0x01000000L     /*!< 再描画フラグ: 拡張ステータス全体 / Display Extra Info */
#define PR_BASIC        0x02000000L     /*!< 再描画フラグ: 基本ステータス全体 / Display Basic Info */
#define PR_MAP          0x04000000L     /*!< 再描画フラグ: ゲームマップ / Display Map */
#define PR_WIPE         0x08000000L     /*!< 再描画フラグ: 画面消去 / Hack -- Total Redraw */
/* xxx */
/* xxx */
/* xxx */
/* xxx */

/*
 * Bit flags for the "p_ptr->window" variable (etc)
 */
#define PW_INVEN        0x00000001L     /*!<サブウィンドウ描画フラグ: 所持品-装備品 / Display inven/equip */
#define PW_EQUIP        0x00000002L     /*!<サブウィンドウ描画フラグ: 装備品-所持品 / Display equip/inven */
#define PW_SPELL        0x00000004L     /*!<サブウィンドウ描画フラグ: 魔法一覧 / Display spell list */
#define PW_PLAYER       0x00000008L     /*!<サブウィンドウ描画フラグ: プレイヤーのステータス / Display character */
#define PW_MONSTER_LIST 0x00000010L     /*!<サブウィンドウ描画フラグ: 視界内モンスターの一覧 / Display monster list */
/* xxx */
/* xxx */
#define PW_MESSAGE      0x00000040L     /*!<サブウィンドウ描画フラグ: メッセージログ / Display messages */
#define PW_OVERHEAD     0x00000080L     /*!<サブウィンドウ描画フラグ: 周辺の光景 / Display overhead view */
#define PW_MONSTER      0x00000100L     /*!<サブウィンドウ描画フラグ: モンスターの思い出 / Display monster recall */
#define PW_OBJECT       0x00000200L     /*!<サブウィンドウ描画フラグ: アイテムの知識 / Display object recall */
#define PW_DUNGEON      0x00000400L     /*!<サブウィンドウ描画フラグ: ダンジョンの地形 / Display dungeon view */
#define PW_SNAPSHOT     0x00000800L     /*!<サブウィンドウ描画フラグ: 記念写真 / Display snap-shot */
/* xxx */
/* xxx */
#define PW_BORG_1       0x00004000L     /*!<サブウィンドウ描画フラグ: ボーグメッセージ / Display borg messages */
#define PW_BORG_2       0x00008000L     /*!<サブウィンドウ描画フラグ: ボーグステータス / Display borg status */

/*
 * Bit flags for the place_monster_???() (etc)
 */
#define PM_ALLOW_SLEEP    0x00000001    /*!< モンスター生成フラグ: 眠っている状態で生成されても良い */
#define PM_ALLOW_GROUP    0x00000002    /*!< モンスター生成フラグ: 集団生成されても良い */
#define PM_FORCE_FRIENDLY 0x00000004    /*!< モンスター生成フラグ: 必ず友好的に生成される */
#define PM_FORCE_PET      0x00000008    /*!< モンスター生成フラグ: 必ずペットとして生成される */
#define PM_NO_KAGE        0x00000010    /*!< モンスター生成フラグ: 必ずあやしい影としては生成されない */
#define PM_NO_PET         0x00000020    /*!< モンスター生成フラグ: 必ずペットとして生成されない */
#define PM_ALLOW_UNIQUE   0x00000040    /*!< モンスター生成フラグ: ユニークの選択生成を許可する */
#define PM_IGNORE_TERRAIN 0x00000080    /*!< モンスター生成フラグ: 侵入可能地形を考慮せずに生成する */
#define PM_HASTE          0x00000100    /*!< モンスター生成フラグ: 加速状態で生成する */
#define PM_KAGE           0x00000200    /*!< モンスター生成フラグ: 必ずあやしい影として生成する */
#define PM_MULTIPLY       0x00000400    /*!< モンスター生成フラグ: 増殖処理時として生成する */


/* Bit flags for monster_desc() */
#define MD_OBJECTIVE      0x00000001 /* Objective (or Reflexive) */
#define MD_POSSESSIVE     0x00000002 /* Possessive (or Reflexive) */
#define MD_INDEF_HIDDEN   0x00000004 /* Use indefinites for hidden monsters ("something") */
#define MD_INDEF_VISIBLE  0x00000008 /* Use indefinites for visible monsters ("a kobold") */
#define MD_PRON_HIDDEN    0x00000010 /* Pronominalize hidden monsters */
#define MD_PRON_VISIBLE   0x00000020 /* Pronominalize visible monsters */
#define MD_ASSUME_HIDDEN  0x00000040 /* Assume the monster is hidden */
#define MD_ASSUME_VISIBLE 0x00000080 /* Assume the monster is visible */
#define MD_TRUE_NAME      0x00000100 /* Chameleon's true name */
#define MD_IGNORE_HALLU   0x00000200 /* Ignore hallucination, and penetrate shape change */


/*
 * Bit flags for object_desc()
 */
#define OD_NAME_ONLY        0x00000001  /* Omit values, pval, inscription */
#define OD_NAME_AND_ENCHANT 0x00000002  /* Omit pval, inscription */
#define OD_OMIT_INSCRIPTION 0x00000004  /* Omit inscription */
#define OD_OMIT_PREFIX      0x00000008  /* Omit numeric prefix */
#define OD_NO_PLURAL        0x00000010  /* Don't use plural */
#define OD_STORE            0x00000020  /* Assume to be aware and known */
#define OD_NO_FLAVOR        0x00000040  /* Allow to hidden flavor */
#define OD_FORCE_FLAVOR     0x00000080  /* Get un-shuffled flavor name */


/*
 * p_ptr->special_attackによるプレイヤーの攻撃状態の定義 / Bit flags for the "p_ptr->special_attack" variable. -LM-
 *
 * Note:  The elemental and poison attacks should be managed using the 
 * function "set_ele_attack", in spell2.c.  This provides for timeouts and
 * prevents the player from getting more than one at a time.
 */
#define ATTACK_CONFUSE	0x00000001 /*!< プレイヤーのステータス:混乱打撃 */
#define ATTACK_XXX1		0x00000002 /*!< プレイヤーのステータス:未使用1 */
#define ATTACK_XXX2		0x00000004 /*!< プレイヤーのステータス:未使用2 */
#define ATTACK_XXX3	    0x00000008 /*!< プレイヤーのステータス:未使用3 */
#define ATTACK_ACID		0x00000010 /*!< プレイヤーのステータス:魔法剣/溶解 */
#define ATTACK_ELEC		0x00000020 /*!< プレイヤーのステータス:魔法剣/電撃 */
#define ATTACK_FIRE		0x00000040 /*!< プレイヤーのステータス:魔法剣/火炎 */
#define ATTACK_COLD		0x00000080 /*!< プレイヤーのステータス:魔法剣/冷凍 */
#define ATTACK_POIS		0x00000100 /*!< プレイヤーのステータス:魔法剣/毒殺 */
#define ATTACK_HOLY		0x00000200 /*!< プレイヤーのステータス:対邪?(未使用) */
#define ATTACK_SUIKEN	0x00000400 /*!< プレイヤーのステータス:酔拳 */

/*
 * p_ptr->special_defenseによるプレイヤーの防御状態の定義 / Bit flags for the "p_ptr->special_defense" variable. -LM-
 */
#define DEFENSE_ACID	0x00000001 /*!< プレイヤーのステータス:酸免疫 */
#define DEFENSE_ELEC	0x00000002 /*!< プレイヤーのステータス:電撃免疫 */
#define DEFENSE_FIRE	0x00000004 /*!< プレイヤーのステータス:火炎免疫 */
#define DEFENSE_COLD	0x00000008 /*!< プレイヤーのステータス:冷気免疫 */
#define DEFENSE_POIS	0x00000010 /*!< プレイヤーのステータス:毒免疫 */
#define KAMAE_GENBU     0x00000020 /*!< プレイヤーのステータス:玄武の構え */
#define KAMAE_BYAKKO    0x00000040 /*!< プレイヤーのステータス:白虎の構え */
#define KAMAE_SEIRYU    0x00000080 /*!< プレイヤーのステータス:青竜の構え */
#define KAMAE_SUZAKU    0x00000100 /*!< プレイヤーのステータス:朱雀の構え */
#define KATA_IAI        0x00000200 /*!< プレイヤーのステータス:居合 */
#define KATA_FUUJIN     0x00000400 /*!< プレイヤーのステータス:風塵 */
#define KATA_KOUKIJIN   0x00000800 /*!< プレイヤーのステータス:降鬼陣 */
#define KATA_MUSOU      0x00001000 /*!< プレイヤーのステータス:無想 */
#define NINJA_KAWARIMI  0x00002000 /*!< プレイヤーのステータス:変わり身 */
#define NINJA_S_STEALTH 0x00004000 /*!< プレイヤーのステータス:超隠密 */

#define MAX_KAMAE 4 /*!< 修行僧の構え最大数 */
#define KAMAE_MASK (KAMAE_GENBU | KAMAE_BYAKKO | KAMAE_SEIRYU | KAMAE_SUZAKU) /*!< 修行僧の構えビット配列 */

#define MAX_KATA 4 /*!< 修行僧の型最大数 */
#define KATA_MASK (KATA_IAI | KATA_FUUJIN | KATA_KOUKIJIN | KATA_MUSOU) /*!< 修行僧の型ビット配列 */


#define ACTION_NONE     0 /*!< 持続行動: なし */
#define ACTION_SEARCH   1 /*!< 持続行動: 探索 */
#define ACTION_REST     2 /*!< 持続行動: 休憩 */
#define ACTION_LEARN    3 /*!< 持続行動: 青魔法ラーニング */
#define ACTION_FISH     4 /*!< 持続行動: 釣り */
#define ACTION_KAMAE    5 /*!< 持続行動: 修行僧の構え */
#define ACTION_KATA     6 /*!< 持続行動: 剣術家の型 */
#define ACTION_SING     7 /*!< 持続行動: 歌 */
#define ACTION_HAYAGAKE 8 /*!< 持続行動: 早駆け */
#define ACTION_SPELL    9 /*!< 持続行動: 呪術 */


/* Empty hand status */
#define EMPTY_HAND_NONE 0x0000 /* Both hands are used */
#define EMPTY_HAND_LARM 0x0001 /* Left hand is empty */
#define EMPTY_HAND_RARM 0x0002 /* Right hand is empty */

/*** General index values ***/


/*
 * summon_specificで取り扱われる、召喚の種別定義 / Legal restrictions for "summon_specific()"
 */
#define SUMMON_ANT                  11 /*!< 召喚タイプ: アリ */
#define SUMMON_SPIDER               12 /*!< 召喚タイプ: 蜘蛛 */
#define SUMMON_HOUND                13 /*!< 召喚タイプ: ハウンド */
#define SUMMON_HYDRA                14 /*!< 召喚タイプ: ヒドラ */
#define SUMMON_ANGEL                15 /*!< 召喚タイプ: 天使 */
#define SUMMON_DEMON                16 /*!< 召喚タイプ: 悪魔 */
#define SUMMON_UNDEAD               17 /*!< 召喚タイプ: アンデッド */
#define SUMMON_DRAGON               18 /*!< 召喚タイプ: ドラゴン */
#define SUMMON_HI_UNDEAD            21 /*!< 召喚タイプ: 強力なアンデッド */
#define SUMMON_HI_DRAGON            22 /*!< 召喚タイプ: 古代ドラゴン */
#define SUMMON_HI_DEMON             23 /*!< 召喚タイプ: 上級デーモン */
#define SUMMON_AMBERITES            31 /*!< 召喚タイプ: アンバーの王族 */
#define SUMMON_UNIQUE               32 /*!< 召喚タイプ: ユニーク */
#define SUMMON_MOLD                 33 /*!< 召喚タイプ: カビ */
#define SUMMON_BAT                  34 /*!< 召喚タイプ: コウモリ */
#define SUMMON_QUYLTHULG            35 /*!< 召喚タイプ: クイルスルグ */
#define SUMMON_XXX1                 36 /*!< 召喚タイプ: 未使用 */
#define SUMMON_COIN_MIMIC           37 /*!< 召喚タイプ: クリーピング・コイン */
#define SUMMON_MIMIC                38 /*!< 召喚タイプ: ミミック */
#define SUMMON_CYBER                39 /*!< 召喚タイプ: サイバーデーモン */
#define SUMMON_KIN                  40 /*!< 召喚タイプ: 召喚者の同族 */
#define SUMMON_DAWN                 41 /*!< 召喚タイプ: 暁の戦士 */
#define SUMMON_ANIMAL               42 /*!< 召喚タイプ: 自然界の動物 */
#define SUMMON_ANIMAL_RANGER        43 /*!< 召喚タイプ: レンジャー向け自然界の動物 */
/*#define SUMMON_HI_UNDEAD_NO_UNIQUES 44*/
/*#define SUMMON_HI_DRAGON_NO_UNIQUES 45*/
/*#define SUMMON_NO_UNIQUES           46*/
#define SUMMON_PHANTOM              47 /*!< 召喚タイプ: ゴースト */
/*#define SUMMON_ELEMENTAL_NO_UNIQUES 48*/
#define SUMMON_BLUE_HORROR          49 /*!< 召喚タイプ: ブルー・ホラー */
#define SUMMON_LIVING               50 /*!< 召喚タイプ: 生命のあるモンスター */
#define SUMMON_HI_DRAGON_LIVING     51 /*!< 召喚タイプ: 生命のある古代ドラゴン */
#define SUMMON_GOLEM                52 /*!< 召喚タイプ: ゴーレム */
#define SUMMON_ELEMENTAL            53 /*!< 召喚タイプ: エレメンタル */
#define SUMMON_VORTEX               54 /*!< 召喚タイプ: ボルテックス */
#define SUMMON_HYBRID               55 /*!< 召喚タイプ: 混合生物 */
#define SUMMON_BIRD                 56 /*!< 召喚タイプ: 鳥 */
/*#define SUMMON_AQUATIC_NO_UNIQUES   57*/
#define SUMMON_KAMIKAZE             58 /*!< 召喚タイプ: 自爆モンスター */
#define SUMMON_KAMIKAZE_LIVING      59 /*!< 召喚タイプ: 生命のある自爆モンスター */
#define SUMMON_MANES                60 /*!< 召喚タイプ: 古代の死霊 */
#define SUMMON_LOUSE                61 /*!< 召喚タイプ: シラミ */
#define SUMMON_GUARDIANS            62 /*!< 召喚タイプ: ダンジョンの主 */
#define SUMMON_KNIGHTS              63 /*!< 召喚タイプ: 聖戦用騎士系モンスター */
#define SUMMON_EAGLES               64 /*!< 召喚タイプ: 鷲系モンスター */
#define SUMMON_PIRANHAS             65 /*!< 召喚タイプ: ピラニア・トラップ用 */
#define SUMMON_ARMAGE_GOOD          66 /*!< 召喚タイプ: ハルマゲドン・トラップ用天使陣営 */
#define SUMMON_ARMAGE_EVIL          67 /*!< 召喚タイプ: ハルマゲドン・トラップ用悪魔陣営 */


/*
 * Spell types used by project(), and related functions.
 */
#define GF_ELEC         1			/*!< 魔法効果: 電撃*/
#define GF_POIS         2			/*!< 魔法効果: 毒*/
#define GF_ACID         3			/*!< 魔法効果: 酸*/
#define GF_COLD         4			/*!< 魔法効果: 冷気*/
#define GF_FIRE         5			/*!< 魔法効果: 火炎*/
#define GF_PSY_SPEAR    9			/*!< 魔法効果: 光の剣*/
#define GF_MISSILE      10			/*!< 魔法効果: 弱魔力*/
#define GF_ARROW        11			/*!< 魔法効果: 射撃*/
#define GF_PLASMA       12			/*!< 魔法効果: プラズマ*/
/* Replaced with GF_HOLY_FIRE and GF_HELL_FIRE */
/* #define GF_HOLY_ORB     13 */
#define GF_WATER        14			/*!< 魔法効果: 水流*/
#define GF_LITE         15			/*!< 魔法効果: 閃光*/
#define GF_DARK         16			/*!< 魔法効果: 暗黒*/
#define GF_LITE_WEAK    17			/*!< 魔法効果: 弱光*/
#define GF_DARK_WEAK    18			/*!< 魔法効果: 弱暗*/
#define GF_SHARDS       20			/*!< 魔法効果: 破片*/
#define GF_SOUND        21			/*!< 魔法効果: 轟音*/
#define GF_CONFUSION    22			/*!< 魔法効果: 混乱*/
#define GF_FORCE        23			/*!< 魔法効果: フォース*/
#define GF_INERTIAL     24			/*!< 魔法効果: 遅鈍*/
#define GF_MANA         26			/*!< 魔法効果: 純粋魔力*/
#define GF_METEOR       27			/*!< 魔法効果: 隕石*/
#define GF_ICE          28			/*!< 魔法効果: 極寒*/
#define GF_CHAOS        30			/*!< 魔法効果: カオス*/
#define GF_NETHER       31			/*!< 魔法効果: 地獄*/
#define GF_DISENCHANT   32			/*!< 魔法効果: 劣化*/
#define GF_NEXUS        33			/*!< 魔法効果: 因果混乱*/
#define GF_TIME         34			/*!< 魔法効果: 時間逆転*/
#define GF_GRAVITY      35			/*!< 魔法効果: 重力*/
#define GF_KILL_WALL    40			/*!< 魔法効果: 岩石溶解*/
#define GF_KILL_DOOR    41			/*!< 魔法効果: ドア破壊*/
#define GF_KILL_TRAP    42			/*!< 魔法効果: トラップ破壊*/
#define GF_MAKE_WALL    45			/*!< 魔法効果: 壁生成*/
#define GF_MAKE_DOOR    46			/*!< 魔法効果: ドア生成*/
#define GF_MAKE_TRAP    47			/*!< 魔法効果: トラップ生成*/
#define GF_MAKE_TREE    48			/*!< 魔法効果: 森林生成*/
#define GF_OLD_CLONE    51			/*!< 魔法効果: クローン・モンスター*/
#define GF_OLD_POLY     52			/*!< 魔法効果: チェンジ・モンスター*/
#define GF_OLD_HEAL     53			/*!< 魔法効果: 回復モンスター*/
#define GF_OLD_SPEED    54			/*!< 魔法効果: スピード・モンスター*/
#define GF_OLD_SLOW     55			/*!< 魔法効果: スロウ・モンスター*/
#define GF_OLD_CONF     56			/*!< 魔法効果: パニック・モンスター*/
#define GF_OLD_SLEEP    57			/*!< 魔法効果: スリープ・モンスター*/
#define GF_HYPODYNAMIA  58			/*!< 魔法効果: 衰弱*/
#define GF_AWAY_UNDEAD  61			/*!< 魔法効果: アンデッド・アウェイ*/
#define GF_AWAY_EVIL    62			/*!< 魔法効果: 邪悪飛ばし*/
#define GF_AWAY_ALL     63			/*!< 魔法効果: テレポート・アウェイ*/
#define GF_TURN_UNDEAD  64			/*!< 魔法効果: アンデッド恐慌*/
#define GF_TURN_EVIL    65			/*!< 魔法効果: 邪悪恐慌*/
#define GF_TURN_ALL     66			/*!< 魔法効果: モンスター恐慌*/
#define GF_DISP_UNDEAD  67			/*!< 魔法効果: アンデッド退散*/
#define GF_DISP_EVIL    68			/*!< 魔法効果: 邪悪退散*/
#define GF_DISP_ALL     69			/*!< 魔法効果: モンスター退散*/
/* New types for Zangband begin here... */
#define GF_DISP_DEMON      70		/*!< 魔法効果: 悪魔退散*/
#define GF_DISP_LIVING     71		/*!< 魔法効果: 生命退散*/
#define GF_ROCKET          72		/*!< 魔法効果: ロケット*/
#define GF_NUKE            73		/*!< 魔法効果: 放射性廃棄物*/
#define GF_MAKE_GLYPH      74		/*!< 魔法効果: 守りのルーン生成*/
#define GF_STASIS          75		/*!< 魔法効果: モンスター拘束*/
#define GF_STONE_WALL      76		/*!< 魔法効果: 壁生成*/
#define GF_DEATH_RAY       77		/*!< 魔法効果: 死の光線*/
#define GF_STUN            78		/*!< 魔法効果: 朦朧*/
#define GF_HOLY_FIRE       79		/*!< 魔法効果: 聖光*/
#define GF_HELL_FIRE       80		/*!< 魔法効果: 地獄の劫火*/
#define GF_DISINTEGRATE    81		/*!< 魔法効果: 分解*/
#define GF_CHARM           82		/*!< 魔法効果: モンスター魅了*/
#define GF_CONTROL_UNDEAD  83		/*!< 魔法効果: アンデッド支配*/
#define GF_CONTROL_ANIMAL  84		/*!< 魔法効果: 動物支配*/
#define GF_PSI             85		/*!< 魔法効果: サイキック攻撃*/
#define GF_PSI_DRAIN       86		/*!< 魔法効果: 精神吸収*/
#define GF_TELEKINESIS     87		/*!< 魔法効果: テレキシネス*/
#define GF_JAM_DOOR        88		/*!< 魔法効果: 施錠*/
#define GF_DOMINATION      89		/*!< 魔法効果: 精神支配*/
#define GF_DISP_GOOD       90		/*!< 魔法効果: 善良退散*/
#define GF_DRAIN_MANA      91		/*!< 魔法効果: 魔力吸収*/
#define GF_MIND_BLAST      92		/*!< 魔法効果: 精神攻撃*/
#define GF_BRAIN_SMASH     93		/*!< 魔法効果: 脳攻撃*/
#define GF_CAUSE_1         94		/*!< 魔法効果: 軽傷の呪い*/
#define GF_CAUSE_2         95		/*!< 魔法効果: 重傷の呪い*/
#define GF_CAUSE_3         96		/*!< 魔法効果: 致命傷の呪い*/
#define GF_CAUSE_4         97		/*!< 魔法効果: 秘孔を突く*/
#define GF_HAND_DOOM       98		/*!< 魔法効果: 破滅の手*/
#define GF_CAPTURE         99		/*!< 魔法効果: 捕縛*/
#define GF_ANIM_DEAD      100		/*!< 魔法効果: 死者復活*/
#define GF_CHARM_LIVING   101		/*!< 魔法効果: 生命魅了*/
#define GF_IDENTIFY       102		/*!< 魔法効果: 鑑定*/
#define GF_ATTACK         103		/*!< 魔法効果: 白兵*/
#define GF_ENGETSU        104		/*!< 魔法効果: 円月*/
#define GF_GENOCIDE       105		/*!< 魔法効果: 抹殺*/
#define GF_PHOTO          106		/*!< 魔法効果: 撮影*/
#define GF_CONTROL_DEMON  107		/*!< 魔法効果: 悪魔支配*/
#define GF_LAVA_FLOW      108		/*!< 魔法効果: 溶岩噴出*/
#define GF_BLOOD_CURSE    109		/*!< 魔法効果: 血の呪い*/
#define GF_SEEKER         110		/*!< 魔法効果: シーカーレイ*/
#define GF_SUPER_RAY      111		/*!< 魔法効果: スーパーレイ*/
#define GF_STAR_HEAL      112		/*!< 魔法効果: 星の癒し*/
#define GF_WATER_FLOW     113		/*!< 魔法効果: 流水*/
#define GF_CRUSADE        114		/*!< 魔法効果: 聖戦*/
#define GF_STASIS_EVIL    115		/*!< 魔法効果: 邪悪拘束*/
#define GF_WOUNDS         116		/*!< 魔法効果: 創傷*/

#define MAX_GF         117

/*
 * Some things which induce learning
 */
#define DRS_ACID         1
#define DRS_ELEC         2
#define DRS_FIRE         3
#define DRS_COLD         4
#define DRS_POIS         5
#define DRS_NETH         6
#define DRS_LITE         7
#define DRS_DARK         8
#define DRS_FEAR         9
#define DRS_CONF        10
#define DRS_CHAOS       11
#define DRS_DISEN       12
#define DRS_BLIND       13
#define DRS_NEXUS       14
#define DRS_SOUND       15
#define DRS_SHARD       16
#define DRS_FREE        30
#define DRS_MANA        31
#define DRS_REFLECT     32



#define DAMAGE_FORCE    1
#define DAMAGE_GENO     2
#define DAMAGE_LOSELIFE 3
#define DAMAGE_ATTACK   4
#define DAMAGE_NOESCAPE 5
#define DAMAGE_USELIFE  6


/*
 * アイテムの簡易鑑定定義 / Game generated inscription indices. These are stored in the object,
 * and are used to index the string array from tables.c.
 */
#define FEEL_NONE              0 /*!< 簡易鑑定: 未鑑定 */
#define FEEL_BROKEN            1 /*!< 簡易鑑定: 壊れている */
#define FEEL_TERRIBLE          2 /*!< 簡易鑑定: 恐ろしい */
#define FEEL_WORTHLESS         3 /*!< 簡易鑑定: 無価値 */
#define FEEL_CURSED            4 /*!< 簡易鑑定: 呪われている */
#define FEEL_UNCURSED          5 /*!< 簡易鑑定: 呪われていない */
#define FEEL_AVERAGE           6 /*!< 簡易鑑定: 並 */
#define FEEL_GOOD              7 /*!< 簡易鑑定: 上質 */
#define FEEL_EXCELLENT         8 /*!< 簡易鑑定: 高級 */
#define FEEL_SPECIAL           9 /*!< 簡易鑑定: 特別 */

#define FEEL_MAX               9 /*!< 簡易鑑定の種別数 */

/*
 * 変愚ver1.5.0以前に使われていたアイテムの追加特性フラグ / Hack -- special "xtra" object powers
 */
#define EGO_XTRA_SUSTAIN        1 /*!< 旧版アイテムフラグ(非推奨): 追加維持能力 / Sustain one stat */
#define EGO_XTRA_POWER          2 /*!< 旧版アイテムフラグ(非推奨): 追加上級耐性 / High resist */
#define EGO_XTRA_ABILITY        3 /*!< 旧版アイテムフラグ(非推奨): 追加能力 / Special ability */


/*** Object flag values ***/


/*!
 * 箱のトラップ定義 Chest trap flags (see "tables.c")
 */
#define CHEST_LOSE_STR          0x0001 /*!< 箱のトラップ: STR減少の毒針 */
#define CHEST_LOSE_CON          0x0002 /*!< 箱のトラップ: CON減少の毒針 */
#define CHEST_POISON            0x0004 /*!< 箱のトラップ: 毒針 */
#define CHEST_PARALYZE          0x0008 /*!< 箱のトラップ: 麻痺ガス */
#define CHEST_EXPLODE           0x0010 /*!< 箱のトラップ: 爆発 */
#define CHEST_SUMMON            0x0020 /*!< 箱のトラップ: 召喚のルーン(モンスター) */
#define CHEST_SCATTER           0x0040 /*!< 箱のトラップ: アイテム散乱 */
#define CHEST_E_SUMMON          0x0080 /*!< 箱のトラップ: 召喚のルーン(エレメンタル) */
#define CHEST_BIRD_STORM        0x0100 /*!< 箱のトラップ: 召喚のルーン(鳥) */
#define CHEST_H_SUMMON          0x0200 /*!< 箱のトラップ: 召喚のルーン(強敵)  */
#define CHEST_RUNES_OF_EVIL     0x0400 /*!< 箱のトラップ: 邪悪なルーン */
#define CHEST_ALARM             0x0800 /*!< 箱のトラップ: 警報装置 */


/*
 * Special Object Flags
 */
#define IDENT_SENSE     0x01    /* Item has been "sensed" */
#define IDENT_FIXED     0x02    /* Item has been "haggled" */
#define IDENT_EMPTY     0x04    /* Item charges are known */
#define IDENT_KNOWN     0x08    /* Item abilities are known */
#define IDENT_STORE     0x10    /* Item is storebought !!!! */
#define IDENT_MENTAL    0x20    /* Item information is known */
#if 0
#define IDENT_CURSED    0x40    /* Item is temporarily cursed */
#endif
#define IDENT_BROKEN    0x80    /* Item is permanently worthless */


/* 
 * How object is marked (flags in object_type.mark)
 * OM_FOUND --- original boolean flag
 * OM_NOMSG --- temporary flag to suppress messages which were
 *              already printed in autopick_pickup_items().
 */
#define OM_FOUND        0x01    /* original boolean flag */
#define OM_NOMSG        0x02    /* temporary flag to suppress messages */
#define OM_NO_QUERY     0x04    /* Query for auto-pick was already answered as 'No' */
#define OM_AUTODESTROY  0x08    /* Destroy later to avoid illegal inventry shift */
#define OM_TOUCHED      0x10    /* Object was touched by player */


/*
 * Special Monster Flags (all temporary)
 */
#define MFLAG_VIEW      0x01    /* Monster is in line of sight */
#define MFLAG_TEMP      0x02    /* Monster is marked for project_hack() */
#define MFLAG_XXX2      0x04    /* (unused) */
#define MFLAG_XXX3      0x08    /* (unused) */
#define MFLAG_BORN      0x10    /* Monster is still being born */
#define MFLAG_NICE      0x20    /* Monster is still being nice */

#define MFLAG2_KAGE      0x01    /* Monster is kage */
#define MFLAG2_NOPET     0x02    /* Cannot make monster pet */
#define MFLAG2_NOGENO    0x04    /* Cannot genocide */
#define MFLAG2_CHAMELEON 0x08    /* Monster is chameleon */
#define MFLAG2_NOFLOW    0x10    /* Monster is in no_flow_by_smell mode */
#define MFLAG2_SHOW      0x20    /* Monster is recently memorized */
#define MFLAG2_MARK      0x40    /* Monster is currently memorized */


/*
 * Object flags
 *
 * Old variables for object flags such as flags1, flags2, and flags3
 * are obsolated.  Now single array flgs[TR_FLAG_SIZE] contains all
 * object flags.  And each flag is refered by single index number
 * instead of a bit mask.
 *
 * Therefore it's very easy to add a lot of new flags; no one need to
 * worry about in which variable a new flag should be put, nor to
 * modify a huge number of files all over the source directory at once
 * to add new flag variables such as flags4, a_ability_flags1, etc...
 *
 * All management of flags is now treated using a set of macros
 * instead of bit operations.
 * Note: These macros are using division, modulo, and bit shift
 * operations, and it seems that these operations are rather slower
 * than original bit operation.  But since index numbers are almost
 * always given as constant, such slow operations are performed in the
 * compile time.  So there is no problem on the speed.
 *
 * Exceptions of new flag management is a set of flags to control
 * object generation and the curse flags.  These are not yet rewritten
 * in new index form; maybe these have no merit of rewriting.
 */

#define have_flag(ARRAY, INDEX) !!((ARRAY)[(INDEX)/32] & (1L << ((INDEX)%32)))
#define add_flag(ARRAY, INDEX) ((ARRAY)[(INDEX)/32] |= (1L << ((INDEX)%32)))
#define remove_flag(ARRAY, INDEX) ((ARRAY)[(INDEX)/32] &= ~(1L << ((INDEX)%32)))
#define is_pval_flag(INDEX) ((TR_STR <= (INDEX) && (INDEX) <= TR_MAGIC_MASTERY) || (TR_STEALTH <= (INDEX) && (INDEX) <= TR_BLOWS))
#define have_pval_flags(ARRAY) !!((ARRAY)[0] & (0x00003f7f))


#define TR_STR                 0      /* STR += "pval" */
#define TR_INT                 1      /* INT += "pval" */
#define TR_WIS                 2      /* WIS += "pval" */
#define TR_DEX                 3      /* DEX += "pval" */
#define TR_CON                 4      /* CON += "pval" */
#define TR_CHR                 5      /* CHR += "pval" */
#define TR_MAGIC_MASTERY       6      /* Later */
#define TR_FORCE_WEAPON        7      /* Later */
#define TR_STEALTH             8      /* Stealth += "pval" */
#define TR_SEARCH              9      /* Search += "pval" */
#define TR_INFRA               10     /* Infra += "pval" */
#define TR_TUNNEL              11     /* Tunnel += "pval" */
#define TR_SPEED               12     /* Speed += "pval" */
#define TR_BLOWS               13     /* Blows += "pval" */
#define TR_CHAOTIC             14
#define TR_VAMPIRIC            15
#define TR_SLAY_ANIMAL         16
#define TR_SLAY_EVIL           17
#define TR_SLAY_UNDEAD         18
#define TR_SLAY_DEMON          19
#define TR_SLAY_ORC            20
#define TR_SLAY_TROLL          21
#define TR_SLAY_GIANT          22
#define TR_SLAY_DRAGON         23
#define TR_KILL_DRAGON         24     /* Execute Dragon */
#define TR_VORPAL              25     /* Later */
#define TR_IMPACT              26     /* Cause Earthquakes */
#define TR_BRAND_POIS          27
#define TR_BRAND_ACID          28
#define TR_BRAND_ELEC          29
#define TR_BRAND_FIRE          30
#define TR_BRAND_COLD          31

#define TR_SUST_STR            32
#define TR_SUST_INT            33
#define TR_SUST_WIS            34
#define TR_SUST_DEX            35
#define TR_SUST_CON            36
#define TR_SUST_CHR            37
#define TR_RIDING              38
#define TR_EASY_SPELL          39
#define TR_IM_ACID             40
#define TR_IM_ELEC             41
#define TR_IM_FIRE             42
#define TR_IM_COLD             43
#define TR_THROW               44     /* Later */
#define TR_REFLECT             45     /* Reflect 'bolts' */
#define TR_FREE_ACT            46     /* Free Action */
#define TR_HOLD_EXP            47     /* Hold EXP */
#define TR_RES_ACID            48
#define TR_RES_ELEC            49
#define TR_RES_FIRE            50
#define TR_RES_COLD            51
#define TR_RES_POIS            52
#define TR_RES_FEAR            53     /* Added for Zangband */
#define TR_RES_LITE            54
#define TR_RES_DARK            55
#define TR_RES_BLIND           56
#define TR_RES_CONF            57
#define TR_RES_SOUND           58
#define TR_RES_SHARDS          59
#define TR_RES_NETHER          60
#define TR_RES_NEXUS           61
#define TR_RES_CHAOS           62
#define TR_RES_DISEN           63

#define TR_SH_FIRE             64     /* Immolation (Fire) */
#define TR_SH_ELEC             65     /* Electric Sheath */
#define TR_SLAY_HUMAN          66     /* Slay human */
#define TR_SH_COLD             67     /* cold aura */
#define TR_NO_TELE             68     /* Anti-teleportation */
#define TR_NO_MAGIC            69     /* Anti-magic */
#define TR_DEC_MANA            70     /* ??? */
#define TR_TY_CURSE            71     /* The Ancient Curse */
#define TR_WARNING             72     /* Warning */
#define TR_HIDE_TYPE           73     /* Hide "pval" description */
#define TR_SHOW_MODS           74     /* Always show Tohit/Todam */
#define TR_SLAY_GOOD           75
#define TR_LEVITATION          76     /* Feather Falling */
#define TR_LITE_1              77     /* Light Radius 1*/
#define TR_SEE_INVIS           78     /* See Invisible */
#define TR_TELEPATHY           79     /* Telepathy */
#define TR_SLOW_DIGEST         80     /* Item slows down digestion */
#define TR_REGEN               81     /* Item induces regeneration */
#define TR_XTRA_MIGHT          82     /* Bows get extra multiplier */
#define TR_XTRA_SHOTS          83     /* Bows get extra shots */
#define TR_IGNORE_ACID         84     /* Item ignores Acid Damage */
#define TR_IGNORE_ELEC         85     /* Item ignores Elec Damage */
#define TR_IGNORE_FIRE         86     /* Item ignores Fire Damage */
#define TR_IGNORE_COLD         87     /* Item ignores Cold Damage */
#define TR_ACTIVATE            88     /* Item can be activated */
#define TR_DRAIN_EXP           89     /* Item drains Experience */
#define TR_TELEPORT            90     /* Item teleports player */
#define TR_AGGRAVATE           91     /* Item aggravates monsters */
#define TR_BLESSED             92     /* Item is Blessed */
#define TR_ES_ATTACK           93     /* Fake flag for Smith */
#define TR_ES_AC               94     /* Fake flag for Smith */
#define TR_KILL_GOOD           95

#define TR_KILL_ANIMAL         96
#define TR_KILL_EVIL           97
#define TR_KILL_UNDEAD         98
#define TR_KILL_DEMON          99
#define TR_KILL_ORC            100
#define TR_KILL_TROLL          101
#define TR_KILL_GIANT          102
#define TR_KILL_HUMAN          103
#define TR_ESP_ANIMAL          104
#define TR_ESP_UNDEAD          105
#define TR_ESP_DEMON           106
#define TR_ESP_ORC             107
#define TR_ESP_TROLL           108
#define TR_ESP_GIANT           109
#define TR_ESP_DRAGON          110
#define TR_ESP_HUMAN           111
#define TR_ESP_EVIL            112
#define TR_ESP_GOOD            113
#define TR_ESP_NONLIVING       114
#define TR_ESP_UNIQUE          115
#define TR_FULL_NAME           116
#define TR_FIXED_FLAVOR        117
#define TR_ADD_L_CURSE         118
#define TR_ADD_H_CURSE         119
#define TR_DRAIN_HP            120
#define TR_DRAIN_MANA          121
#define TR_LITE_2			   122
#define TR_LITE_3			   123
#define TR_LITE_M1			   124    /* Permanent decrease Light Area (-1) */
#define TR_LITE_M2			   125    /* Permanent decrease Light Area (-1) */
#define TR_LITE_M3			   126    /* Permanent decrease Light Area (-1) */
#define TR_LITE_FUEL		   127	  /* Lights need Fuels */

#define TR_CALL_ANIMAL         128
#define TR_CALL_DEMON          129
#define TR_CALL_DRAGON         130
#define TR_CALL_UNDEAD         131
#define TR_COWARDICE           132
#define TR_LOW_MELEE           133
#define TR_LOW_AC              134
#define TR_LOW_MAGIC           135
#define TR_FAST_DIGEST         136
#define TR_SLOW_REGEN          137

#define TR_FLAG_MAX            138
#define TR_FLAG_SIZE           5


#define TRG_INSTA_ART           0x00000001L     /* Item must be an artifact */
#define TRG_QUESTITEM           0x00000002L     /* quest level item -KMW- */
#define TRG_XTRA_POWER          0x00000004L     /* Extra power */
#define TRG_ONE_SUSTAIN         0x00000008L     /* One sustain */
#define TRG_XTRA_RES_OR_POWER   0x00000010L     /* Extra resistance or power */
#define TRG_XTRA_H_RES          0x00000020L     /* Extra high resistance */
#define TRG_XTRA_E_RES          0x00000040L     /* Extra element resistance */
#define TRG_XTRA_L_RES          0x00000080L     /* Extra lordly resistance */
#define TRG_XTRA_D_RES          0x00000100L     /* Extra dragon resistance */
#define TRG_XTRA_RES            0x00000200L     /* Extra resistance */
#define TRG_CURSED              0x00000400L     /* Item is Cursed */
#define TRG_HEAVY_CURSE         0x00000800L     /* Item is Heavily Cursed */
#define TRG_PERMA_CURSE         0x00001000L     /* Item is Perma Cursed */
#define TRG_RANDOM_CURSE0       0x00002000L     /* Item is Random Cursed */
#define TRG_RANDOM_CURSE1       0x00004000L     /* Item is Random Cursed */
#define TRG_RANDOM_CURSE2       0x00008000L     /* Item is Random Cursed */
#define TRG_XTRA_DICE           0x00010000L     /* Extra dice */
#define TRG_POWERFUL            0x00020000L     /* Item has good value even if Cursed */


#define MAX_CURSE 18

#define TRC_CURSED              0x00000001L
#define TRC_HEAVY_CURSE         0x00000002L
#define TRC_PERMA_CURSE         0x00000004L
#define TRC_XXX1                0x00000008L
#define TRC_TY_CURSE            0x00000010L
#define TRC_AGGRAVATE           0x00000020L
#define TRC_DRAIN_EXP           0x00000040L
#define TRC_SLOW_REGEN          0x00000080L
#define TRC_ADD_L_CURSE         0x00000100L
#define TRC_ADD_H_CURSE         0x00000200L
#define TRC_CALL_ANIMAL         0x00000400L
#define TRC_CALL_DEMON          0x00000800L
#define TRC_CALL_DRAGON         0x00001000L
#define TRC_COWARDICE           0x00002000L
#define TRC_TELEPORT            0x00004000L
#define TRC_LOW_MELEE           0x00008000L
#define TRC_LOW_AC              0x00010000L
#define TRC_LOW_MAGIC           0x00020000L
#define TRC_FAST_DIGEST         0x00040000L
#define TRC_DRAIN_HP            0x00080000L
#define TRC_DRAIN_MANA          0x00100000L
#define TRC_CALL_UNDEAD         0x00200000L

#define TRC_TELEPORT_SELF       0x00000001L
#define TRC_CHAINSWORD          0x00000002L

#define TRC_SPECIAL_MASK \
	(TRC_TY_CURSE | TRC_AGGRAVATE)

#define TRC_HEAVY_MASK   \
	(TRC_TY_CURSE | TRC_AGGRAVATE | TRC_DRAIN_EXP | TRC_ADD_H_CURSE | \
	 TRC_CALL_DEMON | TRC_CALL_DRAGON | TRC_CALL_UNDEAD | TRC_TELEPORT)

#define TRC_P_FLAG_MASK  \
	(TRC_TELEPORT_SELF | TRC_CHAINSWORD | \
	 TRC_TY_CURSE | TRC_DRAIN_EXP | TRC_ADD_L_CURSE | TRC_ADD_H_CURSE | \
	 TRC_CALL_ANIMAL | TRC_CALL_DEMON | TRC_CALL_DRAGON | TRC_COWARDICE | \
	 TRC_TELEPORT | TRC_DRAIN_HP | TRC_DRAIN_MANA | TRC_CALL_UNDEAD)



/*
 * Bit flags for apply_magic() (etc)
 */
#define AM_NO_FIXED_ART 0x00000001 /* Don't allow roll for fixed artifacts */
#define AM_GOOD         0x00000002 /* Generate good items */
#define AM_GREAT        0x00000004 /* Generate great items */
#define AM_SPECIAL      0x00000008 /* Generate artifacts (for debug mode only) */
#define AM_CURSED       0x00000010 /* Generate cursed/worthless items */


/*** Monster blow constants ***/


/*!
 * @note モンスターの打撃方法 / New monster blow methods
 * 打撃の種別に応じて傷と朦朧が発生するかがコメントの通りに決まる
 */
#define RBM_HIT          1 /*!< モンスターの攻撃種別:殴る(傷/朦朧が半々) */
#define RBM_TOUCH        2 /*!< モンスターの攻撃種別:触る */
#define RBM_PUNCH        3 /*!< モンスターの攻撃種別:パンチする(朦朧) */
#define RBM_KICK         4 /*!< モンスターの攻撃種別:蹴る(朦朧) */
#define RBM_CLAW         5 /*!< モンスターの攻撃種別:ひっかく(傷) */
#define RBM_BITE         6 /*!< モンスターの攻撃種別:噛む(傷) */
#define RBM_STING        7 /*!< モンスターの攻撃種別:刺す */
#define RBM_SLASH        8 /*!< モンスターの攻撃種別:斬る(傷) */
#define RBM_BUTT         9 /*!< モンスターの攻撃種別:角で突く(朦朧) */
#define RBM_CRUSH       10 /*!< モンスターの攻撃種別:体当たりする(朦朧) */
#define RBM_ENGULF      11 /*!< モンスターの攻撃種別:飲み込む */
#define RBM_CHARGE      12 /*!< モンスターの攻撃種別:請求書を寄越す */
#define RBM_CRAWL       13 /*!< モンスターの攻撃種別:体の上を這い回る */
#define RBM_DROOL       14 /*!< モンスターの攻撃種別:よだれをたらす */
#define RBM_SPIT        15 /*!< モンスターの攻撃種別:つばを吐く */
#define RBM_EXPLODE     16 /*!< モンスターの攻撃種別:爆発する */
#define RBM_GAZE        17 /*!< モンスターの攻撃種別:にらむ */
#define RBM_WAIL        18 /*!< モンスターの攻撃種別:泣き叫ぶ */
#define RBM_SPORE       19 /*!< モンスターの攻撃種別:胞子を飛ばす */
#define RBM_XXX4        20 /*!< モンスターの攻撃種別:未定義 */
#define RBM_BEG         21 /*!< モンスターの攻撃種別:金をせがむ */
#define RBM_INSULT      22 /*!< モンスターの攻撃種別:侮辱する */
#define RBM_MOAN        23 /*!< モンスターの攻撃種別:うめく */
#define RBM_SHOW        24 /*!< モンスターの攻撃種別:歌う */
#define RBM_SHOOT       25 /*!< モンスターの攻撃種別:射撃(非打撃) */


/*!
 * @note モンスターの攻撃効果 / New monster blow effects
 */
#define RBE_HURT         1 /*!< モンスターの攻撃効果: 攻撃する*/
#define RBE_POISON       2 /*!< モンスターの攻撃効果: 毒をくらわす*/
#define RBE_UN_BONUS     3 /*!< モンスターの攻撃効果: 劣化させる*/
#define RBE_UN_POWER     4 /*!< モンスターの攻撃効果: 充填魔力を吸収する*/
#define RBE_EAT_GOLD     5 /*!< モンスターの攻撃効果: 金を盗む*/
#define RBE_EAT_ITEM     6 /*!< モンスターの攻撃効果: アイテムを盗む*/
#define RBE_EAT_FOOD     7 /*!< モンスターの攻撃効果: あなたの食糧を食べる*/
#define RBE_EAT_LITE     8 /*!< モンスターの攻撃効果: 明かりを吸収する*/
#define RBE_ACID         9 /*!< モンスターの攻撃効果: 酸を飛ばす*/
#define RBE_ELEC        10 /*!< モンスターの攻撃効果: 感電させる*/
#define RBE_FIRE        11 /*!< モンスターの攻撃効果: 燃やす*/
#define RBE_COLD        12 /*!< モンスターの攻撃効果: 凍らせる*/
#define RBE_BLIND       13 /*!< モンスターの攻撃効果: 盲目にする*/
#define RBE_CONFUSE     14 /*!< モンスターの攻撃効果: 混乱させる*/
#define RBE_TERRIFY     15 /*!< モンスターの攻撃効果: 恐怖させる*/
#define RBE_PARALYZE    16 /*!< モンスターの攻撃効果: 麻痺させる*/
#define RBE_LOSE_STR    17 /*!< モンスターの攻撃効果: 腕力を減少させる*/
#define RBE_LOSE_INT    18 /*!< モンスターの攻撃効果: 知能を減少させる*/
#define RBE_LOSE_WIS    19 /*!< モンスターの攻撃効果: 賢さを減少させる*/
#define RBE_LOSE_DEX    20 /*!< モンスターの攻撃効果: 器用さを減少させる*/
#define RBE_LOSE_CON    21 /*!< モンスターの攻撃効果: 耐久力を減少させる*/
#define RBE_LOSE_CHR    22 /*!< モンスターの攻撃効果: 魅力を減少させる*/
#define RBE_LOSE_ALL    23 /*!< モンスターの攻撃効果: 全ステータスを減少させる*/
#define RBE_SHATTER     24 /*!< モンスターの攻撃効果: 粉砕する*/
#define RBE_EXP_10      25 /*!< モンスターの攻撃効果: 経験値を減少(10d6+)させる*/
#define RBE_EXP_20      26 /*!< モンスターの攻撃効果: 経験値を減少(20d6+)させる*/
#define RBE_EXP_40      27 /*!< モンスターの攻撃効果: 経験値を減少(40d6+)させる*/
#define RBE_EXP_80      28 /*!< モンスターの攻撃効果: 経験値を減少(80d6+)させる*/
#define RBE_DISEASE     29 /*!< モンスターの攻撃効果: 病気にする*/
#define RBE_TIME        30 /*!< モンスターの攻撃効果: 時間を逆戻りさせる*/
#define RBE_DR_LIFE     31 /*!< モンスターの攻撃効果: 生命力を吸収する*/
#define RBE_DR_MANA     32 /*!< モンスターの攻撃効果: 魔力を奪う*/
#define RBE_SUPERHURT   33 /*!< モンスターの攻撃効果: 強力に攻撃する*/
#define RBE_INERTIA     34 /*!< モンスターの攻撃効果: 減速させる*/
#define RBE_STUN        35 /*!< モンスターの攻撃効果: 朦朧とさせる*/


/*** Monster flag values (hard-coded) ***/


/*
 * New monster race bit flags
 */
#define RF1_UNIQUE              0x00000001  /*!< モンスター特性: ユニーク / Unique Monster */
#define RF1_QUESTOR             0x00000002  /*!< モンスター特性: クエストモンスター / Quest Monster */
#define RF1_MALE                0x00000004  /*!< モンスター特性: 男性 / Male gender */
#define RF1_FEMALE              0x00000008  /*!< モンスター特性: 女性 / Female gender */
#define RF1_CHAR_CLEAR          0x00000010  /*!< モンスター特性: シンボルが完全に透明 / Absorbs symbol */
#define RF1_SHAPECHANGER        0x00000020  /*!< モンスター特性: シンボルアルファベットがランダムになる / TY: shapechanger */
#define RF1_ATTR_CLEAR          0x00000040  /*!< モンスター特性: シンボルカラーが透明色になる(地形と同じ色になる) / Absorbs color */
#define RF1_ATTR_MULTI          0x00000080  /*!< モンスター特性: シンボルカラーがランダムに変化する(基本7色) / Changes color */
#define RF1_FORCE_DEPTH         0x00000100  /*!< モンスター特性: 指定階未満では生成されない / Start at "correct" depth */
#define RF1_FORCE_MAXHP         0x00000200  /*!< モンスター特性: 通常生成時必ずHPがダイス最大値になる / Start with max hitpoints */
#define RF1_FORCE_SLEEP         0x00000400  /*!< モンスター特性: 通常生成時必ず寝ている / Start out sleeping */
#define RF1_FORCE_EXTRA         0x00000800  /*!< モンスター特性: (未使用) / Start out something */
#define RF1_ATTR_SEMIRAND       0x00001000  /*!< モンスター特性: シンボルカラーがランダムに変化する(15色) / Color is determined semi-randomly */
#define RF1_FRIENDS             0x00002000  /*!< モンスター特性: 同種の友軍を用意している / Arrive with some friends */
#define RF1_ESCORT              0x00004000  /*!< モンスター特性: 護衛を用意している/ Arrive with an escort */
#define RF1_ESCORTS             0x00008000  /*!< モンスター特性: さらに大量の護衛を用意している / Arrive with some escorts */
#define RF1_NEVER_BLOW          0x00010000  /*!< モンスター特性: 打撃を一切行わない / Never make physical blow */
#define RF1_NEVER_MOVE          0x00020000  /*!< モンスター特性: 移動を一切行わない / Never make physical move */
#define RF1_RAND_25             0x00040000  /*!< モンスター特性: ランダムに移動する確率+25%/ Moves randomly (25%) */
#define RF1_RAND_50             0x00080000  /*!< モンスター特性: ランダムに移動する確率+50%/ Moves randomly (50%) */
#define RF1_ONLY_GOLD           0x00100000  /*!< モンスター特性: 財宝しか落とさない / Drop only gold */
#define RF1_ONLY_ITEM           0x00200000  /*!< モンスター特性: アイテムしか落とさない / Drop only items */
#define RF1_DROP_60             0x00400000  /*!< モンスター特性: 落とすアイテム数60%で+1/ Drop an item/gold (60%) */
#define RF1_DROP_90             0x00800000  /*!< モンスター特性: 落とすアイテム数90%で+1 / Drop an item/gold (90%) */
#define RF1_DROP_1D2            0x01000000  /*!< モンスター特性: 落とすアイテム数+1d2 / Drop 1d2 items/gold */
#define RF1_DROP_2D2            0x02000000  /*!< モンスター特性: 落とすアイテム数+2d2 / Drop 2d2 items/gold */
#define RF1_DROP_3D2            0x04000000  /*!< モンスター特性: 落とすアイテム数+3d2 / Drop 3d2 items/gold */
#define RF1_DROP_4D2            0x08000000  /*!< モンスター特性: 落とすアイテム数+4d2 / Drop 4d2 items/gold */
#define RF1_DROP_GOOD           0x10000000  /*!< モンスター特性: 必ず上質品をドロップする / Drop good items */
#define RF1_DROP_GREAT          0x20000000  /*!< モンスター特性: 必ず高級品をドロップする / Drop great items */
#define RF1_XXX2                0x40000000  /*!< モンスター特性: 未使用 / XXX */
#define RF1_XXX3                0x80000000  /*!< モンスター特性: 未使用 / XXX */

/*
 * New monster race bit flags
 */
#define RF2_STUPID          0x00000001  /*!< モンスター特性: 愚かな行動を取る / Monster is stupid */
#define RF2_SMART           0x00000002  /*!< モンスター特性: 賢い行動を取る / Monster is smart */
#define RF2_CAN_SPEAK       0x00000004  /*!< モンスター特性: 台詞をしゃべる / TY: can speak */
#define RF2_REFLECTING      0x00000008  /*!< モンスター特性: 矢やボルト魔法を反射する / Reflects bolts */
#define RF2_INVISIBLE       0x00000010  /*!< モンスター特性: 透明視力がないと見えない / Monster avoids vision */
#define RF2_COLD_BLOOD      0x00000020  /*!< モンスター特性: 冷血動物である / Monster avoids infra */
#define RF2_EMPTY_MIND      0x00000040  /*!< モンスター特性: 知性を持たない(テレパシー回避) / Monster avoids telepathy */
#define RF2_WEIRD_MIND      0x00000080  /*!< モンスター特性: 異質な知性(テレパシーで感知づらい) / Monster avoids telepathy? */
#define RF2_MULTIPLY        0x00000100  /*!< モンスター特性: 増殖する / Monster reproduces */
#define RF2_REGENERATE      0x00000200  /*!< モンスター特性: 急激に回復する / Monster regenerates */
#define RF2_CHAR_MULTI      0x00000400  /*!< モンスター特性: 未使用 / (Not implemented) */
#define RF2_ATTR_ANY        0x00000800  /*!< モンスター特性: ATTR_MULTIの色数が増える / TY: Attr_any */
#define RF2_POWERFUL        0x00001000  /*!< モンスター特性: 強力に魔法をあやつる / Monster has strong breath */
#define RF2_ELDRITCH_HORROR 0x00002000  /*!< モンスター特性: 狂気を呼び起こす / Sanity-blasting horror    */
#define RF2_AURA_FIRE       0x00004000  /*!< モンスター特性: 火炎のオーラを持つ / Burns in melee */
#define RF2_AURA_ELEC       0x00008000  /*!< モンスター特性: 電撃のオーラを持つ / Shocks in melee */
#define RF2_OPEN_DOOR       0x00010000  /*!< モンスター特性: ドアを開けることができる / Monster can open doors */
#define RF2_BASH_DOOR       0x00020000  /*!< モンスター特性: ドアを破壊することができる / Monster can bash doors */
#define RF2_PASS_WALL       0x00040000  /*!< モンスター特性: 壁を抜けることができる / Monster can pass walls */
#define RF2_KILL_WALL       0x00080000  /*!< モンスター特性: 壁を破壊して進む / Monster can destroy walls */
#define RF2_MOVE_BODY       0x00100000  /*!< モンスター特性: 道中の弱いモンスターを押しのけることができる / Monster can move monsters */
#define RF2_KILL_BODY       0x00200000  /*!< モンスター特性: 道中の弱いモンスターを殺して進む / Monster can kill monsters */
#define RF2_TAKE_ITEM       0x00400000  /*!< モンスター特性: 道中のアイテムを拾う / Monster can pick up items */
#define RF2_KILL_ITEM       0x00800000  /*!< モンスター特性: 道中のアイテムを破壊する / Monster can crush items */
#define RF2_XXX1            0x01000000  /*!< モンスター特性: 未使用 / XXX */
#define RF2_XXX2            0x02000000  /*!< モンスター特性: 未使用 / XXX */
#define RF2_XXX3            0x04000000  /*!< モンスター特性: 未使用 / XXX */
#define RF2_XXX4            0x08000000  /*!< モンスター特性: 未使用 / XXX */
#define RF2_XXX5            0x10000000  /*!< モンスター特性: 未使用 / XXX */
#define RF2_XXX6            0x20000000  /*!< モンスター特性: 未使用 / XXX */
#define RF2_HUMAN           0x40000000  /*!< モンスター特性: 人間 / Human */
#define RF2_QUANTUM         0x80000000  /*!< モンスター特性: 量子的な振る舞いをする / Monster has quantum behavior */

/*
 * New monster race bit flags
 */
#define RF3_ORC             0x00000001  /*!< モンスター特性: オーク / Orc */
#define RF3_TROLL           0x00000002  /*!< モンスター特性: トロル / Troll */
#define RF3_GIANT           0x00000004  /*!< モンスター特性: 巨人 / Giant */
#define RF3_DRAGON          0x00000008  /*!< モンスター特性: ドラゴン / Dragon */
#define RF3_DEMON           0x00000010  /*!< モンスター特性: 悪魔 / Demon */
#define RF3_UNDEAD          0x00000020  /*!< モンスター特性: アンデッド / Undead */
#define RF3_EVIL            0x00000040  /*!< モンスター特性: 邪悪 / Evil */
#define RF3_ANIMAL          0x00000080  /*!< モンスター特性: 動物 / Animal */
#define RF3_AMBERITE        0x00000100  /*!< モンスター特性: アンバーの血族 / TY: Amberite */
#define RF3_GOOD            0x00000200  /*!< モンスター特性: 善良 / Good */
#define RF3_AURA_COLD       0x00000400  /*!< モンスター特性: 冷気オーラ / Freezes in melee */
#define RF3_NONLIVING       0x00000800  /*!< モンスター特性: 無生物 / TY: Non-Living (?) */
#define RF3_HURT_LITE       0x00001000  /*!< モンスター特性: 通常の光(GF_WEAK_LITE)でダメージを受ける / Hurt by lite */
#define RF3_HURT_ROCK       0x00002000  /*!< モンスター特性: 岩石溶解(GF_KILL_WALL)でダメージを受ける / Hurt by rock remover */
#define RF3_HURT_FIRE       0x00004000  /*!< モンスター特性: 火炎が弱点 / Hurt badly by fire */
#define RF3_HURT_COLD       0x00008000  /*!< モンスター特性: 冷気が弱点 / Hurt badly by cold */
#define RF3_XXX16           0x00010000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX17           0x00020000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX18           0x00040000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX19           0x00080000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX20           0x00100000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX21           0x00200000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX22           0x00400000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX23           0x00800000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX24           0x01000000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX25           0x02000000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX26           0x04000000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_XXX27           0x08000000  /*!< モンスター特性: 未使用 / XXX */
#define RF3_NO_FEAR         0x10000000  /*!< モンスター特性: 恐怖しない / Cannot be scared */
#define RF3_NO_STUN         0x20000000  /*!< モンスター特性: 朦朧としない / Cannot be stunned */
#define RF3_NO_CONF         0x40000000  /*!< モンスター特性: 混乱しない / Cannot be confused and resist confusion */
#define RF3_NO_SLEEP        0x80000000  /*!< モンスター特性: 眠らない / Cannot be slept */

/*
 * New monster race bit flags
 */
#define RF4_SHRIEK          0x00000001  /*!< モンスター能力: 叫ぶ / Shriek for help */
#define RF4_XXX1            0x00000002  /*!< モンスター能力: 未使用 / XXX */
#define RF4_DISPEL          0x00000004  /*!< モンスター能力: 魔力消去 / Dispel magic */
#define RF4_ROCKET          0x00000008  /*!< モンスター能力: ロケット / TY: Rocket */
#define RF4_SHOOT           0x00000010  /*!< モンスター能力: 射撃/ Fire missiles */
#define RF4_XXX2            0x00000020  /*!< モンスター能力: 未使用 / XXX */
#define RF4_XXX3            0x00000040  /*!< モンスター能力: 未使用 / XXX */
#define RF4_XXX4            0x00000080  /*!< モンスター能力: 未使用 / XXX */
#define RF4_BR_ACID         0x00000100  /*!< モンスター能力: 酸のブレス / Breathe Acid */
#define RF4_BR_ELEC         0x00000200  /*!< モンスター能力: 電撃のブレス / Breathe Elec */
#define RF4_BR_FIRE         0x00000400  /*!< モンスター能力: 火炎のブレス / Breathe Fire */
#define RF4_BR_COLD         0x00000800  /*!< モンスター能力: 冷気のブレス / Breathe Cold */
#define RF4_BR_POIS         0x00001000  /*!< モンスター能力: 毒のブレス / Breathe Poison */
#define RF4_BR_NETH         0x00002000  /*!< モンスター能力: 地獄のブレス / Breathe Nether */
#define RF4_BR_LITE         0x00004000  /*!< モンスター能力: 閃光のブレス / Breathe Lite */
#define RF4_BR_DARK         0x00008000  /*!< モンスター能力: 暗黒のブレス / Breathe Dark */
#define RF4_BR_CONF         0x00010000  /*!< モンスター能力: 混乱のブレス / Breathe Confusion */
#define RF4_BR_SOUN         0x00020000  /*!< モンスター能力: 轟音のブレス / Breathe Sound */
#define RF4_BR_CHAO         0x00040000  /*!< モンスター能力: カオスのブレス / Breathe Chaos */
#define RF4_BR_DISE         0x00080000  /*!< モンスター能力: 劣化のブレス / Breathe Disenchant */
#define RF4_BR_NEXU         0x00100000  /*!< モンスター能力: 因果混乱のブレス / Breathe Nexus */
#define RF4_BR_TIME         0x00200000  /*!< モンスター能力: 時間逆転のブレス / Breathe Time */
#define RF4_BR_INER         0x00400000  /*!< モンスター能力: 遅鈍のブレス / Breathe Inertia */
#define RF4_BR_GRAV         0x00800000  /*!< モンスター能力: 重力のブレス / Breathe Gravity */
#define RF4_BR_SHAR         0x01000000  /*!< モンスター能力: 破片のブレス / Breathe Shards */
#define RF4_BR_PLAS         0x02000000  /*!< モンスター能力: プラズマのブレス / Breathe Plasma */
#define RF4_BR_WALL         0x04000000  /*!< モンスター能力: フォースのブレス / Breathe Force */
#define RF4_BR_MANA         0x08000000  /*!< モンスター能力: 魔力のブレス / Breathe Mana */
#define RF4_BA_NUKE         0x10000000  /*!< モンスター能力: 放射能球 / TY: Nuke Ball */
#define RF4_BR_NUKE         0x20000000  /*!< モンスター能力: 放射性廃棄物のブレス / TY: Toxic Breath */
#define RF4_BA_CHAO         0x40000000  /*!< モンスター能力: ログルス球 / TY: Logrus Ball */
#define RF4_BR_DISI         0x80000000  /*!< モンスター能力: 分解のブレス / Breathe Disintegration */

/*
 * New monster race bit flags
 */
#define RF5_BA_ACID         0x00000001  /*!< モンスター能力: アシッド・ボール / Acid Ball */
#define RF5_BA_ELEC         0x00000002  /*!< モンスター能力: サンダー・ボール / Elec Ball */
#define RF5_BA_FIRE         0x00000004  /*!< モンスター能力: ファイア・ボール / Fire Ball */
#define RF5_BA_COLD         0x00000008  /*!< モンスター能力: アイス・ボール / Cold Ball */
#define RF5_BA_POIS         0x00000010  /*!< モンスター能力: 悪臭雲 / Poison Ball */
#define RF5_BA_NETH         0x00000020  /*!< モンスター能力: 地獄球 / Nether Ball */
#define RF5_BA_WATE         0x00000040  /*!< モンスター能力: ウォーター・ボール / Water Ball */
#define RF5_BA_MANA         0x00000080  /*!< モンスター能力: 魔力の嵐 / Mana Storm */
#define RF5_BA_DARK         0x00000100  /*!< モンスター能力: 暗黒の嵐 / Darkness Storm */
#define RF5_DRAIN_MANA      0x00000200  /*!< モンスター能力: 魔力吸収 / Drain Mana */
#define RF5_MIND_BLAST      0x00000400  /*!< モンスター能力: 精神攻撃 / Blast Mind */
#define RF5_BRAIN_SMASH     0x00000800  /*!< モンスター能力: 脳攻撃 / Smash Brain */
#define RF5_CAUSE_1         0x00001000  /*!< モンスター能力: 軽傷の呪い / Cause Light Wound */
#define RF5_CAUSE_2         0x00002000  /*!< モンスター能力: 重症の頃い / Cause Serious Wound */
#define RF5_CAUSE_3         0x00004000  /*!< モンスター能力: 致命傷の呪い / Cause Critical Wound */
#define RF5_CAUSE_4         0x00008000  /*!< モンスター能力: 秘孔を突く / Cause Mortal Wound */
#define RF5_BO_ACID         0x00010000  /*!< モンスター能力: アシッド・ボルト / Acid Bolt */
#define RF5_BO_ELEC         0x00020000  /*!< モンスター能力: サンダー・ボルト / Elec Bolt */
#define RF5_BO_FIRE         0x00040000  /*!< モンスター能力: ファイア・ボルト / Fire Bolt */
#define RF5_BO_COLD         0x00080000  /*!< モンスター能力: アイス・ボルト / Cold Bolt */
#define RF5_BA_LITE         0x00100000  /*!< モンスター能力: スター・バースト / StarBurst */
#define RF5_BO_NETH         0x00200000  /*!< モンスター能力: 地獄の矢 / Nether Bolt */
#define RF5_BO_WATE         0x00400000  /*!< モンスター能力: ウォーター・ボルト / Water Bolt */
#define RF5_BO_MANA         0x00800000  /*!< モンスター能力: 魔力の矢 / Mana Bolt */
#define RF5_BO_PLAS         0x01000000  /*!< モンスター能力: プラズマ・ボルト / Plasma Bolt */
#define RF5_BO_ICEE         0x02000000  /*!< モンスター能力: 極寒の矢 / Ice Bolt */
#define RF5_MISSILE         0x04000000  /*!< モンスター能力: マジック・ミサイルt / Magic Missile */
#define RF5_SCARE           0x08000000  /*!< モンスター能力: 恐慌 / Frighten Player */
#define RF5_BLIND           0x10000000  /*!< モンスター能力: 盲目 / Blind Player */
#define RF5_CONF            0x20000000  /*!< モンスター能力: 混乱 / Confuse Player */
#define RF5_SLOW            0x40000000  /*!< モンスター能力: 減速 / Slow Player */
#define RF5_HOLD            0x80000000  /*!< モンスター能力: 麻痺 / Paralyze Player */

/*
 * New monster race bit flags
 */
#define RF6_HASTE           0x00000001  /* Speed self */
#define RF6_HAND_DOOM       0x00000002  /* Hand of Doom */
#define RF6_HEAL            0x00000004  /* Heal self */
#define RF6_INVULNER        0x00000008  /* INVULNERABILITY! */
#define RF6_BLINK           0x00000010  /* Teleport Short */
#define RF6_TPORT           0x00000020  /* Teleport Long */
#define RF6_WORLD           0x00000040  /* world */
#define RF6_SPECIAL         0x00000080  /* Special Attack */
#define RF6_TELE_TO         0x00000100  /* Move player to monster */
#define RF6_TELE_AWAY       0x00000200  /* Move player far away */
#define RF6_TELE_LEVEL      0x00000400  /* Move player vertically */
#define RF6_PSY_SPEAR       0x00000800  /* Psyco-spear */
#define RF6_DARKNESS        0x00001000  /* Create Darkness */
#define RF6_TRAPS           0x00002000  /* Create Traps */
#define RF6_FORGET          0x00004000  /* Cause amnesia */
#define RF6_RAISE_DEAD      0x00008000  /* Raise Dead */
#define RF6_S_KIN           0x00010000  /* Summon "kin" */
#define RF6_S_CYBER         0x00020000  /* Summon Cyberdemons! */
#define RF6_S_MONSTER       0x00040000  /* Summon Monster */
#define RF6_S_MONSTERS      0x00080000  /* Summon Monsters */
#define RF6_S_ANT           0x00100000  /* Summon Ants */
#define RF6_S_SPIDER        0x00200000  /* Summon Spiders */
#define RF6_S_HOUND         0x00400000  /* Summon Hounds */
#define RF6_S_HYDRA         0x00800000  /* Summon Hydras */
#define RF6_S_ANGEL         0x01000000  /* Summon Angel */
#define RF6_S_DEMON         0x02000000  /* Summon Demon */
#define RF6_S_UNDEAD        0x04000000  /* Summon Undead */
#define RF6_S_DRAGON        0x08000000  /* Summon Dragon */
#define RF6_S_HI_UNDEAD     0x10000000  /* Summon Greater Undead */
#define RF6_S_HI_DRAGON     0x20000000  /* Summon Ancient Dragon */
#define RF6_S_AMBERITES     0x40000000  /* Summon Amberites */
#define RF6_S_UNIQUE        0x80000000  /* Summon Unique Monster */

/*
 * New monster race bit flags
 */
#define RF7_AQUATIC             0x00000001  /* Aquatic monster */
#define RF7_CAN_SWIM            0x00000002  /* Monster can swim */
#define RF7_CAN_FLY             0x00000004  /* Monster can fly */
#define RF7_FRIENDLY            0x00000008  /* Monster is friendly */
#define RF7_NAZGUL              0x00000010  /* Is a "Nazgul" unique */
#define RF7_UNIQUE2             0x00000020  /* Fake unique */
#define RF7_RIDING              0x00000040  /* Good for riding */
#define RF7_KAGE                0x00000080  /* Is kage */
#define RF7_HAS_LITE_1          0x00000100  /* Monster carries light */
#define RF7_SELF_LITE_1         0x00000200  /* Monster lights itself */
#define RF7_HAS_LITE_2          0x00000400  /* Monster carries light */
#define RF7_SELF_LITE_2         0x00000800  /* Monster lights itself */
#define RF7_GUARDIAN            0x00001000  /* Guardian of a dungeon */
#define RF7_CHAMELEON           0x00002000  /* Chameleon can change */
#define RF7_XXXX4XXX            0x00004000  /* Now Empty */
#define RF7_TANUKI              0x00008000  /* Tanuki disguise */
#define RF7_HAS_DARK_1          0x00010000  /* Monster carries darkness */
#define RF7_SELF_DARK_1         0x00020000  /* Monster darkens itself */
#define RF7_HAS_DARK_2          0x00040000  /* Monster carries darkness */
#define RF7_SELF_DARK_2         0x00080000  /* Monster darkens itself */

/*
 * Monster race flags
 */
#define RF8_WILD_ONLY           0x00000001
#define RF8_WILD_TOWN           0x00000002
#define RF8_XXX8X02             0x00000004
#define RF8_WILD_SHORE          0x00000008
#define RF8_WILD_OCEAN          0x00000010
#define RF8_WILD_WASTE          0x00000020
#define RF8_WILD_WOOD           0x00000040
#define RF8_WILD_VOLCANO        0x00000080
#define RF8_XXX8X08             0x00000100
#define RF8_WILD_MOUNTAIN       0x00000200
#define RF8_WILD_GRASS          0x00000400
#define RF8_WILD_ALL            0x80000000

/*
 * Monster drop info
 */
#define RF9_DROP_CORPSE         0x00000001
#define RF9_DROP_SKELETON       0x00000002
#define RF9_EAT_BLIND           0x00000004
#define RF9_EAT_CONF            0x00000008
#define RF9_EAT_MANA            0x00000010
#define RF9_EAT_NEXUS           0x00000020
#define RF9_EAT_SLEEP           0x00000040
#define RF9_EAT_BERSERKER       0x00000080
#define RF9_EAT_ACIDIC          0x00000100
#define RF9_EAT_SPEED           0x00000200
#define RF9_EAT_CURE            0x00000400
#define RF9_EAT_FIRE_RES        0x00000800
#define RF9_EAT_COLD_RES        0x00001000
#define RF9_EAT_ACID_RES        0x00002000
#define RF9_EAT_ELEC_RES        0x00004000
#define RF9_EAT_POIS_RES        0x00008000
#define RF9_EAT_INSANITY        0x00010000
#define RF9_EAT_DRAIN_EXP       0x00020000
#define RF9_EAT_POISONOUS       0x00040000
#define RF9_EAT_GIVE_STR        0x00080000
#define RF9_EAT_GIVE_INT        0x00100000
#define RF9_EAT_GIVE_WIS        0x00200000
#define RF9_EAT_GIVE_DEX        0x00400000
#define RF9_EAT_GIVE_CON        0x00800000
#define RF9_EAT_GIVE_CHR        0x01000000
#define RF9_EAT_LOSE_STR        0x02000000
#define RF9_EAT_LOSE_INT        0x04000000
#define RF9_EAT_LOSE_WIS        0x08000000
#define RF9_EAT_LOSE_DEX        0x10000000
#define RF9_EAT_LOSE_CON        0x20000000
#define RF9_EAT_LOSE_CHR        0x40000000
#define RF9_EAT_DRAIN_MANA      0x80000000

/*
 * Monster bit flags of racial resistances
 * Note: Resist confusion was merged to RFR_NO_CONF
 */
#define RFR_IM_ACID         0x00000001  /* Immunity acid */
#define RFR_IM_ELEC         0x00000002  /* Immunity elec */
#define RFR_IM_FIRE         0x00000004  /* Immunity fire */
#define RFR_IM_COLD         0x00000008  /* Immunity cold */
#define RFR_IM_POIS         0x00000010  /* Immunity poison */
#define RFR_RES_LITE        0x00000020  /* Resist lite */
#define RFR_RES_DARK        0x00000040  /* Resist dark */
#define RFR_RES_NETH        0x00000080  /* Resist nether */
#define RFR_RES_WATE        0x00000100  /* Resist water */
#define RFR_RES_PLAS        0x00000200  /* Resist plasma */
#define RFR_RES_SHAR        0x00000400  /* Resist shards */
#define RFR_RES_SOUN        0x00000800  /* Resist sound */
#define RFR_RES_CHAO        0x00001000  /* Resist chaos */
#define RFR_RES_NEXU        0x00002000  /* Resist nexus */
#define RFR_RES_DISE        0x00004000  /* Resist disenchantment */
#define RFR_RES_WALL        0x00008000  /* Resist force */
#define RFR_RES_INER        0x00010000  /* Resist inertia */
#define RFR_RES_TIME        0x00020000  /* Resist time */
#define RFR_RES_GRAV        0x00040000  /* Resist gravity */
#define RFR_RES_ALL         0x00080000  /* Resist all */
#define RFR_RES_TELE        0x00100000  /* Resist teleportation */
#define RFR_XXX21           0x00200000
#define RFR_XXX22           0x00400000
#define RFR_XXX23           0x00800000
#define RFR_XXX24           0x01000000
#define RFR_XXX25           0x02000000
#define RFR_XXX26           0x04000000
#define RFR_XXX27           0x08000000
#define RFR_XXX28           0x10000000
#define RFR_XXX29           0x20000000
#define RFR_XXX30           0x40000000
#define RFR_XXX31           0x80000000


/*
 * Hack -- choose "intelligent" spells when desperate
 * Including "summon" spells
 */
#define RF4_INT_MASK \
	(RF4_SUMMON_MASK | RF4_DISPEL)

#define RF5_INT_MASK \
	(RF5_SUMMON_MASK | \
	 RF5_HOLD | RF5_SLOW | RF5_CONF | RF5_BLIND | RF5_SCARE)

#define RF6_INT_MASK \
	(RF6_SUMMON_MASK | \
	 RF6_BLINK | RF6_TPORT | RF6_TELE_LEVEL | RF6_TELE_AWAY | \
	 RF6_HEAL | RF6_INVULNER | RF6_HASTE | RF6_TRAPS)

/*
 * Hack -- spells that cannot be used while player riding on the monster
 */
#define RF4_RIDING_MASK \
	(RF4_SHRIEK)

#define RF5_RIDING_MASK 0UL

#define RF6_RIDING_MASK \
	(RF6_BLINK | RF6_TPORT | RF6_TRAPS | RF6_DARKNESS | RF6_SPECIAL)

/*
 * Hack -- "bolt" spells that may hurt fellow monsters
 * Currently "bolt" spells are included in "attack"
 */
#define RF4_BOLT_MASK \
	(RF4_ROCKET | RF4_SHOOT)

#define RF5_BOLT_MASK \
	(RF5_BO_ACID | RF5_BO_ELEC | RF5_BO_FIRE | RF5_BO_COLD | \
	 RF5_BO_NETH | RF5_BO_WATE | RF5_BO_MANA | RF5_BO_PLAS | \
	 RF5_BO_ICEE | RF5_MISSILE)

#define RF6_BOLT_MASK 0UL

/*
 * Hack -- "beam" spells that may hurt fellow monsters
 * Currently "beam" spells are included in "attack"
 */
#define RF4_BEAM_MASK 0UL

#define RF5_BEAM_MASK 0UL

#define RF6_BEAM_MASK (RF6_PSY_SPEAR)

/*
 * Hack -- "ball" spells that may hurt friends
 * Including "radius 4 ball" and "breath" spells
 * Currently "ball" spells are included in "attack"
 */
#define RF4_BALL_MASK \
	(RF4_BIG_BALL_MASK | RF4_BREATH_MASK | \
	 RF4_ROCKET | RF4_BA_NUKE)

#define RF5_BALL_MASK \
	(RF5_BIG_BALL_MASK | RF5_BREATH_MASK | \
	 RF5_BA_ACID | RF5_BA_ELEC | RF5_BA_FIRE | RF5_BA_COLD | \
	 RF5_BA_POIS | RF5_BA_NETH)

#define RF6_BALL_MASK \
	(RF6_BIG_BALL_MASK | RF6_BREATH_MASK)

/*
 * Hack -- "ball" spells with radius 4 that may hurt friends
 * Currently "radius 4 ball" spells are included in "ball"
 */
#define RF4_BIG_BALL_MASK \
	(RF4_BA_CHAO)

#define RF5_BIG_BALL_MASK \
	(RF5_BA_LITE | RF5_BA_DARK | RF5_BA_WATE | RF5_BA_MANA)

#define RF6_BIG_BALL_MASK 0UL

/*
 * Hack -- "breath" spells that may hurt friends
 * Currently "breath" spells are included in "ball" and "non-magic"
 */
#define RF4_BREATH_MASK \
	(RF4_BR_ACID | RF4_BR_ELEC | RF4_BR_FIRE | RF4_BR_COLD | \
	 RF4_BR_POIS | RF4_BR_NETH | RF4_BR_LITE | RF4_BR_DARK | \
	 RF4_BR_CONF | RF4_BR_SOUN | RF4_BR_CHAO | RF4_BR_DISE | \
	 RF4_BR_NEXU | RF4_BR_SHAR | RF4_BR_TIME | RF4_BR_INER | \
	 RF4_BR_GRAV | RF4_BR_PLAS | RF4_BR_WALL | RF4_BR_MANA | \
	 RF4_BR_NUKE | RF4_BR_DISI)

#define RF5_BREATH_MASK 0UL

#define RF6_BREATH_MASK 0UL

/*
 * Hack -- "summon" spells
 * Currently "summon" spells are included in "intelligent" and "indirect"
 */
#define RF4_SUMMON_MASK 0UL

#define RF5_SUMMON_MASK 0UL

#define RF6_SUMMON_MASK \
	(RF6_S_KIN | RF6_S_CYBER | RF6_S_MONSTER | RF6_S_MONSTERS | RF6_S_ANT | \
	 RF6_S_SPIDER | RF6_S_HOUND | RF6_S_HYDRA | RF6_S_ANGEL | RF6_S_DEMON | \
	 RF6_S_UNDEAD | RF6_S_DRAGON | RF6_S_HI_UNDEAD | RF6_S_HI_DRAGON | \
	 RF6_S_AMBERITES | RF6_S_UNIQUE)

/*
 * Hack -- "attack" spells
 * Including "bolt", "beam" and "ball" spells
 */
#define RF4_ATTACK_MASK \
	(RF4_BOLT_MASK | RF4_BEAM_MASK | RF4_BALL_MASK | RF4_DISPEL)

#define RF5_ATTACK_MASK \
	(RF5_BOLT_MASK | RF5_BEAM_MASK | RF5_BALL_MASK | \
	 RF5_DRAIN_MANA | RF5_MIND_BLAST | RF5_BRAIN_SMASH | \
	 RF5_CAUSE_1 | RF5_CAUSE_2 | RF5_CAUSE_3 | RF5_CAUSE_4 | \
	 RF5_SCARE | RF5_BLIND | RF5_CONF | RF5_SLOW | RF5_HOLD)

#define RF6_ATTACK_MASK \
	(RF6_BOLT_MASK | RF6_BEAM_MASK | RF6_BALL_MASK | \
	 RF6_HAND_DOOM | RF6_TELE_TO | RF6_TELE_AWAY | RF6_TELE_LEVEL | \
	 RF6_DARKNESS | RF6_TRAPS | RF6_FORGET)

/*
 * Hack -- "indirect" spells
 * Including "summon" spells
 */
#define RF4_INDIRECT_MASK \
	(RF4_SUMMON_MASK | RF4_SHRIEK)

#define RF5_INDIRECT_MASK \
	(RF5_SUMMON_MASK)

#define RF6_INDIRECT_MASK \
	(RF6_SUMMON_MASK | \
	 RF6_HASTE | RF6_HEAL | RF6_INVULNER | RF6_BLINK | RF6_WORLD | \
	 RF6_TPORT | RF6_RAISE_DEAD)

/*
 * Hack -- "non-magic" spells
 * Including "breath" spells
 */
#define RF4_NOMAGIC_MASK \
	(RF4_BREATH_MASK | RF4_SHRIEK | RF4_ROCKET | RF4_SHOOT)

#define RF5_NOMAGIC_MASK \
	(RF5_BREATH_MASK)

#define RF6_NOMAGIC_MASK \
	(RF6_BREATH_MASK | RF6_SPECIAL)

/*
 * Hack -- "torch" masks
 */
#define RF7_LITE_MASK \
	(RF7_HAS_LITE_1 | RF7_SELF_LITE_1 | RF7_HAS_LITE_2 | RF7_SELF_LITE_2)

#define RF7_DARK_MASK \
	(RF7_HAS_DARK_1 | RF7_SELF_DARK_1 | RF7_HAS_DARK_2 | RF7_SELF_DARK_2)

#define RF7_HAS_LD_MASK \
	(RF7_HAS_LITE_1 | RF7_HAS_LITE_2 | RF7_HAS_DARK_1 | RF7_HAS_DARK_2)

#define RF7_SELF_LD_MASK \
	(RF7_SELF_LITE_1 | RF7_SELF_LITE_2 | RF7_SELF_DARK_1 | RF7_SELF_DARK_2)

/*
 * Hack -- effective elemental and poison immunity mask
 */
#define RFR_EFF_IM_ACID_MASK  (RFR_IM_ACID | RFR_RES_ALL)
#define RFR_EFF_IM_ELEC_MASK  (RFR_IM_ELEC | RFR_RES_ALL)
#define RFR_EFF_IM_FIRE_MASK  (RFR_IM_FIRE | RFR_RES_ALL)
#define RFR_EFF_IM_COLD_MASK  (RFR_IM_COLD | RFR_RES_ALL)
#define RFR_EFF_IM_POIS_MASK  (RFR_IM_POIS | RFR_RES_ALL)
#define RFR_EFF_RES_SHAR_MASK (RFR_RES_SHAR | RFR_RES_ALL)
#define RFR_EFF_RES_CHAO_MASK (RFR_RES_CHAO | RFR_RES_ALL)
#define RFR_EFF_RES_NEXU_MASK (RFR_RES_NEXU | RFR_RES_ALL)


#define MR1_SINKA 0x01


#define is_friendly(A) \
	 (bool)(((A)->smart & SM_FRIENDLY) ? TRUE : FALSE)

#define is_friendly_idx(IDX) \
	 (bool)((IDX) > 0 && is_friendly(&m_list[(IDX)]))

#define is_pet(A) \
	 (bool)(((A)->smart & SM_PET) ? TRUE : FALSE)

#define is_hostile(A) \
	 (bool)((is_friendly(A) || is_pet(A)) ? FALSE : TRUE)

/* Hack -- Determine monster race appearance index is same as race index */
#define is_original_ap(A) \
	 (bool)(((A)->ap_r_idx == (A)->r_idx) ? TRUE : FALSE)

#define is_original_ap_and_seen(A) \
	 (bool)((A)->ml && !p_ptr->image && ((A)->ap_r_idx == (A)->r_idx))

/*
 * Is the monster seen by the player?
 */
#define is_seen(A) \
	((bool)((A)->ml && (!ignore_unview || p_ptr->inside_battle || \
	 (player_can_see_bold((A)->fy, (A)->fx) && projectable(p_ptr->y, p_ptr->x, (A)->fy, (A)->fx)))))


/*** Option Definitions ***/


#define OPT_PAGE_INPUT          1
#define OPT_PAGE_MAPSCREEN      2
#define OPT_PAGE_TEXT           3
#define OPT_PAGE_GAMEPLAY       4
#define OPT_PAGE_DISTURBANCE    5
#define OPT_PAGE_BIRTH          6
#define OPT_PAGE_AUTODESTROY    7
#define OPT_PAGE_PLAYRECORD    10

#define OPT_PAGE_JAPANESE_ONLY 99


/*** Macro Definitions ***/


/*
 * Hack -- The main "screen"
 */
#define term_screen     (angband_term[0])


/*
 * Determine if a given inventory item is "aware"
 */
#define object_is_aware(T) \
    (k_info[(T)->k_idx].aware)

/*
 * Determine if a given inventory item is "tried"
 */
#define object_is_tried(T) \
    (k_info[(T)->k_idx].tried)


/*
 * Determine if a given inventory item is "known"
 * Test One -- Check for special "known" tag
 * Test Two -- Check for "Easy Know" + "Aware"
 */
#define object_is_known(T) \
    (((T)->ident & (IDENT_KNOWN)) || \
     (k_info[(T)->k_idx].easy_know && k_info[(T)->k_idx].aware))


/*
 * Return the "attr" for a given item.
 * Use "flavor" if available.
 * Default to user definitions.
 */
#define object_attr(T) \
	((k_info[(T)->k_idx].flavor) ? \
	 (k_info[k_info[(T)->k_idx].flavor].x_attr) : \
	 ((!(T)->k_idx || ((T)->tval != TV_CORPSE) || ((T)->sval != SV_CORPSE) || \
	   (k_info[(T)->k_idx].x_attr != TERM_DARK)) ? \
	  (k_info[(T)->k_idx].x_attr) : (r_info[(T)->pval].x_attr)))

/*
 * Return the "char" for a given item.
 * Use "flavor" if available.
 * Default to user definitions.
 */
#define object_char(T) \
	((k_info[(T)->k_idx].flavor) ? \
	 (k_info[k_info[(T)->k_idx].flavor].x_char) : \
	 (k_info[(T)->k_idx].x_char))


/*
 * Artifacts use the "name1" field
 */
#define object_is_fixed_artifact(T) \
	((T)->name1 ? TRUE : FALSE)

/*
 * Ego-Items use the "name2" field
 */
#define object_is_ego(T) \
	((T)->name2 ? TRUE : FALSE)


/*
 * Broken items.
 */
#define object_is_broken(T) \
	((T)->ident & (IDENT_BROKEN))

/*
 * Cursed items.
 */
#define object_is_cursed(T) \
	((T)->curse_flags)


/*
 * Convert an "attr"/"char" pair into a "pict" (P)
 */
#define PICT(A,C) \
	((((u16b)(A)) << 8) | ((byte)(C)))

/*
 * Convert a "pict" (P) into an "attr" (A)
 */
#define PICT_A(P) \
	((byte)((P) >> 8))

/*
 * Convert a "pict" (P) into an "char" (C)
 */
#define PICT_C(P) \
	((char)((byte)(P)))


/*
 * Convert a "location" (Y,X) into a "grid" (G)
 */
#define GRID(Y,X) \
	(256 * (Y) + (X))

/*
 * Convert a "grid" (G) into a "location" (Y)
 */
#define GRID_Y(G) \
	((int)((G) / 256U))

/*
 * Convert a "grid" (G) into a "location" (X)
 */
#define GRID_X(G) \
	((int)((G) % 256U))


/*
 * Determines if a map location is fully inside the outer walls
 */
#define in_bounds(Y,X) \
   (((Y) > 0) && ((X) > 0) && ((Y) < cur_hgt-1) && ((X) < cur_wid-1))

/*
 * Determines if a map location is on or inside the outer walls
 */
#define in_bounds2(Y,X) \
   (((Y) >= 0) && ((X) >= 0) && ((Y) < cur_hgt) && ((X) < cur_wid))

/*
 * Determines if a map location is on or inside the outer walls
 * (unsigned version)
 */
#define in_bounds2u(Y,X) \
   (((Y) < cur_hgt) && ((X) < cur_wid))

/*
 * Determines if a map location is currently "on screen" -RAK-
 * Note that "panel_contains(Y,X)" always implies "in_bounds2(Y,X)".
 */
#define panel_contains(Y,X) \
  (((Y) >= panel_row_min) && ((Y) <= panel_row_max) && \
   ((X) >= panel_col_min) && ((X) <= panel_col_max))


/*
 * Determine if player is on this grid
 */
#define player_bold(Y,X) \
	(((Y) == p_ptr->y) && ((X) == p_ptr->x))


/*
 * Grid based version of "player_bold()"
 */
#define player_grid(C) \
	((C) == &cave[p_ptr->y][p_ptr->x])


#define cave_have_flag_bold(Y,X,INDEX) \
	(have_flag(f_info[cave[(Y)][(X)].feat].flags, (INDEX)))


#define cave_have_flag_grid(C,INDEX) \
	(have_flag(f_info[(C)->feat].flags, (INDEX)))


/*
 * Determine if a "feature" supports "los"
 */
#define feat_supports_los(F) \
	(have_flag(f_info[(F)].flags, FF_LOS))


/*
 * Determine if a "legal" grid supports "los"
 */
#define cave_los_bold(Y,X) \
	(feat_supports_los(cave[(Y)][(X)].feat))

#define cave_los_grid(C) \
	(feat_supports_los((C)->feat))


/*
 * Determine if a "legal" grid is a "clean" floor grid
 * Determine if terrain-change spells are allowed in a grid.
 *
 * Line 1 -- forbid non-floors
 * Line 2 -- forbid object terrains
 * Line 3 -- forbid normal objects
 */
#define cave_clean_bold(Y,X) \
	(cave_have_flag_bold((Y), (X), FF_FLOOR) && \
	 !(cave[Y][X].info & CAVE_OBJECT) && \
	  (cave[Y][X].o_idx == 0))


/*
 * Determine if an object can be dropped on a "legal" grid
 *
 * Line 1 -- forbid non-drops
 * Line 2 -- forbid object terrains
 */
#define cave_drop_bold(Y,X) \
	(cave_have_flag_bold((Y), (X), FF_DROP) && \
	 !(cave[Y][X].info & CAVE_OBJECT))


/*
 * Determine if a "legal" grid is an "empty" floor grid
 * Determine if monsters are allowed to move into a grid
 *
 * Line 1 -- forbid non-placement grids
 * Line 2 -- forbid normal monsters
 * Line 3 -- forbid the player
 */
#define cave_empty_bold(Y,X) \
	(cave_have_flag_bold((Y), (X), FF_PLACE) && \
	 !(cave[Y][X].m_idx) && \
	 !player_bold(Y,X))


/*
 * Determine if a "legal" grid is an "empty" floor grid
 * Determine if monster generation is allowed in a grid
 *
 * Line 1 -- forbid non-empty grids
 * Line 2 -- forbid trees while dungeon generation
 */
#define cave_empty_bold2(Y,X) \
	(cave_empty_bold(Y,X) && \
	 (character_dungeon || !cave_have_flag_bold((Y), (X), FF_TREE)))


/*
 * Determine if a "legal" grid is an "naked" floor grid
 *
 * Line 1 -- forbid non-clean gird
 * Line 2 -- forbid monsters
 * Line 3 -- forbid the player
 */
#define cave_naked_bold(Y,X) \
	(cave_clean_bold(Y,X) && \
	 !(cave[Y][X].m_idx) && \
	 !player_bold(Y,X))


/*
 * Determine if a "legal" grid is "permanent"
 *
 * Line 1 -- permanent flag
 */
#define cave_perma_bold(Y,X) \
	(cave_have_flag_bold((Y), (X), FF_PERMANENT))


/*
 * Grid based version of "cave_empty_bold()"
 */
#define cave_empty_grid(C) \
	(cave_have_flag_grid((C), FF_PLACE) && \
	 !((C)->m_idx) && \
	 !player_grid(C))


/*
 * Grid based version of "cave_perma_bold()"
 */
#define cave_perma_grid(C) \
	(cave_have_flag_grid((C), FF_PERMANENT))


#define pattern_tile(Y,X) \
	(cave_have_flag_bold((Y), (X), FF_PATTERN))

/*
 * Does the grid stop disintegration?
 */
#define cave_stop_disintegration(Y,X) \
	(!cave_have_flag_bold((Y), (X), FF_PROJECT) && \
	 (!cave_have_flag_bold((Y), (X), FF_HURT_DISI) || \
	  cave_have_flag_bold((Y), (X), FF_PERMANENT)))


/*
 * Determine if a "legal" grid is within "los" of the player
 *
 * Note the use of comparison to zero to force a "boolean" result
 */
#define player_has_los_grid(C) \
    (((C)->info & (CAVE_VIEW)) != 0)

/*
 * Determine if a "legal" grid is within "los" of the player
 *
 * Note the use of comparison to zero to force a "boolean" result
 */
#define player_has_los_bold(Y,X) \
    (((cave[Y][X].info & (CAVE_VIEW)) != 0) || p_ptr->inside_battle)


/*
 * Determine if a "feature" is "permanent wall"
 */
#define permanent_wall(F) \
	(have_flag((F)->flags, FF_WALL) && \
	 have_flag((F)->flags, FF_PERMANENT))

/*
 * Get feature mimic from f_info[] (applying "mimic" field)
 */
#define get_feat_mimic(C) \
	(f_info[(C)->mimic ? (C)->mimic : (C)->feat].mimic)

/*
 * Hack -- Prepare to use the "Secure" routines
 */
#if defined(SET_UID) && defined(SECURE)
extern int PlayerUID;
# define getuid() PlayerUID
# define geteuid() PlayerUID
#endif



/*** Color constants ***/


/*
 * Angband "attributes" (with symbols, and base (R,G,B) codes)
 *
 * The "(R,G,B)" codes are given in "fourths" of the "maximal" value,
 * and should "gamma corrected" on most (non-Macintosh) machines.
 */
#define TERM_DARK                0  /*!< 'd' - 黒 0,0,0 */
#define TERM_WHITE               1  /*!< 'w' - 白 4,4,4 */
#define TERM_SLATE               2  /*!< 's' - 灰 2,2,2 */
#define TERM_ORANGE              3  /*!< 'o' - 橙 4,2,0 */
#define TERM_RED                 4  /*!< 'r' - 赤 3,0,0 */
#define TERM_GREEN               5  /*!< 'g' - 緑 0,2,1 */
#define TERM_BLUE                6  /*!< 'b' - 青 0,0,4 */
#define TERM_UMBER               7  /*!< 'u' - 琥珀 2,1,0 */
#define TERM_L_DARK              8  /*!< 'D' - 暗い灰 1,1,1 */
#define TERM_L_WHITE             9  /*!< 'W' - 明るい灰 3,3,3 */
#define TERM_VIOLET             10  /*!< 'v' - 紫 4,0,4 */
#define TERM_YELLOW             11  /*!< 'y' - 黄 4,4,0 */
#define TERM_L_RED              12  /*!< 'R' - 明るい赤 4,0,0 */
#define TERM_L_GREEN            13  /*!< 'G' - 明るい緑 0,4,0 */
#define TERM_L_BLUE             14  /*!< 'B' - 明るい青 0,4,4 */
#define TERM_L_UMBER            15  /*!< 'U' - 明るい琥珀 3,2,1 */


/*
 * Not using graphical tiles for this feature?
 */
#define is_ascii_graphics(A) (!((A) & 0x80))

/*** Music constants ***/

#define MUSIC_BASIC_DEFAULT    0
#define MUSIC_BASIC_GAMEOVER   1
#define MUSIC_BASIC_EXIT       2
#define MUSIC_BASIC_TOWN       3
#define MUSIC_BASIC_FIELD1     4
#define MUSIC_BASIC_FIELD2     5
#define MUSIC_BASIC_FIELD3     6
#define MUSIC_BASIC_DUN_LOW    7
#define MUSIC_BASIC_DUN_MED    8
#define MUSIC_BASIC_DUN_HIGH   9
#define MUSIC_BASIC_DUN_FEEL1 10
#define MUSIC_BASIC_DUN_FEEL2 11
#define MUSIC_BASIC_WINNER    12
#define MUSIC_BASIC_BUILD     13
#define MUSIC_BASIC_WILD      14
#define MUSIC_BASIC_QUEST     15
#define MUSIC_BASIC_ARENA     16
#define MUSIC_BASIC_BATTLE    17
#define MUSIC_BASIC_QUEST_CLEAR 18
#define MUSIC_BASIC_FINAL_QUEST_CLEAR 19
#define MUSIC_BASIC_AMBUSH    20
#define MUSIC_BASIC_MAX       21 /*!< BGM定義の最大数 */

/*** Sound constants ***/

/*
 * Mega-Hack -- some primitive sound support (see "main-win.c")
 *
 * Some "sound" constants for "Term_xtra(TERM_XTRA_SOUND, val)"
 */
#define SOUND_HIT        1
#define SOUND_MISS       2
#define SOUND_FLEE       3
#define SOUND_DROP       4
#define SOUND_KILL       5
#define SOUND_LEVEL      6
#define SOUND_DEATH      7
#define SOUND_STUDY      8
#define SOUND_TELEPORT   9
#define SOUND_SHOOT     10
#define SOUND_QUAFF     11
#define SOUND_ZAP       12
#define SOUND_WALK      13
#define SOUND_TPOTHER   14
#define SOUND_HITWALL   15
#define SOUND_EAT       16
#define SOUND_STORE1    17
#define SOUND_STORE2    18
#define SOUND_STORE3    19
#define SOUND_STORE4    20
#define SOUND_DIG       21
#define SOUND_OPENDOOR  22
#define SOUND_SHUTDOOR  23
#define SOUND_TPLEVEL   24
#define SOUND_SCROLL	25
#define SOUND_BUY	    26
#define SOUND_SELL	    27
#define SOUND_WARN	    28
#define SOUND_ROCKET    29 /*!< Somebody's shooting rockets */
#define SOUND_N_KILL    30 /*!< The player kills a non-living/undead monster */
#define SOUND_U_KILL    31 /*!< The player kills a unique */
#define SOUND_QUEST     32 /*!< The player has just completed a quest */
#define SOUND_HEAL      33 /*!< The player was healed a little bit */
#define SOUND_X_HEAL    34 /*!< The player was healed full health */
#define SOUND_BITE      35 /*!< A monster bites you */
#define SOUND_CLAW      36 /*!< A monster claws you */
#define SOUND_M_SPELL   37 /*!< A monster casts a miscellaneous spell */
#define SOUND_SUMMON    38 /*!< A monster casts a summoning spell  */
#define SOUND_BREATH    39 /*!< A monster breathes */
#define SOUND_BALL      40 /*!< A monster casts a ball / bolt spell */
#define SOUND_M_HEAL    41 /*!< A monster heals itself somehow */
#define SOUND_ATK_SPELL 42 /*!< A monster casts a misc. offensive spell */
#define SOUND_EVIL      43 /*!< Something nasty has just happened! */
#define SOUND_TOUCH     44 /*!< A monster touches you */
#define SOUND_STING     45 /*!< A monster stings you */
#define SOUND_CRUSH     46 /*!< A monster crushes / envelopes you */
#define SOUND_SLIME     47 /*!< A monster drools/spits/etc on you */
#define SOUND_WAIL      48 /*!< A monster wails */
#define SOUND_WINNER    49 /*!< Just won the game! */
#define SOUND_FIRE      50 /*!< An item was burned  */
#define SOUND_ACID      51 /*!< An item was destroyed by acid */
#define SOUND_ELEC      52 /*!< An item was destroyed by electricity */
#define SOUND_COLD      53 /*!< An item was shattered */
#define SOUND_ILLEGAL   54 /*!< Illegal command attempted */
#define SOUND_FAIL      55 /*!< Fail to get a spell off / activate an item */
#define SOUND_WAKEUP    56 /*!< A monster wakes up */
#define SOUND_INVULN    57 /*!< Invulnerability! */
#define SOUND_FALL      58 /*!< Falling through a trapdoor... */
#define SOUND_PAIN      59 /*!< A monster is in pain! */
#define SOUND_DESTITEM  60 /*!< An item was destroyed by misc. means */
#define SOUND_MOAN      61 /*!< A monster makes a moan/beg/insult attack */
#define SOUND_SHOW      62 /*!< A monster makes a "show" attack */
#define SOUND_UNUSED    63 /*!< (no sound for gaze attacks) */
#define SOUND_EXPLODE   64 /*!< Something (or somebody) explodes */
#define SOUND_GLASS     65 /*!< A glass feature was crashed */
#define SOUND_REFLECT   66 /*!< A bolt was reflected */

/*
 * Mega-Hack -- maximum known sounds
 */
#define SOUND_MAX 67 /*!< 効果音定義の最大数 */

#define MAX_VIRTUE 18 /*!< 徳定義の最大数 */

#define V_COMPASSION    1
#define V_HONOUR	    2
#define V_JUSTICE	    3
#define V_SACRIFICE	    4
#define V_KNOWLEDGE	    5
#define V_FAITH 	    6
#define V_ENLIGHTEN	    7
#define V_ENCHANT	    8
#define V_CHANCE	    9
#define V_NATURE	   10
#define V_HARMONY	   11
#define V_VITALITY	   12
#define V_UNLIFE	   13
#define V_PATIENCE	   14
#define V_TEMPERANCE	   15
#define V_DILIGENCE	   16
#define V_VALOUR	   17
#define V_INDIVIDUALISM    18


/*** Hack ***/


/*
 * Hack -- attempt to reduce various values
 */
#ifdef ANGBAND_LITE
# undef MACRO_MAX
# define MACRO_MAX      128
# undef QUARK_MAX
# define QUARK_MAX      128
# undef MESSAGE_MAX
# define MESSAGE_MAX    128
# undef MESSAGE_BUF
# define MESSAGE_BUF    4096
#endif


/*
 * Buildings actions
 */
#define BACT_NOTHING                 0
#define BACT_RESEARCH_ITEM           1
#define BACT_TOWN_HISTORY            2
#define BACT_RACE_LEGENDS            3
#define BACT_GREET_KING              4
#define BACT_KING_LEGENDS            5
#define BACT_QUEST                   6
#define BACT_XXX_UNUSED              7
#define BACT_POSTER                  8
#define BACT_ARENA_RULES             9
#define BACT_ARENA                  10
#define BACT_ARENA_LEGENDS          11
#define BACT_IN_BETWEEN             12
#define BACT_GAMBLE_RULES           13
#define BACT_CRAPS                  14
#define BACT_SPIN_WHEEL             15
#define BACT_DICE_SLOTS             16
#define BACT_REST                   17
#define BACT_FOOD                   18
#define BACT_RUMORS                 19
#define BACT_RESEARCH_MONSTER       20
#define BACT_COMPARE_WEAPONS        21
#define BACT_LEGENDS                22
#define BACT_ENCHANT_WEAPON         23
#define BACT_ENCHANT_ARMOR          24
#define BACT_RECHARGE               25
#define BACT_IDENTS                 26
#define BACT_LEARN                  27
#define BACT_HEALING                28
#define BACT_RESTORE                29
#define BACT_ENCHANT_ARROWS         30
#define BACT_ENCHANT_BOW            31
#define BACT_GREET                  32
#define BACT_RECALL                 33
#define BACT_TELEPORT_LEVEL         34
#define BACT_LOSE_MUTATION          35
#define BACT_BATTLE                 36
#define BACT_TSUCHINOKO             37
#define BACT_TARGET                 38
#define BACT_KUBI                   39
#define BACT_KANKIN                 40
#define BACT_HEIKOUKA               41
#define BACT_TELE_TOWN              42
#define BACT_POKER                  43
#define BACT_IDENT_ONE              44
#define BACT_RECHARGE_ALL           45
#define BACT_EVAL_AC		        46
#define BACT_BROKEN_WEAPON          47
#define MAX_BACT                    48

/*
 * Quest status
 */
#define QUEST_STATUS_UNTAKEN              0 /*!< クエストステータス状態：未発生*/
#define QUEST_STATUS_TAKEN                1 /*!< クエストステータス状態：発生中*/
#define QUEST_STATUS_COMPLETED            2 /*!< クエストステータス状態：達成*/
#define QUEST_STATUS_REWARDED             3 /*!< クエストステータス状態：報酬受け取り前*/
#define QUEST_STATUS_FINISHED             4 /*!< クエストステータス状態：完了*/
#define QUEST_STATUS_FAILED               5 /*!< クエストステータス状態：失敗*/
#define QUEST_STATUS_FAILED_DONE          6 /*!< クエストステータス状態：失敗完了*/
#define QUEST_STATUS_STAGE_COMPLETED      7 /*!< クエストステータス状態：ステージ毎達成*/

/*
 * Quest type
 */
#define QUEST_TYPE_KILL_LEVEL                1 /*!< クエスト目的: 特定のユニークモンスターを倒す */
#define QUEST_TYPE_KILL_ANY_LEVEL            2 /*!< クエスト目的: イベント受託時点でランダムで選ばれた特定のユニークモンスターを倒す */
#define QUEST_TYPE_FIND_ARTIFACT             3 /*!< クエスト目的: 特定のアーティファクトを発見する */
#define QUEST_TYPE_FIND_EXIT                 4 /*!< クエスト目的: 脱出する */
#define QUEST_TYPE_KILL_NUMBER               5 /*!< クエスト目的: モンスターを無差別に特定数倒す */
#define QUEST_TYPE_KILL_ALL                  6 /*!< クエスト目的: エリア中のすべてのモンスターを全て倒す */
#define QUEST_TYPE_RANDOM                    7 /*!< クエスト目的: ランダムクエストとして選ばれたユニーク1体を倒す */
#define QUEST_TYPE_TOWER                     8 /*!< クエスト目的: 複数のエリアの全てのモンスターを倒す */

/*
 * Initialization flags
 */
#define INIT_NAME_ONLY          0x01
#define INIT_SHOW_TEXT          0x02
#define INIT_ASSIGN             0x04
#define INIT_CREATE_DUNGEON     0x08
#define INIT_ONLY_FEATURES      0x10
#define INIT_ONLY_BUILDINGS     0x20

/*
 * Quest flags
 */
#define QUEST_FLAG_SILENT  0x01 /*!< クエストフラグ: クエスト進行に関する情報表示を抑止する / no messages from completion */
#define QUEST_FLAG_PRESET  0x02 /*!< クエストフラグ: クエストがダンジョン外で発生する / quest is outside the main dungeon */
#define QUEST_FLAG_ONCE    0x04 /*!< クエストフラグ: クエストがフロアを出た時点で完了する / quest is marked finished after leaving */
#define QUEST_FLAG_TOWER   0x08 /*!< クエストフラグ: クエスト:塔の形式で進行する / Tower quest is special */

/*
 * Available graphic modes
 */
#define GRAPHICS_NONE       0
#define GRAPHICS_ORIGINAL   1
#define GRAPHICS_ADAM_BOLT  2
#define GRAPHICS_HENGBAND   3

/*
 * Modes for the random name generator
 */
#define NAME_DWARF  1
#define NAME_ELF    2
#define NAME_GNOME  3
#define NAME_HOBBIT 4
#define NAME_HUMAN  5
#define NAME_ORC    6

#ifdef JP
#define JVERB_AND 1
#define JVERB_TO  2
#define JVERB_OR  3
#endif

/*
 * Modes for the tokenizer
 */
#define TOKENIZE_CHECKQUOTE 0x01  /* Special handling of single quotes */

/*
 * Parse errors
 */
#define PARSE_ERROR_GENERIC                  1
#define PARSE_ERROR_ABSOLETE_FILE            2
#define PARSE_ERROR_MISSING_RECORD_HEADER    3
#define PARSE_ERROR_NON_SEQUENTIAL_RECORDS   4
#define PARSE_ERROR_INVALID_FLAG             5
#define PARSE_ERROR_UNDEFINED_DIRECTIVE      6
#define PARSE_ERROR_OUT_OF_MEMORY            7
#define PARSE_ERROR_OUT_OF_BOUNDS            8
#define PARSE_ERROR_TOO_FEW_ARGUMENTS        9
#define PARSE_ERROR_UNDEFINED_TERRAIN_TAG   10
#define PARSE_ERROR_MAX                     11

#define GINOU_SUDE      0
#define GINOU_NITOURYU  1
#define GINOU_RIDING    2
#define GINOU_MAX      10

/* Proficiency level */
#define EXP_LEVEL_UNSKILLED 0
#define EXP_LEVEL_BEGINNER  1
#define EXP_LEVEL_SKILLED   2
#define EXP_LEVEL_EXPERT    3
#define EXP_LEVEL_MASTER    4

/* Proficiency of weapons and misc. skills (except riding) */
#define WEAPON_EXP_UNSKILLED     0
#define WEAPON_EXP_BEGINNER   4000
#define WEAPON_EXP_SKILLED    6000
#define WEAPON_EXP_EXPERT     7000
#define WEAPON_EXP_MASTER     8000

/* Proficiency of riding */
#define RIDING_EXP_UNSKILLED     0
#define RIDING_EXP_BEGINNER    500
#define RIDING_EXP_SKILLED    2000
#define RIDING_EXP_EXPERT     5000
#define RIDING_EXP_MASTER     8000

/* Proficiency of spells */
#define SPELL_EXP_UNSKILLED      0
#define SPELL_EXP_BEGINNER     900
#define SPELL_EXP_SKILLED     1200
#define SPELL_EXP_EXPERT      1400
#define SPELL_EXP_MASTER      1600

#define NO_TOWN 6
#define SECRET_TOWN 5

#define NIKKI_HIGAWARI     0
#define NIKKI_BUNSHOU      1
#define NIKKI_ART          2
#define NIKKI_UNIQUE       3
#define NIKKI_FIX_QUEST_C  4
#define NIKKI_FIX_QUEST_F  5
#define NIKKI_RAND_QUEST_C 6
#define NIKKI_RAND_QUEST_F 7
#define NIKKI_MAXDEAPTH    8
#define NIKKI_TRUMP        9
#define NIKKI_STAIR       10
#define NIKKI_RECALL      11
#define NIKKI_TO_QUEST    12
#define NIKKI_TELE_LEV    13
#define NIKKI_BUY         14
#define NIKKI_SELL        15
#define NIKKI_ARENA       16
#define NIKKI_HANMEI      17
#define NIKKI_LEVELUP     18
#define NIKKI_GAMESTART   19
#define NIKKI_WIZ_TELE    20
#define NIKKI_NAMED_PET   21
#define NIKKI_PAT_TELE    22
#define NIKKI_ART_SCROLL  23

#define NIKKI_WIZARD_LOG  24


#define RECORD_NAMED_PET_NAME        0
#define RECORD_NAMED_PET_UNNAME      1
#define RECORD_NAMED_PET_DISMISS     2
#define RECORD_NAMED_PET_DEATH       3
#define RECORD_NAMED_PET_MOVED       4
#define RECORD_NAMED_PET_LOST_SIGHT  5
#define RECORD_NAMED_PET_DESTROY     6
#define RECORD_NAMED_PET_EARTHQUAKE  7
#define RECORD_NAMED_PET_GENOCIDE    8
#define RECORD_NAMED_PET_WIZ_ZAP     9
#define RECORD_NAMED_PET_TELE_LEVEL  10
#define RECORD_NAMED_PET_BLAST       11
#define RECORD_NAMED_PET_HEAL_LEPER  12
#define RECORD_NAMED_PET_COMPACT     13
#define RECORD_NAMED_PET_LOSE_PARENT 14

#define MAX_MANE 16
#define MAX_MONSPELLS 96
#define MONSPELL_TYPE_BOLT 1
#define MONSPELL_TYPE_BALL 2
#define MONSPELL_TYPE_BREATH 3
#define MONSPELL_TYPE_SUMMON 4
#define MONSPELL_TYPE_OTHER 5

#define EATER_EXT 36
#define EATER_CHARGE 0x10000L
#define EATER_ROD_CHARGE 0x10L

#define MAX_KUBI 20

#define DETECT_RAD_DEFAULT 30
#define DETECT_RAD_MAP     30
#define DETECT_RAD_ALL     255

/* Monster Spells */
#define MS_SHRIEK         0
#define MS_XXX1           1
#define MS_DISPEL         2
#define MS_ROCKET         3
#define MS_SHOOT          4
#define MS_XXX2           5
#define MS_XXX3           6
#define MS_XXX4           7
#define MS_BR_ACID        8
#define MS_BR_ELEC        9
#define MS_BR_FIRE        10
#define MS_BR_COLD        11
#define MS_BR_POIS        12
#define MS_BR_NETHER      13
#define MS_BR_LITE        14
#define MS_BR_DARK        15
#define MS_BR_CONF        16
#define MS_BR_SOUND       17
#define MS_BR_CHAOS       18
#define MS_BR_DISEN       19
#define MS_BR_NEXUS       20
#define MS_BR_TIME        21
#define MS_BR_INERTIA     22
#define MS_BR_GRAVITY     23
#define MS_BR_SHARDS      24
#define MS_BR_PLASMA      25
#define MS_BR_FORCE       26
#define MS_BR_MANA        27
#define MS_BALL_NUKE      28
#define MS_BR_NUKE        29
#define MS_BALL_CHAOS     30
#define MS_BR_DISI        31
#define MS_BALL_ACID      32
#define MS_BALL_ELEC      33
#define MS_BALL_FIRE      34
#define MS_BALL_COLD      35
#define MS_BALL_POIS      36
#define MS_BALL_NETHER    37
#define MS_BALL_WATER     38
#define MS_BALL_MANA      39
#define MS_BALL_DARK      40
#define MS_DRAIN_MANA     41
#define MS_MIND_BLAST     42
#define MS_BRAIN_SMASH    43
#define MS_CAUSE_1        44
#define MS_CAUSE_2        45
#define MS_CAUSE_3        46
#define MS_CAUSE_4        47
#define MS_BOLT_ACID      48
#define MS_BOLT_ELEC      49
#define MS_BOLT_FIRE      50
#define MS_BOLT_COLD      51
#define MS_STARBURST      52
#define MS_BOLT_NETHER    53
#define MS_BOLT_WATER     54
#define MS_BOLT_MANA      55
#define MS_BOLT_PLASMA    56
#define MS_BOLT_ICE       57
#define MS_MAGIC_MISSILE  58
#define MS_SCARE          59
#define MS_BLIND          60
#define MS_CONF           61
#define MS_SLOW           62
#define MS_SLEEP          63
#define MS_SPEED          64
#define MS_HAND_DOOM      65
#define MS_HEAL           66
#define MS_INVULNER       67
#define MS_BLINK          68
#define MS_TELEPORT       69
#define MS_WORLD          70
#define MS_SPECIAL        71
#define MS_TELE_TO        72
#define MS_TELE_AWAY      73
#define MS_TELE_LEVEL     74
#define MS_PSY_SPEAR      75
#define MS_DARKNESS       76
#define MS_MAKE_TRAP      77
#define MS_FORGET         78
#define MS_RAISE_DEAD     79
#define MS_S_KIN          80
#define MS_S_CYBER        81
#define MS_S_MONSTER      82
#define MS_S_MONSTERS     83
#define MS_S_ANT          84
#define MS_S_SPIDER       85
#define MS_S_HOUND        86
#define MS_S_HYDRA        87
#define MS_S_ANGEL        88
#define MS_S_DEMON        89
#define MS_S_UNDEAD       90
#define MS_S_DRAGON       91
#define MS_S_HI_UNDEAD    92
#define MS_S_HI_DRAGON    93
#define MS_S_AMBERITE     94
#define MS_S_UNIQUE       95


#define MON_BEGGAR        12
#define MON_LEPER         13
#define MON_BLACK_MARKET  14
#define MON_LION_HEART    19
#define MON_GHB           39
#define MON_NOV_PRIEST    45
#define MON_GRIP          53
#define MON_WOLF          54
#define MON_FANG          55
#define MON_LOUSE         69
#define MON_PIRANHA       70
#define MON_COPPER_COINS  85
#define MON_NOV_PALADIN   97
#define MON_GREEN_G       100
#define MON_NOV_PRIEST_G  109
#define MON_SILVER_COINS  117
#define MON_D_ELF         122
#define MON_MANES         128
#define MON_LOST_SOUL     133
#define MON_ROBIN_HOOD    138
#define MON_NOV_PALADIN_G 147
#define MON_PHANTOM_W     152
#define MON_WOUNDED_BEAR  159
#define MON_D_ELF_MAGE    178
#define MON_D_ELF_WARRIOR 182
#define MON_BLUE_HORROR   189
#define MON_GOLD_COINS    195
#define MON_VORPAL_BUNNY  205
#define MON_MASTER_YEEK   224
#define MON_PRIEST        225
#define MON_D_ELF_PRIEST  226
#define MON_TIGER         230
#define MON_MITHRIL_COINS 239
#define MON_DRUID         241
#define MON_PINK_HORROR   242
#define MON_HILL_GIANT    255
#define MON_WERERAT       270
#define MON_UMBER_HULK    283
#define MON_ORC_CAPTAIN   285
#define MON_BERSERKER     293
#define MON_IMP           296
#define MON_SHAGRAT       314
#define MON_GORBAG        315
#define MON_STONE_GIANT   321
#define MON_LIZARD_KING   332
#define MON_WYVERN        334
#define MON_SABRE_TIGER   339
#define MON_D_ELF_LORD    348
#define MON_FIRE_VOR      354
#define MON_WATER_VOR     355
#define MON_ARCH_VILE     357
#define MON_COLD_VOR      358
#define MON_ENERGY_VOR    359
#define MON_IRON_GOLEM    367
#define MON_JADE_MONK     370
#define MON_D_ELF_WARLOCK 375
#define MON_HAGEN         383
#define MON_MENELDOR      384
#define MON_PHANTOM_B     385
#define MON_C_CRAWLER     395
#define MON_XICLOTLAN     396
#define MON_D_ELF_DRUID   400
#define MON_TROLL_PRIEST  403
#define MON_GWAIHIR       410
#define MON_ANGEL         417
#define MON_ADAMANT_COINS 423
#define MON_COLBRAN       435
#define MON_SPIRIT_NAGA   436
#define MON_GACHAPIN      441
#define MON_BASILISK      453
#define MON_ARCHANGEL     456
#define MON_MITHRIL_GOLEM 464
#define MON_THORONDOR     468
#define MON_SHADOW_DRAKE  471
#define MON_GHOST         477
#define MON_OGRE_SHAMAN   479
#define MON_GHOUL_KING    483
#define MON_NINJA         485
#define MON_BICLOPS       490
#define MON_IVORY_MONK    492
#define MON_LOG_MASTER    498
#define MON_ETHER_DRAKE   504
#define MON_GOEMON        505
#define MON_CHERUB        511
#define MON_WATER_ELEM    512
#define MON_JURT          517
#define MON_LICH          518
#define MON_BLOODLETTER   523
#define MON_HALFLING_S    539
#define MON_GRAV_HOUND    540
#define MON_REVENANT      555
#define MON_RAAL          557
#define MON_COLOSSUS      558
#define MON_NIGHTBLADE    564
#define MON_ELDER_THING   569
#define MON_CRYPT_THING   577
#define MON_NEXUS_VOR     587
#define MON_PLASMA_VOR    588
#define MON_TIME_VOR      589
#define MON_M_MH_DRAGON   593
#define MON_MANDOR        598
#define MON_SHIM_VOR      600
#define MON_SERAPH        605
#define MON_BARON_HELL    609
#define MON_KAVLAX        616
#define MON_ETTIN         621
#define MON_VAMPIRE_LORD  623
#define MON_JUBJUB        640
#define MON_G_C_DRAKE     646
#define MON_CLUB_DEMON    648
#define MON_F_ANGEL       652
#define MON_D_ELF_SORC    657
#define MON_MASTER_LICH   658
#define MON_RINALDO       660
#define MON_ARCHON        661
#define MON_UND_BEHOLDER  664
#define MON_IRON_LICH     666
#define MON_JACK_SHADOWS  670
#define MON_LLOIGOR       682
#define MON_DREADMASTER   690
#define MON_DROLEM        691
#define MON_DAWN          693
#define MON_NAZGUL        696
#define MON_SMAUG         697
#define MON_STORMBRINGER  698
#define MON_ULTRA_PALADIN 699
#define MON_G_TITAN       702
#define MON_S_TYRANNO     705
#define MON_FAFNER        712
#define MON_G_BALROG      720
#define MON_TIME_HOUND    725
#define MON_PLASMA_HOUND  726
#define MON_BULLGATES     732
#define MON_SANTACLAUS    733
#define MON_LORD_CHAOS    737
#define MON_TINDALOS      739
#define MON_DEMILICH      742
#define MON_NIGHTCRAWLER  744
#define MON_CHAOS_VOR     751
#define MON_AETHER_VOR    752
#define MON_FUNDIN        762
#define MON_DWORKIN       763
#define MON_NIGHTWALKER   768
#define MON_RAPHAEL       769
#define MON_SARUMAN       771
#define MON_GANDALF       772
#define MON_BRAND         773
#define MON_SHADOWLORD    774
#define MON_ARCHLICH      776
#define MON_JABBERWOCK    778
#define MON_CHAOS_HOUND   779
#define MON_ULT_BEHOLDER  781
#define MON_SHAMBLER      786
#define MON_BLEYS         789
#define MON_FIONA         791
#define MON_SKY_DRAKE     793
#define MON_JULIAN        794
#define MON_BLACK_REAVER  798
#define MON_CAINE         799
#define MON_GERARD        807
#define MON_UNGOLIANT     808
#define MON_ATLACH_NACHA  809
#define MON_Y_GOLONAC     810
#define MON_AETHER_HOUND  811
#define MON_WARP_DEMON    812
#define MON_ERIC          813
#define MON_UNMAKER       815
#define MON_CYBER         816
#define MON_KLING         819
#define MON_CORWIN        820
#define MON_ANGMAR        825
#define MON_CANTORAS      830
#define MON_GODZILLA      832
#define MON_SPAWN_CTH     836
#define MON_SURTUR        837
#define MON_TARRASQUE     838
#define MON_LUNGORTHIN    839
#define MON_CYBER_KING    843
#define MON_WYRM_POWER    847
#define MON_NODENS        849
#define MON_JORMUNGAND    854
#define MON_DESTROYER     855
#define MON_GOTHMOG       856
#define MON_G_CTHULHU     857
#define MON_SAURON        858
#define MON_UNICORN_ORD   859
#define MON_OBERON        860
#define MON_MORGOTH       861
#define MON_SERPENT       862
#define MON_ONE_RING      864
#define MON_CAAWS         866
#define MON_CULVERIN      867
#define MON_EBONY_MONK    870
#define MON_HAGURE        871
#define MON_OROCHI        872
#define MON_ECHIZEN       873
#define MON_SPECT_WYRM    874
#define MON_DIO           878
#define MON_OHMU          879
#define MON_WONG          880
#define MON_ZOMBI_SERPENT 883
#define MON_D_ELF_SHADE   886
#define MON_MANA_HOUND    887
#define MON_VENOM_WYRM    890
#define MON_TROLL_KING    894
#define MON_BAZOOKER      896
#define MON_SHARD_VOR     897
#define MON_ELF_LORD      900
#define MON_MASTER_MYS    916
#define MON_G_MASTER_MYS  917
#define MON_IE            921
#define MON_TSUCHINOKO    926
#define MON_GCWADL        929
#define MON_LOCKE_CLONE   930
#define MON_CALDARM       931
#define MON_BANORLUPART   932
#define MON_BANOR         933
#define MON_LUPART        934
#define MON_KENSHIROU     936
#define MON_W_KNIGHT      938
#define MON_PLANETAR      942
#define MON_SOLAR         943
#define MON_BIKETAL       945
#define MON_RICH          948
#define MON_IKETA         949
#define MON_B_DEATH_SWORD 953
#define MON_YASE_HORSE    955
#define MON_HORSE         956
#define MON_BOTEI         963
#define MON_KAGE          964
#define MON_JAIAN         967
#define MON_BELD          973
#define MON_THAT_BAT      975
#define MON_SHUTEN        979
#define MON_FENGHUANG     988
#define MON_KIRIN         989
#define MON_BAHAMUT       1000
#define MON_SUKE          1001
#define MON_KAKU          1002
#define MON_GHOST_Q       1003
#define MON_PIP           1004
#define MON_A_GOLD        1010
#define MON_A_SILVER      1011
#define MON_ROLENTO       1013
#define MON_RAOU          1018
#define MON_SHURYUUDAN    1023
#define MON_WAHHA         1031
#define MON_DEBBY         1032
#define MON_KNI_TEMPLAR   1037
#define MON_PALADIN       1038
#define MON_CHAMELEON     1040
#define MON_CHAMELEON_K   1041
#define MON_TOPAZ_MONK    1047
#define MON_M_MINDCRAFTER 1056
#define MON_ELDER_VAMPIRE 1058
#define MON_NOBORTA       1059
#define MON_MORI_TROLL    1060
#define MON_BARNEY        1061
#define MON_GROO          1062
#define MON_LOUSY         1063
#define MON_WYRM_SPACE    1064
#define MON_JIZOTAKO      1065
#define MON_TANUKI        1067
#define MON_HATOPOPPO     1083
#define MON_KOGAN         1096

/* Maximum "Nazguls" number */
#define MAX_NAZGUL_NUM 5

#define DO_AUTOPICK       0x01
#define DO_AUTODESTROY    0x02
#define DO_DISPLAY        0x04
#define DONT_AUTOPICK     0x08
#define ITEM_DISPLAY      0x10
#define DO_QUERY_AUTOPICK 0x20


#define MAGIC_GLOVE_REDUCE_MANA 0x0001
#define MAGIC_FAIL_5PERCENT     0x0002
#define MAGIC_GAIN_EXP          0x0004

#define VIRTUE_LARGE 1
#define VIRTUE_SMALL 2

#define SPELL_DD_S 27
#define SPELL_DD_T 13
#define SPELL_SW   22
#define SPELL_KABE 20

#define KNOW_STAT   0x01
#define KNOW_HPRATE 0x02

/*
 * Music songs
 */
#define MUSIC_NONE              0
#define MUSIC_SLOW              1
#define MUSIC_BLESS             2
#define MUSIC_STUN              3
#define MUSIC_L_LIFE            4
#define MUSIC_FEAR              5
#define MUSIC_HERO              6
#define MUSIC_MIND              7
#define MUSIC_STEALTH           8
#define MUSIC_ID                9
#define MUSIC_CONF              10
#define MUSIC_SOUND             11
#define MUSIC_CHARM             12
#define MUSIC_WALL              13
#define MUSIC_RESIST            14
#define MUSIC_SPEED             15
#define MUSIC_DISPEL            16
#define MUSIC_SARUMAN           17
#define MUSIC_QUAKE             18
#define MUSIC_STASIS            19
#define MUSIC_SHERO             20
#define MUSIC_H_LIFE            21
#define MUSIC_INVULN            22
#define MUSIC_PSI               23

#define MUSIC_DETECT            101

#define SINGING_SONG_EFFECT(P_PTR) ((P_PTR)->magic_num1[0])
#define INTERUPTING_SONG_EFFECT(P_PTR) ((P_PTR)->magic_num1[1])
#define SINGING_COUNT(P_PTR) ((P_PTR)->magic_num1[2])
#define SINGING_SONG_ID(P_PTR) ((P_PTR)->magic_num2[0])
#define music_singing(X) ((p_ptr->pclass == CLASS_BARD) && (p_ptr->magic_num1[0] == (X)))
#define music_singing_any() ((p_ptr->pclass == CLASS_BARD) && p_ptr->magic_num1[0])

#define HISSATSU_NONE   0
#define HISSATSU_2      1
#define HISSATSU_3WAY   2
#define HISSATSU_SUTEMI 3
#define HISSATSU_FIRE   4
#define HISSATSU_COLD   5
#define HISSATSU_POISON 6
#define HISSATSU_ELEC   7
#define HISSATSU_NYUSIN 8
#define HISSATSU_FUKI   9
#define HISSATSU_MAJIN  10
#define HISSATSU_BOOMER 11
#define HISSATSU_DRAIN  12
#define HISSATSU_SEKIRYUKA 13
#define HISSATSU_OTAKEBI 14
#define HISSATSU_SHOUGE 15
#define HISSATSU_CONF   16
#define HISSATSU_ISSEN  17
#define HISSATSU_KYUSHO 18
#define HISSATSU_KONSIN 19
#define HISSATSU_HYAKU  20
#define HISSATSU_MINEUCHI 21
#define HISSATSU_MEKIKI 22
#define HISSATSU_ZANMA  23
#define HISSATSU_UNDEAD 24
#define HISSATSU_HAGAN  25
#define HISSATSU_QUAKE  26
#define HISSATSU_COUNTER 27
#define HISSATSU_HARAI  28
#define HISSATSU_3DAN   29
#define HISSATSU_100NIN 30

#define HISSATSU_IAI    100

/*
 *  Special essence id for Weapon smith
 */
#define MIN_SPECIAL_ESSENCE 200

#define ESSENCE_ATTACK        (MIN_SPECIAL_ESSENCE + 0)
#define ESSENCE_AC            (MIN_SPECIAL_ESSENCE + 1)
#define ESSENCE_TMP_RES_ACID  (MIN_SPECIAL_ESSENCE + 2)
#define ESSENCE_TMP_RES_ELEC  (MIN_SPECIAL_ESSENCE + 3)
#define ESSENCE_TMP_RES_FIRE  (MIN_SPECIAL_ESSENCE + 4)
#define ESSENCE_TMP_RES_COLD  (MIN_SPECIAL_ESSENCE + 5)
#define ESSENCE_SH_FIRE       (MIN_SPECIAL_ESSENCE + 6)
#define ESSENCE_SH_ELEC       (MIN_SPECIAL_ESSENCE + 7)
#define ESSENCE_SH_COLD       (MIN_SPECIAL_ESSENCE + 8)
#define ESSENCE_RESISTANCE    (MIN_SPECIAL_ESSENCE + 9)
#define ESSENCE_SUSTAIN       (MIN_SPECIAL_ESSENCE + 10)
#define ESSENCE_SLAY_GLOVE    (MIN_SPECIAL_ESSENCE + 11)


#define DUNGEON_MODE_NONE       0
#define DUNGEON_MODE_AND        1
#define DUNGEON_MODE_NAND       2
#define DUNGEON_MODE_OR         3
#define DUNGEON_MODE_NOR        4

/*** Dungeon type flags -- DG ***/
#define DF1_WINNER              0x00000001L
#define DF1_MAZE                0x00000002L
#define DF1_SMALLEST            0x00000004L
#define DF1_BEGINNER            0x00000008L
#define DF1_BIG                 0x00000010L
#define DF1_NO_DOORS            0x00000020L
#define DF1_WATER_RIVER         0x00000040L
#define DF1_LAVA_RIVER          0x00000080L
#define DF1_CURTAIN             0x00000100L
#define DF1_GLASS_DOOR          0x00000200L
#define DF1_CAVE                0x00000400L
#define DF1_CAVERN              0x00000800L
#define DF1_ARCADE              0x00001000L
#define DF1_XXX13               0x00002000L
#define DF1_XXX14               0x00004000L
#define DF1_XXX15               0x00008000L
#define DF1_FORGET              0x00010000L
#define DF1_LAKE_WATER          0x00020000L
#define DF1_LAKE_LAVA           0x00040000L
#define DF1_LAKE_RUBBLE         0x00080000L
#define DF1_LAKE_TREE           0x00100000L
#define DF1_NO_VAULT            0x00200000L
#define DF1_ARENA               0x00400000L
#define DF1_DESTROY             0x00800000L
#define DF1_GLASS_ROOM          0x01000000L
#define DF1_NO_CAVE             0x02000000L
#define DF1_NO_MAGIC            0x04000000L
#define DF1_NO_MELEE            0x08000000L
#define DF1_CHAMELEON           0x10000000L
#define DF1_DARKNESS            0x20000000L
#define DF1_XXX30               0x40000000L
#define DF1_XXX31               0x80000000L

#define DF1_LAKE_MASK (DF1_LAKE_WATER | DF1_LAKE_LAVA | DF1_LAKE_RUBBLE | DF1_LAKE_TREE)

#define DUNGEON_ANGBAND  1
#define DUNGEON_GALGALS  2
#define DUNGEON_ORC      3
#define DUNGEON_MAZE     4
#define DUNGEON_DRAGON   5
#define DUNGEON_GRAVE    6
#define DUNGEON_WOOD     7
#define DUNGEON_VOLCANO  8
#define DUNGEON_HELL     9
#define DUNGEON_HEAVEN   10
#define DUNGEON_OCEAN    11
#define DUNGEON_CASTLE   12
#define DUNGEON_CTH      13
#define DUNGEON_MOUNTAIN 14
#define DUNGEON_GOLD     15
#define DUNGEON_NO_MAGIC 16
#define DUNGEON_NO_MELEE 17
#define DUNGEON_CHAMELEON 18
#define DUNGEON_DARKNESS 19


#define DUNGEON_FEAT_PROB_NUM 3


/*
 * Flags for change floor mode
 */
#define CFM_UP        	 0x0001  /* Move up */
#define CFM_DOWN      	 0x0002  /* Move down */
#define CFM_LONG_STAIRS  0x0004  /* Randomly occurred long stairs/shaft */
#define CFM_XXX  	 0x0008  /* XXX */
#define CFM_SHAFT     	 0x0010  /* Shaft */
#define CFM_RAND_PLACE   0x0020  /* Arrive at random grid */
#define CFM_RAND_CONNECT 0x0040  /* Connect with random stairs */
#define CFM_SAVE_FLOORS  0x0080  /* Save floors */
#define CFM_NO_RETURN    0x0100  /* Flee from random quest etc... */
#define CFM_FIRST_FLOOR  0x0200  /* Create exit from the dungeon */


/*
 * Flags for save/load temporal saved floor file
 */
#define SLF_SECOND     	 0x0001  /* Called from another save/load function */
#define SLF_NO_KILL      0x0002  /* Don't kill temporal files */


/*
 * Flags for wr_item()/rd_item()
 */
#define SAVE_ITEM_PVAL         0x00000001
#define SAVE_ITEM_DISCOUNT     0x00000002
#define SAVE_ITEM_NUMBER       0x00000004
#define SAVE_ITEM_NAME1        0x00000008
#define SAVE_ITEM_NAME2        0x00000010
#define SAVE_ITEM_TIMEOUT      0x00000020
#define SAVE_ITEM_TO_H         0x00000040
#define SAVE_ITEM_TO_D         0x00000080
#define SAVE_ITEM_TO_A         0x00000100
#define SAVE_ITEM_AC           0x00000200
#define SAVE_ITEM_DD           0x00000400
#define SAVE_ITEM_DS           0x00000800
#define SAVE_ITEM_IDENT        0x00001000
#define SAVE_ITEM_MARKED       0x00002000
#define SAVE_ITEM_ART_FLAGS0   0x00004000
#define SAVE_ITEM_ART_FLAGS1   0x00008000
#define SAVE_ITEM_ART_FLAGS2   0x00010000
#define SAVE_ITEM_ART_FLAGS3   0x00020000
#define SAVE_ITEM_CURSE_FLAGS  0x00040000
#define SAVE_ITEM_HELD_M_IDX   0x00080000
#define SAVE_ITEM_XTRA1        0x00100000
#define SAVE_ITEM_XTRA2        0x00200000
#define SAVE_ITEM_XTRA3        0x00400000
#define SAVE_ITEM_XTRA4        0x00800000
#define SAVE_ITEM_XTRA5        0x01000000
#define SAVE_ITEM_FEELING      0x02000000
#define SAVE_ITEM_INSCRIPTION  0x04000000
#define SAVE_ITEM_ART_NAME     0x08000000
#define SAVE_ITEM_ART_FLAGS4   0x10000000


/*
 * Flags for wr_monster()/rd_monster()
 */
#define SAVE_MON_AP_R_IDX     0x00000001
#define SAVE_MON_SUB_ALIGN    0x00000002
#define SAVE_MON_CSLEEP       0x00000004
#define SAVE_MON_FAST         0x00000008
#define SAVE_MON_SLOW         0x00000010
#define SAVE_MON_STUNNED      0x00000020
#define SAVE_MON_CONFUSED     0x00000040
#define SAVE_MON_MONFEAR      0x00000080
#define SAVE_MON_TARGET_Y     0x00000100
#define SAVE_MON_TARGET_X     0x00000200
#define SAVE_MON_INVULNER     0x00000400
#define SAVE_MON_SMART        0x00000800
#define SAVE_MON_EXP          0x00001000
#define SAVE_MON_MFLAG2       0x00002000
#define SAVE_MON_NICKNAME     0x00004000
#define SAVE_MON_PARENT       0x00008000


/*
 * Constant for kinds of mimic
 */
#define MIMIC_NONE       0
#define MIMIC_DEMON      1
#define MIMIC_DEMON_LORD 2
#define MIMIC_VAMPIRE    3


#define MIMIC_FLAGS choice
#define MIMIC_IS_NONLIVING 0x00000001
#define MIMIC_IS_DEMON     0x00000002
#define MIMIC_IS_UNDEAD    0x00000004


#define prace_is_(A) (!p_ptr->mimic_form && (p_ptr->prace == A))

/* Sub-alignment flags for neutral monsters */
#define SUB_ALIGN_NEUTRAL 0x0000
#define SUB_ALIGN_EVIL    0x0001
#define SUB_ALIGN_GOOD    0x0002

/* Temporary flags macro */
#define IS_FAST() (p_ptr->fast || music_singing(MUSIC_SPEED) || music_singing(MUSIC_SHERO))
#define IS_INVULN() (p_ptr->invuln || music_singing(MUSIC_INVULN))
#define IS_HERO() (p_ptr->hero || music_singing(MUSIC_HERO) || music_singing(MUSIC_SHERO))
#define IS_BLESSED() (p_ptr->blessed || music_singing(MUSIC_BLESS) || hex_spelling(HEX_BLESS))
#define IS_OPPOSE_ACID() (p_ptr->oppose_acid || music_singing(MUSIC_RESIST) || (p_ptr->special_defense & KATA_MUSOU))
#define IS_OPPOSE_ELEC() (p_ptr->oppose_elec || music_singing(MUSIC_RESIST) || (p_ptr->special_defense & KATA_MUSOU))
#define IS_OPPOSE_FIRE() (p_ptr->oppose_fire || music_singing(MUSIC_RESIST) || (p_ptr->special_defense & KATA_MUSOU))
#define IS_OPPOSE_COLD() (p_ptr->oppose_cold || music_singing(MUSIC_RESIST) || (p_ptr->special_defense & KATA_MUSOU))
#define IS_OPPOSE_POIS() (p_ptr->oppose_pois || music_singing(MUSIC_RESIST) || (p_ptr->special_defense & KATA_MUSOU))
#define IS_TIM_ESP() (p_ptr->tim_esp || music_singing(MUSIC_MIND) || (p_ptr->concent >= CONCENT_TELE_THRESHOLD))
#define IS_TIM_STEALTH() (p_ptr->tim_stealth || music_singing(MUSIC_STEALTH))

#define P_PTR_KI (p_ptr->magic_num1[0])

/* Multishadow effects is determined by turn */
#define CHECK_MULTISHADOW() (p_ptr->multishadow && (turn & 1))

/* Is "teleport level" ineffective to this target? */
#define TELE_LEVEL_IS_INEFF(TARGET) \
	(p_ptr->inside_arena || p_ptr->inside_battle || \
	 (p_ptr->inside_quest && !random_quest_number(dun_level)) || \
	 (((TARGET) <= 0) && (quest_number(dun_level) || (dun_level >= d_info[dungeon_type].maxdepth)) && \
	  (dun_level >= 1) && ironman_downward))


/*
 * Max numbers of macro trigger names
 */
#define MAX_MACRO_MOD 12
#define MAX_MACRO_TRIG 200 /*!< 登録を許すマクロ（トリガー）の最大数 */

/* Max size of screen dump buffer */
#define SCREEN_BUF_MAX_SIZE (4 * 65536)


/*
 * Special key code used for inkey_special()
 */
#define SKEY_MOD_MASK     0x0f00
#define SKEY_MOD_SHIFT    0x0100
#define SKEY_MOD_CONTROL  0x0200

#define SKEY_MASK         0xf000
#define SKEY_DOWN   	  0xf001
#define SKEY_LEFT   	  0xf002
#define SKEY_RIGHT  	  0xf003
#define SKEY_UP     	  0xf004
#define SKEY_PGUP   	  0xf005
#define SKEY_PGDOWN 	  0xf006
#define SKEY_TOP    	  0xf007
#define SKEY_BOTTOM 	  0xf008

/*
 * Bit flags for move_player_effect()
 */
#define MPE_STAYING       0x00000001
#define MPE_FORGET_FLOW   0x00000002
#define MPE_HANDLE_STUFF  0x00000004
#define MPE_ENERGY_USE    0x00000008
#define MPE_DONT_PICKUP   0x00000010
#define MPE_DO_PICKUP     0x00000020
#define MPE_BREAK_TRAP    0x00000040
#define MPE_DONT_SWAP_MON 0x00000080


#define MTIMED_CSLEEP   0 /* Monster is sleeping */
#define MTIMED_FAST     1 /* Monster is temporarily fast */
#define MTIMED_SLOW     2 /* Monster is temporarily slow */
#define MTIMED_STUNNED  3 /* Monster is stunned */
#define MTIMED_CONFUSED 4 /* Monster is confused */
#define MTIMED_MONFEAR  5 /* Monster is afraid */
#define MTIMED_INVULNER 6 /* Monster is temporarily invulnerable */

#define MAX_MTIMED      7

#define MON_CSLEEP(M_PTR)   ((M_PTR)->mtimed[MTIMED_CSLEEP])
#define MON_FAST(M_PTR)     ((M_PTR)->mtimed[MTIMED_FAST])
#define MON_SLOW(M_PTR)     ((M_PTR)->mtimed[MTIMED_SLOW])
#define MON_STUNNED(M_PTR)  ((M_PTR)->mtimed[MTIMED_STUNNED])
#define MON_CONFUSED(M_PTR) ((M_PTR)->mtimed[MTIMED_CONFUSED])
#define MON_MONFEAR(M_PTR)  ((M_PTR)->mtimed[MTIMED_MONFEAR])
#define MON_INVULNER(M_PTR) ((M_PTR)->mtimed[MTIMED_INVULNER])

/*
 * Bit flags for screen_object()
 */
#define SCROBJ_FAKE_OBJECT  0x00000001
#define SCROBJ_FORCE_DETAIL 0x00000002

/*
 * For travel command (auto run)
 */
#define TRAVEL

/* Sniper */
#define SP_NONE          0
#define SP_LITE          1
#define SP_AWAY          2
#define SP_FIRE          3
#define SP_KILL_WALL     4
#define SP_COLD          5
#define SP_KILL_TRAP     6
#define SP_ELEC          7
#define SP_PIERCE        8
#define SP_RUSH          9
#define SP_DOUBLE        10
#define SP_EXPLODE       11
#define SP_EVILNESS      12
#define SP_HOLYNESS      13
#define SP_FINAL         14
#define SP_NEEDLE        15

#define CONCENT_RADAR_THRESHOLD 2
#define CONCENT_TELE_THRESHOLD  5

/* Hex */
#define hex_spelling_any() \
	((p_ptr->realm1 == REALM_HEX) && (p_ptr->magic_num1[0]))
#define hex_spelling(X) \
	((p_ptr->realm1 == REALM_HEX) && (p_ptr->magic_num1[0] & (1L << (X))))
#define CASTING_HEX_FLAGS(P_PTR) ((P_PTR)->magic_num1[0])
#define CASTING_HEX_NUM(P_PTR) ((P_PTR)->magic_num2[0])
#define HEX_REVENGE_POWER(P_PTR) ((P_PTR)->magic_num1[2])
#define HEX_REVENGE_TURN(P_PTR) ((P_PTR)->magic_num2[2])
#define HEX_REVENGE_TYPE(P_PTR) ((P_PTR)->magic_num2[1])

/* 1st book */
#define HEX_BLESS             0
#define HEX_CURE_LIGHT        1
#define HEX_DEMON_AURA        2
#define HEX_STINKING_MIST     3
#define HEX_XTRA_MIGHT        4
#define HEX_CURSE_WEAPON      5
#define HEX_DETECT_EVIL       6
#define HEX_PATIENCE          7
/* 2nd book */
#define HEX_ICE_ARMOR         8
#define HEX_CURE_SERIOUS      9
#define HEX_INHAIL           10
#define HEX_VAMP_MIST        11
#define HEX_RUNESWORD        12
#define HEX_CONFUSION        13
#define HEX_BUILDING         14
#define HEX_ANTI_TELE        15
/* 3rd book */
#define HEX_SHOCK_CLOAK      16
#define HEX_CURE_CRITICAL    17
#define HEX_RECHARGE         18
#define HEX_RAISE_DEAD       19
#define HEX_CURSE_ARMOUR     20
#define HEX_SHADOW_CLOAK     21
#define HEX_PAIN_TO_MANA     22
#define HEX_EYE_FOR_EYE      23
/* 4th book */
#define HEX_ANTI_MULTI       24
#define HEX_RESTORE          25
#define HEX_DRAIN_CURSE      26
#define HEX_VAMP_BLADE       27
#define HEX_STUN_MONSTERS    28
#define HEX_SHADOW_MOVE      29
#define HEX_ANTI_MAGIC       30
#define HEX_REVENGE          31

/*
  Language selection macro
*/
#ifdef JP
#define _(JAPANESE,ENGLISH) (JAPANESE)
#else
#define _(JAPANESE,ENGLISH) (ENGLISH)
#endif

/* Lite flag macro */
#define have_lite_flag(ARRAY) \
	(have_flag(ARRAY, TR_LITE_1) || have_flag(ARRAY, TR_LITE_2) || have_flag(ARRAY, TR_LITE_3))

#define have_dark_flag(ARRAY) \
	(have_flag(ARRAY, TR_LITE_M1) || have_flag(ARRAY, TR_LITE_M2) || have_flag(ARRAY, TR_LITE_M3))

/* Spell Type flag */
#define MONSTER_TO_PLAYER     0x01
#define MONSTER_TO_MONSTER    0x02

/* summoning number */
#define S_NUM_6     (easy_band ? 2 : 6)
#define S_NUM_4     (easy_band ? 1 : 4)

/* monster spell number */
#define RF4_SPELL_START 32 * 3
#define RF5_SPELL_START 32 * 4
#define RF6_SPELL_START 32 * 5

#define RF4_SPELL_SIZE 32
#define RF5_SPELL_SIZE 32
#define RF6_SPELL_SIZE 32

/* Spell Damage Calc Flag*/
#define DAM_ROLL 1
#define DAM_MAX 2
#define DAM_MIN 3
#define DICE_NUM 4
#define DICE_SIDE 5
#define DICE_MULT 6
#define DICE_DIV 7
#define BASE_DAM 8

/* Cheat Info Type */
#define CHEAT_OBJECT 0
#define CHEAT_MONSTER 1
#define CHEAT_DUNGEON 2
#define CHEAT_MISC 3

#define COMMAND_ARG_REST_UNTIL_DONE -2   /*!<休憩コマンド引数 … 必要な分だけ回復 */
#define COMMAND_ARG_REST_FULL_HEALING -1 /*!<休憩コマンド引数 … HPとMPが全回復するまで */
