﻿/*!
 * @file tables.c
 * @brief ゲーム情報テーブル / Angband Tables
 * @date 2014/07/23
 * @author
 * <pre>
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 * This software may be copied and distributed for educational, research,
 * and not for profit purposes provided that this copyright and statement
 * are included in all such copies.  Other copyrights may also apply.
 * </pre>
 */


#include "angband.h"

/*!
 * キーパッドの方向を南から反時計回り順に列挙 / Global array for looping through the "keypad directions"
 */
const s16b ddd[9] =
{ 2, 8, 6, 4, 3, 1, 9, 7, 5 };

/*!
 * dddで定義した順にベクトルのX軸成分を定義 / Global arrays for converting "keypad direction" into offsets
 */
const s16b ddx[10] =
{ 0, -1, 0, 1, -1, 0, 1, -1, 0, 1 };

/*!
 * dddで定義した順にベクトルのY軸成分を定義 / Global arrays for converting "keypad direction" into offsets
 */
const s16b ddy[10] =
{ 0, 1, 1, 1, 0, 0, 0, -1, -1, -1 };

/*!
 * ddd越しにベクトルのX軸成分を定義 / Global arrays for optimizing "ddx[ddd[i]]" and "ddy[ddd[i]]"
 */
const s16b ddx_ddd[9] =
{ 0, 0, 1, -1, 1, -1, 1, -1, 0 };

/*!
 * ddd越しにベクトルのY軸成分を定義 / Global arrays for optimizing "ddx[ddd[i]]" and "ddy[ddd[i]]"
 */
const s16b ddy_ddd[9] =
{ 1, -1, 0, 0, 1, 1, -1, -1, 0 };


/*!
 * キーパッドの円環状方向配列 / Circular keypad direction array
 */
const s16b cdd[8] =
{ 2, 3, 6, 9, 8, 7, 4, 1 };

/*!
 * cdd越しにベクトルのX軸成分を定義 / Global arrays for optimizing "ddx[cdd[i]]" and "ddy[cdd[i]]"
 */
const s16b ddx_cdd[8] =
{ 0, 1, 1, 1, 0, -1, -1, -1 };

/*!
 * cdd越しにベクトルのY軸成分を定義 / Global arrays for optimizing "ddx[cdd[i]]" and "ddy[cdd[i]]"
 */
const s16b ddy_cdd[8] =
{ 1, 1, 0, -1, -1, -1, 0, 1 };



/*!
 * 10進数から16進数への変換テーブル /
 * Global array for converting numbers to uppercase hecidecimal digit
 * This array can also be used to convert a number to an octal digit
 */
const char hexsym[16] =
{
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};


/*!
 * 選択処理用キーコード /
 * Global array for converting numbers to a logical list symbol
 */
const char listsym[] =
{
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
	'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
	'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
	'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
	'\0'
};


/*!
 * スクリーン表示色キャラクタ /
 * Encode the screen colors
 */
const cptr color_char = "dwsorgbuDWvyRGBU";


/*!
 * 知力/賢さによるレベル毎の習得可能魔法数テーブル
 * Stat Table (INT/WIS) -- Number of half-spells per level
 */
const byte adj_mag_study[] =
{
	0	/* 3 */,
	0	/* 4 */,
	0	/* 5 */,
	0	/* 6 */,
	0	/* 7 */,
	1	/* 8 */,
	1	/* 9 */,
	1	/* 10 */,
	1	/* 11 */,
	2	/* 12 */,
	2	/* 13 */,
	2	/* 14 */,
	2	/* 15 */,
	2	/* 16 */,
	2	/* 17 */,
	2	/* 18/00-18/09 */,
	2	/* 18/10-18/19 */,
	2	/* 18/20-18/29 */,
	2	/* 18/30-18/39 */,
	2	/* 18/40-18/49 */,
	3	/* 18/50-18/59 */,
	3	/* 18/60-18/69 */,
	3	/* 18/70-18/79 */,
	3	/* 18/80-18/89 */,
	4	/* 18/90-18/99 */,
	4	/* 18/100-18/109 */,
	4	/* 18/110-18/119 */,
	5	/* 18/120-18/129 */,
	5	/* 18/130-18/139 */,
	5	/* 18/140-18/149 */,
	5	/* 18/150-18/159 */,
	5	/* 18/160-18/169 */,
	5	/* 18/170-18/179 */,
	5	/* 18/180-18/189 */,
	5	/* 18/190-18/199 */,
	5	/* 18/200-18/209 */,
	6	/* 18/210-18/219 */,
	6	/* 18/220+ */
};


/*!
 * 知力/賢さによるMP修正テーブル
 * Stat Table (INT/WIS) -- extra 1/4-mana-points per level
 */
const byte adj_mag_mana[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	2       /* 9 */,
	3       /* 10 */,
	4       /* 11 */,
	5       /* 12 */,
	5       /* 13 */,
	6       /* 14 */,
	7       /* 15 */,
	8       /* 16 */,
	9       /* 17 */,
	10      /* 18/00-18/09 */,
	11      /* 18/10-18/19 */,
	11      /* 18/20-18/29 */,
	12      /* 18/30-18/39 */,
	12      /* 18/40-18/49 */,
	13      /* 18/50-18/59 */,
	14      /* 18/60-18/69 */,
	15      /* 18/70-18/79 */,
	16      /* 18/80-18/89 */,
	17      /* 18/90-18/99 */,
	18      /* 18/100-18/109 */,
	19      /* 18/110-18/119 */,
	20      /* 18/120-18/129 */,
	21      /* 18/130-18/139 */,
	22      /* 18/140-18/149 */,
	23      /* 18/150-18/159 */,
	24      /* 18/160-18/169 */,
	25      /* 18/170-18/179 */,
	26      /* 18/180-18/189 */,
	27      /* 18/190-18/199 */,
	28      /* 18/200-18/209 */,
	29      /* 18/210-18/219 */,
	30      /* 18/220+ */
};


/*!
 * 知力/賢さによる最低魔法失敗率テーブル
 * Stat Table (INT/WIS) -- Minimum failure rate (percentage)
 */
const byte adj_mag_fail[] =
{
	99      /* 3 */,
	99      /* 4 */,
	99      /* 5 */,
	99      /* 6 */,
	99      /* 7 */,
	50      /* 8 */,
	30      /* 9 */,
	20      /* 10 */,
	15      /* 11 */,
	12      /* 12 */,
	11      /* 13 */,
	10      /* 14 */,
	9       /* 15 */,
	8       /* 16 */,
	7       /* 17 */,
	6       /* 18/00-18/09 */,
	6       /* 18/10-18/19 */,
	5       /* 18/20-18/29 */,
	5       /* 18/30-18/39 */,
	5       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	4       /* 18/60-18/69 */,
	4       /* 18/70-18/79 */,
	4       /* 18/80-18/89 */,
	3       /* 18/90-18/99 */,
	3       /* 18/100-18/109 */,
	2       /* 18/110-18/119 */,
	2       /* 18/120-18/129 */,
	2       /* 18/130-18/139 */,
	2       /* 18/140-18/149 */,
	1       /* 18/150-18/159 */,
	1       /* 18/160-18/169 */,
	1       /* 18/170-18/179 */,
	1       /* 18/180-18/189 */,
	1       /* 18/190-18/199 */,
	0       /* 18/200-18/209 */,
	0       /* 18/210-18/219 */,
	0       /* 18/220+ */
};


/*!
 * 知力/賢さによる魔法失敗率修正テーブル
 * Stat Table (INT/WIS) -- Various things
 */
const byte adj_mag_stat[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	3       /* 18/30-18/39 */,
	3       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	4       /* 18/60-18/69 */,
	5       /* 18/70-18/79 */,
	6       /* 18/80-18/89 */,
	7       /* 18/90-18/99 */,
	8       /* 18/100-18/109 */,
	9       /* 18/110-18/119 */,
	10      /* 18/120-18/129 */,
	11      /* 18/130-18/139 */,
	12      /* 18/140-18/149 */,
	13      /* 18/150-18/159 */,
	14      /* 18/160-18/169 */,
	15      /* 18/170-18/179 */,
	16      /* 18/180-18/189 */,
	17      /* 18/190-18/199 */,
	18      /* 18/200-18/209 */,
	19      /* 18/210-18/219 */,
	20      /* 18/220+ */
};


/*!
 * 魅力による店での取引修正テーブル
 * Stat Table (CHR) -- payment percentages
 */
const byte adj_chr_gold[] =
{
	130     /* 3 */,
	125     /* 4 */,
	122     /* 5 */,
	120     /* 6 */,
	118     /* 7 */,
	116     /* 8 */,
	114     /* 9 */,
	112     /* 10 */,
	110     /* 11 */,
	108     /* 12 */,
	106     /* 13 */,
	104     /* 14 */,
	103     /* 15 */,
	102     /* 16 */,
	101     /* 17 */,
	100     /* 18/00-18/09 */,
	99      /* 18/10-18/19 */,
	98      /* 18/20-18/29 */,
	97      /* 18/30-18/39 */,
	96      /* 18/40-18/49 */,
	95      /* 18/50-18/59 */,
	94      /* 18/60-18/69 */,
	93      /* 18/70-18/79 */,
	92      /* 18/80-18/89 */,
	91      /* 18/90-18/99 */,
	90      /* 18/100-18/109 */,
	89      /* 18/110-18/119 */,
	88      /* 18/120-18/129 */,
	87      /* 18/130-18/139 */,
	86      /* 18/140-18/149 */,
	85      /* 18/150-18/159 */,
	84      /* 18/160-18/169 */,
	83      /* 18/170-18/179 */,
	82      /* 18/180-18/189 */,
	81      /* 18/190-18/199 */,
	80      /* 18/200-18/209 */,
	79      /* 18/210-18/219 */,
	78      /* 18/220+ */
};


/*!
 * 知力による魔道具使用修正テーブル
 * Stat Table (INT) -- Magic devices
 */
const byte adj_int_dev[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	4       /* 18/20-18/29 */,
	4       /* 18/30-18/39 */,
	5       /* 18/40-18/49 */,
	5       /* 18/50-18/59 */,
	6       /* 18/60-18/69 */,
	6       /* 18/70-18/79 */,
	7       /* 18/80-18/89 */,
	7       /* 18/90-18/99 */,
	8       /* 18/100-18/109 */,
	9       /* 18/110-18/119 */,
	10      /* 18/120-18/129 */,
	11      /* 18/130-18/139 */,
	12      /* 18/140-18/149 */,
	13      /* 18/150-18/159 */,
	14      /* 18/160-18/169 */,
	15      /* 18/170-18/179 */,
	16      /* 18/180-18/189 */,
	17      /* 18/190-18/199 */,
	18      /* 18/200-18/209 */,
	19      /* 18/210-18/219 */,
	20      /* 18/220+ */
};


/*!
 * 賢さによる魔法防御修正テーブル
 * Stat Table (WIS) -- Saving throw
 */
const byte adj_wis_sav[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	3       /* 18/30-18/39 */,
	3       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	4       /* 18/60-18/69 */,
	5       /* 18/70-18/79 */,
	5       /* 18/80-18/89 */,
	6       /* 18/90-18/99 */,
	7       /* 18/100-18/109 */,
	8       /* 18/110-18/119 */,
	9       /* 18/120-18/129 */,
	10      /* 18/130-18/139 */,
	11      /* 18/140-18/149 */,
	12      /* 18/150-18/159 */,
	13      /* 18/160-18/169 */,
	14      /* 18/170-18/179 */,
	15      /* 18/180-18/189 */,
	16      /* 18/190-18/199 */,
	17      /* 18/200-18/209 */,
	18      /* 18/210-18/219 */,
	19      /* 18/220+ */
};


/*!
 * 器用さによるトラップ解除修正テーブル
 * Stat Table (DEX) -- disarming
 */
const byte adj_dex_dis[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	0       /* 8 */,
	0       /* 9 */,
	0       /* 10 */,
	0       /* 11 */,
	0       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	1       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	4       /* 18/00-18/09 */,
	4       /* 18/10-18/19 */,
	4       /* 18/20-18/29 */,
	4       /* 18/30-18/39 */,
	5       /* 18/40-18/49 */,
	5       /* 18/50-18/59 */,
	5       /* 18/60-18/69 */,
	6       /* 18/70-18/79 */,
	6       /* 18/80-18/89 */,
	7       /* 18/90-18/99 */,
	8       /* 18/100-18/109 */,
	8       /* 18/110-18/119 */,
	8       /* 18/120-18/129 */,
	8       /* 18/130-18/139 */,
	8       /* 18/140-18/149 */,
	9       /* 18/150-18/159 */,
	9       /* 18/160-18/169 */,
	9       /* 18/170-18/179 */,
	9       /* 18/180-18/189 */,
	9       /* 18/190-18/199 */,
	10      /* 18/200-18/209 */,
	10      /* 18/210-18/219 */,
	10      /* 18/220+ */
};


/*!
 * 知力によるトラップ解除修正テーブル
 * Stat Table (INT) -- disarming
 */
const byte adj_int_dis[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	4       /* 18/30-18/39 */,
	4       /* 18/40-18/49 */,
	5       /* 18/50-18/59 */,
	6       /* 18/60-18/69 */,
	7       /* 18/70-18/79 */,
	8       /* 18/80-18/89 */,
	9       /* 18/90-18/99 */,
	10      /* 18/100-18/109 */,
	10      /* 18/110-18/119 */,
	11      /* 18/120-18/129 */,
	12      /* 18/130-18/139 */,
	13      /* 18/140-18/149 */,
	14      /* 18/150-18/159 */,
	15      /* 18/160-18/169 */,
	16      /* 18/170-18/179 */,
	17      /* 18/180-18/189 */,
	18      /* 18/190-18/199 */,
	19      /* 18/200-18/209 */,
	19      /* 18/210-18/219 */,
	20      /* 18/220+ */
};


/*!
 * 器用さによるAC修正テーブル
 * Stat Table (DEX) -- bonus to ac (plus 128)
 */
const byte adj_dex_ta[] =
{
	128 + -4    /*  3 */,
	128 + -3    /*  4 */,
	128 + -2    /*  5 */,
	128 + -1    /*  6 */,
	128 + 0     /*  7 */,
	128 + 0     /*  8 */,
	128 + 0     /*  9 */,
	128 + 0     /* 10 */,
	128 + 0     /* 11 */,
	128 + 0     /* 12 */,
	128 + 0     /* 13 */,
	128 + 0     /* 14 */,
	128 + 1     /* 15 */,
	128 + 1     /* 16 */,
	128 + 1     /* 17 */,
	128 + 2     /* 18/00-18/09 */,
	128 + 2     /* 18/10-18/19 */,
	128 + 2     /* 18/20-18/29 */,
	128 + 2     /* 18/30-18/39 */,
	128 + 2     /* 18/40-18/49 */,
	128 + 3     /* 18/50-18/59 */,
	128 + 3     /* 18/60-18/69 */,
	128 + 3     /* 18/70-18/79 */,
	128 + 4     /* 18/80-18/89 */,
	128 + 5     /* 18/90-18/99 */,
	128 + 6     /* 18/100-18/109 */,
	128 + 7     /* 18/110-18/119 */,
	128 + 8     /* 18/120-18/129 */,
	128 + 9     /* 18/130-18/139 */,
	128 + 9     /* 18/140-18/149 */,
	128 + 10    /* 18/150-18/159 */,
	128 + 11    /* 18/160-18/169 */,
	128 + 12    /* 18/170-18/179 */,
	128 + 13    /* 18/180-18/189 */,
	128 + 14    /* 18/190-18/199 */,
	128 + 15    /* 18/200-18/209 */,
	128 + 15    /* 18/210-18/219 */,
	128 + 16    /* 18/220+ */
};


/*!
 * 腕力によるダメージ修正テーブル
 * Stat Table (STR) -- bonus to dam (plus 128)
 */
const byte adj_str_td[] =
{
	128 + -2    /*  3 */,
	128 + -2    /*  4 */,
	128 + -1    /*  5 */,
	128 + -1    /*  6 */,
	128 + 0     /*  7 */,
	128 + 0     /*  8 */,
	128 + 0     /*  9 */,
	128 + 0     /* 10 */,
	128 + 0     /* 11 */,
	128 + 0     /* 12 */,
	128 + 0     /* 13 */,
	128 + 0     /* 14 */,
	128 + 0     /* 15 */,
	128 + 1     /* 16 */,
	128 + 2     /* 17 */,
	128 + 2     /* 18/00-18/09 */,
	128 + 2     /* 18/10-18/19 */,
	128 + 3     /* 18/20-18/29 */,
	128 + 3     /* 18/30-18/39 */,
	128 + 3     /* 18/40-18/49 */,
	128 + 3     /* 18/50-18/59 */,
	128 + 3     /* 18/60-18/69 */,
	128 + 4     /* 18/70-18/79 */,
	128 + 5     /* 18/80-18/89 */,
	128 + 5     /* 18/90-18/99 */,
	128 + 6     /* 18/100-18/109 */,
	128 + 7     /* 18/110-18/119 */,
	128 + 8     /* 18/120-18/129 */,
	128 + 9     /* 18/130-18/139 */,
	128 + 10    /* 18/140-18/149 */,
	128 + 11    /* 18/150-18/159 */,
	128 + 12    /* 18/160-18/169 */,
	128 + 13    /* 18/170-18/179 */,
	128 + 14    /* 18/180-18/189 */,
	128 + 15    /* 18/190-18/199 */,
	128 + 16    /* 18/200-18/209 */,
	128 + 18    /* 18/210-18/219 */,
	128 + 20    /* 18/220+ */
};


/*!
 * 器用度による命中修正テーブル
 * Stat Table (DEX) -- bonus to hit (plus 128)
 */
const byte adj_dex_th[] =
{
	128 + -3        /* 3 */,
	128 + -2        /* 4 */,
	128 + -2        /* 5 */,
	128 + -1        /* 6 */,
	128 + -1        /* 7 */,
	128 + 0 /* 8 */,
	128 + 0 /* 9 */,
	128 + 0 /* 10 */,
	128 + 0 /* 11 */,
	128 + 0 /* 12 */,
	128 + 0 /* 13 */,
	128 + 0 /* 14 */,
	128 + 0 /* 15 */,
	128 + 1 /* 16 */,
	128 + 2 /* 17 */,
	128 + 3 /* 18/00-18/09 */,
	128 + 3 /* 18/10-18/19 */,
	128 + 3 /* 18/20-18/29 */,
	128 + 3 /* 18/30-18/39 */,
	128 + 3 /* 18/40-18/49 */,
	128 + 4 /* 18/50-18/59 */,
	128 + 4 /* 18/60-18/69 */,
	128 + 4 /* 18/70-18/79 */,
	128 + 4 /* 18/80-18/89 */,
	128 + 5 /* 18/90-18/99 */,
	128 + 6 /* 18/100-18/109 */,
	128 + 7 /* 18/110-18/119 */,
	128 + 8 /* 18/120-18/129 */,
	128 + 9 /* 18/130-18/139 */,
	128 + 9 /* 18/140-18/149 */,
	128 + 10        /* 18/150-18/159 */,
	128 + 11        /* 18/160-18/169 */,
	128 + 12        /* 18/170-18/179 */,
	128 + 13        /* 18/180-18/189 */,
	128 + 14        /* 18/190-18/199 */,
	128 + 15        /* 18/200-18/209 */,
	128 + 15        /* 18/210-18/219 */,
	128 + 16        /* 18/220+ */
};


/*!
 * 腕力による命中修正テーブル
 * Stat Table (STR) -- bonus to hit (plus 128)
 */
const byte adj_str_th[] =
{
	128 + -3        /* 3 */,
	128 + -2        /* 4 */,
	128 + -1        /* 5 */,
	128 + -1        /* 6 */,
	128 + 0 /* 7 */,
	128 + 0 /* 8 */,
	128 + 0 /* 9 */,
	128 + 0 /* 10 */,
	128 + 0 /* 11 */,
	128 + 0 /* 12 */,
	128 + 0 /* 13 */,
	128 + 0 /* 14 */,
	128 + 0 /* 15 */,
	128 + 0 /* 16 */,
	128 + 0 /* 17 */,
	128 + 1 /* 18/00-18/09 */,
	128 + 1 /* 18/10-18/19 */,
	128 + 1 /* 18/20-18/29 */,
	128 + 1 /* 18/30-18/39 */,
	128 + 1 /* 18/40-18/49 */,
	128 + 1 /* 18/50-18/59 */,
	128 + 1 /* 18/60-18/69 */,
	128 + 2 /* 18/70-18/79 */,
	128 + 3 /* 18/80-18/89 */,
	128 + 4 /* 18/90-18/99 */,
	128 + 5 /* 18/100-18/109 */,
	128 + 6 /* 18/110-18/119 */,
	128 + 7 /* 18/120-18/129 */,
	128 + 8 /* 18/130-18/139 */,
	128 + 9 /* 18/140-18/149 */,
	128 + 10        /* 18/150-18/159 */,
	128 + 11        /* 18/160-18/169 */,
	128 + 12        /* 18/170-18/179 */,
	128 + 13        /* 18/180-18/189 */,
	128 + 14        /* 18/190-18/199 */,
	128 + 15        /* 18/200-18/209 */,
	128 + 15        /* 18/210-18/219 */,
	128 + 16        /* 18/220+ */
};


/*!
 * 腕力による基本所持重量値テーブル
 * Stat Table (STR) -- weight limit in deca-pounds
 */
const byte adj_str_wgt[] =
{
	10      /* 3 */,
	11      /* 4 */,
	12      /* 5 */,
	13      /* 6 */,
	14      /* 7 */,
	15      /* 8 */,
	16      /* 9 */,
	17      /* 10 */,
	18      /* 11 */,
	19      /* 12 */,
	20      /* 13 */,
	21      /* 14 */,
	22      /* 15 */,
	23      /* 16 */,
	24      /* 17 */,
	25      /* 18/00-18/09 */,
	26      /* 18/10-18/19 */,
	27      /* 18/20-18/29 */,
	28      /* 18/30-18/39 */,
	29      /* 18/40-18/49 */,
	30      /* 18/50-18/59 */,
	31      /* 18/60-18/69 */,
	31      /* 18/70-18/79 */,
	32      /* 18/80-18/89 */,
	32      /* 18/90-18/99 */,
	33      /* 18/100-18/109 */,
	33      /* 18/110-18/119 */,
	34      /* 18/120-18/129 */,
	34      /* 18/130-18/139 */,
	35      /* 18/140-18/149 */,
	35      /* 18/150-18/159 */,
	36      /* 18/160-18/169 */,
	36      /* 18/170-18/179 */,
	37      /* 18/180-18/189 */,
	37      /* 18/190-18/199 */,
	38      /* 18/200-18/209 */,
	38      /* 18/210-18/219 */,
	39      /* 18/220+ */
};


/*!
 * 腕力による武器重量限界値テーブル
 * Stat Table (STR) -- weapon weight limit in pounds
 */
const byte adj_str_hold[] =
{
	4       /* 3 */,
	5       /* 4 */,
	6       /* 5 */,
	7       /* 6 */,
	8       /* 7 */,
	9       /* 8 */,
	10      /* 9 */,
	11      /* 10 */,
	12      /* 11 */,
	13      /* 12 */,
	14      /* 13 */,
	15      /* 14 */,
	16      /* 15 */,
	17      /* 16 */,
	18      /* 17 */,
	19      /* 18/00-18/09 */,
	20      /* 18/10-18/19 */,
	21      /* 18/20-18/29 */,
	22      /* 18/30-18/39 */,
	23      /* 18/40-18/49 */,
	24      /* 18/50-18/59 */,
	25      /* 18/60-18/69 */,
	26      /* 18/70-18/79 */,
	27      /* 18/80-18/89 */,
	28      /* 18/90-18/99 */,
	30      /* 18/100-18/109 */,
	31      /* 18/110-18/119 */,
	32      /* 18/120-18/129 */,
	33      /* 18/130-18/139 */,
	34      /* 18/140-18/149 */,
	35      /* 18/150-18/159 */,
	37      /* 18/160-18/169 */,
	40      /* 18/170-18/179 */,
	44      /* 18/180-18/189 */,
	48      /* 18/190-18/199 */,
	50     /* 18/200-18/209 */,
	50     /* 18/210-18/219 */,
	50     /* 18/220+ */
};


/*!
 * 腕力による採掘能力修正値テーブル
 * Stat Table (STR) -- digging value
 */
const byte adj_str_dig[] =
{
	0       /* 3 */,
	0       /* 4 */,
	1       /* 5 */,
	2       /* 6 */,
	3       /* 7 */,
	4       /* 8 */,
	4       /* 9 */,
	5       /* 10 */,
	5       /* 11 */,
	6       /* 12 */,
	6       /* 13 */,
	7       /* 14 */,
	7       /* 15 */,
	8       /* 16 */,
	8       /* 17 */,
	9       /* 18/00-18/09 */,
	10      /* 18/10-18/19 */,
	12      /* 18/20-18/29 */,
	15      /* 18/30-18/39 */,
	20      /* 18/40-18/49 */,
	25      /* 18/50-18/59 */,
	30      /* 18/60-18/69 */,
	35      /* 18/70-18/79 */,
	40      /* 18/80-18/89 */,
	45      /* 18/90-18/99 */,
	50      /* 18/100-18/109 */,
	55      /* 18/110-18/119 */,
	60      /* 18/120-18/129 */,
	65      /* 18/130-18/139 */,
	70      /* 18/140-18/149 */,
	75      /* 18/150-18/159 */,
	80      /* 18/160-18/169 */,
	85      /* 18/170-18/179 */,
	90      /* 18/180-18/189 */,
	95      /* 18/190-18/199 */,
	100     /* 18/200-18/209 */,
	100     /* 18/210-18/219 */,
	100     /* 18/220+ */
};


/*!
 * 腕力による攻撃回数算定値テーブル
 * Stat Table (STR) -- help index into the "blow" table
 */
const byte adj_str_blow[] =
{
	3       /* 3 */,
	4       /* 4 */,
	5       /* 5 */,
	6       /* 6 */,
	7       /* 7 */,
	8       /* 8 */,
	9       /* 9 */,
	10      /* 10 */,
	11      /* 11 */,
	12      /* 12 */,
	13      /* 13 */,
	14      /* 14 */,
	15      /* 15 */,
	16      /* 16 */,
	17      /* 17 */,
	20 /* 18/00-18/09 */,
	30 /* 18/10-18/19 */,
	40 /* 18/20-18/29 */,
	50 /* 18/30-18/39 */,
	60 /* 18/40-18/49 */,
	70 /* 18/50-18/59 */,
	80 /* 18/60-18/69 */,
	90 /* 18/70-18/79 */,
	100 /* 18/80-18/89 */,
	110 /* 18/90-18/99 */,
	120 /* 18/100-18/109 */,
	130 /* 18/110-18/119 */,
	140 /* 18/120-18/129 */,
	150 /* 18/130-18/139 */,
	160 /* 18/140-18/149 */,
	170 /* 18/150-18/159 */,
	180 /* 18/160-18/169 */,
	190 /* 18/170-18/179 */,
	200 /* 18/180-18/189 */,
	210 /* 18/190-18/199 */,
	220 /* 18/200-18/209 */,
	230 /* 18/210-18/219 */,
	240 /* 18/220+ */
};


/*!
 * 器用さによる攻撃回数インデックステーブル
 * Stat Table (DEX) -- index into the "blow" table
 */
const byte adj_dex_blow[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	0       /* 8 */,
	0       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	2       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	3       /* 18/30-18/39 */,
	4       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	5       /* 18/60-18/69 */,
	5       /* 18/70-18/79 */,
	6       /* 18/80-18/89 */,
	6       /* 18/90-18/99 */,
	7       /* 18/100-18/109 */,
	7       /* 18/110-18/119 */,
	8       /* 18/120-18/129 */,
	8       /* 18/130-18/139 */,
	9      /* 18/140-18/149 */,
	9      /* 18/150-18/159 */,
	10      /* 18/160-18/169 */,
	10      /* 18/170-18/179 */,
	11      /* 18/180-18/189 */,
	11      /* 18/190-18/199 */,
	12      /* 18/200-18/209 */,
	12      /* 18/210-18/219 */,
	13      /* 18/220+ */
};


/*!
 * 器用さによる盗難防止＆体当たり成功判定修正テーブル
 * Stat Table (DEX) -- chance of avoiding "theft" and "falling"
 */
const byte adj_dex_safe[] =
{
	0       /* 3 */,
	1       /* 4 */,
	2       /* 5 */,
	3       /* 6 */,
	4       /* 7 */,
	5       /* 8 */,
	5       /* 9 */,
	6       /* 10 */,
	6       /* 11 */,
	7       /* 12 */,
	7       /* 13 */,
	8       /* 14 */,
	8       /* 15 */,
	9       /* 16 */,
	9       /* 17 */,
	10      /* 18/00-18/09 */,
	10      /* 18/10-18/19 */,
	15      /* 18/20-18/29 */,
	15      /* 18/30-18/39 */,
	20      /* 18/40-18/49 */,
	25      /* 18/50-18/59 */,
	30      /* 18/60-18/69 */,
	35      /* 18/70-18/79 */,
	40      /* 18/80-18/89 */,
	45      /* 18/90-18/99 */,
	50      /* 18/100-18/109 */,
	60      /* 18/110-18/119 */,
	70      /* 18/120-18/129 */,
	80      /* 18/130-18/139 */,
	90      /* 18/140-18/149 */,
	100     /* 18/150-18/159 */,
	100     /* 18/160-18/169 */,
	100     /* 18/170-18/179 */,
	100     /* 18/180-18/189 */,
	100     /* 18/190-18/199 */,
	100     /* 18/200-18/209 */,
	100     /* 18/210-18/219 */,
	100     /* 18/220+ */
};


/*!
 * 耐久による基本HP自然治癒値テーブル /
 * Stat Table (CON) -- base regeneration rate
 */
const byte adj_con_fix[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	0       /* 8 */,
	0       /* 9 */,
	0       /* 10 */,
	0       /* 11 */,
	0       /* 12 */,
	0       /* 13 */,
	1       /* 14 */,
	1       /* 15 */,
	1       /* 16 */,
	1       /* 17 */,
	2       /* 18/00-18/09 */,
	2       /* 18/10-18/19 */,
	2       /* 18/20-18/29 */,
	2       /* 18/30-18/39 */,
	2       /* 18/40-18/49 */,
	3       /* 18/50-18/59 */,
	3       /* 18/60-18/69 */,
	3       /* 18/70-18/79 */,
	3       /* 18/80-18/89 */,
	3       /* 18/90-18/99 */,
	4       /* 18/100-18/109 */,
	4       /* 18/110-18/119 */,
	5       /* 18/120-18/129 */,
	6       /* 18/130-18/139 */,
	6       /* 18/140-18/149 */,
	7       /* 18/150-18/159 */,
	7       /* 18/160-18/169 */,
	8       /* 18/170-18/179 */,
	8       /* 18/180-18/189 */,
	8       /* 18/190-18/199 */,
	9       /* 18/200-18/209 */,
	9       /* 18/210-18/219 */,
	9       /* 18/220+ */
};


/*!
 * 耐久による基本HP自然治癒値テーブル /
 * Stat Table (CON) -- extra 1/4-hitpoints per level (plus 128)
 */
const byte adj_con_mhp[] =
{
	128 + -8        /* 3 */,
	128 + -6        /* 4 */,
	128 + -4        /* 5 */,
	128 + -2        /* 6 */,
	128 + -1 /* 7 */,
	128 + 0 /* 8 */,
	128 + 0 /* 9 */,
	128 + 0 /* 10 */,
	128 + 0 /* 11 */,
	128 + 0 /* 12 */,
	128 + 0 /* 13 */,
	128 + 1 /* 14 */,
	128 + 1 /* 15 */,
	128 + 2 /* 16 */,
	128 + 3 /* 17 */,
	128 + 4 /* 18/00-18/09 */,
	128 + 5 /* 18/10-18/19 */,
	128 + 6 /* 18/20-18/29 */,
	128 + 7 /* 18/30-18/39 */,
	128 + 8 /* 18/40-18/49 */,
	128 + 9 /* 18/50-18/59 */,
	128 + 10  /* 18/60-18/69 */,
	128 + 11 /* 18/70-18/79 */,
	128 + 12 /* 18/80-18/89 */,
	128 + 14 /* 18/90-18/99 */,
	128 + 17         /* 18/100-18/109 */,
	128 + 20        /* 18/110-18/119 */,
	128 + 23        /* 18/120-18/129 */,
	128 + 26        /* 18/130-18/139 */,
	128 + 29        /* 18/140-18/149 */,
	128 + 32        /* 18/150-18/159 */,
	128 + 35        /* 18/160-18/169 */,
	128 + 38        /* 18/170-18/179 */,
	128 + 40        /* 18/180-18/189 */,
	128 + 42        /* 18/190-18/199 */,
	128 + 44        /* 18/200-18/209 */,
	128 + 46        /* 18/210-18/219 */,
	128 + 48        /* 18/220+ */
};


/*!
 * 魅力による魅了能力修正テーブル /
 * Stat Table (CHR) -- charm
 */
const byte adj_chr_chm[] =
{
	0       /* 3 */,
	0       /* 4 */,
	1       /* 5 */,
	2       /* 6 */,
	3       /* 7 */,
	4       /* 8 */,
	4       /* 9 */,
	5       /* 10 */,
	5       /* 11 */,
	6       /* 12 */,
	6       /* 13 */,
	7       /* 14 */,
	7       /* 15 */,
	8       /* 16 */,
	8       /* 17 */,
	9       /* 18/00-18/09 */,
	10      /* 18/10-18/19 */,
	12      /* 18/20-18/29 */,
	15      /* 18/30-18/39 */,
	18      /* 18/40-18/49 */,
	21      /* 18/50-18/59 */,
	24      /* 18/60-18/69 */,
	28      /* 18/70-18/79 */,
	32      /* 18/80-18/89 */,
	36      /* 18/90-18/99 */,
	39      /* 18/100-18/109 */,
	42      /* 18/110-18/119 */,
	45      /* 18/120-18/129 */,
	49      /* 18/130-18/139 */,
	53      /* 18/140-18/149 */,
	57      /* 18/150-18/159 */,
	61      /* 18/160-18/169 */,
	65      /* 18/170-18/179 */,
	69      /* 18/180-18/189 */,
	73      /* 18/190-18/199 */,
	77      /* 18/200-18/209 */,
	81      /* 18/210-18/219 */,
	85      /* 18/220+ */
};


/*!
 * @brief
 * 魅力による魅了能力修正テーブル /
 * This table is used to help calculate the number of blows the player can
 * make in a single round of attacks (one player turn) with a normal weapon.
 * @details
 * <pre>
 * This number ranges from a single blow/round for weak players to up to six
 * blows/round for powerful warriors.
 *
 * Note that certain artifacts and ego-items give "bonus" blows/round.
 *
 * First, from the player class, we extract some values:
 *
 * Warrior       num = 6; mul = 5; div = MAX(70, weapon_weight);
 * Berserker     num = 6; mul = 7; div = MAX(70, weapon_weight);
 * Mage          num = 3; mul = 2; div = MAX(100, weapon_weight);
 * Priest        num = 5; mul = 3; div = MAX(100, weapon_weight);
 * Mindcrafter   num = 5; mul = 3; div = MAX(100, weapon_weight);
 * Rogue         num = 5; mul = 3; div = MAX(40, weapon_weight);
 * Ranger        num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Paladin       num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Weaponsmith   num = 5; mul = 5; div = MAX(150, weapon_weight);
 * Warrior-Mage  num = 5; mul = 3; div = MAX(70, weapon_weight);
 * Chaos Warrior num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Monk          num = 5; mul = 3; div = MAX(60, weapon_weight);
 * Tourist       num = 4; mul = 3; div = MAX(100, weapon_weight);
 * Imitator      num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Beastmaster   num = 5; mul = 3; div = MAX(70, weapon_weight);
 * Cavalry(Ride) num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Cavalry(Walk) num = 5; mul = 3; div = MAX(100, weapon_weight);
 * Sorcerer      num = 1; mul = 1; div = MAX(1, weapon_weight);
 * Archer        num = 4; mul = 2; div = MAX(70, weapon_weight);
 * Magic eater   num = 4; mul = 2; div = MAX(70, weapon_weight);
 * ForceTrainer  num = 4; mul = 2; div = MAX(60, weapon_weight);
 * Mirror Master num = 3; mul = 3; div = MAX(100, weapon_weight);
 * Ninja         num = 4; mul = 1; div = MAX(20, weapon_weight);
 *
 * To get "P", we look up the relevant "adj_str_blow[]" (see above),
 * multiply it by "mul", and then divide it by "div".
 * Increase P by 1 if you wield a weapon two-handed.
 * Decrease P by 1 if you are a Ninja.
 *
 * To get "D", we look up the relevant "adj_dex_blow[]" (see above),
 *
 * The player gets "blows_table[P][D]" blows/round, as shown below,
 * up to a maximum of "num" blows/round, plus any "bonus" blows/round.
 * </pre>
 */
const byte blows_table[12][12] =
{
	/* P/D */
	/*      0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11+ */
	/*      3   10   15  /10  /40  /60  /80 /100 /120 /140 /160 /180  */
/* 0 */{	1,   1,   1,   1,   1,   2,   2,   2,   2,   3,   3,   4 },
/* 1 */{	1,   1,   1,   2,   2,   2,   3,   3,   3,   4,   4,   4 },
/* 2 */{	1,   1,   2,   2,   3,   3,   4,   4,   4,   5,   5,   5 },
/* 3 */{	1,   1,   2,   3,   3,   4,   4,   4,   5,   5,   5,   5 },
/* 4 */{	1,   1,   2,   3,   3,   4,   4,   5,   5,   5,   5,   5 },
/* 5 */{	1,   1,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6 },
/* 6 */{	1,   1,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6 },
/* 7 */{	1,   2,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6 },
/* 8 */{	1,   2,   3,   3,   4,   4,   4,   5,   5,   5,   6,   6 },
/* 9 */{	1,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6,   6 },
/* 10*/{	2,   2,   3,   4,   4,   4,   5,   5,   5,   6,   6,   6 },
/*11+*/{	2,   2,   3,   4,   4,   4,   5,   5,   6,   6,   6,   6 },

};

/*!
 * @brief 闘技場のモンスターID及び報酬アイテムテーブル
 */
const arena_type arena_info[MAX_ARENA_MONS + 2] =
{
	{ MON_NOBORTA,       TV_AMULET, SV_AMULET_ADORNMENT           },
	{ MON_MORI_TROLL,    TV_FOOD,   SV_FOOD_PINT_OF_WINE          },
	{ MON_IMP,           TV_POTION, SV_POTION_SPEED               },
	{ MON_LION_HEART,    0,         0                             },
	{ MON_MASTER_YEEK,   TV_POTION, SV_POTION_CURING              },
	{ MON_SABRE_TIGER,   TV_WAND,   SV_WAND_STONE_TO_MUD          },
	{ MON_LIZARD_KING,   TV_WAND,   SV_WAND_TELEPORT_AWAY         },
	{ MON_WYVERN,        TV_POTION, SV_POTION_HEALING             },
	{ MON_ARCH_VILE,     TV_POTION, SV_POTION_RESISTANCE          },
	{ MON_ELF_LORD   ,   TV_POTION, SV_POTION_ENLIGHTENMENT       },
	{ MON_GHOUL_KING,    TV_FOOD,   SV_FOOD_RESTORING             },
	{ MON_COLBRAN,       TV_RING,   SV_RING_ELEC                  },
	{ MON_BICLOPS,       TV_WAND,   SV_WAND_ACID_BALL             },
	{ MON_M_MINDCRAFTER, TV_POTION, SV_POTION_SELF_KNOWLEDGE      },
	{ MON_GROO,          TV_SCROLL, SV_SCROLL_ACQUIREMENT         },
	{ MON_RAAL,          TV_SCROLL, SV_SCROLL_STAR_DESTRUCTION    },
	{ MON_DREADMASTER,   TV_WAND,   SV_WAND_DRAIN_LIFE            },
	{ MON_ULTRA_PALADIN, TV_STAFF,  SV_STAFF_DISPEL_EVIL          },
	{ MON_BARNEY,        TV_RING,   SV_RING_RES_CHAOS             },
	{ MON_TROLL_KING,    TV_SCROLL, SV_SCROLL_MASS_GENOCIDE       },
	{ MON_BARON_HELL,    TV_POTION, SV_POTION_AUGMENTATION        },
	{ MON_F_ANGEL,       TV_SCROLL, SV_SCROLL_RUNE_OF_PROTECTION  },
	{ MON_G_C_DRAKE,     TV_WAND,   SV_WAND_DRAGON_FIRE           },
	{ MON_IRON_LICH,     TV_STAFF,  SV_STAFF_DESTRUCTION          },
	{ MON_DROLEM,        TV_POTION, SV_POTION_STAR_HEALING        },
	{ MON_G_TITAN,       TV_WAND,   SV_WAND_GENOCIDE              },
	{ MON_G_BALROG,      TV_POTION, SV_POTION_EXPERIENCE          },
	{ MON_ELDER_VAMPIRE, TV_RING,   SV_RING_SUSTAIN               },
	{ MON_NIGHTWALKER,   TV_WAND,   SV_WAND_STRIKING              },
	{ MON_S_TYRANNO,     TV_SCROLL, SV_SCROLL_STAR_ACQUIREMENT    },
	{ MON_G_MASTER_MYS,  TV_ROD,    SV_ROD_IDENTIFY               },
	{ MON_LORD_CHAOS,    TV_POTION, SV_POTION_LIFE                },
	{ MON_SHADOWLORD,    TV_POTION, SV_POTION_STAR_ENLIGHTENMENT  },
	{ MON_ULT_BEHOLDER,  TV_AMULET, SV_AMULET_REFLECTION          },
	{ MON_JABBERWOCK,    TV_ROD,    SV_ROD_HEALING                },
	{ MON_LOCKE_CLONE,   TV_WAND,   SV_WAND_DISINTEGRATE          },
	{ MON_WYRM_SPACE,    TV_ROD,    SV_ROD_RESTORATION            },
	{ MON_SHAMBLER,      TV_SCROLL, SV_SCROLL_STAR_ACQUIREMENT    },
	{ MON_BLACK_REAVER,  TV_RING,   SV_RING_LORDLY                },
	{ MON_FENGHUANG,     TV_STAFF,  SV_STAFF_THE_MAGI             },
	{ MON_WYRM_POWER,    TV_SCROLL, SV_SCROLL_ARTIFACT            },
	{ 0,                 0,         0                             }, /* Victory prizing */
	{ MON_HAGURE,        TV_SCROLL, SV_SCROLL_ARTIFACT            },
};


/*!
 * @brief 闘技場のモンスターID及び報酬アイテムテーブル /
 * Store owners (exactly four "possible" owners per store, chosen randomly)
 * @details
 * <pre>
 * { name, purse, max greed, min greed, haggle_per, tolerance, race, unused }
 *
 * Lifted extra shopkeepers from CthAngband (don't you just love open source
 * development? ;-)). Since this gave less than 32 unique names for some
 * shops, those have their first x names copied to reach 32.
 *
 * For the weapon and armour shops, several owners have a limit of 5k.
 *
 * I want to do 50k owners, but the purse is currently s16b. Perhaps
 * we should just store 1/10th of the purse?
 * </pre>
 */
const owner_type owners[MAX_STORES][MAX_OWNERS] =
{
	{
		/* General store - 32 unique names */
		/*
		  Raistlin は dragonlance の powerful wizard 。
		  Rincewind the Chicken は Terry Pratchett の Discworld の登場人物 上記のパロディ？、
		  { "憶病者ラストリン",       200,    175,  108,   4,  12,  RACE_HUMAN},
		  { "Raistlin the Chicken",       200,    175, 108,  4, 12, RACE_HUMAN},
		*/

#ifdef JP
		{ "フレンドリーなビルボ",       200,    170,  108,   5,  15,  RACE_HOBBIT},
		{ "憶病者リンスウィンド",       200,    175,  108,   4,  12,  RACE_HUMAN}, 
		{ "背の低いサルタン",             300,    170,  107,   5,  15,  RACE_GNOME},
		{ "ハンサムなライア=エル",      300,    165,  107,   6,  18,  RACE_ELF},
		{ "親切なファリルマウエン",         250,    170, 108,  5, 15, RACE_HOBBIT},
		{ "臆病者ヴォワラン",       500,    175, 108,  4, 12, RACE_HUMAN},
		{ "チビのエラシュナク",          750,    170, 107,  5, 15, RACE_BEASTMAN},
		{ "ハンサムなグラッグ",        1000,    165, 107,  6, 18, RACE_HALF_TITAN},
		{ "ケチなフォロビア",         250,    170, 108,  5, 15, RACE_HUMAN},
		{ "馬鹿のエリス",       500,    175, 108,  4, 12, RACE_HUMAN},
		{ "腹ペコのフィルバート",          750,    170, 107,  5, 15, RACE_VAMPIRE},
		{ "スナーグル・サシグア",        1000,    165, 107,  6, 18, RACE_MIND_FLAYER},
		{ "長死きエロワーズ",         250,    170, 108,  5, 15, RACE_SPECTRE},
		{ "ノロマのフンディ",       500,    175, 108,  4, 12, RACE_ZOMBIE},
		{ "グランサス",          750,    170, 107,  5, 15, RACE_SKELETON},
		{ "丁寧なロラックス",        1000,    165, 107,  6, 18, RACE_VAMPIRE},
		{ "ブッチ",         250,    170, 108,  5, 15, RACE_HALF_ORC},
		{ "美しきエルベレス",       500,    175, 108,  4, 12, RACE_HIGH_ELF},
		{ "こそこそサーレス",          750,    170, 107,  5, 15, RACE_GNOME},
		{ "ナーロック",        1000,    165, 107,  6, 18, RACE_DWARF},
		{ "チビのヘイネッカ",         250,    170, 108,  5, 15, RACE_GNOME},
		{ "きちがいロワラン",       500,    175, 108,  4, 12, RACE_HALF_GIANT},
		{ "毒息のウート",          750,    170, 107,  5, 15, RACE_DRACONIAN},
		{ "でぶっちょアラァカ",        1000,    165, 107,  6, 18, RACE_DRACONIAN},
		{ "低能なプーゴー",         250,    170, 108,  5, 15, RACE_BEASTMAN},
		{ "フェロールフィリアン",       500,    175, 108,  4, 12, RACE_ELF},
		{ "年寄りマロカ",          750,    170, 107,  5, 15, RACE_GNOME},
		{ "勇敢なサシン",        1000,    165, 107,  6, 18, RACE_HALF_GIANT},
		{ "田舎者アビエマール",         250,    170, 108,  5, 15, RACE_HUMAN},
		{ "貧乏なハーク",       500,    175, 108,  4, 12, RACE_HALF_ORC},
		{ "みじめなソアリン",          750,    170, 107,  5, 15, RACE_ZOMBIE},
		{ "質素なメルラ",        1000,    165, 107,  6, 18, RACE_ELF},
#else
		{ "Bilbo the Friendly",         200,    170, 108,  5, 15, RACE_HOBBIT},
		{ "Rincewind the Chicken",       200,    175, 108,  4, 12, RACE_HUMAN},
		{ "Sultan the Midget",          300,    170, 107,  5, 15, RACE_GNOME},
		{ "Lyar-el the Comely",         300,    165, 107,  6, 18, RACE_ELF},
		{ "Falilmawen the Friendly",         250,    170, 108,  5, 15, RACE_HOBBIT},
		{ "Voirin the Cowardly",       500,    175, 108,  4, 12, RACE_HUMAN},
		{ "Erashnak the Midget",          750,    170, 107,  5, 15, RACE_BEASTMAN},
		{ "Grug the Comely",        1000,    165, 107,  6, 18, RACE_HALF_TITAN},
		{ "Forovir the Cheap",         250,    170, 108,  5, 15, RACE_HUMAN},
		{ "Ellis the Fool",       500,    175, 108,  4, 12, RACE_HUMAN},
		{ "Filbert the Hungry",          750,    170, 107,  5, 15, RACE_VAMPIRE},
		{ "Fthnargl Psathiggua",        1000,    165, 107,  6, 18, RACE_MIND_FLAYER},
		{ "Eloise Long-Dead",         250,    170, 108,  5, 15, RACE_SPECTRE},
		{ "Fundi the Slow",       500,    175, 108,  4, 12, RACE_ZOMBIE},
		{ "Granthus",          750,    170, 107,  5, 15, RACE_SKELETON},
		{ "Lorax the Suave",        1000,    165, 107,  6, 18, RACE_VAMPIRE},
		{ "Butch",         250,    170, 108,  5, 15, RACE_HALF_ORC},
		{ "Elbereth the Beautiful",       500,    175, 108,  4, 12, RACE_HIGH_ELF},
		{ "Sarleth the Sneaky",          750,    170, 107,  5, 15, RACE_GNOME},
		{ "Narlock",        1000,    165, 107,  6, 18, RACE_DWARF},
		{ "Haneka the Small",         250,    170, 108,  5, 15, RACE_GNOME},
		{ "Loirin the Mad",       500,    175, 108,  4, 12, RACE_HALF_GIANT},
		{ "Wuto Poisonbreath",          750,    170, 107,  5, 15, RACE_DRACONIAN},
		{ "Araaka the Rotund",        1000,    165, 107,  6, 18, RACE_DRACONIAN},
		{ "Poogor the Dumb",         250,    170, 108,  5, 15, RACE_BEASTMAN},
		{ "Felorfiliand",       500,    175, 108,  4, 12, RACE_ELF},
		{ "Maroka the Aged",          750,    170, 107,  5, 15, RACE_GNOME},
		{ "Sasin the Bold",        1000,    165, 107,  6, 18, RACE_HALF_GIANT},
		{ "Abiemar the Peasant",         250,    170, 108,  5, 15, RACE_HUMAN},
		{ "Hurk the Poor",       500,    175, 108,  4, 12, RACE_HALF_ORC},
		{ "Soalin the Wretched",          750,    170, 107,  5, 15, RACE_ZOMBIE},
		{ "Merulla the Humble",        1000,    165, 107,  6, 18, RACE_ELF},
#endif
	},
	{
		/* Armoury - 28 unique names */
#ifdef JP
		{ "醜悪コン=ダー",      5000,   210,  115,   5,   7,  RACE_HALF_ORC},
		{ "頑固者ダーグ=ロウ",  10000,  190,  111,   4,   9,  RACE_HUMAN},
		{ "賢者デカド",                 25000,  200,  112,   4,  10,  RACE_DUNADAN},
		{ "鍛冶屋のウィーランド",   30000,  200,  112,   4,   5,  RACE_DWARF},
		{ "醜悪コン=ダー",           10000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "頑固者ダーグ=ロウ",          15000,  190, 111,  4,  9, RACE_HUMAN},
		{ "ハンサムなデカド",            25000,  200, 112,  4, 10, RACE_AMBERITE},
		{ "エロー・ドラゴンスケイル",          30000,  200, 112,  4,  5, RACE_ELF},
		{ "デリカトス",           10000,   210, 115,  5,  7, RACE_SPRITE},
		{ "巨大なグルース",          15000,  190, 111,  4,  9, RACE_HALF_GIANT},
		{ "アニムス",            25000,  200, 112,  4, 10, RACE_GOLEM},
		{ "マルヴァス",          30000,  200, 112,  4,  5, RACE_HALF_TITAN},
		{ "セラクシス",           10000,   210, 115,  5,  7, RACE_ZOMBIE},
		{ "デス・チル",          5000,  190, 111,  4,  9, RACE_SPECTRE},
		{ "微かなドリオス",            25000,  200, 112,  4, 10, RACE_SPECTRE},
		{ "冷たいバスリック",          30000,  200, 112,  4,  5, RACE_VAMPIRE},
		{ "冷酷ヴェンジェラ",           10000,   210, 115,  5,  7, RACE_HALF_TROLL},
		{ "強者ウィラナ",          15000,  190, 111,  4,  9, RACE_HUMAN},
		{ "ヨジョ二世",            25000,  200, 112,  4, 10, RACE_DWARF},
		{ "優しいラナラー",          30000,  200, 112,  4,  5, RACE_AMBERITE},
		{ "不浄のホルバグ",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "テレパスのエレレン",          15000,  190, 111,  4,  9, RACE_DARK_ELF},
		{ "イスドリリアス",            25000,  200, 112,  4, 10, RACE_SPRITE},
		{ "一つ目ヴェグナー",          5000,  200, 112,  4,  5, RACE_CYCLOPS},
		{ "混沌のロディッシュ",           10000,   210, 115,  5,  7, RACE_BEASTMAN},
		{ "剣豪ヘジン",          15000,  190, 111,  4,  9, RACE_NIBELUNG},
		{ "ずる屋のエルベレリス",           10000,  200, 112,  4, 10, RACE_DARK_ELF},
		{ "インプのザサス",          30000,  200, 112,  4,  5, RACE_IMP},
		{ "醜悪コン=ダー",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "頑固者ダーグ=ロウ",          10000,  190, 111,  4,  9, RACE_HUMAN},
		{ "ハンサムなデカド",            25000,  200, 112,  4, 10, RACE_AMBERITE},
		{ "鍛冶屋のウィーランド",          30000,  200, 112,  4,  5, RACE_DWARF},
#else
		{ "Kon-Dar the Ugly",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "Darg-Low the Grim",          10000,  190, 111,  4,  9, RACE_HUMAN},
		{ "Decado the Handsome",            25000,  200, 112,  4, 10, RACE_DUNADAN},
		{ "Wieland the Smith",          30000,  200, 112,  4,  5, RACE_DWARF},
		{ "Kon-Dar the Ugly",           10000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "Darg-Low the Grim",          15000,  190, 111,  4,  9, RACE_HUMAN},
		{ "Decado the Handsome",            25000,  200, 112,  4, 10, RACE_AMBERITE},
		{ "Elo Dragonscale",          30000,  200, 112,  4,  5, RACE_ELF},
		{ "Delicatus",           10000,   210, 115,  5,  7, RACE_SPRITE},
		{ "Gruce the Huge",          15000,  190, 111,  4,  9, RACE_HALF_GIANT},
		{ "Animus",            25000,  200, 112,  4, 10, RACE_GOLEM},
		{ "Malvus",          30000,  200, 112,  4,  5, RACE_HALF_TITAN},
		{ "Selaxis",           10000,   210, 115,  5,  7, RACE_ZOMBIE},
		{ "Deathchill",          5000,  190, 111,  4,  9, RACE_SPECTRE},
		{ "Drios the Faint",            25000,  200, 112,  4, 10, RACE_SPECTRE},
		{ "Bathric the Cold",          30000,  200, 112,  4,  5, RACE_VAMPIRE},
		{ "Vengella the Cruel",           10000,   210, 115,  5,  7, RACE_HALF_TROLL},
		{ "Wyrana the Mighty",          15000,  190, 111,  4,  9, RACE_HUMAN},
		{ "Yojo II",            25000,  200, 112,  4, 10, RACE_DWARF},
		{ "Ranalar the Sweet",          30000,  200, 112,  4,  5, RACE_AMBERITE},
		{ "Horbag the Unclean",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "Elelen the Telepath",          15000,  190, 111,  4,  9, RACE_DARK_ELF},
		{ "Isedrelias",            25000,  200, 112,  4, 10, RACE_SPRITE},
		{ "Vegnar One-eye",          5000,  200, 112,  4,  5, RACE_CYCLOPS},
		{ "Rodish the Chaotic",           10000,   210, 115,  5,  7, RACE_BEASTMAN},
		{ "Hesin Swordmaster",          15000,  190, 111,  4,  9, RACE_NIBELUNG},
		{ "Elvererith the Cheat",           10000,  200, 112,  4, 10, RACE_DARK_ELF},
		{ "Zzathath the Imp",          30000,  200, 112,  4,  5, RACE_IMP},
		{ "Kon-Dar the Ugly",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "Darg-Low the Grim",          10000,  190, 111,  4,  9, RACE_HUMAN},
		{ "Decado the Handsome",            25000,  200, 112,  4, 10, RACE_AMBERITE},
		{ "Wieland the Smith",          30000,  200, 112,  4,  5, RACE_DWARF},
#endif
	},

	{
		/* Weapon Smith - 28 unique names */
#ifdef JP
		{ "残忍なるアーノルド",        5000,   210,  115,   6,   6,  RACE_HALF_TROLL},
		{ "獣殺しのアーンダル", 10000,  185,  110,   5,   9,  RACE_HALF_ELF},
		{ "獣マスターのエディー", 25000,  190,  115,   5,   7,  RACE_HOBBIT},
		{ "竜殺しのオグライン", 30000,  195,  112,   4,   8,  RACE_DWARF},
		{ "熟練者ドリュー",      10000,   210, 115,  6,  6, RACE_HUMAN},
		{"龍の子オラックス",        15000,  185, 110,  5,  9, RACE_DRACONIAN},
		{ "病気持ちのアンスラックス",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
		{ "頑丈者アルコス",       30000,  195, 112,  4,  8, RACE_DWARF},
		{ "腐れ者のサリアス",      5000,   210, 115,  6,  6, RACE_ZOMBIE},
		{"晒し骨のトゥエシク",        15000,  185, 110,  5,  9, RACE_SKELETON},
		{ "ビリオス",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
		{ "ファスガル",       30000,  195, 112,  4,  8, RACE_ZOMBIE},
		{ "パラディンのエレフリス",      10000,   210, 115,  6,  6, RACE_BARBARIAN},
		{"キ'トリッ'ク",        15000,  185, 110,  5,  9, RACE_KLACKON},
		{ "蜘蛛の友ドゥロカス",         25000,  190, 115,  5,  7, RACE_DARK_ELF},
		{ "巨人殺しのフングス",       30000,  195, 112,  4,  8, RACE_DWARF},
		{ "デランサ",      10000,   210, 115,  6,  6, RACE_ELF},
		{"レンジャーのソルビスタニ",        15000,  185, 110,  5,  9, RACE_HALF_ELF},
		{ "ノロマのゾリル",         25000,  190, 115,  5,  7, RACE_GOLEM},
		{ "イーオン・フラックス",       20000,  195, 112,  4,  8, RACE_HALF_ELF},
		{ "強者ナドック",      10000,   210, 115,  6,  6, RACE_HOBBIT},
		{"弱虫エラモグ",        15000,  185, 110,  5,  9, RACE_KOBOLD},
		{ "公正なエオウィリス",         25000,  190, 115,  5,  7, RACE_VAMPIRE},
		{ "バルログ殺しのヒュイモグ",       30000,  195, 112,  4,  8, RACE_HALF_ORC},
		{ "冷酷ピーダス",      5000,   210, 115,  6,  6, RACE_HUMAN},
		{ "ヴァモグ スレイヤー",        15000,  185, 110,  5,  9, RACE_HALF_OGRE},
		{ "性悪フーシュナク",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
		{ "舞闘バレン",       30000,  195, 112,  4,  8, RACE_BARBARIAN},
		{ " 残忍なるアーノルド",      5000,   210, 115,  6,  6, RACE_BARBARIAN},
		{ "獣殺しのアーンダル",        10000,  185, 110,  5,  9, RACE_HALF_ELF},
		{ "ビーストマスター・エディー",         25000,  190, 115,  5,  7, RACE_HALF_ORC},
		{ "竜殺しのオグライン",       30000,  195, 112,  4,  8, RACE_DWARF},
#else
		{ "Arnold the Beastly",      5000,   210, 115,  6,  6, RACE_BARBARIAN},
		{ "Arndal Beast-Slayer",        10000,  185, 110,  5,  9, RACE_HALF_ELF},
		{ "Eddie Beast-Master",         25000,  190, 115,  5,  7, RACE_HALF_ORC},
		{ "Oglign Dragon-Slayer",       30000,  195, 112,  4,  8, RACE_DWARF},
		{ "Drew the Skilled",      10000,   210, 115,  6,  6, RACE_HUMAN},
		{"Orrax Dragonson",        15000,  185, 110,  5,  9, RACE_DRACONIAN},
		{ "Anthrax Disease-Carrier",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
		{ "Arkhoth the Stout",       30000,  195, 112,  4,  8, RACE_DWARF},
		{ "Sarlyas the Rotten",      5000,   210, 115,  6,  6, RACE_ZOMBIE},
		{"Tuethic Bare-Bones",        15000,  185, 110,  5,  9, RACE_SKELETON},
		{ "Bilious",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
		{ "Fasgul",       30000,  195, 112,  4,  8, RACE_ZOMBIE},
		{ "Ellefris the Paladin",      10000,   210, 115,  6,  6, RACE_BARBARIAN},
		{"K'trrik'k",        15000,  185, 110,  5,  9, RACE_KLACKON},
		{ "Drocus Spiderfriend",         25000,  190, 115,  5,  7, RACE_DARK_ELF},
		{ "Fungus Giant-Slayer",       30000,  195, 112,  4,  8, RACE_DWARF},
		{ "Delantha",      10000,   210, 115,  6,  6, RACE_ELF},
		{"Solvistani the Ranger",        15000,  185, 110,  5,  9, RACE_HALF_ELF},
		{ "Xoril the Slow",         25000,  190, 115,  5,  7, RACE_GOLEM},
		{ "Aeon Flux",       20000,  195, 112,  4,  8, RACE_HALF_ELF},
		{ "Nadoc the Strong",      10000,   210, 115,  6,  6, RACE_HOBBIT},
		{"Eramog the Weak",        15000,  185, 110,  5,  9, RACE_KOBOLD},
		{ "Eowilith the Fair",         25000,  190, 115,  5,  7, RACE_VAMPIRE},
		{ "Huimog Balrog-Slayer",       30000,  195, 112,  4,  8, RACE_HALF_ORC},
		{ "Peadus the Cruel",      5000,   210, 115,  6,  6, RACE_HUMAN},
		{ "Vamog Slayer",        15000,  185, 110,  5,  9, RACE_HALF_OGRE},
		{ "Hooshnak the Vicious",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
		{ "Balenn War-Dancer",       30000,  195, 112,  4,  8, RACE_BARBARIAN},
		{ "Arnold the Beastly",      5000,   210, 115,  6,  6, RACE_BARBARIAN},
		{ "Arndal Beast-Slayer",        10000,  185, 110,  5,  9, RACE_HALF_ELF},
		{ "Eddie Beast-Master",         25000,  190, 115,  5,  7, RACE_HALF_ORC},
		{ "Oglign Dragon-Slayer",       30000,  195, 112,  4,  8, RACE_DWARF},
#endif
	},
	{
		/* Temple - 22 unique names */
#ifdef JP
		{ "質素なルードヴィヒ",         5000,   175,  109,   6,  15,  RACE_HUMAN},
		{ "パラディンのガンナー",       10000,  185,  110,   5,  23,  RACE_HUMAN},
		{ "選ばれしトリン",                     25000,  180,  107,   6,  20,  RACE_ELF},
		{ "賢明なるサラストロ",                     30000,  185,  109,   5,  15,  RACE_DWARF},
		{ "パーシヴァル卿",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "神聖なるアセナス",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "マッキノン",         10000,   175, 109,  6, 15, RACE_HUMAN},
		{ "謹み婦人",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
		{ "ドルイドのハシュニック",           25000,  180, 107,  6, 20, RACE_HOBBIT},
		{ "フィナク",          30000,  185, 109,  5, 15, RACE_YEEK},
		{ "クリキック",         10000,   175, 109,  6, 15, RACE_KLACKON},
		{ "荒くれ者モリヴァル",         15000,  185, 110,  5, 23, RACE_ELF},
		{ "暗きホシャック",           25000,  180, 107,  6, 20, RACE_IMP},
		{ "賢者アタール",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "清きイベニッド",         10000,   175, 109,  6, 15, RACE_HUMAN},
		{ "エリディシュ",         15000,  185, 110,  5, 23, RACE_HALF_TROLL},
		{ "呪術師ヴルドゥシュ",           25000,  180, 107,  6, 20, RACE_HALF_OGRE},
		{ "狂戦士ハオブ",          30000,  185, 109,  5, 15, RACE_BARBARIAN},
		{ "若きプルーグディシュ",         10000,   175, 109,  6, 15, RACE_HALF_OGRE},
		{ "きちがいラムワイズ",         15000,  185, 110,  5, 23, RACE_YEEK},
		{ "有徳者ムワート",           25000,  180, 107,  6, 20, RACE_KOBOLD},
		{ "弱虫ダードバード",          30000,  185, 109,  5, 15, RACE_SPECTRE},
		{ "質素なルードヴィヒ",         5000,   175,  109,   6,  15,  RACE_HUMAN},
		{ "パラディンのガンナー",       10000,  185,  110,   5,  23,  RACE_HUMAN},
		{ "選ばれしトリン",                     25000,  180,  107,   6,  20,  RACE_ELF},
		{ "賢明なるサラストロ",                     30000,  185,  109,   5,  15,  RACE_DWARF},
		{ "パーシヴァル卿",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "神聖なるアセナス",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "マッキノン",         10000,   175, 109,  6, 15, RACE_HUMAN},
		{ "謹み婦人",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
		{ "ドルイドのハシュニック",           25000,  180, 107,  6, 20, RACE_HOBBIT},
		{ "フィナク",          30000,  185, 109,  5, 15, RACE_YEEK},
#else
		{ "Ludwig the Humble",          5000,   175, 109,  6, 15, RACE_DWARF},
		{ "Gunnar the Paladin",         10000,  185, 110,  5, 23, RACE_HALF_TROLL},
		{ "Torin the Chosen",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "Sarastro the Wise",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "Sir Parsival the Pure",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "Asenath the Holy",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "McKinnon",         10000,   175, 109,  6, 15, RACE_HUMAN},
		{ "Mistress Chastity",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
		{ "Hashnik the Druid",           25000,  180, 107,  6, 20, RACE_HOBBIT},
		{ "Finak",          30000,  185, 109,  5, 15, RACE_YEEK},
		{ "Krikkik",         10000,   175, 109,  6, 15, RACE_KLACKON},
		{ "Morival the Wild",         15000,  185, 110,  5, 23, RACE_ELF},
		{ "Hoshak the Dark",           25000,  180, 107,  6, 20, RACE_IMP},
		{ "Atal the Wise",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "Ibenidd the Chaste",         10000,   175, 109,  6, 15, RACE_HUMAN},
		{ "Eridish",         15000,  185, 110,  5, 23, RACE_HALF_TROLL},
		{ "Vrudush the Shaman",           25000,  180, 107,  6, 20, RACE_HALF_OGRE},
		{ "Haob the Berserker",          30000,  185, 109,  5, 15, RACE_BARBARIAN},
		{ "Proogdish the Youthfull",         10000,   175, 109,  6, 15, RACE_HALF_OGRE},
		{ "Lumwise the Mad",         15000,  185, 110,  5, 23, RACE_YEEK},
		{ "Muirt the Virtuous",           25000,  180, 107,  6, 20, RACE_KOBOLD},
		{ "Dardobard the Weak",          30000,  185, 109,  5, 15, RACE_SPECTRE},
		{ "Ludwig the Humble",          5000,   175, 109,  6, 15, RACE_DWARF},
		{ "Gunnar the Paladin",         10000,  185, 110,  5, 23, RACE_HALF_TROLL},
		{ "Torin the Chosen",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "Sarastro the Wise",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "Sir Parsival the Pure",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "Asenath the Holy",          30000,  185, 109,  5, 15, RACE_HUMAN},
		{ "McKinnon",         10000,   175, 109,  6, 15, RACE_HUMAN},
		{ "Mistress Chastity",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
		{ "Hashnik the Druid",           25000,  180, 107,  6, 20, RACE_HOBBIT},
		{ "Finak",          30000,  185, 109,  5, 15, RACE_YEEK},
#endif
	},
	{
		/* Alchemist - 26 unique names */
#ifdef JP
		{ "化学者マウザー",             10000,  190,  111,   5,   8,  RACE_HALF_ELF},
		{ "カオスのウィズル",   10000,  190,  110,   6,   8,  RACE_HOBBIT},
		{ "強欲ミダス",              15000,  200,  116,   6,   9,  RACE_GNOME},
		{ "貧弱ジャ=ファー",                   15000,  220,  111,   4,   9,  RACE_ELF},/*FIRST*/
		{ "カカルラカカル",           15000,  200, 116,  6,  9, RACE_KLACKON},
		{ "錬金術師ジャル=エス",       15000,  220, 111,  4,  9, RACE_ELF},
		{ "用心深いファネラス",         10000,  190, 111,  5,  8, RACE_DWARF},
		{ "キチガイのルンシー",         10000,  190, 110,  6,  8, RACE_HUMAN},
		{ "グランブルワース",           15000,  200, 116,  6,  9, RACE_GNOME},
		{ "フリッター",       15000,  220, 111,  4,  9, RACE_SPRITE},
		{ "ザリルス",         10000,  190, 111,  5,  8, RACE_HUMAN},
		{ "古きエグバート",         10000,  190, 110,  6,  8, RACE_DWARF},
		{ "誇り高きヴァリンドラ",           15000,  200, 116,  6,  9, RACE_HIGH_ELF},
		{ "錬金術師タエン",       15000,  220, 111,  4,  9, RACE_HUMAN},
		{ "巧言カイド",         10000,  190, 111,  5,  8, RACE_VAMPIRE},
		{ "暗きフリア",         10000,  190, 110,  6,  8, RACE_NIBELUNG},
		{ "質素なドムリ",           15000,  200, 116,  6,  9, RACE_DWARF},
		{ "魔の子ヤァジュッカ",       15000,  220, 111,  4,  9, RACE_IMP},
		{ "薬草師ジェララルドール",         10000,  190, 111,  5,  8, RACE_HIGH_ELF},
		{ "賢者オレラルダン",         10000,  190, 110,  6,  8, RACE_BARBARIAN},
		{ "デモニシストのフゾグロ",           15000,  200, 116,  6,  9, RACE_IMP},
		{ "錬金術師ドゥリアシュ",       15000,  220, 111,  4,  9, RACE_HALF_ORC},
		{ "強者ネリア",         10000,  190, 111,  5,  8, RACE_CYCLOPS},
		{ "辛口リグナス",         10000,  190, 110,  6,  8, RACE_HALF_ORC},
		{ "ティルバ",           15000,  200, 116,  6,  9, RACE_HOBBIT},
		{ "金持ちミリルドリック",       15000,  220, 111,  4,  9, RACE_HUMAN},

		{ "科学者マウザー",         10000,  190, 111,  5,  8, RACE_HALF_ELF},
		{ "カオスのウィズル",         10000,  190, 110,  6,  8, RACE_HOBBIT},
		{ "強欲ミダス",           15000,  200, 116,  6,  9, RACE_GNOME},
		{ "錬金術師ジャ=ファー",       15000,  220, 111,  4,  9, RACE_ELF},
		{ "カカルラカカル",           15000,  200, 116,  6,  9, RACE_KLACKON},
		{ "錬金術師ジャル=エス",       15000,  220, 111,  4,  9, RACE_ELF},
#else
		{ "Mauser the Chemist",         10000,  190, 111,  5,  8, RACE_HALF_ELF},
		{ "Wizzle the Chaotic",         10000,  190, 110,  6,  8, RACE_HOBBIT},
		{ "Midas the Greedy",           15000,  200, 116,  6,  9, RACE_GNOME},
		{ "Ja-Far the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
		{ "Kakalrakakal",           15000,  200, 116,  6,  9, RACE_KLACKON},
		{ "Jal-Eth the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
		{ "Fanelath the Cautious",         10000,  190, 111,  5,  8, RACE_DWARF},
		{ "Runcie the Insane",         10000,  190, 110,  6,  8, RACE_HUMAN},
		{ "Grumbleworth",           15000,  200, 116,  6,  9, RACE_GNOME},
		{ "Flitter",       15000,  220, 111,  4,  9, RACE_SPRITE},
		{ "Xarillus",         10000,  190, 111,  5,  8, RACE_HUMAN},
		{ "Egbert the Old",         10000,  190, 110,  6,  8, RACE_DWARF},
		{ "Valindra the Proud",           15000,  200, 116,  6,  9, RACE_HIGH_ELF},
		{ "Taen the Alchemist",       15000,  220, 111,  4,  9, RACE_HUMAN},
		{ "Cayd the Sweet",         10000,  190, 111,  5,  8, RACE_VAMPIRE},
		{ "Fulir the Dark",         10000,  190, 110,  6,  8, RACE_NIBELUNG},
		{ "Domli the Humble",           15000,  200, 116,  6,  9, RACE_DWARF},
		{ "Yaarjukka Demonspawn",       15000,  220, 111,  4,  9, RACE_IMP},
		{ "Gelaraldor the Herbmaster",         10000,  190, 111,  5,  8, RACE_HIGH_ELF},
		{ "Olelaldan the Wise",         10000,  190, 110,  6,  8, RACE_BARBARIAN},
		{ "Fthoglo the Demonicist",           15000,  200, 116,  6,  9, RACE_IMP},
		{ "Dridash the Alchemist",       15000,  220, 111,  4,  9, RACE_HALF_ORC},
		{ "Nelir the Strong",         10000,  190, 111,  5,  8, RACE_CYCLOPS},
		{ "Lignus the Pungent",         10000,  190, 110,  6,  8, RACE_HALF_ORC},
		{ "Tilba",           15000,  200, 116,  6,  9, RACE_HOBBIT},
		{ "Myrildric the Wealthy",       15000,  220, 111,  4,  9, RACE_HUMAN},

		{ "Mauser the Chemist",         10000,  190, 111,  5,  8, RACE_HALF_ELF},
		{ "Wizzle the Chaotic",         10000,  190, 110,  6,  8, RACE_HOBBIT},
		{ "Midas the Greedy",           15000,  200, 116,  6,  9, RACE_GNOME},
		{ "Ja-Far the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
		{ "Kakalrakakal",           15000,  200, 116,  6,  9, RACE_KLACKON},
		{ "Jal-Eth the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
#endif
	},

	{
		/* Magic Shop - 23 unique names */
#ifdef JP
		{ "ソーサラーのロ=パン",       20000,  200,  110,   7,   8,  RACE_HALF_ELF},
		{ "偉大なるブガービイ",         20000,  215,  113,   6,  10,  RACE_GNOME},
		{ "イェンダーの魔法使い",     30000,  200,  110,   7,  10,  RACE_HUMAN},
		{ "死霊使いリャク",30000,      175,  110,   5,  11,  RACE_HIGH_ELF},
		{ "魔術師スキドゥニー",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "幻術師キリア",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "死霊術師ニッキ",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "ソロストラン",        15000,  200, 110,  7,  8, RACE_SPRITE},
		{ "烏賊口アチシェ",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
		{ "貴族のカザ",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
		{ "暗きファジル",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "偉大なるケルドーン",        15000,  200, 110,  7,  8, RACE_DWARF},
		{ "フィランスロプス",         20000,  215, 113,  6, 10, RACE_HOBBIT},
		{ "魔女のアグナー",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "死霊術師ビュリアンス",       30000,  175, 110,  5, 11, RACE_BEASTMAN},
		{ "ハイメイジのヴイラク",        15000,  200, 110,  7,  8, RACE_BEASTMAN},
		{ "知恵者マディッシュ",         20000,  215, 113,  6, 10, RACE_BEASTMAN},
		{ "ファレブリンボール",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
		{ "陰険フェリル=ガンド",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "呪術師サレゴード",        15000,  200, 110,  7,  8, RACE_BARBARIAN},
		{ "神秘家クトゥアロス",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
		{ "幻術師イベリ",       30000,  200, 110,  7, 10, RACE_SKELETON},
		{ "死霊術師ヘトー",       30000,  175, 110,  5, 11, RACE_YEEK},
		{ "魔術師ロ=パン",        20000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "偉大なるブガービイ",         20000,  215, 113,  6, 10, RACE_GNOME},
		{ "イェンダーの魔法使い",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "死霊術師リャク",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "魔術師スキドゥニー",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "幻術師キリア",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "死霊術師ニッキ",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "ソロストラン",        15000,  200, 110,  7,  8, RACE_SPRITE},
		{ "烏賊口アチシェ",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#else
		{ "Lo Pan the Sorcerer",        20000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "Buggerby the Great",         20000,  215, 113,  6, 10, RACE_GNOME},
		{ "The Wizard of Yendor",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "Rjak the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "Skidney the Sorcerer",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "Kyria the Illusionist",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "Nikki the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "Solostoran",        15000,  200, 110,  7,  8, RACE_SPRITE},
		{ "Achshe the Tentacled",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
		{ "Kaza the Noble",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
		{ "Fazzil the Dark",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "Keldorn the Grand",        15000,  200, 110,  7,  8, RACE_DWARF},
		{ "Philanthropus",         20000,  215, 113,  6, 10, RACE_HOBBIT},
		{ "Agnar the Enchantress",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "Buliance the Necromancer",       30000,  175, 110,  5, 11, RACE_BEASTMAN},
		{ "Vuirak the High-Mage",        15000,  200, 110,  7,  8, RACE_BEASTMAN},
		{ "Madish the Smart",         20000,  215, 113,  6, 10, RACE_BEASTMAN},
		{ "Falebrimbor",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
		{ "Felil-Gand the Subtle",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "Thalegord the Shaman",        15000,  200, 110,  7,  8, RACE_BARBARIAN},
		{ "Cthoaloth the Mystic",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
		{ "Ibeli the Illusionist",       30000,  200, 110,  7, 10, RACE_SKELETON},
		{ "Heto the Necromancer",       30000,  175, 110,  5, 11, RACE_YEEK},
		{ "Lo Pan the Sorcerer",        20000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "Buggerby the Great",         20000,  215, 113,  6, 10, RACE_GNOME},
		{ "The Wizard of Yendor",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "Rjak the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "Skidney the Sorcerer",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "Kyria the Illusionist",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "Nikki the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
		{ "Solostoran",        15000,  200, 110,  7,  8, RACE_SPRITE},
		{ "Achshe the Tentacled",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#endif
	},
	{
		/* Black Market - 32 unique names */
#ifdef JP
		{ "ガリー=ギガズ",            20000,  250,  150,  10,   5,  RACE_HALF_TROLL},
		{ "ゴブリンのヒストーア",       20000,  250,  150,  10,   5,  RACE_HALF_ORC},
		{ "フェレンギ人クアーク",           30000,  250,  150,  10,   5,  RACE_HUMAN},
		{ "公正なる(?)トッピ",                     30000,  250,  150,  10,   5,  RACE_ELF},
		{ "死人ヴァッサ",             20000,  250, 150, 10,  5, RACE_ZOMBIE},
		{ "裏切り者カイン",          20000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "ブボニカス",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
		{ "コープスライト",           30000,  250, 150, 10,  5, RACE_SPECTRE},
		{ "血に飢えしパリッシュ",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "ヴァイル",          20000,  250, 150, 10,  5, RACE_SKELETON},
		{ "信頼のプレンティス",          30000,  250, 150, 10,  5, RACE_SKELETON},
		{ "人間殺しのグリエラ",           30000,  250, 150, 10,  5, RACE_IMP},
		{ "エンジェル",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "水膨れフロツァム",          20000,  250, 150, 10,  5, RACE_ZOMBIE},
		{ "ニーヴァル",          30000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "明るいアナスタシア",           30000,  250, 150, 10,  5, RACE_SPECTRE},
		{ "死霊術師チャリティー", 20000,  250, 150, 10,  5, RACE_DARK_ELF},
		{ "ボクサーのプグナシオス",          20000,  250, 150, 10,  5, RACE_HALF_ORC},
		{ "幸運なフットソア",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
		{ "光指のシドリア",           30000,  250, 150, 10,  5, RACE_HUMAN},
		{ "手品師リアソー",                 20000,  250, 150, 10,  5, RACE_HOBBIT},
		{ "やりくり上手のジャナッカ",          20000,  250, 150, 10,  5, RACE_GNOME},
		{ "悪党シーナ",          30000,  250, 150, 10,  5, RACE_GNOME},
		{ "大爪アルニッキ",           30000,  250, 150, 10,  5, RACE_DRACONIAN},
		{ "貧乏チャエアンド",                 20000,  250, 150, 10,  5, RACE_HUMAN},
		{ "山賊アファードーフ",          20000,  250, 150, 10,  5, RACE_BARBARIAN},
		{ "強欲ラザクスル",          30000,  250, 150, 10,  5, RACE_MIND_FLAYER},
		{ "ファラレウィン",           30000,  250, 150, 10,  5, RACE_SPRITE},
		{ "しわしわヴォスール",                 20000,  250, 150, 10,  5, RACE_NIBELUNG},
		{ "ハンサムなアラオード",          20000,  250, 150, 10,  5, RACE_AMBERITE},
		{ "負け犬セラドフリド",          30000,  250, 150, 10,  5, RACE_HUMAN},
		{ "片足のエルーロ",           30000,  250, 150, 10,  5, RACE_HALF_OGRE},
#else
		{ "Gary Gygaz",                 20000,  250, 150, 10,  5, RACE_HALF_TROLL},
		{ "Histor the Goblin",          20000,  250, 150, 10,  5, RACE_HALF_ORC},
		{ "Quark the Ferengi",          30000,  250, 150, 10,  5, RACE_DWARF},
		{ "Topi the Fair(?)",           30000,  250, 150, 10,  5, RACE_HUMAN},
		{ "Vhassa the Dead",             20000,  250, 150, 10,  5, RACE_ZOMBIE},
		{ "Kyn the Treacherous",          20000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "Bubonicus",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
		{ "Corpselight",           30000,  250, 150, 10,  5, RACE_SPECTRE},
		{ "Parrish the Bloodthirsty",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "Vile",          20000,  250, 150, 10,  5, RACE_SKELETON},
		{ "Prentice the Trusted",          30000,  250, 150, 10,  5, RACE_SKELETON},
		{ "Griella Humanslayer",           30000,  250, 150, 10,  5, RACE_IMP},
		{ "Angel",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "Flotsam the Bloated",          20000,  250, 150, 10,  5, RACE_ZOMBIE},
		{ "Nieval",          30000,  250, 150, 10,  5, RACE_VAMPIRE},
		{ "Anastasia the Luminous",           30000,  250, 150, 10,  5, RACE_SPECTRE},
		{ "Charity the Necromancer", 20000,  250, 150, 10,  5, RACE_DARK_ELF},
		{ "Pugnacious the Pugilist",          20000,  250, 150, 10,  5, RACE_HALF_ORC},
		{ "Footsore the Lucky",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
		{ "Sidria Lighfingered",           30000,  250, 150, 10,  5, RACE_HUMAN},
		{ "Riatho the Juggler",                 20000,  250, 150, 10,  5, RACE_HOBBIT},
		{ "Janaaka the Shifty",          20000,  250, 150, 10,  5, RACE_GNOME},
		{ "Cina the Rogue",          30000,  250, 150, 10,  5, RACE_GNOME},
		{ "Arunikki Greatclaw",           30000,  250, 150, 10,  5, RACE_DRACONIAN},
		{ "Chaeand the Poor",                 20000,  250, 150, 10,  5, RACE_HUMAN},
		{ "Afardorf the Brigand",          20000,  250, 150, 10,  5, RACE_BARBARIAN},
		{ "Lathaxl the Greedy",          30000,  250, 150, 10,  5, RACE_MIND_FLAYER},
		{ "Falarewyn",           30000,  250, 150, 10,  5, RACE_SPRITE},
		{ "Vosur the Wrinkled",                 20000,  250, 150, 10,  5, RACE_NIBELUNG},
		{ "Araord the Handsome",          20000,  250, 150, 10,  5, RACE_AMBERITE},
		{ "Theradfrid the Loser",          30000,  250, 150, 10,  5, RACE_HUMAN},
		{ "One-Legged Eroolo",           30000,  250, 150, 10,  5, RACE_HALF_OGRE},
#endif
	},
	{
		/* Home */
#ifdef JP
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
		{ "我が家",                          0,      100, 100,  0, 99, 99},
#else
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
#endif

	},

	{
		/* Bookstore - 21 unique names */
#ifdef JP
		{ "強欲ドラフ", 10000, 175, 108, 4, 12, RACE_HUMAN},
		{ "賢者オドナー", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "中立のガンダー", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
		{ "忍耐の人ロ=シャ", 30000, 140, 105, 6, 12, RACE_ELF},
		{ "ランドルフ・カーター", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "隼のサライ", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "千里眼ボドリル", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "沈黙のヴェオロイン", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
		{ "学者のヴァンシラス", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
		{ "物書きオセイン", 15000, 175, 108, 4, 12, RACE_SKELETON},
		{ "本の虫オルヴァー", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
		{ "浅井墓男", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
		{ "デスマスク", 30000, 140, 105, 6, 12, RACE_ZOMBIE},
		{ "学者のアスーヌ", 15000, 175, 108, 4, 12, RACE_MIND_FLAYER},
		{ "死人のプリランド", 20000, 120, 105, 6, 16, RACE_ZOMBIE},
		{ "鉄のロナール", 25000, 120, 110, 7, 19, RACE_GOLEM},
#else
		{ "Dolaf the Greedy", 10000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Odnar the Sage", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "Gandar the Neutral", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
		{ "Ro-sha the Patient", 30000, 140, 105, 6, 12, RACE_ELF},
		{ "Randolph Carter", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Sarai the Swift", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Bodril the Seer", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "Veloin the Quiet", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
		{ "Vanthylas the Learned", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
		{ "Ossein the Literate", 15000, 175, 108, 4, 12, RACE_SKELETON},
		{ "Olvar Bookworm", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
		{ "Shallowgrave", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
		{ "Death Mask", 30000, 140, 105, 6, 12, RACE_ZOMBIE},
		{ "Asuunu the Learned", 15000, 175, 108, 4, 12, RACE_MIND_FLAYER},
		{ "Prirand the Dead", 20000, 120, 105, 6, 16, RACE_ZOMBIE},
		{ "Ronar the Iron", 25000, 120, 110, 7, 19, RACE_GOLEM},
#endif
#ifdef JP
		{ "ガリル=ガミル", 30000, 140, 105, 6, 12, RACE_ELF},
		{ "本食いローバグ", 15000, 175, 108, 4, 12, RACE_KOBOLD},
		{ "キリアリキーク", 20000, 120, 105, 6, 16, RACE_KLACKON},
		{ "静かなるリリン", 25000, 120, 110, 7, 19, RACE_DWARF},
		{ "王者イサング", 30000, 140, 105, 6, 12, RACE_HIGH_ELF},
		{ "強欲ドラフ", 10000, 175, 108, 4, 12, RACE_HUMAN},
		{ "賢者オドナー", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "中立のガンダー", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
		{ "忍耐の人ロ=シャ", 30000, 140, 105, 6, 12, RACE_ELF},
		{ "ランドルフ・カーター", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "隼サライ", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "千里眼ボドリル", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "沈黙のヴェオロイン", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
		{ "学者のヴァンシラス", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
		{ "物書きオセイン", 15000, 175, 108, 4, 12, RACE_SKELETON},
		{ "本の虫オルヴァー", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
#else
		{ "Galil-Gamir", 30000, 140, 105, 6, 12, RACE_ELF},
		{ "Rorbag Book-Eater", 15000, 175, 108, 4, 12, RACE_KOBOLD},
		{ "Kiriarikirk", 20000, 120, 105, 6, 16, RACE_KLACKON},
		{ "Rilin the Quiet", 25000, 120, 110, 7, 19, RACE_DWARF},
		{ "Isung the Lord", 30000, 140, 105, 6, 12, RACE_HIGH_ELF},
		{ "Dolaf the Greedy", 10000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Odnar the Sage", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "Gandar the Neutral", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
		{ "Ro-sha the Patient", 30000, 140, 105, 6, 12, RACE_ELF},
		{ "Randolph Carter", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Sarai the Swift", 15000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Bodril the Seer", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "Veloin the Quiet", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
		{ "Vanthylas the Learned", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
		{ "Ossein the Literate", 15000, 175, 108, 4, 12, RACE_SKELETON},
		{ "Olvar Bookworm", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
#endif
	},

	{
		/* Museum */
#ifdef JP
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
		{ "博物館",                          0,      100, 100,  0, 99, 99},
#else
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
#endif

	},

};


/*!
 * @brief 加速値による実質速度修正倍率テーブル /
 * This table allows quick conversion from "speed" to "energy"
 * @details
 * <pre>
 * The basic function WAS ((S>=110) ? (S-110) : (100 / (120-S)))
 * Note that table access is *much* quicker than computation.
 *
 * Note that the table has been changed at high speeds.  From
 * "Slow (-40)" to "Fast (+30)" is pretty much unchanged, but
 * at speeds above "Fast (+30)", one approaches an asymptotic
 * effective limit of 50 energy per turn.  This means that it
 * is relatively easy to reach "Fast (+30)" and get about 40
 * energy per turn, but then speed becomes very "expensive",
 * and you must get all the way to "Fast (+50)" to reach the
 * point of getting 45 energy per turn.  After that point,
 * furthur increases in speed are more or less pointless,
 * except to balance out heavy inventory.
 *
 * Note that currently the fastest monster is "Fast (+30)".
 *
 * It should be possible to lower the energy threshhold from
 * 100 units to 50 units, though this may interact badly with
 * the (compiled out) small random energy boost code.  It may
 * also tend to cause more "clumping" at high speeds.
 * </pre>
 */
const byte extract_energy[200] =
{
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* S-50 */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* S-40 */     2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
	/* S-30 */     2,  2,  2,  2,  2,  2,  2,  3,  3,  3,
	/* S-20 */     3,  3,  3,  3,  3,  4,  4,  4,  4,  4,
	/* S-10 */     5,  5,  5,  5,  6,  6,  7,  7,  8,  9,
	/* Norm */    10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
	/* F+10 */    20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
	/* F+20 */    30, 31, 32, 33, 34, 35, 36, 36, 37, 37,
	/* F+30 */    38, 38, 39, 39, 40, 40, 40, 41, 41, 41,
	/* F+40 */    42, 42, 42, 43, 43, 43, 44, 44, 44, 44,
	/* F+50 */    45, 45, 45, 45, 45, 46, 46, 46, 46, 46,
	/* F+60 */    47, 47, 47, 47, 47, 48, 48, 48, 48, 48,
	/* F+70 */    49, 49, 49, 49, 49, 49, 49, 49, 49, 49,
	/* Fast */    49, 49, 49, 49, 49, 49, 49, 49, 49, 49,
};



/*!
 * @brief 基本必要経験値テーブル /
 * Base experience levels, may be adjusted up for race and/or class
 */
const s32b player_exp[PY_MAX_LEVEL] =
{
	10,
	25,
	45,
	70,
	100,
	140,
	200,
	280,
	380,/*10*/
	500,
	650,
	850,
	1100,
	1400,
	1800,
	2300,
	2900,
	3600,
	4400,/*20*/
	5400,
	6800,
	8400,
	10200,
	12500,
	17500,
	25000,
	35000L,
	50000L,
	75000L,/*30*/
	100000L,
	150000L,
	200000L,
	275000L,
	350000L,
	450000L,
	550000L,
	700000L,
	850000L,
	1000000L,/*40*/
	1250000L,
	1500000L,
	1800000L,
	2100000L,
	2400000L,
	2700000L,
	3000000L,
	3500000L,
	4000000L,
	4500000L,/*50*/
	5000000L
};


/*!
 * @brief 基本必要強化値テーブル(アンドロイド専用)
 */
const s32b player_exp_a[PY_MAX_LEVEL] =
{
	20,
	50,
	100,
	170,
	280,
	430,
	650,
	950,
	1400,/*10*/
	1850,
	2300,
	2900,
	3600,
	4400,
	5400,
	6800,
	8400,
	10400,
	12500,/*20*/
	17500,
	25000,
	35000,
	50000L,
	75000L,
	100000L,
	150000L,
	200000L,
	275000L,
	350000L,/*30*/
	450000L,
	550000L,
	650000L,
	800000L,
	950000L,
	1100000L,
	1250000L,
	1400000L,
	1550000L,
	1700000L,/*40*/
	1900000L,
	2100000L,
	2300000L,
	2550000L,
	2800000L,
	3050000L,
	3300000L,
	3700000L,
	4100000L,
	4500000L,/*50*/
	5000000L
};


/*!
 * @brief 性別表記 /
 * Player Sexes
 * @details
 * <pre>
 *      Title,
 *      Winner
 * </pre>
 */
const player_sex sex_info[MAX_SEXES] =
{
	{
#ifdef JP
		"女性",
		"クイーン",
#endif
		"Female",
		"Queen"
	},
	{
#ifdef JP
		"男性",
		"キング",
#endif
		"Male",
		"King"
	}
};


/*!
 * @brief 種族情報 /
 * Player Races
 * @details
 * <pre>
 *      Title,
 *      {STR,INT,WIS,DEX,CON,CHR},
 *      r_dis, r_dev, r_sav, r_stl, r_srh, r_fos, r_thn, r_thb,
 *      hitdie, exp base,
 *      Age (Base, Mod),
 *      Male (Hgt, Wgt),
 *      Female (Hgt, Wgt)
 *      infra,
 *      class-choices
 * </pre>
 */
const player_race race_info[MAX_RACES] =
{
	{
#ifdef JP
		"人間",
#endif
		"Human",

		{  0,  0,  0,  0,  0,  0 },
		0,  0,  0,  0,  0,  10,  0,  0,
		10,  100,
		14,  6,
		72,  6, 180, 25,
		66,  4, 150, 20,
		0,
		0xFFFFFFF,
	},
	{
#ifdef JP
		"ハーフエルフ",
#endif
		"Half-Elf",

		{ -1,  1,  1,  1, -1,  1 },
		2,  3,  2,  1, 6,  11, -1,  5,
		9,  110,
		24, 16,
		66,  6, 130, 15,
		62,  6, 100, 10,
		2,
		0xE77E7FF,
	},
	{
#ifdef JP
		"エルフ",
#endif
		"Elf",

		{ -1,  2,  0,  1, -2,  2 },
		5,  6,  4,  2, 8,  12, -5, 15,
		8,  120,
		75, 75,
		60,  4, 100,  6,
		54,  4, 80,  6,
		3,
		0xE77E75B,

	},
	{
#ifdef JP
		"ホビット",
#endif
		"Hobbit",

		{ -2,  1,  1,  3,  2,  1 },
		15, 12, 10, 5, 12,  15, -10, 20,
		7,  110,
		21, 12,
		36,  3, 60,  3,
		33,  3, 50,  3,
		4,
		0xF6FFC0B,
	},
	{
#ifdef JP
		"ノーム",
#endif
		"Gnome",

		{ -1,  2,  0,  2,  1, -1 },
		10, 8, 7,  3, 6,  13, -8, 12,
		8,  120,
		50, 40,
		42,  3, 90,  6,
		39,  3, 75,  3,
		4,
		0xF67D60F,
	},
	{
#ifdef JP
		"ドワーフ",
#endif
		"Dwarf",

		{  2, -2,  2, -2,  2, -1 },
		2,  7,  6,  -1,  7,  10, 15,  0,
		11,  125,
		35, 15,
		48,  3, 150, 10,
		46,  3, 120, 10,
		5,
		0x1890005,
	},
	{
#ifdef JP
		"ハーフオーク",
#endif
		"Half-Orc",

		{  2, -1,  0,  0,  1, -2 },
		-3, -3, -2,  -1,  0, 7, 12, -5,
		10,  110,
		11,  4,
		66,  1, 150,  5,
		62,  1, 120,  5,
		3,
		0xDD8818D,
	},
	{
#ifdef JP
		"ハーフトロル",
#endif
		"Half-Troll",

		{ 4, -4, -1, -3,  3, -3 },
		-5, -8, -5, -2,  -1, 5, 20, -10,
		12,  125,
		20, 10,
		96, 10, 250, 50,
		84,  8, 225, 40,
		3,
		0x0880005,
	},
	{
#ifdef JP
		"アンバライト",
#endif
		"Amberite",

		{  1,  2,  2,  2,  3,  2 },
		4,  5,  3,  2, 3, 13, 15, 10,
		10,  225,
		50, 50,
		82, 5, 190, 20,
		78,  6, 180, 15,
		0,
		0xFFFF7FF,
	},
	{
#ifdef JP
		"ハイエルフ",
#endif
		"High-Elf",

		{  1,  3,  -1,  3,  1,  3 },
		4,  13, 12,  4,  3, 14, 10, 25,
		10,  200,
		100, 30,
		90, 10, 190, 20,
		82, 10, 180, 15,
		4,
		0xF77E75B,
	},
	{
#ifdef JP
		"野蛮人",
#endif
		"Barbarian",

		{ 3, -2,  -1,  1,  2, 0 },
		-2, -10, 2,  -1,  1, 7, 12, 10,
		11, 120,
		14, 8,
		82, 5, 200, 20,
		78,  6, 190, 15,
		0,
		0x5C0A09D,
	},
	{
#ifdef JP
		"ハーフオーガ",
#endif
		"Half-Ogre",

		{ 3, -2, 0, -1, 3, -2 },
		-3, -5, -3, -2, -1, 5, 20, 0,
		12,  145,
		40, 10,
		92, 10, 255, 60,
		80,  8, 235, 60,
		3,
		0x0A80407,
	},
	{
#ifdef JP
		"半巨人",
#endif
		"Half-Giant",

		{ 4, -2, -2, -2, 3, -2 },
		-6, -8, -3, -2, -1, 5, 25, 5,
		13, 160,
		40, 10,
		100,10, 255, 65,
		80, 10, 240, 64,
		3,
		0x8880011,
	},
	{
#ifdef JP
		"半タイタン",
#endif
		"Half-Titan",

		{ 5, 1, 2, -2, 3, 1 },
		-5, 5, 1, -2, 1, 8, 25, 0,
		14, 255,
		100,30,
		111, 11, 255, 86,
		99, 11, 250, 86,
		0,
		0x23D4727,
	},
	{
#ifdef JP
		"サイクロプス",
#endif
		"Cyclops",

		{ 4, -3, -2, -3, 4, -3 },
		-4, -5, -3, -2, -2, 5, 20, 12,
		13, 150,
		50, 24,
		92, 10, 255, 60,
		80,  8, 235, 60,
		1,
		0x0888005,
	},
	{
#ifdef JP
		"イーク",
#endif
		"Yeek",

		{ -2, 1, 1, 1, -2, -4 },
		2, 4, 6, 3, 5, 15, -5, -5,
		7, 100,
		14, 3,
		50,  3, 90,  6,
		50,  3, 75,  3,
		2,
		0x667360F,
	},
	{
#ifdef JP
		"クラッコン",
#endif
		"Klackon",

		{ 2, -1, -1, 1, 2, -1 },
		10, 5, 3, 0, -1, 10, 5, 5,
		12, 190,
		20, 3,
		60,  3, 80,  4,
		54,  3, 70,  4,
		2,
		0x04D8011,
	},
	{
#ifdef JP
		"コボルド",
#endif
		"Kobold",

		{ 1, -1, 0, 1, 0, -2 },
		-2, -3, -1, -1, 1, 8, 10, -8,
		9, 125,
		11,  3,
		60,  1, 130,  5,
		55,  1, 100,  5,
		3,
		0x444A009,
	},
	{
#ifdef JP
		"ニーベルング",
#endif
		"Nibelung",

		{ 1, -1, 2, 0, 2, -2 },
		3, 5, 6, 1, 5, 10, 9, 0,
		11, 170,
		40, 12,
		43,  3, 92,  6,
		40,  3, 78,  3,
		5,
		0x569040F,
	},
	{
#ifdef JP
		"ダークエルフ",
#endif
		"Dark-Elf",

		{ -1, 3, 2, 2, -2, 1 },
		5, 10, 12, 3, 8, 12, -5, 10,
		9, 150,
		75, 75,
		60,  4, 100,  6,
		54,  4, 80,  6,
		5,
		0xE77C7DF,
	},
	{
#ifdef JP
		"ドラコニアン",
#endif
		"Draconian",

		{ 2, 1, 1, 1, 2, -1 },
		-2, 5, 2, 0, 1, 10, 5, 5,
		11, 220,
		75, 33,
		76,  1, 160,  5,
		72,  1, 130,  5,
		2,
		0x7FFE757,
	},
	{
#ifdef JP
		"マインドフレア",
#endif
		"Mindflayer",

		{ -3, 4, 4, 0, -2, -3 },
		10, 15, 9, 2, 5, 12, -10, -5,
		9, 140,
		100, 25,
		68,  6, 142, 15,
		63,  6, 112, 10,
		4,
		0x2334746,
	},
	{
#ifdef JP
		"インプ",
#endif
		"Imp",

		{ 0, -1, -1, 1, 2, -1 },
		-3, 2, -1, 1, -1, 10, 5, -5,
		10, 115,
		13,  4,
		68,  1, 150,  5,
		64,  1, 120,  5,
		3,
		0xDB537CB,
	},
	{
#ifdef JP
		"ゴーレム",
#endif
		"Golem",

		{ 4, -5, -5, -2, 4, -2 },
		-5, -5, 6, -1, -1, 8, 20, 0,
		12, 200,
		1, 100,
		66,  1, 200,  6,
		62,  1, 180,  6,
		4,
		0x0800001,
	},
	{
#ifdef JP
		"骸骨",
#endif
		"Skeleton",

		{ 0, 1, -2, 0, 1, -2 },
		-5, 0, 3, -1, -1, 8, 10, 0,
		10, 145,
		100, 35,
		72,  6, 50, 5,
		66,  4, 50, 5,
		2,
		0x234070F,
	},
	{
#ifdef JP
		"ゾンビ",
#endif
		"Zombie",

		{ 2, -6, -6, 1, 4, -3 },
		-5, -5, 5, -1, -1, 5, 15, 0,
		13, 150,
		100, 30,
		72, 6, 100, 25,
		66, 4, 100, 20,
		2,
		0x0800001,
	},
	{
#ifdef JP
		"吸血鬼",
#endif
		"Vampire",

		{ 3, 3, -1, -1, 1, 2 },
		4, 8, 6, 4, 1, 8, 5, 0,
		11, 200,
		100, 30,
		72,  6, 180, 25,
		66,  4, 150, 20,
		5,
		0x67DC7FF,
	},
	{
#ifdef JP
		"幽霊",
#endif
		"Spectre",

		{ -5, 4, -1, 2, 0, -3 },
		10, 15, 12, 5, 5, 14, -15, -5,
		7, 210,
		100, 30,
		72, 6, 100, 25,
		66, 4, 100, 20,
		5,
		0x631474A,
	},
	{
#ifdef JP
		"妖精",
#endif
		"Sprite",

		{ -4, 3, 3, 3, -2, 2 },
		10, 8, 6, 4, 10, 10, -12, 0,
		7, 145,
		50, 25,
		32,  2, 75,  2,
		29,  2, 65,  2,
		4,
		0x623F65E,
	},
	{
#ifdef JP
		"獣人",  
#endif
		"Beastman",

		{ 2, -2, -1, -1, 2, -2 },
		-5, -2, -1, -1, -1, 5, 12, 5,
		11, 140,
		14, 6,
		65,  6, 150, 20,
		61,  6, 120, 15,
		0,
		0x57887CF,
	},
	{
#ifdef JP
		"エント",
#endif
		"Ent",
		{ 2,  0,  2, -3,  2,  0 },
		 -5,  2,  5,  -1, 0, 9,  15, -5,
		 12, 140,
		120,  60,
		111, 11, 255, 50,
		99, 11, 250, 45,
		  0,
		0x0010005,
	},
	{
#ifdef JP
		"アルコン",
#endif
		"Archon",

		{  2,  0,  4,  1,  2,  3 },
		0,  12,  8,  2, 2, 11, 10, 10,
		11,  235,
		150, 150,
		82, 5, 190, 20,
		78,  6, 180, 15,
		3,
		0x779F777,
	},
	{
#ifdef JP
		"バルログ",
		"Balrog",
#else
		"Balrog",
#endif

		{  4,  2,  -10,  2,  3,  -5 },
		-3,  12, 15,  -2,  1, 8, 20, 0,
		12,  250,
		255, 255,
		100,10, 255, 65,
		80, 10, 240, 64,
		5,
		0x7EDC4DB,
	},
	{
#ifdef JP
		"ドゥナダン",
#endif
		"Dunadan",

		{  1,  2,  2,  2,  3,  2 },
		4,  5,  3,  2, 3, 13, 15, 10,
		10,  180,
		50, 20,
		82, 5, 190, 20,
		78,  6, 180, 15,
		0,
		0xFFFF7FF,
	},
	{
#ifdef JP
		"影フェアリー",
#endif
		"Shadow-Fairy",
		{-2,  2,  2,  1, -1,  0 },
		  7,  8,  0,  6, 12, 15, -10, -5,
		  7, 120,
		200, 100,
		 80,  8, 90, 20,
		 73,  8, 80, 15,
		  4,
		0xE33C7DF,
	},
	{
#ifdef JP
		"クター",
#endif
		"Kutar",

		{  0,  -1,  -1,  1,  2,  3 },
		-2,  5,  5,  5,  -2,  6,  0,  -5,
		11,  140,
		14,  6,
		48,  6, 150, 25,
		44,  4, 130, 20,
		0,
		0xC18B7AD,
	},
	{
#ifdef JP
		"アンドロイド",
#endif
		"Android",

		{ 4, -5, -5, 0, 4, -2 },
		0, -5, 0, -2, 3, 14, 20, 10,
		13, 200,
		1, 100,
		72, 12, 240, 64,
		66, 12, 220, 64,
		0,
		0x0800001,
	},
};


/*!
 * @brief 職業情報 /
 * Player Classes
 * @details
 * <pre>
 *      Title,
 *      {STR,INT,WIS,DEX,CON,CHR},
 *      c_dis, c_dev, c_sav, c_stl, c_srh, c_fos, c_thn, c_thb,
 *      x_dis, x_dev, x_sav, x_stl, x_srh, x_fos, x_thn, x_thb,
 *      HD, Exp, pet_upkeep_div
 * </pre>
 */
const player_class class_info[MAX_CLASS] =
{
	{
#ifdef JP
		"戦士",
#endif
		"Warrior",

		{ 4, -2, -2, 2, 2, -1},
		25, 18, 31, 1,  14, 2, 70, 55,
		12, 7,  10, 0,  0,  0,  30, 30,
		9,  0, 40
	},

	{
#ifdef JP
		"メイジ",
#endif
		"Mage",

		{-4, 3, 0, 1, -2, 1},
		30, 40, 38, 3,  16, 20, 34, 20,
		7,  15, 11,  0,  0,  0,  6, 7,
		0, 30, 30
	},

	{
#ifdef JP
		"プリースト",
#endif
		"Priest",

		{-1, -3, 3, -1, 0, 2},
		25, 35, 40, 2,  16, 8, 48, 35,
		7,  11, 12, 0,  0,  0, 13, 11,
		2, 20, 35
	},

	{
#ifdef JP
		"盗賊",
#endif
		"Rogue",

		{ 2, 1, -2, 3, 1, -1},
		45, 37, 36, 5, 32, 24, 60, 66,
		15, 12, 10, 0,  0,  0, 21, 18,
		6, 25, 40
	},

	{
#ifdef JP
		"レンジャー",
#endif
		"Ranger",

		{ 2, 2, 0, 1, 1, 1},
		30, 37, 36, 3,  24, 16, 56, 72,
		8,  11, 10, 0,  0,  0,  18, 28,
		4, 40, 35
	},

	{
#ifdef JP
		"パラディン",
#endif
		"Paladin",

		{ 3, -3, 1, 0, 2, 2},
		20, 24, 34, 1,  12, 2, 68, 40,
		7,  10, 11, 0,  0,  0,  21, 18,
		6, 35, 40
	},

	{
#ifdef JP
		"魔法戦士",
#endif
		"Warrior-Mage",

		{ 2, 2, 0, 1, 0, 1},
		30, 35, 36, 2,  18, 16, 50, 25,
		7,  10, 10, 0,  0,  0,  15, 11,
		4, 40, 35
	},

	{
#ifdef JP
		"混沌の戦士",
#endif
		"Chaos-Warrior",

		{ 2, 1, -1, 0, 2, -2},
		20, 25, 34, 1,  14, 12, 65, 40,
		7,  11, 10, 0,  0,  0,  20, 17,
		6, 25, 40
	},

	{
#ifdef JP
		"修行僧",
#endif
		"Monk",

		{ 2, -1, 1, 3, 2, 1},
		45, 34, 36, 5, 32, 24, 64, 60,
		15, 11, 10, 0,  0,  0, 18, 18,
		6, 30, 35
	},

	{
#ifdef JP
		"超能力者",
#endif
		"Mindcrafter",

		{-1, 0, 3, -1, -1, 2},   /* note: spell stat is Wis */
		30, 33, 38, 3,  22, 16, 50, 40,
		10, 11, 10, 0,   0,  0, 14, 18,
		2, 25, 35
	},

	{
#ifdef JP
		"ハイ=メイジ",
#endif
		"High-Mage",

		{-4, 4, 0, 0, -2, 1},
		30, 40, 38, 3,  16, 20, 34, 20,
		7,  15, 11,  0,  0,  0,  6, 7,
		0, 30, 25
	},

	{
#ifdef JP
		"観光客",
#endif
		"Tourist",
		{ -1, -1, -1, -1, -1, -1},
		15, 18, 28, 1, 12, 2, 40, 20,
		5, 7, 9, 0,  0,  0,  11, 11,
		0, -30, 40
	},

	{
#ifdef JP
		"ものまね師",
#endif
		"Imitator",
		{ 0, 1, -1, 2, 0, 1},
		25, 30, 36, 2,  18, 16, 60, 50,
		7,  10,  10, 0,  0,  0,  18, 20,
		5, 10, 20
	},

	{
#ifdef JP
		"魔獣使い",
#endif
		"BeastMaster",
		{ 1, -1, -1, 1, 0, 2},
		20, 25, 32, 2,  18, 16, 52, 63,
		7,  10, 10, 0,  0,  0,  14, 25,
		3, 20, 10
	},

	{
#ifdef JP
		"スペルマスター",
#endif
		"Sorcerer",

		{-5, 6, -2, 2, 0, -2},
		30, 48, 75, 2,  12, 22,  0, 0,
		 7, 18, 13, 0,  0,  0,  0, 0,
		4, 60, 25
	},

	{
#ifdef JP
		"アーチャー",
#endif
		"Archer",

		{ 2, -1, -1, 2, 1, 0},
		38, 24, 35, 4,  24, 16, 56, 82,
		12, 10, 10, 0,  0,  0,  18, 36,
		6, 10, 40
	},

	{
#ifdef JP
		"魔道具術師",
#endif
		"Magic-Eater",

		{-1, 2, 1, 2, -2, 1},
		25, 42, 36, 2,  20, 16, 48, 35,
		7,  16, 10,  0,  0,  0, 13, 11,
		3, 30, 30
	},

	{
#ifdef JP
		"吟遊詩人",
#endif
		"Bard",              /* Note : spell stat is Charisma */
		{-2, 1, 2, -1, -2, 4},
		20, 33, 34, -5, 16, 20, 34, 20,
		8,  13, 11, 0,  0,  0,  10, 8,
		2, 40, 25
	},

	{
#ifdef JP
		"赤魔道師",
#endif
		"Red-Mage",

		{ 2, 2, -1, 1, 0, -1},
		20, 34, 34, 1,  16, 10, 56, 25,
		7,  11, 11, 0,  0,  0,  18, 11,
		4, 40, 40
	},

	{
#ifdef JP
		"剣術家",
#endif
		"Samurai",

		{ 3, -2, 1, 2, 1, 0},
		25, 18, 32, 2,  16, 6, 70, 40,
		12, 7,  10, 0,  0,  0,  23, 18,
		6,  30, 40
	},

	{
#ifdef JP
		"練気術師",
#endif
		"ForceTrainer",

		{ 0, -1, 3, 2, 1, 1},
		30, 34, 38, 4, 32, 24, 50, 40,
		10, 11, 11, 0,  0,  0, 14, 15,
		2, 35, 40
	},

	{
#ifdef JP
		"青魔道師",
#endif
		"Blue-Mage",

		{-4, 4, -1, 1, -2, -1},
		30, 40, 36, 3,  20, 16, 40, 25,
		7,  16, 11,  0,  0,  0,  6, 7,
		2, 30, 35
	},

	{
#ifdef JP
		"騎兵",
#endif
		"Cavalry",
		{ 2, -2, -2, 2, 2, 0},
		20, 18, 32, 1,  16, 10, 60, 66,
		10,  7, 10, 0,  0,  0,  22, 26,
		5, 20, 35
	},

	{
#ifdef JP
		"狂戦士",
#endif
		"Berserker",

		{ 8, -20, -20, 4, 4, -5},
		-100, -1000, -200, -100,  -100, -100, 120, -2000,
		0, 0,  0, 0,  0,  0,  50, 0,
		11,  60, 255
	},

	{
#ifdef JP
		"鍛冶師",
#endif
		"Weaponsmith",

		{ 3, -1, -1, 1, 0, -1},
		30, 28, 28, 1,  20, 10, 60, 45,
		10, 10,  10, 0,  0,  0,  21, 15,
		6,  30, 40
	},
	{
#ifdef JP
		"鏡使い",
#endif
		"Mirror-Master",

		{ -2,  3, 1, -1, -2, 1},
		30, 33, 40, 3, 14, 16, 34,30,
		10, 11, 12, 0,  0,  0,  6,10,
		2,  30, 30
	},
	{
#ifdef JP
		"忍者",
#endif
		"Ninja",

		{ 0,  -1, -1, 3, 2, -1},
		45, 24, 36, 8, 48, 32, 70,66,
		15, 10, 10, 0,  0,  0, 25,18,
		2,  20, 40
	},

	{
#ifdef JP
		"スナイパー",
#endif
		"Sniper",

		{ 2, -1, -1, 2, 1, 0},
		25, 24, 28, 5, 32, 18, 56,  72,
		12, 10, 10, 0,  0,  0, 18,  28,
		2, 20, 40,
	},
};

/*!
 * @brief 性格情報 /
 * Player Character
 */
const player_seikaku seikaku_info[MAX_SEIKAKU] =
{
	{
#ifdef JP
		"ふつう",
#endif
		"Ordinary",
		{  0,  0,  0,  0,  0,  0 },
		0,  0,  0,  0,  0,  0,  0,  0,
		0, 1, 0
	},

	{
#ifdef JP
		"ちからじまん",
#endif
		"Mighty",
		{  2,  -2,  -1,  0,  1,  0 },
		-5,  -5,  -3,  -1,  -2,  -2,  10,  0,
		1, 1, 0
	},

	{
#ifdef JP
		"きれもの",
#endif
		"Shrewd",
		{  -2,  2,  0,  1,  -1,  -1 },
		3,  8,  2,  0,  -2,  5,  -8,  -5,
		-1, 1, 0
	},

	{
#ifdef JP
		"しあわせもの",
#endif
		"Pious",
		{  0,  -1,  2,  -1,  0,  1 },
		-5,  2,  4,  -1,  3,  -2,  -3,  -6,
		0, 1, 0
	},

	{
#ifdef JP
		"すばしっこい",
#endif
		"Nimble",
		{  -1,  1,  -1,  2,  -1,  -1 },
		7,  2,  -1,  1,  5,  5,  0,  10,
		0, 0, 0
	},

	{
#ifdef JP
		"いのちしらず",
#endif
		"Fearless",
		{  2,  1,  1,  -1,  -1,  0 },
		-5,  5,  -2,  0,  2,  -2,  10,  10,
		-1, 1, 0
	},

	{
#ifdef JP
		"コンバット",
#endif
		"Combat",
		{  1,  -1,  -2,  2,  0,  1 },
		-2,  -3,  -3,  0,  -1,  2,  5,  5,
		0, 0, 0
	},

	{
#ifdef JP
		"なまけもの",
#endif
		"Lazy",
		{  -2,  -2,  -2,  -2,  -2,  -2 },
		-5,  -5,  -3,  -1,  -4,  -2,  -8,  -8,
		-1, 1, 0
	},

	{
#ifdef JP
		"セクシーギャル",
#endif
		"Sexy",
		{  1,  1,  1,  1,  1,  3 },
		10,  5,  3,  0,  4,  2,  10,  10,
		0, 1, 1
	},

	{
#ifdef JP
		"ラッキーマン",
#endif
		"Lucky",
		{  -2,  -2,  -2,  -2,  -2,  2 },
		10,  7,  3,  2, 10,  8,  15,  15,
		0, 1, 2
	},

	{
#ifdef JP
		"がまんづよい",
#endif
		"Patient",
		{  -1,  -1,  1,  -2,  2,  0 },
		-5,  -3,  3,  1,  0,  -3,  -6,  -6,
		1, 0, 0
	},

	{
#ifdef JP
		"いかさま",
#endif
		"Munchkin",
		{ 10,  10,  10,  10,  10,  10 },
		 20, 40, 30, 10, 40, 40,  80, 80,
		 15, 1, 0
	},
};


/*!
 * @brief 変身種族情報
 */
const player_race mimic_info[] =
{
	{
#ifdef JP
		"[標準形態]",
#endif
		"Default",

		{  0,  0,  0,  0,  0,  0 },
		0,  0,  0,  0,  0,  10,  0,  0,
		10,  100,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		0,
		0x000000,
	},
	{
#ifdef JP
		"[悪魔]",
#endif
		"[Demon]",

		{  5,  3,  2,  3,  4,  -6 },
		-5,  18, 20, -2,  3,  10, 40, 20,
		12,  0,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		5,
		0x000003,
	},
	{
#ifdef JP
		"[魔王]",
#endif
		"[Demon lord]",

		{  20,  20,  20,  20,  20,  20 },
		20,  20, 25, -2,  3,  10, 70, 40,
		14,  0,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		20,
		0x000003,
	},
	{
#ifdef JP
		"[吸血鬼]",
#endif
		"[Vampire]",

		{ 4, 4, 1, 1, 2, 3 },
		6, 12, 8, 6, 2, 12, 30, 20,
		11,  0,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		5,
		0x000005,
	},
};


/*!
 * @brief 歌、剣術、呪術領域情報テーブル
 */
const magic_type technic_info[NUM_TECHNIC][32] =
{
	{
		/* Music */
		{ 1,  1,  10,   2},
		{ 2,  1,  10,   2},
		{ 3,  2,  20,   3},
		{ 4,  2,  20,   4},
		{ 5,  2,  20,   6},
		{ 7,  4,  30,   8},
		{ 9,  3,  30,   10},
		{ 10, 2,  30,   12},

		{ 12,  3,   40,   20},
		{ 15, 16,  42,   35},
		{ 17, 18,  40,   25},
		{ 18,  2,  45,   30},
		{ 23,  8,  50,   38},
		{ 28, 30,  50,   41},
		{ 33, 35,  60,   42},
		{ 38, 35,  70,   46},

		{ 10,  4,  20,   13},
		{ 22,  5,  30,   26},
		{ 23,  3,  35,   27},
		{ 26,  28,  37,   29},
		{ 32,  37,  41,   36},
		{ 33,  22,  43,   40},
		{ 37,  35,  46,   42},
		{ 45,  60,  50,   56},

		{ 23,  18,  20,   23},
		{ 30,  30,  30,   26},
		{ 33,  65,  41,   30},
		{ 37,  35,  43,   35},
		{ 40,  30,  46,   50},
		{ 42,  75,  50,   68},
		{ 45,  58,  62,   73},
		{ 49,  48,  70,  200}
	},

	{
		/* Hissatsu */
		{ 1,   15,   0,   0},
		{ 3,   10,   0,   0},
		{ 6,   15,   0,   0},
		{ 9,    8,   0,   0},
		{ 10,  12,   0,   0},
		{ 12,  25,   0,   0},
		{ 14,   7,   0,   0},
		{ 17,  20,   0,   0},

		{ 19,  10,   0,   0},
		{ 22,  20,   0,   0},
		{ 24,  30,   0,   0},
		{ 25,  10,   0,   0},
		{ 27,  15,   0,   0},
		{ 29,  45,   0,   0},
		{ 32,  70,   0,   0},
		{ 35,  50,   0,   0},

		{ 18,  40,   0,   0},
		{ 22,  22,   0,   0},
		{ 24,  30,   0,   0},
		{ 26,  35,   0,   0},
		{ 30,  30,   0,   0},
		{ 32,  60,   0,   0},
		{ 36,  40,   0,   0},
		{ 39,  80,   0,   0},

		{ 26,  20,   0,   0},
		{ 29,  40,   0,   0},
		{ 31,  35,   0,   0},
		{ 36,  80,   0,   0},
		{ 39, 100,   0,   0},
		{ 42, 110,   0,   0},
		{ 45, 130,   0,   0},
		{ 50, 255,   0,   0}
	},

	{
		/* Hex */
		{  1,  2, 20,   2},
		{  1,  2, 20,   2},
		{  3,  2, 30,   3},
		{  5,  3, 30,   4},
		{  7,  3, 40,   6},
		{  8, 10, 60,   8},
		{  9,  3, 30,  10},
		{ 10,  5, 40,  12},

		{ 12,  8, 40,  15},
		{ 12,  9, 35,  15},
		{ 15, 10, 50,  20},
		{ 20, 12, 45,  35},
		{ 25, 15, 50,  50},
		{ 30, 12, 60,  70},
		{ 35, 10, 60,  80},
		{ 40, 16, 70, 100},

		{ 15,  8, 20,  20},
		{ 18, 15, 50,  20},
		{ 22, 10, 65,  35},
		{ 25, 28, 70,  50},
		{ 28, 10, 70,  60},
		{ 30, 20, 60,  60},
		{ 36, 22, 70,  80},
		{ 40, 28, 70, 100},

		{  5,  6, 35,   5},
		{ 22, 24, 70,  40},
		{ 25,  2, 65,  50},
		{ 32, 20, 50,  70},
		{ 35, 35, 70,  80},
		{ 38, 32, 70,  90},
		{ 42, 24, 70, 120},
		{ 46, 45, 80, 200}
	},
};


/*!
 * 魔法領域フラグ管理テーブル /
 * Zangband uses this array instead of the spell flags table, as there
 * are 5 realms of magic, each with 4 spellbooks and 8 spells per book -- TY
 */
const u32b fake_spell_flags[4]=
{
	0x000000ff,
	0x0000ff00,
	0x00ff0000,
	0xff000000
};

/*!
 * 職業毎に選択可能な第一領域魔法テーブル
 */
const s32b realm_choices1[MAX_CLASS] =
{
	(CH_NONE),				/* Warrior */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON |
	 CH_CRUSADE),                              /* Mage */
	(CH_LIFE | CH_DEATH | CH_DAEMON |
	 CH_CRUSADE),                              /* Priest */
	(CH_SORCERY | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT),               /* Rogue */
	(CH_NATURE),                            /* Ranger */
	(CH_CRUSADE | CH_DEATH),                   /* Paladin */
	(CH_ARCANE),                            /* Warrior-Mage */
	(CH_CHAOS | CH_DAEMON),                 /* Chaos-Warrior */
	(CH_LIFE | CH_NATURE | CH_DEATH |
	 CH_ENCHANT),                           /* Monk */
	(CH_NONE),                              /* Mindcrafter */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON |
	 CH_CRUSADE | CH_HEX),                  /* High-Mage */
	(CH_ARCANE),                            /* Tourist */
	(CH_NONE),                              /* Imitator */
	(CH_TRUMP),                             /* Beastmaster */
	(CH_NONE),                              /* Sorcerer */
	(CH_NONE),                              /* Archer */
	(CH_NONE),                              /* Magic eater */
	(CH_MUSIC),                             /* Bard */
	(CH_NONE),                              /* Red Mage */
	(CH_HISSATSU),                          /* Samurai */
	(CH_LIFE | CH_NATURE | CH_DEATH |
	 CH_ENCHANT | CH_CRUSADE),                 /* ForceTrainer */
	(CH_NONE),                              /* Blue Mage */
	(CH_NONE),				/* Cavalry */
	(CH_NONE),				/* Berserker */
	(CH_NONE),				/* Weaponsmith */
	(CH_NONE),				/* Mirror-master */
	(CH_NONE),				/* Ninja */
	(CH_NONE),				/* Sniper */
};

/*!
 * 職業毎に選択可能な第二領域魔法テーブル
 */
const s32b realm_choices2[MAX_CLASS] =
{
	(CH_NONE),                              /* Warrior */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON |
	 CH_CRUSADE),                              /* Mage */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON |
	 CH_CRUSADE),                              /* Priest */
	(CH_NONE),                              /* Rogue */
	(CH_SORCERY | CH_CHAOS | CH_DEATH |
	 CH_TRUMP | CH_ARCANE | CH_DAEMON),     /* Ranger */
	(CH_NONE),                              /* Paladin */
	(CH_LIFE | CH_NATURE | CH_CHAOS |
	 CH_DEATH | CH_TRUMP | CH_ARCANE |
	 CH_SORCERY | CH_ENCHANT | CH_DAEMON |
	 CH_CRUSADE),                              /* Warrior-Mage */
	(CH_NONE),                              /* Chaos-Warrior */
	(CH_NONE),                              /* Monk */
	(CH_NONE),                              /* Mindcrafter */
	(CH_NONE),                              /* High-Mage */
	(CH_NONE),                              /* Tourist */
	(CH_NONE),                              /* Imitator */
	(CH_NONE),                              /* Beastmanster */
	(CH_NONE),                              /* Sorcerer */
	(CH_NONE),                              /* Archer */
	(CH_NONE),                              /* Magic eater */
	(CH_NONE),                              /* Bard */
	(CH_NONE),                              /* Red Mage */
	(CH_NONE),                              /* Samurai */
	(CH_NONE),                              /* ForceTrainer */
	(CH_NONE),                              /* Blue Mage */
	(CH_NONE),				/* Cavalry */
	(CH_NONE),				/* Berserker */
	(CH_NONE),				/* Weaponsmith */
	(CH_NONE),				/* Mirror-master */
	(CH_NONE),				/* Ninja */
	(CH_NONE),				/* Sniper */
};


#ifdef JP
/*!
 * 領域魔法名称
 */
const cptr realm_names[] =
{
	"魔法なし",
	"生命",
	"仙術",
	"自然",
	"カオス",
	"暗黒",
	"トランプ",
	"秘術",
	"匠",
	"悪魔",
	"破邪",
	"不明",
	"不明",
	"不明",
	"不明",
	"不明",
	"歌",
	"武芸",
	"呪術",
	"不明"
};
#endif

#ifdef JP
const cptr E_realm_names[]
#else
const cptr realm_names[]
#endif
= {
	"none",
	"Life",
	"Sorcery",
	"Nature",
	"Chaos",
	"Death",
	"Trump",
	"Arcane",
	"Craft",
	"Daemon",
	"Crusade",
	"unknown",
	"unknown",
	"unknown",
	"unknown",
	"unknown",
	"Music",
	"Kendo",
	"Hex",
	"unknown"
};


/*!
 * @brief 箱のトラップテーブル
 * @details
 * <pre>
 * Each chest has a certain set of traps, determined by pval
 * Each chest has a "pval" from 1 to the chest level (max 55)
 * If the "pval" is negative then the trap has been disarmed
 * The "pval" of a chest determines the quality of its treasure
 * Note that disarming a trap on a chest also removes the lock.
 * </pre>
 */
const int chest_traps[64] =
{
	0,                                      /* 0 == empty */
	(CHEST_POISON),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),                       /* 5 == best small wooden */
	0,
	(CHEST_ALARM),
	(CHEST_ALARM),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),
	(CHEST_POISON),
	(CHEST_SCATTER),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_SUMMON),                 /* 15 == best large wooden */
	0,
	(CHEST_ALARM),
	(CHEST_SCATTER),
	(CHEST_PARALYZE),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_SUMMON),
	(CHEST_PARALYZE),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),
	(CHEST_EXPLODE),                        /* 25 == best small iron */
	0,
	(CHEST_E_SUMMON),
	(CHEST_POISON | CHEST_LOSE_CON),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_BIRD_STORM),
	(CHEST_POISON | CHEST_SUMMON),
	(CHEST_E_SUMMON | CHEST_ALARM),
	(CHEST_EXPLODE),
	(CHEST_EXPLODE | CHEST_SUMMON), /* 35 == best large iron */
	0,
	(CHEST_SUMMON | CHEST_ALARM),
	(CHEST_EXPLODE),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_POISON | CHEST_PARALYZE),
	(CHEST_EXPLODE),
	(CHEST_BIRD_STORM),
	(CHEST_EXPLODE | CHEST_E_SUMMON | CHEST_ALARM),
	(CHEST_H_SUMMON),        /* 45 == best small steel */
	0,
	(CHEST_EXPLODE | CHEST_SUMMON | CHEST_ALARM),
	(CHEST_BIRD_STORM),
	(CHEST_RUNES_OF_EVIL),
	(CHEST_EXPLODE | CHEST_SUMMON | CHEST_ALARM),
	(CHEST_BIRD_STORM | CHEST_ALARM),
	(CHEST_H_SUMMON | CHEST_ALARM),
	(CHEST_RUNES_OF_EVIL),
	(CHEST_H_SUMMON | CHEST_SCATTER | CHEST_ALARM),
	(CHEST_RUNES_OF_EVIL | CHEST_EXPLODE),  /* 55 == best large steel */
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
};


/*!
 * @brief 職業とレベル毎のプレイヤー称号テーブル / Class titles for the player.
 * <pre>
 * The player gets a new title every five levels, so each class
 * needs only ten titles total.
 * </pre>
 */
#ifdef JP
const cptr player_title[MAX_CLASS][PY_MAX_LEVEL / 5] =
{
	/* Warrior */
	{
		"新参兵",
		"兵士",
		"傭兵",
		"古参兵",
		"剣士",
		"闘士",
		"英雄",
		"男爵",
		"伯爵",
		"君主",
	},

	/* Mage */
	{
				/*"見習い",*/
		"練習生",                 /*丁稚、練習生 */
		"奇術師",               /*詐欺師、ペテン師 */
		"幻術師",
		"呪術師",
		"召霊師",
		"召魔師",
		"魔術師",
		"魔道師",
		"イプシシマス",
		"大魔道師",
	},

	/* Priest */
	{
		"信者",                 /*信徒 */
		"侍僧",             /*教会奉仕者、見習い僧、伴僧、従者 */
		"熟練僧",
		"聖職者",                 /*聖職者 */
		"伝道師",               /*司祭評議員、修道会会員 */
		"牧師",               /*ラマ教の僧 */
		"聖人",               /*大司教、総主教、総大司教 */
		"祭司",                 /*祭司、司祭 */
		"祭司長",               /*大祭司、祭司長 */
		"教皇",
	},

	/* Rogues */
	{
		/* "ごろつき",*/                     /*ごろつき、風来坊、浮浪者 */
		"すり",
		"追いはぎ",                     /*追い剥ぎ、強盗、泥棒 */
		"夜盗",                         /*強盗、夜盗、泥棒 */
		"こそ泥",                       /*こそ泥、小泥棒 */
		"ペテン師",                     /*博徒、ペテン師、詐欺師 */
		"ロウシーフ",
		"ハイシーフ",
		"マスター", /* "マスターシーフ", */
		"アサシン",						/* 暗殺者 */
		"頭領", /*"ギルドマスター",*/
	},

	/* Rangers */
	{
		"使い走り",
		"馳夫",
		"斥候",                         /*斥候、見張り、偵察兵 */
		"狩人",
		"追跡者",
		"先導者",
		"探険者",                       /*開拓者、探険者 */
		"野伏",
		"野伏頭",
		"野伏の総領",
	},

	/* Paladins */
	{
		"勇士",                 /*色男、愛人、しゃれ者、勇敢な人 */
		"衛士",
		"保護者",
		"防衛者",
		"護衛者",
		"騎士",
		"重騎士",
		"聖騎士",
		"上級聖騎士",
		"聖騎士団長",
	},

	/* Warrior-Mage */
	{
		"見習い",
		"徒弟",			/*丁稚、練習生 */
		"一人前",
		"古参兵",
		"魔術兵士",
		"魔術闘士",
		"魔術の英雄", /* Mage-Hero */
		"魔男爵",
		/* "魔公爵", */
		"戦闘魔術士",
		"知識の守護者", /* "ウィザードロード", */
	},

	/* Chaos Warrior */
	{
		"新参兵",
		"兵士",
		"傭兵",
		"古参兵",
		"剣士",
		"闘士",
		"混沌の英雄",
		"混沌の男爵",
		"混沌の公爵",
		"混沌の王者",
	},

	/* Monk */
	{
		"入門者",
		"弟子",
		"直弟子",
		"師範代",
		"師範",
		"道場主",
		"名人",
		"大名人",
		"拳聖",
		"拳神",
	},

	/* Mindcrafter */
	{
		"練習生",
		"見習い",
		"熟練士",
		"熟達士",
		"黙想士",
		"心術士",
		"サイキック",
		"サイオニック",
		"超能力者",
		"精神の支配者", /* "マインドマスター", */
	},

	/* High Mage; same as Mage */
	{
				/*"見習い",*/
		"練習生",                 /*丁稚、練習生 */
		"奇術師",               /*詐欺師、ペテン師 */
		"幻術師",
		"呪術師",
		"召霊師",
		"召魔師",
		"魔術師",
		"魔道師",
		"イプシシマス",
		"大魔道師",
	},

	/* Tourist */
	{
		"プー太郎",
		"観光客",
		"周遊旅行者",
		"遍歴者",
		"旅行者",
		"放浪者", /* "旅人", */
		"航海者",
		"探検家",
		"冒険家",
		"スペランカー",
	},

	/* Imitator */
	{
		"これから",
		"いまいち",
		"まだまだ",
		"ぼちぼち",
		"そこそこ",
		"まあまあ",
		"なかなか",
		"いけいけ",
		"そうとう",
		"えらい",
	},

	/* Beastmaster */
	{
		"イモリ使い",
		"ヘビ使い",
		"クモ使い",
		"狼使い",
		"トラ使い",
		"甲虫使い",
		"ヒドラ使い",
		"ハウンド使い",
		"ムーマク使い",
		"ドラゴン使い",
	},

	/* Sorcerer; same as Mage */
	{
				/*"見習い",*/
		"練習生",                 /*丁稚、練習生 */
		"奇術師",               /*詐欺師、ペテン師 */
		"幻術師",
		"呪術師",
		"召霊師",
		"召魔師",
		"魔術師",
		"魔道師",
		"イプシシマス",
		"大魔道師",
	},

	/* Archer */
	{
		"新参兵",
		"兵士",
		"傭兵",
		"古参兵",
		"剣士",
		"闘士",
		"英雄",
		"男爵",
		"伯爵",
		"領主",
	},

	/* Magic eater */
	{
		"無知なる者",
		"入門者",
		"奇術師",
		"秘術師",
		"秘術師",
		"熟練者",
		"達人",
		"達人",
		"魔道師",
		"全てを知る者",
	},

	/* Bard */
	{
		"見習い",	/*"Apprentice"*/
		"作曲家",	/*"Songsmith"*/
		"吟遊詩人",	/*"Bard"*/
		"コンパニオン",	/*"Companion"*/
		"心の癒し手",	/*"Minstrel"*/
		"竪琴師",	/*"Harper"*/
		"伝承の紡ぎ手",	/*"Loreweaver"*/
		"詩神の申し子",	/*"Muse"*/
		"夢紡ぎ",	/*"Dreamweaver"*/
		"マスター",	/*"Master Harper"*/
	},

	/* Red Mage; same as Warrior-Mage */
	{
		"見習い",
		"徒弟",			/*丁稚、練習生 */
		"一人前",
		"古参兵",
		"魔術兵士",
		"魔術闘士",
		"魔術の英雄", /* Mage-Hero */
		"魔男爵",
		/* "魔公爵", */
		"戦闘魔術士",
		"知識の守護者", /* "ウィザードロード", */
	},

	/* Samurai */
	{
		"入門者",
		"弟子",
		"直弟子",
		"師範代",
		"師範",
		"道場主",
		"名人",
		"大名人",
		"剣聖",
		"剣神",
	},

	/* ForceTrainer; same as Monk(?) */
	{
		"入門者",
		"弟子",
		"直弟子",
		"師範代",
		"師範",
		"道場主",
		"名人",
		"大名人",
		"拳聖",
		"拳神",
	},

	/* Blue Mage; same as Mage */
	{
				/*"見習い",*/
		"練習生",                 /*丁稚、練習生 */
		"奇術師",               /*詐欺師、ペテン師 */
		"幻術師",
		"呪術師",
		"召霊師",
		"召魔師",
		"魔術師",
		"魔道師",
		"イプシシマス",
		"大魔道師",
	},

	/* Cavalry */
	{
		"新参兵",
		"兵士",
		"傭兵",
		"古参兵",
		"剣士",
		"闘士",
		"英雄",
		"男爵",
		"伯爵",
		"領主",
	},

	/* Berserker */
	{
		"バーサーカー",
		"バーサーカー",
		"バーサーカー",
		"怒りの公爵",
		"怒りの公爵",
		"怒りの公爵",
		"怒りの王",
		"怒りの王",
		"怒りの王",
		"怒りの化身",
	},

	/* Weaponsmith */
	{
		"銅を鍛えし者",
		"鉄を鍛えし者",
		"鋼を鍛えし者",
		"銀を鍛えし者",
		"竜を鍛えし者",
		"霊を鍛えし者",
		"魔を鍛えし者",
		"魂を鍛えし者",
		"神を鍛えし者",
		"全を鍛えし者",
	},

	/* Mirror Master */
	{
		"鏡を見る人",
		"鏡磨き",
		"鏡職人",
		"鏡術師",
		"鏡導師",
		"鏡の賢者",
		"鏡の王",
		"鏡の皇帝",
		"鏡の化身",
		"ラフノール王",
	},
	/* Ninja */
	{
		"訓練生",
		"仕手",
		"熟達者",
		"短刀使い",
		"切り裂き",
		"凄腕",
		"漆黒の刃",
		"闇の一撃",
		"暗殺者",
		"死の長き腕",
	},

	/* Sniper */
	{
		"新参兵",
		"兵士",
		"傭兵",
		"古参兵",
		"剣士",
		"闘士",
		"英雄",
		"男爵",
		"伯爵",
		"領主",
	},
};

#else
const cptr player_title[MAX_CLASS][PY_MAX_LEVEL / 5] =
{
	/* Warrior */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},

	/* Mage */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Priest */
	{
		"Believer",
		"Acolyte",
		"Adept",
		"Curate",
		"Canon",
		"Priest",
		"High Priest",
		"Cardinal",
		"Inquisitor",
		"Pope",
	},

	/* Rogues */
	{
		"Cutpurse",
		"Robber",
		"Burglar",
		"Filcher",
		"Sharper",
		"Low Thief",
		"High Thief",
		"Master Thief",
		"Assassin",
		"Guildmaster",
	},

	/* Rangers */
	{
		"Runner",
		"Strider",
		"Scout",
		"Courser",
		"Tracker",
		"Guide",
		"Pathfinder",
		"Low Ranger",
		"High Ranger",
		"Ranger Lord",
	},

	/* Paladins */
	{
		"Gallant",
		"Keeper",
		"Protector",
		"Defender",
		"Warder",
		"Knight",
		"Guardian",
		"Low Paladin",
		"High Paladin",
		"Paladin Lord",
	},

	/* Warrior-Mage */
	{
		"Novice",
		"Apprentice",
		"Journeyman",
		"Veteran",
		"Enchanter",
		"Champion",
		"Mage-Hero",
		"Baron Mage",
		"Battlemage",
		"Wizard Lord",
	},

	/* Chaos Warrior */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Chaos Hero",
		"Chaos Baron",
		"Chaos Duke",
		"Chaos Lord",
	},

	/* Monk */
	{
		"Initiate",
		"Brother",
		"Disciple",
		"Immaculate",
		"Master",
		"Soft Master",
		"Hard Master",
		"Flower Master",
		"Dragon Master",
		"Grand Master",
	},

	/* Mindcrafter */
	{
		"Trainee",
		"Acolyte",
		"Adept",
		"Immaculate",
		"Contemplator",
		"Mentalist",
		"Psychic",
		"Psionicist",
		"Esper",
		"Mindmaster",
	},

	/* High Mage; same as Mage */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Tourist */
	{
		"Rambler",
		"Sightseer",
		"Excursionist",
		"Peregrinator",
		"Traveler",
		"Journeyer",
		"Voyager",
		"Explorer",
		"Adventurer",
		"Spelunker",
	},

	/* Imitator */
	{
		"Yet",
		"Lacks",
		"Still more",
		"So so",
		"All right",
		"Not bad",
		"Considerable",
		"Go go",
		"Sizable",
		"Great man",
	},

	/* BeastMaster */
	{
		"Newt Master",
		"Snake Master",
		"Spider Master",
		"Wolf Master",
		"Tiger Master",
		"Beetle Master",
		"Hydra Master",
		"Hound Master",
		"Mumak Master",
		"Dragon Master",
	},

	/* Sorcerer */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Archer */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Bowman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},

	/* Magic eater */
	{
		"Apprentice",
		"Beginner",
		"Jagguler",
		"Skilled",
		"Conjurer",
		"Magician",
		"Master",
		"Master",
		"Wizard",
		"Almighty",
	},

	/* Bard */
	{
		"Apprentice",	/*"Apprentice"*/
		"Songsmith",	/*"Songsmith"*/
		"Bard",	/*"Bard"*/
		"Companion",	/*"Companion"*/
		"Minstrel",	/*"Minstrel"*/
		"Harper",	/*"Harper"*/
		"Loreweaver",	/*"Loreweaver"*/
		"Muse",	/*"Muse"*/
		"Dreamweaver",	/*"Dreamweaver"*/
		"Master Harper",	/*"Master Harper"*/
	},

	/* Red Mage */
	{
		"Novice",
		"Apprentice",
		"Journeyman",
		"Veteran",
		"Enchanter",
		"Champion",
		"Mage-Hero",
		"Baron Mage",
		"Battlemage",
		"Wizard Lord",
	},

	/* Samurai */
	{
		"Initiate",
		"Brother",
		"Disciple",
		"Immaculate",
		"Master",
		"Soft Master",
		"Hard Master",
		"Flower Master",
		"Dragon Master",
		"Grand Master",
	},

	/* ForceTrainer */
	{
		"Initiate",
		"Brother",
		"Disciple",
		"Immaculate",
		"Master",
		"Soft Master",
		"Hard Master",
		"Flower Master",
		"Dragon Master",
		"Grand Master",
	},

	/* Blue Mage */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Warrior */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},

	/* Berserker */
	{
		    "Berserker",
		    "Berserker",
		    "Berserker",
		    "Rage Prince",
		    "Rage Prince",
		    "Rage Prince",
		    "Rage King",
		    "Rage King",
		    "Rage King",
		    "God of Rage",
	},

	/* Weaponsmith */
	{
		"Copper smith",
		"Iron smith",
		"Steel smith",
		"Silver smith",
		"Dragon smith",
		"Spirit smith",
		"Magic smith",
		"Soul smith",
		"God smith",
		"AlmightySmith",
	},

	/* Mirror Master */
	{
		"Mirrorstarer",
		"Mirrorcleaner",
		"Mirrormaker",
		"Mirrormagician",
		"Mirror Guru",
		"Mirror Mage",
		"Mirror King",
		"Mirror Emperor",
		"Mirror Avatar",
		"Ruffnor King",
	},

	/* Ninja */
	{
		"Trainee",
		"Myrmidon",
		"Initiate",
		"Knifer",
		"Bladesman",
		"Hashishin",
		"Black Dagger",
		"Shadowstrike",
		"Assassinator",
		"Death Lord",
	},

	/* Sniper */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},
};
#endif

/*!
 * @brief 青魔法テーブル
 * @details
 * level,  smana,  %fail,  manedam,  %manefail,  use_stat, name
 */
const monster_power monster_powers[MAX_MONSPELLS] =
{
#ifdef JP
{  1,   1,  10,    0,  15, A_CON,  "叫ぶ"},
{ 10,   4,  35,   89,  40, A_INT,  "何か"},
{ 40,  35,  85,    0,  40, A_INT,  "魔力消去"},
{ 35,  30,  80,  800,  70, A_STR,  "ロケット"},
{  5,   1,  20,   18,  15, A_DEX,  "射撃"},
{ 10,   4,  35,   89,  40, A_INT,  "何か"},
{ 10,   4,  35,   89,  40, A_INT,  "何か"},
{ 10,   4,  35,   89,  40, A_INT,  "何か"},
{ 20,  15,  55, 1600,  95, A_CON,  "酸のブレス"},
{ 20,  15,  55, 1600,  95, A_CON,  "電撃のブレス"},
{ 20,  15,  55, 1600,  95, A_CON,  "炎のブレス"},
{ 20,  15,  55, 1600,  95, A_CON,  "冷気のブレス"},
{ 20,  15,  55,  800,  95, A_CON,  "毒のブレス"},
{ 20,  15,  70,  550,  95, A_CON,  "地獄のブレス"},
{ 20,  16,  70,  400,  95, A_CON,  "閃光のブレス"},
{ 20,  16,  70,  400,  95, A_CON,  "暗黒のブレス"},
{ 20,  20,  70,  450,  95, A_CON,  "混乱のブレス"},
{ 20,  20,  70,  450,  95, A_CON,  "轟音のブレス"},
{ 20,  20,  70,  600,  95, A_CON,  "カオスのブレス"},
{ 20,  16,  70,  500,  95, A_CON,  "劣化のブレス"},
{ 30,  25,  80,  250,  95, A_CON,  "因果混乱のブレス"},
{ 35,  18,  80,  150,  95, A_CON,  "時間逆転のブレス"},
{ 30,  25,  80,  200,  95, A_CON,  "遅鈍のブレス"},
{ 30,  28,  90,  200,  95, A_CON,  "重力のブレス"},
{ 20,  15,  70,  500,  95, A_CON,  "破片のブレス"},
{ 35,  15,  80,  150,  95, A_CON,  "プラズマのブレス"},
{ 30,  18,  70,  200,  95, A_CON,  "フォースのブレス"},
{ 30,  28,  80,  250,  95, A_CON,  "魔力のブレス"},
{ 25,  20,  95,  320,  80, A_INT,  "放射能球"},
{ 25,  15,  70,  800,  95, A_CON,  "放射性廃棄物のブレス"},
{ 30,  32,  85,  400,  80, A_INT,  "純ログルス"},
{ 35,  40,  95,  150,  95, A_CON,  "分解のブレス"},
{ 18,  13,  55,  630,  80, A_INT,  "アシッド・ボール"},
{ 14,  10,  45,  316,  60, A_INT,  "サンダー・ボール"},
{ 20,  14,  60,  720,  80, A_INT,  "ファイア・ボール"},
{ 15,  11,  50,  320,  60, A_INT,  "アイス・ボール"},
{  5,   3,  40,   48,  20, A_INT,  "悪臭雲"},
{ 25,  18,  70,  350,  80, A_INT,  "地獄球"},
{ 30,  22,  75,  350,  80, A_INT,  "ウォーター・ボール"},
{ 44,  45,  85,  550,  95, A_INT,  "魔力の嵐"},
{ 40,  42,  90,  550,  95, A_INT,  "暗黒の嵐"},
{ 10,   5,  50,    0,  25, A_INT,  "魔力吸収"},
{ 25,  10,  60,    0,  30, A_INT,  "精神攻撃"},
{ 30,  14,  65,    0,  30, A_INT,  "脳攻撃"},
{  3,   1,  25,   24,  20, A_INT,  "軽傷"},
{ 12,   2,  35,   64,  25, A_INT,  "重傷"},
{ 22,   6,  50,  150,  30, A_INT,  "致命傷"},
{ 32,  10,  70,  225,  35, A_INT,  "秘孔を突く"},
{ 13,   7,  40,  178,  40, A_INT,  "アシッド・ボルト"},
{ 10,   5,  35,  130,  35, A_INT,  "サンダー・ボルト"},
{ 15,   9,  50,  210,  45, A_INT,  "ファイア・ボルト"},
{ 12,   6,  35,  162,  40, A_INT,  "アイス・ボルト"},
{ 40,  42,  90,  550,  95, A_INT,  "スター・バースト"},
{ 25,  17,  60,  255,  60, A_INT,  "地獄の矢"},
{ 25,  20,  65,  250,  60, A_INT,  "ウォーター・ボルト"},
{ 25,  24,  90,  400,  80, A_INT,  "魔力の矢"},
{ 25,  20,  80,  216,  60, A_INT,  "プラズマ・ボルト"},
{ 25,  16,  60,  186,  60, A_INT,  "極寒の矢"},
{  3,   1,  25,   12,  20, A_INT,  "マジック・ミサイル"},
{  5,   3,  35,    0,  20, A_INT,  "恐慌"},
{ 10,   5,  40,    0,  20, A_INT,  "盲目"},
{ 10,   5,  40,    0,  20, A_INT,  "パニック・モンスター"},
{ 10,   5,  40,    0,  20, A_INT,  "スロウ・モンスター"},
{ 10,   5,  40,    0,  20, A_INT,  "スリープ・モンスター"},
{ 20,  10,  70,    0,  40, A_INT,  "スピード"},
{ 45, 120,  95,    0,  60, A_INT,  "破滅の手"},
{ 20,  15,  70,    0,  20, A_WIS,  "体力回復"},
{ 45,  65,  80,    0,  60, A_INT,  "無傷の球"},
{  5,   1,  30,    0,  20, A_INT,  "ショート・テレポート"},
{ 15,   8,  40,    0,  30, A_INT,  "テレポート"},
{ 40, 999,  99,    0,  80, A_INT,  "ザ・ワールド"},
{  1,   0,   0,    0,  15, A_INT,  "何か"},
{ 15,   8,  50,    0,  30, A_INT,  "引きよせる"},
{ 20,  13,  80,    0,  30, A_INT,  "テレポート・アウェイ"},
{ 30,  40,  95,    0,  40, A_INT,  "テレポート・レベル"},
{ 35,  30,  80,  350,  70, A_INT,  "光の剣"},
{  5,   1,  20,    0,  15, A_INT,  "暗闇"},
{  5,   1,  20,    0,  15, A_DEX,  "トラップ創造"},
{ 15,   3,  40,    0,  30, A_INT,  "記憶喪失"},
{ 30,  30,  70,    0,  40, A_INT,  "死者復活"},
{ 40,  70,  85,    0,  45, A_INT,  "援軍を呼ぶ"},
{ 45,  90,  90,    0,  50, A_INT,  "サイバーデーモンの召喚"},
{ 25,  20,  65,    0,  30, A_INT,  "モンスターの召喚"},
{ 35,  30,  75,    0,  40, A_INT,  "複数のモンスターの召喚"},
{ 25,  25,  65,    0,  25, A_INT,  "アリの召喚"},
{ 25,  20,  60,    0,  25, A_INT,  "蜘蛛の召喚"},
{ 35,  26,  75,    0,  40, A_INT,  "ハウンドの召喚"},
{ 30,  23,  70,    0,  35, A_INT,  "ヒドラの召喚"},
{ 40,  50,  85,    0,  40, A_INT,  "天使の召喚"},
{ 35,  50,  80,    0,  35, A_INT,  "デーモンの召喚"},
{ 30,  30,  75,    0,  35, A_INT,  "アンデッドの召喚"},
{ 39,  70,  80,    0,  40, A_INT,  "ドラゴンの召喚"},
{ 43,  85,  85,    0,  45, A_INT,  "上級アンデッドの召喚"},
{ 46,  90,  85,    0,  45, A_INT,  "古代ドラゴンの召喚"},
{ 48, 120,  90,    0,  50, A_INT,  "アンバーの王族の召喚"},
{ 50, 150,  95,    0,  50, A_INT,  "ユニークモンスターの召喚"},
#else
{  1,   1,  10,    0,  15, A_CON,  "shriek"},
{ 10,   4,  35,   89,  40, A_INT,  "something"},
{ 40,  35,  85,    0,  40, A_INT,  "dispel-magic"},
{ 35,  30,  80,  800,  70, A_STR,  "rocket"},
{  2,   1,  15,   10,  15, A_DEX,  "arrow"},
{  5,   2,  20,   18,  20, A_DEX,  "arrows"},
{ 12,   3,  25,   30,  25, A_DEX,  "missile"},
{ 16,   4,  30,   42,  30, A_DEX,  "missiles"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath acid"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath lightning"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath fire"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath cold"},
{ 20,  15,  55,  800,  95, A_CON,  "breath poison"},
{ 20,  15,  70,  550,  95, A_CON,  "breath nether"},
{ 20,  16,  70,  400,  95, A_CON,  "breath light"},
{ 20,  16,  70,  400,  95, A_CON,  "breath dark"},
{ 20,  20,  70,  450,  95, A_CON,  "breath confusion"},
{ 20,  20,  70,  450,  95, A_CON,  "breath sound"},
{ 20,  20,  70,  600,  95, A_CON,  "breath chaos"},
{ 20,  16,  70,  500,  95, A_CON,  "breath disenchantment"},
{ 30,  25,  80,  250,  95, A_CON,  "breath nexus"},
{ 35,  18,  80,  150,  95, A_CON,  "breath time"},
{ 30,  25,  80,  200,  95, A_CON,  "breath inertia"},
{ 30,  28,  90,  200,  95, A_CON,  "breath gravity"},
{ 20,  15,  70,  500,  95, A_CON,  "breath shards"},
{ 35,  15,  80,  150,  95, A_CON,  "breath plasma"},
{ 30,  18,  70,  200,  95, A_CON,  "breath force"},
{ 30,  28,  80,  250,  95, A_CON,  "breath mana"},
{ 25,  20,  95,  320,  80, A_INT,  "nuke ball"},
{ 25,  15,  70,  800,  95, A_CON,  "breath nuke"},
{ 30,  32,  85,  400,  80, A_INT,  "raw Logrus"},
{ 35,  40,  95,  150,  95, A_CON,  "breath disintegrate"},
{ 18,  13,  55,  630,  80, A_INT,  "acid ball"},
{ 14,  10,  45,  316,  60, A_INT,  "lightning ball"},
{ 20,  14,  60,  720,  80, A_INT,  "fire ball"},
{ 15,  11,  50,  320,  60, A_INT,  "frost ball"},
{  5,   3,  40,   48,  20, A_INT,  "stinking cloud"},
{ 25,  18,  70,  350,  80, A_INT,  "nether ball"},
{ 30,  22,  75,  350,  80, A_INT,  "water ball"},
{ 44,  45,  85,  550,  95, A_INT,  "mana storm"},
{ 40,  42,  90,  550,  95, A_INT,  "darkness storm"},
{ 10,   5,  50,    0,  25, A_INT,  "drain mana"},
{ 25,  10,  60,    0,  30, A_INT,  "mind blast"},
{ 30,  14,  65,    0,  30, A_INT,  "brain smash"},
{  3,   1,  25,   24,  20, A_INT,  "cause light wounds"},
{ 12,   2,  35,   64,  25, A_INT,  "cause serious wounds"},
{ 22,   6,  50,  150,  30, A_INT,  "cause critical wounds"},
{ 32,  10,  70,  225,  35, A_INT,  "cause mortal wounds"},
{ 13,   7,  40,  178,  40, A_INT,  "acid bolt"},
{ 10,   5,  35,  130,  35, A_INT,  "lightning bolt"},
{ 15,   9,  50,  210,  45, A_INT,  "fire bolt"},
{ 12,   6,  35,  162,  40, A_INT,  "frost bolt"},
{ 40,  42,  90,  550,  95, A_INT,  "starburst"},
{ 25,  17,  60,  255,  60, A_INT,  "nether bolt"},
{ 25,  20,  65,  250,  60, A_INT,  "water bolt"},
{ 25,  24,  90,  400,  80, A_INT,  "mana bolt"},
{ 25,  20,  80,  216,  60, A_INT,  "plasma bolt"},
{ 25,  16,  60,  186,  60, A_INT,  "ice bolt"},
{  3,   1,  25,   12,  20, A_INT,  "magic missile"},
{  5,   3,  35,    0,  20, A_INT,  "scare"},
{ 10,   5,  40,    0,  20, A_INT,  "blind"},
{ 10,   5,  40,    0,  20, A_INT,  "confuse"},
{ 10,   5,  40,    0,  20, A_INT,  "slow"},
{ 10,   5,  40,    0,  20, A_INT,  "sleep"},
{ 20,  10,  70,    0,  40, A_INT,  "speed"},
{ 45, 120,  95,    0,  60, A_INT,  "the Hand of Doom"},
{ 20,  15,  70,    0,  20, A_WIS,  "heal-self"},
{ 45,  65,  80,    0,  60, A_INT,  "make invulnerable"},
{  5,   1,  30,    0,  20, A_INT,  "blink-self"},
{ 15,   8,  40,    0,  30, A_INT,  "teleport-self"},
{ 40, 999,  99,    0,  80, A_INT,  "The world"},
{  1,   0,   0,    0,  15, A_INT,  "something"},
{ 15,   8,  50,    0,  30, A_INT,  "teleport to"},
{ 20,  13,  80,    0,  30, A_INT,  "teleport away"},
{ 30,  40,  95,    0,  40, A_INT,  "teleport level"},
{ 35,  30,  80,  350,  70, A_INT,  "psycho-spear"},
{  5,   1,  20,    0,  15, A_INT,  "create darkness"},
{  5,   1,  20,    0,  15, A_DEX,  "create traps"},
{ 15,   3,  40,    0,  30, A_INT,  "cause amnesia"},
{ 30,  30,  70,    0,  40, A_INT,  "raise dead"},
{ 40,  70,  85,    0,  45, A_INT,  "summon aid"},
{ 45,  90,  90,    0,  50, A_INT,  "summon Cyberdemons"},
{ 25,  20,  65,    0,  30, A_INT,  "summon a monster"},
{ 35,  30,  75,    0,  40, A_INT,  "summon monsters"},
{ 25,  25,  65,    0,  25, A_INT,  "summon ants"},
{ 25,  20,  60,    0,  25, A_INT,  "summon spiders"},
{ 35,  26,  75,    0,  40, A_INT,  "summon hounds"},
{ 30,  23,  70,    0,  35, A_INT,  "summon hydras"},
{ 40,  50,  85,    0,  40, A_INT,  "summon an angel"},
{ 35,  50,  80,    0,  35, A_INT,  "summon a daemon"},
{ 30,  30,  75,    0,  35, A_INT,  "summon an undead"},
{ 39,  70,  80,    0,  40, A_INT,  "summon a dragon"},
{ 43,  85,  85,    0,  45, A_INT,  "summon Greater Undead"},
{ 46,  90,  85,    0,  45, A_INT,  "summon Ancient Dragon"},
{ 48, 120,  90,    0,  50, A_INT,  "summon Lords of Amber"},
{ 50, 150,  95,    0,  50, A_INT,  "summon Unique Monsters"},
#endif

};

/*!
 * @brief モンスター魔法名テーブル
 */
const cptr monster_powers_short[MAX_MONSPELLS] = {
#ifdef JP

	"叫ぶ", "何か", "魔力消去", "ロケット", "射撃", "何か", "何か", "何か",
	"酸", "電撃", "火炎", "冷気", "毒", "地獄", "閃光", "暗黒",
	"混乱", "轟音", "カオス", "劣化", "因果混乱", "時間逆転", "遅鈍", "重力",
	"破片", "プラズマ", "フォース", "魔力", "放射能球", "放射性廃棄物", "純ログルス", "分解",

	"酸", "電撃", "火炎", "冷気", "悪臭雲", "地獄球", "ウォーター", "魔力の嵐",
	"暗黒の嵐", "魔力吸収", "精神攻撃", "脳攻撃", "軽傷", "重傷", "致命傷", "秘孔を突く",
	"酸", "電撃", "火炎", "冷気", "スターバースト", "地獄の矢", "ウォーター", "魔力の矢",
	"プラズマ", "極寒", "マジックミサイル", "恐慌", "盲目", "混乱", "減速", "睡眠",

	"加速", "破滅の手", "体力回復", "無傷の球", "ショートテレポート", "テレポート", "時を止める", "何か",
	"引きよせる", "テレポートアウェイ", "テレポートレベル", "光の剣", "暗闇", "トラップ創造", "記憶喪失", "死者復活",
	"援軍", "サイバーデーモン", "モンスター", "複数のモンスター", "蟻", "蜘蛛", "ハウンド", "ヒドラ",
	"天使", "悪魔", "アンデッド", "ドラゴン", "上級アンデッド", "古代ドラゴン", "アンバーの王族", "ユニーク"

#else

	"Shriek", "Something", "Dispel-magic", "Rocket", "Arrow", "Arrows", "Missile", "Missiles",
	"Acid", "Lightning", "Fire", "Cold", "Poison", "Nether", "Light", "Dark",
	"Confusion", "Sound", "Chaos", "Disenchantment", "Nexus", "Time", "Inertia", "Gravity",
	"Shards", "Plasma", "Force", "Mana", "Nuke", "Nuke", "Logrus", "Disintergrate",

	"Acid", "Lightning", "Fire", "Frost", "Stinking Cloud", "Nether", "Water", "Mana storm",
	"Darkness storm", "Drain mana", "Mind blast", "Brain smash", "Cause Light Wound", "Cause Serious Wound", "Cause Critical Wound", "Cause Mortal Wound",
	"Acid", "Lightning", "Fire", "Frost", "Starburst", "Nether", "Water", "Mana",
	"Plasm", "Ice", "Magic missile", "Scare", "Blind", "Confuse", "Slow", "Sleep",

	"Speed", "Hand of doom", "Heal-self", "Invulnerable", "Blink", "Teleport", "The world", "Something",
	"Teleport to", "Teleport away", "Teleport level", "Psycho-spear", "Create darkness", "Create traps", "Amnesia", "Raise dead",
	"Aid", "Cyberdeamons", "A monster", "Monsters", "Ants", "Spiders", "Hounds", "Hydras",
	"Angel", "Daemon", "Undead", "Dragon", "Greater Undead", "Ancient Dragon", "Lords of Amber", "Unique monsters"

#endif
};


/*!
 * @brief 色名称テーブル / Hack -- the "basic" color names (see "TERM_xxx")
 */
const cptr color_names[16] =
{
#ifdef JP
	"黒",
	"白",
	"青灰色",
	"オレンジ",
	"赤",
	"緑",
	"青",
	"琥珀色",
	"灰色",
	"明青灰色",
	"紫",
	"黄",
	"明るい赤",
	"明るい緑",
	"明るい青",
	"明琥珀色",
#else
	"Dark",
	"White",
	"Slate",
	"Orange",
	"Red",
	"Green",
	"Blue",
	"Umber",
	"Light Dark",
	"Light Slate",
	"Violet",
	"Yellow",
	"Light Red",
	"Light Green",
	"Light Blue",
	"Light Umber",
#endif

};


/*!
 * @brief 能力値テーブル / Abbreviations of healthy stats
 */
const cptr stat_names[6] =
{
#ifdef JP
	"腕力 :", "知能 :", "賢さ :", "器用 :", "耐久 :", "魅力 :"
#else
	"STR : ", "INT : ", "WIS : ", "DEX : ", "CON : ", "CHR : "
#endif

};

/*!
 * @brief 能力値テーブル(能力低下時) / Abbreviations of damaged stats
 */
const cptr stat_names_reduced[6] =
{
#ifdef JP
	"腕力x:", "知能x:", "賢さx:", "器用x:", "耐久x:", "魅力x:"
#else
	"Str : ", "Int : ", "Wis : ", "Dex : ", "Con : ", "Chr : "
#endif

};


/*!
 * @brief サブウィンドウ名称テーブル
 * @details
 * <pre>
 * Certain "screens" always use the main screen, including News, Birth,
 * Dungeon, Tomb-stone, High-scores, Macros, Colors, Visuals, Options.
 *
 * Later, special flags may allow sub-windows to "steal" stuff from the
 * main window, including File dump (help), File dump (artifacts, uniques),
 * Character screen, Small scale map, Previous Messages, Store screen, etc.
 *
 * The "ctrl-i" (tab) command flips the "Display inven/equip" and "Display
 * equip/inven" flags for all windows.
 *
 * The "ctrl-g" command (or pseudo-command) should perhaps grab a snapshot
 * of the main screen into any interested windows.
 * </pre>
 */
const cptr window_flag_desc[32] =
{
#ifdef JP
	"持ち物/装備一覧",
#else
	"Display inven/equip",
#endif

#ifdef JP
	"装備/持ち物一覧",
#else
	"Display equip/inven",
#endif

#ifdef JP
	"呪文一覧",
#else
	"Display spell list",
#endif

#ifdef JP
	"キャラクタ情報",
#else
	"Display character",
#endif
		
#ifdef JP
	"視界内のモンスター表示",
#else
	"Display monsters in sight",
#endif		

	NULL,
#ifdef JP
	"メッセージ",
#else
	"Display messages",
#endif

#ifdef JP
	"ダンジョン全体図",
#else
	"Display overhead view",
#endif

#ifdef JP
	"モンスターの思い出",
#else
	"Display monster recall",
#endif

#ifdef JP
	"アイテムの詳細",
#else
	"Display object recall",
#endif

#ifdef JP
	"自分の周囲を表示", 
#else
	"Display dungeon view",
#endif

#ifdef JP
	"記念撮影",
#else
	"Display snap-shot",
#endif

	NULL,
	NULL,
#ifdef JP
	"ボーグ・メッセージ",
#else
	"Display borg messages",
#endif

#ifdef JP
	"ボーグ・ステータス",
#else
	"Display borg status",
#endif

	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};


/*!
 * @brief オプションテーブル /
 * Available Options
 */
const option_type option_info[] =
{
	/*** Input Options ***/

	{ &rogue_like_commands,         FALSE, OPT_PAGE_INPUT, 0, 0,
	"rogue_like_commands",          _("ローグ風キー配置を使用する", "Rogue-like commands") },

	{ &always_pickup,               FALSE, OPT_PAGE_INPUT, 0, 5,
	"always_pickup",                _("常にアイテムを拾う" , "Pick things up by default") },

	{ &carry_query_flag,            FALSE, OPT_PAGE_INPUT, 0, 3,
	"carry_query_flag",             _("アイテムを拾う前に確認する", "Prompt before picking things up") },
		
	{ &quick_messages,              TRUE,  OPT_PAGE_INPUT, 0, 1,
	"quick_messages",               _("クイック・メッセージを使用する", "Activate quick messages") },
		
	{ &auto_more,                   FALSE, OPT_PAGE_INPUT, 2, 6,
	"auto_more",                    _("キー待ちしないで連続でメッセージを表示する", "Automatically clear '-more-' prompts") },
	
	{ &command_menu,                TRUE,  OPT_PAGE_INPUT, 2, 7,
	"command_menu",                 _("メニューによりコマンド選択を有効にする", "Enable command selection menu") },
	
	{ &other_query_flag,            FALSE, OPT_PAGE_INPUT, 0, 2,
	"other_query_flag",             _("床上のアイテムを使用するときに確認する", "Prompt for floor item selection") },
	
	{ &use_old_target,              FALSE, OPT_PAGE_INPUT, 0, 4,
	"use_old_target",               _("常に以前のターゲットを指定する", "Use old target by default") },

	{ &always_repeat,               TRUE,  OPT_PAGE_INPUT, 0, 6,
	"always_repeat",                _("コマンド自動繰り返し", "Repeat obvious commands") },

	{ &confirm_destroy,             FALSE, OPT_PAGE_INPUT, 5, 3,
	"confirm_destroy",              _("「無価値」なアイテムを破壊する時確認する", "Prompt for destruction of known worthless items") },

	{ &confirm_wear,                TRUE,  OPT_PAGE_INPUT, 5, 4,
	"confirm_wear",                 _("呪われた物を装備する時確認する", "Confirm to wear/wield known cursed items") },

	{ &confirm_quest,               TRUE,  OPT_PAGE_INPUT, 1, 9,
	"confirm_quest",                _("クエストを諦めて階段で逃げる前に確認する", "Prompt before exiting a quest level") },

	{ &target_pet,                  FALSE, OPT_PAGE_INPUT, 2, 5,
	"target_pet",                   _("ペットをターゲットにする", "Allow targetting pets") },

#ifdef ALLOW_EASY_OPEN
	{ &easy_open,                   TRUE,  OPT_PAGE_INPUT, 5, 7,
	"easy_open",                    _("自動的にドアを開ける", "Automatically open doors") },
#endif /* ALLOW_EASY_OPEN */

#ifdef ALLOW_EASY_DISARM
	{ &easy_disarm,                 TRUE,  OPT_PAGE_INPUT, 5, 8,
	"easy_disarm",                  _("自動的に罠を解除する", "Automatically disarm traps") },
#endif /* ALLOW_EASY_DISARM */

#ifdef ALLOW_EASY_FLOOR /* TNB */
	{ &easy_floor,                  FALSE, OPT_PAGE_INPUT, 5, 9,
	"easy_floor",                   _("床上で重なったアイテムをリストする", "Display floor stacks in a list") },
#endif /* ALLOW_EASY_FLOOR -- TNB */

	{ &use_command,                 FALSE, OPT_PAGE_INPUT, 5, 10,
	"use_command",                  _("「使う(a)」コマンドでアイテムを何でも使える", "Allow unified use command") },

	{ &over_exert,                  FALSE, OPT_PAGE_INPUT, 0, 29,
	"over_exert",                   _("MPが足りなくても魔法に挑戦する", "Allow casting spells when short of mana") },

	{ &numpad_as_cursorkey,         TRUE, OPT_PAGE_INPUT, 2, 31,
	"numpad_as_cursorkey",          _("エディタ内でテンキーをカーソルキーとして使う", "Use numpad keys as cursor keys in editor mode") },

	/*** Map Screen Options ***/

	{ &center_player,               FALSE, OPT_PAGE_MAPSCREEN, 5, 11,
	"center_player",                _("常にプレイヤーを中心に置く(*遅い*)", "Center map while walking (*slow*)") },

	{ &center_running,              TRUE,  OPT_PAGE_MAPSCREEN, 5, 12,
	"center_running",               _("走っている時でも中心に置く", "Centering even while running") },

	{ &view_yellow_lite,            TRUE,  OPT_PAGE_MAPSCREEN, 1, 28,
	"view_yellow_lite",             _("明かりの範囲を特別な色で表示する", "Use special colors for torch-lit grids") },

	{ &view_bright_lite,            TRUE,  OPT_PAGE_MAPSCREEN, 1, 29,
	"view_bright_lite",             _("視界の範囲を特別な色で表示する", "Use special colors for 'viewable' grids") },

	{ &view_granite_lite,           TRUE,  OPT_PAGE_MAPSCREEN, 1, 30,
	"view_granite_lite",            _("壁を特別な色で表示する(重い)", "Use special colors for wall grids (slow)") },

	{ &view_special_lite,           TRUE,  OPT_PAGE_MAPSCREEN, 1, 31,
	"view_special_lite",            _("床を特別な色で表示する(重い)", "Use special colors for floor grids (slow)") },

	{ &view_perma_grids,            TRUE,  OPT_PAGE_MAPSCREEN, 1, 6,
	"view_perma_grids",             _("明るい場所はそのままにする", "Map remembers all perma-lit grids") },

	{ &view_torch_grids,            FALSE, OPT_PAGE_MAPSCREEN, 1, 7,
	"view_torch_grids",             _("明かりで照らした場所はそのままにする", "Map remembers all torch-lit grids") },

	{ &view_unsafe_grids,           FALSE, OPT_PAGE_MAPSCREEN, 1, 8,
	"view_unsafe_grids",            _("トラップ感知済みでない場所を表示する", "Map marked by detect traps") },

	{ &view_reduce_view,            FALSE, OPT_PAGE_MAPSCREEN, 1, 17,
	"view_reduce_view",             _("街では視野を狭くする", "Reduce view-radius in town") },

	{ &fresh_before,                TRUE,  OPT_PAGE_MAPSCREEN, 1, 23,
	"fresh_before",                 _("連続コマンド中に画面を再描画し続ける", "Flush output while continuous command") },

	{ &fresh_after,                 FALSE, OPT_PAGE_MAPSCREEN, 1, 24,
	"fresh_after",                  _("コマンド後に画面を常に再描画し続ける", "Flush output after monster's move") },

	{ &fresh_message,               FALSE, OPT_PAGE_MAPSCREEN, 1, 25,
	"fresh_message",                _("メッセージの後に画面を再描画する", "Flush output after every message") },

	{ &hilite_player,               FALSE, OPT_PAGE_MAPSCREEN, 1, 27,
	"hilite_player",                _("プレイヤーにカーソルを合わせる", "Hilite the player with the cursor") },

	{ &display_path,                FALSE, OPT_PAGE_MAPSCREEN, 2, 8,
	"display_path",                 _("魔法や矢の軌跡を表示する", "Display actual path before shooting") },

	/*** Text Display Options ***/

	{ &plain_descriptions,          TRUE,  OPT_PAGE_TEXT, 5, 1,
	"plain_descriptions",           _("アイテムの記述を簡略にする", "Plain object descriptions") },

	{ &plain_pickup,                FALSE, OPT_PAGE_TEXT, 6, 6,
	"plain_pickup",                 _("「拾った」メッセージを簡略化する", "Plain pickup messages(japanese only)") },

	{ &always_show_list,            TRUE,  OPT_PAGE_TEXT, 4, 0,
	"always_show_list",             _("選択時には常に一覧を表示する", "Always show list when choosing items") },

	{ &depth_in_feet,               FALSE, OPT_PAGE_TEXT, 0, 7,
	"depth_in_feet",                _("ダンジョンの深さをフィートで表示する", "Show dungeon level in feet") },

	{ &show_labels,                 TRUE,  OPT_PAGE_TEXT, 0, 10,
	"show_labels",                  _("装備一覧で装備場所を表示する", "Show labels in object listings") },

	{ &show_weights,                TRUE,  OPT_PAGE_TEXT, 0, 11,
	"show_weights",                 _("アイテム一覧で重量を表示する", "Show weights in object listings") },

	{ &show_item_graph,             TRUE,  OPT_PAGE_TEXT, 2, 0,
	"show_item_graph",              _("アイテムのシンボルを表示する", "Show items graphics") },

	{ &equippy_chars,               TRUE,  OPT_PAGE_TEXT, 1, 12,
	"equippy_chars",                _("ステータスに文字で装備を表示する", "Display 'equippy' chars") },

	{ &display_mutations,           FALSE, OPT_PAGE_TEXT, 5, 0,
	"display_mutations",            _("'C'コマンドで突然変異を表示する", "Display mutations in 'C'haracter Display") },

	{ &compress_savefile,           FALSE, OPT_PAGE_TEXT, 1, 26,
	"compress_savefile",            _("セーブ・ファイル中のメッセージを圧縮する", "Compress messages in savefiles") },

	{ &abbrev_extra,                FALSE, OPT_PAGE_TEXT, 2, 10,
	"abbrev_extra",                 _("アイテムに追加耐性/能力の略称を刻む", "Describe obj's extra resistances by abbreviation") },

	{ &abbrev_all,                  FALSE, OPT_PAGE_TEXT, 2, 11,
	"abbrev_all",                   _("アイテムに全ての耐性/能力の略称を刻む", "Describe obj's all resistances by abbreviation") },

	{ &exp_need,                    FALSE, OPT_PAGE_TEXT, 2, 12,
	"exp_need",                     _("次のレベルに必要な経験値を表示する", "Show the experience needed for next level") },

	{ &ignore_unview,               FALSE, OPT_PAGE_TEXT, 2, 13,
	"ignore_unview",                _("視界外のモンスターの行動を表示しない", "Ignore whenever any monster does") },
	
	{ &show_ammo_detail,            TRUE, OPT_PAGE_TEXT, 2, 14,
	"show_ammo_detail",             _("矢弾のダメージの説明を表示する", "Show description of ammo damage") },
		
	{ &show_ammo_no_crit,           FALSE, OPT_PAGE_TEXT, 2, 15,
	"show_ammo_no_crit",            _("会心を考慮しない場合の矢弾のダメージを表示する", "Show ammo damage with no critical") },

	{ &show_ammo_crit_ratio,           FALSE, OPT_PAGE_TEXT, 2, 16,
	"show_ammo_crit_ratio",            _("矢弾の会心発生率を表示する", "Show critical ratio of ammo") },
		

	/*** Game-Play ***/

	{ &stack_force_notes,           TRUE,  OPT_PAGE_GAMEPLAY, 0, 8,
	"stack_force_notes",            _("異なる銘のアイテムをまとめる", "Merge inscriptions when stacking") },

	{ &stack_force_costs,           FALSE, OPT_PAGE_GAMEPLAY, 0, 9,
	"stack_force_costs",            _("異なる割引表示のアイテムをまとめる", "Merge discounts when stacking") },

	{ &expand_list,                 TRUE,  OPT_PAGE_GAMEPLAY, 1, 5,
	"expand_list",                  _("「一覧」コマンドを拡張する", "Expand the power of the list commands") },

	{ &small_levels,                TRUE,  OPT_PAGE_GAMEPLAY, 0, 30,
	"small_levels",                 _("非常に小さいフロアの生成を可能にする", "Allow unusually small dungeon levels") },

	{ &always_small_levels,         FALSE, OPT_PAGE_GAMEPLAY, 2, 3,
	"always_small_levels",          _("常に非常に小さいフロアを生成する", "Always create unusually small dungeon levels") },

	{ &empty_levels,                TRUE,  OPT_PAGE_GAMEPLAY, 0, 31,
	"empty_levels",                 _("空っぽの「アリーナ」レベルの生成を可能にする", "Allow empty 'arena' levels") },

	{ &bound_walls_perm,            FALSE, OPT_PAGE_GAMEPLAY, 2, 1,
	"bound_walls_perm",             _("ダンジョンの外壁を永久岩にする", "Boundary walls become 'permanent wall'") },

	{ &last_words,                  TRUE,  OPT_PAGE_GAMEPLAY, 0, 28,
	"last_words",                   _("キャラクターが死んだ時遺言をのこす", "Leave last words when your character dies") },

#ifdef WORLD_SCORE
	{ &send_score,                  TRUE,  OPT_PAGE_GAMEPLAY, 4, 6,
	"send_score",                   _("スコアサーバにスコアを送る", "Send score dump to the world score server") },
#endif
	
	{ &allow_debug_opts,            FALSE, OPT_PAGE_GAMEPLAY, 6, 11,
	"allow_debug_opts",             _("デバッグ/詐欺オプションを許可する", "Allow use of debug/cheat options") },

	/*** Disturbance ***/

	{ &find_ignore_stairs,          FALSE, OPT_PAGE_DISTURBANCE, 0, 16,
	"find_ignore_stairs",           _("階段は通過する", "Run past stairs") },

	{ &find_ignore_doors,           TRUE,  OPT_PAGE_DISTURBANCE, 0, 17,
	"find_ignore_doors",            _("ドアは通過する", "Run through open doors") },

	{ &find_cut,                    FALSE, OPT_PAGE_DISTURBANCE, 0, 18,
	"find_cut",                     _("曲り角を斜めに最短距離で通過する", "Run past known corners") },

	{ &check_abort,                 TRUE,  OPT_PAGE_DISTURBANCE, 1, 18,
	"check_abort",                  _("連続コマンドはキー入力で中断する", "Check for user abort while continuous command") },

	{ &flush_failure,               TRUE,  OPT_PAGE_DISTURBANCE, 1, 20,
	"flush_failure",                _("様々なミス発生時に入力をクリアする", "Flush input on various failures") },

	{ &flush_disturb,               FALSE, OPT_PAGE_DISTURBANCE, 1, 21,
	"flush_disturb",                _("障害発生時に入力をクリアする", "Flush input whenever disturbed") },

	{ &disturb_move,                FALSE, OPT_PAGE_DISTURBANCE, 0, 20,
	"disturb_move",                 _("どこのモンスターが動いても行動を中止する", "Disturb whenever any monster moves") },

	{ &disturb_high,                FALSE, OPT_PAGE_DISTURBANCE, 1, 3,
	"disturb_high",                 _("レベルの高いモンスターが動いたら行動を中止する", "Disturb whenever high-level monster moves") },

	{ &disturb_near,                TRUE,  OPT_PAGE_DISTURBANCE, 0, 21,
	"disturb_near",                 _("視界内のモンスターが動いたら行動を中止する", "Disturb whenever viewable monster moves") },

	{ &disturb_pets,                FALSE, OPT_PAGE_DISTURBANCE, 5, 6,
	"disturb_pets",                 _("視界内のペットが動いたら行動を中止する", "Disturb when visible pets move") },

	{ &disturb_panel,               TRUE,  OPT_PAGE_DISTURBANCE, 0, 22,
	"disturb_panel",                _("画面スクロール時に行動を中止する", "Disturb whenever map panel changes") },

	{ &disturb_state,               TRUE,  OPT_PAGE_DISTURBANCE, 0, 23,
	"disturb_state",                _("自分のステータス変化時に行動を中止する", "Disturb whenever player state changes") },

	{ &disturb_minor,               TRUE,  OPT_PAGE_DISTURBANCE, 0, 24,
	"disturb_minor",                _("些細なことが起きても行動を中止する", "Disturb whenever boring things happen") },

	{ &ring_bell,                   FALSE, OPT_PAGE_DISTURBANCE, 0, 14,
	"ring_bell",                    _("エラー時にビープ音を鳴らす", "Audible bell (on errors, etc)") },

	{ &disturb_trap_detect,         TRUE,  OPT_PAGE_DISTURBANCE, 0, 27,
	"disturb_trap_detect",          _("トラップ感知範囲外に出る直前に行動を中止する", "Disturb when leaving trap detected area") },

	{ &alert_trap_detect,           FALSE, OPT_PAGE_DISTURBANCE, 0, 25,
	"alert_trap_detect",            _("トラップ感知範囲外に出る直前に警告する", "Alert when leaving trap detected area") },

	/*** Birth Options ***/
	{ &manual_haggle,               FALSE, OPT_PAGE_BIRTH, 1, 0,
	"manual_haggle",                _("店で値切り交渉をする", "Manually haggle in stores") },

	{ &easy_band,                   FALSE, OPT_PAGE_BIRTH, 6, 31,
	"easy_band",                    _("初心者用簡単モード(*)", "Easy Mode (*)") },

	{ &smart_learn,                 TRUE,  OPT_PAGE_BIRTH, 1, 14,
	"smart_learn",                  _("モンスターは失敗を学習する(*)", "Monsters learn from their mistakes (*)") },
	
	{ &smart_cheat,                 FALSE, OPT_PAGE_BIRTH, 1, 15,
	"smart_cheat",                  _("モンスターはプレイヤーの弱みを突く(*)", "Monsters exploit players weaknesses (*)") },

	{ &vanilla_town,                FALSE, OPT_PAGE_BIRTH, 6, 0,
	"vanilla_town",                 _("元祖の街/クエストと荒野なし", "Use 'vanilla' town without quests and wilderness") },

	{ &lite_town,                   FALSE, OPT_PAGE_BIRTH, 6, 1,
	"lite_town",                    _("小規模な街/荒野なし", "Use 'lite' town without a wilderness") },

	{ &ironman_shops,               FALSE, OPT_PAGE_BIRTH, 6, 2,
	"ironman_shops",                _("(鉄人用)店を使用しない(*)", "Stores are permanently closed (*)") },

	{ &ironman_small_levels,        FALSE, OPT_PAGE_BIRTH, 6, 3,
	"ironman_small_levels",         _("(鉄人用)常に非常に小さいフロアを生成(*)", "Always create unusually small dungeon levels (*)") },

	{ &ironman_downward,            FALSE, OPT_PAGE_BIRTH, 6, 4,
	"ironman_downward",             _("(鉄人用)帰還と上り階段なし(*)", "Disable recall and use of up stairs (*)") },

	{ &ironman_empty_levels,        FALSE, OPT_PAGE_BIRTH, 6, 8,
	"ironman_empty_levels",         _("(鉄人用)常に空っぽのアリーナレベルを生成(*)", "Always create empty 'arena' levels (*)") },

	{ &ironman_rooms,               FALSE, OPT_PAGE_BIRTH, 6, 12,
	"ironman_rooms",                _("(鉄人用)常に普通でない部屋を生成する(*)", "Always generate very unusual rooms (*)") },

	{ &ironman_nightmare,           FALSE, OPT_PAGE_BIRTH, 6, 18,
	"ironman_nightmare",            _("(鉄人用)悪夢モード(これは全く不条理です！)(*)", "Nightmare mode(it isn't even remotely fair!)(*)") },

	{ &left_hander,                 FALSE, OPT_PAGE_BIRTH, 6, 13,
	"left_hander",                  _("左利きである", "Left-Hander") },

	{ &preserve_mode,               TRUE,  OPT_PAGE_BIRTH, 6, 14,
	"preserve_mode",                _("伝説のアイテムを取り逃しても再生成される(*)", "Preserve artifacts (*)") },

	{ &autoroller,                  TRUE,  OPT_PAGE_BIRTH, 6, 15,
	"autoroller",                   _("能力値にオートローラー使用(*)", "Allow use of autoroller for stats (*)") },

	{ &autochara,                   FALSE, OPT_PAGE_BIRTH, 6, 16,
	"autochara",                   _("体格/地位にオートローラー使用", "Autoroll for weight, height and social status") },

	{ &powerup_home,                TRUE,  OPT_PAGE_BIRTH, 4, 3,
	"powerup_home",                 _("我が家を拡張する(*)", "Increase capacity of your home (*)") },

	/*** Easy Object Auto-Destroyer ***/

	{ &destroy_items,               FALSE, OPT_PAGE_AUTODESTROY, 7, 0,
	"destroy_items",                _("アイテムの簡易自動破壊を使用する", "Use easy auto-destroyer") },

	{ &destroy_feeling,             FALSE, OPT_PAGE_AUTODESTROY, 7, 8,
	"destroy_feeling",              _("簡易鑑定したとき自動破壊を適用する", "Apply auto-destroy as sense feeling") },

	{ &destroy_identify,            FALSE, OPT_PAGE_AUTODESTROY, 7, 9,
	"destroy_identify",             _("鑑定したとき自動破壊を適用する", "Apply auto-destroy as identify an item") },

	{ &leave_worth,                 TRUE,  OPT_PAGE_AUTODESTROY, 7, 2,
	"leave_worth",                  _("価値があるアイテムは壊さない", "Auto-destroyer leaves known worthy items") },

	{ &leave_equip,                 FALSE, OPT_PAGE_AUTODESTROY, 7, 3,
	"leave_equip",                  _("武器/防具は壊さない", "Auto-destroyer leaves weapons and armour") },

	{ &leave_chest,                 TRUE,  OPT_PAGE_AUTODESTROY, 7, 7,
	"leave_chest",                  _("開封されていない箱は壊さない", "Auto-destroyer leaves closed chests") },

	{ &leave_wanted,                TRUE,  OPT_PAGE_AUTODESTROY, 7, 4,
	"leave_wanted",                 _("賞金首の死体/骨は壊さない", "Auto-destroyer leaves wanted corpses") },

	{ &leave_corpse,                FALSE, OPT_PAGE_AUTODESTROY, 7, 5,
	"leave_corpse",                 _("死体/骨は壊さない", "Auto-destroyer leaves corpses and skeletons") },

	{ &leave_junk,                  FALSE, OPT_PAGE_AUTODESTROY, 7, 6,
	"leave_junk",                   _("がらくたは壊さない", "Auto-destroyer leaves junk") },

	{ &leave_special,               TRUE,  OPT_PAGE_AUTODESTROY, 7, 1,
	"leave_special",                _("種族/職業で特別に必要なアイテムは壊さない", "Auto-destroyer leaves items your race/class needs") },

	/*** Play-record Options ***/

	{ &record_fix_art,              TRUE,  OPT_PAGE_PLAYRECORD, 4, 11,
	"record_fix_art",               _("固定アーティファクトの入手を記録する", "Record fixed artifacts") },

	{ &record_rand_art,             TRUE,  OPT_PAGE_PLAYRECORD, 4, 12,
	"record_rand_art",              _("ランダムアーティファクトの入手を記録する", "Record random artifacts") },

	{ &record_destroy_uniq,         TRUE,  OPT_PAGE_PLAYRECORD, 4, 13,
	"record_destroy_uniq",          _("ユニークモンスターを倒したときを記録する", "Record when destroy unique monster") },

	{ &record_fix_quest,            TRUE,  OPT_PAGE_PLAYRECORD, 4, 14,
	"record_fix_quest",             _("固定クエストの達成を記録する", "Record fixed quests") },

	{ &record_rand_quest,           TRUE,  OPT_PAGE_PLAYRECORD, 4, 15,
	"record_rand_quest",            _("ランダムクエストの達成を記録する", "Record random quests") },

	{ &record_maxdepth,             TRUE,  OPT_PAGE_PLAYRECORD, 4, 16,
	"record_maxdepth",              _("最深階を更新したときに記録する", "Record movements to deepest level") },

	{ &record_stair,                TRUE,  OPT_PAGE_PLAYRECORD, 4, 17,
	"record_stair",                 _("階の移動を記録する", "Record recall and stair movements") },

	{ &record_buy,                  TRUE,  OPT_PAGE_PLAYRECORD, 4, 18,
	"record_buy",                   _("アイテムの購入を記録する", "Record purchased items") },

	{ &record_sell,                 FALSE, OPT_PAGE_PLAYRECORD, 4, 19,
	"record_sell",                  _("アイテムの売却を記録する", "Record sold items") },

	{ &record_danger,               TRUE,  OPT_PAGE_PLAYRECORD, 4, 20,
	"record_danger",                _("ピンチになったときを記録する", "Record hitpoint warning") },

	{ &record_arena,                TRUE,  OPT_PAGE_PLAYRECORD, 4, 21,
	"record_arena",                 _("アリーナでの勝利を記録する", "Record arena victories") },

	{ &record_ident,                TRUE,  OPT_PAGE_PLAYRECORD, 4, 22,
	"record_ident",                 _("未判明のアイテムの識別を記録する", "Record first identified items") },

	{ &record_named_pet,            FALSE, OPT_PAGE_PLAYRECORD, 4, 23,
	"record_named_pet",             _("名前つきペットの情報を記録する", "Record informations of named pets") },

	/*** End of Table ***/

	{ NULL,                         0, 0, 0, 0,
	NULL,                           NULL }
};


#ifdef JP
/*!
 * @brief カオスパトロン名テーブル
 */
const cptr chaos_patrons[MAX_PATRON] =
{
	"スローター",
	"マベロード",
	"チャードロス",
	"ハイオンハーン",
	"キシオムバーグ",

	"ピアレー",
	"バラン",
	"アリオッチ",
	"イーカー",
	"ナージャン",

	"バロ",
	"コーン",
	"スラーネッシュ",
	"ナーグル",
	"ティーンチ",

	"カイン"
};
#else
const cptr chaos_patrons[MAX_PATRON] =
{
	"Slortar",
	"Mabelode",
	"Chardros",
	"Hionhurn",
	"Xiombarg",

	"Pyaray",
	"Balaan",
	"Arioch",
	"Eequor",
	"Narjhan",

	"Balo",
	"Khorne",
	"Slaanesh",
	"Nurgle",
	"Tzeentch",

	"Khaine"
};
#endif


/*!
 * @brief カオスパトロンの報酬能力値テーブル
 */
const int chaos_stats[MAX_PATRON] =
{
	A_CON,  /* Slortar */
	A_CON,  /* Mabelode */
	A_STR,  /* Chardros */
	A_STR,  /* Hionhurn */
	A_STR,  /* Xiombarg */

	A_INT,  /* Pyaray */
	A_STR,  /* Balaan */
	A_INT,  /* Arioch */
	A_CON,  /* Eequor */
	A_CHR,  /* Narjhan */

	-1,     /* Balo */
	A_STR,  /* Khorne */
	A_CHR,  /* Slaanesh */
	A_CON,  /* Nurgle */
	A_INT,  /* Tzeentch */

	A_STR,  /* Khaine */
};


/*!
 * @brief カオスパトロンの報酬テーブル
 */
const int chaos_rewards[MAX_PATRON][20] =
{
	/* Slortar the Old: */
	{
		REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_POLY_SLF,
		REW_POLY_SLF, REW_POLY_SLF, REW_GAIN_ABL, REW_GAIN_ABL, REW_GAIN_EXP,
		REW_GOOD_OBJ, REW_CHAOS_WP, REW_GREA_OBJ, REW_AUGM_ABL, REW_AUGM_ABL
	},

	/* Mabelode the Faceless: */
	{
		REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_H_SUMMON, REW_SUMMON_M,
		REW_SUMMON_M, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_POLY_WND,
		REW_POLY_SLF, REW_HEAL_FUL, REW_HEAL_FUL, REW_GAIN_ABL, REW_SER_UNDE,
		REW_CHAOS_WP, REW_GOOD_OBJ, REW_GOOD_OBJ, REW_GOOD_OBS, REW_GOOD_OBS
	},

	/* Chardros the Reaper: */
	{
		REW_WRATH, REW_WRATH, REW_HURT_LOT, REW_PISS_OFF, REW_H_SUMMON,
		REW_SUMMON_M, REW_IGNORE, REW_IGNORE, REW_DESTRUCT, REW_SER_UNDE,
		REW_GENOCIDE, REW_MASS_GEN, REW_MASS_GEN, REW_DISPEL_C, REW_GOOD_OBJ,
		REW_CHAOS_WP, REW_GOOD_OBS, REW_GOOD_OBS, REW_AUGM_ABL, REW_AUGM_ABL
	},

	/* Hionhurn the Executioner: */
	{
		REW_WRATH, REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL,
		REW_IGNORE, REW_IGNORE, REW_SER_UNDE, REW_DESTRUCT, REW_GENOCIDE,
		REW_MASS_GEN, REW_MASS_GEN, REW_HEAL_FUL, REW_GAIN_ABL, REW_GAIN_ABL,
		REW_CHAOS_WP, REW_GOOD_OBS, REW_GOOD_OBS, REW_AUGM_ABL, REW_AUGM_ABL
	},

	/* Xiombarg the Sword-Queen: */
	{
		REW_TY_CURSE, REW_TY_CURSE, REW_PISS_OFF, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_WND, REW_POLY_WND,
		REW_GENOCIDE, REW_DISPEL_C, REW_GOOD_OBJ, REW_GOOD_OBJ, REW_SER_MONS,
		REW_GAIN_ABL, REW_CHAOS_WP, REW_GAIN_EXP, REW_AUGM_ABL, REW_GOOD_OBS
	},


	/* Pyaray the Tentacled Whisperer of Impossible Secretes: */
	{
		REW_WRATH, REW_TY_CURSE, REW_PISS_OFF, REW_H_SUMMON, REW_H_SUMMON,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_POLY_SLF,
		REW_POLY_SLF, REW_SER_DEMO, REW_HEAL_FUL, REW_GAIN_ABL, REW_GAIN_ABL,
		REW_CHAOS_WP, REW_DO_HAVOC, REW_GOOD_OBJ, REW_GREA_OBJ, REW_GREA_OBS
	},

	/* Balaan the Grim: */
	{
		REW_TY_CURSE, REW_HURT_LOT, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL,
		REW_SUMMON_M, REW_LOSE_EXP, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_WND,
		REW_SER_UNDE, REW_HEAL_FUL, REW_HEAL_FUL, REW_GAIN_EXP, REW_GAIN_EXP,
		REW_CHAOS_WP, REW_GOOD_OBJ, REW_GOOD_OBS, REW_GREA_OBS, REW_AUGM_ABL
	},

	/* Arioch, Duke of Hell: */
	{
		REW_WRATH, REW_PISS_OFF, REW_RUIN_ABL, REW_LOSE_EXP, REW_H_SUMMON,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_SLF,
		REW_POLY_SLF, REW_MASS_GEN, REW_SER_DEMO, REW_HEAL_FUL, REW_CHAOS_WP,
		REW_CHAOS_WP, REW_GOOD_OBJ, REW_GAIN_EXP, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Eequor, Blue Lady of Dismay: */
	{
		REW_WRATH, REW_TY_CURSE, REW_PISS_OFF, REW_CURSE_WP, REW_RUIN_ABL,
		REW_IGNORE, REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_WND,
		REW_GOOD_OBJ, REW_GOOD_OBJ, REW_SER_MONS, REW_HEAL_FUL, REW_GAIN_EXP,
		REW_GAIN_ABL, REW_CHAOS_WP, REW_GOOD_OBS, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Narjhan, Lord of Beggars: */
	{
		REW_WRATH, REW_CURSE_AR, REW_CURSE_WP, REW_CURSE_WP, REW_CURSE_AR,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF,
		REW_POLY_WND, REW_HEAL_FUL, REW_HEAL_FUL, REW_GAIN_EXP, REW_AUGM_ABL,
		REW_GOOD_OBJ, REW_GOOD_OBJ, REW_CHAOS_WP, REW_GREA_OBJ, REW_GREA_OBS
	},

	/* Balo the Jester: */
	{
		REW_WRATH, REW_SER_DEMO, REW_CURSE_WP, REW_CURSE_AR, REW_LOSE_EXP,
		REW_GAIN_ABL, REW_LOSE_ABL, REW_POLY_WND, REW_POLY_SLF, REW_IGNORE,
		REW_DESTRUCT, REW_MASS_GEN, REW_CHAOS_WP, REW_GREA_OBJ, REW_HURT_LOT,
		REW_AUGM_ABL, REW_RUIN_ABL, REW_H_SUMMON, REW_GREA_OBS, REW_AUGM_ABL
	},

	/* Khorne the Bloodgod: */
	{
		REW_WRATH, REW_HURT_LOT, REW_HURT_LOT, REW_H_SUMMON, REW_H_SUMMON,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_SER_MONS, REW_SER_DEMO,
		REW_POLY_SLF, REW_POLY_WND, REW_HEAL_FUL, REW_GOOD_OBJ, REW_GOOD_OBJ,
		REW_CHAOS_WP, REW_GOOD_OBS, REW_GOOD_OBS, REW_GREA_OBJ, REW_GREA_OBS
	},

	/* Slaanesh: */
	{
		REW_WRATH, REW_PISS_OFF, REW_PISS_OFF, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_LOSE_EXP, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_SER_DEMO,
		REW_POLY_SLF, REW_HEAL_FUL, REW_HEAL_FUL, REW_GOOD_OBJ, REW_GAIN_EXP,
		REW_GAIN_EXP, REW_CHAOS_WP, REW_GAIN_ABL, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Nurgle: */
	{
		REW_WRATH, REW_PISS_OFF, REW_HURT_LOT, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_LOSE_EXP, REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_SLF,
		REW_POLY_SLF, REW_POLY_WND, REW_HEAL_FUL, REW_GOOD_OBJ, REW_GAIN_ABL,
		REW_GAIN_ABL, REW_SER_UNDE, REW_CHAOS_WP, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Tzeentch: */
	{
		REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_LOSE_EXP, REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_SLF,
		REW_POLY_SLF, REW_POLY_WND, REW_HEAL_FUL, REW_CHAOS_WP, REW_GREA_OBJ,
		REW_GAIN_ABL, REW_GAIN_ABL, REW_GAIN_EXP, REW_GAIN_EXP, REW_AUGM_ABL
	},

	/* Khaine: */
	{
		REW_WRATH, REW_HURT_LOT, REW_PISS_OFF, REW_LOSE_ABL, REW_LOSE_EXP,
		REW_IGNORE,   REW_IGNORE,   REW_DISPEL_C, REW_DO_HAVOC, REW_DO_HAVOC,
		REW_POLY_SLF, REW_POLY_SLF, REW_GAIN_EXP, REW_GAIN_ABL, REW_GAIN_ABL,
		REW_SER_MONS, REW_GOOD_OBJ, REW_CHAOS_WP, REW_GREA_OBJ, REW_GOOD_OBS
	}
};

/*!
 * @brief マーシャルアーツ打撃テーブル
 */
const martial_arts ma_blows[MAX_MA] =
{
#ifdef JP
	{ "%sを殴った。",                          1, 0, 1, 5, 0 },
	{ "%sを蹴った。",                           2, 0, 1, 7, 0 },
	{ "%sに正拳突きをくらわした。",                         3, 0, 1, 9, 0 },
	{ "%sに膝蹴りをくらわした。",             5, 5, 2, 4, MA_KNEE },
	{ "%sに肘打ちをくらわした。",            7, 5, 1, 12, 0 },
	{ "%sに体当りした。",                           9, 10, 2, 6, 0 },
	{ "%sを蹴った。",                           11, 10, 3, 6, MA_SLOW },
	{ "%sにアッパーをくらわした。",                       13, 12, 5, 5, 6 },
	{ "%sに二段蹴りをくらわした。",                    16, 15, 5, 6, 8 },
	{ "%sに猫爪撃をくらわした。",          20, 20, 5, 8, 0 },
	{ "%sに跳空脚をくらわした。",           24, 25, 6, 8, 10 },
	{ "%sに鷲爪襲をくらわした。",       28, 25, 7, 9, 0 },
	{ "%sに回し蹴りをくらわした。",         32, 30, 8, 10, 10 },
	{ "%sに鉄拳撃をくらわした。",          35, 35, 8, 11, 10 },
	{ "%sに飛空脚をくらわした。",         39, 35, 8, 12, 12 },
	{ "%sに昇龍拳をくらわした。",         43, 35, 9, 12, 16 },
	{ "%sに石破天驚拳をくらわした。",       48, 40, 10, 13, 18 },
#else
	{ "You punch %s.",                          1, 0, 1, 4, 0 },
	{ "You kick %s.",                           2, 0, 1, 6, 0 },
	{ "You strike %s.",                         3, 0, 1, 7, 0 },
	{ "You hit %s with your knee.",             5, 5, 2, 3, MA_KNEE },
	{ "You hit %s with your elbow.",            7, 5, 1, 8, 0 },
	{ "You butt %s.",                           9, 10, 2, 5, 0 },
	{ "You kick %s.",                           11, 10, 3, 4, MA_SLOW },
	{ "You uppercut %s.",                       13, 12, 4, 4, 6 },
	{ "You double-kick %s.",                    16, 15, 5, 4, 8 },
	{ "You hit %s with a Cat's Claw.",          20, 20, 5, 5, 0 },
	{ "You hit %s with a jump kick.",           25, 25, 5, 6, 10 },
	{ "You hit %s with an Eagle's Claw.",       29, 25, 6, 6, 0 },
	{ "You hit %s with a circle kick.",         33, 30, 6, 8, 10 },
	{ "You hit %s with an Iron Fist.",          37, 35, 8, 8, 10 },
	{ "You hit %s with a flying kick.",         41, 35, 8, 10, 12 },
	{ "You hit %s with a Dragon Fist.",         45, 35, 10, 10, 16 },
	{ "You hit %s with a Crushing Blow.",       48, 35, 10, 12, 18 },
#endif

};

/*!
 * @brief 修行僧のターンダメージ算出テーブル
 */
const int monk_ave_damage[PY_MAX_LEVEL+1][3] =
{
  {0, 0, 0},
  {249, 249, 249},
  {324, 324, 324},
  {382, 438, 382},
  {382, 439, 382},
  {390, 446, 390},
  {394, 473, 394},
  {425, 528, 425},
  {430, 535, 430},
  {505, 560, 435},
  {517, 575, 444},
  {566, 655, 474},
  {585, 713, 486},
  {653, 843, 527},
  {678, 890, 544},
  {703, 973, 558},
  {765, 1096, 596},
  {914, 1146, 614},
  {943, 1240, 629},
  {971, 1276, 643},
  {1018, 1350, 667},
  {1063, 1464, 688},
  {1099, 1515, 705},
  {1128, 1559, 721},
  {1153, 1640, 735},
  {1336, 1720, 757},
  {1387, 1789, 778},
  {1430, 1893, 794},
  {1610, 2199, 863},
  {1666, 2280, 885},
  {1713, 2401, 908},
  {1755, 2465, 925},
  {1909, 2730, 984},
  {2156, 2891, 1009},
  {2218, 2970, 1031},
  {2319, 3107, 1063},
  {2404, 3290, 1098},
  {2477, 3389, 1125},
  {2544, 3483, 1150},
  {2771, 3899, 1228},
  {2844, 3982, 1259},
  {3129, 4064, 1287},
  {3200, 4190, 1313},
  {3554, 4674, 1432},
  {3614, 4738, 1463},
  {3679, 4853, 1485},
  {3741, 4905, 1512},
  {3785, 4943, 1538},
  {4141, 5532, 1652},
  {4442, 5581, 1679},
  {4486, 5636, 1702},
};

/*!
 * @brief アイテムの価値記述テーブル /
 * Table of game-generated inscriptions (indexed by the defines in defines.h). -- RG
 */
const cptr game_inscriptions[] =
{
	NULL,            /* FEEL_NONE */
#ifdef JP
	"壊れている",    /* FEEL_BROKEN */
	"恐ろしい",      /* FEEL_TERRIBLE */
	"無価値",        /* FEEL_WORTHLESS */
	"呪われている",  /* FEEL_CURSED */
	"上質以上",      /* FEEL_UNCURSED */
	"並",            /* FEEL_AVERAGE */
	"上質",          /* FEEL_GOOD */
	"高級品",        /* FEEL_EXCELLENT */
	"特別製",        /* FEEL_SPECIAL */
#else
	"broken",        /* FEEL_BROKEN */
	"terrible",      /* FEEL_TERRIBLE */
	"worthless",     /* FEEL_WORTHLESS */
	"cursed",        /* FEEL_CURSED */
	"uncursed",      /* FEEL_UNCURSED */
	"average",       /* FEEL_AVERAGE */
	"good",          /* FEEL_GOOD */
	"excellent",     /* FEEL_EXCELLENT */
	"special",       /* FEEL_SPECIAL */
#endif

};

/*!
 * @brief 修行僧の構え能力テーブル
 */
const kamae kamae_shurui[MAX_KAMAE] =
{
#ifdef JP
	{"玄武", 25, ""},
	{"白虎", 30, ""},
	{"青竜", 35, ""},
	{"朱雀", 40, ""},
#else
	{"Genbu", 25, "(Black Tortoise) "},
	{"Byakko", 30, "(White Tiger) "},
	{"Seiryuu", 35, "(Blue Dragon) "},
	{"Suzaku", 40, "(Red Phoenix) "},
#endif
};

/*!
 * @brief 剣術家の構え能力テーブル
 */
const kamae kata_shurui[MAX_KATA] =
{
#ifdef JP
	{"居合", 25, ""},
	{"風塵", 30, ""},
	{"降鬼", 35, ""},
	{"無想", 40, ""},
#else
	{"Iai", 25, ""},
	{"Huujin", 30, ""},
	{"Kouki", 35, ""},
	{"Musou", 40, ""},
#endif
};

/*!
 * @brief 技能値到達表記テーブル
 */
const cptr exp_level_str[5]=
#ifdef JP
{"[初心者]", "[入門者]", "[熟練者]", "[エキスパート]", "[達人]"};
#else
{"[Unskilled]", "[Beginner]", "[Skilled]", "[Expert]", "[Master]"};
#endif

/*!
 * @brief 幻覚時の打撃記述テーブル / Weird melee attack types when hallucinating
 */
#ifdef JP
const cptr silly_attacks[MAX_SILLY_ATTACK] =
{
	"に小便をかけられた。",
	"があなたの回りを3回回ってワンと言った。",
	"にしばきまわされた。",
	"に靴をなめさせられた。",
	"にハナクソを飛ばされた。",
	"にジャン拳で攻撃された。",
	"があなたの頬を札束でしばいた。",
	"があなたの前でポージングをした。",
	"にアカンベーされた。",
	"に「神の国」発言の撤回を求められた。",
	"にメッ○ールを飲まされた。",
	"につっこみを入れられた。",
	"はあなたと踊った。",
	"に顔にらく書きをされた。",
	"に借金の返済をせまられた。",
	"にスカートをめくられた。",
	"はあなたの手相を占った。",
	"から役満を上がられた。",
	"から愛の告白をされた。",
	"はあなたを時給500円で雇った。",
	"はあなたの100の秘密について熱く語った。",
	"がニャーと鳴いた。",
	"はあなたに気をつけた。",
	"はあなたをポリゴン化させた。",
	"に少しかじられた。",
	"はアルテマの呪文を唱えた！",
	"はあなたのスパイクをブロックした。",
	"はスライド移動した。",
	"は昇龍拳コマンドの入力に失敗した。",
	"は拡散波動砲を発射した。",
	"はデスラー戦法をしかけた。",
	"にライダーキックで攻撃された。",
	"に二週間以内でビデオを人に見せないと死ぬ呪いをかけられた。",
	"はパルプンテを唱えた。",
	"はスーパーウルトラギャラクティカマグナムを放った。",
	"にしゃがみ小キックでハメられた。",
	"にジェットストリームアタックをかけられた。",
	"はあなたに卍固めをかけて「1、2、3、ダーッ！」と叫んだ。",
	"は「いくじなし！ばかばかばか！」といって駆け出した。",
	"が「ごらん、ルーベンスの絵だよ」と言って静かに目を閉じた。",
	"は言った。「変愚蛮怒、絶賛公開中！」",
};

/*!
 * @brief 幻覚時の打撃記述テーブル(フォーマットつき) / Weird melee attack types when hallucinating (%s for strfmt())
 */
const cptr silly_attacks2[MAX_SILLY_ATTACK] =
{
	"%sに小便をかけた。",
	"%sの回りを3回回ってワンと言った。",
	"%sをしばきまわした。",
	"%sに靴をなめさせた。",
	"%sにハナクソを飛ばした。",
	"%sをジャン拳で攻撃した。",
	"%sの頬を札束でしばいた。",
	"%sの前でポージングをした。",
	"%sにアカンベーした。",
	"%sに「神の国」発言の撤回を求めた。",
	"%sにメッ○ールを飲ませた。",
	"%sにつっこみを入れた。",
	"%sと踊った。",
	"%sの顔にらく書きをした。",
	"%sに借金の返済をせまった。",
	"%sのスカートをめくった。",
	"%sの手相を占った。",
	"%sから役満を上がった。",
	"%sに愛の告白をした。",
	"%sを時給500円で雇った。",
	"%sの100の秘密について熱く語った。",
	"ニャーと鳴いた。",
	"%sに気をつけた。",
	"%sをポリゴン化させた。",
	"%sを少しかじった。",
	"アルテマの呪文を唱えた！",
	"%sのスパイクをブロックした。",
	"スライド移動した。",
	"昇龍拳コマンドの入力に失敗した。",
	"%sに拡散波動砲を発射した。",
	"%sにデスラー戦法をしかけた。",
	"%sをライダーキックで攻撃した。",
	"%sに二週間以内でビデオを人に見せないと死ぬ呪いをかけた。",
	"パルプンテを唱えた。",
	"%sにスーパーウルトラギャラクティカマグナムを放った。",
	"%sをしゃがみ小キックでハメた。",
	"%sにジェットストリームアタックをかけた。",
	"%sに卍固めをかけて「1、2、3、ダーッ！」と叫んだ。",
	"「いくじなし！ばかばかばか！」といって駆け出した。",
	"「ごらん、ルーベンスの絵だよ」と言って静かに目を閉じた。",
	"言った。「変愚蛮怒、絶賛公開中！」",
};
#else
const cptr silly_attacks[MAX_SILLY_ATTACK] =
{
	"smothers",
	"hugs",
	"humiliates",
	"whips",
	"kisses",

	"disgusts",
	"pees all over",
	"passes the gas on",
	"makes obscene gestures at",
	"licks",

	"stomps on",
	"swallows",
	"drools on",
	"misses",
	"shrinks",

	"emasculates",
	"evaporates",
	"solidifies",
	"digitizes",
	"insta-kills",

	"massacres",
	"slaughters",
	"drugs",
	"psychoanalyzes",
	"deconstructs",

	"falsifies",
	"disbelieves",
	"molests",
	"pusupusu",
};
#endif


/*!
 * @brief シンボル解説テーブル /
 * The table of "symbol info" -- each entry is a string of the form "X:desc" where "X" is the trigger, and "desc" is the "info".
 */
const cptr ident_info[] =
{
#ifdef JP
	" :暗闇",
	"!:薬, オイル",
	"\":アミュレット, 頸飾り",
	"#:壁(隠しドア)/植物/気体",
	"$:財宝(金か宝石)",
	"%:鉱脈(溶岩か石英)",
	"&:箱",
	"':開いたドア",
	"(:軟らかい防具",
	"):盾",
	"*:財宝を含んだ鉱脈または球形の怪物",
	"+:閉じたドア",
	",:食べ物, おばけキノコ",
	"-:魔法棒, ロッド",
	".:床",
	"/:竿状武器(アックス/パイク/等)",
	"0:博物館の入口",
	"1:雑貨屋の入口",
	"2:防具屋の入口",
	"3:武器専門店の入口",
	"4:寺院の入口",
	"5:錬金術の店の入口",
	"6:魔法の店の入口",
	"7:ブラックマーケットの入口",
	"8:我が家の入口",
	"9:書店の入口",
	"::岩石",
	";:回避の彫像/爆発のルーン",
	"<:上り階段",
	"=:指輪",
	">:下り階段",
	"?:巻物",
	"@:プレイヤー",
	"A:天使",
	"B:鳥",
	"C:犬",
	"D:古代ドラゴン/ワイアーム",
	"E:エレメンタル",
	"F:トンボ",
	"G:ゴースト",
	"H:雑種",
	"I:昆虫",
	"J:ヘビ",
	"K:キラー・ビートル",
	"L:リッチ",
	"M:多首の爬虫類",
	"N:謎の生物",
	"O:オーガ",
	"P:巨大人間型生物",
	"Q:クイルスルグ(脈打つ肉塊)",
	"R:爬虫類/両生類",
	"S:蜘蛛/サソリ/ダニ",
	"T:トロル",
	"U:上級デーモン",
	"V:バンパイア",
	"W:ワイト/レイス/等",
	"X:ゾーン/ザレン/等",
	"Y:イエティ",
	"Z:ハウンド",
	"[:堅いアーマー",
	"\\:鈍器(メイス/ムチ/等)",
	"]:種々の防具",
	"^:トラップ",
	"_:杖",
	"`:人形，彫像",
	"a:アリ",
	"b:コウモリ",
	"c:ムカデ",
	"d:ドラゴン",
	"e:目玉",
	"f:ネコ",
	"g:ゴーレム",
	"h:ホビット/エルフ/ドワーフ",
	"i:ベトベト",
	"j:ゼリー",
	"k:コボルド",
	"l:水棲生物",
	"m:モルド",
	"n:ナーガ",
	"o:オーク",
	"p:人間",
	"q:四足獣",
	"r:ネズミ",
	"s:スケルトン",
	"t:町の人",
	"u:下級デーモン",
	"v:ボルテックス",
	"w:イモムシ/大群",
	/* "x:unused", */
	"y:イーク",
	"z:ゾンビ/ミイラ",
	"{:飛び道具の弾(矢/弾)",
	"|:刀剣類(ソード/ダガー/等)",
	"}:飛び道具(弓/クロスボウ/スリング)",
	"~:水/溶岩流(種々のアイテム)",
#else
	" :A dark grid",
	"!:A potion (or oil)",
	"\":An amulet (or necklace)",
	"#:A wall (or secret door) / a plant / a gas",
	"$:Treasure (gold or gems)",
	"%:A vein (magma or quartz)",
	"&:A chest",
	"':An open door",
	"(:Soft armor",
	"):A shield",
	"*:A vein with treasure or a ball monster",
	"+:A closed door",
	",:Food (or mushroom patch)",
	"-:A wand (or rod)",
	".:Floor",
	"/:A polearm (Axe/Pike/etc)",
	"0:Entrance to Museum",
	"1:Entrance to General Store",
	"2:Entrance to Armory",
	"3:Entrance to Weaponsmith",
	"4:Entrance to Temple",
	"5:Entrance to Alchemy shop",
	"6:Entrance to Magic store",
	"7:Entrance to Black Market",
	"8:Entrance to your home",
	"9:Entrance to the bookstore",
	"::Rubble",
	";:A glyph of warding / an explosive rune",
	"<:An up staircase",
	"=:A ring",
	">:A down staircase",
	"?:A scroll",
	"@:You",
	"A:Angel",
	"B:Bird",
	"C:Canine",
	"D:Ancient Dragon/Wyrm",
	"E:Elemental",
	"F:Dragon Fly",
	"G:Ghost",
	"H:Hybrid",
	"I:Insect",
	"J:Snake",
	"K:Killer Beetle",
	"L:Lich",
	"M:Multi-Headed Reptile",
	"N:Mystery Living",
	"O:Ogre",
	"P:Giant Humanoid",
	"Q:Quylthulg (Pulsing Flesh Mound)",
	"R:Reptile/Amphibian",
	"S:Spider/Scorpion/Tick",
	"T:Troll",
	"U:Major Demon",
	"V:Vampire",
	"W:Wight/Wraith/etc",
	"X:Xorn/Xaren/etc",
	"Y:Yeti",
	"Z:Zephyr Hound",
	"[:Hard armor",
	"\\:A hafted weapon (mace/whip/etc)",
	"]:Misc. armor",
	"^:A trap",
	"_:A staff",
	"`:A figurine or statue",
	"a:Ant",
	"b:Bat",
	"c:Centipede",
	"d:Dragon",
	"e:Floating Eye",
	"f:Feline",
	"g:Golem",
	"h:Hobbit/Elf/Dwarf",
	"i:Icky Thing",
	"j:Jelly",
	"k:Kobold",
	"l:Aquatic monster",
	"m:Mold",
	"n:Naga",
	"o:Orc",
	"p:Person/Human",
	"q:Quadruped",
	"r:Rodent",
	"s:Skeleton",
	"t:Townsperson",
	"u:Minor Demon",
	"v:Vortex",
	"w:Worm/Worm-Mass",
	/* "x:unused", */
	"y:Yeek",
	"z:Zombie/Mummy",
	"{:A missile (arrow/bolt/shot)",
	"|:An edged weapon (sword/dagger/etc)",
	"}:A launcher (bow/crossbow/sling)",
	"~:Fluid terrain (or miscellaneous item)",
#endif

	NULL
};

/*!
 * @brief モンスターの打撃効力テーブル /
 * The table of monsters' blow effects
 */
const mbe_info_type mbe_info[] =
{
	{  0, 0,             }, /* None      */
	{ 60, GF_MISSILE,    }, /* HURT      */
	{  5, GF_POIS,       }, /* POISON    */
	{ 20, GF_DISENCHANT, }, /* UN_BONUS  */
	{ 15, GF_MISSILE,    }, /* UN_POWER  */ /* ToDo: Apply the correct effects */
	{  5, GF_MISSILE,    }, /* EAT_GOLD  */
	{  5, GF_MISSILE,    }, /* EAT_ITEM  */
	{  5, GF_MISSILE,    }, /* EAT_FOOD  */
	{  5, GF_MISSILE,    }, /* EAT_LITE  */
	{  0, GF_ACID,       }, /* ACID      */
	{ 10, GF_ELEC,       }, /* ELEC      */
	{ 10, GF_FIRE,       }, /* FIRE      */
	{ 10, GF_COLD,       }, /* COLD      */
	{  2, GF_MISSILE,    }, /* BLIND     */
	{ 10, GF_CONFUSION,  }, /* CONFUSE   */
	{ 10, GF_MISSILE,    }, /* TERRIFY   */
	{  2, GF_MISSILE,    }, /* PARALYZE  */
	{  0, GF_MISSILE,    }, /* LOSE_STR  */
	{  0, GF_MISSILE,    }, /* LOSE_INT  */
	{  0, GF_MISSILE,    }, /* LOSE_WIS  */
	{  0, GF_MISSILE,    }, /* LOSE_DEX  */
	{  0, GF_MISSILE,    }, /* LOSE_CON  */
	{  0, GF_MISSILE,    }, /* LOSE_CHR  */
	{  2, GF_MISSILE,    }, /* LOSE_ALL  */
	{ 60, GF_ROCKET,     }, /* SHATTER   */
	{  5, GF_MISSILE,    }, /* EXP_10    */
	{  5, GF_MISSILE,    }, /* EXP_20    */
	{  5, GF_MISSILE,    }, /* EXP_40    */
	{  5, GF_MISSILE,    }, /* EXP_80    */
	{  5, GF_POIS,       }, /* DISEASE   */
	{  5, GF_TIME,       }, /* TIME      */
	{  5, GF_MISSILE,    }, /* EXP_VAMP  */
	{  5, GF_MANA,       }, /* DR_MANA   */
	{ 60, GF_MISSILE,    }, /* SUPERHURT */
};


/*!
 * @brief 地形状態フラグテーブル /
 * The table of features' actions
 */
const byte feature_action_flags[FF_FLAG_MAX] =
{
	0, /* LOS */
	0, /* PROJECT */
	0, /* MOVE */
	0, /* PLACE */
	0, /* DROP */
	0, /* SECRET */
	0, /* NOTICE */
	0, /* REMEMBER */
	0, /* OPEN */
	0, /* CLOSE */
	FAF_CRASH_GLASS, /* BASH */
	0, /* SPIKE */
	FAF_DESTROY, /* DISARM */
	0, /* STORE */
	FAF_DESTROY | FAF_CRASH_GLASS, /* TUNNEL */
	0, /* MAY_HAVE_GOLD */
	0, /* HAS_GOLD */
	0, /* HAS_ITEM */
	0, /* DOOR */
	0, /* TRAP */
	0, /* STAIRS */
	0, /* GLYPH */
	0, /* LESS */
	0, /* MORE */
	0, /* RUN */
	0, /* FLOOR */
	0, /* WALL */
	0, /* PERMANENT */
	0, /* INNER */
	0, /* OUTER */
	0, /* SOLID */
	0, /* HIT_TRAP */

	0, /* BRIDGE */
	0, /* RIVER */
	0, /* LAKE */
	0, /* BRIDGED */
	0, /* COVERED */
	0, /* GLOW */
	0, /* ENSECRET */
	0, /* WATER */
	0, /* LAVA */
	0, /* SHALLOW */
	0, /* DEEP */
	0, /* FILLED */
	FAF_DESTROY | FAF_CRASH_GLASS, /* HURT_ROCK */
	0, /* HURT_FIRE */
	0, /* HURT_COLD */
	0, /* HURT_ACID */
	0, /* ICE */
	0, /* ACID */
	0, /* OIL */
	0, /* XXX04 */
	0, /* CAN_CLIMB */
	0, /* CAN_FLY */
	0, /* CAN_SWIM */
	0, /* CAN_PASS */
	0, /* CAN_OOZE */
	0, /* CAN_DIG */
	0, /* HIDE_ITEM */
	0, /* HIDE_SNEAK */
	0, /* HIDE_SWIM */
	0, /* HIDE_DIG */
	0, /* KILL_HUGE */
	0, /* KILL_MOVE */

	0, /* PICK_TRAP */
	0, /* PICK_DOOR */
	0, /* ALLOC */
	0, /* CHEST */
	0, /* DROP_1D2 */
	0, /* DROP_2D2 */
	0, /* DROP_GOOD */
	0, /* DROP_GREAT */
	0, /* HURT_POIS */
	0, /* HURT_ELEC */
	0, /* HURT_WATER */
	0, /* HURT_BWATER */
	0, /* USE_FEAT */
	0, /* GET_FEAT */
	0, /* GROUND */
	0, /* OUTSIDE */
	0, /* EASY_HIDE */
	0, /* EASY_CLIMB */
	0, /* MUST_CLIMB */
	0, /* TREE */
	0, /* NEED_TREE */
	0, /* BLOOD */
	0, /* DUST */
	0, /* SLIME */
	0, /* PLANT */
	0, /* XXX2 */
	0, /* INSTANT */
	0, /* EXPLODE */
	0, /* TIMED */
	0, /* ERUPT */
	0, /* STRIKE */
	0, /* SPREAD */

	0, /* SPECIAL */
	FAF_DESTROY | FAF_NO_DROP | FAF_CRASH_GLASS, /* HURT_DISI */
	0, /* QUEST_ENTER */
	0, /* QUEST_EXIT */
	0, /* QUEST */
	0, /* SHAFT */
	0, /* MOUNTAIN */
	0, /* BLDG */
	0, /* MINOR_GLYPH */
	0, /* PATTERN */
	0, /* TOWN */
	0, /* ENTRANCE */
	0, /* MIRROR */
	0, /* UNPERM */
	0, /* TELEPORTABLE */
	0, /* CONVERT */
	0, /* GLASS */
};


/*!
 * @brief 装備耐性に準じたブレス効果の選択テーブル /
 * Define flags, effect type, name for dragon breath activation
 */
const dragonbreath_type dragonbreath_info[] = {
	{ TR_RES_ACID, GF_ACID, _("酸", "acid") },
	{ TR_RES_ELEC, GF_ELEC, _("電撃", "lightning") },
	{ TR_RES_FIRE, GF_FIRE, _("火炎", "fire") },
	{ TR_RES_COLD, GF_COLD, _("冷気", "cold") },
	{ TR_RES_POIS, GF_POIS, _("毒", "poison") },
	{ TR_RES_LITE, GF_LITE, _("閃光", "light") },
	{ TR_RES_DARK, GF_DARK, _("暗黒", "dark") },
	{ TR_RES_SHARDS, GF_SHARDS, _("破片", "shard") },
	{ TR_RES_CONF, GF_CONFUSION, _("混乱", "confusion") },
	{ TR_RES_SOUND, GF_SOUND, _("轟音", "sound") },
	{ TR_RES_NEXUS, GF_NEXUS, _("因果混乱", "nexus") },
	{ TR_RES_NETHER, GF_NETHER, _("地獄", "nether") },
	{ TR_RES_CHAOS, GF_CHAOS, _("カオス", "chaos") },
	{ TR_RES_DISEN, GF_DISENCHANT, _("劣化", "disenchant") },
	{ 0, 0, NULL }
};

/*!
 * @brief アイテムの発動効果テーブル /
 * Define flags, levels, values of activations
 */
const activation_type activation_info[] =
{
	{ "SUNLIGHT", ACT_SUNLIGHT, 10, 250, {10, 0},
	  _("太陽光線", "beam of sunlight") },
	{ "BO_MISS_1", ACT_BO_MISS_1, 10, 250, {2, 0},
	  _("マジック・ミサイル(2d6)", "magic missile (2d6)") },
	{ "BA_POIS_1", ACT_BA_POIS_1, 10, 300, {4, 0},
	  _("悪臭雲(12)", "stinking cloud (12)") },
	{ "BO_ELEC_1", ACT_BO_ELEC_1, 20, 250, {5, 0},
	  _("サンダー・ボルト(4d8)", "lightning bolt (4d8)") },
	{ "BO_ACID_1", ACT_BO_ACID_1, 20, 250, {6, 0},
	  _("アシッド・ボルト(5d8)", "acid bolt (5d8)") },
	{ "BO_COLD_1", ACT_BO_COLD_1, 20, 250, {7, 0},
	  _("アイス・ボルト(6d8)", "frost bolt (6d8)") },
	{ "BO_FIRE_1", ACT_BO_FIRE_1, 20, 250, {8, 0},
	  _("ファイア・ボルト(9d8)", "fire bolt (9d8)") },
	{ "BA_COLD_1", ACT_BA_COLD_1, 30, 750, {6, 0},
	  _("アイス・ボール(48)", "ball of cold (48)") },
	{ "BA_COLD_2", ACT_BA_COLD_2, 40, 1000, {12, 0},
	  _("アイス・ボール(100)", "ball of cold (100)") },
	{ "BA_COLD_3", ACT_BA_COLD_3, 70, 2500, {50, 0},
	  _("巨大アイス・ボール(400)", "ball of cold (400)") },
	{ "BA_FIRE_1", ACT_BA_FIRE_1, 30, 1000, {9, 0},
	  _("ファイア・ボール(72)", "ball of fire (72)") },
	{ "BA_FIRE_2", ACT_BA_FIRE_2, 40, 1500, {15, 0},
	  _("巨大ファイア・ボール(120)", "large fire ball (120)") },
	{ "BA_FIRE_3", ACT_BA_FIRE_3, 60, 1750, {40, 0},
	  _("巨大ファイア・ボール(300)", "fire ball (300)") },
	{ "BA_FIRE_4", ACT_BA_FIRE_4, 40, 1000, {12, 0},
	  _("ファイア・ボール(100)", "fire ball (100)") },
	{ "BA_ELEC_2", ACT_BA_ELEC_2, 40, 1000, {12, 0},
	  _("サンダー・ボール(100)", "ball of lightning (100)") },
	{ "BA_ELEC_3", ACT_BA_ELEC_3, 70, 2500, {70, 0},
	  _("巨大サンダー・ボール(500)", "ball of lightning (500)") },
	{ "BA_ACID_1", ACT_BA_ACID_1, 30, 1000, {12, 0},
	  _("アシッド・ボール(100)", "ball of acid (100)") },
	{ "BA_NUKE_1", ACT_BA_NUKE_1, 50, 1000, {12, 0},
	  _("放射能球(100)", "ball of nuke (100)") },
	{ "DRAIN_1", ACT_DRAIN_1, 30, 500, {12, 0},
	  _("窒息攻撃(100)", "a strangling attack (100)") },
	{ "DRAIN_2", ACT_DRAIN_2, 40, 750, {15, 0},
	  _("生命力吸収(120)", "drain life (120)") },
	{ "VAMPIRE_1", ACT_VAMPIRE_1, 40, 1000, {20, 0},
	  _("吸血ドレイン(3*50)", "vampiric drain (3*50)") },
	{ "BO_MISS_2", ACT_BO_MISS_2, 40, 1000, {20, 0},
	  _("矢(150)", "arrows (150)") },
	{ "WHIRLWIND", ACT_WHIRLWIND, 50, 7500, {25, 0},
	  _("カマイタチ", "whirlwind attack") },
	{ "VAMPIRE_2", ACT_VAMPIRE_2, 50, 2500, {40, 0},
	  _("吸血ドレイン(3*100)", "vampiric drain (3*100)") },
	{ "CALL_CHAOS", ACT_CALL_CHAOS, 70, 5000, {35, 0},
	  _("混沌召来", "call chaos") },
	{ "ROCKET", ACT_ROCKET, 70, 5000, {20, 0},
	  _("ロケット(120+レベル)", "launch rocket (120+level)") },
	{ "DISP_EVIL", ACT_DISP_EVIL, 50, 4000, {50, 0},
	  _("邪悪退散(x5)", "dispel evil (x5)") },
	{ "BA_MISS_3", ACT_BA_MISS_3, 50, 1500, {50, 0},
	  _("エレメントのブレス(300)", "elemental breath (300)") },
	{ "DISP_GOOD", ACT_DISP_GOOD, 50, 3500, {50, 0},
	  _("善良退散(x5)", "dispel good (x5)") },
	{ "BO_MANA", ACT_BO_MANA, 40, 1500, {20, 0},
	  _("魔法の矢(150)", "a magical arrow (150)") },
	{ "BA_WATER", ACT_BA_WATER, 50, 2000, {25, 0},
	  _("ウォーター・ボール(200)", "water ball (200)") },
	{ "BA_STAR", ACT_BA_STAR, 50, 2200, {25, 0},
	  _("巨大スター・ボール(200)", "large star ball (200)") },
	{ "BA_DARK", ACT_BA_DARK, 50, 2200, {30, 0},
	  _("暗黒の嵐(250)", "darkness storm (250)") },
	{ "BA_MANA", ACT_BA_MANA, 70, 2500, {30, 0},
	  _("魔力の嵐(250)", "a mana storm (250)") },
	{ "PESTICIDE", ACT_PESTICIDE, 10, 500, {10, 0},
	  _("害虫の駆除", "dispel small life") },
	{ "BLINDING_LIGHT", ACT_BLINDING_LIGHT, 30, 5000, {40, 0},
	  _("眩しい光", "blinding light") },
	{ "BIZARRE", ACT_BIZARRE, 90, 10000, {50, 0},
	  _("信じ難いこと", "bizarre things") },
	{ "CAST_BA_STAR", ACT_CAST_BA_STAR, 70, 7500, {100, 0},
	  _("スター・ボール・ダスト(150)", "cast star balls (150)") },
	{ "BLADETURNER", ACT_BLADETURNER, 80, 20000, {80, 0},
	  _("エレメントのブレス(300), 士気高揚、祝福、耐性", "breathe elements (300), hero, bless, and resistance") },
	{ "BR_FIRE", ACT_BR_FIRE, 50, 5000, {-1, 0},
	  _("火炎のブレス (200)", "fire breath (200)") },
	{ "BR_COLD", ACT_BR_COLD, 50, 5000, {-1, 0},
	  _("冷気のブレス (200)", "cold breath (200)") },
	{ "BR_DRAGON", ACT_BR_DRAGON, 70, 10000, { 30, 0 },
	  "" /* built by item_activation_dragon_breath() */ },

	{ "CONFUSE", ACT_CONFUSE, 10, 500, {10, 0},
	  _("パニック・モンスター", "confuse monster") },
	{ "SLEEP", ACT_SLEEP, 10, 750, {15, 0},
	  _("周囲のモンスターを眠らせる", "sleep nearby monsters") },
	{ "QUAKE", ACT_QUAKE, 30, 600, {20, 0},
	  _("地震", "earthquake") },
	{ "TERROR", ACT_TERROR, 20, 2500, {-1, 0},
	  _("恐慌", "terror") },
	{ "TELE_AWAY", ACT_TELE_AWAY, 20, 2000, {15, 0},
	  _("テレポート・アウェイ", "teleport away") },
	{ "BANISH_EVIL", ACT_BANISH_EVIL, 40, 2000, {250, 0},
	  _("邪悪消滅", "banish evil") },
	{ "GENOCIDE", ACT_GENOCIDE, 50, 10000, {500, 0},
	  _("抹殺", "genocide") },
	{ "MASS_GENO", ACT_MASS_GENO, 50, 10000, {1000, 0},
	  _("周辺抹殺", "mass genocide") },
	{ "SCARE_AREA", ACT_SCARE_AREA, 20, 2500, {20, 0},
	  _("モンスター恐慌", "frighten monsters") },
	{ "AGGRAVATE", ACT_AGGRAVATE, 0, 100, {0, 0},
	  _("モンスターを怒らせる", "aggravete monsters") },

	{ "CHARM_ANIMAL", ACT_CHARM_ANIMAL, 40, 7500, {200, 0},
	  _("動物魅了", "charm animal") },
	{ "CHARM_UNDEAD", ACT_CHARM_UNDEAD, 40, 10000, {333, 0},
	  _("アンデッド従属", "enslave undead") },
	{ "CHARM_OTHER", ACT_CHARM_OTHER, 40, 10000, {400, 0},
	  _("モンスター魅了", "charm monster") },
	{ "CHARM_ANIMALS", ACT_CHARM_ANIMALS, 40, 12500, {500, 0},
	  _("動物友和", "animal friendship") },
	{ "CHARM_OTHERS", ACT_CHARM_OTHERS, 40, 17500, {750, 0},
	  _("周辺魅了", "mass charm") },
	{ "SUMMON_ANIMAL", ACT_SUMMON_ANIMAL, 50, 10000, {200, 300},
	  _("動物召喚", "summon animal") },
	{ "SUMMON_PHANTOM", ACT_SUMMON_PHANTOM, 50, 12000, {200, 200},
	  _("幻霊召喚", "summon phantasmal servant") },
	{ "SUMMON_ELEMENTAL", ACT_SUMMON_ELEMENTAL, 50, 15000, {750, 0},
	  _("エレメンタル召喚", "summon elemental") },
	{ "SUMMON_DEMON", ACT_SUMMON_DEMON, 50, 20000, {666, 0},
	  _("悪魔召喚", "summon demon") },
	{ "SUMMON_UNDEAD", ACT_SUMMON_UNDEAD, 50, 20000, {666, 0},
	  _("アンデッド召喚", "summon undead") },
	{ "SUMMON_HOUND", ACT_SUMMON_HOUND, 50, 15000, {300, 0},
	  _("ハウンド召喚", "summon hound") },
	{ "SUMMON_DAWN", ACT_SUMMON_DAWN, 50, 15000, {500, 0},
	  _("暁の師団召喚", "summon the Legion of the Dawn") },
	{ "SUMMON_OCTOPUS", ACT_SUMMON_OCTOPUS, 50, 15000, {300, 0},
	  _("蛸の大群召喚", "summon octopus") },

	{ "CHOIR_SINGS", ACT_CHOIR_SINGS, 60, 20000, {300, 0},
	  _("回復(777)、癒し、士気高揚", "heal 777 hit points, curing and heloism") },
	{ "CURE_LW", ACT_CURE_LW, 10, 500, {10, 0},
	  _("恐怖除去/体力回復(30)", "remove fear and heal 30 hp") },
	{ "CURE_MW", ACT_CURE_MW, 20, 750, {3, 3},
	  _("傷回復(4d8)", "heal 4d8 and wounds") },
	{ "CURE_POISON", ACT_CURE_POISON, 10, 1000, {5, 0},
	  _("恐怖除去/毒消し", "remove fear and cure poison") },
	{ "REST_LIFE", ACT_REST_EXP, 40, 7500, {450, 0},
	  _("経験値復活", "restore experience") },
	{ "REST_ALL", ACT_REST_ALL, 30, 15000, {750, 0},
	  _("全ステータスと経験値復活", "restore stats and experience") },
	{ "CURE_700", ACT_CURE_700, 40, 10000, {250, 0},
	  _("体力回復(700)", "heal 700 hit points") },
	{ "CURE_1000", ACT_CURE_1000, 50, 15000, {888, 0},
	  _("体力回復(1000)", "heal 1000 hit points") },
	{ "CURING", ACT_CURING, 30, 5000, {100, 0},
	  _("癒し", "curing") },
	{ "CURE_MANA_FULL", ACT_CURE_MANA_FULL, 60, 20000, {777, 0},
	  _("魔力復活", "restore mana") },

	{ "ESP", ACT_ESP, 30, 1500, {100, 0},
	  _("テレパシー(期間 25+d30)", "telepathy (dur 25+d30)") },
	{ "BERSERK", ACT_BERSERK, 10, 800, {75, 0},
	  _("狂戦士化(25+d25ターン)", "berserk (25+d25 turns)") },
	{ "PROT_EVIL", ACT_PROT_EVIL, 30, 5000, {100, 0},
	  _("対邪悪結界(期間 3*レベル+d25)", "protect evil (dur level*3 + d25)") },
	{ "RESIST_ALL", ACT_RESIST_ALL, 30, 5000, {111, 0},
	  _("全耐性(期間 20+d20)", "resist elements (dur 20+d20)") },
	{ "SPEED", ACT_SPEED, 40, 15000, {250, 0},
	  _("加速(期間 20+d20)", "speed (dur 20+d20)") },
	{ "XTRA_SPEED", ACT_XTRA_SPEED, 40, 25000, {200, 200},
	  _("加速(期間 75+d75)", "speed (dur 75+d75)") },
	{ "WRAITH", ACT_WRAITH, 90, 25000, {1000, 0},
	  _("幽体化(期間 (レベル/2)+d(レベル/2))", "wraith form (dur level/2 + d(level/2))") },
	{ "INVULN", ACT_INVULN, 90, 25000, {1000, 0},
	  _("無敵化(期間 8+d8)", "invulnerability (dur 8+d8)") },
	{ "HELO", ACT_HELO, 10, 500, {30, 30},
	  _("士気高揚", "heroism") },
	{ "HELO_SPEED", ACT_HELO_SPEED, 30, 20000, {100, 200},
	  _("士気高揚, スピード(期間 50+d50ターン)", "hero and +10 to speed (50)") },
	{ "RESIST_ACID", ACT_RESIST_ACID, 20, 2000, {40, 40},
	  _("酸への耐性(期間 20+d20)", "resist acid (dur 20+d20)") },
	{ "RESIST_FIRE", ACT_RESIST_FIRE, 20, 2000, {40, 40},
	  _("火炎への耐性(期間 20+d20)", "resist fire (dur 20+d20)") },
	{ "RESIST_COLD", ACT_RESIST_COLD, 20, 2000, {40, 40},
	  _("冷気への耐性(期間 20+d20)", "resist cold (dur 20+d20)") },
	{ "RESIST_ELEC", ACT_RESIST_ELEC, 20, 2000, {40, 40},
	  _("電撃への耐性(期間 20+d20)", "resist elec (dur 20+d20)") },
	{ "RESIST_POIS", ACT_RESIST_POIS, 20, 2000, {40, 40},
	  _("毒への耐性(期間 20+d20)", "resist poison (dur 20+d20)") },

	{ "LIGHT", ACT_LIGHT, 10, 150, {10, 10},
	  _("イルミネーション", "light area (dam 2d15)") },
	{ "MAP_LIGHT", ACT_MAP_LIGHT, 30, 500, {50, 50},
	  _("魔法の地図と光", "light (dam 2d15) & map area") },
	{ "DETECT_ALL", ACT_DETECT_ALL, 30, 1000, {55, 55},
	  _("全感知", "detection") },
	{ "DETECT_XTRA", ACT_DETECT_XTRA, 50, 12500, {100, 0},
	  _("全感知、探索、*鑑定*", "detection, probing and identify true") },
	{ "ID_FULL", ACT_ID_FULL, 50, 10000, {75, 0},
	  _("*鑑定*", "identify true") },
	{ "ID_PLAIN", ACT_ID_PLAIN, 20, 1250, {10, 0},
	  _("鑑定", "identify spell") },
	{ "RUNE_EXPLO", ACT_RUNE_EXPLO, 40, 4000, {200, 0},
	  _("爆発のルーン", "explosive rune") },
	{ "RUNE_PROT", ACT_RUNE_PROT, 60, 10000, {400, 0},
	  _("守りのルーン", "rune of protection") },
	{ "SATIATE", ACT_SATIATE, 10, 2000, {200, 0},
	  _("空腹充足", "satisfy hunger") },
	{ "DEST_DOOR", ACT_DEST_DOOR, 10, 100, {10, 0},
	  _("ドア破壊", "destroy doors") },
	{ "STONE_MUD", ACT_STONE_MUD, 20, 1000, {3, 0},
	  _("岩石溶解", "stone to mud") },
	{ "RECHARGE", ACT_RECHARGE, 30, 1000, {70, 0},
	  _("魔力充填", "recharging") },
	{ "ALCHEMY", ACT_ALCHEMY, 50, 10000, {500, 0},
	  _("錬金術", "alchemy") },
	{ "DIM_DOOR", ACT_DIM_DOOR, 50, 10000, {100, 0},
	  _("次元の扉", "dimension door") },
	{ "TELEPORT", ACT_TELEPORT, 10, 2000, {25, 0},
	  _("テレポート", "teleport") },
	{ "RECALL", ACT_RECALL, 30, 7500, {200, 0},
	  _("帰還の詔", "word of recall") },
	{ "JUDGE", ACT_JUDGE, 90, 50000, {20, 20},
	  _("体力と引き替えに千里眼と帰還", "a telekinesis (500 lb)") },
	{ "TELEKINESIS", ACT_TELEKINESIS, 20, 5500, {25, 25},
	  _("物体を引き寄せる(重量25kgまで)", "clairvoyance and recall, draining you") },
	{ "DETECT_UNIQUE", ACT_DETECT_UNIQUE, 40, 10000, {200, 0},
	  _("この階にいるユニークモンスターを表示", "list of the uniques on the level") },
	{ "ESCAPE", ACT_ESCAPE, 10, 3000, {35, 0},
	  _("逃走", "a getaway") },
	{ "DISP_CURSE_XTRA", ACT_DISP_CURSE_XTRA, 40, 30000, {0, 0},
	  _("*解呪*と調査", "dispel curse and probing") },
	{ "BRAND_FIRE_BOLTS", ACT_BRAND_FIRE_BOLTS, 40, 20000, {999, 0},
	  _("刃先のファイア・ボルト", "fire branding of bolts") },
	{ "RECHARGE_XTRA", ACT_RECHARGE_XTRA, 70, 30000, {200, 0},
	  _("魔力充填", "recharge item") },
	{ "LORE", ACT_LORE, 10, 30000, {0, 0},
	  _("危険を伴う鑑定", "perilous identify") },
	{ "SHIKOFUMI", ACT_SHIKOFUMI, 10, 10000, {100, 100},
	  _("四股踏み", "shiko") },
	{ "PHASE_DOOR", ACT_PHASE_DOOR, 10, 1500, {10, 0},
	  _("ショート・テレポート", "blink") },
	{ "DETECT_ALL_MONS", ACT_DETECT_ALL_MONS, 30, 3000, {150, 0},
	  _("全モンスター感知", "detect all monsters") },
	{ "ULTIMATE_RESIST", ACT_ULTIMATE_RESIST, 90, 20000, {777, 0},
	  _("士気高揚、祝福、究極の耐性", "hero, bless, and ultimate resistance") },

	{ "CAST_OFF", ACT_CAST_OFF, 30, 15000, {100, 0},
	  _("脱衣と小宇宙燃焼", "cast it off and cosmic heroism") },
	{ "FISHING", ACT_FISHING, 0, 100, {0, 0},
	  _("釣りをする", "fishing") },
	{ "INROU", ACT_INROU, 40, 15000, {150, 150},
	  _("例のアレ", "reveal your identity") },
	{ "MURAMASA", ACT_MURAMASA, 0, 0, {-1, 0},
	  _("腕力の上昇", "increase STR") },
	{ "BLOODY_MOON", ACT_BLOODY_MOON, 0, 0, {3333, 0},
	  _("属性変更", "change zokusei") },
	{ "CRIMSON", ACT_CRIMSON, 0, 50000, {15, 0},
	  _("ファイア！", "fire!") },

	{ "STRAIN_HASTE", ACT_STRAIN_HASTE, 10, 1000, {120, 100},
	  _("体力と引き換えに加速", "haste with strain") },
	{ "GRAND_CROSS", ACT_GRAND_CROSS, 30, 15000, {250, 200},
	  _("グランド・クロス", "grand cross") },
	{ "TELEPORT_LEVEL", ACT_TELEPORT_LEVEL, 10, 1500, {100, 200},
	  _("テレポート・レベル", "teleort level") },
	{ "ARTS_FALLING_STAR", ACT_FALLING_STAR, 20, 5500, {30, 50},
	  _("魔剣・流れ星", "blade arts 'falling star'") },
	{ NULL, 0, 0, 0, {0, 0},
	  "" }
};

#ifdef JP
/*!
 * @brief ランダムアーティファクトのバイアス名称テーブル
 */
const cptr artifact_bias_name[MAX_BIAS] =
{
	"なし",
	"電撃",
	"毒",
	"火炎",
	"冷気",
	"酸",
	"腕力",
	"知力",
	"賢さ",
	"器用さ",
	"耐久",
	"魅力",
	"混沌",
	"プリースト",
	"死霊",
	"法",
	"盗賊",
	"メイジ",
	"戦士",
	"レンジャー",
};
#else
const cptr artifact_bias_name[MAX_BIAS] =
{
	"None",
	"Elec",
	"Poison",
	"Fire",
	"Cold",
	"Acid",
	"STR",
	"INT",
	"WIS",
	"DEX",
	"CON",
	"CHA",
	"Chaos",
	"Pristly",
	"Necromantic",
	"Law",
	"Rogue",
	"Mage",
	"Warrior",
	"Ranger",
};
#endif