/* File: tables.c */

/* Purpose: Angband Tables */

/*
 * Copyright (c) 1989 James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research, and
 * not for profit purposes provided that this copyright and statement are
 * included in all such copies.
 */

#include "angband.h"




/*
 * Global array for looping through the "keypad directions"
 */
s16b ddd[9] =
{ 2, 8, 6, 4, 3, 1, 9, 7, 5 };

/*
 * Global arrays for converting "keypad direction" into offsets
 */
s16b ddx[10] =
{ 0, -1, 0, 1, -1, 0, 1, -1, 0, 1 };

s16b ddy[10] =
{ 0, 1, 1, 1, 0, 0, 0, -1, -1, -1 };

/*
 * Global arrays for optimizing "ddx[ddd[i]]" and "ddy[ddd[i]]"
 */
s16b ddx_ddd[9] =
{ 0, 0, 1, -1, 1, -1, 1, -1, 0 };

s16b ddy_ddd[9] =
{ 1, -1, 0, 0, 1, 1, -1, -1, 0 };


/*
 * Circular keypad direction array
 */
s16b cdd[8] =
{ 2, 3, 6, 9, 8, 7, 4, 1 };

/*
 * Global arrays for optimizing "ddx[cdd[i]]" and "ddy[cdd[i]]"
 */
s16b ddx_cdd[8] =
{ 0, 1, 1, 1, 0, -1, -1, -1 };

s16b ddy_cdd[8] =
{ 1, 1, 0, -1, -1, -1, 0, 1 };



/*
 * Global array for converting numbers to uppercase hecidecimal digit
 * This array can also be used to convert a number to an octal digit
 */
char hexsym[16] =
{
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};


/*
 * Global array for converting numbers to a logical list symbol
 */
char listsym[] =
{
	'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
	'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
	'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
	'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
	'\0'
};


/*
 * Encode the screen colors
 */
cptr color_char = "dwsorgbuDWvyRGBU";


/*
 * Stat Table (INT/WIS) -- Number of half-spells per level
 */
byte adj_mag_study[] =
{
	0	/* 3 */,
	0	/* 4 */,
	0	/* 5 */,
	0	/* 6 */,
	0	/* 7 */,
	1	/* 8 */,
	1	/* 9 */,
	1	/* 10 */,
	1	/* 11 */,
	2	/* 12 */,
	2	/* 13 */,
	2	/* 14 */,
	2	/* 15 */,
	2	/* 16 */,
	2	/* 17 */,
	2	/* 18/00-18/09 */,
	2	/* 18/10-18/19 */,
	2	/* 18/20-18/29 */,
	2	/* 18/30-18/39 */,
	2	/* 18/40-18/49 */,
	3	/* 18/50-18/59 */,
	3	/* 18/60-18/69 */,
	3	/* 18/70-18/79 */,
	3	/* 18/80-18/89 */,
	4	/* 18/90-18/99 */,
	4	/* 18/100-18/109 */,
	4	/* 18/110-18/119 */,
	5	/* 18/120-18/129 */,
	5	/* 18/130-18/139 */,
	5	/* 18/140-18/149 */,
	5	/* 18/150-18/159 */,
	5	/* 18/160-18/169 */,
	5	/* 18/170-18/179 */,
	5	/* 18/180-18/189 */,
	5	/* 18/190-18/199 */,
	5	/* 18/200-18/209 */,
	6	/* 18/210-18/219 */,
	6	/* 18/220+ */
};


/*
 * Stat Table (INT/WIS) -- extra 1/4-mana-points per level
 */
byte adj_mag_mana[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	2       /* 9 */,
	3       /* 10 */,
	4       /* 11 */,
	5       /* 12 */,
	5       /* 13 */,
	6       /* 14 */,
	7       /* 15 */,
	8       /* 16 */,
	9       /* 17 */,
	10      /* 18/00-18/09 */,
	11      /* 18/10-18/19 */,
	11      /* 18/20-18/29 */,
	12      /* 18/30-18/39 */,
	12      /* 18/40-18/49 */,
	13      /* 18/50-18/59 */,
	14      /* 18/60-18/69 */,
	15      /* 18/70-18/79 */,
	16      /* 18/80-18/89 */,
	17      /* 18/90-18/99 */,
	18      /* 18/100-18/109 */,
	19      /* 18/110-18/119 */,
	20      /* 18/120-18/129 */,
	21      /* 18/130-18/139 */,
	22      /* 18/140-18/149 */,
	23      /* 18/150-18/159 */,
	24      /* 18/160-18/169 */,
	25      /* 18/170-18/179 */,
	26      /* 18/180-18/189 */,
	27      /* 18/190-18/199 */,
	28      /* 18/200-18/209 */,
	29      /* 18/210-18/219 */,
	30      /* 18/220+ */
};


/*
 * Stat Table (INT/WIS) -- Minimum failure rate (percentage)
 */
byte adj_mag_fail[] =
{
	99      /* 3 */,
	99      /* 4 */,
	99      /* 5 */,
	99      /* 6 */,
	99      /* 7 */,
	50      /* 8 */,
	30      /* 9 */,
	20      /* 10 */,
	15      /* 11 */,
	12      /* 12 */,
	11      /* 13 */,
	10      /* 14 */,
	9       /* 15 */,
	8       /* 16 */,
	7       /* 17 */,
	6       /* 18/00-18/09 */,
	6       /* 18/10-18/19 */,
	5       /* 18/20-18/29 */,
	5       /* 18/30-18/39 */,
	5       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	4       /* 18/60-18/69 */,
	4       /* 18/70-18/79 */,
	4       /* 18/80-18/89 */,
	3       /* 18/90-18/99 */,
	3       /* 18/100-18/109 */,
	2       /* 18/110-18/119 */,
	2       /* 18/120-18/129 */,
	2       /* 18/130-18/139 */,
	2       /* 18/140-18/149 */,
	1       /* 18/150-18/159 */,
	1       /* 18/160-18/169 */,
	1       /* 18/170-18/179 */,
	1       /* 18/180-18/189 */,
	1       /* 18/190-18/199 */,
	0       /* 18/200-18/209 */,
	0       /* 18/210-18/219 */,
	0       /* 18/220+ */
};


/*
 * Stat Table (INT/WIS) -- Various things
 */
byte adj_mag_stat[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	3       /* 18/30-18/39 */,
	3       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	4       /* 18/60-18/69 */,
	5       /* 18/70-18/79 */,
	6       /* 18/80-18/89 */,
	7       /* 18/90-18/99 */,
	8       /* 18/100-18/109 */,
	9       /* 18/110-18/119 */,
	10      /* 18/120-18/129 */,
	11      /* 18/130-18/139 */,
	12      /* 18/140-18/149 */,
	13      /* 18/150-18/159 */,
	14      /* 18/160-18/169 */,
	15      /* 18/170-18/179 */,
	16      /* 18/180-18/189 */,
	17      /* 18/190-18/199 */,
	18      /* 18/200-18/209 */,
	19      /* 18/210-18/219 */,
	20      /* 18/220+ */
};


/*
 * Stat Table (CHR) -- payment percentages
 */
byte adj_chr_gold[] =
{
	130     /* 3 */,
	125     /* 4 */,
	122     /* 5 */,
	120     /* 6 */,
	118     /* 7 */,
	116     /* 8 */,
	114     /* 9 */,
	112     /* 10 */,
	110     /* 11 */,
	108     /* 12 */,
	106     /* 13 */,
	104     /* 14 */,
	103     /* 15 */,
	102     /* 16 */,
	101     /* 17 */,
	100     /* 18/00-18/09 */,
	99      /* 18/10-18/19 */,
	98      /* 18/20-18/29 */,
	97      /* 18/30-18/39 */,
	96      /* 18/40-18/49 */,
	95      /* 18/50-18/59 */,
	94      /* 18/60-18/69 */,
	93      /* 18/70-18/79 */,
	92      /* 18/80-18/89 */,
	91      /* 18/90-18/99 */,
	90      /* 18/100-18/109 */,
	89      /* 18/110-18/119 */,
	88      /* 18/120-18/129 */,
	87      /* 18/130-18/139 */,
	86      /* 18/140-18/149 */,
	85      /* 18/150-18/159 */,
	84      /* 18/160-18/169 */,
	83      /* 18/170-18/179 */,
	82      /* 18/180-18/189 */,
	81      /* 18/190-18/199 */,
	80      /* 18/200-18/209 */,
	79      /* 18/210-18/219 */,
	78      /* 18/220+ */
};


/*
 * Stat Table (INT) -- Magic devices
 */
byte adj_int_dev[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	4       /* 18/20-18/29 */,
	4       /* 18/30-18/39 */,
	5       /* 18/40-18/49 */,
	5       /* 18/50-18/59 */,
	6       /* 18/60-18/69 */,
	6       /* 18/70-18/79 */,
	7       /* 18/80-18/89 */,
	7       /* 18/90-18/99 */,
	8       /* 18/100-18/109 */,
	9       /* 18/110-18/119 */,
	10      /* 18/120-18/129 */,
	11      /* 18/130-18/139 */,
	12      /* 18/140-18/149 */,
	13      /* 18/150-18/159 */,
	14      /* 18/160-18/169 */,
	15      /* 18/170-18/179 */,
	16      /* 18/180-18/189 */,
	17      /* 18/190-18/199 */,
	18      /* 18/200-18/209 */,
	19      /* 18/210-18/219 */,
	20      /* 18/220+ */
};


/*
 * Stat Table (WIS) -- Saving throw
 */
byte adj_wis_sav[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	3       /* 18/30-18/39 */,
	3       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	4       /* 18/60-18/69 */,
	5       /* 18/70-18/79 */,
	5       /* 18/80-18/89 */,
	6       /* 18/90-18/99 */,
	7       /* 18/100-18/109 */,
	8       /* 18/110-18/119 */,
	9       /* 18/120-18/129 */,
	10      /* 18/130-18/139 */,
	11      /* 18/140-18/149 */,
	12      /* 18/150-18/159 */,
	13      /* 18/160-18/169 */,
	14      /* 18/170-18/179 */,
	15      /* 18/180-18/189 */,
	16      /* 18/190-18/199 */,
	17      /* 18/200-18/209 */,
	18      /* 18/210-18/219 */,
	19      /* 18/220+ */
};


/*
 * Stat Table (DEX) -- disarming
 */
byte adj_dex_dis[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	0       /* 8 */,
	0       /* 9 */,
	0       /* 10 */,
	0       /* 11 */,
	0       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	1       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	4       /* 18/00-18/09 */,
	4       /* 18/10-18/19 */,
	4       /* 18/20-18/29 */,
	4       /* 18/30-18/39 */,
	5       /* 18/40-18/49 */,
	5       /* 18/50-18/59 */,
	5       /* 18/60-18/69 */,
	6       /* 18/70-18/79 */,
	6       /* 18/80-18/89 */,
	7       /* 18/90-18/99 */,
	8       /* 18/100-18/109 */,
	8       /* 18/110-18/119 */,
	8       /* 18/120-18/129 */,
	8       /* 18/130-18/139 */,
	8       /* 18/140-18/149 */,
	9       /* 18/150-18/159 */,
	9       /* 18/160-18/169 */,
	9       /* 18/170-18/179 */,
	9       /* 18/180-18/189 */,
	9       /* 18/190-18/199 */,
	10      /* 18/200-18/209 */,
	10      /* 18/210-18/219 */,
	10      /* 18/220+ */
};


/*
 * Stat Table (INT) -- disarming
 */
byte adj_int_dis[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	1       /* 8 */,
	1       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	3       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	4       /* 18/30-18/39 */,
	4       /* 18/40-18/49 */,
	5       /* 18/50-18/59 */,
	6       /* 18/60-18/69 */,
	7       /* 18/70-18/79 */,
	8       /* 18/80-18/89 */,
	9       /* 18/90-18/99 */,
	10      /* 18/100-18/109 */,
	10      /* 18/110-18/119 */,
	11      /* 18/120-18/129 */,
	12      /* 18/130-18/139 */,
	13      /* 18/140-18/149 */,
	14      /* 18/150-18/159 */,
	15      /* 18/160-18/169 */,
	16      /* 18/170-18/179 */,
	17      /* 18/180-18/189 */,
	18      /* 18/190-18/199 */,
	19      /* 18/200-18/209 */,
	19      /* 18/210-18/219 */,
	20      /* 18/220+ */
};


/*
 * Stat Table (DEX) -- bonus to ac (plus 128)
 */
byte adj_dex_ta[] =
{
	128 + -4    /*  3 */,
	128 + -3    /*  4 */,
	128 + -2    /*  5 */,
	128 + -1    /*  6 */,
	128 + 0     /*  7 */,
	128 + 0     /*  8 */,
	128 + 0     /*  9 */,
	128 + 0     /* 10 */,
	128 + 0     /* 11 */,
	128 + 0     /* 12 */,
	128 + 0     /* 13 */,
	128 + 0     /* 14 */,
	128 + 1     /* 15 */,
	128 + 1     /* 16 */,
	128 + 1     /* 17 */,
	128 + 2     /* 18/00-18/09 */,
	128 + 2     /* 18/10-18/19 */,
	128 + 2     /* 18/20-18/29 */,
	128 + 2     /* 18/30-18/39 */,
	128 + 2     /* 18/40-18/49 */,
	128 + 3     /* 18/50-18/59 */,
	128 + 3     /* 18/60-18/69 */,
	128 + 3     /* 18/70-18/79 */,
	128 + 4     /* 18/80-18/89 */,
	128 + 5     /* 18/90-18/99 */,
	128 + 6     /* 18/100-18/109 */,
	128 + 7     /* 18/110-18/119 */,
	128 + 8     /* 18/120-18/129 */,
	128 + 9     /* 18/130-18/139 */,
	128 + 9     /* 18/140-18/149 */,
	128 + 10    /* 18/150-18/159 */,
	128 + 11    /* 18/160-18/169 */,
	128 + 12    /* 18/170-18/179 */,
	128 + 13    /* 18/180-18/189 */,
	128 + 14    /* 18/190-18/199 */,
	128 + 15    /* 18/200-18/209 */,
	128 + 15    /* 18/210-18/219 */,
	128 + 16    /* 18/220+ */
};


/*
 * Stat Table (STR) -- bonus to dam (plus 128)
 */
byte adj_str_td[] =
{
	128 + -2    /*  3 */,
	128 + -2    /*  4 */,
	128 + -1    /*  5 */,
	128 + -1    /*  6 */,
	128 + 0     /*  7 */,
	128 + 0     /*  8 */,
	128 + 0     /*  9 */,
	128 + 0     /* 10 */,
	128 + 0     /* 11 */,
	128 + 0     /* 12 */,
	128 + 0     /* 13 */,
	128 + 0     /* 14 */,
	128 + 0     /* 15 */,
	128 + 1     /* 16 */,
	128 + 2     /* 17 */,
	128 + 2     /* 18/00-18/09 */,
	128 + 2     /* 18/10-18/19 */,
	128 + 3     /* 18/20-18/29 */,
	128 + 3     /* 18/30-18/39 */,
	128 + 3     /* 18/40-18/49 */,
	128 + 3     /* 18/50-18/59 */,
	128 + 3     /* 18/60-18/69 */,
	128 + 4     /* 18/70-18/79 */,
	128 + 5     /* 18/80-18/89 */,
	128 + 5     /* 18/90-18/99 */,
	128 + 6     /* 18/100-18/109 */,
	128 + 7     /* 18/110-18/119 */,
	128 + 8     /* 18/120-18/129 */,
	128 + 9     /* 18/130-18/139 */,
	128 + 10    /* 18/140-18/149 */,
	128 + 11    /* 18/150-18/159 */,
	128 + 12    /* 18/160-18/169 */,
	128 + 13    /* 18/170-18/179 */,
	128 + 14    /* 18/180-18/189 */,
	128 + 15    /* 18/190-18/199 */,
	128 + 16    /* 18/200-18/209 */,
	128 + 18    /* 18/210-18/219 */,
	128 + 20    /* 18/220+ */
};


/*
 * Stat Table (DEX) -- bonus to hit (plus 128)
 */
byte adj_dex_th[] =
{
	128 + -3        /* 3 */,
	128 + -2        /* 4 */,
	128 + -2        /* 5 */,
	128 + -1        /* 6 */,
	128 + -1        /* 7 */,
	128 + 0 /* 8 */,
	128 + 0 /* 9 */,
	128 + 0 /* 10 */,
	128 + 0 /* 11 */,
	128 + 0 /* 12 */,
	128 + 0 /* 13 */,
	128 + 0 /* 14 */,
	128 + 0 /* 15 */,
	128 + 1 /* 16 */,
	128 + 2 /* 17 */,
	128 + 3 /* 18/00-18/09 */,
	128 + 3 /* 18/10-18/19 */,
	128 + 3 /* 18/20-18/29 */,
	128 + 3 /* 18/30-18/39 */,
	128 + 3 /* 18/40-18/49 */,
	128 + 4 /* 18/50-18/59 */,
	128 + 4 /* 18/60-18/69 */,
	128 + 4 /* 18/70-18/79 */,
	128 + 4 /* 18/80-18/89 */,
	128 + 5 /* 18/90-18/99 */,
	128 + 6 /* 18/100-18/109 */,
	128 + 7 /* 18/110-18/119 */,
	128 + 8 /* 18/120-18/129 */,
	128 + 9 /* 18/130-18/139 */,
	128 + 9 /* 18/140-18/149 */,
	128 + 10        /* 18/150-18/159 */,
	128 + 11        /* 18/160-18/169 */,
	128 + 12        /* 18/170-18/179 */,
	128 + 13        /* 18/180-18/189 */,
	128 + 14        /* 18/190-18/199 */,
	128 + 15        /* 18/200-18/209 */,
	128 + 15        /* 18/210-18/219 */,
	128 + 16        /* 18/220+ */
};


/*
 * Stat Table (STR) -- bonus to hit (plus 128)
 */
byte adj_str_th[] =
{
	128 + -3        /* 3 */,
	128 + -2        /* 4 */,
	128 + -1        /* 5 */,
	128 + -1        /* 6 */,
	128 + 0 /* 7 */,
	128 + 0 /* 8 */,
	128 + 0 /* 9 */,
	128 + 0 /* 10 */,
	128 + 0 /* 11 */,
	128 + 0 /* 12 */,
	128 + 0 /* 13 */,
	128 + 0 /* 14 */,
	128 + 0 /* 15 */,
	128 + 0 /* 16 */,
	128 + 0 /* 17 */,
	128 + 1 /* 18/00-18/09 */,
	128 + 1 /* 18/10-18/19 */,
	128 + 1 /* 18/20-18/29 */,
	128 + 1 /* 18/30-18/39 */,
	128 + 1 /* 18/40-18/49 */,
	128 + 1 /* 18/50-18/59 */,
	128 + 1 /* 18/60-18/69 */,
	128 + 2 /* 18/70-18/79 */,
	128 + 3 /* 18/80-18/89 */,
	128 + 4 /* 18/90-18/99 */,
	128 + 5 /* 18/100-18/109 */,
	128 + 6 /* 18/110-18/119 */,
	128 + 7 /* 18/120-18/129 */,
	128 + 8 /* 18/130-18/139 */,
	128 + 9 /* 18/140-18/149 */,
	128 + 10        /* 18/150-18/159 */,
	128 + 11        /* 18/160-18/169 */,
	128 + 12        /* 18/170-18/179 */,
	128 + 13        /* 18/180-18/189 */,
	128 + 14        /* 18/190-18/199 */,
	128 + 15        /* 18/200-18/209 */,
	128 + 15        /* 18/210-18/219 */,
	128 + 16        /* 18/220+ */
};


/*
 * Stat Table (STR) -- weight limit in deca-pounds
 */
byte adj_str_wgt[] =
{
	10      /* 3 */,
	11      /* 4 */,
	12      /* 5 */,
	13      /* 6 */,
	14      /* 7 */,
	15      /* 8 */,
	16      /* 9 */,
	17      /* 10 */,
	18      /* 11 */,
	19      /* 12 */,
	20      /* 13 */,
	21      /* 14 */,
	22      /* 15 */,
	23      /* 16 */,
	24      /* 17 */,
	25      /* 18/00-18/09 */,
	26      /* 18/10-18/19 */,
	27      /* 18/20-18/29 */,
	28      /* 18/30-18/39 */,
	29      /* 18/40-18/49 */,
	30      /* 18/50-18/59 */,
	31      /* 18/60-18/69 */,
	31      /* 18/70-18/79 */,
	32      /* 18/80-18/89 */,
	32      /* 18/90-18/99 */,
	33      /* 18/100-18/109 */,
	33      /* 18/110-18/119 */,
	34      /* 18/120-18/129 */,
	34      /* 18/130-18/139 */,
	35      /* 18/140-18/149 */,
	35      /* 18/150-18/159 */,
	36      /* 18/160-18/169 */,
	36      /* 18/170-18/179 */,
	37      /* 18/180-18/189 */,
	37      /* 18/190-18/199 */,
	38      /* 18/200-18/209 */,
	38      /* 18/210-18/219 */,
	39      /* 18/220+ */
};


/*
 * Stat Table (STR) -- weapon weight limit in pounds
 */
byte adj_str_hold[] =
{
	4       /* 3 */,
	5       /* 4 */,
	6       /* 5 */,
	7       /* 6 */,
	8       /* 7 */,
	9       /* 8 */,
	10      /* 9 */,
	11      /* 10 */,
	12      /* 11 */,
	13      /* 12 */,
	14      /* 13 */,
	15      /* 14 */,
	16      /* 15 */,
	17      /* 16 */,
	18      /* 17 */,
	19      /* 18/00-18/09 */,
	20      /* 18/10-18/19 */,
	21      /* 18/20-18/29 */,
	22      /* 18/30-18/39 */,
	23      /* 18/40-18/49 */,
	24      /* 18/50-18/59 */,
	25      /* 18/60-18/69 */,
	26      /* 18/70-18/79 */,
	27      /* 18/80-18/89 */,
	28      /* 18/90-18/99 */,
	30      /* 18/100-18/109 */,
	31      /* 18/110-18/119 */,
	32      /* 18/120-18/129 */,
	33      /* 18/130-18/139 */,
	34      /* 18/140-18/149 */,
	35      /* 18/150-18/159 */,
	37      /* 18/160-18/169 */,
	40      /* 18/170-18/179 */,
	44      /* 18/180-18/189 */,
	48      /* 18/190-18/199 */,
	50     /* 18/200-18/209 */,
	50     /* 18/210-18/219 */,
	50     /* 18/220+ */
};


/*
 * Stat Table (STR) -- digging value
 */
byte adj_str_dig[] =
{
	0       /* 3 */,
	0       /* 4 */,
	1       /* 5 */,
	2       /* 6 */,
	3       /* 7 */,
	4       /* 8 */,
	4       /* 9 */,
	5       /* 10 */,
	5       /* 11 */,
	6       /* 12 */,
	6       /* 13 */,
	7       /* 14 */,
	7       /* 15 */,
	8       /* 16 */,
	8       /* 17 */,
	9       /* 18/00-18/09 */,
	10      /* 18/10-18/19 */,
	12      /* 18/20-18/29 */,
	15      /* 18/30-18/39 */,
	20      /* 18/40-18/49 */,
	25      /* 18/50-18/59 */,
	30      /* 18/60-18/69 */,
	35      /* 18/70-18/79 */,
	40      /* 18/80-18/89 */,
	45      /* 18/90-18/99 */,
	50      /* 18/100-18/109 */,
	55      /* 18/110-18/119 */,
	60      /* 18/120-18/129 */,
	65      /* 18/130-18/139 */,
	70      /* 18/140-18/149 */,
	75      /* 18/150-18/159 */,
	80      /* 18/160-18/169 */,
	85      /* 18/170-18/179 */,
	90      /* 18/180-18/189 */,
	95      /* 18/190-18/199 */,
	100     /* 18/200-18/209 */,
	100     /* 18/210-18/219 */,
	100     /* 18/220+ */
};


/*
 * Stat Table (STR) -- help index into the "blow" table
 */
byte adj_str_blow[] =
{
	3       /* 3 */,
	4       /* 4 */,
	5       /* 5 */,
	6       /* 6 */,
	7       /* 7 */,
	8       /* 8 */,
	9       /* 9 */,
	10      /* 10 */,
	11      /* 11 */,
	12      /* 12 */,
	13      /* 13 */,
	14      /* 14 */,
	15      /* 15 */,
	16      /* 16 */,
	17      /* 17 */,
	20 /* 18/00-18/09 */,
	30 /* 18/10-18/19 */,
	40 /* 18/20-18/29 */,
	50 /* 18/30-18/39 */,
	60 /* 18/40-18/49 */,
	70 /* 18/50-18/59 */,
	80 /* 18/60-18/69 */,
	90 /* 18/70-18/79 */,
	100 /* 18/80-18/89 */,
	110 /* 18/90-18/99 */,
	120 /* 18/100-18/109 */,
	130 /* 18/110-18/119 */,
	140 /* 18/120-18/129 */,
	150 /* 18/130-18/139 */,
	160 /* 18/140-18/149 */,
	170 /* 18/150-18/159 */,
	180 /* 18/160-18/169 */,
	190 /* 18/170-18/179 */,
	200 /* 18/180-18/189 */,
	210 /* 18/190-18/199 */,
	220 /* 18/200-18/209 */,
	230 /* 18/210-18/219 */,
	240 /* 18/220+ */
};


/*
 * Stat Table (DEX) -- index into the "blow" table
 */
byte adj_dex_blow[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	0       /* 8 */,
	0       /* 9 */,
	1       /* 10 */,
	1       /* 11 */,
	1       /* 12 */,
	1       /* 13 */,
	1       /* 14 */,
	2       /* 15 */,
	2       /* 16 */,
	2       /* 17 */,
	2       /* 18/00-18/09 */,
	3       /* 18/10-18/19 */,
	3       /* 18/20-18/29 */,
	3       /* 18/30-18/39 */,
	4       /* 18/40-18/49 */,
	4       /* 18/50-18/59 */,
	5       /* 18/60-18/69 */,
	5       /* 18/70-18/79 */,
	6       /* 18/80-18/89 */,
	6       /* 18/90-18/99 */,
	7       /* 18/100-18/109 */,
	7       /* 18/110-18/119 */,
	8       /* 18/120-18/129 */,
	8       /* 18/130-18/139 */,
	9      /* 18/140-18/149 */,
	9      /* 18/150-18/159 */,
	10      /* 18/160-18/169 */,
	10      /* 18/170-18/179 */,
	11      /* 18/180-18/189 */,
	11      /* 18/190-18/199 */,
	12      /* 18/200-18/209 */,
	12      /* 18/210-18/219 */,
	13      /* 18/220+ */
};


/*
 * Stat Table (DEX) -- chance of avoiding "theft" and "falling"
 */
byte adj_dex_safe[] =
{
	0       /* 3 */,
	1       /* 4 */,
	2       /* 5 */,
	3       /* 6 */,
	4       /* 7 */,
	5       /* 8 */,
	5       /* 9 */,
	6       /* 10 */,
	6       /* 11 */,
	7       /* 12 */,
	7       /* 13 */,
	8       /* 14 */,
	8       /* 15 */,
	9       /* 16 */,
	9       /* 17 */,
	10      /* 18/00-18/09 */,
	10      /* 18/10-18/19 */,
	15      /* 18/20-18/29 */,
	15      /* 18/30-18/39 */,
	20      /* 18/40-18/49 */,
	25      /* 18/50-18/59 */,
	30      /* 18/60-18/69 */,
	35      /* 18/70-18/79 */,
	40      /* 18/80-18/89 */,
	45      /* 18/90-18/99 */,
	50      /* 18/100-18/109 */,
	60      /* 18/110-18/119 */,
	70      /* 18/120-18/129 */,
	80      /* 18/130-18/139 */,
	90      /* 18/140-18/149 */,
	100     /* 18/150-18/159 */,
	100     /* 18/160-18/169 */,
	100     /* 18/170-18/179 */,
	100     /* 18/180-18/189 */,
	100     /* 18/190-18/199 */,
	100     /* 18/200-18/209 */,
	100     /* 18/210-18/219 */,
	100     /* 18/220+ */
};


/*
 * Stat Table (CON) -- base regeneration rate
 */
byte adj_con_fix[] =
{
	0       /* 3 */,
	0       /* 4 */,
	0       /* 5 */,
	0       /* 6 */,
	0       /* 7 */,
	0       /* 8 */,
	0       /* 9 */,
	0       /* 10 */,
	0       /* 11 */,
	0       /* 12 */,
	0       /* 13 */,
	1       /* 14 */,
	1       /* 15 */,
	1       /* 16 */,
	1       /* 17 */,
	2       /* 18/00-18/09 */,
	2       /* 18/10-18/19 */,
	2       /* 18/20-18/29 */,
	2       /* 18/30-18/39 */,
	2       /* 18/40-18/49 */,
	3       /* 18/50-18/59 */,
	3       /* 18/60-18/69 */,
	3       /* 18/70-18/79 */,
	3       /* 18/80-18/89 */,
	3       /* 18/90-18/99 */,
	4       /* 18/100-18/109 */,
	4       /* 18/110-18/119 */,
	5       /* 18/120-18/129 */,
	6       /* 18/130-18/139 */,
	6       /* 18/140-18/149 */,
	7       /* 18/150-18/159 */,
	7       /* 18/160-18/169 */,
	8       /* 18/170-18/179 */,
	8       /* 18/180-18/189 */,
	8       /* 18/190-18/199 */,
	9       /* 18/200-18/209 */,
	9       /* 18/210-18/219 */,
	9       /* 18/220+ */
};


/*
 * Stat Table (CON) -- extra 1/4-hitpoints per level (plus 128)
 */
byte adj_con_mhp[] =
{
	128 + -8        /* 3 */,
	128 + -6        /* 4 */,
	128 + -4        /* 5 */,
	128 + -2        /* 6 */,
	128 + -1 /* 7 */,
	128 + 0 /* 8 */,
	128 + 0 /* 9 */,
	128 + 0 /* 10 */,
	128 + 0 /* 11 */,
	128 + 0 /* 12 */,
	128 + 0 /* 13 */,
	128 + 1 /* 14 */,
	128 + 1 /* 15 */,
	128 + 2 /* 16 */,
	128 + 3 /* 17 */,
	128 + 4 /* 18/00-18/09 */,
	128 + 5 /* 18/10-18/19 */,
	128 + 6 /* 18/20-18/29 */,
	128 + 7 /* 18/30-18/39 */,
	128 + 8 /* 18/40-18/49 */,
	128 + 9 /* 18/50-18/59 */,
	128 + 10  /* 18/60-18/69 */,
	128 + 11 /* 18/70-18/79 */,
	128 + 12 /* 18/80-18/89 */,
	128 + 14 /* 18/90-18/99 */,
	128 + 17         /* 18/100-18/109 */,
	128 + 20        /* 18/110-18/119 */,
	128 + 23        /* 18/120-18/129 */,
	128 + 26        /* 18/130-18/139 */,
	128 + 29        /* 18/140-18/149 */,
	128 + 32        /* 18/150-18/159 */,
	128 + 35        /* 18/160-18/169 */,
	128 + 38        /* 18/170-18/179 */,
	128 + 40        /* 18/180-18/189 */,
	128 + 42        /* 18/190-18/199 */,
	128 + 44        /* 18/200-18/209 */,
	128 + 46        /* 18/210-18/219 */,
	128 + 48        /* 18/220+ */
};


/*
 * Stat Table (CHR) -- charm
 */
byte adj_chr_chm[] =
{
	0       /* 3 */,
	0       /* 4 */,
	1       /* 5 */,
	2       /* 6 */,
	3       /* 7 */,
	4       /* 8 */,
	4       /* 9 */,
	5       /* 10 */,
	5       /* 11 */,
	6       /* 12 */,
	6       /* 13 */,
	7       /* 14 */,
	7       /* 15 */,
	8       /* 16 */,
	8       /* 17 */,
	9       /* 18/00-18/09 */,
	10      /* 18/10-18/19 */,
	12      /* 18/20-18/29 */,
	15      /* 18/30-18/39 */,
	18      /* 18/40-18/49 */,
	21      /* 18/50-18/59 */,
	24      /* 18/60-18/69 */,
	28      /* 18/70-18/79 */,
	32      /* 18/80-18/89 */,
	36      /* 18/90-18/99 */,
	39      /* 18/100-18/109 */,
	42      /* 18/110-18/119 */,
	45      /* 18/120-18/129 */,
	49      /* 18/130-18/139 */,
	53      /* 18/140-18/149 */,
	57      /* 18/150-18/159 */,
	61      /* 18/160-18/169 */,
	65      /* 18/170-18/179 */,
	69      /* 18/180-18/189 */,
	73      /* 18/190-18/199 */,
	77      /* 18/200-18/209 */,
	81      /* 18/210-18/219 */,
	85      /* 18/220+ */
};


/*
 * This table is used to help calculate the number of blows the player can
 * make in a single round of attacks (one player turn) with a normal weapon.
 *
 * This number ranges from a single blow/round for weak players to up to six
 * blows/round for powerful warriors.
 *
 * Note that certain artifacts and ego-items give "bonus" blows/round.
 *
 * First, from the player class, we extract some values:
 *
 * Warrior       num = 6; mul = 5; div = MAX(70, weapon_weight);
 * Berserker     num = 6; mul = 7; div = MAX(70, weapon_weight);
 * Mage          num = 3; mul = 2; div = MAX(100, weapon_weight);
 * Priest        num = 5; mul = 3; div = MAX(100, weapon_weight);
 * Mindcrafter   num = 5; mul = 3; div = MAX(100, weapon_weight);
 * Rogue         num = 5; mul = 3; div = MAX(40, weapon_weight);
 * Ranger        num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Paladin       num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Kaji          num = 5; mul = 5; div = MAX(150, weapon_weight);
 * Warrior-Mage  num = 5; mul = 3; div = MAX(70, weapon_weight);
 * Chaos Warrior num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Monk          num = 5; mul = 3; div = MAX(60, weapon_weight);
 * Tourist       num = 4; mul = 3; div = MAX(100, weapon_weight);
 * Imitator      num = 5; mul = 4; div = MAX(70, weapon_weight);
 * Beastmaster   num = 5; mul = 3; div = MAX(70, weapon_weight);
 * Sorcerer      num = 1; mul = 1; div = MAX(1, weapon_weight);
 * Archer        num = 4; mul = 2; div = MAX(70, weapon_weight);
 * Magic eater   num = 4; mul = 2; div = MAX(70, weapon_weight);
 * ForceTrainer  num = 4; mul = 2; div = MAX(60, weapon_weight);
 * Mirror Master num = 3; mul = 3; div = MAX(100, weapon_weight);
 * Ninja         num = 4; mul = 1; div = MAX(20, weapon_weight);
 *
 * To get "P", we look up the relevant "adj_str_blow[]" (see above),
 * multiply it by "mul", and then divide it by "div".
 * Increase P by 1 if you wield a weapon two-handed.
 * Decrease P by 1 if you are a Ninja.
 *
 * To get "D", we look up the relevant "adj_dex_blow[]" (see above),
 *
 * The player gets "blows_table[P][D]" blows/round, as shown below,
 * up to a maximum of "num" blows/round, plus any "bonus" blows/round.
 */
byte blows_table[12][12] =
{
	/* P/D */
	/*      0,   1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11+ */
	/*      3   10   15  /10  /40  /60  /80 /100 /120 /140 /160 /180  */
/* 0 */{	1,   1,   1,   1,   1,   2,   2,   2,   2,   3,   3,   4 },
/* 1 */{	1,   1,   1,   2,   2,   2,   3,   3,   3,   4,   4,   4 },
/* 2 */{	1,   1,   2,   2,   3,   3,   4,   4,   4,   5,   5,   5 },
/* 3 */{	1,   1,   2,   3,   3,   4,   4,   4,   5,   5,   5,   5 },
/* 4 */{	1,   1,   2,   3,   3,   4,   4,   5,   5,   5,   5,   5 },
/* 5 */{	1,   1,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6 },
/* 6 */{	1,   1,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6 },
/* 7 */{	1,   2,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6 },
/* 8 */{	1,   2,   3,   3,   4,   4,   4,   5,   5,   5,   6,   6 },
/* 9 */{	1,   2,   3,   4,   4,   4,   5,   5,   5,   5,   6,   6 },
/* 10*/{	2,   2,   3,   4,   4,   4,   5,   5,   5,   6,   6,   6 },
/*11+*/{	2,   2,   3,   4,   4,   4,   5,   5,   6,   6,   6,   6 },

};

s16b arena_monsters[MAX_ARENA_MONS+2] =
{
	1059,  1060,	230,	241,	255,	270,	285,
	293,	321,	348,	367,	395,	396,	403,
	417,	436,	453,	464,	471,	479,	490,
	504,	539,	569,	593,	621,	623,	1061,
	1062,	657,	666,	702,	720,	768,	786,
	798,	988,	847,	  0,	871
};



s16b arena_shouhin[MAX_ARENA_MONS+2] =
{
	  0,	  0,	  0,	  0,	249,	  0,	  0,
	  0,	  0,	242,	  0,	  0,	  0,	  0,
	583,	  0,	  0,	  0,	  0,	198,	  0,
	190,	  0,	363,	  0,	212,	  0,	207,
	  0,	628,	  0,	422,	  0,	372,	327,
	199,	325,	374,	  0,	205
};



/*
 * Store owners (exactly four "possible" owners per store, chosen randomly)
 * { name, purse, max greed, min greed, haggle_per, tolerance, race, unused }
 *
 * Lifted extra shopkeepers from CthAngband (don't you just love open source
 * development? ;-)). Since this gave less than 32 unique names for some
 * shops, those have their first x names copied to reach 32.
 *
 * For the weapon and armour shops, several owners have a limit of 5k.
 *
 * I want to do 50k owners, but the purse is currently s16b. Perhaps
 * we should just store 1/10th of the purse?
 */
owner_type owners[MAX_STORES][MAX_OWNERS] =
{
	{
		/* General store - 32 unique names */
/*
   Raistlin  dragonlance  powerful wizard 
   Rincewind the Chicken  Terry Pratchett  Discworld оʪ 嵭Υѥǥ
 */

#ifdef JP
                { "եɥ꡼ʥӥ",       200,    170,  108,   5,  15,  RACE_HOBBIT},
/*                { "¼ԥ饹ȥ",       200,    175,  108,   4,  12,  RACE_HUMAN},  */
                { "¼ԥ󥹥",       200,    175,  108,   4,  12,  RACE_HUMAN}, 
                { "ؤ㤤륿",             300,    170,  107,   5,  15,  RACE_GNOME},
                { "ϥ󥵥ʥ饤=",      300,    165,  107,   6,  18,  RACE_ELF},
#else
		{ "Bilbo the Friendly",         200,    170, 108,  5, 15, RACE_HOBBIT},
/*		{ "Raistlin the Chicken",       200,    175, 108,  4, 12, RACE_HUMAN}, */
		{ "Rincewind the Chicken",       200,    175, 108,  4, 12, RACE_HUMAN},
		{ "Sultan the Midget",          300,    170, 107,  5, 15, RACE_GNOME},
		{ "Lyar-el the Comely",         300,    165, 107,  6, 18, RACE_ELF},
#endif

#ifdef JP
{ "ڤʥեޥ",         250,    170, 108,  5, 15, RACE_HOBBIT},
#else
		{ "Falilmawen the Friendly",         250,    170, 108,  5, 15, RACE_HOBBIT},
#endif
#ifdef JP
{ "¼ԥ",       500,    175, 108,  4, 12, RACE_HUMAN},
#else
		{ "Voirin the Cowardly",       500,    175, 108,  4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "ӤΥ饷ʥ",          750,    170, 107,  5, 15, RACE_BEASTMAN},
#else
		{ "Erashnak the Midget",          750,    170, 107,  5, 15, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ϥ󥵥ʥå",        1000,    165, 107,  6, 18, RACE_HALF_TITAN},
#else
		{ "Grug the Comely",        1000,    165, 107,  6, 18, RACE_HALF_TITAN},
#endif
#ifdef JP
{ "ʥեӥ",         250,    170, 108,  5, 15, RACE_HUMAN},
#else
		{ "Forovir the Cheap",         250,    170, 108,  5, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ϼΥꥹ",       500,    175, 108,  4, 12, RACE_HUMAN},
#else
		{ "Ellis the Fool",       500,    175, 108,  4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "ʢڥΥեС",          750,    170, 107,  5, 15, RACE_VAMPIRE},
#else
		{ "Filbert the Hungry",          750,    170, 107,  5, 15, RACE_VAMPIRE},
#endif
#ifdef JP
{ "ʡ롦",        1000,    165, 107,  6, 18, RACE_MIND_FLAYER},
#else
		{ "Fthnargl Psathiggua",        1000,    165, 107,  6, 18, RACE_MIND_FLAYER},
#endif
#ifdef JP
{ "Ĺभ",         250,    170, 108,  5, 15, RACE_SPECTRE},
#else
		{ "Eloise Long-Dead",         250,    170, 108,  5, 15, RACE_SPECTRE},
#endif
#ifdef JP
{ "ΥޤΥեǥ",       500,    175, 108,  4, 12, RACE_ZOMBIE},
#else
		{ "Fundi the Slow",       500,    175, 108,  4, 12, RACE_ZOMBIE},
#endif
#ifdef JP
{ "󥵥",          750,    170, 107,  5, 15, RACE_SKELETON},
#else
		{ "Granthus",          750,    170, 107,  5, 15, RACE_SKELETON},
#endif
#ifdef JP
{ "ǫʥå",        1000,    165, 107,  6, 18, RACE_VAMPIRE},
#else
		{ "Lorax the Suave",        1000,    165, 107,  6, 18, RACE_VAMPIRE},
#endif
#ifdef JP
{ "֥å",         250,    170, 108,  5, 15, RACE_HALF_ORC},
#else
		{ "Butch",         250,    170, 108,  5, 15, RACE_HALF_ORC},
#endif
#ifdef JP
{ "٥쥹",       500,    175, 108,  4, 12, RACE_HIGH_ELF},
#else
		{ "Elbereth the Beautiful",       500,    175, 108,  4, 12, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "쥹",          750,    170, 107,  5, 15, RACE_GNOME},
#else
		{ "Sarleth the Sneaky",          750,    170, 107,  5, 15, RACE_GNOME},
#endif
#ifdef JP
{ "ʡå",        1000,    165, 107,  6, 18, RACE_DWARF},
#else
		{ "Narlock",        1000,    165, 107,  6, 18, RACE_DWARF},
#endif
#ifdef JP
{ "ӤΥإͥå",         250,    170, 108,  5, 15, RACE_GNOME},
#else
		{ "Haneka the Small",         250,    170, 108,  5, 15, RACE_GNOME},
#endif
#ifdef JP
{ "",       500,    175, 108,  4, 12, RACE_HALF_GIANT},
#else
		{ "Loirin the Mad",       500,    175, 108,  4, 12, RACE_HALF_GIANT},
#endif
#ifdef JP
{ "©Υ",          750,    170, 107,  5, 15, RACE_DRACONIAN},
#else
		{ "Wuto Poisonbreath",          750,    170, 107,  5, 15, RACE_DRACONIAN},
#endif
#ifdef JP
{ "Ǥ֤ä祢饡",        1000,    165, 107,  6, 18, RACE_DRACONIAN},
#else
		{ "Araaka the Rotund",        1000,    165, 107,  6, 18, RACE_DRACONIAN},
#endif
#ifdef JP
{ "ǽʥס",         250,    170, 108,  5, 15, RACE_BEASTMAN},
#else
		{ "Poogor the Dumb",         250,    170, 108,  5, 15, RACE_BEASTMAN},
#endif
#ifdef JP
{ "եեꥢ",       500,    175, 108,  4, 12, RACE_ELF},
#else
		{ "Felorfiliand",       500,    175, 108,  4, 12, RACE_ELF},
#endif
#ifdef JP
{ "ǯޥ",          750,    170, 107,  5, 15, RACE_GNOME},
#else
		{ "Maroka the Aged",          750,    170, 107,  5, 15, RACE_GNOME},
#endif
#ifdef JP
{ " ͦʥ",        1000,    165, 107,  6, 18, RACE_HALF_GIANT},
#else
		{ "Sasin the Bold",        1000,    165, 107,  6, 18, RACE_HALF_GIANT},
#endif
#ifdef JP
{ "ļ˼ԥӥޡ",         250,    170, 108,  5, 15, RACE_HUMAN},
#else
		{ "Abiemar the Peasant",         250,    170, 108,  5, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "˳ʥϡ",       500,    175, 108,  4, 12, RACE_HALF_ORC},
#else
		{ "Hurk the Poor",       500,    175, 108,  4, 12, RACE_HALF_ORC},
#endif
#ifdef JP
{ "ߤʥ",          750,    170, 107,  5, 15, RACE_ZOMBIE},
#else
		{ "Soalin the Wretched",          750,    170, 107,  5, 15, RACE_ZOMBIE},
#endif
#ifdef JP
{ "Ǥʥ",        1000,    165, 107,  6, 18, RACE_ELF},
#else
		{ "Merulla the Humble",        1000,    165, 107,  6, 18, RACE_ELF},
#endif
	},
	{
		/* Armoury - 28 unique names */
#ifdef JP
                { "=",      5000,   210,  115,   5,   7,  RACE_HALF_ORC},
                { "Ǽԥ=",  10000,  190,  111,   4,   9,  RACE_HUMAN},
                { "ԥǥ",                 25000,  200,  112,   4,  10,  RACE_DUNADAN},
                { "결Υ",   30000,  200,  112,   4,   5,  RACE_DWARF},
#else
		{ "Kon-Dar the Ugly",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
		{ "Darg-Low the Grim",          10000,  190, 111,  4,  9, RACE_HUMAN},
		{ "Decado the Handsome",            25000,  200, 112,  4, 10, RACE_DUNADAN},
		{ "Wieland the Smith",          30000,  200, 112,  4,  5, RACE_DWARF},
#endif

#ifdef JP
{ "=",           10000,   210, 115,  5,  7, RACE_HALF_ORC},
#else
		{ "Kon-Dar the Ugly",           10000,   210, 115,  5,  7, RACE_HALF_ORC},
#endif
#ifdef JP
{ "Ǽԥ=",          15000,  190, 111,  4,  9, RACE_HUMAN},
#else
		{ "Darg-Low the Grim",          15000,  190, 111,  4,  9, RACE_HUMAN},
#endif
#ifdef JP
{ "ϥ󥵥ʥǥ",            25000,  200, 112,  4, 10, RACE_AMBERITE},
#else
		{ "Decado the Handsome",            25000,  200, 112,  4, 10, RACE_AMBERITE},
#endif
#ifdef JP
        { "ɥ饴󥹥",          30000,  200, 112,  4,  5, RACE_ELF},
#else
        { "Elo Dragonscale",          30000,  200, 112,  4,  5, RACE_ELF},
#endif
#ifdef JP
{ "ǥꥫȥ",           10000,   210, 115,  5,  7, RACE_SPRITE},
#else
		{ "Delicatus",           10000,   210, 115,  5,  7, RACE_SPRITE},
#endif
#ifdef JP
{ "ʥ롼",          15000,  190, 111,  4,  9, RACE_HALF_GIANT},
#else
		{ "Gruce the Huge",          15000,  190, 111,  4,  9, RACE_HALF_GIANT},
#endif
#ifdef JP
{ "˥ॹ",            25000,  200, 112,  4, 10, RACE_GOLEM},
#else
		{ "Animus",            25000,  200, 112,  4, 10, RACE_GOLEM},
#endif
#ifdef JP
        { "ޥ",          30000,  200, 112,  4,  5, RACE_HALF_TITAN},
#else
        { "Malvus",          30000,  200, 112,  4,  5, RACE_HALF_TITAN},
#endif
#ifdef JP
{ "饯",           10000,   210, 115,  5,  7, RACE_ZOMBIE},
#else
		{ "Selaxis",           10000,   210, 115,  5,  7, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ǥ",          5000,  190, 111,  4,  9, RACE_SPECTRE},
#else
		{ "Deathchill",          5000,  190, 111,  4,  9, RACE_SPECTRE},
#endif
#ifdef JP
{ "ʥɥꥪ",            25000,  200, 112,  4, 10, RACE_SPECTRE},
#else
		{ "Drios the Faint",            25000,  200, 112,  4, 10, RACE_SPECTRE},
#endif
#ifdef JP
        { "䤿Хå",          30000,  200, 112,  4,  5, RACE_VAMPIRE},
#else
        { "Bathric the Cold",          30000,  200, 112,  4,  5, RACE_VAMPIRE},
#endif
#ifdef JP
{ "󥸥",           10000,   210, 115,  5,  7, RACE_HALF_TROLL},
#else
		{ "Vengella the Cruel",           10000,   210, 115,  5,  7, RACE_HALF_TROLL},
#endif
#ifdef JP
{ "ԥ",          15000,  190, 111,  4,  9, RACE_HUMAN},
#else
		{ "Wyrana the Mighty",          15000,  190, 111,  4,  9, RACE_HUMAN},
#endif
#ifdef JP
{ "襸",            25000,  200, 112,  4, 10, RACE_DWARF},
#else
		{ "Yojo II",            25000,  200, 112,  4, 10, RACE_DWARF},
#endif
#ifdef JP
        { "ͥʥ顼",          30000,  200, 112,  4,  5, RACE_AMBERITE},
#else
        { "Ranalar the Sweet",          30000,  200, 112,  4,  5, RACE_AMBERITE},
#endif
#ifdef JP
{ "ԾΥۥХ",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
#else
		{ "Horbag the Unclean",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
#endif
#ifdef JP
{ "ƥѥΥ",          15000,  190, 111,  4,  9, RACE_DARK_ELF},
#else
		{ "Elelen the Telepath",          15000,  190, 111,  4,  9, RACE_DARK_ELF},
#endif
#ifdef JP
{ "ɥꥢ",            25000,  200, 112,  4, 10, RACE_SPRITE},
#else
		{ "Isedrelias",            25000,  200, 112,  4, 10, RACE_SPRITE},
#endif
#ifdef JP
        { "ܥʡ",          5000,  200, 112,  4,  5, RACE_CYCLOPS},
#else
        { "Vegnar One-eye",          5000,  200, 112,  4,  5, RACE_CYCLOPS},
#endif
#ifdef JP
{ "٤Υǥå",           10000,   210, 115,  5,  7, RACE_BEASTMAN},
#else
		{ "Rodish the Chaotic",           10000,   210, 115,  5,  7, RACE_BEASTMAN},
#endif
#ifdef JP
{ "إ",          15000,  190, 111,  4,  9, RACE_NIBELUNG},
#else
		{ "Hesin Swordmaster",          15000,  190, 111,  4,  9, RACE_NIBELUNG},
#endif
#ifdef JP
{ "벰Υ٥ꥹ",           10000,  200, 112,  4, 10, RACE_DARK_ELF},
#else
		{ "Elvererith the Cheat",           10000,  200, 112,  4, 10, RACE_DARK_ELF},
#endif
#ifdef JP
        { "פΥ",          30000,  200, 112,  4,  5, RACE_IMP},
#else
        { "Zzathath the Imp",          30000,  200, 112,  4,  5, RACE_IMP},
#endif
#ifdef JP
{ "=",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
#else
		{ "Kon-Dar the Ugly",           5000,   210, 115,  5,  7, RACE_HALF_ORC},
#endif
#ifdef JP
{ "Ǽԥ=",          10000,  190, 111,  4,  9, RACE_HUMAN},
#else
		{ "Darg-Low the Grim",          10000,  190, 111,  4,  9, RACE_HUMAN},
#endif
#ifdef JP
{ "ϥ󥵥ʥǥ",            25000,  200, 112,  4, 10, RACE_AMBERITE},
#else
		{ "Decado the Handsome",            25000,  200, 112,  4, 10, RACE_AMBERITE},
#endif
#ifdef JP
{ "결Υ",          30000,  200, 112,  4,  5, RACE_DWARF},
#else
		{ "Wieland the Smith",          30000,  200, 112,  4,  5, RACE_DWARF},
#endif
	},
	{
		/* Weapon Smith - 28 unique names */
#ifdef JP
                { "Ǧʤ륢Υ",        5000,   210,  115,   6,   6,  RACE_HALF_TROLL},
                { "ûΥ", 10000,  185,  110,   5,   9,  RACE_HALF_ELF},
                { "åޥΥǥ", 25000,  190,  115,   5,   7,  RACE_HOBBIT},
                { "εΥ饤", 30000,  195,  112,   4,   8,  RACE_DWARF},
#else
		{ "Arnold the Beastly",      5000,   210, 115,  6,  6, RACE_BARBARIAN},
		{ "Arndal Beast-Slayer",        10000,  185, 110,  5,  9, RACE_HALF_ELF},
		{ "Eddie Beast-Master",         25000,  190, 115,  5,  7, RACE_HALF_ORC},
		{ "Oglign Dragon-Slayer",       30000,  195, 112,  4,  8, RACE_DWARF},
#endif

#ifdef JP
{ "ԥɥ塼",      10000,   210, 115,  6,  6, RACE_HUMAN},
#else
		{ "Drew the Skilled",      10000,   210, 115,  6,  6, RACE_HUMAN},
#endif
#ifdef JP
{"ζλҥå",        15000,  185, 110,  5,  9, RACE_DRACONIAN},
#else
		{"Orrax Dragonson",        15000,  185, 110,  5,  9, RACE_DRACONIAN},
#endif
#ifdef JP
{ "µΥ󥹥å",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
#else
		{ "Anthrax Disease-Carrier",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ԥ륳",       30000,  195, 112,  4,  8, RACE_DWARF},
#else
		{ "Arkhoth the Stout",       30000,  195, 112,  4,  8, RACE_DWARF},
#endif
#ifdef JP
{ "ԤΥꥢ",      5000,   210, 115,  6,  6, RACE_ZOMBIE},
#else
		{ "Sarlyas the Rotten",      5000,   210, 115,  6,  6, RACE_ZOMBIE},
#endif
#ifdef JP
{"Υȥ",        15000,  185, 110,  5,  9, RACE_SKELETON},
#else
		{"Tuethic Bare-Bones",        15000,  185, 110,  5,  9, RACE_SKELETON},
#endif
#ifdef JP
{ "ӥꥪ",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
#else
		{ "Bilious",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ե",       30000,  195, 112,  4,  8, RACE_ZOMBIE},
#else
		{ "Fasgul",       30000,  195, 112,  4,  8, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ѥǥΥեꥹ",      10000,   210, 115,  6,  6, RACE_BARBARIAN},
#else
		{ "Ellefris the Paladin",      10000,   210, 115,  6,  6, RACE_BARBARIAN},
#endif
#ifdef JP
{"'ȥ'",        15000,  185, 110,  5,  9, RACE_KLACKON},
#else
		{"K'trrik'k",        15000,  185, 110,  5,  9, RACE_KLACKON},
#endif
#ifdef JP
{ "ͧɥ",         25000,  190, 115,  5,  7, RACE_DARK_ELF},
#else
		{ "Drocus Spiderfriend",         25000,  190, 115,  5,  7, RACE_DARK_ELF},
#endif
#ifdef JP
{ "ͻΥե󥰥",       30000,  195, 112,  4,  8, RACE_DWARF},
#else
		{ "Fungus Giant-Slayer",       30000,  195, 112,  4,  8, RACE_DWARF},
#endif
#ifdef JP
{ "ǥ",      10000,   210, 115,  6,  6, RACE_ELF},
#else
		{ "Delantha",      10000,   210, 115,  6,  6, RACE_ELF},
#endif
#ifdef JP
{"󥸥㡼Υӥ",        15000,  185, 110,  5,  9, RACE_HALF_ELF},
#else
		{"Solvistani the Ranger",        15000,  185, 110,  5,  9, RACE_HALF_ELF},
#endif
#ifdef JP
{ "ΥޤΥ",         25000,  190, 115,  5,  7, RACE_GOLEM},
#else
		{ "Xoril the Slow",         25000,  190, 115,  5,  7, RACE_GOLEM},
#endif
#ifdef JP
{ "󡦥եå",       20000,  195, 112,  4,  8, RACE_HALF_ELF},
#else
		{ "Aeon Flux",       20000,  195, 112,  4,  8, RACE_HALF_ELF},
#endif
#ifdef JP
{ "ԥʥɥå",      10000,   210, 115,  6,  6, RACE_HOBBIT},
#else
		{ "Nadoc the Strong",      10000,   210, 115,  6,  6, RACE_HOBBIT},
#endif
#ifdef JP
{"⥰",        15000,  185, 110,  5,  9, RACE_KOBOLD},
#else
		{"Eramog the Weak",        15000,  185, 110,  5,  9, RACE_KOBOLD},
#endif
#ifdef JP
{ "ʥꥹ",         25000,  190, 115,  5,  7, RACE_VAMPIRE},
#else
		{ "Eowilith the Fair",         25000,  190, 115,  5,  7, RACE_VAMPIRE},
#endif
#ifdef JP
{ "ХΥҥ奤⥰",       30000,  195, 112,  4,  8, RACE_HALF_ORC},
#else
		{ "Huimog Balrog-Slayer",       30000,  195, 112,  4,  8, RACE_HALF_ORC},
#endif
#ifdef JP
{ "ԡ",      5000,   210, 115,  6,  6, RACE_HUMAN},
#else
		{ "Peadus the Cruel",      5000,   210, 115,  6,  6, RACE_HUMAN},
#endif
#ifdef JP
{ "⥰ 쥤䡼",        15000,  185, 110,  5,  9, RACE_HALF_OGRE},
#else
		{ "Vamog Slayer",        15000,  185, 110,  5,  9, RACE_HALF_OGRE},
#endif
#ifdef JP
{ "աʥ",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
#else
		{ "Hooshnak the Vicious",         25000,  190, 115,  5,  7, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ƮХ",       30000,  195, 112,  4,  8, RACE_BARBARIAN},
#else
		{ "Balenn War-Dancer",       30000,  195, 112,  4,  8, RACE_BARBARIAN},
#endif
#ifdef JP
{ " Ǧʤ륢Υ",      5000,   210, 115,  6,  6, RACE_BARBARIAN},
#else
		{ "Arnold the Beastly",      5000,   210, 115,  6,  6, RACE_BARBARIAN},
#endif
#ifdef JP
{ "ûΥ",        10000,  185, 110,  5,  9, RACE_HALF_ELF},
#else
		{ "Arndal Beast-Slayer",        10000,  185, 110,  5,  9, RACE_HALF_ELF},
#endif
#ifdef JP
{ "ӡȥޥǥ",         25000,  190, 115,  5,  7, RACE_HALF_ORC},
#else
		{ "Eddie Beast-Master",         25000,  190, 115,  5,  7, RACE_HALF_ORC},
#endif
#ifdef JP
{ "εΥ饤",       30000,  195, 112,  4,  8, RACE_DWARF},
#else
		{ "Oglign Dragon-Slayer",       30000,  195, 112,  4,  8, RACE_DWARF},
#endif
	},
	{
		/* Temple - 22 unique names */
#ifdef JP
                { "Ǥʥ롼ɥ",         5000,   175,  109,   6,  15,  RACE_HUMAN},
                { "ѥǥΥʡ",       10000,  185,  110,   5,  23,  RACE_HUMAN},
                { "Ф줷ȥ",                     25000,  180,  107,   6,  20,  RACE_ELF},
                { "ʤ륵饹ȥ",                     30000,  185,  109,   5,  15,  RACE_DWARF},
#else
		{ "Ludwig the Humble",          5000,   175, 109,  6, 15, RACE_DWARF},
		{ "Gunnar the Paladin",         10000,  185, 110,  5, 23, RACE_HALF_TROLL},
		{ "Torin the Chosen",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
		{ "Sarastro the Wise",          30000,  185, 109,  5, 15, RACE_HUMAN},
#endif

#ifdef JP
{ "ѡ붪",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
#else
		{ "Sir Parsival the Pure",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ʤ륢ʥ",          30000,  185, 109,  5, 15, RACE_HUMAN},
#else
		{ "Asenath the Holy",          30000,  185, 109,  5, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ޥåΥ",         10000,   175, 109,  6, 15, RACE_HUMAN},
#else
		{ "McKinnon",         10000,   175, 109,  6, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ؿ",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
#else
		{ "Mistress Chastity",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ɥ륤ɤΥϥ˥å",           25000,  180, 107,  6, 20, RACE_HOBBIT},
#else
		{ "Hashnik the Druid",           25000,  180, 107,  6, 20, RACE_HOBBIT},
#endif
#ifdef JP
{ "եʥ",          30000,  185, 109,  5, 15, RACE_YEEK},
#else
		{ "Finak",          30000,  185, 109,  5, 15, RACE_YEEK},
#endif
#ifdef JP
{ "ꥭå",         10000,   175, 109,  6, 15, RACE_KLACKON},
#else
		{ "Krikkik",         10000,   175, 109,  6, 15, RACE_KLACKON},
#endif
#ifdef JP
{ "Ӥԥ",         15000,  185, 110,  5, 23, RACE_ELF},
#else
		{ "Morival the Wild",         15000,  185, 110,  5, 23, RACE_ELF},
#endif
#ifdef JP
{ "Ťۥå",           25000,  180, 107,  6, 20, RACE_IMP},
#else
		{ "Hoshak the Dark",           25000,  180, 107,  6, 20, RACE_IMP},
#endif
#ifdef JP
{ "ԥ",          30000,  185, 109,  5, 15, RACE_HUMAN},
#else
		{ "Atal the Wise",          30000,  185, 109,  5, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "٥˥å",         10000,   175, 109,  6, 15, RACE_HUMAN},
#else
		{ "Ibenidd the Chaste",         10000,   175, 109,  6, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ǥ",         15000,  185, 110,  5, 23, RACE_HALF_TROLL},
#else
		{ "Eridish",         15000,  185, 110,  5, 23, RACE_HALF_TROLL},
#endif
#ifdef JP
{ "ѻեɥ",           25000,  180, 107,  6, 20, RACE_HALF_OGRE},
#else
		{ "Vrudush the Shaman",           25000,  180, 107,  6, 20, RACE_HALF_OGRE},
#endif
#ifdef JP
{ "Υϥ",          30000,  185, 109,  5, 15, RACE_BARBARIAN},
#else
		{ "Haob the Berserker",          30000,  185, 109,  5, 15, RACE_BARBARIAN},
#endif
#ifdef JP
{ "㤭ץ롼ǥ",         10000,   175, 109,  6, 15, RACE_HALF_OGRE},
#else
		{ "Proogdish the Youthfull",         10000,   175, 109,  6, 15, RACE_HALF_OGRE},
#endif
#ifdef JP
{ "磻",         15000,  185, 110,  5, 23, RACE_YEEK},
#else
		{ "Lumwise the Mad",         15000,  185, 110,  5, 23, RACE_YEEK},
#endif
#ifdef JP
{ "ͭԥ",           25000,  180, 107,  6, 20, RACE_KOBOLD},
#else
		{ "Muirt the Virtuous",           25000,  180, 107,  6, 20, RACE_KOBOLD},
#endif
#ifdef JP
{ "ɥС",          30000,  185, 109,  5, 15, RACE_SPECTRE},
#else
		{ "Dardobard the Weak",          30000,  185, 109,  5, 15, RACE_SPECTRE},
#endif
#ifdef JP
{ "Ǥʥ롼ɥ",          5000,   175, 109,  6, 15, RACE_DWARF},
#else
		{ "Ludwig the Humble",          5000,   175, 109,  6, 15, RACE_DWARF},
#endif
#ifdef JP
{ "ѥǥΥʡ",         10000,  185, 110,  5, 23, RACE_HALF_TROLL},
#else
		{ "Gunnar the Paladin",         10000,  185, 110,  5, 23, RACE_HALF_TROLL},
#endif
#ifdef JP
{ "Ф줷ȥ",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
#else
		{ "Torin the Chosen",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ʤ륵饹ȥ",          30000,  185, 109,  5, 15, RACE_HUMAN},
#else
		{ "Sarastro the Wise",          30000,  185, 109,  5, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ѡ붪",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
#else
		{ "Sir Parsival the Pure",           25000,  180, 107,  6, 20, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ʤ륢ʥ",          30000,  185, 109,  5, 15, RACE_HUMAN},
#else
		{ "Asenath the Holy",          30000,  185, 109,  5, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ޥåΥ",         10000,   175, 109,  6, 15, RACE_HUMAN},
#else
		{ "McKinnon",         10000,   175, 109,  6, 15, RACE_HUMAN},
#endif
#ifdef JP
{ "ؿ",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
#else
		{ "Mistress Chastity",         15000,  185, 110,  5, 23, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ɥ륤 ϥ˥å",           25000,  180, 107,  6, 20, RACE_HOBBIT},
#else
		{ "Hashnik the Druid",           25000,  180, 107,  6, 20, RACE_HOBBIT},
#endif
#ifdef JP
{ "եʥ",        15000,  200, 110,  7,  8, RACE_BARBARIAN},
#else
		{ "Finak",          30000,  185, 109,  5, 15, RACE_YEEK},
#endif
	},
	{
		/* Alchemist - 26 unique names */
#ifdef JP
                { "ؼԥޥ",             10000,  190,  111,   5,   8,  RACE_HALF_ELF},
                { "Υ",   10000,  190,  110,   6,   8,  RACE_HOBBIT},
                { "ߥߥ",              15000,  200,  116,   6,   9,  RACE_GNOME},
                { "ϼ奸=ե",                   15000,  220,  111,   4,   9,  RACE_ELF},/*FIRST*/
#else
		{ "Mauser the Chemist",         10000,  190, 111,  5,  8, RACE_HALF_ELF},
		{ "Wizzle the Chaotic",         10000,  190, 110,  6,  8, RACE_HOBBIT},
		{ "Midas the Greedy",           15000,  200, 116,  6,  9, RACE_GNOME},
		{ "Ja-Far the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
#endif

#ifdef JP
        { "饫",           15000,  200, 116,  6,  9, RACE_KLACKON},
#else
        { "Kakalrakakal",           15000,  200, 116,  6,  9, RACE_KLACKON},
#endif
#ifdef JP
        { "ϣѻե=",       15000,  220, 111,  4,  9, RACE_ELF},
#else
        { "Jal-Eth the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
#endif
#ifdef JP
{ "ѿեͥ饹",         10000,  190, 111,  5,  8, RACE_DWARF},
#else
		{ "Fanelath the Cautious",         10000,  190, 111,  5,  8, RACE_DWARF},
#endif
#ifdef JP
{ "Υ󥷡",         10000,  190, 110,  6,  8, RACE_HUMAN},
#else
		{ "Runcie the Insane",         10000,  190, 110,  6,  8, RACE_HUMAN},
#endif
#ifdef JP
        { "֥",           15000,  200, 116,  6,  9, RACE_GNOME},
#else
        { "Grumbleworth",           15000,  200, 116,  6,  9, RACE_GNOME},
#endif
#ifdef JP
        { "եå",       15000,  220, 111,  4,  9, RACE_SPRITE},
#else
        { "Flitter",       15000,  220, 111,  4,  9, RACE_SPRITE},
#endif
#ifdef JP
{ "륹",         10000,  190, 111,  5,  8, RACE_HUMAN},
#else
		{ "Xarillus",         10000,  190, 111,  5,  8, RACE_HUMAN},
#endif
#ifdef JP
{ "ŤС",         10000,  190, 110,  6,  8, RACE_DWARF},
#else
		{ "Egbert the Old",         10000,  190, 110,  6,  8, RACE_DWARF},
#endif
#ifdef JP
        { "ؤ⤭ɥ",           15000,  200, 116,  6,  9, RACE_HIGH_ELF},
#else
        { "Valindra the Proud",           15000,  200, 116,  6,  9, RACE_HIGH_ELF},
#endif
#ifdef JP
        { "ϣѻե",       15000,  220, 111,  4,  9, RACE_HUMAN},
#else
        { "Taen the Alchemist",       15000,  220, 111,  4,  9, RACE_HUMAN},
#endif
#ifdef JP
{ "",         10000,  190, 111,  5,  8, RACE_VAMPIRE},
#else
		{ "Cayd the Sweet",         10000,  190, 111,  5,  8, RACE_VAMPIRE},
#endif
#ifdef JP
{ "Ťեꥢ",         10000,  190, 110,  6,  8, RACE_NIBELUNG},
#else
		{ "Fulir the Dark",         10000,  190, 110,  6,  8, RACE_NIBELUNG},
#endif
#ifdef JP
        { "Ǥʥɥ",           15000,  200, 116,  6,  9, RACE_DWARF},
#else
        { "Domli the Humble",           15000,  200, 116,  6,  9, RACE_DWARF},
#endif
#ifdef JP
        { "λҥ䥡å",       15000,  220, 111,  4,  9, RACE_IMP},
#else
        { "Yaarjukka Demonspawn",       15000,  220, 111,  4,  9, RACE_IMP},
#endif
#ifdef JP
{ "եɡ",         10000,  190, 111,  5,  8, RACE_HIGH_ELF},
#else
		{ "Gelaraldor the Herbmaster",         10000,  190, 111,  5,  8, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ԥ",         10000,  190, 110,  6,  8, RACE_BARBARIAN},
#else
		{ "Olelaldan the Wise",         10000,  190, 110,  6,  8, RACE_BARBARIAN},
#endif
#ifdef JP
        { "ǥ˥ȤΥե",           15000,  200, 116,  6,  9, RACE_IMP},
#else
        { "Fthoglo the Demonicist",           15000,  200, 116,  6,  9, RACE_IMP},
#endif
#ifdef JP
        { "ϣѻեɥꥢ",       15000,  220, 111,  4,  9, RACE_HALF_ORC},
#else
        { "Dridash the Alchemist",       15000,  220, 111,  4,  9, RACE_HALF_ORC},
#endif
#ifdef JP
{ "ԥͥꥢ",         10000,  190, 111,  5,  8, RACE_CYCLOPS},
#else
		{ "Nelir the Strong",         10000,  190, 111,  5,  8, RACE_CYCLOPS},
#endif
#ifdef JP
{ "ɸꥰʥ",         10000,  190, 110,  6,  8, RACE_HALF_ORC},
#else
		{ "Lignus the Pungent",         10000,  190, 110,  6,  8, RACE_HALF_ORC},
#endif
#ifdef JP
        { "ƥ",           15000,  200, 116,  6,  9, RACE_HOBBIT},
#else
        { "Tilba",           15000,  200, 116,  6,  9, RACE_HOBBIT},
#endif
#ifdef JP
        { "ߥɥå",       15000,  220, 111,  4,  9, RACE_HUMAN},
#else
        { "Myrildric the Wealthy",       15000,  220, 111,  4,  9, RACE_HUMAN},
#endif
#ifdef JP
{ "ʳؼԥޥ",         10000,  190, 111,  5,  8, RACE_HALF_ELF},
#else
		{ "Mauser the Chemist",         10000,  190, 111,  5,  8, RACE_HALF_ELF},
#endif
#ifdef JP
{ "Υ",         10000,  190, 110,  6,  8, RACE_HOBBIT},
#else
		{ "Wizzle the Chaotic",         10000,  190, 110,  6,  8, RACE_HOBBIT},
#endif
#ifdef JP
{ "ߥߥ",           15000,  200, 116,  6,  9, RACE_GNOME},
#else
		{ "Midas the Greedy",           15000,  200, 116,  6,  9, RACE_GNOME},
#endif
#ifdef JP
{ "ϣѻե=ե",       15000,  220, 111,  4,  9, RACE_ELF},
#else
		{ "Ja-Far the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
#endif
#ifdef JP
        { "饫",           15000,  200, 116,  6,  9, RACE_KLACKON},
#else
        { "Kakalrakakal",           15000,  200, 116,  6,  9, RACE_KLACKON},
#endif
#ifdef JP
        { "ϣѻե=",       15000,  220, 111,  4,  9, RACE_ELF},
#else
        { "Jal-Eth the Alchemist",       15000,  220, 111,  4,  9, RACE_ELF},
#endif
	},
	{
		/* Magic Shop - 23 unique names */
#ifdef JP
                { "顼Υ=ѥ",       20000,  200,  110,   7,   8,  RACE_HALF_ELF},
                { "ʤ֥ӥ",         20000,  215,  113,   6,  10,  RACE_GNOME},
                { "ˡȤ",     30000,  200,  110,   7,  10,  RACE_HUMAN},
                { "Ȥ㥯",30000,      175,  110,   5,  11,  RACE_HIGH_ELF},
#else
		{ "Lo Pan the Sorcerer",        20000,  200, 110,  7,  8, RACE_HALF_ELF},
		{ "Buggerby the Great",         20000,  215, 113,  6, 10, RACE_GNOME},
		{ "The Wizard of Yendor",       30000,  200, 110,  7, 10, RACE_HUMAN},
		{ "Rjak the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#endif

#ifdef JP
        { "ѻեɥˡ",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
#else
        { "Skidney the Sorcerer",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
#endif
#ifdef JP
        { "ѻեꥢ",       30000,  200, 110,  7, 10, RACE_HUMAN},
#else
        { "Kyria the Illusionist",       30000,  200, 110,  7, 10, RACE_HUMAN},
#endif
#ifdef JP
        { "ѻե˥å",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#else
        { "Nikki the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#endif
#ifdef JP
        { "ȥ",        15000,  200, 110,  7,  8, RACE_SPRITE},
#else
        { "Solostoran",        15000,  200, 110,  7,  8, RACE_SPRITE},
#endif
#ifdef JP
{ "±",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#else
		{ "Achshe the Tentacled",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#endif
#ifdef JP
        { "²Υ",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
#else
        { "Kaza the Noble",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
#endif
#ifdef JP
        { "Ťե",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#else
        { "Fazzil the Dark",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#endif
#ifdef JP
        { "ʤ륱ɡ",        15000,  200, 110,  7,  8, RACE_DWARF},
#else
        { "Keldorn the Grand",        15000,  200, 110,  7,  8, RACE_DWARF},
#endif
#ifdef JP
{ "ե󥹥ץ",         20000,  215, 113,  6, 10, RACE_HOBBIT},
#else
		{ "Philanthropus",         20000,  215, 113,  6, 10, RACE_HOBBIT},
#endif
#ifdef JP
        { "Υʡ",       30000,  200, 110,  7, 10, RACE_HUMAN},
#else
        { "Agnar the Enchantress",       30000,  200, 110,  7, 10, RACE_HUMAN},
#endif
#ifdef JP
        { "ѻեӥꥢ",       30000,  175, 110,  5, 11, RACE_BEASTMAN},
#else
        { "Buliance the Necromancer",       30000,  175, 110,  5, 11, RACE_BEASTMAN},
#endif
#ifdef JP
        { "ϥᥤΥ饯",        15000,  200, 110,  7,  8, RACE_BEASTMAN},
#else
        { "Vuirak the High-Mage",        15000,  200, 110,  7,  8, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ηüԥޥǥå",         20000,  215, 113,  6, 10, RACE_BEASTMAN},
#else
		{ "Madish the Smart",         20000,  215, 113,  6, 10, RACE_BEASTMAN},
#endif
#ifdef JP
        { "ե֥ܡ",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
#else
        { "Falebrimbor",       30000,  200, 110,  7, 10, RACE_HIGH_ELF},
#endif
#ifdef JP
        { "ե=",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#else
        { "Felil-Gand the Subtle",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#endif
#ifdef JP
        { "ѻե쥴",        15000,  200, 110,  7,  8, RACE_BARBARIAN},
#else
        { "Thalegord the Shaman",        15000,  200, 110,  7,  8, RACE_BARBARIAN},
#endif
#ifdef JP
{ "ȥȥ",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#else
		{ "Cthoaloth the Mystic",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#endif
#ifdef JP
        { "ѻե٥",       30000,  200, 110,  7, 10, RACE_SKELETON},
#else
        { "Ibeli the Illusionist",       30000,  200, 110,  7, 10, RACE_SKELETON},
#endif
#ifdef JP
        { "ѻեإȡ",       30000,  175, 110,  5, 11, RACE_YEEK},
#else
        { "Heto the Necromancer",       30000,  175, 110,  5, 11, RACE_YEEK},
#endif
#ifdef JP
{ "ѻե=ѥ",        20000,  200, 110,  7,  8, RACE_HALF_ELF},
#else
		{ "Lo Pan the Sorcerer",        20000,  200, 110,  7,  8, RACE_HALF_ELF},
#endif
#ifdef JP
{ "ʤ֥ӥ",         20000,  215, 113,  6, 10, RACE_GNOME},
#else
		{ "Buggerby the Great",         20000,  215, 113,  6, 10, RACE_GNOME},
#endif
#ifdef JP
{ "ˡȤ",       30000,  200, 110,  7, 10, RACE_HUMAN},
#else
		{ "The Wizard of Yendor",       30000,  200, 110,  7, 10, RACE_HUMAN},
#endif
#ifdef JP
{ "ѻե㥯",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#else
		{ "Rjak the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#endif
#ifdef JP
        { "ѻեɥˡ",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
#else
        { "Skidney the Sorcerer",        15000,  200, 110,  7,  8, RACE_HALF_ELF},
#endif
#ifdef JP
        { "ѻեꥢ",       30000,  200, 110,  7, 10, RACE_HUMAN},
#else
        { "Kyria the Illusionist",       30000,  200, 110,  7, 10, RACE_HUMAN},
#endif
#ifdef JP
        { "ѻե˥å",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#else
        { "Nikki the Necromancer",       30000,  175, 110,  5, 11, RACE_DARK_ELF},
#endif
#ifdef JP
        { "ȥ",        15000,  200, 110,  7,  8, RACE_SPRITE},
#else
        { "Solostoran",        15000,  200, 110,  7,  8, RACE_SPRITE},
#endif
#ifdef JP
{ "±",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#else
		{ "Achshe the Tentacled",         20000,  215, 113,  6, 10, RACE_MIND_FLAYER},
#endif
	},
	{
		/* Black Market - 32 unique names */
#ifdef JP
                { "꡼=",            20000,  250,  150,  10,   5,  RACE_HALF_TROLL},
                { "֥Υҥȡ",       20000,  250,  150,  10,   5,  RACE_HALF_ORC},
                { "ե󥮿ͥ",           30000,  250,  150,  10,   5,  RACE_HUMAN},
                { "ʤ(?)ȥå",                     30000,  250,  150,  10,   5,  RACE_ELF},
#else
		{ "Gary Gygaz",                 20000,  250, 150, 10,  5, RACE_HALF_TROLL},
		{ "Histor the Goblin",          20000,  250, 150, 10,  5, RACE_HALF_ORC},
		{ "Quark the Ferengi",          30000,  250, 150, 10,  5, RACE_DWARF},
		{ "Topi the Fair(?)",           30000,  250, 150, 10,  5, RACE_HUMAN},
#endif

#ifdef JP
{ "ͥå",             20000,  250, 150, 10,  5, RACE_ZOMBIE},
#else
		{ "Vhassa the Dead",             20000,  250, 150, 10,  5, RACE_ZOMBIE},
#endif
#ifdef JP
{ "΢ڤԥ",          20000,  250, 150, 10,  5, RACE_VAMPIRE},
#else
		{ "Kyn the Treacherous",          20000,  250, 150, 10,  5, RACE_VAMPIRE},
#endif
#ifdef JP
{ "֥ܥ˥",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
#else
		{ "Bubonicus",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ץ饤",           30000,  250, 150, 10,  5, RACE_SPECTRE},
#else
		{ "Corpselight",           30000,  250, 150, 10,  5, RACE_SPECTRE},
#endif
#ifdef JP
{ "˵ѥå",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
#else
		{ "Parrish the Bloodthirsty",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
#endif
#ifdef JP
{ "",          20000,  250, 150, 10,  5, RACE_SKELETON},
#else
		{ "Vile",          20000,  250, 150, 10,  5, RACE_SKELETON},
#endif
#ifdef JP
{ "Υץƥ",          30000,  250, 150, 10,  5, RACE_SKELETON},
#else
		{ "Prentice the Trusted",          30000,  250, 150, 10,  5, RACE_SKELETON},
#endif
#ifdef JP
{ "ʹֻΥꥨ",           30000,  250, 150, 10,  5, RACE_IMP},
#else
		{ "Griella Humanslayer",           30000,  250, 150, 10,  5, RACE_IMP},
#endif
#ifdef JP
{ "󥸥",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
#else
		{ "Angel",                 20000,  250, 150, 10,  5, RACE_VAMPIRE},
#endif
#ifdef JP
{ "Ĥեĥ",          20000,  250, 150, 10,  5, RACE_ZOMBIE},
#else
		{ "Flotsam the Bloated",          20000,  250, 150, 10,  5, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ˡ",          30000,  250, 150, 10,  5, RACE_VAMPIRE},
#else
		{ "Nieval",          30000,  250, 150, 10,  5, RACE_VAMPIRE},
#endif
#ifdef JP
{ "뤤ʥ",           30000,  250, 150, 10,  5, RACE_SPECTRE},
#else
		{ "Anastasia the Luminous",           30000,  250, 150, 10,  5, RACE_SPECTRE},
#endif
#ifdef JP
{ "ѻեƥ", 20000,  250, 150, 10,  5, RACE_DARK_ELF},
#else
		{ "Charity the Necromancer", 20000,  250, 150, 10,  5, RACE_DARK_ELF},
#endif
#ifdef JP
{ "ܥΥץʥ",          20000,  250, 150, 10,  5, RACE_HALF_ORC},
#else
		{ "Pugnacious the Pugilist",          20000,  250, 150, 10,  5, RACE_HALF_ORC},
#endif
#ifdef JP
{ "ʥեåȥ",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
#else
		{ "Footsore the Lucky",          30000,  250, 150, 10,  5, RACE_BEASTMAN},
#endif
#ifdef JP
{ "ؤΥɥꥢ",           30000,  250, 150, 10,  5, RACE_HUMAN},
#else
		{ "Sidria Lighfingered",           30000,  250, 150, 10,  5, RACE_HUMAN},
#endif
#ifdef JP
{ "ʻեꥢ",                 20000,  250, 150, 10,  5, RACE_HOBBIT},
#else
		{ "Riatho the Juggler",                 20000,  250, 150, 10,  5, RACE_HOBBIT},
#endif
#ifdef JP
{ "꤯Υʥå",          20000,  250, 150, 10,  5, RACE_GNOME},
#else
		{ "Janaaka the Shifty",          20000,  250, 150, 10,  5, RACE_GNOME},
#endif
#ifdef JP
{ "ޥ",          30000,  250, 150, 10,  5, RACE_GNOME},
#else
		{ "Cina the Rogue",          30000,  250, 150, 10,  5, RACE_GNOME},
#endif
#ifdef JP
{ "ޥ˥å",           30000,  250, 150, 10,  5, RACE_DRACONIAN},
#else
		{ "Arunikki Greatclaw",           30000,  250, 150, 10,  5, RACE_DRACONIAN},
#endif
#ifdef JP
{ "˳㥨",                 20000,  250, 150, 10,  5, RACE_HUMAN},
#else
		{ "Chaeand the Poor",                 20000,  250, 150, 10,  5, RACE_HUMAN},
#endif
#ifdef JP
{ "±եɡ",          20000,  250, 150, 10,  5, RACE_BARBARIAN},
#else
		{ "Afardorf the Brigand",          20000,  250, 150, 10,  5, RACE_BARBARIAN},
#endif
#ifdef JP
{ "ߥ饶",          30000,  250, 150, 10,  5, RACE_MIND_FLAYER},
#else
		{ "Lathaxl the Greedy",          30000,  250, 150, 10,  5, RACE_MIND_FLAYER},
#endif
#ifdef JP
{ "ե쥦",           30000,  250, 150, 10,  5, RACE_SPRITE},
#else
		{ "Falarewyn",           30000,  250, 150, 10,  5, RACE_SPRITE},
#endif
#ifdef JP
{ "路",                 20000,  250, 150, 10,  5, RACE_NIBELUNG},
#else
		{ "Vosur the Wrinkled",                 20000,  250, 150, 10,  5, RACE_NIBELUNG},
#endif
#ifdef JP
{ "ϥ󥵥ʥ饪",          20000,  250, 150, 10,  5, RACE_AMBERITE},
#else
		{ "Araord the Handsome",          20000,  250, 150, 10,  5, RACE_AMBERITE},
#endif
#ifdef JP
{ "餱ɥե",          30000,  250, 150, 10,  5, RACE_HUMAN},
#else
		{ "Theradfrid the Loser",          30000,  250, 150, 10,  5, RACE_HUMAN},
#endif
#ifdef JP
{ "­Υ롼",           30000,  250, 150, 10,  5, RACE_HALF_OGRE},
#else
		{ "One-Legged Eroolo",           30000,  250, 150, 10,  5, RACE_HALF_OGRE},
#endif
	},
	{
		/* Home */
#ifdef JP
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
		{ "椬",                          0,      100, 100,  0, 99, 99},
#else
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
		{ "Your home",                          0,      100, 100,  0, 99, 99},
#endif

	},

	{
		/* Bookstore - 21 unique names */
#ifdef JP
		{ "ߥɥ", 10000, 175, 108, 4, 12, RACE_HUMAN},
		{ "ԥɥʡ", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "ΩΥ", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
		{ "ǦѤοͥ=", 30000, 140, 105, 6, 12, RACE_ELF},
#else
		{ "Dolaf the Greedy", 10000, 175, 108, 4, 12, RACE_HUMAN},
		{ "Odnar the Sage", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
		{ "Gandar the Neutral", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
		{ "Ro-sha the Patient", 30000, 140, 105, 6, 12, RACE_ELF},
#endif

#ifdef JP
{ "ɥա", 15000, 175, 108, 4, 12, RACE_HUMAN},
#else
		{ "Randolph Carter", 15000, 175, 108, 4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "ȻΥ饤", 15000, 175, 108, 4, 12, RACE_HUMAN},
#else
		{ "Sarai the Swift", 15000, 175, 108, 4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "Τܥɥ", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
#else
		{ "Bodril the Seer", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ۤΥ", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
#else
		{ "Veloin the Quiet", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ؼԤΥ󥷥饹", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
#else
		{ "Vanthylas the Learned", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
#endif
#ifdef JP
{ "ʪ񤭥", 15000, 175, 108, 4, 12, RACE_SKELETON},
#else
		{ "Ossein the Literate", 15000, 175, 108, 4, 12, RACE_SKELETON},
#endif
#ifdef JP
{ "ܤ", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
#else
		{ "Olvar Bookworm", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
#endif
#ifdef JP
{ "", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
#else
		{ "Shallowgrave", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ǥޥ", 30000, 140, 105, 6, 12, RACE_ZOMBIE},
#else
		{ "Death Mask", 30000, 140, 105, 6, 12, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ؼԤΥ", 15000, 175, 108, 4, 12, RACE_MIND_FLAYER},
#else
		{ "Asuunu the Learned", 15000, 175, 108, 4, 12, RACE_MIND_FLAYER},
#endif
#ifdef JP
{ "ͤΥץ", 20000, 120, 105, 6, 16, RACE_ZOMBIE},
#else
		{ "Prirand the Dead", 20000, 120, 105, 6, 16, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ŴΥʡ", 25000, 120, 110, 7, 19, RACE_GOLEM},
#else
		{ "Ronar the Iron", 25000, 120, 110, 7, 19, RACE_GOLEM},
#endif
#ifdef JP
{ "=ߥ", 30000, 140, 105, 6, 12, RACE_ELF},
#else
		{ "Galil-Gamir", 30000, 140, 105, 6, 12, RACE_ELF},
#endif
#ifdef JP
{ "ܿХ", 15000, 175, 108, 4, 12, RACE_KOBOLD},
#else
		{ "Rorbag Book-Eater", 15000, 175, 108, 4, 12, RACE_KOBOLD},
#endif
#ifdef JP
{ "ꥢꥭ", 20000, 120, 105, 6, 16, RACE_KLACKON},
#else
		{ "Kiriarikirk", 20000, 120, 105, 6, 16, RACE_KLACKON},
#endif
#ifdef JP
{ "Ťʤ", 25000, 120, 110, 7, 19, RACE_DWARF},
#else
		{ "Rilin the Quiet", 25000, 120, 110, 7, 19, RACE_DWARF},
#endif
#ifdef JP
{ "ԥ", 30000, 140, 105, 6, 12, RACE_HIGH_ELF},
#else
		{ "Isung the Lord", 30000, 140, 105, 6, 12, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ߥɥ", 10000, 175, 108, 4, 12, RACE_HUMAN},
#else
		{ "Dolaf the Greedy", 10000, 175, 108, 4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "ԥɥʡ", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
#else
		{ "Odnar the Sage", 15000, 120, 105, 6, 16, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ΩΥ", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
#else
		{ "Gandar the Neutral", 25000, 120, 110, 7, 19, RACE_DARK_ELF},
#endif
#ifdef JP
{ "ǦѤοͥ=", 30000, 140, 105, 6, 12, RACE_ELF},
#else
		{ "Ro-sha the Patient", 30000, 140, 105, 6, 12, RACE_ELF},
#endif
#ifdef JP
{ "ɥա", 15000, 175, 108, 4, 12, RACE_HUMAN},
#else
		{ "Randolph Carter", 15000, 175, 108, 4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "Ȼ饤", 15000, 175, 108, 4, 12, RACE_HUMAN},
#else
		{ "Sarai the Swift", 15000, 175, 108, 4, 12, RACE_HUMAN},
#endif
#ifdef JP
{ "Τܥɥ", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
#else
		{ "Bodril the Seer", 20000, 120, 105, 6, 16, RACE_HIGH_ELF},
#endif
#ifdef JP
{ "ۤΥ", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
#else
		{ "Veloin the Quiet", 25000, 120, 110, 7, 19, RACE_ZOMBIE},
#endif
#ifdef JP
{ "ؼԤΥ󥷥饹", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
#else
		{ "Vanthylas the Learned", 30000, 140, 105, 6, 12, RACE_MIND_FLAYER},
#endif
#ifdef JP
{ "ʪ񤭥", 15000, 175, 108, 4, 12, RACE_SKELETON},
#else
		{ "Ossein the Literate", 15000, 175, 108, 4, 12, RACE_SKELETON},
#endif
#ifdef JP
{ "ܤ", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
#else
		{ "Olvar Bookworm", 20000, 120, 105, 6, 16, RACE_VAMPIRE},
#endif
	},

	{
		/* Museum */
#ifdef JP
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
		{ "ʪ",                          0,      100, 100,  0, 99, 99},
#else
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
		{ "Museum",                          0,      100, 100,  0, 99, 99},
#endif

	},

};




/*
 * This table allows quick conversion from "speed" to "energy"
 * The basic function WAS ((S>=110) ? (S-110) : (100 / (120-S)))
 * Note that table access is *much* quicker than computation.
 *
 * Note that the table has been changed at high speeds.  From
 * "Slow (-40)" to "Fast (+30)" is pretty much unchanged, but
 * at speeds above "Fast (+30)", one approaches an asymptotic
 * effective limit of 50 energy per turn.  This means that it
 * is relatively easy to reach "Fast (+30)" and get about 40
 * energy per turn, but then speed becomes very "expensive",
 * and you must get all the way to "Fast (+50)" to reach the
 * point of getting 45 energy per turn.  After that point,
 * furthur increases in speed are more or less pointless,
 * except to balance out heavy inventory.
 *
 * Note that currently the fastest monster is "Fast (+30)".
 *
 * It should be possible to lower the energy threshhold from
 * 100 units to 50 units, though this may interact badly with
 * the (compiled out) small random energy boost code.  It may
 * also tend to cause more "clumping" at high speeds.
 */
byte extract_energy[200] =
{
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* Slow */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* S-50 */     1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	/* S-40 */     2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
	/* S-30 */     2,  2,  2,  2,  2,  2,  2,  3,  3,  3,
	/* S-20 */     3,  3,  3,  3,  3,  4,  4,  4,  4,  4,
	/* S-10 */     5,  5,  5,  5,  6,  6,  7,  7,  8,  9,
	/* Norm */    10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
	/* F+10 */    20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
	/* F+20 */    30, 31, 32, 33, 34, 35, 36, 36, 37, 37,
	/* F+30 */    38, 38, 39, 39, 40, 40, 40, 41, 41, 41,
	/* F+40 */    42, 42, 42, 43, 43, 43, 44, 44, 44, 44,
	/* F+50 */    45, 45, 45, 45, 45, 46, 46, 46, 46, 46,
	/* F+60 */    47, 47, 47, 47, 47, 48, 48, 48, 48, 48,
	/* F+70 */    49, 49, 49, 49, 49, 49, 49, 49, 49, 49,
	/* Fast */    49, 49, 49, 49, 49, 49, 49, 49, 49, 49,
};




/*
 * Base experience levels, may be adjusted up for race and/or class
 */
s32b player_exp[PY_MAX_LEVEL] =
{
	10,
	25,
	45,
	70,
	100,
	140,
	200,
	280,
	380,/*10*/
	500,
	650,
	850,
	1100,
	1400,
	1800,
	2300,
	2900,
	3600,
	4400,/*20*/
	5400,
	6800,
	8400,
	10200,
	12500,
	17500,
	25000,
	35000L,
	50000L,
	75000L,/*30*/
	100000L,
	150000L,
	200000L,
	275000L,
	350000L,
	450000L,
	550000L,
	700000L,
	850000L,
	1000000L,/*40*/
	1250000L,
	1500000L,
	1800000L,
	2100000L,
	2400000L,
	2700000L,
	3000000L,
	3500000L,
	4000000L,
	4500000L,/*50*/
	5000000L
};


s32b player_exp_a[PY_MAX_LEVEL] =
{
	20,
	50,
	100,
	170,
	280,
	430,
	650,
	950,
	1400,/*10*/
	1850,
	2300,
	2900,
	3600,
	4400,
	5400,
	6800,
	8400,
	10400,
	12500,/*20*/
	17500,
	25000,
	35000,
	50000L,
	75000L,
	100000L,
	150000L,
	200000L,
	275000L,
	350000L,/*30*/
	450000L,
	550000L,
	650000L,
	800000L,
	950000L,
	1100000L,
	1250000L,
	1400000L,
	1550000L,
	1700000L,/*40*/
	1900000L,
	2100000L,
	2300000L,
	2550000L,
	2800000L,
	3050000L,
	3300000L,
	3700000L,
	4100000L,
	4500000L,/*50*/
	5000000L
};


/*
 * Player Sexes
 *
 *      Title,
 *      Winner
 */
player_sex sex_info[MAX_SEXES] =
{
	{
#ifdef JP
                "",
                "",
#endif
		"Female",
		"Queen"
	},
	{
#ifdef JP
                "",
                "",
#endif
		"Male",
		"King"
	}
};


/*
 * Player Races
 *
 *      Title,
 *      {STR,INT,WIS,DEX,CON,CHR},
 *      r_dis, r_dev, r_sav, r_stl, r_srh, r_fos, r_thn, r_thb,
 *      hitdie, exp base,
 *      Age (Base, Mod),
 *      Male (Hgt, Wgt),
 *      Female (Hgt, Wgt)
 *      infra,
 *      class-choices
 */
player_race race_info[MAX_RACES] =
{
	{
#ifdef JP
		"ʹ",
#endif
		"Human",

		{  0,  0,  0,  0,  0,  0 },
		0,  0,  0,  0,  0,  10,  0,  0,
		10,  100,
		14,  6,
		72,  6, 180, 25,
		66,  4, 150, 20,
		0,
                0x7FFFFFF,
	},
	{
#ifdef JP
	        "ϡե",
#endif
		"Half-Elf",

		{ -1,  1,  1,  1, -1,  1 },
		2,  3,  2,  1, 6,  11, -1,  5,
		9,  110,
		24, 16,
		66,  6, 130, 15,
		62,  6, 100, 10,
		2,
                0x677E7FF,
	},
	{
#ifdef JP
	        "",
#endif
		"Elf",

		{ -1,  2,  0,  1, -2,  2 },
		5,  6,  4,  2, 8,  12, -5, 15,
		8,  120,
		75, 75,
		60,  4, 100,  6,
		54,  4, 80,  6,
		3,
                0x677E75B,

	},
	{
#ifdef JP
		"ۥӥå",
#endif
		"Hobbit",

		{ -2,  1,  1,  3,  2,  1 },
		15, 12, 10, 5, 12,  15, -10, 20,
		7,  110,
		21, 12,
		36,  3, 60,  3,
		33,  3, 50,  3,
		4,
                0x76FFC0B,
	},
	{
#ifdef JP
		"Ρ",
#endif
		"Gnome",

		{ -1,  2,  0,  2,  1, -1 },
		10, 8, 7,  3, 6,  13, -8, 12,
		8,  120,
		50, 40,
		42,  3, 90,  6,
		39,  3, 75,  3,
		4,
                0x767D60F,
	},
	{
#ifdef JP
		"ɥ",
#endif
		"Dwarf",

		{  2, -2,  2, -2,  2, -1 },
		2,  7,  6,  -1,  7,  10, 15,  0,
		11,  125,
		35, 15,
		48,  3, 150, 10,
		46,  3, 120, 10,
		5,
                0x1890005,
	},
	{
#ifdef JP
		"ϡե",
#endif
		"Half-Orc",

		{  2, -1,  0,  0,  1, -2 },
		-3, -3, -2,  -1,  0, 7, 12, -5,
		10,  110,
		11,  4,
		66,  1, 150,  5,
		62,  1, 120,  5,
		3,
                0x5D8818D,
	},
	{
#ifdef JP
		"ϡեȥ",
#endif
		"Half-Troll",

		{ 4, -4, -1, -3,  3, -3 },
		-5, -8, -5, -2,  -1, 5, 20, -10,
		12,  125,
		20, 10,
		96, 10, 250, 50,
		84,  8, 225, 40,
		3,
                0x0880005,
	},
	{
#ifdef JP
                "Х饤",
#endif
		"Amberite",

		{  1,  2,  2,  2,  3,  2 },
		4,  5,  3,  2, 3, 13, 15, 10,
		10,  225,
		50, 50,
		82, 5, 190, 20,
		78,  6, 180, 15,
		0,
                0x7FFF7FF,
	},
	{
#ifdef JP
                "ϥ",
#endif
		"High-Elf",

		{  1,  3,  -1,  3,  1,  3 },
		4,  13, 12,  4,  3, 14, 10, 25,
		10,  200,
		100, 30,
		90, 10, 190, 20,
		82, 10, 180, 15,
		4,
		0x777E75B,
	},
	{
#ifdef JP
		"ڿ",
#endif
		"Barbarian",

		{ 3, -2,  -1,  1,  2, 0 },
		-2, -10, 2,  -1,  1, 7, 12, 10,
		11, 120,
		14, 8,
		82, 5, 200, 20,
		78,  6, 190, 15,
		0,
		0x5C0A09D,
	},
	{
#ifdef JP
		"ϡե",
#endif
		"Half-Ogre",

		{ 3, -2, 0, -1, 3, -2 },
		-3, -5, -3, -2, -1, 5, 20, 0,
		12,  145,
		40, 10,
		92, 10, 255, 60,
		80,  8, 235, 60,
		3,
		0x0A80407,
	},
	{
#ifdef JP
		"Ⱦ",
#endif
		"Half-Giant",

		{ 4, -2, -2, -2, 3, -2 },
		-6, -8, -3, -2, -1, 5, 25, 5,
		13, 160,
		40, 10,
		100,10, 255, 65,
		80, 10, 240, 64,
		3,
		0x0880011,
	},
	{
#ifdef JP
		"Ⱦ",
#endif
		"Half-Titan",

		{ 5, 1, 2, -2, 3, 1 },
		-5, 5, 1, -2, 1, 8, 25, 0,
		14, 255,
		100,30,
		111, 11, 255, 86,
		99, 11, 250, 86,
		0,
		0x23D4727,
	},
	{
#ifdef JP
		"ץ",
#endif
		"Cyclops",

		{ 4, -3, -2, -3, 4, -3 },
		-4, -5, -3, -2, -2, 5, 20, 12,
		13, 150,
		50, 24,
		92, 10, 255, 60,
		80,  8, 235, 60,
		1,
		0x0888005,
	},
	{
#ifdef JP
		"",
#endif
		"Yeek",

		{ -2, 1, 1, 1, -2, -4 },
		2, 4, 6, 3, 5, 15, -5, -5,
		7, 100,
		14, 3,
		50,  3, 90,  6,
		50,  3, 75,  3,
		2,
		0x667360F,
	},
	{
#ifdef JP
		"å",
#endif
		"Klackon",

		{ 2, -1, -1, 1, 2, -1 },
		10, 5, 3, 0, -1, 10, 5, 5,
		12, 190,
		20, 3,
		60,  3, 80,  4,
		54,  3, 70,  4,
		2,
		0x04D8011,
	},
	{
#ifdef JP
		"ܥ",
#endif
		"Kobold",

		{ 1, -1, 0, 1, 0, -2 },
		-2, -3, -1, -1, 1, 8, 10, -8,
		9, 125,
		11,  3,
		60,  1, 130,  5,
		55,  1, 100,  5,
		3,
		0x444A009,
	},
	{
#ifdef JP
		"ˡ٥",
#endif
		"Nibelung",

		{ 1, -1, 2, 0, 2, -2 },
		3, 5, 6, 1, 5, 10, 9, 0,
		11, 170,
		40, 12,
		43,  3, 92,  6,
		40,  3, 78,  3,
		5,
		0x569040F,
	},
	{
#ifdef JP
		"",
#endif
		"Dark-Elf",

		{ -1, 3, 2, 2, -2, 1 },
		5, 10, 12, 3, 8, 12, -5, 10,
		9, 150,
		75, 75,
		60,  4, 100,  6,
		54,  4, 80,  6,
		5,
		0x677C7DF,
	},
	{
#ifdef JP
		"ɥ饳˥",
#endif
		"Draconian",

		{ 2, 1, 1, 1, 2, -1 },
		-2, 5, 2, 0, 1, 10, 5, 5,
		11, 220,
		75, 33,
		76,  1, 160,  5,
		72,  1, 130,  5,
		2,
		0x7FFE757,
	},
	{
#ifdef JP
		"ޥɥե쥢",
#endif
		"Mindflayer",

		{ -3, 4, 4, 0, -2, -3 },
		10, 15, 9, 2, 5, 12, -10, -5,
		9, 140,
		100, 25,
		68,  6, 142, 15,
		63,  6, 112, 10,
		4,
		0x2334746,
	},
	{
#ifdef JP
		"",
#endif
		"Imp",

		{ 0, -1, -1, 1, 2, -1 },
		-3, 2, -1, 1, -1, 10, 5, -5,
		10, 115,
		13,  4,
		68,  1, 150,  5,
		64,  1, 120,  5,
		3,
		0x6B537CB,
	},
	{
#ifdef JP
		"",
#endif
		"Golem",

		{ 4, -5, -5, -2, 4, -2 },
		-5, -5, 6, -1, -1, 8, 20, 0,
		12, 200,
		1, 100,
		66,  1, 200,  6,
		62,  1, 180,  6,
		4,
		0x0800001,
	},
	{
#ifdef JP
		"",
#endif
		"Skeleton",

		{ 0, 1, -2, 0, 1, -2 },
		-5, 0, 3, -1, -1, 8, 10, 0,
		10, 145,
		100, 35,
		72,  6, 50, 5,
		66,  4, 50, 5,
		2,
		0x234070F,
	},
	{
#ifdef JP
		"",
#endif
		"Zombie",

		{ 2, -6, -6, 1, 4, -3 },
		-5, -5, 5, -1, -1, 5, 15, 0,
		13, 150,
		100, 30,
		72, 6, 100, 25,
		66, 4, 100, 20,
		2,
		0x0800001,
	},
	{
#ifdef JP
		"۷쵴",
#endif
		"Vampire",

		{ 3, 3, -1, -1, 1, 2 },
		4, 8, 6, 4, 1, 8, 5, 0,
		11, 200,
		100, 30,
		72,  6, 180, 25,
		66,  4, 150, 20,
		5,
		0x67DC7FF,
	},
	{
#ifdef JP
		"ͩ",
#endif
		"Spectre",

		{ -5, 4, -1, 2, 0, -3 },
		10, 15, 12, 5, 5, 14, -15, -5,
		7, 180,
		100, 30,
		72, 6, 100, 25,
		66, 4, 100, 20,
		5,
		0x631474A,
	},
	{
#ifdef JP
		"",
#endif
		"Sprite",

		{ -4, 3, 3, 3, -2, 2 },
		10, 8, 6, 4, 10, 10, -12, 0,
		7, 145,
		50, 25,
		32,  2, 75,  2,
		29,  2, 65,  2,
		4,
		0x623F65E,
	},
	{
#ifdef JP
		"ÿ",  
#endif
		"Beastman",

		{ 2, -2, -1, -1, 2, -2 },
		-5, -2, -1, -1, -1, 5, 12, 5,
		11, 140,
		14, 6,
		65,  6, 150, 20,
		61,  6, 120, 15,
		0,
		0x57887CF,
	},
	{
#ifdef JP
		"",
#endif
		"Ent",
		{ 2,  0,  2, -3,  2,  0 },
		 -5,  2,  5,  -1, 0, 9,  15, -5,
		 12, 140,
		120,  60,
		111, 11, 255, 50,
		99, 11, 250, 45,
		  0,
                0x0010005,
	},
	{
#ifdef JP
                "륳",
#endif
		"Archon",

		{  2,  0,  4,  1,  2,  3 },
		0,  12,  8,  2, 2, 11, 10, 10,
		11,  235,
		150, 150,
		82, 5, 190, 20,
		78,  6, 180, 15,
		3,
                0x779F777,
	},
	{
#ifdef JP
                "Х",
		"Balrog",
#else
		"Balrog",
#endif

		{  4,  2,  -10,  2,  3,  -5 },
		-3,  12, 15,  -2,  1, 8, 20, 0,
		12,  250,
		255, 255,
		100,10, 255, 65,
		80, 10, 240, 64,
		5,
                0x7EDC4DB,
	},
	{
#ifdef JP
                "ɥʥ",
#endif
		"Dunadan",

		{  1,  2,  2,  2,  3,  2 },
		4,  5,  3,  2, 3, 13, 15, 10,
		10,  180,
		50, 20,
		82, 5, 190, 20,
		78,  6, 180, 15,
		0,
                0x7FFF7FF,
	},
	{
#ifdef JP
		"ƥե꡼",
#endif
		"Shadow-Fairy",
		{-2,  2,  2,  1, -1,  0 },
		  7,  8,  0,  6, 12, 15, -10, -5,
		  7, 120,
		200, 100,
		 80,  8, 90, 20,
		 73,  8, 80, 15,
		  4,
		0x633C7DF,
	},
	{
#ifdef JP
		"",
#endif
		"Kutar",

		{  0,  -1,  -1,  1,  2,  3 },
		-2,  5,  5,  5,  -2,  6,  0,  -5,
		11,  140,
		14,  6,
		48,  6, 150, 25,
		44,  4, 130, 20,
		0,
                0x418B7AD,
	},
	{
#ifdef JP
		"ɥ",
#endif
		"Android",

		{ 4, -5, -5, 0, 4, -2 },
		0, -5, 0, -2, 3, 14, 20, 10,
		13, 200,
		1, 100,
		72, 12, 240, 64,
		66, 12, 220, 64,
		0,
		0x0800001,
	},
};


/*
 * Player Classes
 *
 *      Title,
 *      {STR,INT,WIS,DEX,CON,CHR},
 *      c_dis, c_dev, c_sav, c_stl, c_srh, c_fos, c_thn, c_thb,
 *      x_dis, x_dev, x_sav, x_stl, x_srh, x_fos, x_thn, x_thb,
 *      HD, Exp, pet_upkeep_div
 */
player_class class_info[MAX_CLASS] =
{
	{
#ifdef JP
		"",
#endif
		"Warrior",

		{ 4, -2, -2, 2, 2, -1},
		25, 18, 31, 1,  14, 2, 70, 55,
		12, 7,  10, 0,  0,  0,  30, 30,
		9,  0, 40
	},

	{
#ifdef JP
		"ᥤ",
#endif
		"Mage",

		{-4, 3, 0, 1, -2, 1},
		30, 40, 38, 3,  16, 20, 34, 20,
		7,  15, 11,  0,  0,  0,  6, 7,
		0, 30, 30
	},

	{
#ifdef JP
		"ץ꡼",
#endif
		"Priest",

		{-1, -3, 3, -1, 0, 2},
		25, 35, 40, 2,  16, 8, 48, 35,
		7,  11, 12, 0,  0,  0, 13, 11,
		2, 20, 35
	},

	{
#ifdef JP
		"±",
#endif
		"Rogue",

		{ 2, 1, -2, 3, 1, -1},
		45, 37, 36, 5, 32, 24, 60, 66,
		15, 12, 10, 0,  0,  0, 21, 18,
		6, 25, 40
	},

	{
#ifdef JP
		"󥸥㡼",
#endif
		"Ranger",

		{ 2, 2, 0, 1, 1, 1},
		30, 37, 36, 3,  24, 16, 56, 72,
		8,  11, 10, 0,  0,  0,  18, 28,
		4, 40, 35
	},

	{
#ifdef JP
		"ѥǥ",
#endif
		"Paladin",

		{ 3, -3, 1, 0, 2, 2},
		20, 24, 34, 1,  12, 2, 68, 40,
		7,  10, 11, 0,  0,  0,  21, 18,
		6, 35, 40
	},

	{
#ifdef JP
		"ˡ",
#endif
		"Warrior-Mage",

		{ 2, 2, 0, 1, 0, 1},
		30, 35, 36, 2,  18, 16, 50, 25,
		7,  10, 10, 0,  0,  0,  15, 11,
		4, 40, 35
	},

	{
#ifdef JP
		"٤",
#endif
		"Chaos-Warrior",

		{ 2, 1, -1, 0, 2, -2},
		20, 25, 34, 1,  14, 12, 65, 40,
		7,  11, 10, 0,  0,  0,  20, 17,
		6, 25, 40
	},

	{
#ifdef JP
		"",
#endif
		"Monk",

		{ 2, -1, 1, 3, 2, 1},
		45, 34, 36, 5, 32, 24, 64, 60,
		15, 11, 10, 0,  0,  0, 18, 18,
		6, 30, 35
	},

	{
#ifdef JP
		"Ķǽϼ",
#endif
		"Mindcrafter",

		{-1, 0, 3, -1, -1, 2},   /* note: spell stat is Wis */
		30, 33, 38, 3,  22, 16, 50, 40,
		10, 11, 10, 0,   0,  0, 14, 18,
		2, 25, 35
	},

	{
#ifdef JP
		"ϥ=ᥤ",
#endif
		"High-Mage",

		{-4, 4, 0, 0, -2, 1},
		30, 40, 38, 3,  16, 20, 34, 20,
		7,  15, 11,  0,  0,  0,  6, 7,
		0, 30, 25
	},

	{
#ifdef JP
		"Ѹ",
#endif
		"Tourist",
		{ -1, -1, -1, -1, -1, -1},
		15, 18, 28, 1, 12, 2, 40, 20,
		5, 7, 9, 0,  0,  0,  11, 11,
		0, -30, 40
	},

	{
#ifdef JP
		"Τޤͻ",
#endif
		"Imitator",
		{ 0, 1, -1, 2, 0, 1},
		25, 30, 36, 2,  18, 16, 60, 50,
		7,  10,  10, 0,  0,  0,  18, 20,
		5, 10, 20
	},

	{
#ifdef JP
		"ûȤ",
#endif
		"BeastMaster",
		{ 1, -1, -1, 1, 0, 2},
		20, 25, 32, 2,  18, 16, 52, 63,
		7,  10, 10, 0,  0,  0,  14, 25,
		3, 20, 10
	},

	{
#ifdef JP
		"ڥޥ",
#endif
		"Sorcerer",

		{-5, 6, -2, 2, 0, -2},
		30, 48, 75, 2,  12, 22,  0, 0,
		 7, 18, 13, 0,  0,  0,  0, 0,
		4, 60, 25
	},

	{
#ifdef JP
		"㡼",
#endif
		"Archer",

		{ 2, -1, -1, 2, 1, 0},
		38, 24, 35, 4,  24, 16, 56, 82,
		12, 10, 10, 0,  0,  0,  18, 36,
		6, 10, 40
	},

	{
#ifdef JP
		"ƻѻ",
#endif
		"Magic-Eater",

		{-1, 2, 1, 2, -2, 1},
		25, 42, 36, 2,  20, 16, 48, 35,
		7,  16, 10,  0,  0,  0, 13, 11,
		3, 30, 30
	},

	{
#ifdef JP
		"ͷ",
#endif
                "Bard",              /* Note : spell stat is Charisma */
                {-2, 1, 2, -1, -2, 4},
                20, 33, 34, -5, 16, 20, 34, 20,
                8,  13, 11, 0,  0,  0,  10, 8,
                2, 40, 25
	},

	{
#ifdef JP
		"ƻ",
#endif
		"Red-Mage",

		{ 2, 2, -1, 1, 0, -1},
		20, 34, 34, 1,  16, 10, 56, 25,
		7,  11, 11, 0,  0,  0,  18, 11,
		4, 40, 40
	},

	{
#ifdef JP
		"Ѳ",
#endif
		"Samurai",

		{ 3, -2, 1, 2, 1, 0},
		25, 18, 32, 2,  16, 6, 70, 40,
		12, 7,  10, 0,  0,  0,  23, 18,
		6,  30, 40
	},

	{
#ifdef JP
		"ѻ",
#endif
		"ForceTrainer",

		{ 0, -1, 3, 2, 1, 1},
		30, 34, 38, 4, 32, 24, 50, 40,
		10, 11, 11, 0,  0,  0, 14, 15,
		2, 35, 40
	},

	{
#ifdef JP
		"ƻ",
#endif
		"Blue-Mage",

		{-4, 4, -1, 1, -2, -1},
		30, 40, 36, 3,  20, 16, 40, 25,
		7,  16, 11,  0,  0,  0,  6, 7,
		2, 30, 35
	},

	{
#ifdef JP
		"ʼ",
#endif
		"Cavalry",
		{ 2, -2, -2, 2, 2, 0},
		20, 18, 32, 1,  16, 10, 60, 66,
		10,  7, 10, 0,  0,  0,  22, 26,
		5, 20, 35
	},

	{
#ifdef JP
		"",
#endif
		"Berserker",

		{ 8, -20, -20, 4, 4, -5},
		-100, -1000, -200, -100,  -100, -100, 120, -2000,
		0, 0,  0, 0,  0,  0,  50, 0,
		11,  60, 255
	},

	{
#ifdef JP
		"",
#endif
		"Weaponsmith",

		{ 3, -1, -1, 1, 0, -1},
		30, 28, 28, 1,  20, 10, 60, 45,
		10, 10,  10, 0,  0,  0,  21, 15,
		6,  30, 40
	},
	{
#ifdef JP
		"Ȥ",
#endif
		"Mirror-Master",

		{ -2,  3, 1, -1, -2, 1},
		30, 33, 40, 3, 14, 16, 34,30,
		10, 11, 12, 0,  0,  0,  6,10,
		2,  30, 30
	},
	{
#ifdef JP
		"Ǧ",
#endif
		"Ninja",

		{ 0,  -1, -1, 3, 2, -1},
		45, 24, 36, 8, 48, 32, 70,66,
		15, 10, 10, 0,  0,  0, 25,18,
		2,  20, 40
	},
};

player_seikaku seikaku_info[MAX_SEIKAKU] =
{
	{
#ifdef JP
		"դĤ",
#endif
		"Ordinary",
		{  0,  0,  0,  0,  0,  0 },
		0,  0,  0,  0,  0,  0,  0,  0,
		0, 1, 0
	},

	{
#ifdef JP
		"餸ޤ",
#endif
		"Mighty",
		{  2,  -2,  -1,  0,  1,  0 },
		-5,  -5,  -3,  -1,  -2,  -2,  10,  0,
		1, 1, 0
	},

	{
#ifdef JP
		"",
#endif
		"Shrewd",
		{  -2,  2,  0,  1,  -1,  -1 },
		3,  8,  2,  0,  -2,  5,  -8,  -5,
		-1, 1, 0
	},

	{
#ifdef JP
		"碌",
#endif
		"Pious",
		{  0,  -1,  2,  -1,  0,  1 },
		-5,  2,  4,  -1,  3,  -2,  -3,  -6,
		0, 1, 0
	},

	{
#ifdef JP
		"Фä",
#endif
		"Nimble",
		{  -1,  1,  -1,  2,  -1,  -1 },
		7,  2,  -1,  1,  5,  5,  0,  10,
		0, 0, 0
	},

	{
#ifdef JP
		"Τ餺",
#endif
		"Fearless",
		{  2,  1,  1,  -1,  -1,  0 },
		-5,  5,  -2,  0,  2,  -2,  10,  10,
		-1, 1, 0
	},

	{
#ifdef JP
		"Хå",
#endif
		"Combat",
		{  1,  -1,  -2,  2,  0,  1 },
		-2,  -3,  -3,  0,  -1,  2,  5,  5,
		0, 0, 0
	},

	{
#ifdef JP
		"ʤޤ",
#endif
		"Lazy",
		{  -2,  -2,  -2,  -2,  -2,  -2 },
		-5,  -5,  -3,  -1,  -4,  -2,  -8,  -8,
		-1, 1, 0
	},

	{
#ifdef JP
		"",
#endif
		"Sexy",
		{  1,  1,  1,  1,  1,  3 },
		10,  5,  3,  0,  4,  2,  10,  10,
		0, 1, 1
	},

	{
#ifdef JP
		"åޥ",
#endif
		"Lucky",
		{  -2,  -2,  -2,  -2,  -2,  2 },
		10,  7,  3,  2, 10,  8,  15,  15,
		0, 1, 2
	},

	{
#ifdef JP
		"ޤŤ褤",
#endif
		"Patient",
		{  -1,  -1,  1,  -2,  2,  0 },
		-5,  -3,  3,  1,  0,  -3,  -6,  -6,
		1, 0, 0
	},

	{
#ifdef JP
		"",
#endif
		"Munchkin",
		{ 10,  10,  10,  10,  10,  10 },
		 20, 40, 30, 10, 40, 40,  80, 80,
		 15, 1, 0
	},
};



player_race mimic_info[] =
{
	{
#ifdef JP
		"[ɸ]",
#endif
		"Default",

		{  0,  0,  0,  0,  0,  0 },
		0,  0,  0,  0,  0,  10,  0,  0,
		10,  100,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		0,
                0x000000,
	},
	{
#ifdef JP
		"[]",
#endif
		"[Demon]",

		{  5,  3,  2,  3,  4,  -6 },
		-5,  18, 20, -2,  3,  10, 40, 20,
		12,  0,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		5,
                0x000003,
	},
	{
#ifdef JP
		"[Ⲧ]",
#endif
		"[Demon lord]",

		{  20,  20,  20,  20,  20,  20 },
		20,  20, 25, -2,  3,  10, 70, 40,
		14,  0,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		20,
                0x000003,
	},
	{
#ifdef JP
		"[۷쵴]",
#endif
		"[Vampire]",

		{ 4, 4, 1, 1, 2, 3 },
		6, 12, 8, 6, 2, 12, 30, 20,
		11,  0,
		0,  0,
		0,  0, 0, 0,
		0,  0, 0, 0,
		5,
                0x000005,
	},
};


/*
 * Hack -- the spell information table.
 *
 *   Class Name
 *
 *   Spell Book
 *   Spell Xtra
 *
 *   Spell Stat,
 *   Spell Type,
 *
 *   Spell Level,
 *   Spell Encumbrance,
 *
 *   Array of { Lev, Mana, Fail, Exp/Lev }
 */
player_magic magic_info[MAX_CLASS] =
{
	{
		/*** Warrior ***/

		0,
		0x00,

		A_STR,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Mage ***/

		TV_SORCERY_BOOK,
		0x05,

		A_INT,
		0,

		1,
		430,

		{
			{
				/* Mage: Life magic */

				{ 1, 1, 30, 4 },
				{ 3, 2, 35, 4 },
				{ 4, 3, 35, 4 },
				{ 5, 5, 35, 4 },
				{ 7, 7, 35, 4 },
				{ 9, 8, 40, 4 },
				{ 12, 12, 40, 3 },
				{ 15, 14, 45, 3 },

				{ 16, 16, 45, 4},
				{ 17, 17, 50, 4},
				{ 18, 18, 50, 4},
				{ 19, 19, 50, 4},
				{ 20, 20, 50, 4},
				{ 23, 23, 50, 4},
				{ 30, 30, 55, 5},
				{ 35, 70, 75, 5},

				{ 26, 30, 50, 75 },
				{ 28, 25, 70, 150 },
				{ 33, 33, 60, 75 },
				{ 35, 35, 60, 75 },
				{ 35, 35, 70, 75 },
				{ 35, 55, 80, 115 },
				{ 39, 40, 80, 125 },
				{ 46, 70, 80, 150 },

				{ 9, 9, 50, 40 },
				{ 25, 25, 50, 50 },
				{ 35, 55, 80, 115 },
				{ 42, 100, 80, 225 },
				{ 45, 90, 80, 115 },
				{ 48, 50, 80, 100 },
				{ 49, 100, 80, 250 },
				{ 50, 100, 80, 250 }
			},

			/* Mage: Sorcery */

			{
				{ 1, 1, 23, 4 },
				{ 1, 2, 24, 4 },
				{ 3, 3, 25, 1 },
				{ 3, 3, 30, 1 },
				{ 4, 4, 30, 1 },
				{ 5, 5, 35, 5 },
				{ 6, 5, 30, 4 },
				{ 7, 7, 75, 9 },

				{ 9, 7, 75, 8 },
				{ 10, 7, 75, 8 },
				{ 11, 7, 75, 7 },
				{ 13, 7, 50, 6 },
				{ 18, 12, 60, 8 },
				{ 22, 12, 60, 8 },
				{ 28, 20, 70, 15 },
				{ 33, 30, 75, 20 },

				{ 3, 3, 25, 15 },
				{ 10, 10, 80, 40 },
				{ 14, 10, 60, 25 },
				{ 18, 30, 70, 40 },
				{ 20, 18, 85, 50 },
				{ 20, 18, 60, 25 },
				{ 25, 25, 75, 19 },
				{ 28, 30, 80, 100 },

				{ 10, 10, 40, 20 },
				{ 25, 25, 75, 70 },
				{ 30, 40, 80, 120 },
				{ 33, 35, 60, 100 },
				{ 40, 40, 80, 100 },
				{ 42, 50, 90, 175 },
				{ 43, 40, 55, 200 },
				{ 45, 70, 85, 250 },
			},

			/* Mage: Nature Magic */

			{
				{ 1, 1, 23, 4 },
				{ 3, 3, 25, 3 },
				{ 3, 3, 25, 1 },
				{ 4, 4, 35, 4 },
				{ 4, 4, 50, 5 },
				{ 4, 5, 50, 5 },
				{ 5, 5, 50, 5 },
				{ 5, 5, 35, 4 },

				{ 5, 5, 40, 6 },
				{ 5, 5, 30, 6 },
				{ 7, 6, 45, 6 },
				{ 7, 6, 40, 6 },
				{ 9, 6, 30, 5 },
				{ 19, 12, 55, 8 },
				{ 25, 25, 90, 50 },
				{ 40, 60, 90, 50 },

				{ 9, 12, 40, 44 },
				{ 10, 12, 75, 120 },
				{ 15, 20, 85, 60 },
				{ 20, 22, 80, 40 },
				{ 30, 30, 90, 100 },
				{ 37, 40, 90, 200 },
				{ 38, 45, 75, 200},
				{ 40, 90, 90, 250 },


				{ 20, 18, 60, 25 },
				{ 23, 23, 80, 50 },
				{ 25, 25, 75, 29 },
				{ 30, 27, 75, 35 },
				{ 35, 30, 85, 65 },
				{ 37, 35, 90, 100 },
				{ 40, 90, 95, 250 },
				{ 40, 75, 65, 150 }
			},

				/* Mage: Chaos Magic */

			{
				{ 1, 1, 20, 4 },
				{ 1, 2, 22, 4 },
				{ 2, 2, 25, 4 },
				{ 5, 5, 30, 1 },
				{ 9, 6, 50, 1 },
				{ 13, 9, 45, 6 },
				{ 14, 9, 45, 6 },
				{ 15, 9, 35, 5 },

				{ 17, 10, 25, 5 },
				{ 19, 12, 45, 9 },
				{ 21, 13, 45, 10 },
				{ 23, 15, 50, 11 },
				{ 25, 16, 50, 12 },
				{ 25, 18, 60, 8 },
				{ 30, 20, 80, 15 },
				{ 35, 40, 85, 40 },

				{ 11, 7, 45, 9 },
				{ 15, 15, 80, 35 },
				{ 16, 14, 80, 35 },
				{ 25, 25, 85, 100 },
				{ 30, 25, 85, 150 },
				{ 40, 41, 80, 150 },
				{ 45, 90, 80, 250 },
				{ 47, 100, 90, 250 },

				{ 20, 20, 66, 8 },
				{ 35, 32, 85, 35 },
				{ 37, 34, 75, 40 },
				{ 39, 42, 85, 100 },
				{ 42, 50, 85, 250 },
				{ 45, 58, 85, 200 },
				{ 47, 75, 80, 200 },
				{ 49, 100, 85, 250 }
			},

				/* Mage: Death Magic */

			{
				{ 1, 1, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 3, 3, 27, 3 },
				{ 5, 5, 30, 4 },
				{ 7, 10, 75, 6 },
				{ 9, 9, 30, 4 },
				{ 10, 10, 30, 4 },

				{ 12, 12, 40, 5 },
				{ 13, 12, 30, 4 },
				{ 18, 15, 50, 10 },
				{ 24, 21, 60, 30 },
				{ 30, 75, 80, 30 },
				{ 32, 30, 60, 16 },
				{ 36, 35, 80, 70 },
				{ 39, 30, 95, 25 },

				{ 10, 20, 80, 180 },
				{ 10, 15, 80, 30 },
				{ 11, 11, 30, 15 },
				{ 30, 25, 75, 50 },
				{ 34, 90, 70, 90 },
				{ 36, 35, 60, 125 },
				{ 38, 35, 70, 40 },
				{ 40, 40, 70, 200 },

				{ 20, 20, 75, 50 },
				{ 25, 66, 95 , 250 },
				{ 30, 40, 95, 250 },
				{ 35, 40, 80, 100 },
				{ 42, 50, 75, 150 },
				{ 43, 75, 80, 100 },
				{ 45, 95, 90, 250 },
				{ 47, 100, 90, 250 }
			},

				/* Mage: Trump Magic */

			{
				{ 1, 1, 25, 3 },
				{ 3, 3, 25, 4 },
				{ 5, 5, 37, 8 },
				{ 6, 6, 40, 8 },
				{ 7, 7, 20, 4 },
				{ 11, 12, 30, 6 },
				{ 14, 15, 30, 5 },
				{ 18, 20, 30, 8 },

				{ 20, 20, 40, 8 },
				{ 23, 20, 35, 9 },
				{ 25, 24, 30, 8 },
				{ 30, 28, 40, 12 },
				{ 33, 30, 35, 10 },
				{ 35, 33, 45, 12 },
				{ 40, 35, 40, 15 },
				{ 42, 40, 35, 12 },

				{ 22, 35, 70, 40 },
				{ 24, 24, 35, 25 },
				{ 26, 26, 35, 30 },
				{ 30, 25, 35, 9 },
				{ 30, 30, 35, 35 },
				{ 35, 70, 40, 100 },
				{ 40, 100, 45, 250 },
				{ 45, 100, 45, 200 },

				{ 30, 30, 30, 50 },
				{ 35, 50, 45, 100 },
				{ 36, 55, 40, 150 },
				{ 39, 80, 40, 150 },
				{ 41, 45, 70, 100 },
				{ 47, 100, 40, 150 },
				{ 48, 100, 40, 200 },
				{ 49, 100, 40, 220 }
			},

				/* Mage: Arcane Magic */

			{
				{ 1, 1, 20, 4 },
				{ 1, 1, 33, 5 },
				{ 1, 1, 33, 4 },
				{ 2, 1, 33, 5 },
				{ 2, 2, 33, 5 },
				{ 4, 4, 40, 6 },
				{ 5, 5, 33, 7 },
				{ 6, 5, 44, 5 },

				{ 7, 6, 40, 7 },
				{ 8, 8, 60, 7 },
				{ 9, 8, 50, 6 },
				{ 9, 9, 50, 6 },
				{ 9, 9, 50, 6 },
				{ 11, 10, 50, 6 },
				{ 12, 12, 50, 5 },
				{ 13, 12, 50, 5 },

				{ 14, 12, 50, 5 },
				{ 15, 12, 50, 5 },
				{ 16, 14, 33, 6 },
				{ 18, 15, 50, 8 },
				{ 20, 20, 60, 25 },
				{ 22, 16, 60, 9 },
				{ 25, 18, 60, 9 },
				{ 28, 20, 70, 12 },

				{ 28, 20, 60, 13 },
				{ 31, 25, 70, 30 },
				{ 35, 35, 80, 25 },
				{ 40, 30, 70, 25 },
				{ 41, 30, 66, 30 },
				{ 42, 30, 80, 40 },
				{ 45, 50, 70, 50 },
				{ 49, 100, 80, 200 }
			},

			/* Mage: Craft Magic */

			{

				{  1,  1, 20,   4},
				{  2,  2, 20,   4},
				{  3,  3, 30,   3},
				{  5,  5, 35,   5},
				{  6,  5, 35,   5},
				{  8,  6, 45,   6},
				{  9,  5, 35,   5},
				{ 10,  5, 35,   5},

				{ 12, 12, 45,   4},
				{ 15, 14, 40,   4},
				{ 17, 15, 60,   6},
				{ 20, 17, 75,  80},
				{ 23, 18, 85,  50},
				{ 26, 22, 50,   4},
				{ 29, 30, 60,   5},
				{ 38, 65, 95,  50},

				{ 12, 12, 45,  12},
				{ 14, 10, 75, 120},
				{ 20, 20, 85,  60},
				{ 24, 12, 60,  10},
				{ 30, 55, 80,  50},
				{ 34, 70, 80, 150},
				{ 41, 80, 90, 200},
				{ 45, 55, 70, 200},

				{ 14, 14, 65,  50},
				{ 30, 32, 75, 150},
				{ 36, 45, 90, 200},
				{ 38, 80, 95, 200},
				{ 38,100, 95, 200},
				{ 40,100, 90, 250},
				{ 43,100, 70, 250},
				{ 47, 70, 80, 250}
			},

				/* Mage: Daemon Magic */

			{
				{ 1, 1, 20, 4 },
				{ 2, 1, 22, 1 },
				{ 3, 2, 25, 4 },
				{ 4, 5, 30, 4 },
				{ 7, 5, 45, 4 },
				{ 10, 7, 40, 6 },
				{ 11, 9, 35, 5 },
				{ 13, 13, 40, 10 },

				{ 15, 14, 45, 9 },
				{ 17, 15, 60, 10 },
				{ 20, 18, 50, 11 },
				{ 23, 15, 50, 12 },
				{ 25, 16, 50, 12 },
				{ 30, 75, 80, 30 },
				{ 32, 28, 70, 15 },
				{ 36, 70, 85, 40 },

				{ 12, 12, 45, 9 },
				{ 15, 20, 60, 20 },
				{ 25, 22, 65, 35 },
				{ 34, 30, 80, 100 },
				{ 35, 40, 80, 100 },
				{ 37, 35, 85, 100 },
				{ 40, 40, 85, 40 },
				{ 46, 80, 80, 250 },

				{  9,  9, 50, 8 },
				{ 26, 25, 85, 35 },
				{ 36, 35, 80, 200 },
				{ 41, 50, 85, 100 },
				{ 43,100, 90, 250 },
				{ 45, 58, 85, 200 },
				{ 47, 85, 80, 200 },
				{ 49, 80, 85, 250 }
			},
		}
	},

	{
		/*** Priest ***/

		TV_LIFE_BOOK,
		0x04,

		A_WIS,
		1,

		1,
		430,
		{
			/* Priest: Life Magic */
			{
				{ 1, 1, 10, 4 },
				{ 1, 2, 15, 4 },
				{ 1, 2, 20, 4 },
				{ 3, 2, 25, 1 },
				{ 3, 3, 27, 2 },
				{ 4, 4, 28, 2 },
				{ 5, 4, 32, 4 },
				{ 7, 5, 38, 4 },

				{ 7, 6, 38, 5 },
				{ 9, 6, 38, 4 },
				{ 9, 7, 40, 4 },
				{ 10, 8, 38, 4 },
				{ 10, 8, 40, 4 },
				{ 11, 8, 42, 4 },
				{ 20, 16, 60, 7 },
				{ 33, 55, 90, 15 },

				{ 15, 14, 50, 50 },
				{ 16, 14, 80, 60 },
				{ 17, 14, 55, 70 },
				{ 24, 20, 55, 70 },
				{ 25, 20, 70, 120 },
				{ 25, 25, 80, 250 },
				{ 39, 32, 95, 200 },
				{ 44, 44, 80, 250 },

				{ 5, 5, 50, 80 },
				{ 15, 14, 50, 100 },
				{ 30, 30, 80, 130 },
				{ 35, 70, 90, 250 },
				{ 40, 50, 80, 130 },
				{ 40, 40, 80, 200 },
				{ 42, 90, 85, 200 },
				{ 45, 90, 85, 250 },
			},

			/* Priest: Sorcery */
			{
				{ 2, 1, 18, 4 },
				{ 3, 2, 19, 4 },
				{ 4, 3, 20, 1 },
				{ 5, 4, 25, 1 },
				{ 6, 5, 25, 1 },
				{ 7, 6, 30, 5 },
				{ 9, 7, 25, 4 },
				{ 11, 10, 70, 9 },

				{ 13, 11, 70, 8 },
				{ 14, 12, 70, 6 },
				{ 15, 13, 70, 7 },
				{ 16, 14, 45, 6 },
				{ 22, 15, 55, 8 },
				{ 27, 17, 60, 10 },
				{ 30, 22, 65, 15 },
				{ 36, 33, 70, 20 },

				{ 7, 7, 20, 15 },
				{ 14, 14, 75, 40 },
				{ 18, 18, 55, 25 },
				{ 20, 33, 65, 40 },
				{ 20, 20, 80, 50 },
				{ 22, 22, 55, 25 },
				{ 27, 27, 70, 19 },
				{ 30, 30, 75, 100 },

				{ 13, 13, 35, 20 },
				{ 24, 24, 70, 70 },
				{ 33, 40, 75, 120 },
				{ 35, 40, 60, 100 },
				{ 42, 45, 75, 100 },
				{ 45, 50, 85, 175 },
				{ 46, 45, 55, 200 },
				{ 48, 70, 80, 250 },
			},

			/* Priest: Nature Magic */
			{
				{ 2, 1, 20, 4 },
				{ 5, 3, 20, 4 },
				{ 5, 4, 20, 1 },
				{ 6, 5, 30, 4 },
				{ 6, 5, 45, 5 },
				{ 6, 6, 45, 5 },
				{ 7, 7, 45, 5 },
				{ 7, 7, 30, 4 },

				{ 7, 7, 35, 6 },
				{ 8, 7, 25, 6 },
				{ 9, 10, 35, 6 },
				{ 10, 10, 35, 6 },
				{ 11, 11, 25, 5 },
				{ 20, 20, 60, 7 },
				{ 30, 30, 50, 8 },
				{ 42, 60, 85, 50 },

				{ 11, 12, 35, 44 },
				{ 12, 13, 70, 120 },
				{ 18, 20, 80, 60 },
				{ 23, 25, 75, 40 },
				{ 35, 35, 75, 50 },
				{ 39, 40, 85, 200 },
				{ 40, 50, 80, 250 },
				{ 42, 90, 85, 250 },

				{ 22, 22, 55, 24 },
				{ 25, 25, 55, 25 },
				{ 27, 27, 70, 29 },
				{ 32, 30, 70, 29 },
				{ 37, 32, 80, 65 },
				{ 39, 37, 85, 100 },
				{ 42, 90, 90, 250 },
				{ 44, 80, 60, 150 },
			},

			/* Priest: Chaos Magic */
			{
				{ 2, 1, 17, 4 },
				{ 3, 2, 19, 4 },
				{ 4, 3, 21, 4 },
				{ 5, 4, 25, 6 },
				{ 10, 6, 25, 5 },
				{ 11, 6, 45, 5 },
				{ 16, 11, 45, 6 },
				{ 17, 11, 30, 5 },

				{ 19, 15, 45, 7 },
				{ 21, 16, 45, 9 },
				{ 23, 18, 75, 20 },
				{ 25, 18, 45, 11 },
				{ 27, 20, 60, 12 },
				{ 29, 22, 55, 8 },
				{ 33, 23, 75, 15},
				{ 37, 42, 80, 40 },


				{ 14, 11, 40, 9 },
				{ 17, 17, 65, 20 },
				{ 20, 18, 75, 35 },
				{ 27, 27, 65, 35 },
				{ 35, 30, 80, 150 },
				{ 44, 45, 85, 150 },
				{ 47, 90, 85, 250 },
				{ 49, 100, 85, 250 },

				{ 25, 25, 61, 8 },
				{ 37, 37, 80, 35 },
				{ 39, 37, 70, 50 },
				{ 41, 45, 80, 100 },
				{ 45, 55, 90, 250 },
				{ 47, 60, 90, 200 },
				{ 49, 95, 75, 200 },
				{ 50, 100, 90, 250 },
			},

			/* Priest: Death Magic */
			{
				{ 1, 1, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 3, 3, 25, 4 },
				{ 5, 4, 27, 4 },
				{ 7, 7, 30, 4 },
				{ 9, 11, 75, 6 },
				{ 11, 11, 30, 4 },
				{ 12, 12, 40, 5 },

				{ 14, 14, 40, 5 },
				{ 16, 16, 30, 4 },
				{ 21, 20, 50, 10 },
				{ 26, 25, 60, 30 },
				{ 33, 75, 90, 30 },
				{ 35, 32, 60, 16 },
				{ 41, 40, 80, 70 },
				{ 43, 35, 95, 100 },

				{ 13, 20, 80, 180 },
				{ 13, 15, 80, 30 },
				{ 14, 15, 30, 15 },
				{ 33, 33, 70, 33 },
				{ 36, 95, 70, 90 },
				{ 38, 35, 60, 125 },
				{ 40, 40, 70, 40 },
				{ 44, 44, 80, 200 },

				{ 25, 25, 75, 50 },
				{ 30, 75, 95, 250 },
				{ 35, 45, 95, 250 },
				{ 37, 50, 85, 150 },
				{ 44, 52, 75, 150 },
				{ 47, 75, 80, 100 },
				{ 48, 100, 90, 250 },
				{ 50, 111, 90, 250 },
			},

			/* Priest: Trump Magic */
			{
				{ 1, 1, 20, 3 },
				{ 4, 4, 20, 4 },
				{ 6, 5, 32, 8 },
				{ 7, 7, 35, 8 },
				{ 9, 9, 15, 4 },
				{ 13, 14, 25, 6 },
				{ 17, 17, 25, 5 },
				{ 29, 22, 25, 8 },

				{ 22, 22, 35, 8 },
				{ 26, 22, 30, 9 },
				{ 28, 25, 25, 8 },
				{ 32, 30, 35, 12 },
				{ 35, 35, 30, 10 },
				{ 38, 37, 25, 6 },
				{ 42, 40, 35, 15 },
				{ 45, 45, 30, 12 },

				{ 24, 38, 65, 40 },
				{ 27, 25, 30, 25 },
				{ 29, 27, 30, 30 },
				{ 32, 30, 30, 9 },
				{ 33, 30, 30, 35 },
				{ 38, 75, 35, 100 },
				{ 41, 110, 40, 250 },
				{ 49, 125, 40, 200 },

				{ 32, 30, 25, 50 },
				{ 38, 55, 40, 100 },
				{ 40, 60, 35, 150 },
				{ 43, 85, 35, 150 },
				{ 43, 50, 65, 100 },
				{ 48, 115, 35, 150 },
				{ 49, 120, 35, 200 },
				{ 50, 125, 35, 220 }
			},

			/* Priest: Arcane Magic */
			{
				{ 1, 1, 15, 4 },
				{ 1, 1, 28, 5 },
				{ 2, 1, 28, 4 },
				{ 2, 2, 28, 5 },
				{ 3, 3, 28, 5 },
				{ 5, 5, 35, 6 },
				{ 6, 6, 28, 7 },
				{ 7, 6, 39, 5 },

				{ 8, 7, 35, 7 },
				{ 9, 8, 55, 7 },
				{ 10, 9, 45, 6 },
				{ 11, 10, 45, 6 },
				{ 12, 11, 45, 6 },
				{ 13, 12, 45, 6 },
				{ 14, 13, 45, 5 },
				{ 15, 14, 45, 5 },

				{ 16, 15, 45, 5 },
				{ 17, 16, 45, 5 },
				{ 18, 17, 28, 6 },
				{ 19, 18, 45, 8 },
				{ 22, 25, 55, 25 },
				{ 24, 22, 55, 9 },
				{ 27, 24, 55, 9 },
				{ 29, 26, 65, 12 },

				{ 33, 30, 55, 13 },
				{ 37, 32, 75, 50 },
				{ 40, 36, 75, 25 },
				{ 42, 38, 65, 25 },
				{ 44, 39, 61, 30 },
				{ 46, 40, 75, 40 },
				{ 47, 55, 65, 50 },
				{ 50, 120, 75, 200 }
			},

			/* Priest: Craft Magic */
			{

				{  1,  1, 20,   4},
				{  3,  3, 20,   4},
				{  4,  4, 30,   3},
				{  6,  6, 35,   5},
				{  7,  6, 35,   5},
				{  9,  7, 45,   6},
				{ 10,  6, 35,   5},
				{ 11,  6, 35,   5},

				{ 13, 15, 45,   4},
				{ 15, 13, 30,   4},
				{ 18, 17, 60,   6},
				{ 22, 20, 75,  80},
				{ 23, 21, 85,  50},
				{ 26, 24, 50,   4},
				{ 29, 28, 60,   5},
				{ 42, 70, 85,  50},

				{ 14, 14, 45,  12},
				{ 16, 12, 70, 120},
				{ 22, 23, 85,  60},
				{ 27, 17, 60,  10},
				{ 33, 58, 75,  50},
				{ 36, 80, 85, 150},
				{ 43, 90, 90, 200},
				{ 46, 60, 70, 200},

				{ 15, 15, 65,  50},
				{ 30, 28, 60, 150},
				{ 38, 50, 85, 200},
				{ 40, 80, 90, 200},
				{ 40,100, 90, 200},
				{ 42,100, 90, 250},
				{ 45,110, 75, 250},
				{ 48, 70, 75, 250}
			},

			/* Priest: Daemon Magic */
			{
				{ 2, 1, 22, 4 },
				{ 3, 2, 24, 4 },
				{ 4, 3, 26, 4 },
				{ 5, 6, 30, 6 },
				{ 10, 7, 30, 5 },
				{ 12, 9, 50, 6 },
				{ 14, 12, 35, 5 },
				{ 15, 15, 50, 7 },

				{ 17, 17, 50, 9 },
				{ 20, 19, 65, 20 },
				{ 23, 23, 60, 11 },
				{ 25, 18, 65, 12 },
				{ 27, 20, 65, 12 },
				{ 33, 75, 90, 30 },
				{ 35, 32, 70, 30 },
				{ 39, 85, 85, 40 },


				{ 14, 14, 45, 9 },
				{ 18, 20, 70, 20 },
				{ 27, 25, 75, 35 },
				{ 36, 34, 80, 100 },
				{ 37, 50, 85, 150 },
				{ 39, 40, 80, 100 },
				{ 44, 44, 75, 50 },
				{ 47, 95, 90, 250 },

				{  9,  9, 66, 8 },
				{ 28, 28, 85, 35 },
				{ 38, 40, 80, 200 },
				{ 43, 58, 85, 100 },
				{ 45, 100, 90, 250 },
				{ 47, 60, 90, 150 },
				{ 49, 95, 80, 200 },
				{ 50, 80, 85, 250 },
			},
		}
	},

	{
		/*** Rogue ***/
		TV_SORCERY_BOOK,
		0x07,

		A_INT,
		0,

		5,
		400,
		{
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},

			/* Rogue (Burglar): Sorcery */
			{
				{ 5, 1, 50, 1 },
				{ 7, 2, 55, 1 },
				{ 8, 3, 65, 1 },
				{ 9, 3, 65, 1 },
				{ 13, 6, 75, 1 },
				{ 15, 7, 75, 1 },
				{ 17, 9, 75, 1 },
				{ 21, 12, 80, 1 },

				{ 25, 14, 80, 1 },
				{ 27, 15, 80, 1 },
				{ 29, 17, 75, 2 },
				{ 30, 20, 80, 4 },
				{ 31, 23, 80, 5 },
				{ 32, 25, 70, 6 },
				{ 35, 30, 80, 12 },
				{ 40, 35, 75, 20 },

				{ 9, 3, 65, 5 },
				{ 14, 10, 80, 8 },
				{ 16, 10, 60, 10 },
				{ 22, 40, 80, 20 },
				{ 17, 20, 80, 20 },
				{ 18, 17, 60, 30 },
				{ 30, 35, 75, 15 },
				{ 35, 40, 80, 60 },

				{ 15, 15, 40, 10 },
				{ 20, 20, 70, 50 },
				{ 37, 40, 80, 100 },
				{ 44, 45, 80, 80 },
				{ 44, 45, 70, 100 },
				{ 45, 50, 70, 100 },
				{ 49, 50, 70, 100 },
				{ 99, 0, 0, 0 },
			},

			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},

			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},

			/* Rogue (Assassin): Death Magic */
			{
				{ 5, 3, 40, 1 },
				{ 7, 4, 40, 1 },
				{ 9, 5, 50, 1 },
				{ 13, 7, 60, 1 },
				{ 15, 7, 80, 1 },
				{ 17, 15, 80, 1 },
				{ 19, 17, 30, 1 },
				{ 19, 19, 30, 1 },

				{ 21, 21, 60, 3 },
				{ 23, 23, 75, 4 },
				{ 27, 25, 75, 4 },
				{ 31, 30, 85, 5 },
				{ 35, 35, 75, 5 },
				{ 40, 40, 75, 4 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },

				{ 20, 25, 80, 100 },
				{ 23, 20, 40, 20 },
				{ 28, 28, 75, 25 },
				{ 32, 32, 80, 50 },
				{ 46, 100, 90, 100 },
				{ 48, 45, 75, 40 },
				{ 99, 0, 0, 0 },
				{ 50, 50, 75, 50 },

				{ 30, 30, 80, 50 },
				{ 31, 80, 80, 250 },
				{ 32, 40, 90, 150 },
				{ 41, 55, 80, 200 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 50, 125, 90, 250 },
			},

			/* Rogue (Card Shark): Trump Magic */
			{
				{ 5, 2, 25, 3 },
				{ 7, 5, 25, 4 },
				{ 9, 7, 37, 8 },
				{ 11, 9, 40, 8 },
				{ 13, 11, 20, 4 },
				{ 15, 15, 30, 6 },
				{ 19, 20, 30, 5 },
				{ 22, 21, 30, 8 },

				{ 25, 22, 40, 8 },
				{ 28, 23, 35, 9 },
				{ 30, 26, 30, 8 },
				{ 35, 35, 40, 12 },
				{ 39, 38, 35, 10 },
				{ 41, 40, 45, 12 },
				{ 46, 44, 40, 15 },
				{ 49, 50, 35, 12 },

				{ 26, 40, 80, 40 },
				{ 30, 30, 35, 25 },
				{ 33, 30, 35, 30 },
				{ 35, 32, 35, 9 },
				{ 38, 33, 35, 35 },
				{ 42, 90, 40, 100 },
				{ 45, 150, 45, 250 },
				{ 99, 0, 0, 0 },

				{ 35, 30, 30, 50 },
				{ 42, 65, 45, 100 },
				{ 44, 70, 40, 150 },
				{ 46, 100, 40, 150 },
				{ 46, 60, 70, 75 },
				{ 49, 125, 40, 150 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
			},

			/* Rogue (Thief): Arcane Magic */
			{
				{ 1, 1, 20, 4 },
				{ 1, 1, 33, 5 },
				{ 2, 1, 33, 4 },
				{ 2, 2, 33, 5 },
				{ 3, 3, 33, 5 },
				{ 5, 5, 40, 6 },
				{ 6, 6, 33, 7 },
				{ 7, 7, 44, 5 },

				{ 8, 8, 40, 7 },
				{ 9, 9, 60, 7 },
				{ 10, 10, 50, 6 },
				{ 11, 11, 50, 6 },
				{ 13, 11, 50, 6 },
				{ 14, 12, 50, 6 },
				{ 15, 13, 50, 5 },
				{ 16, 14, 50, 5 },

				{ 17, 15, 50, 5 },
				{ 18, 16, 50, 5 },
				{ 19, 17, 33, 6 },
				{ 20, 20, 50, 8 },
				{ 23, 28, 60, 25 },
				{ 25, 24, 60, 9 },
				{ 28, 25, 60, 9 },
				{ 30, 28, 70, 12 },

				{ 33, 30, 60, 13 },
				{ 37, 33, 80, 40 },
				{ 40, 36, 80, 25 },
				{ 44, 38, 70, 25 },
				{ 46, 40, 66, 30 },
				{ 47, 42, 80, 40 },
				{ 48, 60, 70, 50 },
				{ 50, 125, 80, 200 }
			},

			/* Rogue ( ): Craft Magic */
			{

				{  5,  2, 40,   2},
				{  7,  3, 40,   2},
				{  8,  5, 45,   1},
				{ 11, 10, 50,   2},
				{ 13, 10, 50,   2},
				{ 15, 12, 55,   3},
				{ 18, 10, 50,   2},
				{ 20, 10, 50,   2},

				{ 23, 18, 55,   2},
				{ 27, 20, 60,   2},
				{ 28, 24, 65,   3},
				{ 30, 27, 75,  40},
				{ 25, 22, 70,  25},
				{ 35, 35, 70,   2},
				{ 99,  0,  0,   0},
				{ 50, 90, 85,  25},

				{ 23, 23, 60,   6},
				{ 24, 18, 80,  60},
				{ 28, 30, 90,  30},
				{ 32, 25, 70,   5},
				{ 37, 65, 80,  25},
				{ 41, 90, 90,  75},
				{ 48,120, 95, 100},
				{ 99,  0,  0,   0},

				{ 24, 24, 75,  25},
				{ 37, 40, 80,  75},
				{ 42, 60, 90, 100},
				{ 42, 80, 80, 100},
				{ 43,100, 80, 100},
				{ 46,120, 80, 125},
				{ 49,150, 90, 125},
				{ 99,  0,  0,   0}
			},

			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},
		}
	},

	{
		/*** Ranger ***/

		TV_SORCERY_BOOK,
		0x07,

		A_INT,
		0,

		3,
		450,

		{
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},

			/* Ranger: Sorcery */
			{
				{ 3, 2, 35, 2 },
				{ 3, 3, 35, 2 },
				{ 5, 4, 40, 2 },
				{ 7, 4, 40, 2 },
				{ 9, 6, 40, 2 },
				{ 11, 8, 45, 2 },
				{ 13, 8, 40, 3 },
				{ 17, 17, 90, 4 },

				{ 20, 19, 85, 4 },
				{ 23, 25, 90, 3 },
				{ 25, 25, 60, 3 },
				{ 27, 25, 85, 3 },
				{ 31, 27, 70, 3 },
				{ 34, 35, 70, 4 },
				{ 38, 37, 70, 8 },
				{ 42, 40, 90, 10 },

				{ 15, 7, 75, 20 },
				{ 17, 17, 70, 25 },
				{ 19, 25, 65, 20 },
				{ 22, 40, 70, 40 },
				{ 23, 25, 60, 20 },
				{ 27, 27, 60, 15 },
				{ 35, 35, 75, 13 },
				{ 38, 60, 80, 60 },

				{ 20, 20, 45, 5 },
				{ 27, 27, 70, 50 },
				{ 40, 40, 95, 120 },
				{ 45, 50, 80, 100 },
				{ 47, 50, 80, 100 },
				{ 50, 50, 90, 175 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
			},

			/* Ranger: Nature Magic */
			{
				{ 3, 1, 35, 2 },
				{ 4, 3, 40, 2 },
				{ 4, 4, 40, 3 },
				{ 5, 7, 55, 2 },
				{ 6, 7, 50, 3 },
				{ 7, 7, 50, 3 },
				{ 8, 7, 50, 3 },
				{ 9, 7, 45, 3 },

				{ 9, 7, 80, 4 },
				{ 10, 7, 40, 3 },
				{ 11, 9, 40, 4 },
				{ 12, 9, 55, 4 },
				{ 14, 9, 55, 4 },
				{ 18, 20, 65, 8 },
				{ 23, 23, 65, 10 },
				{ 40, 60, 90, 50 },

				{ 12, 12, 50, 25 },
				{ 14, 15, 70, 60 },
				{ 20, 30, 85, 70 },
				{ 22, 24, 80, 30 },
				{ 35, 30, 80, 75 },
				{ 38, 40, 90, 200 },
				{ 40, 55, 90, 250 },
				{ 42, 80, 90, 250 },

				{ 25, 28, 60, 150 },
				{ 26, 26, 60, 100 },
				{ 30, 35, 75, 125 },
				{ 32, 29, 75, 35 },
				{ 36, 33, 75, 45 },
				{ 40, 35, 75, 100 },
				{ 41, 80, 95, 250 },
				{ 42, 80, 75, 150 },
			},

			/* Ranger: Chaos Magic */
			{
				{ 3, 1, 20, 1 },
				{ 3, 3, 35, 2 },
				{ 5, 3, 35, 2 },
				{ 7, 5, 45, 2 },
				{ 14, 12, 40, 2 },
				{ 20, 16, 50, 6 },
				{ 25, 21, 60, 3 },
				{ 25, 22, 60, 3 },

				{ 27, 23, 60, 5 },
				{ 30, 25, 60, 8 },
				{ 33, 30, 70, 13 },
				{ 35, 31, 70, 10 },
				{ 37, 35, 75, 15 },
				{ 39, 29, 65, 5 },
				{ 43, 30, 95, 15 },
				{ 48, 50, 85, 30 },

				{ 22, 20, 60, 30 },
				{ 25, 25, 70, 20 },
				{ 28, 25, 80, 45 },
				{ 35, 32, 70, 35 },
				{ 38, 35, 85, 150 },
				{ 50, 50, 90, 150 },
				{ 48, 100, 90, 250 },
				{ 99, 0, 0, 0 },

				{ 33, 33, 66, 8 },
				{ 40, 45, 85, 35 },
				{ 42, 42, 75, 42 },
				{ 48, 48, 85, 100 },
				{ 42, 75, 95, 250 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
			},

			/* Ranger: Death Magic */
			{
				{ 5, 2, 40, 3 },
				{ 5, 3, 40, 3 },
				{ 7, 4, 50, 3 },
				{ 9, 5, 40, 3 },
				{ 11, 8, 40, 3 },
				{ 17, 25, 75, 4 },
				{ 19, 19, 50, 3 },
				{ 22, 22, 50, 3 },

				{ 24, 24, 55, 3 },
				{ 26, 26, 50, 3 },
				{ 28, 28, 75, 4 },
				{ 34, 35, 85, 10 },
				{ 42, 40, 80, 5 },
				{ 50, 50, 90, 75 },
				{ 44, 39, 76, 50 },
				{ 99, 0, 0, 0 },

				{ 25, 30, 80, 125 },
				{ 25, 25, 80, 100 },
				{ 27, 27, 40, 40 },
				{ 41, 60, 80, 150 },
				{ 45, 40, 60, 9 },
				{ 45, 100, 90, 100 },
				{ 47, 45, 80, 100 },
				{ 99, 0, 0, 0 },

				{ 35, 35, 75, 50 },
				{ 38, 90, 80, 100 },
				{ 40, 45, 95, 200 },
				{ 48, 50, 30, 75 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 }
			},

			/* Ranger: Trump Magic */
			{
				{ 3, 1, 25, 3 },
				{ 6, 6, 25, 4 },
				{ 9, 7, 37, 8 },
				{ 10, 8, 40, 8 },
				{ 13, 10, 20, 4 },
				{ 17, 17, 30, 6 },
				{ 20, 20, 30, 5 },
				{ 23, 22, 30, 8 },

				{ 24, 22, 40, 8 },
				{ 27, 23, 35, 9 },
				{ 30, 26, 30, 8 },
				{ 35, 33, 40, 12 },
				{ 39, 38, 35, 10 },
				{ 42, 40, 45, 12 },
				{ 45, 42, 40, 15 },
				{ 99, 0, 0, 0 },

				{ 25, 40, 70, 40 },
				{ 28, 26, 35, 25 },
				{ 31, 30, 35, 30 },
				{ 36, 32, 35, 9 },
				{ 36, 33, 35, 35 },
				{ 41, 80, 40, 100 },
				{ 44, 120, 45, 250 },
				{ 99, 0, 0, 0 },

				{ 35, 33, 30, 50 },
				{ 40, 65, 45, 100 },
				{ 44, 65, 45, 100 },
				{ 48, 95, 40, 150 },
				{ 46, 60, 70, 100 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
			},

			/* Ranger: Arcane Magic */
			{
				{ 3, 2, 20, 4 },
				{ 3, 2, 33, 5 },
				{ 4, 3, 33, 4 },
				{ 4, 3, 33, 5 },
				{ 5, 4, 33, 5 },
				{ 6, 6, 40, 6 },
				{ 7, 7, 33, 7 },
				{ 8, 8, 44, 5 },

				{ 9, 9, 40, 7 },
				{ 10, 10, 60, 7 },
				{ 11, 11, 50, 6 },
				{ 12, 12, 50, 6 },
				{ 14, 13, 50, 6 },
				{ 15, 14, 50, 6 },
				{ 16, 15, 50, 5 },
				{ 17, 16, 50, 5 },

				{ 18, 17, 50, 5 },
				{ 19, 18, 50, 5 },
				{ 20, 19, 33, 6 },
				{ 22, 20, 50, 8 },
				{ 25, 29, 60, 25 },
				{ 27, 26, 60, 9 },
				{ 29, 27, 60, 9 },
				{ 33, 30, 70, 12 },

				{ 35, 33, 60, 13 },
				{ 38, 36, 80, 40 },
				{ 42, 38, 80, 25 },
				{ 46, 40, 70, 25 },
				{ 47, 42, 66, 30 },
				{ 48, 44, 80, 40 },
				{ 49, 65, 70, 50 },
				{ 99, 0, 0, 0 }
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Ranger: Daemon Magic */
			{
				{ 3, 1, 20, 1 },
				{ 4, 3, 35, 2 },
				{ 6, 4, 35, 2 },
				{ 7, 8, 45, 2 },
				{ 12, 8, 40, 2 },
				{ 17, 11, 60, 3 },
				{ 20, 15, 60, 3 },
				{ 25, 25, 60, 5 },

				{ 27, 27, 60, 8 },
				{ 29, 26, 70, 13 },
				{ 32, 30, 70, 10 },
				{ 35, 31, 75, 15 },
				{ 37, 35, 75, 15 },
				{ 38, 80, 95, 20 },
				{ 44, 40, 80, 15 },
				{ 47, 100, 90, 30 },

				{ 18, 18, 60, 30 },
				{ 22, 25, 70, 20 },
				{ 37, 38, 80, 45 },
				{ 40, 42, 80, 150 },
				{ 41, 60, 80, 150 },
				{ 44, 50, 80, 35 },
				{ 48, 48, 80, 42 },
				{ 99, 0, 0, 0 },

				{ 17, 14, 66, 8 },
				{ 35, 35, 85, 35 },
				{ 45, 55, 85, 200 },
				{ 50, 70, 85, 100 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
			},
		}
	},

	{
		/*** Paladin ***/

		TV_LIFE_BOOK,
		0x06,

		A_WIS,
		1,

		1,
		450,
		{
			/* Paladin: Life Magic */
			{
				{ 1, 1, 30, 4 },
				{ 2, 2, 35, 4 },
				{ 3, 3, 35, 4 },
				{ 4, 3, 35, 4 },
				{ 5, 4, 35, 4 },
				{ 8, 5, 40, 4 },
				{ 11, 9, 40, 3 },
				{ 13, 10, 45, 3 },

				{ 14, 11, 45, 4},
				{ 15, 15, 50, 4},
				{ 17, 15, 50, 4},
				{ 18, 15, 50, 4},
				{ 18, 15, 50, 4},
				{ 19, 15, 50, 4},
				{ 30, 25, 55, 5},
				{ 35, 70, 75, 5},

				{ 25, 22, 50, 75 },
				{ 28, 24, 70, 150 },
				{ 30, 25, 60, 75 },
				{ 33, 30, 60, 75 },
				{ 35, 32, 70, 75 },
				{ 35, 55, 80, 115 },
				{ 39, 38, 80, 125 },
				{ 46, 60, 80, 150 },

				{ 9, 9, 50, 40 },
				{ 25, 20, 50, 50 },
				{ 35, 45, 80, 115 },
				{ 40, 80, 80, 225 },
				{ 45, 80, 80, 115 },
				{ 45, 45, 80, 100 },
				{ 48, 100, 90, 250 },
				{ 50, 100, 80, 250 }
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Paladin: Death Magic */
			{
				{ 1, 1, 25, 4 },
				{ 3, 2, 25, 4 },
				{ 4, 3, 25, 4 },
				{ 6, 5, 27, 4 },
				{ 8, 8, 30, 4 },
				{ 10, 11, 75, 6 },
				{ 12, 12, 30, 4 },
				{ 15, 15, 40, 5 },

				{ 17, 17, 40, 5 },
				{ 19, 19, 30, 4 },
				{ 23, 23, 50, 10 },
				{ 35, 75, 90, 30 },
				{ 38, 35, 60, 16 },
				{ 46, 45, 80, 70 },
				{ 45, 38, 70, 50 },
				{ 47, 35, 95, 100 },

				{ 15, 20, 80, 180 },
				{ 15, 20, 80, 30 },
				{ 18, 20, 30, 15 },
				{ 38, 53, 80, 150 },
				{ 40, 35, 60, 16 },
				{ 42, 100, 70, 100 },
				{ 44, 40, 60, 125 },
				{ 48, 50, 80, 200 },


				{ 30, 35, 75, 50 },
				{ 36, 85, 90, 200 },
				{ 38, 45, 95, 250 },
				{ 45, 45, 70, 40 },
				{ 47, 52, 75, 150 },
				{ 48, 75, 80, 100 },
				{ 50, 115, 95, 250 },
				{ 50, 111, 95, 250 }
			},

			/* Paladin: No Trump magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},
			/* Paladin: No Arcane Magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		}
	},

	{
		/*** Warrior-Mage ***/

		TV_SORCERY_BOOK,
		0x07,

		A_INT,
		0,

		1,
		430,

		{
			{
				{ 2, 2, 30, 4 },
				{ 4, 4, 35, 4 },
				{ 5, 5, 35, 4 },
				{ 6, 6, 35, 4 },
				{ 8, 8, 35, 4 },
				{ 9, 9, 40, 4 },
				{ 14, 14, 40, 3 },
				{ 16, 16, 45, 3 },

				{ 18, 18, 45, 4},
				{ 20, 20, 50, 4},
				{ 22, 22, 50, 4},
				{ 24, 24, 50, 4},
				{ 26, 26, 50, 4},
				{ 28, 28, 50, 4},
				{ 33, 33, 55, 5},
				{ 40, 70, 75, 5},

				{ 28, 28, 50, 75 },
				{ 30, 30, 70, 150 },
				{ 34, 34, 60, 75 },
				{ 36, 36, 60, 75 },
				{ 38, 38, 70, 75 },
				{ 42, 55, 80, 115 },
				{ 45, 45, 80, 125 },
				{ 50, 70, 80, 150 },

				{ 10, 10, 50, 40 },
				{ 28, 28, 50, 50 },
				{ 38, 85, 80, 115 },
				{ 45, 90, 80, 225 },
				{ 46, 90, 80, 115 },
				{ 48, 50, 80, 100 },
				{ 49, 100, 90, 100 },
				{ 50, 100, 80, 250 }
			},

			/* Warrior-Mage: Sorcery */
			{
				{ 1, 1, 23, 4 },
				{ 2, 2, 24, 4 },
				{ 3, 3, 25, 1 },
				{ 4, 4, 30, 1 },
				{ 5, 5, 30, 1 },
				{ 6, 6, 35, 5 },
				{ 7, 7, 30, 4 },
				{ 8, 8, 75, 9 },

				{ 10, 9, 75, 8 },
				{ 11, 10, 75, 8 },
				{ 12, 11, 75, 7 },
				{ 13, 12, 50, 6 },
				{ 20, 15, 60, 8 },
				{ 27, 18, 60, 8 },
				{ 33, 25, 70, 15 },
				{ 40, 40, 75, 20 },

				{ 4, 4, 25, 15 },
				{ 14, 12, 80, 40 },
				{ 16, 14, 60, 25 },
				{ 21, 35, 70, 40 },
				{ 19, 19, 85, 50 },
				{ 24, 22, 60, 25 },
				{ 28, 28, 75, 19 },
				{ 32, 35, 80, 100 },

				{ 12, 12, 40, 20 },
				{ 19, 19, 75, 70 },
				{ 35, 45, 80, 120 },
				{ 39, 40, 60, 100 },
				{ 44, 45, 70, 100 },
				{ 46, 55, 90, 175 },
				{ 48, 50, 80, 200 },
				{ 48, 75, 85, 250 },
			},

			/* Warrior-Mage: Nature Magic */
			{
				{ 2, 2, 23, 4 },
				{ 3, 3, 25, 3 },
				{ 4, 4, 25, 1 },
				{ 5, 5, 35, 4 },
				{ 6, 6, 50, 5 },
				{ 7, 7, 50, 5 },
				{ 8, 8, 50, 5 },
				{ 9, 9, 35, 4 },

				{ 10, 10, 40, 6 },
				{ 11, 11, 30, 6 },
				{ 12, 12, 45, 6 },
				{ 13, 13, 40, 6 },
				{ 14, 14, 30, 5 },
				{ 19, 15, 65, 7 },
				{ 31, 31, 65, 10 },
				{ 45, 60, 85, 50 },

				{ 12, 12, 40, 44 },
				{ 15, 15, 75, 120 },
				{ 20, 22, 85, 60 },
				{ 25, 30, 85, 40 },
				{ 38, 38, 85, 80 },
				{ 40, 42, 90, 200 },
				{ 45, 48, 75, 200},
				{ 49, 95, 90, 250 },

				{ 25, 25, 60, 25 },
				{ 27, 27, 60, 25 },
				{ 28, 28, 75, 29 },
				{ 33, 33, 75, 35 },
				{ 38, 38, 85, 65 },
				{ 41, 41, 90, 100 },
				{ 45, 95, 95, 250 },
				{ 50, 85, 65, 150 },
			},

			/* Warrior-Mage: Chaos Magic */
			{
				{ 2, 2, 20, 4 },
				{ 3, 3, 22, 4 },
				{ 4, 4, 25, 4 },
				{ 5, 5, 30, 6 },
				{ 8, 8, 30, 1 },
				{ 11, 11, 45, 5 },
				{ 17, 15, 45, 6 },
				{ 18, 17, 35, 5 },

				{ 21, 21, 45, 7 },
				{ 23, 22, 45, 9 },
				{ 27, 25, 50, 20 },
				{ 29, 30, 50, 11 },
				{ 33, 33, 50, 12 },
				{ 37, 35, 60, 8 },
				{ 41, 40, 80, 15 },
				{ 48, 50, 85, 40 },

				{ 12, 12, 45, 9 },
				{ 17, 16, 60, 20 },
				{ 20, 18, 80, 35 },
				{ 27, 25, 60, 35 },
				{ 35, 30, 85, 150 },
				{ 46, 46, 85, 100 },
				{ 49, 95, 80, 250 },
				{ 50, 111, 80, 250 },

				{ 24, 20, 66, 8 },
				{ 40, 35, 85, 35 },
				{ 41, 39, 75, 40 },
				{ 45, 55, 85, 250 },
				{ 48, 48, 80, 150 },
				{ 49, 60, 85, 200 },
				{ 50, 100, 80, 250 },
				{ 50, 100, 85, 250 },
			},

			/* Warrior-Mage: Death Magic */
			{
				{ 1, 1, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 3, 3, 25, 4 },
				{ 4, 4, 27, 3 },
				{ 7, 7, 30, 4 },
				{ 9, 10, 75, 6 },
				{ 10, 10, 30, 4 },
				{ 12, 12, 30, 4 },

				{ 14, 14, 40, 5 },
				{ 16, 16, 30, 4 },
				{ 21, 21, 50, 10 },
				{ 28, 28, 85, 30 },
				{ 35, 75, 80, 30 },
				{ 38, 35, 60, 16 },
				{ 44, 55, 80, 70 },
				{ 46, 45, 95, 25 },

				{ 10, 22, 80, 180 },
				{ 12, 18, 80, 30 },
				{ 14, 18, 30, 15 },
				{ 30, 30, 75, 50 },
				{ 42, 90, 70, 90 },
				{ 43, 40, 60, 125 },
				{ 43, 50, 70, 40 },
				{ 45, 50, 70, 200 },

				{ 24, 24, 75, 50 },
				{ 33, 80, 75, 150 },
				{ 35, 45, 95, 250 },
				{ 39, 55, 80, 200 },
				{ 46, 55, 75, 150 },
				{ 48, 85, 80, 100 },
				{ 50, 110, 95, 250 },
				{ 50, 123, 95, 250 },
			},

			/* Warrior-Mage: Trump Magic */
			{
				{ 1, 1, 25, 3 },
				{ 5, 5, 25, 4 },
				{ 7, 7, 37, 8 },
				{ 8, 7, 40, 8 },
				{ 10, 10, 20, 4 },
				{ 14, 15, 30, 6 },
				{ 18, 18, 30, 5 },
				{ 21, 23, 30, 8 },

				{ 24, 23, 40, 8 },
				{ 27, 24, 35, 9 },
				{ 29, 26, 30, 8 },
				{ 35, 32, 40, 12 },
				{ 38, 38, 35, 10 },
				{ 41, 40, 45, 12 },
				{ 44, 42, 40, 15 },
				{ 48, 46, 35, 12 },

				{ 23, 35, 70, 40 },
				{ 29, 27, 35, 25 },
				{ 31, 30, 35, 30 },
				{ 33, 30, 35, 9 },
				{ 35, 33, 35, 35 },
				{ 40, 80, 40, 100 },
				{ 42, 120, 45, 250 },
				{ 50, 135, 45, 200 },

				{ 33, 30, 30, 50 },
				{ 40, 60, 45, 100 },
				{ 42, 65, 40, 150 },
				{ 45, 95, 40, 150 },
				{ 45, 50, 70, 100 },
				{ 48, 125, 40, 150 },
				{ 49, 130, 40, 200 },
				{ 50, 135, 40, 220 }
			},

			/* Warrior-Mage: Arcane Magic */
			{
				{ 1, 1, 20, 4 },
				{ 2, 1, 33, 5 },
				{ 2, 2, 33, 4 },
				{ 3, 3, 33, 5 },
				{ 4, 4, 33, 5 },
				{ 5, 5, 40, 6 },
				{ 6, 6, 33, 7 },
				{ 7, 7, 44, 5 },

				{ 8, 8, 40, 7 },
				{ 9, 9, 60, 7 },
				{ 11, 10, 50, 6 },
				{ 12, 11, 50, 6 },
				{ 13, 12, 50, 6 },
				{ 14, 13, 50, 6 },
				{ 15, 14, 50, 5 },
				{ 16, 15, 50, 5 },

				{ 17, 16, 50, 5 },
				{ 18, 17, 50, 5 },
				{ 19, 18, 33, 6 },
				{ 20, 20, 50, 8 },
				{ 23, 30, 60, 25 },
				{ 25, 23, 60, 9 },
				{ 28, 25, 60, 9 },
				{ 30, 27, 70, 12 },

				{ 33, 30, 60, 13 },
				{ 35, 35, 80, 50 },
				{ 39, 38, 80, 25 },
				{ 43, 42, 70, 25 },
				{ 45, 44, 66, 30 },
				{ 47, 45, 80, 40 },
				{ 48, 65, 70, 50 },
				{ 50, 140, 80, 200 }
			},

			/* Warrior-Mage: Craft Magic */
			{

				{  2,  2, 25,   4},
				{  3,  3, 25,   4},
				{  5,  4, 35,   3},
				{  7,  8, 40,   5},
				{  8,  8, 40,   5},
				{ 10, 10, 50,   6},
				{ 12,  8, 40,   5},
				{ 13,  8, 40,   5},

				{ 15, 17, 55,   4},
				{ 20, 20, 60,   4},
				{ 22, 22, 65,   6},
				{ 25, 24, 80,  80},
				{ 26, 25, 85,  50},
				{ 31, 30, 65,   4},
				{ 34, 33, 70,   5},
				{ 44, 80, 90,  50},

				{ 16, 16, 50,  12},
				{ 18, 13, 75, 120},
				{ 24, 25, 90,  60},
				{ 28, 18, 60,  10},
				{ 35, 65, 80,  50},
				{ 38, 85, 90, 150},
				{ 45,100, 90, 200},
				{ 48, 65, 70, 200},

				{ 17, 17, 70,  50},
				{ 33, 35, 70, 150},
				{ 40, 55, 90, 200},
				{ 40, 85, 95, 200},
				{ 43,100, 95, 200},
				{ 45,110, 95, 250},
				{ 47,120, 80, 250},
				{ 50, 75, 80, 250}
			},

			/* Warrior-Mage: Daemon Magic */
			{
				{ 2, 2, 20, 4 },
				{ 3, 2, 22, 4 },
				{ 5, 4, 25, 4 },
				{ 7, 8, 30, 6 },
				{ 11, 8, 30, 1 },
				{ 13, 10, 50, 6 },
				{ 15, 13, 35, 5 },
				{ 16, 16, 45, 7 },

				{ 19, 19, 45, 9 },
				{ 24, 20, 60, 20 },
				{ 26, 25, 50, 11 },
				{ 29, 30, 50, 12 },
				{ 33, 33, 50, 12 },
				{ 35, 75, 80, 8 },
				{ 42, 34, 70, 15 },
				{ 44, 90, 85, 40 },

				{ 16, 16, 45, 9 },
				{ 19, 22, 60, 20 },
				{ 37, 50, 60, 35 },
				{ 38, 35, 80, 200 },
				{ 39, 55, 80, 200 },
				{ 42, 38, 80, 35 },
				{ 46, 45, 75, 40 },
				{ 50, 108, 80, 250 },

				{ 10, 10, 66, 8 },
				{ 32, 32, 85, 35 },
				{ 41, 48, 80, 200 },
				{ 46, 62, 85, 100 },
				{ 48, 111, 85, 250 },
				{ 49, 60, 80, 200 },
				{ 50, 105, 80, 200 },
				{ 50, 80, 85, 250 },
			},
		}
	},

	{
		/*** Chaos Warrior ***/
		TV_SORCERY_BOOK,
		0x07,

		A_INT,
		0,

		2,
		450,

		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Chaos Warrior: Chaos Magic */
			{
				{ 2, 1, 20, 4 },
				{ 3, 2, 22, 4 },
				{ 4, 3, 25, 4 },
				{ 5, 4, 30, 6 },
				{ 7, 7, 30, 1 },
				{ 8, 7, 45, 5 },
				{ 15, 9, 45, 6 },
				{ 16, 10, 35, 5 },

				{ 19, 12, 45, 7 },
				{ 22, 14, 45, 9 },
				{ 25, 17, 50, 20 },
				{ 28, 18, 50, 11 },
				{ 30, 20, 50, 12 },
				{ 33, 24, 60, 8 },
				{ 36, 26, 80, 15 },
				{ 40, 45, 85, 40 },

				{ 11, 11, 45, 9 },
				{ 14, 14, 60, 20 },
				{ 16, 15, 80, 35 },
				{ 23, 23, 60, 35 },
				{ 30, 30, 85, 150 },
				{ 40, 41, 80, 150 },
				{ 45, 90, 80, 250 },
				{ 47, 100, 80, 250 },

				{ 23, 23, 66, 10},
				{ 35, 35, 85, 35 },
				{ 37, 37, 75, 40 },
				{ 39, 40, 85, 100 },
				{ 42, 50, 85, 250 },
				{ 45, 58, 85, 200 },
				{ 48, 100, 80, 220 },
				{ 49, 100, 85, 250 },
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Chaos Warrior: No Trump magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},
			/* Chaos Warrior: No Arcane Magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Chaos Warrior: Daemon Magic */
			{
				{ 2, 1, 20, 4 },
				{ 3, 2, 22, 4 },
				{ 5, 4, 25, 4 },
				{ 6, 6, 30, 6 },
				{ 8, 7, 30, 1 },
				{ 10, 8, 50, 6 },
				{ 12, 10, 35, 5 },
				{ 13, 13, 45, 7 },

				{ 16, 15, 45, 9 },
				{ 23, 22, 50, 20 },
				{ 24, 24, 50, 11 },
				{ 28, 18, 50, 12 },
				{ 30, 20, 50, 12 },
				{ 33, 75, 90, 8 },
				{ 38, 32, 70, 15 },
				{ 39, 85, 90, 40 },

				{ 15, 15, 45, 9 },
				{ 18, 20, 60, 20 },
				{ 27, 27, 80, 35 },
				{ 35, 32, 80, 150 },
				{ 36, 45, 85, 200 },
				{ 37, 40, 75, 40 },
				{ 40, 43, 60, 35 },
				{ 46, 90, 90, 250 },

				{ 11, 11, 66, 10},
				{ 29, 28, 85, 35 },
				{ 36, 37, 80, 200 },
				{ 41, 55, 85, 100 },
				{ 43, 100, 85, 250 },
				{ 45, 58, 80, 200 },
				{ 48, 105, 80, 200 },
				{ 49, 80, 85, 250 },
			},
		}
	},
	{
		/*** Monk ***/

		TV_LIFE_BOOK,
		0x06,

		A_WIS,
		0,

		1,
		350,

		{
			{
				{ 1, 1, 30, 4 },
				{ 2, 2, 35, 4 },
				{ 3, 3, 35, 4 },
				{ 4, 4, 35, 4 },
				{ 5, 5, 35, 4 },
				{ 8, 6, 40, 4 },
				{ 11, 10, 40, 3 },
				{ 13, 12, 45, 3 },

				{ 15, 12, 45, 4},
				{ 16, 15, 50, 4},
				{ 17, 15, 50, 4},
				{ 18, 16, 50, 4},
				{ 19, 16, 50, 4},
				{ 20, 18, 50, 4},
				{ 30, 25, 55, 5},
				{ 35, 70, 75, 5},

				{ 26, 26, 50, 75 },
				{ 28, 28, 70, 150 },
				{ 32, 32, 60, 75 },
				{ 36, 35, 60, 75 },
				{ 38, 35, 70, 75 },
				{ 40, 60, 80, 115 },
				{ 45, 45, 80, 125 },
				{ 48, 64, 80, 150 },

				{ 10, 10, 50, 40 },
				{ 25, 25, 50, 50 },
				{ 40, 45, 80, 115 },
				{ 44, 84, 80, 225 },
				{ 46, 64, 80, 115 },
				{ 48, 40, 80, 100 },
				{ 49, 100, 90, 200 },
				{ 50, 100, 80, 250 }
			},

			/* Monk: Sorcery */
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Monk: Nature Magic */
			{
				{ 1, 1, 35, 2 },
				{ 3, 3, 40, 2 },
				{ 4, 4, 40, 3 },
				{ 7, 7, 55, 2 },
				{ 7, 7, 50, 3 },
				{ 7, 7, 50, 3 },
				{ 8, 8, 50, 3 },
				{ 9, 9, 45, 3 },

				{ 10, 8, 80, 4 },
				{ 11, 9, 40, 3 },
				{ 12, 10, 40, 4 },
				{ 14, 12, 55, 4 },
				{ 16, 12, 55, 4 },
				{ 18, 22, 65, 8 },
				{ 31, 31, 75, 10 },
				{ 40, 60, 90, 50 },

				{ 14, 14, 50, 25 },
				{ 16, 16, 70, 60 },
				{ 22, 30, 85, 70 },
				{ 27, 33, 85, 40 },
				{ 35, 35, 80, 80 },
				{ 40, 40, 90, 200 },
				{ 45, 55, 90, 250 },
				{ 50, 80, 90, 250 },

				{ 28, 28, 60, 150 },
				{ 30, 30, 60, 160 },
				{ 35, 50, 75, 125 },
				{ 33, 33, 75, 35 },
				{ 38, 38, 75, 45 },
				{ 42, 40, 75, 100 },
				{ 45, 85, 95, 250 },
				{ 48, 85, 75, 150 }
			},

			/* Monk: Chaos Magic */
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* Monk: Death Magic */
			{
				{ 1, 1, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 3, 3, 25, 4 },
				{ 5, 5, 27, 4 },
				{ 7, 7, 30, 4 },
				{ 11, 11, 75, 6 },
				{ 12, 12, 30, 4 },
				{ 14, 14, 40, 5 },

				{ 16, 16, 40, 5 },
				{ 19, 19, 30, 4 },
				{ 22, 22, 50, 10 },
				{ 28, 30, 85, 30},
				{ 33, 80, 90, 30 },
				{ 37, 35, 60, 16 },
				{ 45, 50, 80, 70 },
				{ 47, 45, 95, 100 },

				{ 15, 20, 80, 180 },
				{ 16, 16, 80, 30 },
				{ 18, 18, 30, 15 },
				{ 35, 35, 75, 50 },
				{ 42, 95, 70, 90 },
				{ 44, 40, 60, 125 },
				{ 45, 44, 70, 40 },
				{ 48, 50, 80, 200 },

				{ 30, 30, 75, 50 },
				{ 37, 85, 85, 220 },
				{ 38, 50, 95, 250 },
				{ 42, 60, 80, 200 },
				{ 48, 60, 75, 150 },
				{ 49, 80, 80, 125 },
				{ 50, 110, 95, 250 },
				{ 50, 115, 95, 250 }
			},

			/* Monk: No Trump magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},

			/* Monk: No Arcane Magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0}
			},

			/* Monk: Craft Magic */
			{

				{  2,  2, 30,   4},
				{  3,  3, 35,   4},
				{  5,  5, 40,   3},
				{  7, 10, 45,   5},
				{  9, 10, 45,   5},
				{ 11, 12, 50,   6},
				{ 13, 10, 45,   5},
				{ 14, 10, 45,   5},

				{ 17, 18, 55,   4},
				{ 21, 21, 60,   4},
				{ 23, 23, 65,   6},
				{ 26, 25, 80,  80},
				{ 28, 26, 85,  50},
				{ 32, 30, 60,   4},
				{ 37, 28, 55,   5},
				{ 45, 90, 80,  50},

				{ 18, 18, 55,  12},
				{ 20, 17, 70, 120},
				{ 26, 28, 85,  60},
				{ 30, 23, 70,  10},
				{ 37, 60, 80,  50},
				{ 40, 85, 90, 150},
				{ 46,100, 95, 200},
				{ 49, 70, 75, 200},

				{ 20, 20, 70,  50},
				{ 36, 34, 70, 150},
				{ 41, 55, 90, 200},
				{ 45, 80, 95, 200},
				{ 46,100, 95, 200},
				{ 48,120, 80, 250},
				{ 47,120, 90, 250},
				{ 50,100, 90, 250}
			},

			/* Monk: Daemon Magic */
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		}
	},

	{
		/*** Mindcrafter ***/

		TV_LIFE_BOOK,
		0x00,

		A_WIS,
		0,

		99,
		400,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** High Mage ***/

		TV_SORCERY_BOOK,
		0x05,

		A_INT,
		0,

		1,
		430,

		{
			/* High Mage: Life Magic */
			{
				{  1,  1, 20,   4 },
				{  2,  2, 25,   4 },
				{  3,  3, 25,   4 },
				{  4,  4, 25,   4 },
				{  5,  5, 25,   4 },
				{  6,  5, 30,   4 },
				{  9,  9, 30,   3 },
				{ 12, 10, 35,   3 },

				{ 14, 12, 35,   4 },
				{ 15, 14, 40,   4 },
				{ 15, 15, 40,   4 },
				{ 17, 15, 40,   4 },
				{ 19, 17, 40,   4 },
				{ 21, 19, 40,   4 },
				{ 25, 25, 45,   5 },
				{ 30, 50, 55,   5 },

				{ 20, 20, 40,  75 },
				{ 24, 24, 60, 150 },
				{ 30, 30, 50,  75 },
				{ 31, 30, 50,  75 },
				{ 32, 30, 60,  75 },
				{ 33, 40, 60, 115 },
				{ 35, 35, 60, 125 },
				{ 40, 70, 70, 150 },

				{  5,  5, 40,  40 },
				{ 20, 20, 40,  50 },
				{ 30, 50, 60, 115 },
				{ 40, 80, 60, 225 },
				{ 42, 75, 60, 115 },
				{ 45, 40, 60, 100 },
				{ 47, 90, 70, 250 },
				{ 49, 90, 70, 250 }
			},

			/* High Mage: Sorcery */
			{
				{  1,  1, 15,   4 },
				{  1,  1, 15,   4 },
				{  2,  2, 15,   1 },
				{  2,  2, 20,   1 },
				{  3,  3, 20,   1 },
				{  4,  3, 25,   5 },
				{  5,  4, 20,   4 },
				{  5,  5, 65,   9 },

				{  7,  5, 65,   8 },
				{  7,  5, 65,   8 },
				{  9,  5, 65,   7 },
				{  9,  5, 40,   6 },
				{ 13,  8, 50,   8 },
				{ 17, 10, 50,   8 },
				{ 24, 15, 60,  15 },
				{ 28, 20, 65,  20 },

				{  2,  2, 20,  15 },
				{  8,  8, 70,  40 },
				{ 12,  9, 50,  25 },
				{ 15, 25, 60,  40 },
				{ 15, 12, 65,  50 },
				{ 17, 12, 50,  25 },
				{ 20, 20, 65,  19 },
				{ 24, 25, 75, 100 },

				{  8,  8, 30,  20 },
				{ 20, 20, 65,  70 },
				{ 25, 30, 70, 120 },
				{ 33, 30, 50, 100 },
				{ 36, 35, 70, 100 },
				{ 40, 40, 80, 175 },
				{ 41, 43, 50, 200 },
				{ 42, 65, 75, 250 },
			},

			/* High Mage: Nature Magic */
			{
				{ 1, 1, 15, 4 },
				{ 2, 1, 15, 3 },
				{ 2, 2, 15, 1 },
				{ 3, 2, 25, 4 },
				{ 3, 3, 40, 5 },
				{ 4, 3, 40, 5 },
				{ 4, 4, 40, 5 },
				{ 5, 4, 25, 4 },

				{ 5, 4, 30, 6 },
				{ 5, 4, 20, 6 },
				{ 5, 5, 35, 6 },
				{ 5, 5, 30, 6 },
				{ 7, 5, 30, 5 },
				{ 14, 10, 35, 8 },
				{ 20, 20, 80, 50 },
				{ 35, 50, 80, 50 },

				{ 7, 7, 20, 44 },
				{ 8, 8, 65, 120 },
				{ 12, 15, 75, 60 },
				{ 17, 20, 60, 40 },
				{ 25, 25, 80, 100 },
				{ 33, 35, 80, 200 },
				{ 35, 40, 65, 200},
				{ 37, 65, 80, 250 },

				{ 15, 15, 50, 25 },
				{ 20, 20, 70, 50 },
				{ 22, 22, 65, 29 },
				{ 28, 25, 65, 35 },
				{ 32, 28, 75, 65 },
				{ 34, 30, 80, 100 },
				{ 36, 80, 85, 250 },
				{ 39, 65, 55, 150 }
			},

			/* High Mage: Chaos Magic */
			{
				{ 1, 1, 15, 4 },
				{ 1, 1, 15, 4 },
				{ 2, 1, 15, 4 },
				{ 4, 2, 20, 1 },
				{ 6, 4, 40, 1 },
				{ 10, 5, 35, 6 },
				{ 12, 6, 35, 6 },
				{ 14, 7, 25, 5 },

				{ 15, 9, 20, 5 },
				{ 17, 10, 35, 9 },
				{ 19, 11, 35, 10 },
				{ 21, 12, 40, 11 },
				{ 22, 13, 40, 12 },
				{ 23, 15, 50, 8 },
				{ 27, 17, 70, 15 },
				{ 30, 35, 75, 40 },

				{ 9, 5, 35, 9 },
				{ 12, 12, 70, 35 },
				{ 14, 12, 70, 35 },
				{ 21, 21, 75, 100 },
				{ 26, 22, 75, 150 },
				{ 36, 36, 70, 150 },
				{ 42, 80, 70, 250 },
				{ 44, 90, 80, 250 },

				{ 16, 16, 55, 8 },
				{ 32, 30, 75, 35 },
				{ 34, 32, 65, 40 },
				{ 35, 34, 75, 100 },
				{ 39, 40, 75, 250 },
				{ 40, 53, 75, 200 },
				{ 43, 60, 70, 200 },
				{ 46, 90, 75, 250 }
			},

			/* High Mage: Death Magic */
			{
				{ 1, 1, 20, 4 },
				{ 1, 1, 20, 4 },
				{ 2, 1, 20, 4 },
				{ 2, 2, 20, 3 },
				{ 3, 3, 20, 4 },
				{ 5, 9, 55, 6 },
				{ 7, 7, 20, 4 },
				{ 8, 8, 20, 4 },

				{ 10, 10, 30, 5 },
				{ 11, 10, 20, 4 },
				{ 14, 12, 40, 10 },
				{ 22, 20, 75, 30 },
				{ 26, 65, 70, 30 },
				{ 30, 25, 50, 16 },
				{ 32, 30, 70, 70 },
				{ 35, 30, 85, 25 },

				{ 8, 15, 70, 180 },
				{ 8, 10, 70, 30 },
				{ 9, 9, 20, 15 },
				{ 25, 20, 65, 50 },
				{ 31, 80, 60, 90 },
				{ 33, 30, 50, 125 },
				{ 34, 30, 60, 40 },
				{ 36, 35, 60, 200 },

				{ 16, 16, 65, 50 },
				{ 22, 60, 85 , 250 },
				{ 26, 35, 85, 250 },
				{ 32, 35, 75, 200 },
				{ 37, 40, 65, 150 },
				{ 38, 66, 70, 100 },
				{ 42, 85, 85, 250 },
				{ 45, 75, 80, 250 }
			},

			/* High Mage: Trump Magic */
			{
				{ 1, 1, 20, 3 },
				{ 2, 2, 20, 4 },
				{ 4, 4, 32, 8 },
				{ 5, 5, 35, 8 },
				{ 6, 5, 15, 4 },
				{ 9, 10, 25, 6 },
				{ 12, 12, 25, 5 },
				{ 15, 16, 25, 8 },

				{ 16, 16, 35, 8 },
				{ 20, 17, 30, 9 },
				{ 23, 22, 25, 8 },
				{ 25, 26, 35, 12 },
				{ 28, 28, 30, 10 },
				{ 32, 30, 35, 12 },
				{ 35, 30, 35, 15 },
				{ 39, 36, 30, 12 },

				{ 17, 25, 60, 40 },
				{ 21, 21, 30, 25 },
				{ 23, 23, 30, 30 },
				{ 28, 24, 30, 9 },
				{ 25, 25, 30, 35 },
				{ 31, 65, 35, 100 },
				{ 36, 90, 40, 250 },
				{ 42, 90, 40, 200 },

				{ 25, 25, 25, 50 },
				{ 32, 45, 40, 100 },
				{ 34, 50, 35, 150 },
				{ 36, 75, 35, 150 },
				{ 37, 40, 60, 100 },
				{ 42, 90, 35, 150 },
				{ 44, 90, 35, 200 },
				{ 46, 90, 35, 220 }
			},

			/* High Mage: Arcane Magic */
			{
				{ 1, 1, 15, 4 },
				{ 1, 1, 23, 5 },
				{ 1, 1, 23, 4 },
				{ 1, 1, 23, 5 },
				{ 2, 1, 23, 5 },
				{ 3, 2, 33, 6 },
				{ 4, 4, 23, 7 },
				{ 5, 4, 33, 5 },

				{ 6, 5, 30, 7 },
				{ 7, 7, 50, 7 },
				{ 8, 7, 40, 6 },
				{ 8, 8, 40, 6 },
				{ 9, 8, 40, 6 },
				{ 10, 9, 40, 6 },
				{ 10, 10, 40, 5 },
				{ 11, 10, 40, 5 },

				{ 12, 10, 40, 5 },
				{ 13, 10, 40, 5 },
				{ 14, 11, 22, 6 },
				{ 15, 12, 40, 8 },
				{ 17, 17, 50, 25 },
				{ 19, 15, 50, 9 },
				{ 20, 16, 50, 9 },
				{ 22, 18, 60, 12 },

				{ 24, 22, 50, 13 },
				{ 26, 26, 60, 30 },
				{ 30, 30, 70, 25 },
				{ 35, 28, 60, 25 },
				{ 40, 28, 55, 30 },
				{ 41, 28, 70, 40 },
				{ 43, 40, 60, 50 },
				{ 46, 80, 70, 200 }
			},

			/* High Mage: Craft Magic */
			{

				{  1,  1, 15,   4},
				{  1,  1, 15,   4},
				{  3,  2, 20,   3},
				{  4,  4, 30,   5},
				{  5,  4, 30,   5},
				{  6,  5, 40,   6},
				{  7,  4, 30,   5},
				{  8,  4, 30,   5},

				{ 10, 10, 40,   4},
				{ 13, 12, 35,   4},
				{ 14, 13, 55,   6},
				{ 16, 15, 70,  80},
				{ 20, 12, 65,  50},
				{ 21, 19, 40,   4},
				{ 25, 25, 45,   5},
				{ 35, 60, 85,  50},

				{  9, 10, 35,  12},
				{ 11,  8, 65, 120},
				{ 16, 15, 75,  60},
				{ 20, 10, 50,  10},
				{ 27, 45, 70,  50},
				{ 31, 60, 70, 150},
				{ 38, 70, 80, 200},
				{ 42, 50, 60, 200},

				{ 10, 10, 50,  50},
				{ 27, 24, 60, 150},
				{ 33, 40, 80, 200},
				{ 30, 65, 85, 200},
				{ 34, 80, 85, 200},
				{ 37, 80, 70, 250},
				{ 40, 90, 50, 250},
				{ 44, 65, 70, 250}
			},

			/* High Mage: Daemon Magic */
			{
				{ 1, 1, 15, 4 },
				{ 1, 1, 15, 4 },
				{ 2, 2, 15, 4 },
				{ 3, 4, 20, 1 },
				{ 5, 4, 30, 1 },
				{ 7, 5, 40, 6 },
				{ 9, 8, 25, 5 },
				{ 11, 11, 40, 5 },

				{ 13, 11, 35, 9 },
				{ 15, 13, 35, 10 },
				{ 17, 15, 40, 11 },
				{ 21, 12, 40, 12 },
				{ 22, 13, 40, 12 },
				{ 26, 65, 70, 8 },
				{ 28, 25, 55, 15 },
				{ 33, 65, 75, 40 },

				{  9, 10, 35, 9 },
				{ 12, 15, 70, 35 },
				{ 22, 19, 70, 35 },
				{ 31, 26, 75, 150 },
				{ 32, 35, 75, 200 },
				{ 34, 32, 75, 100 },
				{ 36, 35, 65, 40 },
				{ 44, 70, 80, 250 },

				{  8,  8, 55, 8 },
				{ 23, 20, 75, 35 },
				{ 33, 30, 70, 200 },
				{ 36, 44, 75, 100 },
				{ 38, 90, 75, 250 },
				{ 40, 53, 70, 200 },
				{ 43, 75, 70, 200 },
				{ 46, 70, 75, 250 }
			},
		}
	},

	{
		/*** Tourist ***/

		TV_SORCERY_BOOK,
		0x07,

		A_INT,
		0,

		5,
		450,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99, 0, 0, 0 },
				{ 1, 1, 33, 5 },
				{ 2, 1, 33, 4 },
				{ 2, 2, 33, 5 },
				{ 3, 3, 33, 5 },
				{ 5, 5, 40, 6 },
				{ 6, 6, 33, 7 },
				{ 99, 0, 0, 0 },

				{ 8, 8, 40, 7 },
				{ 9, 9, 60, 7 },
				{ 10, 10, 50, 6 },
				{ 99, 0,  0, 0 },
				{ 13, 11, 50, 6 },
				{ 14, 12, 50, 6 },
				{ 15, 13, 50, 5 },
				{ 16, 14, 50, 5 },

				{ 17, 15, 50, 5 },
				{ 18, 16, 50, 5 },
				{ 99, 0, 0, 0 },
				{ 20, 20, 50, 8 },
				{  5,  3, 30, 8 },
				{ 25, 22, 60, 9 },
				{ 28, 24, 60, 9 },
				{ 20, 25, 70, 12 },

				{ 30, 28, 60, 13 },
				{ 35, 33, 80, 40 },
				{ 99, 0, 0, 0 },
				{ 44, 40, 70, 25 },
				{ 99, 0, 0, 0 },
				{ 99, 0, 0, 0 },
				{ 48, 70, 70, 50 },
				{ 99, 0, 0, 0 }
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Imitator ***/

		0,
		0x00,

		A_STR,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Beastmaster ***/

		TV_SORCERY_BOOK,
		0x06,

		A_CHR,
		0,

		3,
		430,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			/* Trump */
			{
				{ 5, 3, 25, 3 },
				{ 3, 3, 25, 4 },
				{ 8, 7, 37, 8 },
				{ 12, 12, 40, 8 },
				{ 16, 15, 20, 4 },
				{ 20, 25, 30, 6 },
				{ 24, 30, 30, 5 },
				{ 21, 22, 30, 8 },

				{ 32, 30, 40, 8 },
				{ 23, 25, 35, 9 },
				{ 99,  0,  0, 0 },
				{ 29, 28, 40, 12 },
				{ 37, 50, 35, 10 },
				{ 40, 45, 45, 12 },
				{ 42, 60, 40, 15 },
				{ 44, 80, 35, 12 },

				{ 20, 32, 65, 40 },
				{ 99,  0,  0,  0 },
				{ 23, 26, 35, 30 },
				{ 25, 25, 35, 9 },
				{ 27, 30, 35, 35 },
				{ 40, 100, 40, 100 },
				{ 45, 120, 45, 250 },
				{ 99,  0,  0,  0 },

				{ 33, 35, 30, 50 },
				{ 38, 60, 45, 100 },
				{ 32, 50, 40, 150 },
				{ 36, 80, 40, 150 },
				{ 99,  0,  0,  0 },
				{ 99,   0,  0,   0 },
				{ 99,   0,  0,   0 },
				{ 47, 100, 40, 220 }
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Sorcerer ***/

		TV_SORCERY_BOOK,
		0x01,

		A_INT,
		0,

		1,
		40,
		{
			{
				/* Sorcerer: Life magic */

				{ 1, 1, 30, 4 },
				{ 3, 2, 35, 4 },
				{ 4, 3, 35, 4 },
				{ 5, 5, 35, 4 },
				{ 7, 7, 35, 4 },
				{ 9, 8, 40, 4 },
				{ 12, 12, 40, 3 },
				{ 15, 14, 45, 3 },

				{ 16, 16, 45, 4},
				{ 17, 17, 50, 4},
				{ 18, 18, 50, 4},
				{ 19, 19, 50, 4},
				{ 20, 20, 50, 4},
				{ 23, 23, 50, 4},
				{ 30, 30, 55, 5},
				{ 35, 70, 75, 5},

				{ 26, 30, 50, 75 },
				{ 28, 25, 70, 150 },
				{ 33, 33, 60, 75 },
				{ 35, 35, 60, 75 },
				{ 35, 35, 70, 75 },
				{ 35, 55, 80, 115 },
				{ 39, 40, 80, 125 },
				{ 46, 70, 80, 150 },

				{ 9, 9, 50, 40 },
				{ 25, 25, 50, 50 },
				{ 35, 55, 80, 115 },
				{ 42, 100, 80, 225 },
				{ 45, 90, 80, 115 },
				{ 48, 50, 80, 100 },
				{ 49, 100, 80, 250 },
				{ 50, 100, 80, 250 }
			},

			/* Sorcerer: Sorcery */

			{
				{ 1, 1, 23, 4 },
				{ 1, 2, 24, 4 },
				{ 3, 3, 25, 1 },
				{ 3, 3, 30, 1 },
				{ 4, 4, 30, 1 },
				{ 5, 5, 35, 5 },
				{ 6, 5, 30, 4 },
				{ 7, 7, 75, 9 },

				{ 9, 7, 75, 8 },
				{ 10, 7, 75, 8 },
				{ 11, 7, 75, 7 },
				{ 13, 7, 50, 6 },
				{ 18, 12, 60, 8 },
				{ 22, 12, 60, 8 },
				{ 28, 20, 70, 15 },
				{ 33, 30, 75, 20 },

				{ 3, 3, 25, 15 },
				{ 10, 10, 80, 40 },
				{ 14, 10, 60, 25 },
				{ 18, 30, 70, 40 },
				{ 20, 18, 85, 50 },
				{ 20, 18, 60, 25 },
				{ 25, 25, 75, 19 },
				{ 28, 30, 80, 100 },

				{ 10, 10, 40, 20 },
				{ 25, 25, 75, 70 },
				{ 30, 40, 80, 120 },
				{ 33, 35, 60, 100 },
				{ 40, 40, 80, 100 },
				{ 42, 50, 90, 175 },
				{ 43, 40, 55, 200 },
				{ 45, 70, 85, 250 },
			},

			/* Sorcerer: Nature Magic */

			{
				{ 1, 1, 23, 4 },
				{ 3, 3, 25, 3 },
				{ 3, 3, 25, 1 },
				{ 4, 4, 35, 4 },
				{ 4, 4, 50, 5 },
				{ 4, 5, 50, 5 },
				{ 5, 5, 50, 5 },
				{ 5, 5, 35, 4 },

				{ 5, 5, 40, 6 },
				{ 5, 5, 30, 6 },
				{ 7, 6, 45, 6 },
				{ 7, 6, 40, 6 },
				{ 9, 6, 30, 5 },
				{ 19, 12, 55, 8 },
				{ 25, 25, 90, 50 },
				{ 40, 60, 90, 50 },

				{ 9, 12, 40, 44 },
				{ 10, 12, 75, 120 },
				{ 15, 20, 85, 60 },
				{ 20, 22, 80, 40 },
				{ 30, 30, 90, 100 },
				{ 37, 40, 90, 200 },
				{ 38, 45, 75, 200},
				{ 40, 90, 90, 250 },


				{ 20, 18, 60, 25 },
				{ 23, 23, 80, 50 },
				{ 25, 25, 75, 29 },
				{ 30, 27, 75, 35 },
				{ 35, 30, 85, 65 },
				{ 37, 35, 90, 100 },
				{ 40, 90, 95, 250 },
				{ 40, 75, 65, 150 }
			},

				/* Sorcerer: Chaos Magic */

			{
				{ 1, 1, 20, 4 },
				{ 1, 2, 22, 4 },
				{ 2, 2, 25, 4 },
				{ 5, 5, 30, 1 },
				{ 9, 6, 50, 1 },
				{ 13, 9, 45, 6 },
				{ 14, 9, 45, 6 },
				{ 15, 9, 35, 5 },

				{ 17, 10, 25, 5 },
				{ 19, 12, 45, 9 },
				{ 21, 13, 45, 10 },
				{ 23, 15, 50, 11 },
				{ 25, 16, 50, 12 },
				{ 25, 18, 60, 8 },
				{ 30, 20, 80, 15 },
				{ 35, 40, 85, 40 },

				{ 11, 7, 45, 9 },
				{ 15, 15, 80, 35 },
				{ 16, 14, 80, 35 },
				{ 25, 25, 85, 100 },
				{ 30, 25, 85, 150 },
				{ 40, 41, 80, 150 },
				{ 45, 90, 80, 250 },
				{ 47, 100, 90, 250 },

				{ 20, 20, 66, 8 },
				{ 35, 32, 85, 35 },
				{ 37, 34, 75, 40 },
				{ 39, 42, 85, 100 },
				{ 42, 50, 85, 250 },
				{ 45, 58, 85, 200 },
				{ 47, 75, 80, 200 },
				{ 49, 100, 85, 250 }
			},

				/* Sorcerer: Death Magic */

			{
				{ 1, 1, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 3, 3, 27, 3 },
				{ 5, 5, 30, 4 },
				{ 7, 10, 75, 6 },
				{ 9, 9, 30, 4 },
				{ 10, 10, 30, 4 },

				{ 12, 12, 40, 5 },
				{ 13, 12, 30, 4 },
				{ 18, 15, 50, 10 },
				{ 24, 21, 60, 30 },
				{ 30, 75, 80, 30 },
				{ 32, 30, 60, 16 },
				{ 36, 35, 80, 70 },
				{ 39, 30, 95, 25 },

				{ 10, 20, 80, 180 },
				{ 10, 15, 80, 30 },
				{ 11, 11, 30, 15 },
				{ 30, 25, 75, 50 },
				{ 34, 90, 70, 90 },
				{ 36, 35, 60, 125 },
				{ 38, 35, 70, 40 },
				{ 40, 40, 70, 200 },

				{ 20, 20, 75, 50 },
				{ 25, 66, 95 , 250 },
				{ 30, 40, 95, 250 },
				{ 35, 40, 80, 100 },
				{ 42, 50, 75, 150 },
				{ 43, 75, 80, 100 },
				{ 45, 95, 90, 250 },
				{ 47, 100, 90, 250 }
			},

				/* Sorcerer: Trump Magic */

			{
				{ 1, 1, 25, 3 },
				{ 3, 3, 25, 4 },
				{ 5, 5, 37, 8 },
				{ 6, 6, 40, 8 },
				{ 7, 7, 20, 4 },
				{ 11, 12, 30, 6 },
				{ 14, 15, 30, 5 },
				{ 18, 20, 30, 8 },

				{ 20, 20, 40, 8 },
				{ 23, 20, 35, 9 },
				{ 25, 24, 30, 8 },
				{ 30, 28, 40, 12 },
				{ 33, 30, 35, 10 },
				{ 35, 33, 45, 12 },
				{ 40, 35, 40, 15 },
				{ 42, 40, 35, 12 },

				{ 22, 35, 70, 40 },
				{ 24, 24, 35, 25 },
				{ 26, 26, 35, 30 },
				{ 30, 25, 35, 9 },
				{ 30, 30, 35, 35 },
				{ 35, 70, 40, 100 },
				{ 40, 100, 45, 250 },
				{ 45, 100, 45, 200 },

				{ 30, 30, 30, 50 },
				{ 35, 50, 45, 100 },
				{ 36, 55, 40, 150 },
				{ 39, 80, 40, 150 },
				{ 41, 45, 70, 100 },
				{ 47, 100, 40, 150 },
				{ 48, 100, 40, 200 },
				{ 49, 100, 40, 220 }
			},

				/* Sorcerer: Arcane Magic */

			{
				{ 1, 1, 20, 4 },
				{ 1, 1, 33, 5 },
				{ 1, 1, 33, 4 },
				{ 2, 1, 33, 5 },
				{ 2, 2, 33, 5 },
				{ 4, 4, 40, 6 },
				{ 5, 5, 33, 7 },
				{ 6, 5, 44, 5 },

				{ 7, 6, 40, 7 },
				{ 8, 8, 60, 7 },
				{ 9, 8, 50, 6 },
				{ 9, 9, 50, 6 },
				{ 9, 9, 50, 6 },
				{ 11, 10, 50, 6 },
				{ 12, 12, 50, 5 },
				{ 13, 12, 50, 5 },

				{ 14, 12, 50, 5 },
				{ 15, 12, 50, 5 },
				{ 16, 14, 33, 6 },
				{ 18, 15, 50, 8 },
				{ 20, 20, 60, 25 },
				{ 22, 16, 60, 9 },
				{ 25, 18, 60, 9 },
				{ 28, 20, 70, 12 },

				{ 28, 20, 60, 13 },
				{ 31, 25, 70, 30 },
				{ 35, 35, 80, 25 },
				{ 40, 30, 70, 25 },
				{ 41, 30, 66, 30 },
				{ 42, 30, 80, 40 },
				{ 45, 50, 70, 50 },
				{ 49, 100, 80, 200 }
			},

			/* Sorcerer: Craft Magic */

			{

				{  1,  1, 20,   4},
				{  2,  2, 20,   4},
				{  3,  3, 30,   3},
				{  5,  5, 35,   5},
				{  6,  5, 35,   5},
				{  8,  6, 45,   6},
				{  9,  5, 35,   5},
				{ 10,  5, 35,   5},

				{ 12, 12, 45,   4},
				{ 15, 14, 40,   4},
				{ 17, 15, 60,   6},
				{ 20, 17, 75,  80},
				{ 23, 18, 85,  50},
				{ 26, 22, 50,   4},
				{ 29, 30, 60,   5},
				{ 38, 65, 95,  50},

				{ 12, 12, 45,  12},
				{ 14, 10, 75, 120},
				{ 20, 20, 85,  60},
				{ 24, 12, 60,  10},
				{ 30, 55, 80,  50},
				{ 34, 70, 80, 150},
				{ 41, 80, 90, 200},
				{ 45, 55, 70, 200},

				{ 14, 14, 65,  50},
				{ 30, 32, 75, 150},
				{ 36, 45, 90, 200},
				{ 38, 80, 95, 200},
				{ 38,100, 95, 200},
				{ 40,100, 90, 250},
				{ 43,100, 70, 250},
				{ 47, 70, 80, 250}
			},

				/* Sorcerer: Daemon Magic */

			{
				{ 1, 1, 20, 4 },
				{ 2, 1, 22, 1 },
				{ 3, 2, 25, 4 },
				{ 4, 5, 30, 4 },
				{ 7, 5, 45, 4 },
				{ 10, 7, 40, 6 },
				{ 11, 9, 35, 5 },
				{ 13, 13, 40, 10 },

				{ 15, 14, 45, 9 },
				{ 17, 15, 60, 10 },
				{ 20, 18, 50, 11 },
				{ 23, 15, 50, 12 },
				{ 25, 16, 50, 12 },
				{ 30, 75, 80, 30 },
				{ 32, 28, 70, 15 },
				{ 36, 70, 85, 40 },

				{ 12, 12, 45, 9 },
				{ 15, 20, 60, 20 },
				{ 25, 22, 65, 35 },
				{ 34, 30, 80, 100 },
				{ 35, 40, 80, 100 },
				{ 37, 35, 85, 100 },
				{ 40, 40, 85, 40 },
				{ 46, 80, 80, 250 },

				{  9,  9, 50, 8 },
				{ 26, 25, 85, 35 },
				{ 36, 35, 80, 200 },
				{ 41, 50, 85, 100 },
				{ 43,100, 90, 250 },
				{ 45, 58, 85, 200 },
				{ 47, 85, 80, 200 },
				{ 49, 80, 85, 250 }
			},
		},
	},

	{
		/*** Archer ***/

		0,
		0x00,

		A_STR,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Magic eater ***/

		0,
		0x00,

		A_INT,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Harper ***/

		TV_MUSIC_BOOK,
		0x04,

		A_CHR,
		0,

		1,
		400,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Red Mage ***/

		TV_SORCERY_BOOK,
		0x03,

		A_INT,
		0,

		1,
		430,

		{
			/* Red Mage: Life Magic */
			{
				{ 2, 2, 30, 4 },
				{ 4, 4, 35, 4 },
				{ 5, 5, 35, 4 },
				{ 7, 8, 35, 4 },
				{ 9, 9, 35, 4 },
				{ 11, 10, 40, 4 },
				{ 15, 15, 40, 3 },
				{ 18, 18, 45, 3 },

				{ 20, 20, 45, 4},
				{ 23, 23, 50, 4},
				{ 26, 25, 50, 4},
				{ 30, 28, 50, 4},
				{ 34, 30, 50, 4},
				{ 38, 32, 50, 4},
				{ 42, 38, 55, 5},
				{ 45, 80, 75, 5},

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Sorcery */
			{
				{ 3, 1, 50, 1 },
				{ 5, 2, 55, 1 },
				{ 7, 3, 65, 1 },
				{ 8, 4, 65, 1 },
				{ 12, 7, 75, 1 },
				{ 15, 8, 75, 1 },
				{ 18, 10, 75, 1 },
				{ 22, 13, 80, 1 },

				{ 26, 16, 80, 1 },
				{ 29, 17, 80, 1 },
				{ 32, 19, 75, 2 },
				{ 34, 23, 80, 4 },
				{ 35, 26, 80, 5 },
				{ 39, 29, 70, 6 },
				{ 42, 35, 80, 12 },
				{ 46, 40, 75, 20 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Nature Magic */
			{
				{ 3, 1, 35, 2 },
				{ 4, 3, 40, 2 },
				{ 6, 5, 40, 3 },
				{ 7, 8, 55, 2 },
				{ 9, 8, 50, 3 },
				{ 10, 8, 50, 3 },
				{ 12, 9, 50, 3 },
				{ 14, 10, 45, 3 },

				{ 16, 10, 80, 4 },
				{ 18, 12, 40, 3 },
				{ 19, 13, 40, 4 },
				{ 21, 15, 55, 4 },
				{ 23, 15, 55, 4 },
				{ 26, 27, 65, 8 },
				{ 37, 38, 75, 10 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Chaos Magic */
			{
				{ 3, 2, 20, 1 },
				{ 4, 3, 35, 2 },
				{ 6, 4, 35, 2 },
				{ 9, 6, 45, 2 },
				{ 16, 14, 40, 2 },
				{ 23, 18, 50, 6 },
				{ 26, 24, 60, 3 },
				{ 28, 25, 60, 3 },

				{ 29, 26, 60, 5 },
				{ 32, 29, 60, 8 },
				{ 35, 35, 70, 13 },
				{ 38, 36, 70, 10 },
				{ 41, 40, 75, 15 },
				{ 43, 35, 65, 5 },
				{ 46, 38, 95, 15 },
				{ 49, 60, 85, 30 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Death Magic */
			{
				{ 2, 2, 25, 4 },
				{ 4, 2, 25, 4 },
				{ 6, 4, 25, 4 },
				{ 8, 6, 27, 4 },
				{ 11, 10, 30, 4 },
				{ 14, 13, 75, 6 },
				{ 16, 14, 30, 4 },
				{ 20, 18, 40, 5 },

				{ 22, 21, 40, 5 },
				{ 25, 22, 30, 4 },
				{ 27, 27, 50, 10 },
				{ 35, 35, 85, 10 },
				{ 38, 90, 90, 30 },
				{ 41, 38, 60, 16 },
				{ 48, 50, 80, 70 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Trump Magic */
			{
				{ 3, 2, 25, 3 },
				{ 7, 5, 25, 4 },
				{ 9, 8, 37, 8 },
				{ 12, 10, 40, 8 },
				{ 15, 12, 20, 4 },
				{ 18, 17, 30, 6 },
				{ 21, 23, 30, 5 },
				{ 25, 30, 30, 8 },

				{ 29, 25, 40, 8 },
				{ 32, 27, 35, 9 },
				{ 35, 30, 30, 8 },
				{ 38, 40, 40, 12 },
				{ 42, 44, 35, 10 },
				{ 44, 45, 45, 12 },
				{ 47, 51, 40, 15 },
				{ 49, 55, 35, 12 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Arcane Magic */
			{
				{ 1, 1, 20, 4 },
				{ 2, 1, 33, 5 },
				{ 3, 2, 33, 4 },
				{ 4, 3, 33, 5 },
				{ 5, 4, 33, 5 },
				{ 7, 6, 40, 6 },
				{ 8, 7, 33, 7 },
				{ 9, 8, 44, 5 },

				{ 10, 9, 40, 7 },
				{ 11, 10, 60, 7 },
				{ 13, 12, 50, 6 },
				{ 14, 13, 50, 6 },
				{ 15, 13, 50, 6 },
				{ 16, 14, 50, 6 },
				{ 18, 15, 50, 5 },
				{ 19, 16, 50, 5 },

				{ 20, 17, 50, 5 },
				{ 21, 18, 50, 5 },
				{ 22, 19, 33, 6 },
				{ 24, 23, 50, 8 },
				{ 26, 32, 60, 25 },
				{ 27, 28, 60, 9 },
				{ 30, 29, 60, 9 },
				{ 33, 32, 70, 12 },

				{ 36, 35, 60, 13 },
				{ 39, 38, 80, 40 },
				{ 42, 41, 80, 25 },
				{ 46, 43, 70, 25 },
				{ 48, 46, 66, 30 },
				{ 49, 48, 80, 40 },
				{ 50, 70, 70, 50 },
				{ 99,  0,  0,  0 }
			},

			/* Red Mage: Craft Magic */
			{
				{  2,  2, 40,   2},
				{  4,  4, 40,   2},
				{  7,  6, 45,   1},
				{ 10, 12, 50,   2},
				{ 13, 12, 50,   2},
				{ 16, 14, 55,   3},
				{ 19, 12, 50,   2},
				{ 21, 12, 50,   2},

				{ 25, 20, 55,   2},
				{ 29, 23, 60,   2},
				{ 32, 27, 65,   3},
				{ 36, 31, 75,  40},
				{ 38, 26, 70,  25},
				{ 41, 40, 70,   2},
				{ 44, 38, 55,   3},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},

			/* Red Mage: Daemon Magic */
			{
				{ 3, 2, 20, 1 },
				{ 5, 3, 35, 2 },
				{ 7, 5, 35, 2 },
				{ 10, 8, 45, 2 },
				{ 14, 9, 40, 2 },
				{ 20, 13, 60, 3 },
				{ 23, 18, 60, 3 },
				{ 25, 25, 60, 5 },

				{ 28, 27, 60, 8 },
				{ 32, 29, 70, 13 },
				{ 35, 33, 70, 10 },
				{ 38, 36, 75, 15 },
				{ 41, 40, 75, 15 },
				{ 42, 90, 90, 30 },
				{ 47, 50, 75, 15 },
				{ 49, 111, 90, 30 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },

				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
				{ 99,  0,  0,  0 },
			},
		}
	},

	{
		/*** Samurai ***/

		TV_HISSATSU_BOOK,
		0x00,

		A_WIS,
		0,

		0,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** ForceTrainer ***/

		TV_LIFE_BOOK,
		0x06,

		A_WIS,
		0,

		1,
		350,

		{
			{
				{ 1, 1, 30, 4 },
				{ 2, 2, 35, 4 },
				{ 3, 3, 35, 4 },
				{ 4, 4, 35, 4 },
				{ 5, 5, 35, 4 },
				{ 8, 6, 40, 4 },
				{ 11, 10, 40, 3 },
				{ 13, 12, 45, 3 },

				{ 15, 12, 45, 4},
				{ 16, 15, 50, 4},
				{ 17, 15, 50, 4},
				{ 18, 16, 50, 4},
				{ 19, 16, 50, 4},
				{ 20, 18, 50, 4},
				{ 30, 25, 55, 5},
				{ 35, 70, 75, 5},

				{ 26, 26, 50, 75 },
				{ 28, 25, 70, 150 },
				{ 32, 32, 60, 75 },
				{ 35, 35, 60, 75 },
				{ 35, 35, 70, 75 },
				{ 35, 55, 80, 115 },
				{ 39, 40, 80, 125 },
				{ 46, 64, 80, 150 },

				{ 9,  9, 50, 40 },
				{ 25, 25, 50, 50 },
				{ 35, 45, 80, 115 },
				{ 42, 84, 80, 225 },
				{ 45, 64, 80, 115 },
				{ 48, 40, 80, 100 },
				{ 49, 100, 90, 200 },
				{ 50, 100, 80, 250 }
			},

			/* ForceTrainer: Sorcery */
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* ForceTrainer: Nature Magic */
			{
				{ 1, 1, 35, 2 },
				{ 3, 3, 40, 2 },
				{ 4, 4, 40, 3 },
				{ 5, 5, 55, 2 },
				{ 6, 6, 50, 3 },
				{ 7, 7, 50, 3 },
				{ 8, 8, 50, 3 },
				{ 9, 9, 45, 3 },

				{ 10, 8, 80, 4 },
				{ 11, 9, 40, 3 },
				{ 12, 10, 40, 4 },
				{ 14, 12, 55, 4 },
				{ 16, 12, 55, 4 },
				{ 18, 15, 65, 8 },
				{ 31, 31, 75, 10 },
				{ 40, 60, 90, 50 },

				{ 12, 12, 50, 25 },
				{ 15, 15, 70, 60 },
				{ 20, 22, 85, 70 },
				{ 25, 30, 85, 40 },
				{ 35, 35, 80, 80 },
				{ 40, 40, 90, 200 },
				{ 45, 48, 90, 250 },
				{ 49, 80, 90, 250 },

				{ 25, 25, 60, 150 },
				{ 27, 27, 60, 160 },
				{ 28, 28, 75, 125 },
				{ 33, 33, 75, 35 },
				{ 38, 38, 75, 45 },
				{ 41, 40, 75, 100 },
				{ 45, 85, 95, 250 },
				{ 48, 85, 75, 150 }
			},

			/* ForceTrainer: Chaos Magic */
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			/* ForceTrainer: Death Magic */
			{
				{ 1, 1, 25, 4 },
				{ 2, 2, 25, 4 },
				{ 3, 3, 25, 4 },
				{ 4, 4, 27, 4 },
				{ 7, 7, 30, 4 },
				{ 9, 10, 75, 6 },
				{ 10, 10, 30, 4 },
				{ 12, 12, 40, 5 },

				{ 14, 14, 40, 5 },
				{ 16, 16, 30, 4 },
				{ 21, 21, 50, 10 },
				{ 27, 27, 85, 30},
				{ 33, 75, 90, 30 },
				{ 36, 35, 60, 16 },
				{ 44, 55, 80, 70 },
				{ 46, 45, 95, 100 },

				{ 10, 22, 80, 180 },
				{ 12, 18, 80, 30 },
				{ 14, 18, 30, 15 },
				{ 30, 30, 75, 50 },
				{ 42, 90, 70, 90 },
				{ 43, 40, 60, 125 },
				{ 43, 44, 70, 40 },
				{ 45, 50, 80, 200 },

				{ 24, 24, 75, 50 },
				{ 33, 85, 85, 220 },
				{ 35, 45, 95, 250 },
				{ 39, 55, 80, 200 },
				{ 46, 55, 75, 150 },
				{ 48, 80, 80, 125 },
				{ 50, 110, 95, 250 },
				{ 50, 115, 95, 250 }
			},

			/* ForceTrainer: No Trump magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
			},

			/* ForceTrainer: No Arcane Magic */
			{
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},

				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0},
				{ 99, 0, 0, 0}
			},

			/* ForceTrainer: Craft Magic */
			{

				{  1,  2, 30,   4},
				{  3,  3, 35,   4},
				{  5,  4, 40,   3},
				{  7,  8, 45,   5},
				{  8,  8, 45,   5},
				{ 10, 10, 50,   6},
				{ 12,  8, 45,   5},
				{ 13,  8, 45,   5},

				{ 15, 17, 55,   4},
				{ 20, 20, 60,   4},
				{ 22, 22, 65,   6},
				{ 25, 24, 80,  80},
				{ 26, 25, 85,  50},
				{ 31, 30, 60,   4},
				{ 34, 28, 55,   5},
				{ 44, 80, 80,  50},

				{ 16, 16, 55,  12},
				{ 18, 13, 70, 120},
				{ 24, 25, 85,  60},
				{ 28, 18, 70,  10},
				{ 35, 60, 80,  50},
				{ 38, 85, 90, 150},
				{ 45,100, 95, 200},
				{ 48, 65, 75, 200},

				{ 17, 17, 70,  50},
				{ 33, 34, 70, 150},
				{ 40, 55, 90, 200},
				{ 40, 85, 95, 200},
				{ 43,100, 95, 200},
				{ 45,110, 80, 250},
				{ 47,120, 90, 250},
				{ 50, 75, 90, 250}
			},

			/* ForceTrainer: Daemon Magic */
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		}
	},

	{
		/*** Blue Mage ***/

		TV_SORCERY_BOOK,
		0x01,

		A_INT,
		0,

		99,
		430,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Kihei ***/

		0,
		0x00,

		A_STR,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Berserker ***/

		0,
		0x00,

		A_STR,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},

	{
		/*** Kaji ***/

		0,
		0x00,

		A_STR,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},
	{
		/*** Mirror Master ***/

		TV_LIFE_BOOK,
		0x00,

		A_INT,
		0,

		99,
		400,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},
	{
		/*** Ninja ***/

		0,
		0x00,

		A_DEX,
		0,

		99,
		0,
		{
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},

			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
			{

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},

				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0},
				{ 99,  0,  0,   0}
			},
		},
	},
};


magic_type technic_info[NUM_TECHNIC][32] =
{
	{
		/* Music */
		{ 1,  1,  10,   2},
		{ 2,  1,  10,   2},
		{ 3,  2,  20,   3},
		{ 4,  2,  20,   4},
		{ 5,  2,  20,   6},
		{ 7,  4,  30,   8},
		{ 9,  3,  30,   10},
		{ 10, 2,  30,   12},

		{ 12,  3,   40,   20},
		{ 15, 16,  42,   35},
		{ 17, 18,  40,   25},
		{ 18,  2,  45,   30},
		{ 23,  8,  50,   38},
		{ 28, 30,  50,   41},
		{ 33, 35,  60,   42},
		{ 38, 35,  70,   46},

		{ 10,  4,  20,   13},
		{ 22,  5,  30,   26},
		{ 23,  3,  35,   27},
		{ 26,  28,  37,   29},
		{ 32,  37,  41,   36},
		{ 33,  22,  43,   40},
		{ 37,  35,  46,   42},
		{ 45,  60,  50,   56},

		{ 23,  18,  20,   23},
		{ 30,  30,  30,   26},
		{ 33,  65,  41,   30},
		{ 37,  35,  43,   35},
		{ 40,  30,  46,   50},
		{ 42,  75,  50,   68},
		{ 45,  58,  62,   73},
		{ 49,  48,  70,  200}
	},

	{
		/* Hissatsu */
		{ 1,   15,   0,   0},
		{ 3,   10,   0,   0},
		{ 6,   15,   0,   0},
		{ 9,    8,   0,   0},
		{ 10,  12,   0,   0},
		{ 12,  25,   0,   0},
		{ 14,   7,   0,   0},
		{ 17,  20,   0,   0},

		{ 19,  10,   0,   0},
		{ 22,  20,   0,   0},
		{ 24,  30,   0,   0},
		{ 25,  10,   0,   0},
		{ 27,  15,   0,   0},
		{ 29,  45,   0,   0},
		{ 32,  70,   0,   0},
		{ 35,  50,   0,   0},

		{ 18,  40,   0,   0},
		{ 22,  22,   0,   0},
		{ 24,  30,   0,   0},
		{ 26,  35,   0,   0},
		{ 30,  30,   0,   0},
		{ 32,  60,   0,   0},
		{ 36,  40,   0,   0},
		{ 39,  80,   0,   0},

		{ 26,  20,   0,   0},
		{ 29,  40,   0,   0},
		{ 31,  35,   0,   0},
		{ 36,  80,   0,   0},
		{ 39, 100,   0,   0},
		{ 42, 110,   0,   0},
		{ 45, 130,   0,   0},
		{ 50, 255,   0,   0}
	},
};


/*
 * Zangband uses this array instead of the spell flags table, as there
 * are 5 realms of magic, each with 4 spellbooks and 8 spells per book -- TY
 */
u32b fake_spell_flags[4]=
{
	0x000000ff,
	0x0000ff00,
	0x00ff0000,
	0xff000000
};


s32b realm_choices1[] =
{
	(CH_NONE),				/* Warrior */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON),   /* Mage */
	(CH_LIFE | CH_DEATH | CH_DAEMON),       /* Priest */
	(CH_SORCERY | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT),               /* Rogue */
	(CH_NATURE),                            /* Ranger */
	(CH_LIFE | CH_DEATH),                   /* Paladin */
	(CH_ARCANE),                            /* Warrior-Mage */
	(CH_CHAOS | CH_DAEMON),                 /* Chaos-Warrior */
	(CH_LIFE | CH_NATURE | CH_DEATH |
	 CH_ENCHANT),                           /* Monk */
	(CH_NONE),                              /* Mindcrafter */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON),   /* High-Mage */
	(CH_ARCANE),                            /* Tourist */
	(CH_NONE),                              /* Imitation */
	(CH_TRUMP),                             /* Beastmaster */
	(CH_NONE),                              /* Sorcerer */
	(CH_NONE),                              /* Archer */
	(CH_NONE),                              /* Magic eater */
	(CH_MUSIC),                             /* Harper */
	(CH_NONE),                              /* Red Mage */
	(CH_HISSATSU),                          /* Samurai */
	(CH_LIFE | CH_NATURE | CH_DEATH |
	 CH_ENCHANT),                           /* ForceTrainer */
	(CH_NONE),                              /* Blue Mage */
	(CH_NONE),				/* Kihei */
	(CH_NONE),				/* Berserker */
	(CH_NONE),				/* Kaji */
	(CH_NONE),				/* Mirror-master */
};


s32b realm_choices2[] =
{
	(CH_NONE),                              /* Warrior */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON),   /* Mage */
	(CH_LIFE | CH_SORCERY | CH_NATURE |
	 CH_CHAOS | CH_DEATH | CH_TRUMP |
	 CH_ARCANE | CH_ENCHANT | CH_DAEMON),   /* Priest */
	(CH_NONE),                              /* Rogue */
	(CH_SORCERY | CH_CHAOS | CH_DEATH |
	 CH_TRUMP | CH_ARCANE | CH_DAEMON),     /* Ranger */
	(CH_NONE),                              /* Paladin */
	(CH_LIFE | CH_NATURE | CH_CHAOS |
	 CH_DEATH | CH_TRUMP | CH_ARCANE |
	 CH_SORCERY | CH_ENCHANT | CH_DAEMON),  /* Warrior-Mage */
	(CH_NONE),                              /* Chaos-Warrior */
	(CH_NONE),                              /* Monk */
	(CH_NONE),                              /* Mindcrafter */
	(CH_NONE),                              /* High-Mage */
	(CH_NONE),                              /* Tourist */
	(CH_NONE),                              /* Imitation */
	(CH_NONE),                              /* Beastmanster */
	(CH_NONE),                              /* Sorcerer */
	(CH_NONE),                              /* Archer */
	(CH_NONE),                              /* Magic eater */
	(CH_NONE),                              /* Harper */
	(CH_NONE),                              /* Red Mage */
	(CH_NONE),                              /* Samurai */
	(CH_NONE),                              /* ForceTrainer */
	(CH_NONE),                              /* Blue Mage */
	(CH_NONE),				/* Kihei */
	(CH_NONE),				/* Berserker */
	(CH_NONE),				/* Kaji */
	(CH_NONE),				/* Mirror-master */
};


#ifdef JP
cptr realm_names[] =
{
	"ˡʤ",
	"̿",
	"",
	"",
	"",
	"Ź",
	"ȥ",
	"",
	"",
	"",
	"",
	"",
	"",
	"",
	"",
	"",
	"",
	"",
	""
};
#endif

#ifdef JP
cptr E_realm_names[]
#else
cptr realm_names[]
#endif
= {
	"no magic",
	"Life",
	"Sorcery",
	"Nature",
	"Chaos",
	"Death",
	"Trump",
	"Arcane",
	"Craft",
	"Daemon",
	"unknown",
	"unknown",
	"unknown",
	"unknown",
	"unknown",
	"unknown",
	"Music",
	"Kendo",
	"unknown"
};

/*
 * Names of the spells (mage spells then priest spells)
 */
#ifdef JP
cptr spell_names[VALID_REALM][32] =
{
	/*** Life Spells ***/
	{
		/* Common Life Spellbooks */
"ٰ¸ߴ",
"ڽμ",
"ʡ",
"ݽ",
"ξ",
" & ⴶ",
"Žμ",
"ʢ­",

"",
"",
"̿μ",
"Ʃǧ",
"ʤ",
"мٰ볦",
"ϲ",
"볦",

		/* Rare Life Spellbooks */
"ʧ",
"**",
"ǥå & ໶",
"ι",
"ٰ໶",
"ٰ",
"ʤ",
"볦",

"ε",
"",
"ǥåɾ",
"",
"*ϲ*",
"ʤӥ",
"",
"ˤ"
	},

	/*** Sorcery Spells ***/

	{
		/* Common Sorcery Spellbooks */
"󥹥",
"硼ȡƥݡ",
"櫤ⴶ",
"饤ȡꥢ",
"ѥ˥å󥹥",
"ƥݡ",
"꡼ס󥹥",
"ϽŶ",

"ˡϿ",
"",
"󥹥",
"ե꡼",
"ƥݡȡ󥹥",
"ԡ",
"",
"",

		/* Rare Sorcery Spellbooks */
"ʪΤȺ",
"㡼ࡦ󥹥",
"",
"ư",
"ʬ",
"ƥݡȡ٥",
"Ԥμʸ",
"ȯΥ롼",

"Ĵ",
"ǰư",
"Τ",
"̥λλ",
"",
"ϣ",
"ʪ",
"̵ε"

	},

	/*** Nature Spells ***/

	{
		/* Common Nature Spellbooks */
"󥹥",
"",
"櫤ⴶ",
"",
"θ",
"ưʪ",
"Ķؤ",
"Ǽ",

"ϲ",
"ܥ",
"γ",
"եܥ",
"۸",
"­",
"ưʪ",
"",

		/* Rare Nature Spellbooks */
"",
"ȩв",
"",
"¤",
"ưʪͧ",
"",
"Ф",
"忩ɻ",

"Ͽ",
"ޥ",
"֥ꥶ",
"",
"Ĭ",
"۸",
"ο",
"ζ"

	},

	/*** Chaos Spells ***/

	{
		/* Common Chaos Spellbooks */
"ޥåߥ",
"ȥå/ɥ˲",
"",
"μ",
"",
"եܥ",
"Ϥη",
"ƥݡ",

"",
"ܥ",
"˥å֡",
"Ǥ",
"եܡ",
"ƥݡȡ",
"˲θ",
"륹ȯư",

		/* Rare Chaos Spellbooks */
"¾",
"Ϣ",
"",
"ʬ",
"",
"ޥåå",
"٤ο",
"⾤",

"ϸ",
"ή",
"ΰ",
"پ",
"",
"Ϥ",
"륹Υ֥쥹",
"̵"

	},

	/*** Death Spells ***/

	{
		/* Common Death Spellbooks */
"̵̿",
"",
"ٰ",
"",
"̲",
"",
"",
"ǥåɽ°",

"ȥԡε",
"Ϲ",
"٤",
"󥹥",
"Ǥο",
"۷ɥ쥤",
"ȿν",
"",

		/* Rare Death Spellbooks */
"β",
"",
"Ź",
"",
"۷ο",
"۷",
"θ",
"Ź",

"θ",
"Ծ",
"Ԥ",
"۷쵴Ѳ",
"̿",
"",
"Ϲι",
"ͩβ"

	},

	/*** Trump Spells ***/

	{
		/* Common Trump Spellbooks */
"硼ȡƥݡ",
"Υ",
"åե",
"եꥻå",
"ƥݡ",
"ΤΥ",
"ƥݡȡ󥹥",
"ưʪΥ",

"ưΥ",
"ߥΥ",
"",
"ԡɡ󥹥",
"ƥݡȡ٥",
"",
"Ԥμʸ",
"ʪ",

		/* Rare Trump Spellbooks */
"ָ򴹤Υ",
"ǥåɤΥ",
"Υ",
"󥹥Υ",
"ϥɤΥ",
"ȥפο",
"ʹ֥ȥ",
"СǡΥ",

"ͽΥ",
"μΥ",
"󥹥",
"ɥ饴Υ",
"ФΥ",
"ǡΥ",
"ϹΥ",
"ɥ饴Υ"

	},

	/*** Arcane Spells (_only_ common spells) ***/

	{
"ŷ",
"ˡλܾ",
"Ʃδ",
"󥹥",
"硼ȡƥݡ",
"饤ȡꥢ",
"櫤 ˲",
"ڽμ",

"櫤 ",
"ǳ",
"",
"ˡ ",
"ƥശ",
"",
"",
"Ѳ",

"",
"ѻ",
"Žμ",
"ƥݡ",
"",
"ϲ",
"",
"ʢ­",

"Ʃǧ",
"󥿥뾤",
"ƥݡȡ٥",
"ƥݡȡ󥹥",
"Ǥε",
"",
"Ԥμʸ",
"Τ",

	},

	/*** Craft Spells ***/

	{
		/* Common Craft Spellbooks */
"ֳ",
"϶",
"ʢ­",
"䵤",
"Ѳб",
"ε",
"ŷ",
"ѻ",

"Ʃǧ",
"",
"",
"β",
"ʬ",
"мٰ볦",
"",
"ˡ",

		/* Rare Craft Spellbooks */
"ƥѥ",
"ȩв",
"",
"ԡ",
"ȴ",
"᤭",
"¤",
"ˡγ",

"̵ϲ",
"ʴ",
"ʤμ",
"ﶯ",
"ɶ񶯲",
"°Ϳ",
"ʹ֥ȥ",
"°ؤȱ",

	},

	/*** Daemon Spells ***/

	{
		/* Common Daemon Spellbooks */
"ޥåߥ",
"̵̿",
"٤ʤʡ",
"Ѳб",
"",
"Ϲ",
"λ",
"Ϲα",

"ǡ",
"ӥ",
"Ϲ",
"ץ饺ޡܥ",
"եܡ",
"ο",
"Ϲ",
"ǡ󾤴",

		/* Rare Daemon Spellbooks */
"",
"Υ",
"ϴή",
"ץ饺޵",
"Ѳ",
"Ϲư",
"Хʭ",
"Ǥμ",

"ε",
"Ǥ",
"αߴ",
"ڥå",
"졼ǡ󾤴",
"Ϲ",
"μ",
"ⲦѲ",

	},

	/*** Music Spells ***/

	/* 
	   Song : , Note : , Pattern : Χ, 
	   Ballad : (ضʢֱפ?), Chant : , */
	{
		/******* Apprentice Handbook *******/
		"ߤβ",	/*"Song of Holding"*/
		"ʡβ",	/*"Song of Blessing"*/
		"β",	/*"Wracking Note"*/
		"ۯ۰Χ",	/*"Stun Pattern"*/
		"̿ή",	/*"Flow of Life"*/
		"ۤβ",	/*"Song of the Sun"*/
		"ݤβ",
		"襤β",	/*"Heroic Ballad"*/

		/******* Minstrel's Music *******/
		"Ūγ",	/*"Clairaudience"*/
		"β",
		"μβ",	/*"Song of Lore"*/
		"ۤβ",
		"ƤΧ",	/*"Illusion Pattern"*/
		"Ǥζ",	/*"Doomcall"*/
		"եꥨβ",	/*"Firiel's Song"*/
		"ι",	/*"Fellowship Chant"*/

		/******* Harps of Rivendell *******/
		"ʬ",
		"",
		"ۥӥåȤΥǥ",
		"Ĥ",
		"໶β",
		"ޥδŸ",
		"β",
		"⤦Ĥ",

		/******* Lays of Beleriand *******/
		"˲Χ",
		"ڤβ",
		"β",
		"ͺλ",
		"ʤν",
		"β",
		"",
		"ե󥴥եĩ",
	},

	/*** Hissatsuwaza ***/

	{
		"ӹ",
		"޷¤",
		"֡",
		"",
		"",
		"ߤǤ",
		"󥿡",
		"ʧȴ",

		"ڥĥ",
		"",
		"",
		"ᾢ",
		"˴",
		"",
		"޽ͤ",
		"¤",

		"Τƿ",
		"޻",
		"",
		"ή",
		"̷",
		"",
		"ͺ",
		"̵л",

		"۷쵴β",
		"",
		"ɴͻ¤",
		"ŷζ",
		"Ťη",
		"",
		"ıǦ",
		"ʢ",
	},
};

#else

cptr spell_names[VALID_REALM][32] =
{
	/*** Life Spells ***/
	{
		/* Common Life Spellbooks */
		"Detect Evil",
		"Cure Light Wounds",
		"Bless",
		"Remove Fear",
		"Call Light",
		"Detect Doors & Traps",
		"Cure Medium Wounds",
		"Satisfy Hunger",

		"Remove Curse",
		"Cure Poison",   
		"Cure Critical Wounds",
		"Sense Unseen",
		"Holy Orb",
		"Protection from Evil",
		"Healing",
		"Glyph of Warding",

		/* Rare Life Spellbooks */
		"Exorcism",
		"Dispel Curse",
		"Dispel Undead & Demons",
		"Day of the Dove",
		"Dispel Evil",
		"Banishment",
		"Holy Word",
		"Warding True",

		"Heroism",
		"Prayer",
		"Turn Undead",
		"Restoration",
		"Healing True",
		"Holy Vision",
		"Divine Intervention",
		"Ultimate Resistance",
	},

	/*** Sorcery Spells ***/

	{
		/* Common Sorcery Spellbooks */
		"Detect Monsters",
		"Phase Door",
		"Detect Doors and Traps",
		"Light Area",
		"Confuse Monster",
		"Teleport",
		"Sleep Monster",
		"Recharging",

		"Magic Mapping",
		"Identify",
		"Slow Monster",
		"Mass Sleep",
		"Teleport Away",
		"Haste Self",
		"Detection True",
		"Identify True",

		/* Rare Sorcery Spellbooks */
		"Detect items and Treasure",
		"Charm Monster",
		"Sense Minds",
		"Teleport to town",
		"Self Knowledge",
		"Teleport Level",
		"Word of Recall",
		"Explosive Rune",

		"Probing",
		"Telekinesis",
		"Clairvoyance",
		"Charm monsters",
		"Dimension Door",
		"Alchemy",
		"Banishment",
		"Globe of Invulnerability",
	},

	/*** Nature Spells ***/

	{
		/* Common Nature Spellbooks */
		"Detect Creatures",
		"Lightning",
		"Detect Doors and Traps",
		"Produce Food",
		"Daylight",
		"Animal Taming",
		"Resist Environment",
		"Cure Wounds & Poison",

		"Stone to Mud",
		"Frost Bolt",
		"Nature Awareness",
		"Fire Bolt",
		"Ray of Sunlight",
		"Entangle",
		"Summon Animal",
		"Herbal Healing",

		/* Rare Nature Spellbooks */
		"Stair Building",
		"Stone Skin",
		"Resistance True",
		"Forest Creation",
		"Animal Friendship",
		"Stone Tell",
		"Wall of Stone",
		"Protect from Corrosion",

		"Earthquake",
		"Cyclone",
		"Blizzard",
		"Lightning Storm",
		"Whirlpool",
		"Call Sunlight",
		"Elemental Branding",
		"Nature's Wrath",
	},

	/*** Chaos Spells ***/

	{
		/* Common Chaos Spellbooks */
		"Magic Missile",
		"Trap / Door Destruction",
		"Flash of Light",
		"Touch of Confusion",
		"Mana Burst",
		"Fire Bolt",
		"Fist of Force",
		"Teleport Self",

		"Wonder",
		"Chaos Bolt",
		"Sonic Boom",
		"Doom Bolt",
		"Fire Ball",
		"Teleport Other",
		"Word of Destruction",
		"Invoke Logrus",

		/* Rare Chaos Spellbooks */
		"Polymorph Other",
		"Chain Lightning",
		"Arcane Binding",
		"Disintegrate",
		"Alter Reality",
		"Magic Rocket",
		"Chaos Branding",
		"Summon Demon",

		"Beam of Gravity",
		"Meteor Swarm",
		"Flame Strike",
		"Call Chaos",
		"Polymorph Self",
		"Mana Storm",
		"Breathe Logrus",
		"Call the Void",
	},

	/*** Death Spells ***/

	{
		/* Common Death Spellbooks */
		"Detect Unlife",
		"Malediction",
		"Detect Evil",
		"Stinking Cloud",
		"Black Sleep",
		"Resist Poison",
		"Horrify",
		"Enslave Undead",

		"Orb of Entropy",
		"Nether Bolt",
		"Cloud kill",
		"Genocide One",
		"Poison Branding",
		"Vampiric Drain",
		"Animate dead",
		"Genocide",

		/* Rare Death Spellbooks */
		"Berserk",
		"Invoke Spirits",
		"Dark Bolt",
		"Battle Frenzy",
		"Vampiric Branding",
		"Vampirism True",
		"Nether Wave",
		"Darkness Storm",

		"Death Ray",
		"Raise the Dead",
		"Esoteria",
		"Polymorph Vampire",
		"Restore Life",
		"Mass Genocide",
		"Hellfire",
		"Wraithform",
	},

	/*** Trump Spells ***/

	{
		/* Common Trump Spellbooks */
		"Phase Door",
		"Trump Spiders",
		"Shuffle",
		"Reset Recall",
		"Teleport",
		"Trump Spying",
		"Teleport Away",
		"Trump Animals",

		"Trump Reach",
		"Trump Kamikaze",
		"Phantasmal Servant",
		"Haste Monster",
		"Teleport Level",
		"Dimension Door",
		"Word of Recall",
		"Banish",

		/* Rare Trump Spellbooks */
		"Swap Position",
		"Trump Undead",
		"Trump Reptiles",
		"Trump Monsters",
		"Trump Hounds",
		"Trump Branding",
		"Living Trump",
		"Trump Cyberdemon",

		"Trump Divination",
		"Trump Lore",
		"Heal Monster",
		"Trump Dragon",
		"Trump Meteor",
		"Trump Demon",
		"Trump Greater Undead",
		"Trump Ancient Dragon",
	},

	/*** Arcane Spells (_only_ common spells) ***/

	{
		"Zap",
		"Wizard Lock",
		"Detect Invisibility",
		"Detect Monsters",
		"Blink",
		"Light Area",
		"Trap & Door Destruction",
		"Cure Light Wounds",

		"Detect Doors & Traps",
		"Phlogiston",
		"Detect Treasure",
		"Detect Enchantment",
		"Detect Objects",
		"Cure Poison",
		"Resist Cold",
		"Resist Fire",

		"Resist Lightning",
		"Resist Acid",
		"Cure Medium Wounds",
		"Teleport",
		"Identify",
		"Stone to Mud",
		"Ray of Light",
		"Satisfy Hunger",

		"See Invisible",
		"Conjure Elemental",
		"Teleport Level",
		"Teleport Away",
		"Elemental Ball",
		"Detection",
		"Word of Recall",
		"Clairvoyance",
	},

	/*** Craft Spells ***/

	{
		/* Common Craft Spellbooks */
		"Infravision",
		"Regeneration",
		"Satisfy Hunger",
		"Resist Cold",
		"Resist Fire",
		"Heroism",
		"Resist Lightning",
		"Resist Acid",

		"See Invisibility",
		"Remove Curse",
		"Resist Poison",
		"Berserk",
		"Self Knowledge",
		"Protection from Evil",
		"Cure",
		"Mana Branding",

		/* Rare Craft Spellbooks */
		"Telepathy",
		"Stone Skin",
		"Resistance",
		"Haste",
		"Walk through Wall",
		"Pulish Shield",
		"Create Golem",
		"Magical armor",

		"Remove Enchantment",
		"Remove All Curse",
		"Knowledge True", /* "Total Knowledge", */
		"Enchant Weapon",
		"Enchant Armor",
		"Brand Weapon",
		"Living Trump",
		"Immunity",
	},

	/*** Daemon Spells ***/

	{
		/* Common Daemon Spellbooks */
		"Magic Missile",
		"Detect Unlife",
		"Evil Bless",
		"Resist Fire",
		"Horrify",
		"Nether Bolt",
		"Summon Manes",
		"Hellish Flame",

		"Dominate Demon",
		"Vision",
		"Resist Nether",
		"Plasma bolt",
		"Fire Ball",
		"Fire Branding",
		"Nether Ball",
		"Summon Demon",

		/* Rare Daemon Spellbooks */
		"Devilish Eye",
		"Devil Cloak",
		"The Flow of Lava",
		"Plasma Ball",
		"Polymorph Demon",
		"Nather Wave",
		"Kiss of Saccubus",
		"Doom Hand",

		"Raise the Morale",
		"Immortal Body",
		"Insanity Circle",
		"Explode Pets",
		"Summon Greater Demon",
		"Nether Storm",
		"Bloody Curse",
		"Polymorph Demonlord",
	},

	/*** Music Spells ***/

	{
		/******* Apprentice Handbook *******/
		"Song of Holding",
		"Song of Blessing",
		"Wrecking Note",
		"Stun Pattern",
		"Flow of Life",
		"Song of the Sun",
		"Song of Fear", /* ݤβ */
		"Heroic Ballad",

		/******* Minstrel's Music *******/
		"Clairaudience",
		"Soul Shriek", /* β */
		"Song of Lore",
		"Hiding Tune", /* "ۤβ", */
		"Illusion Pattern",
		"Doomcall",
		"Firiel's Song",
		"Fellowship Chant", /* "ι" */

		/******* Harps of Rivendell *******/
		"Sound of disintegration", /* ʬ */
		"Finrod's Resistance", /*  */
		"Hobbit Melodies", /* ۥӥåȤΥǥ */
		"World Contortion", /* Ĥ */
		"Dispeling chant", /* ໶β */
		"The Voice of Saruman", /* ޥδŸ */
		"Song of the Tempest", /* β */
		"Ambarkanta", /* ⤦Ĥ */

		/******* Lays of Beleriand *******/
		"Wrecking Pattern", /* ˲Χ */
		"Stationary Shriek", /* ڤβ */
		"Endurance", /* β */
		"The Helo's Poem", /* ͺλ */
		"Relief of Yavanna", /* "ʤν" */
		"Goddess' rebirth", /* "β" */
		"Wizardry of Sauron", /* "", */
		"Fingolfin's Challenge", /* "ե󥴥եĩ", */
	},

	/*** Hissatsuwaza ***/
	{
		"Tobi-Izuna",
		"3-Way Attack",
		"Boomerang",
		"Burning Strike",
		"Detect Ferocity",
		"Strike to Stun",
		"Counter",
		"Harainuke",

		"Serpent's Tongue",
		"Zammaken",
		"Wind Blast",
		"Judge",
		"Rock Smash",
		"Midare-Setsugekka",
		"Spot Aiming",
		"Majingiri",

		"Desperate Attack",
		"Lightning Eagle",
		"Rush Attack",
		"Bloody Maelstrom",
		"Earthquake Blow",
		"Crack",
		"War Cry",
		"Musou-Sandan",

		"Vampire's Fang",
		"Moon Dazzling",
		"Hundred Slaughter",
		"Dragonic Flash",
		"Twin Slash",
		"Kofuku-Zettousei",
		"Keiun-Kininken",
		"Harakiri",
	},
};
#endif

/*
 * Each chest has a certain set of traps, determined by pval
 * Each chest has a "pval" from 1 to the chest level (max 55)
 * If the "pval" is negative then the trap has been disarmed
 * The "pval" of a chest determines the quality of its treasure
 * Note that disarming a trap on a chest also removes the lock.
 */
int chest_traps[64] =
{
	0,                                      /* 0 == empty */
	(CHEST_POISON),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),                       /* 5 == best small wooden */
	0,
	(CHEST_POISON),
	(CHEST_POISON),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),
	(CHEST_POISON),
	(CHEST_SCATTER),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_SUMMON),                 /* 15 == best large wooden */
	0,
	(CHEST_LOSE_STR),
	(CHEST_SCATTER),
	(CHEST_PARALYZE),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_SUMMON),
	(CHEST_PARALYZE),
	(CHEST_LOSE_STR),
	(CHEST_LOSE_CON),
	(CHEST_EXPLODE),                        /* 25 == best small iron */
	0,
	(CHEST_E_SUMMON),
	(CHEST_POISON | CHEST_LOSE_CON),
	(CHEST_LOSE_STR | CHEST_LOSE_CON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_BIRD_STORM),
	(CHEST_POISON | CHEST_SUMMON),
	(CHEST_E_SUMMON),
	(CHEST_EXPLODE),
	(CHEST_EXPLODE | CHEST_SUMMON), /* 35 == best large iron */
	0,
	(CHEST_SUMMON),
	(CHEST_EXPLODE),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_POISON | CHEST_PARALYZE),
	(CHEST_EXPLODE),
	(CHEST_BIRD_STORM),
	(CHEST_EXPLODE | CHEST_E_SUMMON),
	(CHEST_H_SUMMON),        /* 45 == best small steel */
	0,
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_BIRD_STORM),
	(CHEST_RUNES_OF_EVIL),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_BIRD_STORM),
	(CHEST_H_SUMMON),
	(CHEST_RUNES_OF_EVIL),
	(CHEST_H_SUMMON | CHEST_SCATTER),
	(CHEST_RUNES_OF_EVIL | CHEST_EXPLODE),  /* 55 == best large steel */
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
	(CHEST_EXPLODE | CHEST_SUMMON),
};


/*
 * Class titles for the player.
 *
 * The player gets a new title every five levels, so each class
 * needs only ten titles total.
 */
#ifdef JP
cptr player_title[MAX_CLASS][PY_MAX_LEVEL / 5] =
{
	/* Warrior */
	{
                "ʼ",
                "ʼ",
                "ʼ",
                "Żʼ",
                "",
                "Ʈ",
                "ͺ",
                "˼",
                "",
                "μ",
	},

	/* Mage */
	{
				/*"",*/
                "",                 /*ա */
                "ѻ",               /*աڥƥ */
                "ѻ",
                "ѻ",
                "",
                "",
                "ѻ",
                "ƻ",
                "ץޥ",
                "ƻ",
	},

	/* Priest */
	{
                "",                 /* */
                "",             /*żԡΡȼΡ */
                "",
                "",                 /* */
                "ƻ",               /*ʺɾİƻ */
                "һ",               /*޶ */
                "",               /*ʶ綵ʶ */
                "׻",                 /*׻ʡʺ */
                "׻Ĺ",               /*׻ʡ׻Ĺ */
                "",
	},

	/* Rogues */
	{
		/* "Ĥ",*/                     /*Ĥ˷ϲ */
                "",
                "ɤϤ",                     /*ɤť */
                "",                         /*ť */
                "ť",                       /*ťť */
                "ڥƥ",                     /*̡ڥƥա */
                "",
                "ϥ",
                "ޥ",
                "",						/* Ż */
                "ɥޥ",
	},

	/* Rangers */
	{
                "Ȥ",
                "",
                "͸",                         /*͸ĥꡢ廡ʼ */
                "",
                "׼",
                "Ƴ",
                "õ",                       /*ԡõ */
                "",
                "Ƭ",
                "",
	},

	/* Paladins */
	{
                "ͦ",                 /*ˡ͡ԡͦʿ */
                "һ",
                "ݸ",
                "ɱҼ",
                "Ҽ",
                "",
                "ŵ",
                "",
                "",
                "Ĺ",
	},

	/* Warrior-Mage */
	{
		"",
		"",			/*ա */
		"",
		"Żʼ",
		"ʼ",
		"Ʈ",
		"Ѥαͺ", /* Mage-Hero */
		"˼",
		/* "", */
		"Ʈѻ",
		"ɥ",
	},

	/* Chaos Warrior */
	{
                "ʼ",
                "ʼ",
                "ʼ",
                "Żʼ",
                "",
                "Ʈ",
                "٤αͺ",
                "٤˼",
                "٤θ",
                "٤β",
	},

	/* Monk */
	{
		"",
		"",
		"ľ",
		"",
		"",
		"ƻ",
		"̾",
		"̾",
		"",
		"",
	},

	/* Mindcrafter */
	{
		"",
		"",
		"",
		"ã",
		"ۻ",
		"ѻ",
		"å",
		"˥å",
		"Ķǽϼ",
		"ޥɥޥ",
	},

	/* High Mage; same as Mage */
	{
				/*"",*/
                "",                 /*ա */
                "ѻ",               /*աڥƥ */
                "ѻ",
                "ѻ",
                "",
                "",
                "ѻ",
                "ƻ",
                "ץޥ",
                "ƻ",
	},

	/* Tourist */
	{
		"סϺ",
		"Ѹ",
		"ͷιԼ",
		"",
		"ιԼ",
		"ϲ", /* "ι", */
		"ҳ",
		"õ",
		"",
		"ڥ󥫡",
	},

	/* Imitator */
	{
		"줫",
		"ޤ",
		"ޤޤ",
		"ܤܤ",
		"",
		"ޤޤ",
		"ʤʤ",
		"",
		"Ȥ",
		"餤",
	},

	/* Beastmaster */
	{
		"Ȥ",
		"إӻȤ",
		"Ȥ",
		"ϵȤ",
		"ȥȤ",
		"Ȥ",
		"ҥɥȤ",
		"ϥɻȤ",
		"ࡼޥȤ",
		"ɥ饴Ȥ",
	},

	/* Sorcerer; same as Mage */
	{
				/*"",*/
                "",                 /*ա */
                "ѻ",               /*աڥƥ */
                "ѻ",
                "ѻ",
                "",
                "",
                "ѻ",
                "ƻ",
                "ץޥ",
                "ƻ",
	},

	/* Archer */
	{
                "ʼ",
                "ʼ",
                "ʼ",
                "Żʼ",
                "",
                "Ʈ",
                "ͺ",
                "˼",
                "",
                "μ",
	},

	/* Magic eater */
	{
                "̵Τʤ",
                "",
                "ѻ",
                "ѻ",
                "ѻ",
                "",
                "ã",
                "ã",
                "ƻ",
                "ƤΤ",
	},

        /* Harper */
	{
                "",	/*"Apprentice"*/
                "ʲ",	/*"Songsmith"*/
                "ͷ",	/*"Bard"*/
                "ѥ˥",	/*"Companion"*/
                "",	/*"Minstrel"*/
                "è׻",	/*"Harper"*/
                "¤",	/*"Loreweaver"*/
                "ο",	/*"Muse"*/
                "̴¤",	/*"Dreamweaver"*/
                "ޥ",	/*"Master Harper"*/
	},

	/* Red Mage; same as Warrior-Mage */
	{
		"",
		"",			/*ա */
		"",
		"Żʼ",
		"ʼ",
		"Ʈ",
		"Ѥαͺ", /* Mage-Hero */
		"˼",
		/* "", */
		"Ʈѻ",
		"ɥ",
	},

	/* Samurai */
	{
		"",
		"",
		"ľ",
		"",
		"",
		"ƻ",
		"̾",
		"̾",
		"",
		"",
	},

	/* ForceTrainer; same as Monk(?) */
	{
		"",
		"",
		"ľ",
		"",
		"",
		"ƻ",
		"̾",
		"̾",
		"",
		"",
	},

	/* Blue Mage; same as Mage */
	{
				/*"",*/
                "",                 /*ա */
                "ѻ",               /*աڥƥ */
                "ѻ",
                "ѻ",
                "",
                "",
                "ѻ",
                "ƻ",
                "ץޥ",
                "ƻ",
	},

	/* Kihei */
	{
                "ʼ",
                "ʼ",
                "ʼ",
                "Żʼ",
                "",
                "Ʈ",
                "ͺ",
                "˼",
                "",
                "μ",
	},

	/* Berserker */
	{
		"С",
		"С",
		"С",
		"ܤθ",
		"ܤθ",
		"ܤθ",
		"ܤβ",
		"ܤβ",
		"ܤβ",
		"ܤβ",
	},

	/* Kaji */
	{
		"Ƽä",
		"Ŵä",
		"ݤä",
		"ä",
		"εä",
		"ä",
		"ä",
		"ä",
		"ä",
		"ä",
	},

	/* Mirror Master */
	{
		"򸫤",
		"᤭",
		"",
		"ѻ",
		"Ƴ",
		"θ",
		"β",
		"ι",
		"β",
		"եΡ벦",
	},
	/* Ninja */
	{
		"",
		"ż",
		"ã",
		"ûȤ",
		"ڤ",
		"",
		"ο",
		"Ǥΰ",
		"Ż",
		"Ĺ",
	},
};

#else
cptr player_title[MAX_CLASS][PY_MAX_LEVEL / 5] =
{
	/* Warrior */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},

	/* Mage */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Priest */
	{
		"Believer",
		"Acolyte",
		"Adept",
		"Curate",
		"Canon",
		"Priest",
		"High Priest",
		"Cardinal",
		"Inquisitor",
		"Pope",
	},

	/* Rogues */
	{
		"Cutpurse",
		"Robber",
		"Burglar",
		"Filcher",
		"Sharper",
		"Low Thief",
		"High Thief",
		"Master Thief",
		"Assassin",
		"Guildmaster",
	},

	/* Rangers */
	{
		"Runner",
		"Strider",
		"Scout",
		"Courser",
		"Tracker",
		"Guide",
		"Pathfinder",
		"Low Ranger",
		"High Ranger",
		"Ranger Lord",
	},

	/* Paladins */
	{
		"Gallant",
		"Keeper",
		"Protector",
		"Defender",
		"Warder",
		"Knight",
		"Guardian",
		"Low Paladin",
		"High Paladin",
		"Paladin Lord",
	},

	/* Warrior-Mage */
	{
		"Novice",
		"Apprentice",
		"Journeyman",
		"Veteran",
		"Enchanter",
		"Champion",
		"Mage-Hero",
		"Baron Mage",
		"Battlemage",
		"Wizard Lord",
	},

	/* Chaos Warrior */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Chaos Hero",
		"Chaos Baron",
		"Chaos Duke",
		"Chaos Lord",
	},

	/* Monk */
	{
		"Initiate",
		"Brother",
		"Disciple",
		"Immaculate",
		"Master",
		"Soft Master",
		"Hard Master",
		"Flower Master",
		"Dragon Master",
		"Grand Master",
	},

	/* Mindcrafter */
	{
		"Trainee",
		"Acolyte",
		"Adept",
		"Immaculate",
		"Contemplator",
		"Mentalist",
		"Psychic",
		"Psionicist",
		"Esper",
		"Mindmaster",
	},

	/* High Mage; same as Mage */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Tourist */
	{
		"Rambler",
		"Sightseer",
		"Excursionist",
		"Peregrinator",
		"Traveler",
		"Journeyer",
		"Voyager",
		"Explorer",
		"Adventurer",
		"Spelunker",
	},

	/* Imitator */
	{
		"Yet",
		"Lacks",
		"Still more",
		"So so",
		"All right",
		"Not bad",
		"Considerable",
		"Go go",
		"Sizable",
		"Great man",
	},

	/* BeastMaster */
	{
		"Newt Master",
		"Snake Master",
		"Spider Master",
		"Wolf Master",
		"Tiger Master",
		"Beetle Master",
		"Hydra Master",
		"Hound Master",
		"Mumak Master",
		"Dragon Master",
	},

	/* Sorcerer */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Archer */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},

	/* Magic eater */
	{
                "Apprentice",
                "Beginner",
                "Jagguler",
                "Skilled",
                "Conjurer",
                "Magician",
                "Master",
                "Master",
                "Wizard",
                "Almighty",
	},

        /* Harper */
	{
                "Apprentice",	/*"Apprentice"*/
                "Songsmith",	/*"Songsmith"*/
                "Bard",	/*"Bard"*/
                "Companion",	/*"Companion"*/
                "Minstrel",	/*"Minstrel"*/
                "Harper",	/*"Harper"*/
                "Loreweaver",	/*"Loreweaver"*/
                "Muse",	/*"Muse"*/
                "Dreamweaver",	/*"Dreamweaver"*/
                "Master Harper",	/*"Master Harper"*/
	},

	/* Red Mage */
	{
		"Novice",
		"Apprentice",
		"Journeyman",
		"Veteran",
		"Enchanter",
		"Champion",
		"Mage-Hero",
		"Baron Mage",
		"Battlemage",
		"Wizard Lord",
	},

	/* Samurai */
	{
		"Initiate",
		"Brother",
		"Disciple",
		"Immaculate",
		"Master",
		"Soft Master",
		"Hard Master",
		"Flower Master",
		"Dragon Master",
		"Grand Master",
	},

	/* ForceTrainer */
	{
		"Initiate",
		"Brother",
		"Disciple",
		"Immaculate",
		"Master",
		"Soft Master",
		"Hard Master",
		"Flower Master",
		"Dragon Master",
		"Grand Master",
	},

	/* Blue Mage */
	{
		"Apprentice",
		"Trickster",
		"Illusionist",
		"Spellbinder",
		"Evoker",
		"Conjurer",
		"Warlock",
		"Sorcerer",
		"Ipsissimus",
		"Archimage",
	},

	/* Warrior */
	{
		"Rookie",
		"Soldier",
		"Mercenary",
		"Veteran",
		"Swordsman",
		"Champion",
		"Hero",
		"Baron",
		"Duke",
		"Lord",
	},

	/* Berserker */
	{
		    "Berserker",
		    "Berserker",
		    "Berserker",
		    "Rage Prince",
		    "Rage Prince",
		    "Rage Prince",
		    "Rage King",
		    "Rage King",
		    "Rage King",
		    "God of Rage",
	},

	/* Kaji */
	{
                "Copper smith",
                "Iron smith",
                "Steel smith",
                "Silver smith",
                "Dragon smith",
                "Spirit smith",
                "Magic smith",
                "Soul smith",
                "God smith",
                "AlmightySmith",
	},

	/* Mirror Master */
	{
		"Mirrorstarer",
		"Mirrorpolisher",
		"Mirrormaker",
		"Mirrormagician",
		"Guru of Mirror ",
		"Mirror Mage",
		"Mirror King",
		"Mirror Emperror",
		"Avatar of Mirror",
		"Ruffnor King",
	},

	/* Ninja */
	{
		"Trainee",
		"Myrmidon",
		"Initiate",
		"Knifer",
		"Bladesman",
		"Hashishin",
		"Black Dagger",
		"Shadowstrike",
		"Assassinator",
		"Death Lord",
	},
};
#endif

monster_power monster_powers[MAX_MONSPELLS] =
{
/* level,  smana,  %fail,  manedam,  %manefail,  use_stat, name */
#ifdef JP
{  1,   1,  10,    0,  15, A_CON,  "" 		, "shriek"},
{ 10,   4,  35,   89,  40, A_INT,  ""		, "XXX1"},
{ 40,  35,  85,    0,  40, A_INT,  "Ͼõ"		, "dispel magic"},
{ 35,  30,  80,  800,  70, A_STR,  "å"		, "rocket"},
{  2,   1,  15,   10,  15, A_DEX,  ""			, "arrow_1"},
{  5,   2,  20,   18,  20, A_DEX,  "Ϥ"		, "arrow_2"},
{ 12,   3,  25,   30,  25, A_DEX,  "ܥ"		, "arrow_3"},
{ 16,   4,  30,   42,  30, A_DEX,  "Ϥʥܥ"	, "arrow_4"},
{ 20,  15,  55, 1600,  95, A_CON,  "Υ֥쥹"		, "breath acid"},
{ 20,  15,  55, 1600,  95, A_CON,  "ŷΥ֥쥹"	, "breath elec"},
{ 20,  15,  55, 1600,  95, A_CON,  "Υ֥쥹"		, "breath fire"},
{ 20,  15,  55, 1600,  95, A_CON,  "䵤Υ֥쥹"	, "breath cold"},
{ 20,  15,  55,  800,  95, A_CON,  "ǤΥ֥쥹"		, "breath poison"},
{ 20,  15,  70,  550,  95, A_CON,  "ϹΥ֥쥹"	, "breath nether"},
{ 20,  16,  70,  400,  95, A_CON,  "Υ֥쥹"	, "breath light"},
{ 20,  16,  70,  400,  95, A_CON,  "ŹΥ֥쥹"	, "breath dark"},
{ 20,  20,  70,  450,  95, A_CON,  "Υ֥쥹"	, "breath confusion"},
{ 20,  20,  70,  450,  95, A_CON,  "첻Υ֥쥹"	, "breath sound"},
{ 20,  20,  70,  600,  95, A_CON,  "Υ֥쥹"	, "breath chaos"},
{ 20,  16,  70,  500,  95, A_CON,  "Υ֥쥹"	, "breath disenchantment"},
{ 30,  25,  80,  250,  95, A_CON,  "̺Υ֥쥹"	, "breath nexus"},
{ 35,  18,  80,  150,  95, A_CON,  "ֵžΥ֥쥹"	, "breath time"},
{ 30,  25,  80,  200,  95, A_CON,  "ߤΥ֥쥹"	, "breath inertia"},
{ 30,  28,  90,  200,  95, A_CON,  "ϤΥ֥쥹"	, "breath gravity"},
{ 20,  15,  70,  500,  95, A_CON,  "ҤΥ֥쥹"	, "breath shards"},
{ 35,  15,  80,  150,  95, A_CON,  "ץ饺ޤΥ֥쥹"	, "breath plasma"},
{ 30,  18,  70,  200,  95, A_CON,  "եΥ֥쥹"	, "breath force"},
{ 30,  28,  80,  250,  95, A_CON,  "ϤΥ֥쥹"	, "breath mana"},
{ 25,  20,  95,  320,  80, A_INT,  "ǽ"		, "nuke ball"},
{ 25,  15,  70,  800,  95, A_CON,  "ѴʪΥ֥쥹"	, "breath nuke"},
{ 30,  32,  85,  400,  80, A_INT,  "륹"		, "raw Logrus"},
{ 35,  40,  95,  150,  95, A_CON,  "ʬΥ֥쥹"	, "breath disintegrate"},
{ 18,  13,  55,  630,  80, A_INT,  "åɡܡ"	, "acid ball"},
{ 14,  10,  45,  316,  60, A_INT,  "ܡ"	, "lightning ball"},
{ 20,  14,  60,  720,  80, A_INT,  "եܡ"	, "fire ball"},
{ 15,  11,  50,  320,  60, A_INT,  "ܡ"	, "frost ball"},
{  5,   3,  40,   48,  20, A_INT,  ""		, "stinking cloud"},
{ 25,  18,  70,  350,  80, A_INT,  "Ϲ"		, "nether ball"},
{ 30,  22,  75,  350,  80, A_INT,  "ܡ"	, "water ball"},
{ 44,  45,  85,  550,  95, A_INT,  "Ϥ"		, "mana storm"},
{ 40,  42,  90,  550,  95, A_INT,  "Ź"		, "darkness storm"},
{ 10,   5,  50,    0,  25, A_INT,  "ϵۼ"		, "drain mana"},
{ 25,  10,  60,    0,  30, A_INT,  ""		, "mind blast"},
{ 30,  14,  65,    0,  30, A_INT,  "Ǿ"		, "brain smash"},
{  3,   1,  25,   24,  20, A_INT,  "ڽ"		, "cause light wounds"},
{ 12,   2,  35,   64,  25, A_INT,  "Ž"		, "cause serious wounds"},
{ 22,   6,  50,  150,  30, A_INT,  "̿"		, "cause critical wounds"},
{ 32,  10,  70,  225,  35, A_INT,  "빦ͤ"		, "cause mortal wounds"},
{ 13,   7,  40,  178,  40, A_INT,  "åɡܥ"	, "acid bolt"},
{ 10,   5,  35,  130,  35, A_INT,  "ܥ"	, "lightning bolt"},
{ 15,   9,  50,  210,  45, A_INT,  "եܥ"	, "fire bolt"},
{ 12,   6,  35,  162,  40, A_INT,  "ܥ"	, "frost bolt"},
{ 40,  42,  90,  550,  95, A_INT,  "С"	, "starburst"},
{ 25,  17,  60,  255,  60, A_INT,  "Ϲ"		, "nether bolt"},
{ 25,  20,  65,  250,  60, A_INT,  "ܥ"	, "water bolt"},
{ 25,  24,  90,  400,  80, A_INT,  "Ϥ"		, "mana bolt"},
{ 25,  20,  80,  216,  60, A_INT,  "ץ饺ޡܥ"	, "plasma bolt"},
{ 25,  16,  60,  186,  60, A_INT,  "˴"		, "ice bolt"},
{  3,   1,  25,   12,  20, A_INT,  "ޥåߥ"	, "magic missile"},
{  5,   3,  35,    0,  20, A_INT,  ""		, "scare"},
{ 10,   5,  40,    0,  20, A_INT,  ""		, "blind"},
{ 10,   5,  40,    0,  20, A_INT,  "ѥ˥å󥹥"	, "confuse"},
{ 10,   5,  40,    0,  20, A_INT,  "󥹥"	, "slow"},
{ 10,   5,  40,    0,  20, A_INT,  "꡼ס󥹥"	, "sleep"},
{ 20,  10,  70,    0,  40, A_INT,  "ԡ"		, "speed"},
{ 45, 120,  95,    0,  60, A_INT,  "Ǥμ"		, "hand of doom"},
{ 20,  15,  70,    0,  20, A_WIS,  "ϲ"		, "heal"},
{ 45,  65,  80,    0,  60, A_INT,  "̵ε"		, "invulnerability"},
{  5,   1,  30,    0,  20, A_INT,  "硼ȡƥݡ"	, "blink"},
{ 15,   8,  40,    0,  30, A_INT,  "ƥݡ"		, "teleport"},
{ 40, 999,  99,    0,  80, A_INT,  ""		, "The world"},
{  1,   0,   0,    0,  15, A_INT,  ""			, "something"},
{ 15,   8,  50,    0,  30, A_INT,  "褻"		, "teleport to"},
{ 20,  13,  80,    0,  30, A_INT,  "ƥݡȡ"	, "teleport away"},
{ 30,  40,  95,    0,  40, A_INT,  "ƥݡȡ٥"	, "teleport level"},
{ 35,  30,  80,  350,  70, A_INT,  "η"		, "psycho-spear"},
{  5,   1,  20,    0,  15, A_INT,  "Ű"		, "create darkness"},
{  5,   1,  20,    0,  15, A_DEX,  "ȥå¤"	, "create trap"},
{ 15,   3,  40,    0,  30, A_INT,  "Ӽ"		, "forget"},
{ 30,  30,  70,    0,  40, A_INT,  ""		, "raise dead"},
{ 40,  70,  85,    0,  45, A_INT,  "緳Ƥ"		, "summon kin"},
{ 45,  90,  90,    0,  50, A_INT,  "Сǡξ"	, "summon cyber"},
{ 25,  20,  65,    0,  30, A_INT,  "󥹥ξ"	, "summon monster"},
{ 35,  30,  75,    0,  40, A_INT,  "ʣΥ󥹥ξ"	, "summon monsters"},
{ 25,  25,  65,    0,  25, A_INT,  "ξ"		, "summon ant"},
{ 25,  20,  60,    0,  25, A_INT,  "ξ"		, "summon spider"},
{ 35,  26,  75,    0,  40, A_INT,  "ϥɤξ"	, "summon hound"},
{ 30,  23,  70,    0,  35, A_INT,  "ҥɥξ"	, "summon hydra"},
{ 40,  50,  85,    0,  40, A_INT,  "ŷȤξ"		, "summon angel"},
{ 35,  50,  80,    0,  35, A_INT,  "ǡξ"	, "summon deamon"},
{ 30,  30,  75,    0,  35, A_INT,  "ǥåɤξ"	, "summon undead"},
{ 39,  70,  80,    0,  40, A_INT,  "ɥ饴ξ"	, "summon dragon"},
{ 43,  85,  85,    0,  45, A_INT,  "饢ǥåɤξ"	, "summon greater undead"},
{ 46,  90,  85,    0,  45, A_INT,  "ɥ饴ξ"	, "summon ancient dragon"},
{ 48, 120,  90,    0,  50, A_INT,  "Сβξ"	, "summon amberites"},
{ 50, 150,  95,    0,  50, A_INT,  "ˡ󥹥ξ"	, "summon unique"},
#else
{  1,   1,  10,    0,  15, A_CON,  "shriek"},
{ 10,   4,  35,   89,  40, A_INT,  "XXX1"},
{ 40,  35,  85,    0,  40, A_INT,  "dispel magic"},
{ 35,  30,  80,  800,  70, A_STR,  "rocket"},
{  2,   1,  15,   10,  15, A_DEX,  "arrow_1"},
{  5,   2,  20,   18,  20, A_DEX,  "arrow_2"},
{ 12,   3,  25,   30,  25, A_DEX,  "arrow_3"},
{ 16,   4,  30,   42,  30, A_DEX,  "arrow_4"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath acid"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath elec"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath fire"},
{ 20,  15,  55, 1600,  95, A_CON,  "breath cold"},
{ 20,  15,  55,  800,  95, A_CON,  "breath poison"},
{ 20,  15,  70,  550,  95, A_CON,  "breath nether"},
{ 20,  16,  70,  400,  95, A_CON,  "breath light"},
{ 20,  16,  70,  400,  95, A_CON,  "breath dark"},
{ 20,  20,  70,  450,  95, A_CON,  "breath confusion"},
{ 20,  20,  70,  450,  95, A_CON,  "breath sound"},
{ 20,  20,  70,  600,  95, A_CON,  "breath chaos"},
{ 20,  16,  70,  500,  95, A_CON,  "breath disenchantment"},
{ 30,  25,  80,  250,  95, A_CON,  "breath nexus"},
{ 35,  18,  80,  150,  95, A_CON,  "breath time"},
{ 30,  25,  80,  200,  95, A_CON,  "breath inertia"},
{ 30,  28,  90,  200,  95, A_CON,  "breath gravity"},
{ 20,  15,  70,  500,  95, A_CON,  "breath shards"},
{ 35,  15,  80,  150,  95, A_CON,  "breath plasma"},
{ 30,  18,  70,  200,  95, A_CON,  "breath force"},
{ 30,  28,  80,  250,  95, A_CON,  "breath mana"},
{ 25,  20,  95,  320,  80, A_INT,  "nuke ball"},
{ 25,  15,  70,  800,  95, A_CON,  "breath nuke"},
{ 30,  32,  85,  400,  80, A_INT,  "raw Logrus"},
{ 35,  40,  95,  150,  95, A_CON,  "breath disintegrate"},
{ 18,  13,  55,  630,  80, A_INT,  "acid ball"},
{ 14,  10,  45,  316,  60, A_INT,  "lightning ball"},
{ 20,  14,  60,  720,  80, A_INT,  "fire ball"},
{ 15,  11,  50,  320,  60, A_INT,  "frost ball"},
{  5,   3,  40,   48,  20, A_INT,  "stinking cloud"},
{ 25,  18,  70,  350,  80, A_INT,  "nether ball"},
{ 30,  22,  75,  350,  80, A_INT,  "water ball"},
{ 44,  45,  85,  550,  95, A_INT,  "mana storm"},
{ 40,  42,  90,  550,  95, A_INT,  "darkness storm"},
{ 10,   5,  50,    0,  25, A_INT,  "drain mana"},
{ 25,  10,  60,    0,  30, A_INT,  "mind blast"},
{ 30,  14,  65,    0,  30, A_INT,  "brain smash"},
{  3,   1,  25,   24,  20, A_INT,  "cause light wounds"},
{ 12,   2,  35,   64,  25, A_INT,  "cause serious wounds"},
{ 22,   6,  50,  150,  30, A_INT,  "cause critical wounds"},
{ 32,  10,  70,  225,  35, A_INT,  "cause mortal wounds"},
{ 13,   7,  40,  178,  40, A_INT,  "acid bolt"},
{ 10,   5,  35,  130,  35, A_INT,  "lightning bolt"},
{ 15,   9,  50,  210,  45, A_INT,  "fire bolt"},
{ 12,   6,  35,  162,  40, A_INT,  "frost bolt"},
{ 40,  42,  90,  550,  95, A_INT,  "starburst"},
{ 25,  17,  60,  255,  60, A_INT,  "nether bolt"},
{ 25,  20,  65,  250,  60, A_INT,  "water bolt"},
{ 25,  24,  90,  400,  80, A_INT,  "mana bolt"},
{ 25,  20,  80,  216,  60, A_INT,  "plasma bolt"},
{ 25,  16,  60,  186,  60, A_INT,  "ice bolt"},
{  3,   1,  25,   12,  20, A_INT,  "magic missile"},
{  5,   3,  35,    0,  20, A_INT,  "scare"},
{ 10,   5,  40,    0,  20, A_INT,  "blind"},
{ 10,   5,  40,    0,  20, A_INT,  "confuse"},
{ 10,   5,  40,    0,  20, A_INT,  "slow"},
{ 10,   5,  40,    0,  20, A_INT,  "sleep"},
{ 20,  10,  70,    0,  40, A_INT,  "speed"},
{ 45, 120,  95,    0,  60, A_INT,  "hand of doom"},
{ 20,  15,  70,    0,  20, A_WIS,  "heal"},
{ 45,  65,  80,    0,  60, A_INT,  "invulnerability"},
{  5,   1,  30,    0,  20, A_INT,  "blink"},
{ 15,   8,  40,    0,  30, A_INT,  "teleport"},
{ 40, 999,  99,    0,  80, A_INT,  "The world"},
{  1,   0,   0,    0,  15, A_INT,  "something"},
{ 15,   8,  50,    0,  30, A_INT,  "teleport to"},
{ 20,  13,  80,    0,  30, A_INT,  "teleport away"},
{ 30,  40,  95,    0,  40, A_INT,  "teleport level"},
{ 35,  30,  80,  350,  70, A_INT,  "psycho-spear"},
{  5,   1,  20,    0,  15, A_INT,  "create darkness"},
{  5,   1,  20,    0,  15, A_DEX,  "create trap"},
{ 15,   3,  40,    0,  30, A_INT,  "forget"},
{ 30,  30,  70,    0,  40, A_INT,  "raise dead"},
{ 40,  70,  85,    0,  45, A_INT,  "summon kin"},
{ 45,  90,  90,    0,  50, A_INT,  "summon cyber"},
{ 25,  20,  65,    0,  30, A_INT,  "summon monster"},
{ 35,  30,  75,    0,  40, A_INT,  "summon monsters"},
{ 25,  25,  65,    0,  25, A_INT,  "summon ant"},
{ 25,  20,  60,    0,  25, A_INT,  "summon spider"},
{ 35,  26,  75,    0,  40, A_INT,  "summon hound"},
{ 30,  23,  70,    0,  35, A_INT,  "summon hydra"},
{ 40,  50,  85,    0,  40, A_INT,  "summon angel"},
{ 35,  50,  80,    0,  35, A_INT,  "summon deamon"},
{ 30,  30,  75,    0,  35, A_INT,  "summon undead"},
{ 39,  70,  80,    0,  40, A_INT,  "summon dragon"},
{ 43,  85,  85,    0,  45, A_INT,  "summon greater undead"},
{ 46,  90,  85,    0,  45, A_INT,  "summon ancient dragon"},
{ 48, 120,  90,    0,  50, A_INT,  "summon amberites"},
{ 50, 150,  95,    0,  50, A_INT,  "summon unique"},
#endif

};



/*
 * Hack -- the "basic" color names (see "TERM_xxx")
 */
cptr color_names[16] =
{
#ifdef JP
        "",
        "",
        "ĳ",
        "",
        "",
        "",
        "",
        "ῧ",
        "",
        "ĳ",
        "",
        "",
        "뤤",
        "뤤",
        "뤤",
        "ῧ",
#else
	"Dark",
	"White",
	"Slate",
	"Orange",
	"Red",
	"Green",
	"Blue",
	"Umber",
	"Light Dark",
	"Light Slate",
	"Violet",
	"Yellow",
	"Light Red",
	"Light Green",
	"Light Blue",
	"Light Umber",
#endif

};


/*
 * Abbreviations of healthy stats
 */
cptr stat_names[6] =
{
#ifdef JP
        " :", "ǽ :", " :", " :", "ѵ :", "̥ :"
#else
	"STR: ", "INT: ", "WIS: ", "DEX: ", "CON: ", "CHR: "
#endif

};

/*
 * Abbreviations of damaged stats
 */
cptr stat_names_reduced[6] =
{
#ifdef JP
        "x:", "ǽx:", "x:", "x:", "ѵx:", "̥x:"
#else
	"Str: ", "Int: ", "Wis: ", "Dex: ", "Con: ", "Chr: "
#endif

};


/*
 * Certain "screens" always use the main screen, including News, Birth,
 * Dungeon, Tomb-stone, High-scores, Macros, Colors, Visuals, Options.
 *
 * Later, special flags may allow sub-windows to "steal" stuff from the
 * main window, including File dump (help), File dump (artifacts, uniques),
 * Character screen, Small scale map, Previous Messages, Store screen, etc.
 *
 * The "ctrl-i" (tab) command flips the "Display inven/equip" and "Display
 * equip/inven" flags for all windows.
 *
 * The "ctrl-g" command (or pseudo-command) should perhaps grab a snapshot
 * of the main screen into any interested windows.
 */
cptr window_flag_desc[32] =
{
#ifdef JP
        "ʪ/",
#else
	"Display inven/equip",
#endif

#ifdef JP
        "/ʪ",
#else
	"Display equip/inven",
#endif

#ifdef JP
        "ʸ",
#else
	"Display spell list",
#endif

#ifdef JP
        "饯",
#else
	"Display character",
#endif

	NULL,
	NULL,
#ifdef JP
        "å",
#else
	"Display messages",
#endif

#ifdef JP
        "󥸥ο",
#else
	"Display overhead view",
#endif

#ifdef JP
        "󥹥λפ",
#else
	"Display monster recall",
#endif

#ifdef JP
        "ƥξܺ",
#else
	"Display object recall",
#endif

#ifdef JP
        "ʬμϤɽ", 
#else
	"Display dungeon view",
#endif

#ifdef JP
        "ǰ",
#else
	"Display snap-shot",
#endif

	NULL,
	NULL,
#ifdef JP
        "ܡå",
#else
	"Display borg messages",
#endif

#ifdef JP
        "ܡơ",
#else
	"Display borg status",
#endif

	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};


/*
 * Available Options
 *
 * Option Screen Sets:
 *
 *      Set 1: User Interface
 *      Set 2: Disturbance
 *      Set 3: Inventory
 *      Set 4: Game Play
 *
 * Note that bits 28-31 of set 0 are currently unused.
 */
option_type option_info[] =
{
	/*** Input Options ***/

#ifdef JP
	{ &rogue_like_commands,         FALSE, 1, 0, 0,
        "rogue_like_commands",          "֤Ѥ" },
#else
	{ &rogue_like_commands,         FALSE, 1, 0, 0,
	"rogue_like_commands",          "Rogue-like commands" },
#endif


#ifdef JP
	{ &quick_messages,              TRUE,  1, 0, 1,
     "quick_messages",  "ååѤ" },
#else
	{ &quick_messages,              TRUE,  1, 0, 1,
	"quick_messages",               "Activate quick messages" },
#endif


#ifdef JP
	{ &auto_more,                   FALSE, 1, 2, 6,
        "auto_more",                    "ԤʤϢ³ǥåɽ" },
#else
	{ &auto_more,                   FALSE, 1, 2, 6,
	"auto_more",                    "Automatically clear '-more-' prompts" },
#endif


#ifdef JP
	{ &command_menu,                TRUE,  1, 2, 7,
        "command_menu",                 "˥塼ˤꥳޥͭˤ" },
#else
	{ &command_menu,                TRUE,  1, 2, 7,
	"command_menu",                 "Enable command selection menu" },
#endif


#ifdef JP
	{ &other_query_flag,            FALSE, 1, 0, 2,
	"other_query_flag",             "ΥƥѤȤ˳ǧ" },
#else
	{ &other_query_flag,            FALSE, 1, 0, 2,
	"other_query_flag",             "Prompt for floor item selection" },
#endif


#ifdef JP
	{ &carry_query_flag,            FALSE, 1, 0, 3,
	"carry_query_flag",  "ƥ򽦤˳ǧ" },
#else
	{ &carry_query_flag,            FALSE, 1, 0, 3,
	"carry_query_flag",             "Prompt before picking things up" },
#endif


#ifdef JP
	{ &use_old_target,              FALSE, 1, 0, 4,
     "use_old_target",  "˰ΥåȤꤹ" },
#else
	{ &use_old_target,              FALSE, 1, 0, 4,
	"use_old_target",               "Use old target by default" },
#endif


#ifdef JP
	{ &always_pickup,               FALSE, 1, 0, 5,
        "always_pickup",                "˥ƥ򽦤" },
#else
	{ &always_pickup,               FALSE,  1, 0, 5,
	"always_pickup",                "Pick things up by default" },
#endif


#ifdef JP
	{ &always_repeat,               TRUE,  1, 0, 6,
        "always_repeat",                "ޥɼư֤" },
#else
	{ &always_repeat,               TRUE,  1, 0, 6,
	"always_repeat",                "Repeat obvious commands" },
#endif


#ifdef JP
	{ &confirm_destroy,             FALSE, 1, 5, 3,
	"confirm_destroy",              "̵͡פʥƥ˲ǧ" },
#else
	{ &confirm_destroy,             FALSE, 1, 5, 3,
	"confirm_destroy",              "Prompt for destruction of known worthless items" },
#endif


#ifdef JP
	{ &wear_confirm,                TRUE, 1, 5, 4,
	"confirm_wear",                 "줿ʪǧ" },
#else
	{ &wear_confirm,                TRUE, 1, 5, 4,
	"confirm_wear",                 "Confirm to wear/wield known cursed items" },
#endif


#ifdef JP
	{ &confirm_stairs,              FALSE, 1, 5, 5,
	"confirm_stairs",               "̤γ˹Ԥǧ" },
#else
	{ &confirm_stairs,              FALSE, 1, 5, 5,
	"confirm_stairs",               "Prompt before exiting a dungeon level" },
#endif


#ifdef JP
	{ &target_pet,                  TRUE,  1, 2, 5,
        "target_pet",                   "ڥåȤ򥿡åȤˤ" },
#else
	{ &target_pet,                  TRUE,  1, 2, 5,
	"target_pet",                   "Allow targetting pets" },
#endif


	/*** Output Options ***/

#ifdef JP
	{ &depth_in_feet,               FALSE, 2, 0, 7,
        "depth_in_feet",                "󥸥οեȤɽ" },
#else
	{ &depth_in_feet,               FALSE, 2, 0, 7,
	"depth_in_feet",                "Show dungeon level in feet" },
#endif


#ifdef JP
	{ &show_labels,                 TRUE,  2, 0, 10,
        "show_labels",                  "ɽ" },
#else
	{ &show_labels,                 TRUE,  2, 0, 10,
	"show_labels",                  "Show labels in object listings" },
#endif


#ifdef JP
	{ &show_weights,                TRUE,  2, 0, 11,
        "show_weights",                 "ƥǽ̤ɽ" },
#else
	{ &show_weights,                TRUE,  2, 0, 11,
	"show_weights",                 "Show weights in object listings" },
#endif


#ifdef JP
	{ &show_item_graph,             TRUE,  2, 2, 0,
	"show_item_graph",              "ƥΥܥɽ" },
#else
	{ &show_item_graph,             TRUE,  2, 2, 0,
	"show_item_graph",              "Show items graphics" },
#endif


#ifdef JP
	{ &show_choices,                TRUE,  2, 0, 12,
        "show_choices",                 "ʪ/ɥ򥢥ƥɽ" },
#else
	{ &show_choices,                TRUE,  2, 0, 12,
	"show_choices",                 "Show choices in certain sub-windows" },
#endif


#ifdef JP
	{ &plain_pickup,                FALSE, 2, 6, 6,
        "plain_pickup",                 "ֽäץåά" },
#else
	{ &plain_pickup,                FALSE,99, 6, 6,
	"plain_pickup",                 "Plain pickup messages(japanese only)" },
#endif


#ifdef JP
	{ &player_symbols,              FALSE, 2, 1, 13,
	"player_symbols",               "ץ쥤䡼̤ʵɽ"},
#else
	{ &player_symbols,              FALSE, 2, 1, 13,
	"player_symbols",               "Use special symbols for the player char"},
#endif


#ifdef JP
	{ &equippy_chars,               TRUE,  2, 1, 12,
	"equippy_chars",                "ơʸɽ" },
#else
	{ &equippy_chars,               TRUE,  2, 1, 12,
	"equippy_chars",                "Display 'equippy' chars" },
#endif


#ifdef JP
	{ &skip_mutations,              FALSE, 2, 5, 0,
	"skip_mutations",               "'C'ޥɤѰۤɽ" },
#else
	{ &skip_mutations,              FALSE, 2, 5, 0,
	"skip_mutations",               "Display mutations in 'C'haracter Display" },
#endif


#ifdef JP
	{ &plain_descriptions,          TRUE, 2, 5, 1,
	"plain_descriptions",           "ƥεҤάˤ" },
#else
	{ &plain_descriptions,          TRUE, 2, 5, 1,
	"plain_descriptions",           "Plain object descriptions" },
#endif


#ifdef JP
        { &center_player,               FALSE, 2, 5, 11,
        "center_player",                "˥ץ쥤䡼濴֤(*٤*)" },
#else
	{ &center_player,               FALSE, 2, 5, 11,
	"center_player",                "Center map while walking (*slow*)" },
#endif


#ifdef JP
        { &center_running,              TRUE,  2, 5, 12,
        "center_running",               "äƤǤ濴֤" },
#else
	{ &center_running,              TRUE,  2, 5, 12,
	"center_running",               "Centering even while running" },
#endif


#ifdef JP
	{ &view_yellow_lite,            TRUE,  2, 1, 28,
        "view_yellow_lite",              "ϰϤ̤ʿɽ" },
#else
	{ &view_yellow_lite,            TRUE,  2, 1, 28,
	"view_yellow_lite",             "Use special colors for torch-lit grids" },
#endif


#ifdef JP
	{ &view_bright_lite,            TRUE,  2, 1, 29,
        "view_bright_lite",             "볦ϰϤ̤ʿɽ" },
#else
	{ &view_bright_lite,            TRUE,  2, 1, 29,
	"view_bright_lite",             "Use special colors for 'viewable' grids" },
#endif


#ifdef JP
	{ &view_granite_lite,           TRUE,  2, 1, 30,
        "view_granite_lite",            "ɤ̤ʿɽ(Ť)" },
#else
	{ &view_granite_lite,           TRUE,  2, 1, 30,
	"view_granite_lite",            "Use special colors for wall grids (slow)" },
#endif


#ifdef JP
	{ &view_special_lite,           TRUE,  2, 1, 31,
        "view_special_lite",            "̤ʿɽ(Ť)" },
#else
	{ &view_special_lite,           TRUE,  2, 1, 31,
	"view_special_lite",            "Use special colors for floor grids (slow)" },
#endif

#ifdef JP
	{ &new_ascii_graphics,          FALSE, 2, 2, 4,
        "new_ascii_graphics",           "ϷŤϤäɽ" },
#else
	{ &new_ascii_graphics,          FALSE, 2, 2, 4,
	"new_ascii_graphics",           "Show a clear contrast between light and dark" },
#endif

#ifdef JP
	{ &display_path,                FALSE, 2, 2, 8,
        "display_path",                 "ˡεפɽ" },
#else
	{ &display_path,                FALSE, 2, 2, 8,
	"display_path",                 "Display actual path before shooting" },
#endif


#ifdef JP
	{ &always_show_list,            TRUE,  2, 4, 0,
        "always_show_list",             "ˤϾ˰ɽ" },
#else
	{ &always_show_list,            TRUE,  2, 4, 0,
        "always_show_list",             "Always show list at first when select items" },
#endif


#ifdef JP
	{ &old_way_of_kaz,              TRUE,  2, 4, 5,
        "old_way_of_kaz",               "ƥμˤäƿѤ" },
#else
	{ &old_way_of_kaz,              TRUE,  99, 4, 5,
        "old_way_of_kaz",               "Change a way to counting (japanese only)" },
#endif


	/*** Game-Play ***/

#ifdef JP
	{ &manual_haggle,               FALSE, 3, 1, 0,
        "manual_haggle",                "ŹڤĤ򤹤" },
#else
	{ &manual_haggle,               FALSE, 3, 1, 0,
	"manual_haggle",                "Manualy haggle in stores" },
#endif


#ifdef JP
	{ &auto_scum,                   FALSE, 3, 1, 1,
        "auto_scum",                    "ʷϵư깥" },
#else
	{ &auto_scum,                   FALSE, 3, 1, 1,
	"auto_scum",                    "Auto-scum for good levels" },
#endif


#ifdef JP
	{ &stack_force_notes,           TRUE,  3, 0, 8,
        "stack_force_notes",            "ۤʤäΥƥޤȤ" },
#else
	{ &stack_force_notes,           TRUE,  3, 0, 8,
	"stack_force_notes",            "Merge inscriptions when stacking" },
#endif


#ifdef JP
	{ &stack_force_costs,           FALSE, 3, 0, 9,
        "stack_force_costs",            "ۤʤɽΥƥޤȤ" },
#else
	{ &stack_force_costs,           FALSE, 3, 0, 9,
	"stack_force_costs",            "Merge discounts when stacking" },
#endif


#ifdef JP
	{ &stack_allow_items,           TRUE,  3, 1, 2,
	"stack_allow_items",            "ƱɶޤȤ" },
#else
	{ &stack_allow_items,           TRUE,  3, 1, 2,
	"stack_allow_items",            "Allow weapons and armor to stack" },
#endif


#ifdef JP
	{ &stack_allow_wands,           TRUE,  3, 1, 3,
	"stack_allow_wands",            "ƱΥå//ˡޤȤ" },
#else
	{ &stack_allow_wands,           TRUE,  3, 1, 3,
	"stack_allow_wands",            "Allow wands/staffs/rods to stack" },
#endif


#ifdef JP
	{ &expand_look,                 TRUE,  3, 1, 4,
        "expand_look",                  "ֲĴ٤ץޥɤĥ" },
#else
	{ &expand_look,                 TRUE,  3, 1, 4,
	"expand_look",                  "Expand the power of the look command" },
#endif


#ifdef JP
	{ &expand_list,                 TRUE,  3, 1, 5,
        "expand_list",                  "ְץޥɤĥ" },
#else
	{ &expand_list,                 TRUE,  3, 1, 5,
	"expand_list",                  "Expand the power of the list commands" },
#endif


#ifdef JP
	{ &view_perma_grids,            TRUE,  3, 1, 6,
        "view_perma_grids",             "뤤ϤΤޤޤˤ" },
#else
	{ &view_perma_grids,            TRUE,  3, 1, 6,
	"view_perma_grids",             "Map remembers all perma-lit grids" },
#endif


#ifdef JP
	{ &view_torch_grids,            FALSE, 3, 1, 7,
        "view_torch_grids",             "ǾȤ餷ϤΤޤޤˤ" },
#else
	{ &view_torch_grids,            FALSE, 3, 1, 7,
	"view_torch_grids",             "Map remembers all torch-lit grids" },
#endif


#ifdef JP
	{ &dungeon_stair,               TRUE,  3, 1, 9,
        "dungeon_stair",                "ʤĤʤƥ󥸥" },
#else
	{ &dungeon_stair,               TRUE,  3, 1, 9,
	"dungeon_stair",                "Generate dungeons with connected stairs" },
#endif


#ifdef JP
	{ &small_levels,                TRUE,  3, 0, 30,
	"small_levels",                 "˾եǽˤ" },
#else
	{ &small_levels,                TRUE,  3, 0, 30,
	"small_levels",                 "Allow unusually small dungeon levels" },
#endif


#ifdef JP
	{ &always_small_levels,         FALSE, 3, 2, 3,
	"always_small_levels",          "˾ե" },
#else
	{ &always_small_levels,         FALSE, 3, 2, 3,
	"always_small_levels",          "Always create unusually small dungeon levels" },
#endif


#ifdef JP
	{ &empty_levels,                TRUE,  3, 0, 31,
        "empty_levels",                 "äݤΡ֥꡼ʡץ٥ǽˤ" },
#else
	{ &empty_levels,                TRUE,  3, 0, 31,
	"empty_levels",                 "Allow empty 'arena' levels" },
#endif


#ifdef ALLOW_EASY_OPEN
#ifdef JP
	{ &easy_open,                   TRUE,  3, 5, 7,
	"easy_open",                    "ưŪ˥ɥ򳫤" },
#else
	{ &easy_open,                   TRUE,  3, 5, 7,
	"easy_open",                    "Automatically open doors" },
#endif
#endif /* ALLOW_EASY_OPEN */


#ifdef ALLOW_EASY_DISARM
#ifdef JP
	{ &easy_disarm,                 TRUE,  3, 5, 8,
	"easy_disarm",                  "ưŪ櫤" },
#else
	{ &easy_disarm,                 TRUE,  3, 5, 8,
	"easy_disarm",                  "Automatically disarm traps" },
#endif
#endif /* ALLOW_EASY_DISARM */


#ifdef ALLOW_EASY_FLOOR /* TNB */
#ifdef JP
	{ &easy_floor,                  FALSE, 3, 5, 9,
	"easy_floor",                   "ǽŤʤäƥꥹȤ" },
#else
	{ &easy_floor,                  FALSE, 3, 5, 9,
	"easy_floor",                   "Display floor stacks in a list" },
#endif
#endif /* ALLOW_EASY_FLOOR -- TNB */


#ifdef JP
	{ &use_command,                 FALSE, 3, 5, 10,
	"use_command",                  "ֻȤ(a)ץޥɤǥƥ򲿤ǤȤ" },
#else
	{ &use_command,                 FALSE, 3, 5, 10,
	"use_command",                  "Allow unified use command" },
#endif


#ifdef JP
	{ &last_words,                  FALSE, 3, 0, 28,
	"last_words",                   "饯Τ" },
#else
	{ &last_words,                  FALSE, 3, 0, 28,
	"last_words",                   "Leave last words when your character dies" },
#endif


#ifdef JP
	{ &over_exert,                  TRUE,  3, 0, 29,
	"over_exert",                   "MP­ʤƤˡĩ魯" },
#else
	{ &over_exert,                  TRUE,  3, 0, 29,
	"over_exert",                   "Allow casting spells when short of mana" },
#endif


#ifdef WORLD_SCORE
#ifdef JP
	{ &send_score,                  TRUE,  3, 4, 6,
        "send_score",                   "Ф˥" },
#else
	{ &send_score,                  TRUE,  3, 4, 6,
        "send_score",                   "Send score dump to the world score server" },
#endif
#endif


	/*** Disturbance ***/

#ifdef JP
	{ &find_ignore_stairs,          FALSE, 4, 0, 16,
	"find_ignore_stairs",           "ʤ̲᤹" },
#else
	{ &find_ignore_stairs,          FALSE, 4, 0, 16,
	"find_ignore_stairs",           "Run past stairs" },
#endif


#ifdef JP
	{ &find_ignore_doors,           TRUE,  4, 0, 17,
	"find_ignore_doors",            "ɥ̲᤹" },
#else
	{ &find_ignore_doors,           TRUE,  4, 0, 17,
	"find_ignore_doors",            "Run through open doors" },
#endif


#ifdef JP
	{ &find_cut,                    FALSE, 4, 0, 18,
	"find_cut",                     "ΤζʤѤ̲᤹" },
#else
	{ &find_cut,                    FALSE, 4, 0, 18,
	"find_cut",                     "Run past known corners" },
#endif


#ifdef JP
	{ &find_examine,                TRUE,  4, 0, 19,
	"find_examine",                 "ʤѤ⤷ʤ̲᤹" },
#else
	{ &find_examine,                TRUE,  4, 0, 19,
	"find_examine",                 "Run into potential corners" },
#endif


#ifdef JP
	{ &disturb_move,                TRUE,  4, 0, 20,
        "disturb_move",                 "ɤΥ󥹥ưƤưߤ" },
#else
	{ &disturb_move,                TRUE,  4, 0, 20,
	"disturb_move",                 "Disturb whenever any monster moves" },
#endif


#ifdef JP
	{ &disturb_near,                TRUE,  4, 0, 21,
        "disturb_near",                 "볦Υ󥹥ưưߤ" },
#else
	{ &disturb_near,                TRUE,  4, 0, 21,
	"disturb_near",                 "Disturb whenever viewable monster moves" },
#endif


#ifdef JP
	{ &disturb_pets,                FALSE, 4, 5, 6,
	"disturb_pets",                 "볦ΥڥåȤưưߤ" },
#else
	{ &disturb_pets,                FALSE, 4, 5, 6,
	"disturb_pets",                 "Disturb when visible pets move" },
#endif


#ifdef JP
	{ &disturb_panel,               TRUE,  4, 0, 22,
        "disturb_panel",                "̥˹ưߤ" },
#else
	{ &disturb_panel,               TRUE,  4, 0, 22,
	"disturb_panel",                "Disturb whenever map panel changes" },
#endif


#ifdef JP
	{ &disturb_state,               TRUE,  4, 0, 23,
        "disturb_state",                "ʬΥơѲ˹ưߤ" },
#else
	{ &disturb_state,               TRUE,  4, 0, 23,
	"disturb_state",                "Disturb whenever player state changes" },
#endif


#ifdef JP
	{ &disturb_minor,               TRUE,  4, 0, 24,
        "disturb_minor",                "٤ʤȤƤưߤ" },
#else
	{ &disturb_minor,               TRUE,  4, 0, 24,
	"disturb_minor",                "Disturb whenever boring things happen" },
#endif


#ifdef JP
	{ &ring_bell,                   FALSE, 4, 0, 14,
        "ring_bell",                    "顼˥ӡײĤ餹" },
#else
	{ &ring_bell,                   FALSE, 4, 0, 14,
	"ring_bell",                    "Audible bell (on errors, etc)" },
#endif


#ifdef JP
	{ &alert_hitpoint,              FALSE, 4, 0, 26,
        "alert_hitpoint",               "ҥåȥݥȤηٹ˥ӡפĤ餹" },
#else
	{ &alert_hitpoint,              FALSE, 4, 0, 26,
	"alert_hitpoint",               "Alert user to critical hitpoints" },
#endif


	/*** Efficiency ***/

#ifdef JP
	{ &view_reduce_lite,            FALSE, 5, 1, 16,
     "view_reduce_lite",  "äƤȤȾ¤򸺾" },
#else
	{ &view_reduce_lite,            FALSE, 5, 1, 16,
	"view_reduce_lite",             "Reduce lite-radius when running" },
#endif


#ifdef JP
	{ &view_reduce_view,            FALSE, 5, 1, 17,
	"view_reduce_view",             "Ǥϻ򶹤" },
#else
	{ &view_reduce_view,            FALSE, 5, 1, 17,
	"view_reduce_view",             "Reduce view-radius in town" },
#endif


#ifdef JP
	{ &avoid_abort,                 TRUE,  5, 1, 18,
        "avoid_abort",                  "Ϣ³ޥɤϥϤǤ" },
#else
	{ &avoid_abort,                 TRUE,  5, 1, 18,
	"avoid_abort",                  "Check for user abort while continuous command" },
#endif


#ifdef JP
	{ &flush_failure,               TRUE,  5, 1, 20,
        "flush_failure",                "͡ʥߥȯϤ򥯥ꥢ" },
#else
	{ &flush_failure,               TRUE,  5, 1, 20,
	"flush_failure",                "Flush input on various failures" },
#endif


#ifdef JP
	{ &flush_disturb,               FALSE, 5, 1, 21,
        "flush_disturb",                "㳲ȯϤ򥯥ꥢ" },
#else
	{ &flush_disturb,               FALSE, 5, 1, 21,
	"flush_disturb",                "Flush input whenever disturbed" },
#endif


#ifdef JP
	{ &fresh_before,                TRUE,  5, 1, 23,
        "fresh_before",                 "Ϣ³ޥ˲̤褷³" },
#else
	{ &fresh_before,                TRUE,  5, 1, 23,
        "fresh_before",                 "Flush output while continuous command" },
#endif


#ifdef JP
	{ &fresh_after,                 FALSE, 5, 1, 24,
        "fresh_after",                  "ޥɸ˲̤˺褷³" },
#else
	{ &fresh_after,                 FALSE, 5, 1, 24,
	"fresh_after",                  "Flush output after monster's move" },
#endif


#ifdef JP
	{ &fresh_message,               FALSE, 5, 1, 25,
	"fresh_message",                "åθ˲̤褹" },
#else
	{ &fresh_message,               FALSE, 5, 1, 25,
	"fresh_message",                "Flush output after every message" },
#endif


#ifdef JP
	{ &compress_savefile,           FALSE, 5, 1, 26,
        "compress_savefile",            "֡եΥå򰵽̤" },
#else
	{ &compress_savefile,           FALSE, 5, 1, 26,
	"compress_savefile",            "Compress messages in savefiles" },
#endif


#ifdef JP
	{ &hilite_player,               TRUE,  5, 1, 27,
	"hilite_player",                "ץ쥤䡼˥碌" },
#else
	{ &hilite_player,               TRUE,  5, 1, 27,
	"hilite_player",                "Hilite the player with the cursor" },
#endif


	/*** Birth Options ***/

#ifdef JP
	{ &easy_band,                   FALSE, 6, 6, 31,
        "easy_band",                    "鿴Ѵñ⡼(*)" },
#else
	{ &easy_band,                   FALSE, 6, 6, 31,
	"easy_band",                    "Easy Mode (*)" },
#endif


#ifdef JP
	{ &smart_learn,                 TRUE,  6, 1, 14,
        "smart_learn",                  "󥹥ϼԤؽ(*)" },
#else
	{ &smart_learn,                 TRUE,  6, 1, 14,
	"smart_learn",                  "Monsters learn from their mistakes (*)" },
#endif


#ifdef JP
	{ &smart_cheat,                 FALSE, 6, 1, 15,
        "smart_cheat",                  "󥹥ϥץ쥤䡼μߤͤ(*)" },
#else
	{ &smart_cheat,                 FALSE, 6, 1, 15,
	"smart_cheat",                  "Monsters exploit players weaknesses (*)" },
#endif


#ifdef JP
	{ &stupid_monsters,             FALSE, 6, 5, 2,
	"stupid_monsters",              "󥹥ȴʹưȤ(*)" },
#else
	{ &stupid_monsters,             FALSE, 6, 5, 2,
	"stupid_monsters",              "Monsters behave stupidly (*)" },
#endif



#ifdef JP
	{ &vanilla_town,                FALSE, 6, 6, 0,
	"vanilla_town",                 "Ĥγ/Ȥȹʤ" },
#else
	{ &vanilla_town,                FALSE, 6, 6, 0,
	"vanilla_town",                 "Use 'vanilla' town without quests and wilderness" },
#endif


#ifdef JP
	{ &lite_town,                   FALSE, 6, 6, 1,
	"lite_town",                    "Ϥʳ/ʤ" },
#else
	{ &lite_town,                   FALSE, 6, 6, 1,
	"lite_town",                    "Use 'lite' town without a wilderness" },
#endif


#ifdef JP
	{ &ironman_shops,               FALSE, 6, 6, 2,
	"ironman_shops",                "ŴѡŹѤʤ(*)" },
#else
	{ &ironman_shops,               FALSE, 6, 6, 2,
	"ironman_shops",                "Stores are permanently closed (*)" },
#endif


#ifdef JP
	{ &ironman_small_levels,        FALSE, 6, 6, 3,
	"ironman_small_levels",         "Ŵѡ˾˾ե(*)" },
#else
	{ &ironman_small_levels,        FALSE, 6, 6, 3,
	"ironman_small_levels",         "Always create unusually small dungeon levels (*)" },
#endif


#ifdef JP
	{ &ironman_downward,            FALSE, 6, 6, 4,
	"ironman_downward",             "Ŵѡ˵ԤȾ곬ʤʤ(*)" },
#else
	{ &ironman_downward,            FALSE, 6, 6, 4,
	"ironman_downward",             "Disable recall and use of up stairs (*)" },
#endif


#ifdef JP
	{ &ironman_autoscum,            FALSE, 6, 6, 5,
	"ironman_autoscum",             "Ŵѡ˾˳ʷϵư깥ߤ" },
#else
	{ &ironman_autoscum,            FALSE, 6, 6, 5,
	"ironman_autoscum",             "Permanently enable the autoscummer" },
#endif


#ifdef JP
       { &ironman_empty_levels,         FALSE, 6, 6, 8,
       "ironman_empty_levels",          "Ŵѡ˾˶äݤΥ꡼ʥ٥(*)" },
#else
	{ &ironman_empty_levels,        FALSE, 6, 6, 8,
	"ironman_empty_levels",         "Always create empty 'arena' levels (*)" },
#endif


#ifdef JP
       { &terrain_streams,              TRUE,  6, 6, 9,
       "terrain_streams",               "俹ʤɤ󥸥(*)" },
#else
       { &terrain_streams,              TRUE,  6, 6, 9,
       "terrain_streams",               "Create terrain 'streamers' in the dungeon (*)" },
#endif


#ifdef JP
       { &munchkin_death,               FALSE, 6, 6, 11,
       "munchkin_death",                "ʤ˻֤뤫ǧ(*)" },
#else
	{ &munchkin_death,              FALSE, 6, 6, 11,
	"munchkin_death",               "(Cheating) Ask for death saves (*)" },
#endif


#ifdef JP
       { &ironman_rooms,                FALSE, 6, 6, 12,
       "ironman_rooms",                 "Ŵѡ˾̤Ǥʤ(*)" },
#else
	{ &ironman_rooms,               FALSE, 6, 6, 12,
	"ironman_rooms",                "Always generate very unusual rooms (*)" },
#endif


#ifdef JP
	{ &ironman_nightmare,           FALSE, 6, 6, 18,
        "ironman_nightmare",	        "Ŵѡ˰̴⡼(ԾǤ)(*)" },
#else
	{ &ironman_nightmare,           FALSE, 6, 6, 18,
        "ironman_nightmare",	        "Nightmare mode(it isn't even remotely fair!)(*)" },
#endif

#ifdef JP
        { &left_hander,                 FALSE, 6, 6, 13,
        "left_hander",                  "Ǥ" },
#else
	{ &left_hander,                 FALSE, 6, 6, 13,
	"left_hander",                  "Left-Hander" },
#endif


#ifdef JP
        { &preserve_mode,               TRUE,  6, 6, 14,
        "preserve_mode",                "ΥƥƨƤ(*)" },
#else
	{ &preserve_mode,               TRUE,  6, 6, 14,
	"preserve_mode",                "Preserve artifacts (*)" },
#endif


#ifdef JP
        { &autoroller,                  TRUE,  6, 6, 15,
        "autoroller",                   "ǽͤ˥ȥ顼(*)" },
#else
	{ &autoroller,                  TRUE,  6, 6, 15,
	"autoroller",                   "Allow use of autoroller for stats (*)" },
#endif


#ifdef JP
        { &autochara,                   FALSE, 6, 6, 16,
        "autochara",                   "γ/ϰ̤˥ȥ顼" },
#else
	{ &autochara,                   FALSE, 6, 6, 16,
	"autochara",                    "Autorole for weight, height and social status" },
#endif


#ifdef JP
	{ &powerup_home,                TRUE,  6, 4, 3,
        "powerup_home",                 "椬Ȥĥ(*)" },

#else
	{ &powerup_home,                TRUE,  6, 4, 3,
        "powerup_home",                 "Increase capacity of your home (*)" },
#endif


	/*** Easy Object Auto-Destroyer ***/

#ifdef JP
        { &destroy_items,               FALSE, 7, 7, 0,
        "destroy_items",                "ƥμư˲Ѥ" },
#else
        { &destroy_items,               FALSE, 7, 7, 0,
        "destroy_items",                "Use easy auto-destroyer" },
#endif

#ifdef JP
        { &leave_worth,                 FALSE, 7, 7, 2,
        "leave_worth",                  "ͤ륢ƥϲʤ" },
#else
        { &leave_worth,                 FALSE, 7, 7, 2,
        "leave_worth",                  "Auto-destroyer leaves known worthy items" },
#endif

#ifdef JP
        { &leave_equip,                 FALSE, 7, 7, 3,
        "leave_equip",                  "/ɶϲʤ" },
#else
        { &leave_equip,                 FALSE, 7, 7, 3,
        "leave_equip",                  "Auto-destroyer leaves weapons and armour" },
#endif

#ifdef JP
        { &leave_chest,                 FALSE, 7, 7, 7,
        "leave_chest",                  "ƤʤȢϲʤ" },
#else
        { &leave_chest,                 FALSE, 7, 7, 7,
        "leave_chest",                  "Auto-destroyer leaves closed chests" },
#endif

#ifdef JP
        { &leave_wanted,                FALSE, 7, 7, 4,
        "leave_wanted",                 "޶λ/ϲʤ" },
#else
        { &leave_wanted,                FALSE, 7, 7, 4,
        "leave_wanted",                 "Auto-destroyer leaves wanted corpses" },
#endif

#ifdef JP
        { &leave_corpse,                FALSE, 7, 7, 5,
        "leave_corpse",                 "/ϲʤ" },
#else
        { &leave_corpse,                FALSE, 7, 7, 5,
        "leave_corpse",                 "Auto-destroyer leaves corpses and skeletons" },
#endif

#ifdef JP
        { &leave_junk,                  FALSE, 7, 7, 6,
        "leave_junk",                   "餯ϲʤ" },
#else
        { &leave_junk,                  FALSE, 7, 7, 6,
        "leave_junk",                   "Auto-destroyer leaves junks" },
#endif


	/*** Play-record Options ***/

#ifdef JP
        { &record_fix_art,              TRUE, 10, 4, 11,
        "record_fix_art",               "ꥢƥեȤϿ" },
#else
        { &record_fix_art,              TRUE, 10, 4, 11,
        "record_fix_art",               "Record fixed artifacts" },
#endif

#ifdef JP
        { &record_rand_art,             TRUE, 10, 4, 12,
        "record_rand_art",              "ॢƥեȤϿ" },
#else
        { &record_rand_art,             TRUE, 10, 4, 12,
        "record_rand_art",              "Record random artifacts" },
#endif

#ifdef JP
        { &record_destroy_uniq,         TRUE, 10, 4, 13,
        "record_destroy_uniq",          "ˡ󥹥ݤȤϿ" },
#else
        { &record_destroy_uniq,         TRUE, 10, 4, 13,
        "record_destroy_uniq",          "Record when destroy unique monster" },
#endif

#ifdef JP
        { &record_fix_quest,            TRUE, 10, 4, 14,
        "record_fix_quest",             "ꥯȤãϿ" },
#else
        { &record_fix_quest,            TRUE, 10, 4, 14,
        "record_fix_quest",             "Record fixed quests" },
#endif

#ifdef JP
        { &record_rand_quest,           TRUE, 10, 4, 15,
        "record_rand_quest",            "९ȤãϿ" },
#else
        { &record_rand_quest,           TRUE, 10, 4, 15,
        "record_rand_quest",            "Record random quests" },
#endif

#ifdef JP
        { &record_maxdeapth,            TRUE, 10, 4, 16,
        "record_maxdeapth",             "ǿ򹹿Ȥ˵Ͽ" },
#else
        { &record_maxdeapth,            TRUE, 10, 4, 16,
        "record_maxdeapth",             "Record movements to deepest level" },
#endif

#ifdef JP
        { &record_stair,                TRUE, 10, 4, 17,
        "record_stair",                 "ΰưϿ" },
#else
        { &record_stair,                TRUE, 10, 4, 17,
        "record_stair",                 "Record recall and stair movements" },
#endif

#ifdef JP
        { &record_buy,                  TRUE, 10, 4, 18,
        "record_buy",                   "ƥιϿ" },
#else
        { &record_buy,                  TRUE, 10, 4, 18,
        "record_buy",                   "Record purchased items" },
#endif

#ifdef JP
        { &record_sell,                 FALSE, 10, 4, 19,
        "record_sell",                  "ƥѤϿ" },
#else
        { &record_sell,                 FALSE, 10, 4, 19,
        "record_sell",                  "Record sold items" },
#endif

#ifdef JP
        { &record_danger,               TRUE, 10, 4, 20,
        "record_danger",                "ԥˤʤäȤϿ" },
#else
        { &record_danger,               TRUE, 10, 4, 20,
        "record_danger",                "Record hitpoint warning" },
#endif

#ifdef JP
        { &record_arena,                TRUE, 10, 4, 21,
        "record_arena",                 "꡼ʤǤξϿ" },
#else
        { &record_arena,                TRUE, 10, 4, 21,
        "record_arena",                 "Record arena victories" },
#endif

#ifdef JP
        { &record_ident,                TRUE, 10, 4, 22,
        "record_ident",                 "̤ȽΥƥμ̤Ͽ" },
#else
        { &record_ident,                TRUE, 10, 4, 22,
        "record_ident",                 "Record first identified items" },
#endif

#ifdef JP
        { &record_named_pet,            FALSE, 10, 4, 23,
        "record_named_pet",             "̾ĤڥåȤξϿ" },
#else
        { &record_named_pet,            FALSE, 10, 4, 23,
        "record_named_pet",             "Record infomations of named pets" },
#endif


	/*** End of Table ***/

	{ NULL,                         0, 0, 0, 0,
	NULL,                           NULL }
};


#ifdef JP
cptr chaos_patrons[MAX_PATRON] =
{
	"",
	"ޥ٥",
	"㡼ɥ",
	"ϥϡ",
	"С",

	"ԥ졼",
	"Х",
	"ꥪå",
	"",
	"ʡ",

	"Х",
	"",
	"顼ͥå",
	"ʡ",
	"ƥ",

	""
};
#else
cptr chaos_patrons[MAX_PATRON] =
{
	"Slortar",
	"Mabelode",
	"Chardros",
	"Hionhurn",
	"Xiombarg",

	"Pyaray",
	"Balaan",
	"Arioch",
	"Eequor",
	"Narjhan",

	"Balo",
	"Khorne",
	"Slaanesh",
	"Nurgle",
	"Tzeentch",

	"Khaine"
};
#endif


int chaos_stats[MAX_PATRON] =
{
	A_CON,  /* Slortar */
	A_CON,  /* Mabelode */
	A_STR,  /* Chardros */
	A_STR,  /* Hionhurn */
	A_STR,  /* Xiombarg */

	A_INT,  /* Pyaray */
	A_STR,  /* Balaan */
	A_INT,  /* Arioch */
	A_CON,  /* Eequor */
	A_CHR,  /* Narjhan */

	-1,     /* Balo */
	A_STR,  /* Khorne */
	A_CHR,  /* Slaanesh */
	A_CON,  /* Nurgle */
	A_INT,  /* Tzeentch */

	A_STR,  /* Khaine */
};




int chaos_rewards[MAX_PATRON][20] =
{
	/* Slortar the Old: */
	{
		REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_POLY_SLF,
		REW_POLY_SLF, REW_POLY_SLF, REW_GAIN_ABL, REW_GAIN_ABL, REW_GAIN_EXP,
		REW_GOOD_OBJ, REW_CHAOS_WP, REW_GREA_OBJ, REW_AUGM_ABL, REW_AUGM_ABL
	},

	/* Mabelode the Faceless: */
	{
		REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_H_SUMMON, REW_SUMMON_M,
		REW_SUMMON_M, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_POLY_WND,
		REW_POLY_SLF, REW_HEAL_FUL, REW_HEAL_FUL, REW_GAIN_ABL, REW_SER_UNDE,
		REW_CHAOS_WP, REW_GOOD_OBJ, REW_GOOD_OBJ, REW_GOOD_OBS, REW_GOOD_OBS
	},

	/* Chardros the Reaper: */
	{
		REW_WRATH, REW_WRATH, REW_HURT_LOT, REW_PISS_OFF, REW_H_SUMMON,
		REW_SUMMON_M, REW_IGNORE, REW_IGNORE, REW_DESTRUCT, REW_SER_UNDE,
		REW_GENOCIDE, REW_MASS_GEN, REW_MASS_GEN, REW_DISPEL_C, REW_GOOD_OBJ,
		REW_CHAOS_WP, REW_GOOD_OBS, REW_GOOD_OBS, REW_AUGM_ABL, REW_AUGM_ABL
	},

	/* Hionhurn the Executioner: */
	{
		REW_WRATH, REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL,
		REW_IGNORE, REW_IGNORE, REW_SER_UNDE, REW_DESTRUCT, REW_GENOCIDE,
		REW_MASS_GEN, REW_MASS_GEN, REW_HEAL_FUL, REW_GAIN_ABL, REW_GAIN_ABL,
		REW_CHAOS_WP, REW_GOOD_OBS, REW_GOOD_OBS, REW_AUGM_ABL, REW_AUGM_ABL
	},

	/* Xiombarg the Sword-Queen: */
	{
		REW_TY_CURSE, REW_TY_CURSE, REW_PISS_OFF, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_WND, REW_POLY_WND,
		REW_GENOCIDE, REW_DISPEL_C, REW_GOOD_OBJ, REW_GOOD_OBJ, REW_SER_MONS,
		REW_GAIN_ABL, REW_CHAOS_WP, REW_GAIN_EXP, REW_AUGM_ABL, REW_GOOD_OBS
	},


	/* Pyaray the Tentacled Whisperer of Impossible Secretes: */
	{
		REW_WRATH, REW_TY_CURSE, REW_PISS_OFF, REW_H_SUMMON, REW_H_SUMMON,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_POLY_SLF,
		REW_POLY_SLF, REW_SER_DEMO, REW_HEAL_FUL, REW_GAIN_ABL, REW_GAIN_ABL,
		REW_CHAOS_WP, REW_DO_HAVOC, REW_GOOD_OBJ, REW_GREA_OBJ, REW_GREA_OBS
	},

	/* Balaan the Grim: */
	{
		REW_TY_CURSE, REW_HURT_LOT, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL,
		REW_SUMMON_M, REW_LOSE_EXP, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_WND,
		REW_SER_UNDE, REW_HEAL_FUL, REW_HEAL_FUL, REW_GAIN_EXP, REW_GAIN_EXP,
		REW_CHAOS_WP, REW_GOOD_OBJ, REW_GOOD_OBS, REW_GREA_OBS, REW_AUGM_ABL
	},

	/* Arioch, Duke of Hell: */
	{
		REW_WRATH, REW_PISS_OFF, REW_RUIN_ABL, REW_LOSE_EXP, REW_H_SUMMON,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_SLF,
		REW_POLY_SLF, REW_MASS_GEN, REW_SER_DEMO, REW_HEAL_FUL, REW_CHAOS_WP,
		REW_CHAOS_WP, REW_GOOD_OBJ, REW_GAIN_EXP, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Eequor, Blue Lady of Dismay: */
	{
		REW_WRATH, REW_TY_CURSE, REW_PISS_OFF, REW_CURSE_WP, REW_RUIN_ABL,
		REW_IGNORE, REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_WND,
		REW_GOOD_OBJ, REW_GOOD_OBJ, REW_SER_MONS, REW_HEAL_FUL, REW_GAIN_EXP,
		REW_GAIN_ABL, REW_CHAOS_WP, REW_GOOD_OBS, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Narjhan, Lord of Beggars: */
	{
		REW_WRATH, REW_CURSE_AR, REW_CURSE_WP, REW_CURSE_WP, REW_CURSE_AR,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF,
		REW_POLY_WND, REW_HEAL_FUL, REW_HEAL_FUL, REW_GAIN_EXP, REW_AUGM_ABL,
		REW_GOOD_OBJ, REW_GOOD_OBJ, REW_CHAOS_WP, REW_GREA_OBJ, REW_GREA_OBS
	},

	/* Balo the Jester: */
	{
		REW_WRATH, REW_SER_DEMO, REW_CURSE_WP, REW_CURSE_AR, REW_LOSE_EXP,
		REW_GAIN_ABL, REW_LOSE_ABL, REW_POLY_WND, REW_POLY_SLF, REW_IGNORE,
		REW_DESTRUCT, REW_MASS_GEN, REW_CHAOS_WP, REW_GREA_OBJ, REW_HURT_LOT,
		REW_AUGM_ABL, REW_RUIN_ABL, REW_H_SUMMON, REW_GREA_OBS, REW_AUGM_ABL
	},

	/* Khorne the Bloodgod: */
	{
		REW_WRATH, REW_HURT_LOT, REW_HURT_LOT, REW_H_SUMMON, REW_H_SUMMON,
		REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_SER_MONS, REW_SER_DEMO,
		REW_POLY_SLF, REW_POLY_WND, REW_HEAL_FUL, REW_GOOD_OBJ, REW_GOOD_OBJ,
		REW_CHAOS_WP, REW_GOOD_OBS, REW_GOOD_OBS, REW_GREA_OBJ, REW_GREA_OBS
	},

	/* Slaanesh: */
	{
		REW_WRATH, REW_PISS_OFF, REW_PISS_OFF, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_LOSE_EXP, REW_IGNORE, REW_IGNORE, REW_POLY_WND, REW_SER_DEMO,
		REW_POLY_SLF, REW_HEAL_FUL, REW_HEAL_FUL, REW_GOOD_OBJ, REW_GAIN_EXP,
		REW_GAIN_EXP, REW_CHAOS_WP, REW_GAIN_ABL, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Nurgle: */
	{
		REW_WRATH, REW_PISS_OFF, REW_HURT_LOT, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_LOSE_EXP, REW_IGNORE, REW_IGNORE, REW_IGNORE, REW_POLY_SLF,
		REW_POLY_SLF, REW_POLY_WND, REW_HEAL_FUL, REW_GOOD_OBJ, REW_GAIN_ABL,
		REW_GAIN_ABL, REW_SER_UNDE, REW_CHAOS_WP, REW_GREA_OBJ, REW_AUGM_ABL
	},

	/* Tzeentch: */
	{
		REW_WRATH, REW_CURSE_WP, REW_CURSE_AR, REW_RUIN_ABL, REW_LOSE_ABL,
		REW_LOSE_EXP, REW_IGNORE, REW_POLY_SLF, REW_POLY_SLF, REW_POLY_SLF,
		REW_POLY_SLF, REW_POLY_WND, REW_HEAL_FUL, REW_CHAOS_WP, REW_GREA_OBJ,
		REW_GAIN_ABL, REW_GAIN_ABL, REW_GAIN_EXP, REW_GAIN_EXP, REW_AUGM_ABL
	},

	/* Khaine: */
	{
		REW_WRATH, REW_HURT_LOT, REW_PISS_OFF, REW_LOSE_ABL, REW_LOSE_EXP,
		REW_IGNORE,   REW_IGNORE,   REW_DISPEL_C, REW_DO_HAVOC, REW_DO_HAVOC,
		REW_POLY_SLF, REW_POLY_SLF, REW_GAIN_EXP, REW_GAIN_ABL, REW_GAIN_ABL,
		REW_SER_MONS, REW_GOOD_OBJ, REW_CHAOS_WP, REW_GREA_OBJ, REW_GOOD_OBS
	}
};

martial_arts ma_blows[MAX_MA] =
{
#ifdef JP
	{ "%s򲥤ä",                          1, 0, 1, 5, 0 },
	{ "%s򽳤ä",                           2, 0, 1, 7, 0 },
	{ "%sͤ򤯤路",                         3, 0, 1, 9, 0 },
	{ "%sɨ򤯤路",             5, 5, 2, 4, MA_KNEE },
	{ "%sɪǤ򤯤路",            7, 5, 1, 12, 0 },
	{ "%sꤷ",                           9, 10, 2, 6, 0 },
	{ "%s򽳤ä",                           11, 10, 3, 6, MA_SLOW },
	{ "%s˥åѡ򤯤路",                       13, 12, 5, 5, 6 },
	{ "%sʽ򤯤路",                    16, 15, 5, 6, 8 },
	{ "%sǭ޷򤯤路",          20, 20, 5, 8, 0 },
	{ "%sķӤ򤯤路",           24, 25, 6, 8, 10 },
	{ "%s޽򤯤路",       28, 25, 7, 9, 0 },
	{ "%s˲󤷽򤯤路",         32, 30, 8, 10, 10 },
	{ "%sŴ򤯤路",          35, 35, 8, 11, 10 },
	{ "%sӤ򤯤路",         39, 35, 8, 12, 12 },
	{ "%s˾ζ򤯤路",         43, 35, 9, 12, 16 },
	{ "%sŷ÷򤯤路",       48, 40, 10, 13, 18 },
#else
	{ "You punch %s.",                          1, 0, 1, 4, 0 },
	{ "You kick %s.",                           2, 0, 1, 6, 0 },
	{ "You strike %s.",                         3, 0, 1, 7, 0 },
	{ "You hit %s with your knee.",             5, 5, 2, 3, MA_KNEE },
	{ "You hit %s with your elbow.",            7, 5, 1, 8, 0 },
	{ "You butt %s.",                           9, 10, 2, 5, 0 },
	{ "You kick %s.",                           11, 10, 3, 4, MA_SLOW },
	{ "You uppercut %s.",                       13, 12, 4, 4, 6 },
	{ "You double-kick %s.",                    16, 15, 5, 4, 8 },
	{ "You hit %s with a Cat's Claw.",          20, 20, 5, 5, 0 },
	{ "You hit %s with a jump kick.",           25, 25, 5, 6, 10 },
	{ "You hit %s with an Eagle's Claw.",       29, 25, 6, 6, 0 },
	{ "You hit %s with a circle kick.",         33, 30, 6, 8, 10 },
	{ "You hit %s with an Iron Fist.",          37, 35, 8, 8, 10 },
	{ "You hit %s with a flying kick.",         41, 35, 8, 10, 12 },
	{ "You hit %s with a Dragon Fist.",         45, 35, 10, 10, 16 },
	{ "You hit %s with a Crushing Blow.",       48, 35, 10, 12, 18 },
#endif

};

int monk_ave_damage[PY_MAX_LEVEL+1][3] =
{
  {0, 0, 0},
  {249, 249, 249},
  {324, 324, 324},
  {382, 438, 382},
  {382, 439, 382},
  {390, 446, 390},
  {394, 473, 394},
  {425, 528, 425},
  {430, 535, 430},
  {505, 560, 435},
  {517, 575, 444},
  {566, 655, 474},
  {585, 713, 486},
  {653, 843, 527},
  {678, 890, 544},
  {703, 973, 558},
  {765, 1096, 596},
  {914, 1146, 614},
  {943, 1240, 629},
  {971, 1276, 643},
  {1018, 1350, 667},
  {1063, 1464, 688},
  {1099, 1515, 705},
  {1128, 1559, 721},
  {1153, 1640, 735},
  {1336, 1720, 757},
  {1387, 1789, 778},
  {1430, 1893, 794},
  {1610, 2199, 863},
  {1666, 2280, 885},
  {1713, 2401, 908},
  {1755, 2465, 925},
  {1909, 2730, 984},
  {2156, 2891, 1009},
  {2218, 2970, 1031},
  {2319, 3107, 1063},
  {2404, 3290, 1098},
  {2477, 3389, 1125},
  {2544, 3483, 1150},
  {2771, 3899, 1228},
  {2844, 3982, 1259},
  {3129, 4064, 1287},
  {3200, 4190, 1313},
  {3554, 4674, 1432},
  {3614, 4738, 1463},
  {3679, 4853, 1485},
  {3741, 4905, 1512},
  {3785, 4943, 1538},
  {4141, 5532, 1652},
  {4442, 5581, 1679},
  {4486, 5636, 1702},
};

/*
 * Table of game-generated inscriptions (indexed by the defines in
 * defines.h). -- RG
 */
cptr game_inscriptions[] =
{
	NULL,            /* FEEL_NONE */
#ifdef JP
	"Ƥ",    /* FEEL_BROKEN */
	"",      /* FEEL_TERRIBLE */
	"̵",        /* FEEL_WORTHLESS */
	"Ƥ",  /* FEEL_CURSED */
	"ʤ",      /* FEEL_UNCURSED */
	"",            /* FEEL_AVERAGE */
	"",          /* FEEL_GOOD */
	"",        /* FEEL_EXCELLENT */
	"",        /* FEEL_SPECIAL */
#else
	"broken",        /* FEEL_BROKEN */
	"terrible",      /* FEEL_TERRIBLE */
	"worthless",     /* FEEL_WORTHLESS */
	"cursed",        /* FEEL_CURSED */
	"uncursed",      /* FEEL_UNCURSED */
	"average",       /* FEEL_AVERAGE */
	"good",          /* FEEL_GOOD */
	"excellent",     /* FEEL_EXCELLENT */
	"special",       /* FEEL_SPECIAL */
#endif

};

kamae kamae_shurui[MAX_KAMAE] =
{
#ifdef JP
	{"", 25, ""},
	{"", 30, ""},
	{"ε", 35, ""},
	{"", 40, ""},
#else
	{"Baihu", 25, "(Black Turtle) "},
	{"Zuan-wu", 30, "(White Tiger) "},
	{"Qinglong", 35, "(Blue Chinese Dragon) "},
	{"Zhuque", 40, "(Red Chinese Phoenix) "},
#endif
};

kamae kata_shurui[MAX_KATA] =
{
#ifdef JP
	{"", 25, ""},
	{"", 30, ""},
	{"ߵ", 35, ""},
	{"̵", 40, ""},
#else
	{"Iai", 25, ""},
	{"Huujin", 30, ""},
	{"Kouki", 35, ""},
	{"Musou", 40, ""},
#endif
};

cptr shougou_moji[5]=
#ifdef JP
{"[鿴]", "[]", "[]", "[ѡ]", "[ã]"};
#else
{"[Unskilled]", "[Beginer]", "[Skilled]", "[Expert]", "[Master]"};
#endif

s16b weapon_exp_settei[MAX_CLASS][5][64][2] =
{
  /* WARRIOR */
  {
    /* Bow */
    {
      {4000,8000},/* 0 */
      {4000,8000},/*  */
      {4000,7000},/* SLING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {4000,7000},/* SHORT_BOW */
      {4000,7000},/* LONG_BOW */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,7000},/* LIGHT_XBOW */
      {4000,7000},/* HEAVY_XBOW */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Digging */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* SHOVEL */
      {4000,8000},/* GNOMISH_SHOVEL */
      {4000,8000},/* DWARVEN_SHOVEL */
      {4000,8000},/* PICK */
      {4000,8000},/* ORCISH_PICK */
      {4000,8000},/* DWARVEN_PICK */
      {4000,8000},/* MATTOCK */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Hafted */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* CLUB */
      {4000,8000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {0,6000},/* NUNCHAKU */
      {4000,8000},/* MACE */
      {4000,8000},/* BALL_AND_CHAIN */
      {4000,8000},/* JO_STAFF */
      {4000,8000},/* WAR_HAMMER */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/* THREE_PIECE_ROD */
      {4000,8000},/* MORNING_STAR */
      {4000,8000},/* FLAIL */
      {4000,8000},/* BO_STAFF */
      {4000,8000},/* LEAD_FILLED_MACE */
      {4000,8000},/* TETSUBO */
      {4000,8000},/*  */
      {4000,8000},/* TWO_HANDED_FLAIL */
      {4000,8000},/* GREAT_HAMMER */
      {4000,8000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,0},/* TSURIZAO */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* GROND */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* HATCHET */
      {4000,8000},/* SPEAR */
      {4000,8000},/* SICKLE */
      {4000,8000},/* AWL_PIKE */
      {4000,8000},/* TRIDENT */
      {4000,8000},/* FAUCHARD */
      {4000,8000},/* BROAD_SPEAR */
      {4000,8000},/* PIKE */
      {4000,8000},/* NAGINATA */
      {4000,8000},/* BEAKED_AXE */
      {4000,8000},/* BROAD_AXE */
      {4000,8000},/* LUCERNE_HAMMER */
      {4000,8000},/* GLAIVE */
      {4000,8000},/* LAJATANG */
      {4000,8000},/* HALBERD */
      {4000,8000},/* GUISARME */
      {4000,8000},/* SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* LANCE */
      {4000,8000},/*  */
      {4000,8000},/* BATTLE_AXE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* GREAT_AXE */
      {4000,8000},/* TRIFURCATE_SPEAR */
      {4000,8000},/*  */
      {4000,8000},/* LOCHABER_AXE */
      {4000,8000},/* HEAVY_LANCE */
      {4000,8000},/* SCYTHE_OF_SLICING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* DEATH_SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Sword */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* BROKEN_DAGGER */
      {4000,8000},/* BROKEN_SWORD */
      {4000,8000},/*  */
      {4000,8000},/* DAGGER */
      {4000,8000},/* MAIN_GAUCHE */
      {4000,8000},/* TANTO */
      {4000,8000},/* RAPIER */
      {4000,8000},/* SMALL_SWORD */
      {4000,8000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {4000,8000},/* SABRE */
      {4000,8000},/* CUTLASS */
      {4000,8000},/* WAKIZASHI */
      {4000,8000},/* KHOPESH */
      {4000,8000},/* TULWAR */
      {4000,8000},/* BROAD_SWORD */
      {4000,8000},/* LONG_SWORD */
      {4000,8000},/* SCIMITAR */
      {4000,8000},/* NINJATO */
      {4000,8000},/* KATANA */
      {4000,8000},/* BASTARD_SWORD */
      {4000,8000},/* GREAT_SCIMITAR */
      {4000,8000},/* CLAYMORE */
      {4000,8000},/* ESPADON */
      {4000,8000},/* TWO_HANDED_SWORD */
      {4000,8000},/* FLAMBERGE */
      {4000,8000},/* NO_DACHI */
      {4000,8000},/* EXECUTIOERS_SWORD */
      {4000,8000},/* ZWEIHANDER */
      {4000,8000},/* BLADE_OF_CHAOS */
      {4000,8000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    }
  },
  /* MAGE */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {4000,6000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,4000},/* 0 */
      {0,4000},/* CLUB */
      {0,4000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {0,0},/* NUNCHAKU */
      {0,4000},/* MACE */
      {0,4000},/* BALL_AND_CHAIN */
      {0,4000},/* JO_STAFF */
      {0,4000},/* WAR_HAMMER */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/* THREE_PIECE_ROD */
      {0,4000},/* MORNING_STAR */
      {0,4000},/* FLAIL */
      {0,4000},/* BO_STAFF */
      {0,4000},/* LEAD_FILLED_MACE */
      {0,4000},/* TETSUBO */
      {0,4000},/*  */
      {0,4000},/* TWO_HANDED_FLAIL */
      {0,4000},/* GREAT_HAMMER */
      {0,4000},/* MACE_OF_DISRUPTION */
      {4000,8000},/* WIZSTAFF */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,0},/* TSURIZAO */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GROND */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,4000},/* 0 */
      {0,4000},/* HATCHET */
      {0,4000},/* SPEAR */
      {0,4000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,4000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LANCE */
      {0,4000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,4000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,4000},/* HEAVY_LANCE */
      {0,4000},/* SCYTHE_OF_SLICING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* DEATH_SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Sword */
    {
      {0,4000},/* 0 */
      {0,4000},/* BROKEN_DAGGER */
      {0,4000},/* BROKEN_SWORD */
      {0,4000},/*  */
      {4000,8000},/* DAGGER */
      {0,4000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,4000},/* RAPIER */
      {0,4000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {0,4000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,4000},/* CUTLASS */
      {0,4000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,4000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,4000},/* LONG_SWORD */
      {0,4000},/* SCIMITAR */
      {0,4000},/* NINJATO */
      {0,4000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,4000},/* EXECUTIOERS_SWORD */
      {0,4000},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    }
  },
  /* PRIEST */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {0,4000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,7000},/* 0 */
      {0,7000},/* CLUB */
      {0,7000},/* WHIP */
      {0,7000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {4000,7000},/* MACE */
      {0,7000},/* BALL_AND_CHAIN */
      {0,7000},/* JO_STAFF */
      {0,7000},/* WAR_HAMMER */
      {0,7000},/*  */
      {0,7000},/* 10 */
      {0,7000},/* THREE_PIECE_ROD */
      {0,7000},/* MORNING_STAR */
      {0,7000},/* FLAIL */
      {0,7000},/* BO_STAFF */
      {0,7000},/* LEAD_FILLED_MACE */
      {0,7000},/* TETSUBO */
      {0,7000},/*  */
      {0,7000},/* TWO_HANDED_FLAIL */
      {0,7000},/* GREAT_HAMMER */
      {0,7000},/* MACE_OF_DISRUPTION */
      {4000,7000},/* WIZSTAFF */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 30 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,0},/* TSURIZAO */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* GROND */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,4000},/* 0 */
      {0,4000},/* HATCHET */
      {0,4000},/* SPEAR */
      {0,4000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,4000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LANCE */
      {0,4000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,4000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,4000},/* HEAVY_LANCE */
      {0,4000},/* SCYTHE_OF_SLICING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* DEATH_SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Sword */
    {
      {0,4000},/* 0 */
      {0,4000},/* BROKEN_DAGGER */
      {0,4000},/* BROKEN_SWORD */
      {0,4000},/*  */
      {0,4000},/* DAGGER */
      {0,4000},/* MAIN_GAUCHE */
      {0,4000},/* TANTO */
      {0,4000},/* RAPIER */
      {0,4000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {0,4000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,4000},/* CUTLASS */
      {0,4000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,4000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,4000},/* LONG_SWORD */
      {0,4000},/* SCIMITAR */
      {0,4000},/* NINJATO */
      {0,4000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,4000},/* EXECUTIOERS_SWORD */
      {0,4000},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,4000},/* DOKUBARI */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    }
  },
  /* ROGUE */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {4000,8000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {4000,7000},/* SHORT_BOW */
      {4000,7000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* LIGHT_XBOW */
      {0,7000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,7000},/* 0 */
      {0,7000},/* SHOVEL */
      {0,7000},/* GNOMISH_SHOVEL */
      {0,7000},/* DWARVEN_SHOVEL */
      {0,7000},/* PICK */
      {0,7000},/* ORCISH_PICK */
      {0,7000},/* DWARVEN_PICK */
      {0,7000},/* MATTOCK */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 10 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 20 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 30 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 40 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 50 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,7000},/* CLUB */
      {4000,8000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {4000,7000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,7000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,4000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,4000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,7000},/* HATCHET */
      {0,7000},/* SPEAR */
      {0,7000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,7000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {4000,8000},/* BROKEN_DAGGER */
      {4000,8000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {4000,8000},/* DAGGER */
      {4000,8000},/* MAIN_GAUCHE */
      {4000,8000},/* TANTO */
      {0,8000},/* RAPIER */
      {4000,8000},/* SMALL_SWORD */
      {0,7000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {0,8000},/* SABRE */
      {0,8000},/* CUTLASS */
      {0,8000},/* WAKIZASHI */
      {0,8000},/* KHOPESH */
      {0,8000},/* TULWAR */
      {0,7000},/* BROAD_SWORD */
      {0,7000},/* LONG_SWORD */
      {0,7000},/* SCIMITAR */
      {0,7000},/* NINJATO */
      {0,8000},/* KATANA */
      {0,7000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,6000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* RANGER */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {0,7000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {4000,8000},/* SHORT_BOW */
      {4000,8000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* LIGHT_XBOW */
      {0,7000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {4000,8000},/* DAGGER */
      {0,7000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {0,6000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {4000,8000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* PALADIN */
  {
    /* Bow */
    {
      {0,6000},/* 0 */
      {0,6000},/*  */
      {0,6000},/* SLING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,8000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,8000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* BROKEN_DAGGER */
      {4000,8000},/* BROKEN_SWORD */
      {4000,8000},/*  */
      {0,6000},/* DAGGER */
      {4000,8000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {4000,8000},/* RAPIER */
      {4000,8000},/* SMALL_SWORD */
      {4000,8000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {4000,8000},/* SABRE */
      {4000,8000},/* CUTLASS */
      {4000,8000},/* WAKIZASHI */
      {4000,8000},/* KHOPESH */
      {4000,8000},/* TULWAR */
      {4000,8000},/* BROAD_SWORD */
      {4000,8000},/* LONG_SWORD */
      {4000,8000},/* SCIMITAR */
      {4000,8000},/* NINJATO */
      {4000,8000},/* KATANA */
      {4000,8000},/* BASTARD_SWORD */
      {4000,8000},/* GREAT_SCIMITAR */
      {4000,8000},/* CLAYMORE */
      {4000,8000},/* ESPADON */
      {4000,8000},/* TWO_HANDED_SWORD */
      {4000,8000},/* FLAMBERGE */
      {4000,8000},/* NO_DACHI */
      {4000,8000},/* EXECUTIOERS_SWORD */
      {4000,8000},/* ZWEIHANDER */
      {4000,8000},/* BLADE_OF_CHAOS */
      {4000,8000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    }
  },
  /* WARRIOR_MAGE */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {0,4000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {0,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {4000,8000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {4000,8000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,6000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* CHAOS_WARRIOR */
  {
    /* Bow */
    {
      {0,6000},/* 0 */
      {0,6000},/*  */
      {0,6000},/* SLING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,7000},/* 0 */
      {0,7000},/* CLUB */
      {0,7000},/* WHIP */
      {0,7000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,7000},/* MACE */
      {0,7000},/* BALL_AND_CHAIN */
      {0,7000},/* JO_STAFF */
      {0,7000},/* WAR_HAMMER */
      {0,7000},/*  */
      {0,7000},/* 10 */
      {0,7000},/* THREE_PIECE_ROD */
      {0,7000},/* MORNING_STAR */
      {0,7000},/* FLAIL */
      {0,7000},/* BO_STAFF */
      {0,7000},/* LEAD_FILLED_MACE */
      {0,7000},/* TETSUBO */
      {0,7000},/*  */
      {0,7000},/* TWO_HANDED_FLAIL */
      {0,7000},/* GREAT_HAMMER */
      {0,7000},/* MACE_OF_DISRUPTION */
      {0,7000},/* WIZSTAFF */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 30 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,0},/* TSURIZAO */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* GROND */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,7000},/* 0 */
      {0,7000},/* HATCHET */
      {0,7000},/* SPEAR */
      {0,7000},/* SICKLE */
      {0,7000},/* AWL_PIKE */
      {0,7000},/* TRIDENT */
      {0,7000},/* FAUCHARD */
      {0,7000},/* BROAD_SPEAR */
      {0,7000},/* PIKE */
      {0,7000},/* NAGINATA */
      {0,7000},/* BEAKED_AXE */
      {0,7000},/* BROAD_AXE */
      {0,7000},/* LUCERNE_HAMMER */
      {0,7000},/* GLAIVE */
      {0,7000},/* LAJATANG */
      {0,7000},/* HALBERD */
      {0,7000},/* GUISARME */
      {0,7000},/* SCYTHE */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* LANCE */
      {0,7000},/*  */
      {0,7000},/* BATTLE_AXE */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* GREAT_AXE */
      {0,7000},/* TRIFURCATE_SPEAR */
      {0,7000},/*  */
      {0,7000},/* LOCHABER_AXE */
      {0,7000},/* HEAVY_LANCE */
      {0,7000},/* SCYTHE_OF_SLICING */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 40 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* DEATH_SCYTHE */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000} /*  */
    },
    /* Sword */
    {
      {0,7000},/* 0 */
      {0,7000},/* BROKEN_DAGGER */
      {0,7000},/* BROKEN_SWORD */
      {0,7000},/*  */
      {0,7000},/* DAGGER */
      {0,7000},/* MAIN_GAUCHE */
      {0,7000},/* TANTO */
      {0,7000},/* RAPIER */
      {0,7000},/* SMALL_SWORD */
      {0,7000},/* BASILLARD */
      {0,7000},/* SHORT_SWORD */
      {0,7000},/* SABRE */
      {0,7000},/* CUTLASS */
      {0,7000},/* WAKIZASHI */
      {0,7000},/* KHOPESH */
      {0,7000},/* TULWAR */
      {4000,8000},/* BROAD_SWORD */
      {0,7000},/* LONG_SWORD */
      {0,7000},/* SCIMITAR */
      {0,7000},/* NINJATO */
      {0,7000},/* KATANA */
      {0,7000},/* BASTARD_SWORD */
      {0,7000},/* GREAT_SCIMITAR */
      {0,7000},/* CLAYMORE */
      {0,7000},/* ESPADON */
      {0,7000},/* TWO_HANDED_SWORD */
      {0,7000},/* FLAMBERGE */
      {0,7000},/* NO_DACHI */
      {0,7000},/* EXECUTIOERS_SWORD */
      {0,7000},/* ZWEIHANDER */
      {4000,8000},/* BLADE_OF_CHAOS */
      {0,7000},/* DIAMOND_EDGE */
      {0,7000},/* DOKUBARI */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 40 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 50 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000} /*  */
    }
  },
  /* MONK */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {0,7000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {4000,7000},/* SHORT_BOW */
      {4000,7000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* LIGHT_XBOW */
      {0,7000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,0},/* 0 */
      {0,0},/* SHOVEL */
      {0,0},/* GNOMISH_SHOVEL */
      {0,0},/* DWARVEN_SHOVEL */
      {0,0},/* PICK */
      {0,0},/* ORCISH_PICK */
      {0,0},/* DWARVEN_PICK */
      {0,0},/* MATTOCK */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 10 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 20 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Hafted */
    {
      {0,0},/* 0 */
      {0,0},/* CLUB */
      {0,0},/* WHIP */
      {0,0},/* QUARTERSTAFF */
      {4000,8000},/* NUNCHAKU */
      {0,0},/* MACE */
      {0,0},/* BALL_AND_CHAIN */
      {0,0},/* JO_STAFF */
      {0,0},/* WAR_HAMMER */
      {0,0},/*  */
      {0,0},/* 10 */
      {4000,8000},/* THREE_PIECE_ROD */
      {0,0},/* MORNING_STAR */
      {0,0},/* FLAIL */
      {0,0},/* BO_STAFF */
      {0,0},/* LEAD_FILLED_MACE */
      {0,0},/* TETSUBO */
      {0,0},/*  */
      {0,0},/* TWO_HANDED_FLAIL */
      {0,0},/* GREAT_HAMMER */
      {0,0},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* GROND */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,0},/* 0 */
      {0,0},/* HATCHET */
      {0,0},/* SPEAR */
      {0,0},/* SICKLE */
      {0,0},/* AWL_PIKE */
      {0,0},/* TRIDENT */
      {0,0},/* FAUCHARD */
      {0,0},/* BROAD_SPEAR */
      {0,0},/* PIKE */
      {0,0},/* NAGINATA */
      {0,0},/* BEAKED_AXE */
      {0,0},/* BROAD_AXE */
      {0,0},/* LUCERNE_HAMMER */
      {0,0},/* GLAIVE */
      {0,0},/* LAJATANG */
      {0,0},/* HALBERD */
      {0,0},/* GUISARME */
      {0,0},/* SCYTHE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* LANCE */
      {0,0},/*  */
      {0,0},/* BATTLE_AXE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* GREAT_AXE */
      {0,0},/* TRIFURCATE_SPEAR */
      {0,0},/*  */
      {0,0},/* LOCHABER_AXE */
      {0,0},/* HEAVY_LANCE */
      {0,0},/* SCYTHE_OF_SLICING */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* DEATH_SCYTHE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Sword */
    {
      {0,0},/* 0 */
      {0,0},/* BROKEN_DAGGER */
      {0,0},/* BROKEN_SWORD */
      {0,0},/*  */
      {0,0},/* DAGGER */
      {0,0},/* MAIN_GAUCHE */
      {0,0},/* TANTO */
      {0,0},/* RAPIER */
      {0,0},/* SMALL_SWORD */
      {0,0},/* BASILLARD */
      {0,0},/* SHORT_SWORD */
      {0,0},/* SABRE */
      {0,0},/* CUTLASS */
      {0,0},/* WAKIZASHI */
      {0,0},/* KHOPESH */
      {0,0},/* TULWAR */
      {0,0},/* BROAD_SWORD */
      {0,0},/* LONG_SWORD */
      {0,0},/* SCIMITAR */
      {0,0},/* NINJATO */
      {0,0},/* KATANA */
      {0,0},/* BASTARD_SWORD */
      {0,0},/* GREAT_SCIMITAR */
      {0,0},/* CLAYMORE */
      {0,0},/* ESPADON */
      {0,0},/* TWO_HANDED_SWORD */
      {0,0},/* FLAMBERGE */
      {0,0},/* NO_DACHI */
      {0,0},/* EXECUTIOERS_SWORD */
      {0,0},/* ZWEIHANDER */
      {0,0},/* BLADE_OF_CHAOS */
      {0,0},/* DIAMOND_EDGE */
      {0,0},/* DOKUBARI */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    }
  },
  /* MINDCRAFTER */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {0,6000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {4000,8000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {0,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {4000,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {0,6000},/* SHORT_SWORD */
      {0,6000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,6000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* HIGH_MAGE */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {4000,6000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,4000},/* 0 */
      {0,4000},/* CLUB */
      {0,4000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,4000},/* MACE */
      {0,4000},/* BALL_AND_CHAIN */
      {0,4000},/* JO_STAFF */
      {0,4000},/* WAR_HAMMER */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/* THREE_PIECE_ROD */
      {0,4000},/* MORNING_STAR */
      {0,4000},/* FLAIL */
      {0,4000},/* BO_STAFF */
      {0,4000},/* LEAD_FILLED_MACE */
      {0,4000},/* TETSUBO */
      {0,4000},/*  */
      {0,4000},/* TWO_HANDED_FLAIL */
      {0,4000},/* GREAT_HAMMER */
      {0,4000},/* MACE_OF_DISRUPTION */
      {4000,8000},/* WIZSTAFF */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,0},/* TSURIZAO */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GROND */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,4000},/* 0 */
      {0,4000},/* HATCHET */
      {0,4000},/* SPEAR */
      {0,4000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,4000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LANCE */
      {0,4000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,4000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,4000},/* HEAVY_LANCE */
      {0,4000},/* SCYTHE_OF_SLICING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* DEATH_SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Sword */
    {
      {0,4000},/* 0 */
      {0,4000},/* BROKEN_DAGGER */
      {0,4000},/* BROKEN_SWORD */
      {0,4000},/*  */
      {4000,8000},/* DAGGER */
      {0,4000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,4000},/* RAPIER */
      {0,4000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {0,4000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,4000},/* CUTLASS */
      {0,4000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,4000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,4000},/* LONG_SWORD */
      {0,4000},/* SCIMITAR */
      {0,4000},/* NINJATO */
      {0,4000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,4000},/* EXECUTIOERS_SWORD */
      {0,4000},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    }
  },
  /* TOURIST */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {4000,8000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {0,8000},/* SHORT_BOW */
      {0,8000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* LIGHT_XBOW */
      {0,8000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,8000},/* 0 */
      {0,8000},/* SHOVEL */
      {0,8000},/* GNOMISH_SHOVEL */
      {0,8000},/* DWARVEN_SHOVEL */
      {0,8000},/* PICK */
      {0,8000},/* ORCISH_PICK */
      {0,8000},/* DWARVEN_PICK */
      {0,8000},/* MATTOCK */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Hafted */
    {
      {0,8000},/* 0 */
      {0,8000},/* CLUB */
      {0,8000},/* WHIP */
      {0,8000},/* QUARTERSTAFF */
      {0,8000},/* NUNCHAKU */
      {0,8000},/* MACE */
      {0,8000},/* BALL_AND_CHAIN */
      {0,8000},/* JO_STAFF */
      {0,8000},/* WAR_HAMMER */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/* THREE_PIECE_ROD */
      {0,8000},/* MORNING_STAR */
      {0,8000},/* FLAIL */
      {0,8000},/* BO_STAFF */
      {0,8000},/* LEAD_FILLED_MACE */
      {0,8000},/* TETSUBO */
      {0,8000},/*  */
      {0,8000},/* TWO_HANDED_FLAIL */
      {0,8000},/* GREAT_HAMMER */
      {0,8000},/* MACE_OF_DISRUPTION */
      {0,8000},/* WIZSTAFF */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* TSURIZAO */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* GROND */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,8000},/* 0 */
      {0,8000},/* HATCHET */
      {0,8000},/* SPEAR */
      {0,8000},/* SICKLE */
      {0,8000},/* AWL_PIKE */
      {0,8000},/* TRIDENT */
      {0,8000},/* FAUCHARD */
      {0,8000},/* BROAD_SPEAR */
      {0,8000},/* PIKE */
      {0,8000},/* NAGINATA */
      {0,8000},/* BEAKED_AXE */
      {0,8000},/* BROAD_AXE */
      {0,8000},/* LUCERNE_HAMMER */
      {0,8000},/* GLAIVE */
      {0,8000},/* LAJATANG */
      {0,8000},/* HALBERD */
      {0,8000},/* GUISARME */
      {0,8000},/* SCYTHE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* LANCE */
      {0,8000},/*  */
      {0,8000},/* BATTLE_AXE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* GREAT_AXE */
      {0,8000},/* TRIFURCATE_SPEAR */
      {0,8000},/*  */
      {0,8000},/* LOCHABER_AXE */
      {0,8000},/* HEAVY_LANCE */
      {0,8000},/* SCYTHE_OF_SLICING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* DEATH_SCYTHE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Sword */
    {
      {0,8000},/* 0 */
      {0,8000},/* BROKEN_DAGGER */
      {0,8000},/* BROKEN_SWORD */
      {0,8000},/*  */
      {0,8000},/* DAGGER */
      {0,8000},/* MAIN_GAUCHE */
      {0,8000},/* TANTO */
      {0,8000},/* RAPIER */
      {0,8000},/* SMALL_SWORD */
      {0,8000},/* BASILLARD */
      {0,8000},/* SHORT_SWORD */
      {0,8000},/* SABRE */
      {0,8000},/* CUTLASS */
      {0,8000},/* WAKIZASHI */
      {0,8000},/* KHOPESH */
      {0,8000},/* TULWAR */
      {0,8000},/* BROAD_SWORD */
      {0,8000},/* LONG_SWORD */
      {0,8000},/* SCIMITAR */
      {0,8000},/* NINJATO */
      {0,8000},/* KATANA */
      {0,8000},/* BASTARD_SWORD */
      {0,8000},/* GREAT_SCIMITAR */
      {0,8000},/* CLAYMORE */
      {0,8000},/* ESPADON */
      {0,8000},/* TWO_HANDED_SWORD */
      {0,8000},/* FLAMBERGE */
      {0,8000},/* NO_DACHI */
      {0,8000},/* EXECUTIOERS_SWORD */
      {0,8000},/* ZWEIHANDER */
      {0,8000},/* BLADE_OF_CHAOS */
      {0,8000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    }
  },
  /* IMITATOR */
  {
    /* Bow */
    {
      {0,6000},/* 0 */
      {0,6000},/*  */
      {4000,8000},/* SLING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,7000},/* 0 */
      {4000,7000},/* CLUB */
      {0,7000},/* WHIP */
      {0,7000},/* QUARTERSTAFF */
      {0,6000},/* NUNCHAKU */
      {4000,7000},/* MACE */
      {0,7000},/* BALL_AND_CHAIN */
      {0,7000},/* JO_STAFF */
      {0,7000},/* WAR_HAMMER */
      {0,7000},/*  */
      {0,7000},/* 10 */
      {0,7000},/* THREE_PIECE_ROD */
      {0,7000},/* MORNING_STAR */
      {0,7000},/* FLAIL */
      {0,7000},/* BO_STAFF */
      {0,7000},/* LEAD_FILLED_MACE */
      {0,7000},/* TETSUBO */
      {0,7000},/*  */
      {0,7000},/* TWO_HANDED_FLAIL */
      {0,7000},/* GREAT_HAMMER */
      {0,7000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 30 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,0},/* TSURIZAO */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* GROND */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,7000},/* 0 */
      {4000,7000},/* HATCHET */
      {0,7000},/* SPEAR */
      {0,7000},/* SICKLE */
      {0,7000},/* AWL_PIKE */
      {0,7000},/* TRIDENT */
      {0,7000},/* FAUCHARD */
      {0,7000},/* BROAD_SPEAR */
      {0,7000},/* PIKE */
      {0,7000},/* NAGINATA */
      {0,7000},/* BEAKED_AXE */
      {0,7000},/* BROAD_AXE */
      {0,7000},/* LUCERNE_HAMMER */
      {0,7000},/* GLAIVE */
      {0,7000},/* LAJATANG */
      {0,7000},/* HALBERD */
      {0,7000},/* GUISARME */
      {0,7000},/* SCYTHE */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* LANCE */
      {0,7000},/*  */
      {0,7000},/* BATTLE_AXE */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* GREAT_AXE */
      {0,7000},/* TRIFURCATE_SPEAR */
      {0,7000},/*  */
      {0,7000},/* LOCHABER_AXE */
      {0,7000},/* HEAVY_LANCE */
      {0,7000},/* SCYTHE_OF_SLICING */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 40 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* DEATH_SCYTHE */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000} /*  */
    },
    /* Sword */
    {
      {0,7000},/* 0 */
      {0,7000},/* BROKEN_DAGGER */
      {0,7000},/* BROKEN_SWORD */
      {0,7000},/*  */
      {4000,7000},/* DAGGER */
      {0,7000},/* MAIN_GAUCHE */
      {0,7000},/* TANTO */
      {0,7000},/* RAPIER */
      {4000,7000},/* SMALL_SWORD */
      {0,7000},/* BASILLARD */
      {4000,7000},/* SHORT_SWORD */
      {0,7000},/* SABRE */
      {0,7000},/* CUTLASS */
      {0,7000},/* WAKIZASHI */
      {0,7000},/* KHOPESH */
      {0,7000},/* TULWAR */
      {0,7000},/* BROAD_SWORD */
      {4000,7000},/* LONG_SWORD */
      {0,7000},/* SCIMITAR */
      {0,7000},/* NINJATO */
      {0,7000},/* KATANA */
      {0,7000},/* BASTARD_SWORD */
      {0,7000},/* GREAT_SCIMITAR */
      {0,7000},/* CLAYMORE */
      {0,7000},/* ESPADON */
      {0,7000},/* TWO_HANDED_SWORD */
      {0,7000},/* FLAMBERGE */
      {0,7000},/* NO_DACHI */
      {0,7000},/* EXECUTIOERS_SWORD */
      {0,7000},/* ZWEIHANDER */
      {0,7000},/* BLADE_OF_CHAOS */
      {0,7000},/* DIAMOND_EDGE */
      {0,7000},/* DOKUBARI */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 40 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 50 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000} /*  */
    }
  },
  /* BEASTMASTER */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {0,7000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {4000,8000},/* SHORT_BOW */
      {4000,8000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,8000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Hafted */
    {
      {0,8000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,7000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,7000},/* WAR_HAMMER */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,7000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,8000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,7000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,0},/* TSURIZAO */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* GROND */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,8000},/* 0 */
      {0,8000},/* HATCHET */
      {4000,8000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,7000},/* TRIDENT */
      {0,7000},/* FAUCHARD */
      {4000,8000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,7000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,8000},/*  */
      {0,8000},/*  */
      {4000,8000},/* LANCE */
      {0,8000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,8000},/* TRIFURCATE_SPEAR */
      {0,8000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,8000},/* HEAVY_LANCE */
      {0,7000},/* SCYTHE_OF_SLICING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* DEATH_SCYTHE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Sword */
    {
      {0,8000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,8000},/*  */
      {0,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {0,6000},/* SHORT_SWORD */
      {0,8000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,7000},/* TULWAR */
      {0,7000},/* BROAD_SWORD */
      {0,7000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,7000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,6000},/* DOKUBARI */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    }
  },
  /* SORCERER */
  {
    /* Bow */
    {
      {0,0},/* 0 */
      {0,0},/*  */
      {0,0},/* SLING */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 10 */
      {0,0},/*  */
      {0,0},/* SHORT_BOW */
      {0,0},/* LONG_BOW */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 20 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* LIGHT_XBOW */
      {0,0},/* HEAVY_XBOW */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Digging */
    {
      {0,0},/* 0 */
      {0,0},/* SHOVEL */
      {0,0},/* GNOMISH_SHOVEL */
      {0,0},/* DWARVEN_SHOVEL */
      {0,0},/* PICK */
      {0,0},/* ORCISH_PICK */
      {0,0},/* DWARVEN_PICK */
      {0,0},/* MATTOCK */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 10 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 20 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Hafted */
    {
      {0,0},/* 0 */
      {0,0},/* CLUB */
      {0,0},/* WHIP */
      {0,0},/* QUARTERSTAFF */
      {0,0},/* NUNCHAKU */
      {0,0},/* MACE */
      {0,0},/* BALL_AND_CHAIN */
      {0,0},/* JO_STAFF */
      {0,0},/* WAR_HAMMER */
      {0,0},/*  */
      {0,0},/* 10 */
      {0,0},/* THREE_PIECE_ROD */
      {0,0},/* MORNING_STAR */
      {0,0},/* FLAIL */
      {0,0},/* BO_STAFF */
      {0,0},/* LEAD_FILLED_MACE */
      {0,0},/* TETSUBO */
      {0,0},/*  */
      {0,0},/* TWO_HANDED_FLAIL */
      {0,0},/* GREAT_HAMMER */
      {0,0},/* MACE_OF_DISRUPTION */
      {8000,8000},/* WIZSTAFF */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* TSURIZAO */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* GROND */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,0},/* 0 */
      {0,0},/* HATCHET */
      {0,0},/* SPEAR */
      {0,0},/* SICKLE */
      {0,0},/* AWL_PIKE */
      {0,0},/* TRIDENT */
      {0,0},/* FAUCHARD */
      {0,0},/* BROAD_SPEAR */
      {0,0},/* PIKE */
      {0,0},/* NAGINATA */
      {0,0},/* BEAKED_AXE */
      {0,0},/* BROAD_AXE */
      {0,0},/* LUCERNE_HAMMER */
      {0,0},/* GLAIVE */
      {0,0},/* LAJATANG */
      {0,0},/* HALBERD */
      {0,0},/* GUISARME */
      {0,0},/* SCYTHE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* LANCE */
      {0,0},/*  */
      {0,0},/* BATTLE_AXE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* GREAT_AXE */
      {0,0},/* TRIFURCATE_SPEAR */
      {0,0},/*  */
      {0,0},/* LOCHABER_AXE */
      {0,0},/* HEAVY_LANCE */
      {0,0},/* SCYTHE_OF_SLICING */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* DEATH_SCYTHE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Sword */
    {
      {0,0},/* 0 */
      {0,0},/* BROKEN_DAGGER */
      {0,0},/* BROKEN_SWORD */
      {0,0},/*  */
      {0,0},/* DAGGER */
      {0,0},/* MAIN_GAUCHE */
      {0,0},/* TANTO */
      {0,0},/* RAPIER */
      {0,0},/* SMALL_SWORD */
      {0,0},/* BASILLARD */
      {0,0},/* SHORT_SWORD */
      {0,0},/* SABRE */
      {0,0},/* CUTLASS */
      {0,0},/* WAKIZASHI */
      {0,0},/* KHOPESH */
      {0,0},/* TULWAR */
      {0,0},/* BROAD_SWORD */
      {0,0},/* LONG_SWORD */
      {0,0},/* SCIMITAR */
      {0,0},/* NINJATO */
      {0,0},/* KATANA */
      {0,0},/* BASTARD_SWORD */
      {0,0},/* GREAT_SCIMITAR */
      {0,0},/* CLAYMORE */
      {0,0},/* ESPADON */
      {0,0},/* TWO_HANDED_SWORD */
      {0,0},/* FLAMBERGE */
      {0,0},/* NO_DACHI */
      {0,0},/* EXECUTIOERS_SWORD */
      {0,0},/* ZWEIHANDER */
      {0,0},/* BLADE_OF_CHAOS */
      {0,0},/* DIAMOND_EDGE */
      {0,0},/* DOKUBARI */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    }
  },
  /* ARCHER */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {4000,8000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {4000,8000},/* SHORT_BOW */
      {4000,8000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {4000,8000},/* LIGHT_XBOW */
      {4000,8000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {0,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {0,6000},/* SHORT_SWORD */
      {0,6000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,6000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,6000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* MAGIC_EATER */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {0,6000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,4000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,0},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,4000},/* BALL_AND_CHAIN */
      {0,4000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/* THREE_PIECE_ROD */
      {0,4000},/* MORNING_STAR */
      {0,4000},/* FLAIL */
      {0,4000},/* BO_STAFF */
      {0,4000},/* LEAD_FILLED_MACE */
      {0,4000},/* TETSUBO */
      {0,4000},/*  */
      {0,4000},/* TWO_HANDED_FLAIL */
      {0,4000},/* GREAT_HAMMER */
      {0,4000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,0},/* TSURIZAO */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GROND */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,4000},/* 0 */
      {0,4000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LANCE */
      {0,4000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,4000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,4000},/* HEAVY_LANCE */
      {0,4000},/* SCYTHE_OF_SLICING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* DEATH_SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Sword */
    {
      {0,4000},/* 0 */
      {0,4000},/* BROKEN_DAGGER */
      {0,4000},/* BROKEN_SWORD */
      {0,4000},/*  */
      {4000,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {4000,6000},/* SHORT_SWORD */
      {0,6000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,4000},/* EXECUTIOERS_SWORD */
      {0,4000},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    }
  },
  /* HARPER */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {0,4000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {0,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {4000,6000},/* SHORT_SWORD */
      {0,6000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,6000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* RED_MAGE */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {0,4000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,6000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 20 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,6000},/* CLUB */
      {0,6000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,6000},/* MACE */
      {0,6000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {0,6000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {0,6000},/* MORNING_STAR */
      {0,6000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {0,6000},/* LEAD_FILLED_MACE */
      {0,6000},/* TETSUBO */
      {0,6000},/*  */
      {0,6000},/* TWO_HANDED_FLAIL */
      {0,6000},/* GREAT_HAMMER */
      {0,6000},/* MACE_OF_DISRUPTION */
      {0,6000},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {0,6000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {0,6000},/* NAGINATA */
      {0,6000},/* BEAKED_AXE */
      {0,6000},/* BROAD_AXE */
      {0,6000},/* LUCERNE_HAMMER */
      {0,6000},/* GLAIVE */
      {0,6000},/* LAJATANG */
      {0,6000},/* HALBERD */
      {0,6000},/* GUISARME */
      {0,6000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* LANCE */
      {0,6000},/*  */
      {0,6000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* GREAT_AXE */
      {0,6000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,6000},/* LOCHABER_AXE */
      {0,6000},/* HEAVY_LANCE */
      {0,6000},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {0,6000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {4000,6000},/* SHORT_SWORD */
      {0,8000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,6000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
  /* SAMURAI */
  {
    /* Bow */
    {
      {4000,8000},/* 0 */
      {4000,8000},/*  */
      {0,6000},/* SLING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Digging */
    {
      {4000,8000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Hafted */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* CLUB */
      {4000,8000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {0,6000},/* NUNCHAKU */
      {4000,8000},/* MACE */
      {4000,8000},/* BALL_AND_CHAIN */
      {4000,8000},/* JO_STAFF */
      {4000,8000},/* WAR_HAMMER */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/* THREE_PIECE_ROD */
      {4000,8000},/* MORNING_STAR */
      {4000,8000},/* FLAIL */
      {4000,8000},/* BO_STAFF */
      {4000,8000},/* LEAD_FILLED_MACE */
      {4000,8000},/* TETSUBO */
      {4000,8000},/*  */
      {4000,8000},/* TWO_HANDED_FLAIL */
      {4000,8000},/* GREAT_HAMMER */
      {4000,8000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,0},/* TSURIZAO */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* GROND */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* HATCHET */
      {4000,8000},/* SPEAR */
      {4000,8000},/* SICKLE */
      {4000,8000},/* AWL_PIKE */
      {4000,8000},/* TRIDENT */
      {4000,8000},/* FAUCHARD */
      {4000,8000},/* BROAD_SPEAR */
      {4000,8000},/* PIKE */
      {4000,8000},/* NAGINATA */
      {4000,8000},/* BEAKED_AXE */
      {4000,8000},/* BROAD_AXE */
      {4000,8000},/* LUCERNE_HAMMER */
      {4000,8000},/* GLAIVE */
      {4000,8000},/* LAJATANG */
      {4000,8000},/* HALBERD */
      {4000,8000},/* GUISARME */
      {4000,8000},/* SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* LANCE */
      {4000,8000},/*  */
      {4000,8000},/* BATTLE_AXE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* GREAT_AXE */
      {4000,8000},/* TRIFURCATE_SPEAR */
      {4000,8000},/*  */
      {4000,8000},/* LOCHABER_AXE */
      {4000,8000},/* HEAVY_LANCE */
      {4000,8000},/* SCYTHE_OF_SLICING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* DEATH_SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Sword */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* BROKEN_DAGGER */
      {4000,8000},/* BROKEN_SWORD */
      {4000,8000},/*  */
      {4000,8000},/* DAGGER */
      {4000,8000},/* MAIN_GAUCHE */
      {4000,8000},/* TANTO */
      {4000,8000},/* RAPIER */
      {4000,8000},/* SMALL_SWORD */
      {4000,8000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {4000,8000},/* SABRE */
      {4000,8000},/* CUTLASS */
      {4000,8000},/* WAKIZASHI */
      {4000,8000},/* KHOPESH */
      {4000,8000},/* TULWAR */
      {4000,8000},/* BROAD_SWORD */
      {4000,8000},/* LONG_SWORD */
      {4000,8000},/* SCIMITAR */
      {4000,8000},/* NINJATO */
      {4000,8000},/* KATANA */
      {4000,8000},/* BASTARD_SWORD */
      {4000,8000},/* GREAT_SCIMITAR */
      {4000,8000},/* CLAYMORE */
      {4000,8000},/* ESPADON */
      {4000,8000},/* TWO_HANDED_SWORD */
      {4000,8000},/* FLAMBERGE */
      {4000,8000},/* NO_DACHI */
      {4000,8000},/* EXECUTIOERS_SWORD */
      {4000,8000},/* ZWEIHANDER */
      {4000,8000},/* BLADE_OF_CHAOS */
      {4000,8000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    }
  },
  /* ForceTrainer */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {0,6000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,0},/* 0 */
      {0,0},/* SHOVEL */
      {0,0},/* GNOMISH_SHOVEL */
      {0,0},/* DWARVEN_SHOVEL */
      {0,0},/* PICK */
      {0,0},/* ORCISH_PICK */
      {0,0},/* DWARVEN_PICK */
      {0,0},/* MATTOCK */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 10 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 20 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Hafted */
    {
      {0,0},/* 0 */
      {0,0},/* CLUB */
      {0,0},/* WHIP */
      {0,0},/* QUARTERSTAFF */
      {4000,8000},/* NUNCHAKU */
      {0,0},/* MACE */
      {0,0},/* BALL_AND_CHAIN */
      {0,7000},/* JO_STAFF */
      {0,0},/* WAR_HAMMER */
      {0,0},/*  */
      {0,0},/* 10 */
      {4000,8000},/* THREE_PIECE_ROD */
      {0,0},/* MORNING_STAR */
      {0,0},/* FLAIL */
      {0,7000},/* BO_STAFF */
      {0,0},/* LEAD_FILLED_MACE */
      {0,7000},/* TETSUBO */
      {0,0},/*  */
      {0,0},/* TWO_HANDED_FLAIL */
      {0,0},/* GREAT_HAMMER */
      {0,0},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* GROND */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,0},/* 0 */
      {0,0},/* HATCHET */
      {0,0},/* SPEAR */
      {0,0},/* SICKLE */
      {0,0},/* AWL_PIKE */
      {0,0},/* TRIDENT */
      {0,0},/* FAUCHARD */
      {0,0},/* BROAD_SPEAR */
      {0,0},/* PIKE */
      {0,0},/* NAGINATA */
      {0,0},/* BEAKED_AXE */
      {0,0},/* BROAD_AXE */
      {0,0},/* LUCERNE_HAMMER */
      {0,0},/* GLAIVE */
      {0,0},/* LAJATANG */
      {0,0},/* HALBERD */
      {0,0},/* GUISARME */
      {0,0},/* SCYTHE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* LANCE */
      {0,0},/*  */
      {0,0},/* BATTLE_AXE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* GREAT_AXE */
      {0,0},/* TRIFURCATE_SPEAR */
      {0,0},/*  */
      {0,0},/* LOCHABER_AXE */
      {0,0},/* HEAVY_LANCE */
      {0,0},/* SCYTHE_OF_SLICING */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* DEATH_SCYTHE */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Sword */
    {
      {0,0},/* 0 */
      {0,0},/* BROKEN_DAGGER */
      {0,0},/* BROKEN_SWORD */
      {0,0},/*  */
      {0,0},/* DAGGER */
      {0,0},/* MAIN_GAUCHE */
      {0,0},/* TANTO */
      {0,0},/* RAPIER */
      {0,0},/* SMALL_SWORD */
      {0,0},/* BASILLARD */
      {0,0},/* SHORT_SWORD */
      {0,0},/* SABRE */
      {0,0},/* CUTLASS */
      {0,0},/* WAKIZASHI */
      {0,0},/* KHOPESH */
      {0,0},/* TULWAR */
      {0,0},/* BROAD_SWORD */
      {0,0},/* LONG_SWORD */
      {0,0},/* SCIMITAR */
      {0,0},/* NINJATO */
      {0,0},/* KATANA */
      {0,0},/* BASTARD_SWORD */
      {0,0},/* GREAT_SCIMITAR */
      {0,0},/* CLAYMORE */
      {0,0},/* ESPADON */
      {0,0},/* TWO_HANDED_SWORD */
      {0,0},/* FLAMBERGE */
      {0,0},/* NO_DACHI */
      {0,0},/* EXECUTIOERS_SWORD */
      {0,0},/* ZWEIHANDER */
      {0,0},/* BLADE_OF_CHAOS */
      {0,0},/* DIAMOND_EDGE */
      {0,0},/* DOKUBARI */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    }
  },
  /* BLUE MAGE */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {4000,6000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LIGHT_XBOW */
      {0,4000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,4000},/* 0 */
      {0,4000},/* CLUB */
      {0,4000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {0,0},/* NUNCHAKU */
      {0,4000},/* MACE */
      {0,4000},/* BALL_AND_CHAIN */
      {0,4000},/* JO_STAFF */
      {0,4000},/* WAR_HAMMER */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/* THREE_PIECE_ROD */
      {0,4000},/* MORNING_STAR */
      {0,4000},/* FLAIL */
      {0,4000},/* BO_STAFF */
      {0,4000},/* LEAD_FILLED_MACE */
      {0,4000},/* TETSUBO */
      {0,4000},/*  */
      {0,4000},/* TWO_HANDED_FLAIL */
      {0,4000},/* GREAT_HAMMER */
      {0,4000},/* MACE_OF_DISRUPTION */
      {4000,8000},/* WIZSTAFF */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,0},/* TSURIZAO */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GROND */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,4000},/* 0 */
      {0,4000},/* HATCHET */
      {0,4000},/* SPEAR */
      {0,4000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,4000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LANCE */
      {0,4000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,4000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,4000},/* HEAVY_LANCE */
      {0,4000},/* SCYTHE_OF_SLICING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* DEATH_SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Sword */
    {
      {0,4000},/* 0 */
      {0,4000},/* BROKEN_DAGGER */
      {0,4000},/* BROKEN_SWORD */
      {0,4000},/*  */
      {4000,8000},/* DAGGER */
      {0,4000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,4000},/* RAPIER */
      {0,4000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {0,4000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,4000},/* CUTLASS */
      {0,4000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,4000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,4000},/* LONG_SWORD */
      {0,4000},/* SCIMITAR */
      {0,4000},/* NINJATO */
      {0,4000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,4000},/* EXECUTIOERS_SWORD */
      {0,4000},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    }
  },
  /* KIHEI */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {0,7000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {4000,8000},/* SHORT_BOW */
      {4000,8000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* LIGHT_XBOW */
      {0,7000},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,8000},/* 0 */
      {0,7000},/* SHOVEL */
      {0,7000},/* GNOMISH_SHOVEL */
      {0,7000},/* DWARVEN_SHOVEL */
      {0,7000},/* PICK */
      {0,7000},/* ORCISH_PICK */
      {0,7000},/* DWARVEN_PICK */
      {0,7000},/* MATTOCK */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Hafted */
    {
      {0,8000},/* 0 */
      {0,7000},/* CLUB */
      {0,7000},/* WHIP */
      {0,7000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,7000},/* MACE */
      {0,8000},/* BALL_AND_CHAIN */
      {0,7000},/* JO_STAFF */
      {0,8000},/* WAR_HAMMER */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,7000},/* THREE_PIECE_ROD */
      {0,7000},/* MORNING_STAR */
      {0,8000},/* FLAIL */
      {0,7000},/* BO_STAFF */
      {0,7000},/* LEAD_FILLED_MACE */
      {0,7000},/* TETSUBO */
      {0,8000},/*  */
      {0,7000},/* TWO_HANDED_FLAIL */
      {0,7000},/* GREAT_HAMMER */
      {0,8000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,0},/* TSURIZAO */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* GROND */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,8000},/* 0 */
      {0,8000},/* HATCHET */
      {4000,8000},/* SPEAR */
      {0,7000},/* SICKLE */
      {0,7000},/* AWL_PIKE */
      {0,8000},/* TRIDENT */
      {0,8000},/* FAUCHARD */
      {4000,8000},/* BROAD_SPEAR */
      {0,7000},/* PIKE */
      {0,7000},/* NAGINATA */
      {0,7000},/* BEAKED_AXE */
      {0,7000},/* BROAD_AXE */
      {0,7000},/* LUCERNE_HAMMER */
      {0,8000},/* GLAIVE */
      {0,7000},/* LAJATANG */
      {0,7000},/* HALBERD */
      {0,7000},/* GUISARME */
      {0,7000},/* SCYTHE */
      {0,8000},/*  */
      {0,8000},/*  */
      {4000,8000},/* LANCE */
      {0,8000},/*  */
      {0,7000},/* BATTLE_AXE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,7000},/* GREAT_AXE */
      {0,8000},/* TRIFURCATE_SPEAR */
      {0,8000},/*  */
      {0,7000},/* LOCHABER_AXE */
      {4000,8000},/* HEAVY_LANCE */
      {0,8000},/* SCYTHE_OF_SLICING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* DEATH_SCYTHE */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Sword */
    {
      {0,8000},/* 0 */
      {0,7000},/* BROKEN_DAGGER */
      {0,7000},/* BROKEN_SWORD */
      {0,8000},/*  */
      {0,7000},/* DAGGER */
      {0,7000},/* MAIN_GAUCHE */
      {0,7000},/* TANTO */
      {0,7000},/* RAPIER */
      {0,7000},/* SMALL_SWORD */
      {0,7000},/* BASILLARD */
      {0,7000},/* SHORT_SWORD */
      {0,8000},/* SABRE */
      {0,7000},/* CUTLASS */
      {0,7000},/* WAKIZASHI */
      {0,7000},/* KHOPESH */
      {0,8000},/* TULWAR */
      {0,8000},/* BROAD_SWORD */
      {0,8000},/* LONG_SWORD */
      {0,7000},/* SCIMITAR */
      {0,7000},/* NINJATO */
      {0,7000},/* KATANA */
      {0,7000},/* BASTARD_SWORD */
      {0,8000},/* GREAT_SCIMITAR */
      {0,7000},/* CLAYMORE */
      {0,7000},/* ESPADON */
      {0,7000},/* TWO_HANDED_SWORD */
      {0,7000},/* FLAMBERGE */
      {0,7000},/* NO_DACHI */
      {0,7000},/* EXECUTIOERS_SWORD */
      {0,7000},/* ZWEIHANDER */
      {0,7000},/* BLADE_OF_CHAOS */
      {0,7000},/* DIAMOND_EDGE */
      {0,7000},/* DOKUBARI */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    }
  },
  /* BERSERKER */
  {
    /* Bow */
    {
      {0,0},/* 0 */
      {0,0},/*  */
      {0,0},/* SLING */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 10 */
      {0,0},/*  */
      {0,0},/* SHORT_BOW */
      {0,0},/* LONG_BOW */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 20 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* LIGHT_XBOW */
      {0,0},/* HEAVY_XBOW */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 30 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 40 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 50 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/*  */
      {0,0},/* 60 */
      {0,0},/*  */
      {0,0},/*  */
      {0,0} /*  */
    },
    /* Digging */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* SHOVEL */
      {4000,8000},/* GNOMISH_SHOVEL */
      {4000,8000},/* DWARVEN_SHOVEL */
      {4000,8000},/* PICK */
      {4000,8000},/* ORCISH_PICK */
      {4000,8000},/* DWARVEN_PICK */
      {4000,8000},/* MATTOCK */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Hafted */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* CLUB */
      {4000,8000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {4000,8000},/* NUNCHAKU */
      {4000,8000},/* MACE */
      {4000,8000},/* BALL_AND_CHAIN */
      {4000,8000},/* JO_STAFF */
      {4000,8000},/* WAR_HAMMER */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/* THREE_PIECE_ROD */
      {4000,8000},/* MORNING_STAR */
      {4000,8000},/* FLAIL */
      {4000,8000},/* BO_STAFF */
      {4000,8000},/* LEAD_FILLED_MACE */
      {4000,8000},/* TETSUBO */
      {4000,8000},/*  */
      {4000,8000},/* TWO_HANDED_FLAIL */
      {4000,8000},/* GREAT_HAMMER */
      {4000,8000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,0},/* TSURIZAO */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* GROND */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* HATCHET */
      {4000,8000},/* SPEAR */
      {4000,8000},/* SICKLE */
      {4000,8000},/* AWL_PIKE */
      {4000,8000},/* TRIDENT */
      {4000,8000},/* FAUCHARD */
      {4000,8000},/* BROAD_SPEAR */
      {4000,8000},/* PIKE */
      {4000,8000},/* NAGINATA */
      {4000,8000},/* BEAKED_AXE */
      {4000,8000},/* BROAD_AXE */
      {4000,8000},/* LUCERNE_HAMMER */
      {4000,8000},/* GLAIVE */
      {4000,8000},/* LAJATANG */
      {4000,8000},/* HALBERD */
      {4000,8000},/* GUISARME */
      {4000,8000},/* SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* LANCE */
      {4000,8000},/*  */
      {4000,8000},/* BATTLE_AXE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* GREAT_AXE */
      {4000,8000},/* TRIFURCATE_SPEAR */
      {4000,8000},/*  */
      {4000,8000},/* LOCHABER_AXE */
      {4000,8000},/* HEAVY_LANCE */
      {4000,8000},/* SCYTHE_OF_SLICING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* DEATH_SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Sword */
    {
      {4000,8000},/* 0 */
      {4000,8000},/* BROKEN_DAGGER */
      {4000,8000},/* BROKEN_SWORD */
      {4000,8000},/*  */
      {4000,8000},/* DAGGER */
      {4000,8000},/* MAIN_GAUCHE */
      {4000,8000},/* TANTO */
      {4000,8000},/* RAPIER */
      {4000,8000},/* SMALL_SWORD */
      {4000,8000},/* BASILLARD */
      {4000,8000},/* SHORT_SWORD */
      {4000,8000},/* SABRE */
      {4000,8000},/* CUTLASS */
      {4000,8000},/* WAKIZASHI */
      {4000,8000},/* KHOPESH */
      {4000,8000},/* TULWAR */
      {4000,8000},/* BROAD_SWORD */
      {4000,8000},/* LONG_SWORD */
      {4000,8000},/* SCIMITAR */
      {4000,8000},/* NINJATO */
      {4000,8000},/* KATANA */
      {4000,8000},/* BASTARD_SWORD */
      {4000,8000},/* GREAT_SCIMITAR */
      {4000,8000},/* CLAYMORE */
      {4000,8000},/* ESPADON */
      {4000,8000},/* TWO_HANDED_SWORD */
      {4000,8000},/* FLAMBERGE */
      {4000,8000},/* NO_DACHI */
      {4000,8000},/* EXECUTIOERS_SWORD */
      {4000,8000},/* ZWEIHANDER */
      {4000,8000},/* BLADE_OF_CHAOS */
      {4000,8000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    }
  },
  /* KAJI */
  {
    /* Bow */
    {
      {4000,8000},/* 0 */
      {4000,8000},/*  */
      {0,6000},/* SLING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,7000},/* LIGHT_XBOW */
      {4000,7000},/* HEAVY_XBOW */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Digging */
    {
      {4000,8000},/* 0 */
      {0,6000},/* SHOVEL */
      {0,6000},/* GNOMISH_SHOVEL */
      {0,6000},/* DWARVEN_SHOVEL */
      {0,6000},/* PICK */
      {0,6000},/* ORCISH_PICK */
      {0,6000},/* DWARVEN_PICK */
      {0,6000},/* MATTOCK */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 20 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Hafted */
    {
      {4000,8000},/* 0 */
      {0,6000},/* CLUB */
      {0,4000},/* WHIP */
      {0,6000},/* QUARTERSTAFF */
      {0,6000},/* NUNCHAKU */
      {4000,7000},/* MACE */
      {4000,7000},/* BALL_AND_CHAIN */
      {0,6000},/* JO_STAFF */
      {4000,8000},/* WAR_HAMMER */
      {4000,8000},/*  */
      {4000,8000},/* 10 */
      {0,6000},/* THREE_PIECE_ROD */
      {4000,7000},/* MORNING_STAR */
      {4000,7000},/* FLAIL */
      {0,6000},/* BO_STAFF */
      {4000,7000},/* LEAD_FILLED_MACE */
      {4000,8000},/* TETSUBO */
      {4000,8000},/*  */
      {4000,8000},/* TWO_HANDED_FLAIL */
      {4000,8000},/* GREAT_HAMMER */
      {4000,8000},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 30 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,0},/* TSURIZAO */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* GROND */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {4000,8000},/* 0 */
      {0,6000},/* HATCHET */
      {0,6000},/* SPEAR */
      {0,6000},/* SICKLE */
      {0,6000},/* AWL_PIKE */
      {0,6000},/* TRIDENT */
      {0,6000},/* FAUCHARD */
      {4000,7000},/* BROAD_SPEAR */
      {0,6000},/* PIKE */
      {4000,7000},/* NAGINATA */
      {4000,8000},/* BEAKED_AXE */
      {4000,8000},/* BROAD_AXE */
      {4000,7000},/* LUCERNE_HAMMER */
      {4000,7000},/* GLAIVE */
      {4000,7000},/* LAJATANG */
      {4000,8000},/* HALBERD */
      {4000,8000},/* GUISARME */
      {4000,8000},/* SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,7000},/* LANCE */
      {4000,8000},/*  */
      {4000,8000},/* BATTLE_AXE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* GREAT_AXE */
      {4000,7000},/* TRIFURCATE_SPEAR */
      {4000,8000},/*  */
      {4000,8000},/* LOCHABER_AXE */
      {4000,7000},/* HEAVY_LANCE */
      {4000,8000},/* SCYTHE_OF_SLICING */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {0,8000},/* DEATH_SCYTHE */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    },
    /* Sword */
    {
      {4000,8000},/* 0 */
      {0,0},/* BROKEN_DAGGER */
      {0,0},/* BROKEN_SWORD */
      {4000,8000},/*  */
      {0,4000},/* DAGGER */
      {0,4000},/* MAIN_GAUCHE */
      {0,4000},/* TANTO */
      {0,4000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,6000},/* BASILLARD */
      {0,6000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,6000},/* CUTLASS */
      {0,4000},/* WAKIZASHI */
      {0,6000},/* KHOPESH */
      {0,6000},/* TULWAR */
      {0,6000},/* BROAD_SWORD */
      {0,6000},/* LONG_SWORD */
      {0,6000},/* SCIMITAR */
      {0,6000},/* NINJATO */
      {0,6000},/* KATANA */
      {0,6000},/* BASTARD_SWORD */
      {0,6000},/* GREAT_SCIMITAR */
      {0,6000},/* CLAYMORE */
      {0,6000},/* ESPADON */
      {0,6000},/* TWO_HANDED_SWORD */
      {0,6000},/* FLAMBERGE */
      {0,6000},/* NO_DACHI */
      {0,6000},/* EXECUTIOERS_SWORD */
      {0,6000},/* ZWEIHANDER */
      {0,6000},/* BLADE_OF_CHAOS */
      {0,6000},/* DIAMOND_EDGE */
      {0,6000},/* DOKUBARI */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 40 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 50 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000},/* 60 */
      {4000,8000},/*  */
      {4000,8000},/*  */
      {4000,8000} /*  */
    }
  },
  /* MIRROR MASTER */
  {
    /* Bow */
    {
      {0,4000},/* 0 */
      {0,4000},/*  */
      {4000,6000},/* SLING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,6000},/* SHORT_BOW */
      {0,6000},/* LONG_BOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,6000},/* LIGHT_XBOW */
      {0,6000},/* HEAVY_XBOW */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Digging */
    {
      {0,4000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 20 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Hafted */
    {
      {0,4000},/* 0 */
      {0,4000},/* CLUB */
      {0,4000},/* WHIP */
      {4000,8000},/* QUARTERSTAFF */
      {0,0},/* NUNCHAKU */
      {0,4000},/* MACE */
      {0,4000},/* BALL_AND_CHAIN */
      {0,4000},/* JO_STAFF */
      {0,4000},/* WAR_HAMMER */
      {0,4000},/*  */
      {0,4000},/* 10 */
      {0,4000},/* THREE_PIECE_ROD */
      {0,4000},/* MORNING_STAR */
      {0,4000},/* FLAIL */
      {0,4000},/* BO_STAFF */
      {0,4000},/* LEAD_FILLED_MACE */
      {0,4000},/* TETSUBO */
      {0,4000},/*  */
      {0,4000},/* TWO_HANDED_FLAIL */
      {0,4000},/* GREAT_HAMMER */
      {0,4000},/* MACE_OF_DISRUPTION */
      {4000,8000},/* WIZSTAFF */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 30 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,0},/* TSURIZAO */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GROND */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,4000},/* 0 */
      {0,4000},/* HATCHET */
      {0,4000},/* SPEAR */
      {0,4000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,4000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* LANCE */
      {0,4000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,4000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,4000},/* HEAVY_LANCE */
      {0,4000},/* SCYTHE_OF_SLICING */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* DEATH_SCYTHE */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    },
    /* Sword */
    {
      {0,4000},/* 0 */
      {0,4000},/* BROKEN_DAGGER */
      {0,4000},/* BROKEN_SWORD */
      {0,4000},/*  */
      {4000,8000},/* DAGGER */
      {0,4000},/* MAIN_GAUCHE */
      {0,6000},/* TANTO */
      {0,4000},/* RAPIER */
      {0,4000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {0,4000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,4000},/* CUTLASS */
      {0,4000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,4000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,4000},/* LONG_SWORD */
      {0,4000},/* SCIMITAR */
      {0,4000},/* NINJATO */
      {0,4000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,4000},/* EXECUTIOERS_SWORD */
      {0,4000},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 40 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 50 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000},/* 60 */
      {0,4000},/*  */
      {0,4000},/*  */
      {0,4000} /*  */
    }
  },
  /* NINJA */
  {
    /* Bow */
    {
      {0,8000},/* 0 */
      {0,8000},/*  */
      {0,7000},/* SLING */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 10 */
      {0,8000},/*  */
      {0,7000},/* SHORT_BOW */
      {0,4000},/* LONG_BOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 20 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,0},/* LIGHT_XBOW */
      {0,0},/* HEAVY_XBOW */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 30 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 40 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 50 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000},/* 60 */
      {0,8000},/*  */
      {0,8000},/*  */
      {0,8000} /*  */
    },
    /* Digging */
    {
      {0,7000},/* 0 */
      {0,4000},/* SHOVEL */
      {0,4000},/* GNOMISH_SHOVEL */
      {0,4000},/* DWARVEN_SHOVEL */
      {0,4000},/* PICK */
      {0,4000},/* ORCISH_PICK */
      {0,4000},/* DWARVEN_PICK */
      {0,4000},/* MATTOCK */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 10 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 20 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 30 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 40 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 50 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000},/* 60 */
      {0,7000},/*  */
      {0,7000},/*  */
      {0,7000} /*  */
    },
    /* Hafted */
    {
      {0,6000},/* 0 */
      {0,4000},/* CLUB */
      {0,4000},/* WHIP */
      {0,4000},/* QUARTERSTAFF */
      {0,4000},/* NUNCHAKU */
      {0,4000},/* MACE */
      {0,4000},/* BALL_AND_CHAIN */
      {0,4000},/* JO_STAFF */
      {0,4000},/* WAR_HAMMER */
      {0,6000},/*  */
      {0,6000},/* 10 */
      {0,4000},/* THREE_PIECE_ROD */
      {0,4000},/* MORNING_STAR */
      {0,4000},/* FLAIL */
      {0,4000},/* BO_STAFF */
      {0,4000},/* LEAD_FILLED_MACE */
      {0,4000},/* TETSUBO */
      {0,6000},/*  */
      {0,4000},/* TWO_HANDED_FLAIL */
      {0,0},/* GREAT_HAMMER */
      {0,0},/* MACE_OF_DISRUPTION */
      {0,0},/* WIZSTAFF */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 30 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* TSURIZAO */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* GROND */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,8000} /* NAMAKE */
    },
    /* Polearm */
    {
      {0,6000},/* 0 */
      {0,4000},/* HATCHET */
      {0,4000},/* SPEAR */
      {0,4000},/* SICKLE */
      {0,4000},/* AWL_PIKE */
      {0,4000},/* TRIDENT */
      {0,4000},/* FAUCHARD */
      {0,4000},/* BROAD_SPEAR */
      {0,4000},/* PIKE */
      {0,4000},/* NAGINATA */
      {0,4000},/* BEAKED_AXE */
      {0,4000},/* BROAD_AXE */
      {0,4000},/* LUCERNE_HAMMER */
      {0,4000},/* GLAIVE */
      {0,4000},/* LAJATANG */
      {0,4000},/* HALBERD */
      {0,4000},/* GUISARME */
      {0,4000},/* SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* LANCE */
      {0,6000},/*  */
      {0,4000},/* BATTLE_AXE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* GREAT_AXE */
      {0,4000},/* TRIFURCATE_SPEAR */
      {0,6000},/*  */
      {0,4000},/* LOCHABER_AXE */
      {0,0},/* HEAVY_LANCE */
      {0,0},/* SCYTHE_OF_SLICING */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,0},/* DEATH_SCYTHE */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    },
    /* Sword */
    {
      {0,6000},/* 0 */
      {0,6000},/* BROKEN_DAGGER */
      {0,6000},/* BROKEN_SWORD */
      {0,6000},/*  */
      {4000,8000},/* DAGGER */
      {0,6000},/* MAIN_GAUCHE */
      {4000,8000},/* TANTO */
      {0,6000},/* RAPIER */
      {0,6000},/* SMALL_SWORD */
      {0,4000},/* BASILLARD */
      {0,6000},/* SHORT_SWORD */
      {0,4000},/* SABRE */
      {0,4000},/* CUTLASS */
      {0,6000},/* WAKIZASHI */
      {0,4000},/* KHOPESH */
      {0,4000},/* TULWAR */
      {0,4000},/* BROAD_SWORD */
      {0,4000},/* LONG_SWORD */
      {0,4000},/* SCIMITAR */
      {4000,8000},/* NINJATO */
      {0,4000},/* KATANA */
      {0,4000},/* BASTARD_SWORD */
      {0,4000},/* GREAT_SCIMITAR */
      {0,4000},/* CLAYMORE */
      {0,4000},/* ESPADON */
      {0,4000},/* TWO_HANDED_SWORD */
      {0,4000},/* FLAMBERGE */
      {0,4000},/* NO_DACHI */
      {0,0},/* EXECUTIOERS_SWORD */
      {0,0},/* ZWEIHANDER */
      {0,4000},/* BLADE_OF_CHAOS */
      {0,4000},/* DIAMOND_EDGE */
      {0,8000},/* DOKUBARI */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 40 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 50 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000},/* 60 */
      {0,6000},/*  */
      {0,6000},/*  */
      {0,6000} /*  */
    }
  },
};

s16b skill_exp_settei[MAX_CLASS][10][2] =
{
  /* WARRIOR */
  {
    {0,8000},/* SUDE */
    {4000,8000},/* NITOURYU */
    {0,5000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* MAGE */
  {
    {0,4000},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* PRIEST */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,500},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* ROGUE */
  {
    {0,8000},/* SUDE */
    {4000,8000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* RANGER */
  {
    {0,6000},/* SUDE */
    {0,6000},/* NITOURYU */
    {0,5000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* PALADIN */
  {
    {0,4000},/* SUDE */
    {0,7000},/* NITOURYU */
    {0,5000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* WARRIOR_MAGE */
  {
    {0,4000},/* SUDE */
    {0,6000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* CHAOS_WARRIOR */
  {
    {0,7000},/* SUDE */
    {0,7000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* MONK */
  {
    {4000,8000},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
  },
  /* MINDCRAFTER */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,500},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* HIGE_MAGE */
  {
    {0,4000},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* TOURIST */
  {
    {0,8000},/* SUDE */
    {0,8000},/* NITOURYU */
    {0,5000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* IMITATOR */
  {
    {0,7000},/* SUDE */
    {0,7000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* BEASTMASTER */
  {
    {0,6000},/* SUDE */
    {0,6000},/* NITOURYU */
    {500,8000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* SORCERER */
  {
    {0,0},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* ARCHER */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* MAGIC_EATER */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* HARPER */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,500},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* RED_MAGE */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* SAMURAI */
  {
    {0,4000},/* SUDE */
    {4000,8000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* ForceTrainer */
  {
    {4000,8000},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
  },
  /* BLUE MAGE */
  {
    {0,4000},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* KIHEI */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {500,8000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* BERSERKER */
  {
    {4000,8000},/* SUDE */
    {4000,8000},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* KAJI */
  {
    {0,4000},/* SUDE */
    {0,4000},/* NITOURYU */
    {0,2000},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* MIRROR MASTER */
  {
    {0,4000},/* SUDE */
    {0,0},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
  /* NINJA */
  {
    {0,7000},/* SUDE */
    {4000,8000},/* NITOURYU */
    {0,0},/* RIDING */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0},/*  */
    {0,0} /*  */
  },
};

byte conv_terrain2feat[MAX_WILDERNESS] =
{
  FEAT_PERM_EXTRA,
  FEAT_TOWN,
  FEAT_DEEP_WATER,
  FEAT_SHAL_WATER,
  FEAT_SWAMP,
  FEAT_DIRT,
  FEAT_GRASS,
  FEAT_TREES,
  FEAT_DIRT,
  FEAT_SHAL_LAVA,
  FEAT_DEEP_LAVA,
  FEAT_MOUNTAIN
};

#ifdef JP
cptr silly_attacks[MAX_SILLY_ATTACK] =
{
  "˾ؤ򤫤줿",
  "ʤβ3äƥȸä",
  "ˤФޤ蘆줿",
  "˷ʤᤵ줿",
  "˥ϥʥФ줿",
  "˥ǹ⤵줿",
  "ʤˤ«ǤФ",
  "ʤǥݡ󥰤򤷤",
  "˥١줿",
  "ˡֿιȯű줿",
  "˥áޤ줿",
  "ˤĤäߤ줿",
  "Ϥʤ٤ä",
  "˴ˤ餯񤭤򤵤줿",
  "˼ڶֺѤ򤻤ޤ줿",
  "˥Ȥ᤯줿",
  "Ϥʤμä",
  "夬줿",
  "鰦ι򤵤줿",
  "Ϥʤ500ߤǸۤä",
  "Ϥʤ100̩ˤĤǮä",
  "˥㡼Ĥ",
  "Ϥʤ˵Ĥ",
  "Ϥʤݥꥴ󲽤",
  "˾줿",
  "ϥƥޤμʸ򾧤",
  "ϤʤΥѥ֥å",
  "ϥ饤ɰư",
  "ϾζޥɤϤ˼Ԥ",
  "ϳȻưˤȯͤ",
  "ϥǥ顼ˡ򤷤",
  "˥饤åǹ⤵줿",
  "󽵴ְǥӥǥͤ˸ʤȻ̼򤫤줿",
  "ϥѥץƤ򾧤",
  "ϥѡȥ饮饯ƥޥʥä",
  "ˤ㤬߾åǥϥ줿",
  "˥åȥȥ꡼ॢå򤫤줿",
  "ϤʤĸǤ򤫤ơ123áפȶ",
  "ϡ֤ʤФФФפȤäƶФ",
  "֤󡢥롼٥󥹤γפȸäŤܤĤ",
  "ϸäѶܡ仿桪ס",
};
#else

/* Weird melee attack types when hallucinating */
cptr silly_attacks[MAX_SILLY_ATTACK] =
{
	"smothers",
	"hugs",
	"humiliates",
	"whips",
	"kisses",

	"disgusts",
	"pees all over",
	"passes the gas on",
	"makes obscene gestures at",
	"licks",

	"stomps on",
	"swallows",
	"drools on",
	"misses",
	"shrinks",

	"emasculates",
	"evaporates",
	"solidifies",
	"digitizes",
	"insta-kills",

	"massacres",
	"slaughters",
	"drugs",
	"psychoanalyzes",
	"deconstructs",

	"falsifies",
	"disbelieves",
	"molests",
	"pusupusu"
};
#endif






