package com.interpress_project.modernshare.client.controller.delegate;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;

import org.apache.commons.io.FileUtils;
import org.tigris.subversion.svnclientadapter.ISVNInfo;
import org.tigris.subversion.svnclientadapter.ISVNStatus;
import org.tigris.subversion.svnclientadapter.SVNClientException;
import org.tigris.subversion.svnclientadapter.SVNInfoUnversioned;
import org.tigris.subversion.svnclientadapter.SVNNodeKind;
import org.tigris.subversion.svnclientadapter.SVNRevision;
import org.tigris.subversion.svnclientadapter.SVNStatusKind;
import org.tigris.subversion.svnclientadapter.SVNUrl;

import com.interpress_project.modernshare.client.controller.delegate.exceptions.DeleteFailException;
import com.interpress_project.modernshare.client.controller.delegate.exceptions.InternalFailException;
import com.interpress_project.modernshare.client.controller.delegate.exceptions.NoNeedSyncException;
import com.interpress_project.modernshare.client.controller.delegate.exceptions.RenameFailException;
import com.interpress_project.modernshare.client.controller.model.ISVNFileItem;
import com.interpress_project.modernshare.client.controller.model.SVNFile;
import com.interpress_project.modernshare.client.controller.model.SVNFolder;
import com.interpress_project.modernshare.client.events.LogEvent;
import com.interpress_project.modernshare.client.events.LogEventType;

/**
 * Represents a singleton.
 * @stereotype Singleton factory
 */
public class LocalSVNManager extends ISVNManager {
	/**
	 * Holds singleton instance
	 */
	private static LocalSVNManager instance;

	private final SVNFolderUtil futil = SVNFolderUtil.getInstance();

	/**
	 * prevents instantiation
	 */
	private LocalSVNManager() {
	}

	/**
	 * Returns the singleton instance.
	 @return	the singleton instance
	 */
	static public LocalSVNManager getInstance() {
		if (instance == null) {
			instance = new LocalSVNManager();
		}
		return instance;
	}

	/**
	 * initialCheckoutFolder
	 * @param rootpath
	 * @param tagName
	 * @throws InternalFailException
	 */
	public void initialCheckoutFolder(File rootpath, String tagName) throws InternalFailException {
		SVNUrl u = createSVNUrl(accounturl + "/" + tagName);

		try {
			svnClient.mkdir(u, "Initial Import" + new Date().toString());
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(rootpath + "ɑ΂鏉Imports܂B", ex);
		}

		try {
			svnClient.checkout(u, rootpath, SVNRevision.HEAD, false);
			addIgnoredFilter(rootpath);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(rootpath + "ɑ΂鏉checkouts܂B", ex);
		}
	}

	/**
	 * initialRegisterFiles
	 * @param rootpath
	 * @return
	 * @throws InternalFailException
	 */
	public File[] initialRegisterFiles(File rootpath) throws InternalFailException {
		ISVNStatus[] status = null;

		ArrayList<File> list = futil.getDirListInTarget(rootpath);
		for (int i = 0; i < list.size(); i++) {
			File file = list.get(i);
			try {
				svnClient.addDirectory(file, false);
				addIgnoredFilter(file);
			}
			catch (SVNClientException ex) {
				throw new InternalFailException("fBNg" + rootpath.getAbsolutePath() + "̒ǉɎs܂B(addDirectory)", ex);
			}
		}

		try {
			status = this.checkDirectoryStatus(rootpath, true);
		}
		catch (NoNeedSyncException ex) {
		}

		/**
		 * LŎ擾ς݂statuŝ܂ܗpāAt@Co^B
		 */
		for (int i = 0; i < status.length; i++) {
			ISVNStatus st = status[i];
			/**
			 * VKǉꂽt@C - SVNɂo[WǗɂȂB 
			 * ? file_or_dir
			 */
			if (SVNStatusKind.UNVERSIONED == st.getTextStatus() && (st.getFile().isDirectory() == false)) {
				this.registerFile(st.getFile());
			}
		}
		/**
		 * status񂩂폜ꂽFileXg쐬
		 */
		list = new ArrayList<File>();
		for (int i = 0; i < status.length; i++) {
			if (status[i].getTextStatus() == SVNStatusKind.DELETED) {
				continue;
			}
			list.add(status[i].getFile());
		}
		return (File[]) list.toArray(new File[list.size()]);
	}

	/**
	 * registerFolder
	 * w肳ꂽdirz̃fBNgo^B̍svntH_̍폜ƃtB^̐ݒsB
	 * svñtB^́AsvnǗɂtH_ɂ̂ݐݒ肷邱ƂłBāA܂
	 * fBNgaddɁAtB^ݒ肷邱ƁB
	 * @param dir
	 * @throws InternalFailException
	 */
	private void registerFolder(File dir) throws InternalFailException {
		if (!dir.isDirectory()) {
			return;
		}
		try {
			futil.deleteSVNFolders(dir);
		}
		catch (DeleteFailException ex) {
			throw new InternalFailException("fBNg" + dir.getAbsolutePath() + "svnt@C̍폜Ɏs܂B", ex);
		}

		ArrayList<File> list = futil.getDirListInTarget(dir);

		try {
			svnClient.addDirectory(dir, false);
			addIgnoredFilter(dir);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException("fBNg" + dir.getAbsolutePath() + "̒ǉɎs܂B(addDirectory)", ex);
		}

		for (int i = 0; i < list.size(); i++) {
			File file = list.get(i);
			try {
				svnClient.addDirectory(file, false);
				addIgnoredFilter(file);
			}
			catch (SVNClientException ex) {
				throw new InternalFailException("fBNg" + dir.getAbsolutePath() + "̒ǉɎs܂B(addDirectory)", ex);
			}
		}
	}

	/**
	 * registerFile
	 * PɃt@CaddB
	 * @param file
	 * @throws InternalFailException
	 */
	private void registerFile(File file) throws InternalFailException {
		if (file.isFile()) {
			try {
				svnClient.addFile(file);
			}
			catch (SVNClientException ex) {
				throw new InternalFailException("t@C" + file.getAbsolutePath() + "̒ǉɎs܂B(addFile)", ex);
			}
		}
	}

	/**
	 * unregisterFile
	 * @param file
	 * @throws InternalFailException
	 * @throws DeleteFailException 
	 */
	private void unregisterFile(File file) throws InternalFailException, DeleteFailException {
		if (file.isDirectory()) {
			if (svnClient.isAdminDirectory(file.getAbsolutePath())) {
				throw new DeleteFailException(file.getAbsolutePath() + "͊ǗfBNgłB폜邱Ƃ͂ł܂B");
			}
		}
		try {
			svnClient.remove(new File[] { file }, true);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(file.getAbsolutePath() + "̍폜Ɏs܂B(remove)", ex);
		}
	}

	/**
	 * checkoutFiles
	 * @param rootfile
	 * @param svnurl
	 * @throws InternalFailException
	 */
	public void checkoutFiles(File rootfile, String svnurl) throws InternalFailException {
		SVNUrl u = createSVNUrl(svnurl);
		try {
			svnClient.checkout(u, rootfile, SVNRevision.HEAD, true);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(rootfile.getAbsolutePath() + "ɑ΂`FbNAEgɎs܂B(" + svnurl + ")", ex);
		}
		addIgnoredFilterToDirectory(rootfile);
	}

	/**
	 * addIgnoredFilterToDirectory 
	 * @param rootfile
	 * @throws InternalFailException
	 */
	public void addIgnoredFilterToDirectory(File rootpath) throws InternalFailException {
		ArrayList<File> list = futil.getDirListInTarget(rootpath);
		addIgnoredFilter(rootpath);
		for (int i = 0; i < list.size(); i++) {
			addIgnoredFilter(list.get(i));
		}
	}

	/**
	 * getRevisionInfo
	 * @param fileName
	 * @return
	 * @throws InternalFailException
	 */
	public ArrayList<ISVNFileItem> getRevisionInfo(String fileName) throws InternalFailException {
		RemoteSVNManager rsvnmgr = RemoteSVNManager.getInstance();
		ISVNInfo info = null;
		try {
			info = svnClient.getInfo(new File(fileName));
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(fileName + "ɑ΂擾Ɏs܂B(getInfo)", ex);
		}
		return rsvnmgr.getRevisionInfo(info.getUrlString());
	}

	/**
	 * getSVNItems
	 * @param file
	 * @return
	 * @throws InternalFailException
	 */
	public ISVNFileItem[] getSVNItems(String path) throws InternalFailException {
		ArrayList<ISVNFileItem> arrItems = new ArrayList<ISVNFileItem>();
		ISVNInfo entry = null;
		File file = new File(path);
		String abspath = file.getAbsolutePath();

		/**
		 * Check if the file exist.
		 */
		if (!file.exists()) {
			throw new InternalFailException("[J\[X" + abspath + "́A[JPCɑ݂Ă܂B");
		}

		try {
			entry = svnClient.getInfoFromWorkingCopy(file);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException("[J\[X" + abspath + "̏擾Ɏs܂B(getInfoFromWorkingCopy)", ex);
		}
		if (entry instanceof SVNInfoUnversioned) {
			throw new InternalFailException("[J\[X" + abspath + "͎񓯊҂ԂΏۊOt@CłB");
		}
		/**
		 * [Us\[XNbN`FbN
		 */
		if (entry.getNodeKind() == SVNNodeKind.DIR && file.isFile()) {
			throw new InternalFailException("[J\[X" + abspath + "͕s\[XłB̓ŏC܂B");
		}
		if (entry.getNodeKind() == SVNNodeKind.FILE && file.isDirectory()) {
			throw new InternalFailException("[J\[X" + abspath + "͕s\[XłB̓ŏC܂B");
		}

		ISVNFileItem item = null;
		try {
			String name = entry.getFile().getName();
			long fsize = entry.getFile().length();
			String author = entry.getLastCommitAuthor();
			Date lcdate = entry.getLastChangedDate();
			long revision = entry.getRevision().getNumber();
			long lcrevision = entry.getLastChangedRevision().getNumber();
			String url = entry.getUrlString();
			String uuid = entry.getUuid();

			if (file.isDirectory()) {
				item = new SVNFolder(name, fsize, author, lcdate, revision, lcrevision, url, uuid);
			}
			else {
				item = new SVNFile(name, fsize, author, lcdate, revision, lcrevision, url, uuid);
			}
			arrItems.add(item);
		}
		catch (Exception ex) {
			throw new InternalFailException("[J\[X" + abspath + "̏擾Ɏs܂B", ex);
		}
		return arrItems.toArray(new ISVNFileItem[arrItems.size()]);
	}

	/**
	 * cleanupLocalFiles - 
	 * @param path
	 * @throws InternalFailException
	 */
	public void cleanupFiles(File path) throws InternalFailException {
		sb.getLogger().debug("Enter cleanupFiles()");
		try {
			svnClient.cleanup(path);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException("[J\[X" + path.getAbsolutePath() + "ɑ΂CleanupɎs܂B", ex);
		}
	}

	/**
	 * getStatus
	 * @param path
	 * @param bCheckRemote
	 * @return
	 * @throws InternalFailException
	 */
	private ISVNStatus[] getStatus(File path, boolean bCheckRemote) throws InternalFailException {
		sb.getLogger().debug("Enter getStatus()");

		ISVNStatus[] status = null;
		try {
			status = svnClient.getStatus(path, true, false, bCheckRemote);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException("[J\[X" + path.getAbsolutePath() + "̕ύXԎ擾Ɏs܂B(getStatus)", ex);
		}
		return status;
	}

	/**
	 * checkDirectoryStatus 
	 * @param path
	 * @param bCheckRemote
	 * @return
	 * @throws InternalFailException
	 * @throws NoNeedSyncException
	 */
	private ISVNStatus[] checkDirectoryStatus(File path, boolean bCheckRemote) throws InternalFailException,
	    NoNeedSyncException {
		sb.getLogger().debug("Enter checkDirectoryStatus()");

		ArrayList<ISVNStatus> list = new ArrayList<ISVNStatus>();
		ISVNStatus[] status = this.getStatus(path, bCheckRemote);

		// root path ɂĂ͏Ȃ̂Ŏ菜
		// ܂NONE		
		for (int i = 0; i < status.length; i++) {
			if (status[i].getFile().getAbsolutePath().equalsIgnoreCase(path.getAbsolutePath())) {
				continue;
			}
			if (status[i].getTextStatus() == SVNStatusKind.NONE) {
				continue;
			}
			list.add(status[i]);
		}
		status = (ISVNStatus[]) list.toArray(new ISVNStatus[list.size()]);
		if (status.length == 0) { // ύXꂽt@CoȂꍇ
			throw new NoNeedSyncException();
		}

		// oꂽASătH_Ń[gύXNONEANORMALł΂͖łB
		boolean bNoNeedUpdate = true;
		for (int i = 0; i < status.length; i++) {
			if (status[i].getNodeKind() == SVNNodeKind.DIR && status[i].getTextStatus() == SVNStatusKind.NORMAL
			    && status[i].getRepositoryTextStatus() == SVNStatusKind.NONE) {
				continue;
			}
			bNoNeedUpdate = false;
			break;
		}
		if (bNoNeedUpdate) {
			throw new NoNeedSyncException();
		}
		return status;
	}

	/**
	 * syncLocalFilesWithRemote - [gt@CŃ[J𓯊B 
	 * @param path
	 * @throws InternalFailException 
	 */
	public void syncLocalFilesWithRemote(File path) throws InternalFailException {
		ISVNStatus[] status = null;
		/*
		 *  t@CVXeύX\[XoB
		 *  ܂̓T[oւ̖₢킹{čŐVXe[^X擾Bʏ͂Ŗ͂ȂA
		 *  OBSTRUCTEDԂꍇɂ͎擾G[͂B̏ꍇɂ͍ēxA
		 *  T[oւ̖₢킹Ŏ擾sB
		 */
		try {
			status = this.checkDirectoryStatus(path, true);
		}
		catch (InternalFailException ex) {
			String msg = path + "̕ύXԂ̎擾Ɏs܂B[JɒuꂽpčĎs܂B(syncLocalFilesWithRemote)";
			logEventSource.fireEvent(new LogEvent(this, msg, LogEventType.INFO));

			try {
				status = this.checkDirectoryStatus(path, false);
			}
			catch (NoNeedSyncException e) {
				return;
			}
		}
		catch (NoNeedSyncException e) {
			return;
		}

		/**
		 * ~file_or_dir̕ - |WgƍƃRs[ŎނقȂĂꍇɂ́A
		 * [gt@CŃ[Jt@CŏCB
		 */
		logEventSource.fireEvent(new LogEvent(this, path + "ɂsԂ̃\[XoĂ܂...", LogEventType.INFO));
		for (int i = 0; i < status.length; i++) {
			ISVNStatus st = status[i];
			// ~file_or_dir
			if (SVNStatusKind.OBSTRUCTED == st.getTextStatus()) {
				File localfile = st.getFile();
				try {
					logEventSource.fireEvent(new LogEvent(this, localfile.getAbsolutePath()
					    + "͕sԂ̃\[XłBf[^Z^Ɋi[ς݂̃\[XŏCs܂B", LogEventType.INFO));
					// Ƀ[Jɂt@C/tH_.HotShot_bkupgqđҔĂB
					futil.renameBkupFile(localfile);
				}
				catch (RenameFailException ex) {
					logEventSource.fireEvent(new LogEvent(this, ex.getMessage(), LogEventType.ERROR, ex));
					continue;
				}
			}
		}

		/**
		 *  OBSTRUCTED\[XCStatusēx擾B
		 */
		try {
			status = this.checkDirectoryStatus(path, true);
		}
		catch (NoNeedSyncException ex) {
			return;
		}

		/**
		 * ^[QbgtH_zɑ΂W
		 */
		for (int i = 0; i < status.length; i++) {
			ISVNStatus st = status[i];
			/**
			 * RtNg - ł̓[gt@Cpă[Jt@CCB
			 */
			if (SVNStatusKind.CONFLICTED == st.getTextStatus()) {
				logEventSource.fireEvent(new LogEvent(this, st.getFile().getAbsolutePath()
				    + "̓f[^Z^ƃRtNgԂɂȂĂ܂Bf[^Z^Ɋi[ꂽt@CŃ[Jt@CXV܂B", LogEventType.INFO));
				try {
					futil.renameBkupFile(st.getFile());
				}
				catch (RenameFailException ex) {
					logEventSource.fireEvent(new LogEvent(this, ex.getMessage(), LogEventType.ERROR, ex));
				}
				continue;
			}
			/**
			 * 폜ꂽt@C- t@C̓o[WǗɂ邪AsSȏԂłB
			 * SVNOOSR}hɂ폜Ă܂B
			 * ! file_or_dir
			 */
			if (SVNStatusKind.MISSING == st.getTextStatus()) {
				// svn update will recover files.
				logEventSource.fireEvent(new LogEvent(this, st.getFile().getAbsolutePath()
				    + "͍폜Ă܂Bf[^Z^Ɋi[ꂽt@Cŕ܂B", LogEventType.INFO));
				continue;
			}
			/**
			 * VKǉꂽt@C - SVNɂo[WǗɂȂB 
			 * ? file_or_dir
			 */
			if (SVNStatusKind.UNVERSIONED == st.getTextStatus() || SVNStatusKind.MODIFIED == st.getTextStatus()) {
				logEventSource.fireEvent(new LogEvent(this, st.getFile().getAbsolutePath()
				    + "̓[JɒǉĂ܂Bf[^Z^Ɠ邽߂ɑҔ܂B", LogEventType.INFO));
				try {
					futil.renameBkupFile(st.getFile());
				}
				catch (RenameFailException ex) {
					logEventSource.fireEvent(new LogEvent(this, ex.getMessage(), LogEventType.ERROR, ex));
				}
				continue;
			}
		}
		/**
		 * [gŃAbvf[gOcleanupĂƁB
		 * OBSTRUCTȂǁAzbgVbgtH_Lock|Ăꍇ邽
		 */
		cleanupFiles(path);

		/**
		 * [gŃ[JAbvf[g
		 */
		remoteUpdate(path);
	}

	/**
	 * remoteUpdate
	 * @param localfile
	 * @throws InternalFailException 
	 */
	private void remoteUpdate(File localfile) throws InternalFailException {
		try {
			svnClient.update(localfile, SVNRevision.HEAD, true);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException("[J\[X" + localfile.getAbsolutePath() + "̍XVɎs܂B(update)", ex);
		}
	}

	/**
	 * prepareSyncRemoteFiles - [Jt@CŃT[o𓯊B
	 * @param path
	 * @throws InternalFailException
	 */
	public File[] prepareSyncRemoteFiles(File path) throws InternalFailException {
		ISVNStatus[] status = null;

		sb.getLogger().debug("Enter prepareSyncRemoteFiles().");
		/**
		 *  t@CVXeύX\[XoB
		 *  ܂̓T[oւ̖₢킹{čŐVXe[^X擾Bʏ͂Ŗ͂ȂA
		 *  OBSTRUCTEDԂꍇɂ͎擾G[͂B̏ꍇɂ͍ēxA
		 *  T[oւ̖₢킹Ŏ擾sB
		 */
		try {
			removeNotifyListener();
			status = this.checkDirectoryStatus(path, true);
		}
		catch (InternalFailException ex) {
			String msg = path + "̕ύXԂ̎擾Ɏs܂B[JɒuꂽpčĎs܂B(prepareSyncRemoteFiles)";
			logEventSource.fireEvent(new LogEvent(this, msg, LogEventType.INFO));
			try {
				status = this.checkDirectoryStatus(path, false);
			}
			catch (NoNeedSyncException e) {
				return new File[] {};
			}
		}
		catch (NoNeedSyncException e) {
			return new File[] {};
		}
		finally {
			addNotifyListener();
		}

		logEventSource.fireEvent(new LogEvent(this, path + "ɂsԂ̃\[XoĂ܂...", LogEventType.INFO));
		/**
		 * ~file_or_dir̕ - |WgƍƃRs[ŎނقȂĂꍇɂ
		 *  [Jt@CŃ[gt@CCB
		 */
		for (int i = 0; i < status.length; i++) {
			ISVNStatus st = status[i];
			// ~file_or_dir
			if (SVNStatusKind.OBSTRUCTED == st.getTextStatus()) {
				File localfile = st.getFile();
				try {
					String msg = localfile.getAbsolutePath() + "͕sԂ̃\[XłB[Jɒuꂽ\[XŏCs܂B";
					logEventSource.fireEvent(new LogEvent(this, msg, LogEventType.INFO));

					if (localfile.isDirectory()) {
						RecoverFolderToFile(localfile);
					}
					else {
						RecoverFileToFolder(localfile);
					}
				}
				catch (RenameFailException ex) {
					throw new InternalFailException(ex);
				}
			}
		}

		boolean bRepeat;
		do {
			bRepeat = false;
			/*
			 *  OBSTRUCTED\[XCStatusĎ擾
			 */
			try {
				status = this.checkDirectoryStatus(path, true);
			}
			catch (NoNeedSyncException ex) {
			}

			/**
			 * ^[QbgtH_zɑ΂W
			 */
			for (int i = 0; i < status.length; i++) {
				ISVNStatus st = status[i];
				/**
				 * |WgŃt@CXVĂ - ł̓[Jt@CpāA
				 * [gt@CCB
				 */
				if (SVNStatusKind.MODIFIED == st.getRepositoryTextStatus()) {
					logEventSource.fireEvent(new LogEvent(this, st.getFile().getAbsolutePath()
					    + "̓f[^Z^ŕύXĂ܂B[Jt@CŃf[^Z^XV܂B", LogEventType.INFO));
					try {
						RecoverModifiedRemoteFileWithLocal(st);
					}
					catch (Exception ex) {
						logEventSource.fireEvent(new LogEvent(this, ex.getMessage(), LogEventType.ERROR, ex));
					}
					continue;
				}
				/**
				 * RtNg - ł̓[Jt@Cpă[gt@CCB
				 */
				if (SVNStatusKind.CONFLICTED == st.getTextStatus()) {
					logEventSource.fireEvent(new LogEvent(this, st.getFile().getAbsolutePath()
					    + "̓f[^Z^ƃRtNgԂɂȂĂ܂B[Jt@CŃf[^Z^XV܂B", LogEventType.INFO));
					try {
						RecoverModifiedRemoteFileWithLocal(st);
					}
					catch (Exception ex) {
						logEventSource.fireEvent(new LogEvent(this, ex.getMessage(), LogEventType.ERROR, ex));
					}
					continue;
				}

				/**
				 * 폜ꂽt@C- t@C̓o[WǗɂ邪AsSȏԂłB
				 * SVNOOSR}hɂ폜Ă܂B
				 * ! file_or_dir
				 */
				if (SVNStatusKind.MISSING == st.getTextStatus()) {
					logEventSource.fireEvent(new LogEvent(this, st.getFile().getAbsolutePath()
					    + "͍폜Ă܂Bf[^Z^Ɋi[ꂽt@C폜܂B", LogEventType.INFO));
					try {
						this.unregisterFile(st.getFile());
					}
					catch (DeleteFailException ex) {
						logEventSource.fireEvent(new LogEvent(this, ex.getMessage(), LogEventType.ERROR, ex));
					}
					catch (InternalFailException ex) {
						sb.getLogger().error(ex.getMessage());
					}
					continue;
				}

				/**
				 * VKǉꂽt@C - SVNɂo[WǗɂȂB 
				 * ? file_or_dir
				 */
				if (SVNStatusKind.UNVERSIONED == st.getTextStatus()) {
					logEventSource.fireEvent(new LogEvent(this,
					  st.getFile().getAbsolutePath() + "̓[Jɒǉ܂Bf[^Z^ɐVKo^܂B", LogEventType.INFO));
					File file = st.getFile();
					if (file.isDirectory()) {
						/**
						 * tH_ǉĂ邽ߍēxstatus̎蒼KvB
						 * ɂătH_z̃t@CXVLƂȂB
						 */
						this.registerFolder(file);
						bRepeat = true;
					}
					else {
						this.registerFile(file);
					}
					continue;
				}
			} // End of for loop.			
		}
		while (bRepeat); // End of do-while(bRpeat) loop

		/**
		 * ŏIIstatus擾B
		 */
		ArrayList<File> list = new ArrayList<File>();
		try {
			status = this.checkDirectoryStatus(path, false);
		}
		catch (NoNeedSyncException e) {
			return new File[] {};
		}
		/**
		 * status񂩂폜ꂽFileXg쐬
		 */
		for (int i = 0; i < status.length; i++) {
			if (status[i].getTextStatus() == SVNStatusKind.DELETED) {
				continue;
			}
			list.add(status[i].getFile());
		}
		return (File[]) list.toArray(new File[list.size()]);
	}

	/**
	 * RecoverModifiedRemoteFileWithLocal
	 * T[oɊi[ꂽt@CύXĂꍇ̕B([Jt@CŃ[gt@CC)
	 * @param st
	 * @throws RenameFailException
	 * @throws DeleteFailException
	 * @throws InternalFailException
	 */
	private void RecoverModifiedRemoteFileWithLocal(ISVNStatus st) throws RenameFailException, DeleteFailException,
	    InternalFailException {
		String src = st.getFile().getAbsolutePath();
		String dest = src + ".orig." + System.currentTimeMillis();
		File localfile = st.getFile();
		File destfile = new File(dest);

		if (localfile.renameTo(destfile) == false) {
			throw new RenameFailException(src + "ɑ΂t@CύXɎs܂B(RecoverModifiedRemoteFileWithLocal)");
		}
		try {
			svnClient.update(localfile, SVNRevision.HEAD, true);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(ex);
		}

		try {
			//			FileUtils.deleteDirectory(localfile);
			FileUtils.forceDelete(localfile);
		}
		catch (IOException ex) {
			throw new DeleteFailException(src + "ɑ΂t@C폜Ɏs܂B(RecoverModifiedRemoteFileWithLocal)");
		}

		sb.gc(); // To gain time ...

		if (destfile.renameTo(localfile) == false) {
			throw new RenameFailException(dest + "ɑ΂t@CύXɎs܂B(RecoverModifiedRemoteFileWithLocal)");
		}

		try {
			FileUtils.touch(localfile);
		}
		catch (IOException ex) {
			throw new RenameFailException(src + "ɑ΂touch()Ɏs܂B(RecoverModifiedRemoteFileWithLocal)");
		}
	}

	/**
	 * RecoverFolderToFile - Plain file in server and folder in client.
	 * 
	 * Recovery step is below.
	 * ren a a.bkup(OS Command)
	 * svn delete a
	 * svn add a.bkup
	 * svn -m msg commit a a.bkup
	 * svn move a.bkup a
	 * svn -m msg commit a a.bkup
	 * 
	 * @param localfile
	 * @throws RenameFailException 
	 * @throws InternalFailException 
	 */
	private void RecoverFolderToFile(File localfile) throws RenameFailException, InternalFailException {
		String src = null, dist = null;
		src = localfile.getAbsolutePath();
		dist = src + ".orig." + System.currentTimeMillis();
		File bkupfile = new File(dist);
		if (localfile.renameTo(bkupfile) == false) {
			throw new RenameFailException(src + "ɑ΂t@CύXɎs܂B(RecoverFolderToFile)");
		}
		try {
			svnClient.remove(new File[] { localfile }, true);
			svnClient.addFile(bkupfile);
			svnClient.commit(new File[] { localfile, bkupfile }, "Recovery step.1", true);
			svnClient.move(bkupfile, localfile, true);
			svnClient.commit(new File[] { localfile, bkupfile }, "Recovery step.2", true);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(ex);
		}
	}

	/**
	 * RecoverFileToFolder - Folder in server, plain file in client.
	 * 
	 * Recovery step is below.
	 * ren a a.bkup (OS Command)
	 * svn update -N a
	 * svn delete a
	 * svn add a.bkup
	 * svn -m msg commit a a.bkup
	 * svn move a.bkup a
	 * svn -m msg commit a a.bkup
	 * 
	 * @param localfile
	 * @throws RenameFailException 
	 * @throws InternalFailException 
	 */
	private void RecoverFileToFolder(File localfile) throws RenameFailException, InternalFailException {
		String src = null, dist = null;
		src = localfile.getAbsolutePath();
		dist = src + ".orig." + System.currentTimeMillis();
		File bkupfile = new File(dist);
		if (localfile.renameTo(bkupfile) == false) {
			throw new RenameFailException("sԃ\[X̃G[: " + src + "ɑ΂t@CύXɎs܂B(RecoverFileToFolder)");
		}
		try {
			svnClient.update(localfile, SVNRevision.HEAD, false);
			svnClient.remove(new File[] { localfile }, true);
			svnClient.addFile(bkupfile);
			svnClient.commit(new File[] { localfile, bkupfile }, "Recovery step.1", true);
			svnClient.move(bkupfile, localfile, true);
			svnClient.commit(new File[] { localfile, bkupfile }, "Recovery step.2", true);
		}
		catch (SVNClientException ex) {
			throw new InternalFailException(ex);
		}
	}
}
