//	Copyright (c) 2008 itok ( http://itok.jp/ , http://110k.net/ )
//	All rights reserved.
//
//	Redistribution and use in source and binary forms, with or without modification, 
//	are permitted provided that the following conditions are met:
//
//	- Redistributions of source code must retain the above copyright notice, 
//	  this list of conditions and the following disclaimer.
//	- Redistributions in binary form must reproduce the above copyright notice, 
//	  this list of conditions and the following disclaimer in the documentation 
//	  and/or other materials provided with the distribution.
//	- Neither the name of itok nor the names of its contributors may be used to endorse 
//	  or promote products derived from this software without specific prior written permission.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS 
//	AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
//	INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
//	AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
//	IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
//	INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
//	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
//	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
//	ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
//	EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#import "PUImageProcessor.h"

#define COMMAND_SIPS	@"/usr/bin/sips"

#define SS_LONG		160
#define SS_SHORT	120
#define S_LONG		320
#define S_SHORT		240
#define M_LONG		640
#define M_SHORT		480
#define L_LONG		1024
#define L_SHORT		768
#define LL_LONG		1600
#define LL_SHORT	1200

@interface PUImageProcessor (Private)

+(NSString*) execCmd:(NSString*)cmd withArgs:(NSArray*)args returnError:(NSString**)error;
+(NSString*) execSipsWithArgs:(NSArray*)args returnError:(NSString**)error;

@end

@implementation PUImageProcessor

+(PUImageFormat) formatWithPath:(NSString*)path
{
	NSString* result = [PUImageProcessor execSipsWithArgs:[NSArray arrayWithObjects:@"-g", @"format", path, nil] returnError:nil];
	NSRange range = [result rangeOfString:@"format: "];
	if (range.location == NSNotFound && range.length == 0) {
		return kPUImageFormat_Invalid;
	}
	NSRange formatRange;
	formatRange.location = range.location + range.length;
	formatRange.length = [result length] - formatRange.location - 1;
	return [PUImageProcessor formatFromString:[result substringWithRange:formatRange]];
//	return @""; // テスト用
}

+(PUImageFormat) formatFromString:(NSString*)str
{
	if ([str length] <= 0) {
		return kPUImageFormat_Invalid;
	} else if ([str caseInsensitiveCompare:@"jpeg"] == NSOrderedSame || 
		[str caseInsensitiveCompare:@"jpg"] == NSOrderedSame) {
		return kPUImageFormat_Jpeg;
	} else if ([str caseInsensitiveCompare:@"png"] == NSOrderedSame) {
		return kPUImageFormat_Png;		
	} else if ([str caseInsensitiveCompare:@"giff"] == NSOrderedSame || 
		[str caseInsensitiveCompare:@"gif"] == NSOrderedSame) {
		return kPUImageFormat_Gif;
	}
	return kPUImageFormat_Other;
}

+(NSString*) stringFromFormat:(PUImageFormat)format
{
	switch (format) {
		case kPUImageFormat_Jpeg :
			return @"JPEG";
			break;
		case kPUImageFormat_Png :
			return @"PNG";
			break;
		case kPUImageFormat_Gif :
			return @"GIF";
			break;	
		case kPUImageFormat_Other :
		default :
			return @"";
			break;
	}
	return @"";
}

+(BOOL) convertImage:(NSString*)origPath toPath:(NSString*)toPath format:(PUImageFormat)format size:(PUImageSize)size
{
	static const int size_tbl[] = {
		SS_LONG,
		S_LONG,
		M_LONG,
		L_LONG,
		LL_LONG,
	};
	
	NSString* err = nil;
	NSMutableArray* args = [NSMutableArray array];
	NSString* formatStr = [[PUImageProcessor stringFromFormat:format] lowercaseString];
	if ([formatStr length] > 0) {
		// フォーマット変換
		[args addObjectsFromArray:[NSArray arrayWithObjects:@"-s", @"format", formatStr, nil]];
	}
	if (size >= kPUImageSize_SS && size <= kPUImageSize_LL) {
		// サイズ変換
		[args addObjectsFromArray:[NSArray arrayWithObjects:@"-Z", [NSString stringWithFormat:@"%d", size_tbl[size]], nil]];
	}
	[args addObjectsFromArray:[NSArray arrayWithObjects:origPath, @"-o", toPath, nil]];

	[PUImageProcessor execSipsWithArgs:args returnError:&err];
	if (err) {
		NSLog(@"ERR: %@", err);
		return NO;
	} 
	return YES;
}

@end

@implementation PUImageProcessor (Private)

+(NSString*) execSipsWithArgs:(NSArray*)args returnError:(NSString**)error
{
	return [PUImageProcessor execCmd:COMMAND_SIPS withArgs:args returnError:error];
}

+(NSString*) execCmd:(NSString*)cmd withArgs:(NSArray*)args returnError:(NSString**)error
{
	NSTask* task = [[NSTask alloc] init];
	NSPipe* pipe = [NSPipe pipe];
	NSFileHandle* hdl = [pipe fileHandleForReading];
	
	NSPipe* errPipe = nil;
	NSFileHandle* errHdl = nil;
	if (error) {
		errPipe = [NSPipe pipe];
		errHdl = [errPipe fileHandleForReading];
	}
	
	NSData* data;
	NSString* result;
	
	@try {
		[task setLaunchPath:cmd];
		[task setStandardOutput:pipe];
		if (errPipe) {
			[task setStandardError:errPipe];
		}
		if (args) {
			[task setArguments:args];
		}
		
		[task launch];
		[task waitUntilExit];
		
		data = [hdl readDataToEndOfFile];
		result = [[[NSString alloc] initWithData:data encoding:NSASCIIStringEncoding] autorelease];
		
		if (errPipe) {
			data = [errHdl readDataToEndOfFile];
			if ([task terminationStatus] && error && [data length] > 0) {
				*error = [[[NSString alloc] initWithData:data encoding:NSASCIIStringEncoding] autorelease];
			}
		}
	} @catch (NSException* excep) {
		result = nil;
		if (error) {
			*error = [NSString stringWithFormat:@"%@:%@", [excep name], [excep reason]];
		}
	}
	[task release];
	
	return result;	
}

@end
