﻿using System;
using System.Collections.Generic;
using System.Text;
using IronPython.Hosting;
using System.Reflection;
using System.Text.RegularExpressions;
using System.IO;

namespace Editor.DefaultParser
{
    class IronPythonDefaultDictionary : IIntelisenceDictionary
    {
        private IronPython.Hosting.PythonEngine _engine;
        readonly string[] _reserveWords = new[] { 
            "and",
            "del",
            "for",
            "is",
            "raise",
            "assert",
            "elif",
            "from",
            "lambda",
            "return",
            "break",
            "else",
            "global",
            "not",
            "try",
            "class",
            "except",
            "if",
            "or",
            "while",
            "continue",
            "exec",
            "import",
            "pass",
            "yield",
            "def",
            "finally",
            "in",
            "print",
            "help"};

        internal IronPythonDefaultDictionary(IronPython.Hosting.PythonEngine engine)
        {
            _engine = engine;
        }

        #region IIntelisenceDictionary メンバ

        public string[] GetChild(IParseResult parseResult)
        {
            var parseResultTmp = parseResult as Editor.DefaultParser.ParseResult;
            string[] tokenList = parseResultTmp.GetPickUpTokens();
            Type dotNetType;
            bool isInstance;
            bool result = GetReturnType(tokenList, out dotNetType, out isInstance);

            if(result==false)
            {
                return new string[0];
            }
            if(dotNetType==null)
            {
                string[] strList = GetPythonDir(CreateFullText(tokenList, 0, tokenList.Length));
                Array.Sort<string>(strList);
                return strList;
            }

            BindingFlags bindingFlags = GetBindingFlags(isInstance);

            List<MemberInfo> memberList = GetMembersInfoList(dotNetType, bindingFlags);

            List<string> list = new List<string>();
            foreach (MemberInfo info in memberList)
            {
                if (list.Contains(info.Name) == false)
                {
                    list.Add(info.Name);
                }
            }
            list.Sort();
            return list.ToArray();
        }

        private List<MemberInfo> GetMembersInfoList(Type dotNetType, BindingFlags bindingFlags)
        {
            List<MemberInfo> memberList = new List<MemberInfo>();
            MethodInfo[] methods = dotNetType.GetMethods(bindingFlags);
            memberList.AddRange(methods);
            PropertyInfo[] properties = dotNetType.GetProperties(bindingFlags);
            memberList.AddRange(properties);
            EventInfo[] events = dotNetType.GetEvents(bindingFlags);
            memberList.AddRange(events);
            ConstructorInfo[] constructors = dotNetType.GetConstructors(bindingFlags | BindingFlags.Instance);
            memberList.AddRange(constructors);
            return memberList;
        }

        private BindingFlags GetBindingFlags(bool isInstance)
        {
            BindingFlags bindingFlags2;
            if (isInstance)
            {
                bindingFlags2 = BindingFlags.Instance | BindingFlags.Public;
            }
            else
            {
                bindingFlags2 = BindingFlags.Static | BindingFlags.Public;
            }
            return bindingFlags2;
        }

        private bool GetReturnType(string[] tokenList, out Type dotNetType, out bool isInstance)
        {
            dotNetType = null;
            isInstance = false;
            bool result = true;

            for (int i = 0; i < tokenList.Length;i++ )
            {
                var token = tokenList[i];
                if (dotNetType == null)
                {
                    object obj;
                    try
                    {
                        obj = _engine.Evaluate(CreateFullText(tokenList, 0, i + 1));

                    }
                    catch (Exception)
                    {
                        result = false;
                        break;
                    }
                    if (obj.GetType() == typeof(IronPython.Runtime.Calls.BuiltinFunction))
                    {
                        //IronPythonの関数
                        result = false;
                        break;
                    }
                    else if (obj.GetType() == typeof(IronPython.Runtime.PythonModule))
                    {
                        //ネームスペース
                    }
                    else if (obj is IronPython.Runtime.Types.DynamicType)
                    {
                        //クラス
                        var dynamicType = (IronPython.Runtime.Types.DynamicType) obj;
                        dotNetType = dynamicType.type;
                    }
                    else
                    {
                        //.NETのインスタンス
                        dotNetType = obj.GetType();
                        isInstance = true;
                    }
                }
                else
                {
                    string checkToken = token;
                    if (IsMethod(checkToken))
                    {
                        checkToken = GetMethodName(checkToken);
                    }

                    BindingFlags bindingFlags;
                    if (isInstance)
                    {
                        bindingFlags = BindingFlags.Instance | BindingFlags.Public;
                    }
                    else
                    {
                        bindingFlags = BindingFlags.Static | BindingFlags.Public;
                    }
                    bool found = false;
                    if (found == false)
                    {
                        MemberInfo[] methods = dotNetType.FindMembers(MemberTypes.Method, bindingFlags, delegate(MemberInfo m, object filterCriteria) { return ((string)filterCriteria) == m.Name; }, checkToken);
                        if (methods.Length != 0)
                        {
                            dotNetType = ((MethodInfo)methods[0]).ReturnType;
                            isInstance = true;
                            found = true;
                        }
                    }
                    if (found == false)
                    {
                        PropertyInfo property = dotNetType.GetProperty(checkToken, bindingFlags);
                        if (property != null)
                        {
                            dotNetType = property.PropertyType;
                            isInstance = true;
                            found = true;
                        }
                    }
                    if (found == false)
                    {
                        result = false;
                        break;
                    }
                }
            }
            return result;
        }

        private string CreateFullText(string[] tokenList, int index, int count)
        {
            StringBuilder stringBuilder = new StringBuilder();
            for (int i = 0; i < count; i++)
            {
                if (stringBuilder.Length != 0)
                {
                    stringBuilder.Append(".");
                }
                stringBuilder.Append(tokenList[i + index]);
            }
            return stringBuilder.ToString();
        }

        public string[] GetPythonDir(string cls)
        {
            object result;
            string resultStr;
            try
            {
                result = _engine.Evaluate(string.Format("dir({0})", cls));
            }
            catch
            {
                return new string[0];
            }
            resultStr = result.ToString();

            Regex _regex = new Regex("'([a-zA-Z0-9_]+)'");
            MatchCollection parentClassMatches = _regex.Matches(resultStr);
            return CreateArray(parentClassMatches, cls == "");
        }

        private string[] CreateArray(MatchCollection matches, bool isAddReservedAdd)
        {
            List<string> list = new List<string>();
            for (int i = 0; i < matches.Count; i++)
            {
                list.Add(matches[i].Groups[1].Value);
            }
            if (isAddReservedAdd == false)
            {
                return list.ToArray();
            }
            for (int i = 0; i < _reserveWords.Length; i++)
            {
                list.Add(_reserveWords[i]);
            }
            list.Sort();
            return list.ToArray();
        }

        private static string GetMethodName(string token)
        {
            Regex regex = new Regex(@"^(?<name>[^\(\)]+)\(");
            Match match = regex.Match(token);
            return match.Groups["name"].Value;
        }

        private static bool IsMethod(string token)
        {
            Regex regex = new Regex(@"^(?<name>[^\(\)]+)\(");
            return regex.IsMatch(token);
        }

        public HelpContentCollection GetHelp(IParseResult parseResult)
        {
            var parseResultTmp = parseResult as Editor.DefaultParser.ParseResult;
            string[] tokenList = parseResultTmp.GetHelpTokens();
            if (tokenList.Length == 0)
            {
                return new HelpContentCollection();
            }

            Type dotNetType;
            bool isInstance;
            bool result;

            //通常関数の場合、関数名は検索対象から外すので最後のトークンを削除してチェック
            string[] tokenList2 = new string[tokenList.Length - 1];
            Array.Copy(tokenList, 0, tokenList2, 0, tokenList.Length - 1);
            result = GetReturnType(tokenList2, out dotNetType, out isInstance);

            if (result == false)
            {
                return new HelpContentCollection();
            }
            if (dotNetType != null)
            {
                return GetDotNetHelp(tokenList, dotNetType, isInstance);
            }

            //コンストラクタの場合があるのでとりあえず全部のトークンでチェック
            result = GetReturnType(tokenList, out dotNetType, out isInstance);
            if (dotNetType != null)
            {
                return GetDotNetHelp(tokenList, dotNetType, isInstance);
            }

            return new HelpContentCollection();// GetPythonHelp(CreateFullText(tokenList, 0, tokenList.Length));
            
        }

        private HelpContentCollection GetDotNetHelp(string[] tokenList, Type dotNetType, bool isInstance)
        {
            BindingFlags bindingFlags = GetBindingFlags(isInstance);
            List<MemberInfo> memberList;
            memberList = GetMembersInfoList(dotNetType, bindingFlags);

            string targetName = tokenList[tokenList.Length - 1];
            StringBuilder stringBuilder = new StringBuilder();
            foreach (var info in memberList)
            {
                if (info.Name == targetName)
                {
                    var method = info as MethodInfo;
                    if (method != null)
                    {
                        stringBuilder.AppendLine(CreateMethodHelp(method));
                    }
                    var property = info as PropertyInfo;
                    if (property != null)
                    {
                        stringBuilder.AppendLine(CreatePropertyHelp(property));
                    }
                    var eventInfo = info as EventInfo;
                    if (eventInfo != null)
                    {
                        stringBuilder.AppendLine(CreateEventHelp(eventInfo));
                    }
                }
                else if (info.Name == ".ctor")
                {
                    var constructor = info as ConstructorInfo;
                    stringBuilder.AppendLine(CreateConstructorHelp(constructor));
                }
            }
            return new HelpContentCollection() {new HelpContent(stringBuilder.ToString())};
        }

        private string CreateConstructorHelp(ConstructorInfo constructor)
        {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.Append(constructor.DeclaringType.Name);
            stringBuilder.Append(" ");
            stringBuilder.Append(constructor.DeclaringType.Name);
            stringBuilder.Append("(");
            ParameterInfo[] parameterInfo = constructor.GetParameters();
            for (int i = 0; i < parameterInfo.Length; i++)
            {
                if(i!=0)
                {
                    stringBuilder.Append(", ");
                }
                stringBuilder.Append(parameterInfo[i].ParameterType.Name);
                stringBuilder.Append(" ");
                stringBuilder.Append(parameterInfo[i].Name);
            }
            stringBuilder.Append(")");
            return stringBuilder.ToString();
        }

        private string CreateEventHelp(EventInfo eventInfo)
        {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.Append(eventInfo.ReflectedType.Name);
            stringBuilder.Append(" ");
            stringBuilder.Append(eventInfo.Name);
            return stringBuilder.ToString();
        }

        private string CreatePropertyHelp(PropertyInfo property)
        {
            StringBuilder stringBuilder = new StringBuilder();
            if (property.CanRead && property.CanWrite) { stringBuilder.Append("[R/W] "); }
            if (!property.CanRead && property.CanWrite) { stringBuilder.Append("[W] "); }
            if (property.CanRead && !property.CanWrite) { stringBuilder.Append("[R] "); }

            stringBuilder.Append(property.PropertyType.Name);
            stringBuilder.Append(" ");
            stringBuilder.Append(property.Name);

            return stringBuilder.ToString();
        }

        private string CreateMethodHelp(MethodInfo method)
        {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.Append(method.ReturnType.Name);
            stringBuilder.Append(" ");
            stringBuilder.Append(method.Name);
            stringBuilder.Append("(");
            ParameterInfo[] parameterInfo = method.GetParameters();
            for (int i = 0; i < parameterInfo.Length; i++)
            {
                if(i!=0)
                {
                    stringBuilder.Append(", ");
                }
                stringBuilder.Append(parameterInfo[i].ParameterType.Name);
                stringBuilder.Append(" ");
                stringBuilder.Append(parameterInfo[i].Name);
            }
            stringBuilder.Append(")");
            return stringBuilder.ToString();
        }

        private string GetPythonHelp(string cls)
        {
            MemoryStream ms = new MemoryStream();
            _engine.Sys.DefaultEncoding = Encoding.Default;
            _engine.SetStandardOutput(ms);
            try
            {
                _engine.Execute(string.Format("help({0})", cls));
            }
            catch
            {
                return "";
            }
            ms.Position = 0;
            StreamReader sr = new StreamReader(ms, Encoding.Default, false);
            string text = sr.ReadToEnd();
            text = text.Replace(" | ", "");
            text = text.Replace("    ", " ");
            Regex regex = new Regex(@"^Help on.+\r\n");
            text = regex.Replace(text, "");
            regex = new Regex(@"^ *\r?\r\n", RegexOptions.Multiline);
            text = regex.Replace(text, "");
            return text;
        }

        #endregion

    }
}
