﻿using System;
using System.Windows.Forms;
using System.Text;
using System.IO;

namespace Editor
{
    public class IntelisenceCore
    {

        private IParseResult _result;

        int _currentIndex;
        string[] _pickList = new string[0];
        HelpContentCollection _help = new HelpContentCollection();

        bool _isFormVisible;
        bool _isHelpVisible;

        readonly IIntelisenceDictionary _dictionary;
        readonly IPythonParser _parser;
        readonly HelpController _helpController;
        readonly IntelisenceController _intelisenceController;
        readonly PythonTextBoxController _textBoxController;
        readonly ITextOutout _textOutput;
        readonly IEngine _engine;

        bool _disableKeyProc = false;
        int[] _operatorList = new int[]{
            '(',
            ')',
            '[',
            ']',
            '{',
            '}',
            '@',
            ',',
            ':',
            '.',
            '`',
            '=',
            ';',
            '+',
            '-',
            '*',
            '/',
            '%',
            '&',
            '|',
            '^',
            '>',
            '<',
            '\'',
            '\"',
            '#',
            '\\',
            '$',
            '?',
            ' ',
            '\t'
        };

        internal IntelisenceCore(IIntelisenceDictionary dictionary, PythonTextBoxController textBoxCont, HelpController helpCont, IntelisenceController inteliCont, IEngine engine, ITextOutout textOutput, IPythonParser parser)
        {
            _dictionary = dictionary;
            _parser = parser;
            _result = _parser.Parse("", 0, null);

            _helpController = helpCont;
            _intelisenceController = inteliCont;
            _engine = engine;
            _textOutput = textOutput;
            _engine.SetTextOutput(_textOutput);

            _textBoxController = textBoxCont;
            _textBoxController.TextChanged += new EventHandler<TextChangedEventArgs>(_textBoxController_TextChanged);
            _textBoxController.Initialized += new EventHandler(_textBoxController_Initialized);
            _textBoxController.KeyDown += new KeyEventHandler(_textBoxController_KeyDown);
            _textBoxController.KeyPress += new KeyPressEventHandler(_textBoxController_KeyPress);
            _textBoxController.LocationChangedAndWindowLocationChanged += new EventHandler(_textBoxController_LocationChangedAndWindowLocationChanged);
        }

        void _textBoxController_LocationChangedAndWindowLocationChanged(object sender, EventArgs e)
        {
            _intelisenceController.UpdatePosition(_textBoxController.GetBasePositoin());
            _helpController.SetWindowPos(_textBoxController.GetBasePositoin());
        }

        void _textBoxController_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (_disableKeyProc)
            {
                _disableKeyProc = false;
                e.Handled = true;
                return;
            }

            char keyValue = e.KeyChar;

            if (keyValue == '\r')
            {
                if (IsFormVisible == false || _intelisenceController.GetSelectedString() == "")
                {
                    ChangeFormVisibleFromBoolean(false);

                    string input;
                    int cursor;
                    _textBoxController.GetText(out input, out cursor);

                    _textBoxController.SetCommandHistory(input);

                    _parser.ResetStatus();
                    _engine.Execute(input);

                    _textBoxController.SetText("", 0);
                    EndHelp();
                    e.Handled = true;
                }
                else
                {
                    string selectedText;
                    selectedText = _intelisenceController.GetSelectedString();
                    if (selectedText != "")
                    {
                        EntrySelectItem(selectedText);
                    }
                }
            }
            else if (keyValue == 0x1B) //ESC
            {
                if (IsHelpVisible)
                {
                    IsHelpVisible = false;
                }
                else if (IsFormVisible)
                {
                    IsFormVisible = false;
                }
            }
            else if (Array.IndexOf(_operatorList, keyValue) >= 0)
            {
                if (_intelisenceController.Visible == false)
                {
                    return;
                }
                string selectedText;
                selectedText = _intelisenceController.GetSelectedString();
                if (selectedText != "")
                {
                    EntrySelectItem(selectedText);
                }
            }
        }





        void _textBoxController_KeyDown(object sender, KeyEventArgs e)
        {

            Keys _lastKey = e.KeyCode;

            if (IsFormVisible)
            {
                if (_lastKey == Keys.PageDown)
                {
                    _intelisenceController.ScrollTo(10);
                }
                else if (_lastKey == Keys.PageUp)
                {
                    _intelisenceController.ScrollTo(-10);
                }
                else if (_lastKey == Keys.Down)
                {
                    _intelisenceController.ScrollTo(1);
                    e.Handled = true;
                }
                else if (_lastKey == Keys.Up)
                {
                    _intelisenceController.ScrollTo(-1);
                    e.Handled = true;
                }
                else if (_lastKey == Keys.Left || _lastKey == Keys.Right)
                {
                    string text;
                    int cursorIndex;
                    _textBoxController.GetText(out text, out cursorIndex);
                    ChangeCursorPos(cursorIndex);
                }
                else if (e.KeyCode == Keys.Enter)
                {
                    e.Handled = true;
                }
            }
            else
            {
                if (_lastKey == Keys.Space && e.Control)
                {
                    e.Handled = true;
                    _disableKeyProc = true;
                    IsFormVisible = true;
                }
            }
        }


        void _textBoxController_Initialized(object sender, EventArgs e)
        {
            ChangeText("", Keys.Enter, 0);
        }

        void _textBoxController_TextChanged(object sender, TextChangedEventArgs e)
        {
            ChangeText(e.Text, e.LastKey, e.CursorIndex);
            _intelisenceController.EditChanged(EditWord);
        }

        public string EditWord
        {
            get { return _result.GetWord(); }
        }

        public string[] PickUpList
        {
            get { return _pickList; }
        }

        public bool IsFormVisible
        {
            get { return _isFormVisible; }
            set
            {
                ChangeFormVisibleFromBoolean(value);
            }
        }

        public bool IsHelpVisible
        {
            get { return _isHelpVisible; }
            set
            {
                ChangeHelpVisibleFromBoolean(value);
            }
        }

        public void ChangeCursorPos(int index)
        {
            _currentIndex = index;
            ChangeFormVisibleFromBoolean(false);
        }

        public void ChangeText(string text, Keys key, int index)
        {
            IParseResult result= _parser.Parse(text, index, _result);
            if (result.IsShowHelp)
            {
                StartHelp(result);
            }
            else
            {
                EndHelp();
            }
            _result = result;
            _currentIndex = index;

            if (_result.IsShowIntelisence())
            {
                ChangeFormVisibleFromBoolean(true);
            }
            else
            {
                ChangeFormVisibleFromBoolean(false);
            }
            if (_isFormVisible)
            {
                _pickList = _dictionary.GetChild(_result);

                _intelisenceController.SetList(PickUpList);
            }
        }

        public void EntrySelectItem(string selectedItem)
        {
            _currentIndex = _result.Replace(selectedItem);

            _textBoxController.SetText(Text, _currentIndex);

            ChangeFormVisibleFromBoolean(false);
        }
        
        private void ChangeFormVisibleFromBoolean(bool isVisible)
        {
            if (_isFormVisible != isVisible)
            {
                _isFormVisible = isVisible;


                _intelisenceController.UpdatePosition(_textBoxController.GetBasePositoin());
                if (IsFormVisible)
                {
                    _intelisenceController.Visible = true;
                    //if (IsHelpVisible)
                    //{
                    //    _helpController.Visible = true;
                    //    _helpController.SetText(_help);
                    //    _helpController.SetWindowPos(_textBoxController.GetBasePositoin());
                    //}
                }
                else
                {
                    _intelisenceController.Visible = false;
                }
            }
        }

        private void ChangeHelpVisibleFromBoolean(bool isVisible)
        {
            //if (_isHelpVisible != isVisible)
            //{
                _isHelpVisible = isVisible;
                _helpController.Visible = isVisible;
            //}
        }

        public string Text
        {
            get { return _result.Text; }
        }

        internal void StartHelp(IParseResult result)
        {
            _help = _dictionary.GetHelp(result);
            _helpController.SetText(_help);
            _helpController.SetWindowPos(_textBoxController.GetBasePositoin());

            ChangeHelpVisibleFromBoolean(true);
        }

        internal void EndHelp()
        {
            _help = new HelpContentCollection();
            ChangeHelpVisibleFromBoolean(false);
        }
    }
}
