﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Text.RegularExpressions;

namespace Editor.Parser.IronPythonParser
{
    class PyExpression : IPyExpression
    {
        readonly List<IPyExpression> _child = new List<IPyExpression>();
        readonly ExpressionType _type;

        internal PyExpression(ExpressionType type)
            : this(type, null, null, null)
        {

        }
        internal PyExpression(ExpressionType type, IPyExpression child)
            : this(type, child, null, null)
        {
        }
        internal PyExpression(ExpressionType type, IPyExpression child1, IPyExpression child2)
            : this(type, child1, child2, null)
        {

        }
        internal PyExpression(ExpressionType type, IPyExpression child1, IPyExpression child2, IPyExpression child3)
        {
            _type = type;
            if (child1 != null)
            {
                _child.Add(child1);
            }
            if (child2 != null)
            {
                _child.Add(child2);
            }
            if (child3 != null)
            {
                _child.Add(child3);
            }
        }
        public ExpressionType Type { get { return _type; } }
        public List<IPyExpression> Child
        {
            get { return _child; }
        }

        #region IExpression メンバ

        public string ToText()
        {
            StringBuilder str = new StringBuilder();
            str.Append(_type.ToString());
            str.Append("( ");
            foreach (var item in _child)
            {
                str.Append(item.ToText());
                str.Append(" ");
            }
            str.Append(")");

            return str.ToString();
        }

        public PyTypeInfo GetTypeInfo(INameResolve resolver)
        {
            string example = "";
            Type type = null;
            KindOfType kindOfType = KindOfType.Unknown;
            MemberInfo memberInfo = null;

            if (IsBinaryArithmeticOperations(_type))
            {
                PyTypeInfo c1Type = _child[0].GetTypeInfo(resolver);
                PyTypeInfo c2Type = _child[1].GetTypeInfo(resolver);

                if (c1Type.DotNetType != null)
                {
                    type = GetBinaryOperationType(_type, c1Type.DotNetType);
                    if (type != null)
                    {
                        kindOfType = KindOfType.DotNetVariable;
                    }
                    else
                    {
                        kindOfType = KindOfType.Unknown;
                    }
                }
                else
                {
                    type = null;
                    kindOfType = KindOfType.PythonVariable;
                }
                string ope = GetBinaryArithmeticOperation(_type);
                example = c1Type.ExampleText + ope + c2Type.ExampleText;
            }
            if (IsComparison(_type))
            {
                example = "True";
                type = typeof(bool);
                kindOfType = KindOfType.DotNetVariable;
            }
            if (IsGroup(_type))
            {
                example = GetGroupExample(_type);
                type = resolver.GetIdentifier(example).DotNetType;
                kindOfType = KindOfType.PythonVariable;
            }
            if (IsSameResult(_type))
            {
                PyTypeInfo c1Type = _child[0].GetTypeInfo(resolver);
                example = c1Type.ExampleText;
                type = c1Type.DotNetType;
                kindOfType = c1Type.KindOfType;
            }
            if (IsCall(_type))
            {
                PyTypeInfo info = _child[0].GetTypeInfo(resolver);
                if (info.KindOfType == KindOfType.DotNetMethod)
                {
                    type = info.DotNetType;
                    example = info.ExampleText;
                    kindOfType = info.KindOfType;
                    memberInfo = info.MemberInfo;
                }
                else if (info.KindOfType == KindOfType.DotNetClass)
                {
                    type = info.DotNetType;
                    example = info.ExampleText;
                    kindOfType = KindOfType.DotNetMethod;
                    memberInfo = info.MemberInfo;
                }
                else if (info.KindOfType == KindOfType.PythonMethod)
                {
                    type = info.DotNetType;
                    example = info.ExampleText;
                    kindOfType = info.KindOfType;
                }
            }
            if (IsAggregate(_type))
            {
                PyTypeInfo info = _child[0].GetTypeInfo(resolver);
                string lastSearchText = "";
                string searchText = info.ExampleText;
                for (int i = 1; i < _child.Count; i++)
                {
                    lastSearchText = searchText;
                    searchText = searchText + "." + _child[i].ToOriginalText();

                    if (info.DotNetType != null)
                    {
                        if (info.KindOfType == KindOfType.DotNetClass || info.KindOfType == KindOfType.PythonClass)
                        {
                            MemberInfo[] members = info.DotNetType.GetMember(_child[i].ToOriginalText(), BindingFlags.Static | BindingFlags.Public);
                            info = CreateMemberTypeInfo(members, searchText);
                        }
                        else if (info.KindOfType == KindOfType.DotNetVariable || info.KindOfType == KindOfType.PythonVariable)
                        {
                            MemberInfo[] members = info.DotNetType.GetMember(_child[i].ToOriginalText(), BindingFlags.Instance | BindingFlags.Public);
                            if (members.Length != 0)
                            {
                                info = CreateMemberTypeInfo(members, searchText);
                            }
                            else
                            {
                                if (info.ExampleText != "")
                                {
                                    info = resolver.GetIdentifier(info.ExampleText + "." + _child[i].ToOriginalText());
                                }
                            }
                        }
                        else if (info.KindOfType == KindOfType.DotNetMethod)
                        {
                            MemberInfo[] members = info.DotNetType.GetMember(_child[i].ToOriginalText(), BindingFlags.Instance | BindingFlags.Public);
                            if (members.Length != 0)
                            {
                                info = CreateMemberTypeInfo(members, searchText);
                            }
                            else
                            {
                                info = UnknownTypeInfo();
                            }
                        }
                    }
                    else
                    {
                        if (info.KindOfType == KindOfType.DotNetNamaSpace)
                        {
                            info = resolver.GetIdentifier(searchText);
                        }
                        else if (info.KindOfType == KindOfType.PythonNameSpace)
                        {
                            info = resolver.GetIdentifier(searchText);
                        }
                        else if (info.KindOfType == KindOfType.PythonClass)
                        {
                            info = resolver.GetIdentifier(searchText);
                        }
                        else if (info.KindOfType == KindOfType.PythonVariable)
                        {
                            info = resolver.GetIdentifier(searchText);
                        }
                        else if (info.KindOfType == KindOfType.PythonMethod)
                        {
                            info = UnknownTypeInfo();
                        }
                    }
                }
                type = info.DotNetType;
                example = info.ExampleText;
                kindOfType = info.KindOfType;
                memberInfo = info.MemberInfo;
            }
            if (IsLambda(_type))
            {
                if (_child.Count <= 1)
                {
                    example = "lambda :1";
                    kindOfType = KindOfType.PythonVariable;
                }
                else
                {
                    example = string.Format("lambda {0}:1", _child[1].ToOriginalText());
                    kindOfType = KindOfType.PythonVariable;
                }
            }
            if (IsStringConversion(_type))
            {
                PyTypeInfo typeInfo = _child[0].GetTypeInfo(resolver);
                type = typeInfo.DotNetType;
                example = typeInfo.ExampleText;
                kindOfType = typeInfo.KindOfType;
            }
            if (IsSubscription(_type))
            {
                PyTypeInfo c1Type = _child[0].GetTypeInfo(resolver);
                if (c1Type.DotNetType != null && c1Type.KindOfType == KindOfType.DotNetVariable)
                {
                    //配列＋添え字
                    type = GetUnaryOperationType(_type, c1Type.DotNetType);
                    if (type != null)
                    {
                        kindOfType = KindOfType.DotNetVariable;
                    }
                    else
                    {
                        kindOfType = KindOfType.Unknown;
                    }
                }
                else if (c1Type.DotNetType != null && c1Type.KindOfType == KindOfType.DotNetClass)
                {
                    //Generic型
                    bool isAllDotNetType = true;
                    StringBuilder str = new StringBuilder();
                    var c2Type = _child[1].GetTypeInfo(resolver);

                    if (c2Type.DotNetType != null)
                    {
                        str.Append(c2Type.ExampleText);
                    }
                    else if (c2Type.KindOfType == KindOfType.Unknown)
                    {
                        //式リストの可能性がある
                        PyExpression exp = _child[1] as PyExpression;
                        if (exp!=null && exp.Type == ExpressionType.ExpressionList)
                        {
                            for (int i = 0; i < exp._child.Count; i++)
                            {
                                var c3Type = exp._child[i].GetTypeInfo(resolver);
                                if (i != 0)
                                {
                                    str.Append(",");
                                }
                                str.Append(c3Type.ExampleText);
                            }
                        }
                    }
                    else
                    {
                        isAllDotNetType = false;
                    }

                    if (isAllDotNetType)
                    {
                        Regex regex = new Regex(@"`.+");
                        string typeFullName = regex.Replace(c1Type.DotNetType.FullName, "");
                        string identifier = string.Format("{0}[{1}]", typeFullName, str.ToString());
                        PyTypeInfo type3 = resolver.GetIdentifier(identifier);
                        type = type3.DotNetType;
                        example = type3.ExampleText;
                        kindOfType = type3.KindOfType;
                        memberInfo = type3.MemberInfo;
                    }
                    else
                    {
                        example = string.Format("{0}[{1}]", c1Type.DotNetType.FullName, str.ToString());
                        kindOfType = KindOfType.PythonClass;
                    }
                }
                else
                {
                    example = _child[0].ToOriginalText() + "[0]";
                    kindOfType = KindOfType.PythonVariable;
                }
            }
            return new PyTypeInfo(type, example, kindOfType, memberInfo);
        }

        private Type GetUnaryOperationType(ExpressionType _type, Type type)
        {
            if (type == typeof(string))
            {
                return typeof(char);
            }
            else
            {
                MethodInfo method = type.GetMethod("get_Item", BindingFlags.Public | BindingFlags.Instance);
                if (method != null)
                {
                    return method.ReturnType;
                }
            }
            return null;
        }

        private Type GetBinaryOperationType(ExpressionType exType, Type xType)
        {
            if (xType.IsPrimitive)
            {
                return GetPrimitiveReturnType(exType, xType);
            }
            else
            {
                string methodName = GetOperatorMethod(exType);
                if(methodName=="")
                {
                    return null;
                }
                MethodInfo method = xType.GetMethod(methodName, BindingFlags.Public | BindingFlags.Static);
                if (method == null)
                {
                    return null;
                }
                else
                {
                    return method.ReturnType;
                }
            }
        }

        private string GetOperatorMethod(ExpressionType exType)
        {
            if (exType == ExpressionType.Add) { return "op_Addition"; }
            if (exType == ExpressionType.Sub) { return "op_Subtraction"; }
            if (exType == ExpressionType.Minus) { return "op_UnaryNegation"; }
            if (exType == ExpressionType.Mult) { return "op_Multiply"; }
            if (exType == ExpressionType.Div) { return "op_Division"; }
            if (exType == ExpressionType.Modulo) { return "op_Modulus"; }
            if (exType == ExpressionType.Power) { return "op_Multiply"; }
            if (exType == ExpressionType.LeftShift) { return "op_LeftShift"; }
            if (exType == ExpressionType.RightShift) { return "op_RightShift"; }
            return "";
        }

        private Type GetPrimitiveReturnType(ExpressionType exType, Type xType)
        {
            if (xType == typeof(string))
            {
                if (exType == ExpressionType.Add) { return typeof(string); }
                if (exType == ExpressionType.Mult) { return typeof(string); }
                if (exType == ExpressionType.Subscription) { return typeof(char); }
            }
            else
            {
                if (exType == ExpressionType.Add) { return xType; }
                if (exType == ExpressionType.Sub) { return xType; }
                if (exType == ExpressionType.Minus) { return xType; }
                if (exType == ExpressionType.Mult) { return xType; }
                if (exType == ExpressionType.Div) { return xType; }
                if (exType == ExpressionType.FloorDiv) { return xType; }
                if (exType == ExpressionType.Modulo) { return xType; }
                if (exType == ExpressionType.Power) { return xType; }
                if (exType == ExpressionType.LeftShift) { return xType; }
                if (exType == ExpressionType.RightShift) { return xType; }
            }
            return null;
        }

        private PyTypeInfo CreateMemberTypeInfo(MemberInfo[] members, string searchText)
        {
            PyTypeInfo info = null;
            if (members.Length == 0)
            {
                info = UnknownTypeInfo();
                return info;
            }
            if (members[0].MemberType == MemberTypes.Event)
            {
                EventInfo ev = members[0] as EventInfo;
                info = new PyTypeInfo(ev.EventHandlerType, searchText, KindOfType.DotNetEvent, ev);
            }
            else if (members[0].MemberType == MemberTypes.Field)
            {
                FieldInfo field = members[0] as FieldInfo;
                info = new PyTypeInfo(field.FieldType, searchText, KindOfType.DotNetVariable, field);
            }
            else if (members[0].MemberType == MemberTypes.Method)
            {
                MethodInfo method = members[0] as MethodInfo;
                info = new PyTypeInfo(method.ReturnType, searchText, KindOfType.DotNetMethod, method);
            }
            else if (members[0].MemberType == MemberTypes.NestedType)
            {
                Type memberType = members[0] as Type;
                info = new PyTypeInfo(memberType, searchText, KindOfType.DotNetClass, memberType);
            }
            else if (members[0].MemberType == MemberTypes.Property)
            {
                PropertyInfo prop = members[0] as PropertyInfo;
                info = new PyTypeInfo(prop.PropertyType, searchText, KindOfType.DotNetVariable, prop);
            }

            return info;
        }

        private static PyTypeInfo UnknownTypeInfo()
        {
            return new PyTypeInfo(null, "", KindOfType.Unknown);
        }

        private bool IsAggregate(ExpressionType type)
        {
            return type == ExpressionType.Aggregate;
        }

        private bool IsStringConversion(ExpressionType type)
        {
            return type == ExpressionType.StringConversion;
        }

        private bool IsLambda(ExpressionType type)
        {
            return type == ExpressionType.Lambda;
        }

        private bool IsSubscription(ExpressionType type)
        {
            return type == ExpressionType.Subscription;
        }

        private bool IsCall(ExpressionType type)
        {
            return type == ExpressionType.Call;
        }

        private bool IsSameResult(ExpressionType type)
        {
            return type == ExpressionType.BitAnd ||
                type == ExpressionType.BitOR ||
                type == ExpressionType.BitWiseInvert ||
                type == ExpressionType.BitXOR ||
                type == ExpressionType.Trinomial ||
                type == ExpressionType.Slicing;
        }

        private string GetGroupExample(ExpressionType type)
        {
            if (type == ExpressionType.Dictionary) { return "{1:0, 2:3}"; }
            if (type == ExpressionType.Generator) { return "(x for x in range(2))"; }
            if (type == ExpressionType.List) { return "[0,1]"; }
            if (type == ExpressionType.ParenthForm) { return "(0,1)"; }
            return "";
        }

        private bool IsGroup(ExpressionType type)
        {
            return type == ExpressionType.Dictionary ||
                type == ExpressionType.Generator ||
                type == ExpressionType.List ||
                type == ExpressionType.ParenthForm;
        }

        private bool IsComparison(ExpressionType type)
        {
            return type == ExpressionType.AndTest ||
                type == ExpressionType.Equal ||
                type == ExpressionType.GraterEqual ||
                type == ExpressionType.GraterThan ||
                type == ExpressionType.In ||
                type == ExpressionType.Is ||
                type == ExpressionType.IsNot ||
                type == ExpressionType.LessEqual ||
                type == ExpressionType.LessThan ||
                type == ExpressionType.NotEqual ||
                type == ExpressionType.NotIn ||
                type == ExpressionType.NotTest ||
                type == ExpressionType.OrTest;
        }

        private string GetBinaryArithmeticOperation(ExpressionType type)
        {
            if (type == ExpressionType.Add) { return "+"; }
            if (type == ExpressionType.Div) { return "/"; }
            if (type == ExpressionType.FloorDiv) { return "//"; }
            if (type == ExpressionType.Minus) { return "-"; }
            if (type == ExpressionType.Modulo) { return "%"; }
            if (type == ExpressionType.Mult) { return "*"; }
            if (type == ExpressionType.Power) { return "**"; }
            if (type == ExpressionType.Sub) { return "-"; }
            if (type == ExpressionType.LeftShift) { return "<<"; }
            if (type == ExpressionType.RightShift) { return ">>"; }
            return "";
        }

        private bool IsBinaryArithmeticOperations(ExpressionType type)
        {
            return type == ExpressionType.Add ||
                type == ExpressionType.Div ||
                type == ExpressionType.FloorDiv ||
                type == ExpressionType.Minus ||
                type == ExpressionType.Modulo ||
                type == ExpressionType.Mult ||
                type == ExpressionType.Power ||
                type == ExpressionType.Sub ||
                type == ExpressionType.LeftShift ||
                type == ExpressionType.RightShift;
        }

        #endregion

        public override string ToString()
        {
            return "Expression: " + ToText();
        }


        #region IExpression メンバ


        public string ToOriginalText()
        {
            string formatBinary = "";
            string formatTrino = "";
            bool isCSV = false;
            string formatCSV = "";
            char charCSV = ',';
            switch (_type)
            {
                case ExpressionType.Add:
                    formatBinary = "{0} + {1}";
                    break;
                case ExpressionType.Sub:
                    formatBinary = "{0} - {1}";
                    break;
                case ExpressionType.Mult:
                    formatBinary = "{0} * {1}";
                    break;
                case ExpressionType.Div:
                    formatBinary = "{0} / {1}";
                    break;
                case ExpressionType.FloorDiv:
                    formatBinary = "{0} // {1}";
                    break;
                case ExpressionType.Modulo:
                    formatBinary = "{0} % {1}";
                    break;
                case ExpressionType.Minus:
                    formatBinary = "-{0}{1}";
                    break;
                case ExpressionType.BitWiseInvert:
                    formatBinary = "~{0}{1}";
                    break;
                case ExpressionType.LeftShift:
                    formatBinary = "{0} << {1}";
                    break;
                case ExpressionType.RightShift:
                    formatBinary = "{0} >> {1}";
                    break;
                case ExpressionType.Power:
                    formatBinary = "{0} ** {1}";
                    break;
                case ExpressionType.Aggregate:
                    isCSV = true;
                    charCSV = '.';
                    formatCSV = "{0}";
                    break;
                case ExpressionType.BitAnd:
                    formatBinary = "{0} & {1}";
                    break;
                case ExpressionType.BitXOR:
                    formatBinary = "{0} ^ {1}";
                    break;
                case ExpressionType.BitOR:
                    formatBinary = "{0} | {1}";
                    break;
                case ExpressionType.NotEqual:
                    formatBinary = "{0} != {1}";
                    break;
                case ExpressionType.GraterEqual:
                    formatBinary = "{0} >= {1}";
                    break;
                case ExpressionType.LessEqual:
                    formatBinary = "{0} <= {1}";
                    break;
                case ExpressionType.Equal:
                    formatBinary = "{0} == {1}";
                    break;
                case ExpressionType.LessThan:
                    formatBinary = "{0} < {1}";
                    break;
                case ExpressionType.GraterThan:
                    formatBinary = "{0} > {1}";
                    break;
                case ExpressionType.Is:
                    formatBinary = "{0} is {1}";
                    break;
                case ExpressionType.In:
                    formatBinary = "{0} in {1}";
                    break;
                case ExpressionType.IsNot:
                    formatBinary = "{0} is not {1}";
                    break;
                case ExpressionType.NotIn:
                    formatBinary = "{0} not in {1}";
                    break;
                case ExpressionType.NotTest:
                    formatBinary = "not {0}{1}";
                    break;
                case ExpressionType.OrTest:
                    formatBinary = "{0} or {1}";
                    break;
                case ExpressionType.AndTest:
                    formatBinary = "{0} and {1}";
                    break;
                case ExpressionType.Lambda:
                    formatBinary = "lambda {1} : {0}";
                    break;
                case ExpressionType.ParameterList:
                    //カンマ区切り
                    isCSV = true;
                    formatCSV = "{0}";
                    break;
                case ExpressionType.ParamAster:
                    formatBinary = "*{0}{1}";
                    break;
                case ExpressionType.ParamAster2:
                    formatBinary = "**{0}{1}";
                    break;
                case ExpressionType.ParamEqual:
                    formatBinary = "{0} = {1}";
                    break;
                case ExpressionType.SubList:
                    //カンマ区切り
                    isCSV = true;
                    //({0})
                    formatCSV = "({0})";
                    break;
                case ExpressionType.Trinomial:
                    formatTrino = "{1} if {0} else {2}";
                    break;
                case ExpressionType.ExpressionList:
                    //カンマ区切り
                    isCSV = true;
                    formatCSV = "{0}";
                    break;
                case ExpressionType.Empty:
                    //空
                    formatBinary = "{0}{1}";
                    break;
                case ExpressionType.ParenthForm:
                    isCSV = true;
                    formatCSV = "({0})";
                    break;
                case ExpressionType.List:
                    formatBinary = "[{0} {1}]";
                    break;
                case ExpressionType.ListFor:
                    formatTrino = "for {0} in {1} {2}";
                    break;
                case ExpressionType.ListIf:
                    formatTrino = "if {0} {1}";
                    break;
                case ExpressionType.Generator:
                    formatBinary = "({0} {1})";
                    break;
                case ExpressionType.GeneratorFor:
                    formatTrino = "for {0} in {1} {2}";
                    break;
                case ExpressionType.GeneratorIf:
                    formatTrino = "if {0} {1}";
                    break;
                case ExpressionType.Dictionary:
                    //カンマ区切り
                    isCSV = true;
                    //"\{ {0} \}"
                    formatCSV = "{{{0}}}";
                    break;
                case ExpressionType.KeyDatum:
                    formatBinary = "{0}:{1}";
                    break;
                case ExpressionType.StringConversion:
                    formatBinary = "`{0}`{1}";
                    break;
                case ExpressionType.Subscription:
                    formatBinary = "{0}[{1}]";
                    break;
                case ExpressionType.Slicing:
                    formatBinary = "{0}[{1}]";
                    break;
                case ExpressionType.SliceItemList:
                    //カンマ区切り
                    isCSV = true;
                    formatCSV = "{0}";
                    break;
                case ExpressionType.SliceItem:
                    //a:b:c a:b a:
                    if(_child.Count==2){formatBinary = "{0}:{1}";}
                    if(_child.Count==3){formatTrino = "{0}:{1}:{2}";}
                    break;
                case ExpressionType.TargetList:
                    //カンマ区切り
                    isCSV = true;
                    formatCSV = "{0}";
                    break;
                case ExpressionType.Call:
                    formatBinary = "{0}({1})";
                    break;
                case ExpressionType.CallArguments:
                    //カンマ区切り
                    isCSV = true;
                    formatCSV = "{0}";
                    break;
                case ExpressionType.KeywordArguments:
                    //カンマ区切り
                    isCSV = true;
                    formatCSV = "{0}";
                    break;
                case ExpressionType.KeywordItem:
                    formatBinary = "{0}={1}";
                    break;
                case ExpressionType.AsterArg:
                    formatBinary = "*{0}{1}";
                    break;
                case ExpressionType.AsterArg2:
                    formatBinary = "**{0}{1}";
                    break;
                default:
                    break;
            }
            string child1 = _child.Count > 0 ? _child[0].ToOriginalText() : "";
            string child2 = _child.Count > 1 ? _child[1].ToOriginalText() : "";
            string child3 = _child.Count > 2 ? _child[2].ToOriginalText() : "";
            if (formatBinary != "")
            {
                return string.Format(formatBinary, child1, child2);
            }
            if (formatTrino != "")
            {
                return string.Format(formatTrino, child1, child2, child3);
            }
            if (isCSV)
            {
                StringBuilder str = new StringBuilder();
                for (int i = 0; i < _child.Count; i++)
                {
                    if (i != 0) { str.Append(charCSV); }
                    str.Append(_child[i].ToOriginalText());
                }
                return string.Format(formatCSV, str.ToString());
            }
            throw new NotSupportedException();
        }

        #endregion
    }

    enum ExpressionType
    {
        Add,
        Sub,
        Mult,
        Div,
        FloorDiv,
        Modulo,
        Minus,
        BitWiseInvert,
        LeftShift,
        RightShift,
        Power,
        Aggregate,
        BitAnd,
        BitXOR,
        BitOR,
        NotEqual,
        GraterEqual,
        LessEqual,
        Equal,
        LessThan,
        GraterThan,
        Is,
        In,
        IsNot,
        NotIn,
        NotTest,
        OrTest,
        AndTest,
        Lambda,
        ParameterList,
        ParamAster,
        ParamAster2,
        ParamEqual,
        SubList,
        Trinomial,
        ExpressionList,
        Empty,
        ParenthForm,
        List,
        ListFor,
        ListIf,
        Generator,
        GeneratorFor,
        GeneratorIf,
        Dictionary,
        KeyDatum,
        StringConversion,
        Subscription,
        Slicing,
        SliceItemList,
        SliceItem,
        TargetList,
        Call,
        CallArguments,
        KeywordArguments,
        KeywordItem,
        AsterArg,
        AsterArg2
    }
}
