﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using Editor.Parser.IronPythonParser;

namespace Editor.Parser
{
    class PythonParser : IPythonParser
    {
        ParserCache _parserCache = new ParserCache();
        PyLexer _lexer = new PyLexer();

        #region IPythonParser メンバ

        public IParseResult Parse(string text, int index)
        {
            return Parse(text, index, null);
        }

        public IParseResult Parse(string text, int index, IParseResult lastResult)
        {
            bool isTextContinue = false;
            ParseResult lastResult2 = lastResult as ParseResult;

            PyTokenCollection tokensTmp = new PyTokenCollection();
            string remainText = text;
            if (lastResult2 != null)
            {
                isTextContinue = lastResult2.IsTextContinue;
                tokensTmp = lastResult2.Tokens.GetSubTokens(text);
                if (tokensTmp.Count >= 2)
                {
                    tokensTmp.RemoveAt(tokensTmp.Count - 1);
                    remainText = text.Replace(tokensTmp.GetOriginalText(), "");
                }
                else
                {
                    tokensTmp.Clear();
                }
            }
            
            PyTokenCollection tokens = _lexer.Lex(remainText, isTextContinue);
            tokens.InsertRange(0, tokensTmp);

            Primary[] intelisencePrimaries = null;
            var token = tokens.GetTokenAt(index - 1);
            if (token == null)
            {
                //先頭
                return new ParseResult(tokens, index, null, null, null);
            }

            PyTokenCollection tokensWithoutWhiteSpece = tokens.CreateWithoutWhiteSpace();
            int index2 = tokensWithoutWhiteSpece.IndexOf(token);
            if (index2 == -1)
            {
                //indexが見つからない場合は、現在箇所が空白なので、右方向に探索する
                int index3 = tokens.IndexOf(token);
                for (int i = index3; i < tokens.Count; i++)
                {
                    index2 = tokensWithoutWhiteSpece.IndexOf(tokens[i]);
                    if (index2 != -1)
                    {
                        break;
                    }
                }
            }

            if (token.Type == TokenType.dot)
            {
                intelisencePrimaries = ParseIntelisence(tokensWithoutWhiteSpece, index2);
            }
            else if (token.Type == TokenType.identifier)
            {
                intelisencePrimaries = ParseIntelisence(tokensWithoutWhiteSpece, index2 - 1);
            }
            else
            {
                //直前の文字が文字でも'.'でもないならちょうど切れ目の位置or記号、数字の途中にいる
                intelisencePrimaries = new Primary[0];
            }

            int parenOpenIndex;
            if (index2 != -1)
            {
                parenOpenIndex = SearchParenOpen(tokensWithoutWhiteSpece, index2 + 1);
            }
            else
            {
                //index2が-1の場合は、テキストの最後か、テキストの最後に空白がある場合
                //その場合は、一番最後から探索する
                parenOpenIndex = SearchParenOpen(tokensWithoutWhiteSpece, tokensWithoutWhiteSpece.Count);
            }

            Primary[] helpPrimaries = new Primary[0];
            if (parenOpenIndex > 0)
            {
                helpPrimaries = ParseIntelisence(tokensWithoutWhiteSpece, parenOpenIndex - 1);
            }

            string intelisenceText = CreatePrimaryText(intelisencePrimaries);
            IPyExpression intelisenceExpression = null;
            if (intelisenceText != "")
            {
                intelisenceExpression = _parserCache.Parse(intelisenceText);
            }
            string helpText = CreatePrimaryText(helpPrimaries);
            IPyExpression helpExpression = null;
            if (helpText != "")
            {
                helpExpression = _parserCache.Parse(helpText);
            }

            var result = new ParseResult(tokens, index, intelisenceExpression, helpExpression, _lexer);
            return result;
        }

        private string CreatePrimaryText(Primary[] primaries)
        {
            if (primaries.Length == 0)
            {
                return "";
            }
            StringBuilder str = new StringBuilder();
            str.Append(primaries[0].ToText());
            for (int i = 1; i < primaries.Length; i++)
            {
                if(primaries[i].Type != PrimaryType.Subscription)
                {
                    str.Append(".");
                }
                str.Append(primaries[i].ToText());
            }
            return str.ToString();
        }

        private Primary[] ParseIntelisence(PyTokenCollection tokens, int index)
        {
            List<Primary> result = new List<Primary>();
            if (index < 0)
            {
                return result.ToArray();
            }

            if (tokens[index].Type == TokenType.dot)
            {
                index--;
            }
            while (index >= 0)
            {
                if (tokens[index].Type == TokenType.identifier)
                {
                    //文字列なら、identifierと認識
                    var pri = new Primary(tokens[index], PrimaryType.identifier);
                    pri.Name = tokens[index].Text;
                    result.Insert(0, pri);
                    index--;
                }
                else if (tokens[index].Type == TokenType.parenClose)
                {
                    //')'なら関数か、(○,○)
                    if (IsMethod(tokens, index))
                    {
                        //関数の場合
                        PyTokenCollection tokensTmp = GetMethodTokens(tokens, index);
                        var pri = new Primary(tokensTmp, PrimaryType.Method);
                        pri.Name = tokensTmp[0].Text;
                        result.Insert(0, pri);
                        index -= pri.TokenCount;
                    }
                    else if(IsParenExpression(tokens, index))
                    {
                        //(○)の場合
                        PyTokenCollection tokensTmp = GetParenExpressionTokens(tokens, index);
                        var pri = new Primary(tokensTmp, PrimaryType.ParenExpression);
                        result.Insert(0, pri);
                        index -= pri.TokenCount;
                    }
                    else
                    {
                        //それ以外なら終了
                        break;
                    }
                }
                else if (tokens[index].Type == TokenType.bracketClose)
                {
                    //']'なら配列か、[○,○]
                    if (IsSubscription(tokens, index))
                    {
                        PyTokenCollection tokensTmp = GetSubscriptionTokens(tokens, index);
                        var pri = new Primary(tokensTmp, PrimaryType.Subscription);
                        result.Insert(0, pri);
                        index -= pri.TokenCount;
                        //次は、文字か)か]のはず
                        continue;
                    }
                    else
                    {
                        //[○,○]なら終了
                        PyTokenCollection tokensTmp = GetSubscriptionTokens(tokens, index);
                        var pri = new Primary(tokensTmp, PrimaryType.Terminal);
                        result.Insert(0, pri);
                        index -= pri.TokenCount;
                        break;
                    }
                }
                else if(tokens[index].Type == TokenType.braceClose)
                {
                    PyTokenCollection tokensTmp = GetDictionaryTokens(tokens, index);
                    var pri = new Primary(tokensTmp, PrimaryType.Dictionary);
                    result.Insert(0, pri);
                    index -= pri.TokenCount;
                    break;
                }
                else if (IsStringEnd(tokens[index].Type))
                {
                    //文字列の終端なら空文字列を入れて終了
                    var pri = new Primary(new PyToken("''", TokenType.shortstring2), PrimaryType.Text);
                    result.Insert(0, pri);
                    break;
                }
                else if (IsTerminal(tokens[index].Type))
                {
                    //終端文字ならそれを最後に挿入して終わり
                    var pri = new Primary(tokens[index], PrimaryType.Terminal);
                    result.Insert(0, pri);
                    break;
                }
                if (index < 0)
                {
                    break;
                }
                if (tokens[index].Type == TokenType.dot)
                {
                    //次が"."なら1字進めてさらに検索
                    index--;
                }
                else
                {
                    //そうでないなら終了
                    break;
                }
            }
            return result.ToArray();
        }

        private PyTokenCollection GetDictionaryTokens(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            PyTokenCollection result = new PyTokenCollection();

            int startIndex = SearchBraceOpen(tokensWithoutWhiteSpece, index);
            if (startIndex < 0) { return result; }

            for (int i = startIndex; i <= index; i++)
            {
                result.Add(tokensWithoutWhiteSpece[i]);
            }

            return result;
        }

        private int SearchBraceOpen(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            int count = 1;
            for (int i = index - 1; i >= 0; i--)
            {
                if (tokensWithoutWhiteSpece[i].Type == TokenType.braceClose)
                {
                    count++;
                }
                else if (tokensWithoutWhiteSpece[i].Type == TokenType.braceOpen)
                {
                    count--;
                }
                if (count == 0)
                {
                    return i;
                }
            }
            return -1;
        }

        private bool IsTerminal(TokenType tokenType)
        {
            return tokenType == TokenType.floatnumber1
                || tokenType == TokenType.floatnumber2
                || tokenType == TokenType.floatnumber3
                || tokenType == TokenType.hexinteger
                || tokenType == TokenType.integer
                || tokenType == TokenType.octinteger;
        }

        private bool IsStringEnd(TokenType tokenType)
        {
            return tokenType == TokenType.shortstring1
                || tokenType == TokenType.shortstring2
                || tokenType == TokenType.longstring1
                || tokenType == TokenType.longstring2
                || tokenType == TokenType.longstring5
                || tokenType == TokenType.longstring6;
        }

        private PyTokenCollection GetParenExpressionTokens(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            PyTokenCollection result = new PyTokenCollection();

            int startIndex = SearchParenOpen(tokensWithoutWhiteSpece, index);
            if (startIndex < 0) { return result; }

            for (int i = startIndex; i <= index; i++)
            {
                result.Add(tokensWithoutWhiteSpece[i]);
            }

            return result;
        }

        private bool IsParenExpression(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            int startIndex = SearchParenOpen(tokensWithoutWhiteSpece, index);
            if (startIndex < 0) { return false; }

            return true;
        }

        private PyTokenCollection GetSubscriptionTokens(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            PyTokenCollection result = new PyTokenCollection();

            int startIndex = SearchBracketOpen(tokensWithoutWhiteSpece, index);
            if (startIndex < 0) { return result; }

            for (int i = startIndex; i <= index ; i++)
            {
                result.Add(tokensWithoutWhiteSpece[i]);
            }

            return result;
        }

        private bool IsSubscription(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            int startIndex = SearchBracketOpen(tokensWithoutWhiteSpece, index);
            if (startIndex <= 0) { return false; }

            if (tokensWithoutWhiteSpece[startIndex - 1].Type == TokenType.identifier ||
                tokensWithoutWhiteSpece[startIndex - 1].Type == TokenType.bracketClose || 
                tokensWithoutWhiteSpece[startIndex - 1].Type == TokenType.parenClose)
            {
                return true;
            }
            return false;
        }

        private static int SearchBracketOpen(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            int count = 1;
            for (int i = index - 1; i >= 0; i--)
            {
                if (tokensWithoutWhiteSpece[i].Type == TokenType.bracketClose)
                {
                    count++;
                }
                else if (tokensWithoutWhiteSpece[i].Type == TokenType.bracketOpen)
                {
                    count--;
                }
                if (count == 0)
                {
                    return i;
                }
            }
            return -1;
        }

        private PyTokenCollection GetMethodTokens(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            PyTokenCollection result = new PyTokenCollection();

            int startIndex = SearchParenOpen(tokensWithoutWhiteSpece, index);
            if (startIndex <= 0) { return result; }

            if (tokensWithoutWhiteSpece[startIndex - 1].Type == TokenType.identifier)
            {
                for (int i = startIndex - 1; i <= index; i++)
                {
                    result.Add(tokensWithoutWhiteSpece[i]);
                }
            }
            return result;
        }

        private bool IsMethod(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            int startIndex = SearchParenOpen(tokensWithoutWhiteSpece, index);
            if (startIndex <= 0) { return false; }

            if (tokensWithoutWhiteSpece[startIndex - 1].Type == TokenType.identifier)
            {
                return true;
            }
            return false;
        }

        private static int SearchParenOpen(PyTokenCollection tokensWithoutWhiteSpece, int index)
        {
            int count = 1;
            for (int i = index - 1; i >= 0; i--)
            {
                if (tokensWithoutWhiteSpece[i].Type == TokenType.parenClose)
                {
                    count++;
                }
                else if (tokensWithoutWhiteSpece[i].Type == TokenType.parenOpen)
                {
                    count--;
                }
                if (count == 0)
                {
                    return i;
                }
            }
            return -1;
        }

        #endregion

        #region IPythonParser メンバ


        public void ResetStatus()
        {
            _parserCache.Clear();
        }

        #endregion
    }
}
