// License: GPL2

#include "iomonitor.h"

#include <errno.h>
#include <iostream>

using namespace CORE;

#define FIFO_FILE "test_fifo"

enum IOMONITOR_FLAGS
{
    INIT_AGAIN = -2,
    INIT_ERROR = -1,
    INIT_OK = 0,
    COMMAND_MAX_LENGTH = 1024
};


/*-------------------------------------------------------------------*/
// コンストラクタ
/*-------------------------------------------------------------------*/
IOMonitor::IOMonitor()
  : m_fifo_fd( -1 ),
    m_iochannel( NULL ),
    m_main_process( false )
{
    // 初期化( FIFO_AGAINが戻ったら制限回数の範囲で繰り返す )
    size_t count = 0;
    while( init() == INIT_AGAIN && count < 10 ) ++count;
}


/*-------------------------------------------------------------------*/
// デストラクタ
/*-------------------------------------------------------------------*/
IOMonitor::~IOMonitor()
{
    // メインプロセスの終了時にFIFOを消去する
    if( m_main_process && unlink( FIFO_FILE ) < 0 )
    {
        std::cerr << "IOMonitor::~IOMonitor(): fifo unlink failed." << std::endl;
    }

    if( m_iochannel ) m_iochannel->close(); // close( m_fifo_fd )
}


/*-------------------------------------------------------------------*/
// このクラスの初期化( FIFO作成、FIFOオープン、Glib::IOChannelの作成 )
//
// 戻り値: 成功 INIT_OK, エラー INIT_ERROR, やり直し INIT_AGAIN
/*-------------------------------------------------------------------*/
int IOMonitor::init()
{
    // FIFOを作成
    int mkfifo_status = -1;
    mkfifo_status = mkfifo( FIFO_FILE, O_RDWR | S_IRUSR | S_IWUSR );

    // FIFO作成でエラーになった( 基本的に既にメインプロセスがある )
    if( mkfifo_status != 0 )
    {
        // FIFOが存在しない
        if( errno != EEXIST )
        {
            std::cerr << "IOMonitor::init(): fifo create failed." << std::endl;
            return INIT_ERROR;
        }

        // FIFOを書き込み専用モードでオープン( ノンブロック )
        while( ( m_fifo_fd = open( FIFO_FILE, O_WRONLY | O_NONBLOCK ) ) == -1 )
        {
            // ノンブロックなので、これらのフラグが立っていたら戻す
            if( errno & ( EAGAIN | EINTR ) != 0 ) continue;

            // 反対側が既にオープンされていない( 異常終了などでメインプロセスがない )
            if( errno == ENXIO )
            {
                // 残っているFIFOを消す
                if( unlink( FIFO_FILE ) < 0 )
                {
                    std::cerr << "IOMonitor::init(): fifo unlink failed." << std::endl;
                    return INIT_ERROR;
                }

                // 最初からやり直す
                return INIT_AGAIN;
            }
            // その他のエラー
            else
            {
                std::cerr << "IOMonitor::init(): " << strerror( errno ) << std::endl;
                return INIT_ERROR;
            }
        }
    }
    // メインプロセス
    else
    {
        // FIFOを読み込み専用モードでオープン( ノンブロック )
        while( ( m_fifo_fd = open( FIFO_FILE, O_RDWR | O_NONBLOCK ) ) == -1 )
        {
            // ノンブロックなので、これらのフラグが立っていたら戻す
            if( errno & ( EAGAIN | EINTR ) != 0 ) continue;

            // エラーなのでFIFOを消す
            if( unlink( FIFO_FILE ) < 0 )
            {
                std::cerr << "IOMonitor::init(): fifo unlink failed." << std::endl;
            }

            std::cerr << "IOMonitor::init(): " << strerror( errno ) << std::endl;
            return INIT_ERROR;
        }

        // メインプロセスである
        m_main_process = true;

        // Glib::IOChannel
        Glib::signal_io().connect( sigc::mem_fun( this, &IOMonitor::slot_ioin ), m_fifo_fd, Glib::IO_IN );
        m_iochannel = Glib::IOChannel::create_from_fd( m_fifo_fd );
    }

    return INIT_OK;
}


/*-------------------------------------------------------------------*/
// FIFOに書き込む
//
// 引数 1: 書き込む文字列
//
// 戻り値: 全て書き込まれたか否か
/*-------------------------------------------------------------------*/
bool IOMonitor::send_command( const char* command )
{
    const size_t command_length = strlen( command );

    if( command_length > COMMAND_MAX_LENGTH ) return false;

    g_assert( m_fifo_fd >= 0 );

    int status = -1;
    status = write( m_fifo_fd, command, command_length );

    return ( (unsigned long)status == command_length );
}


/*-------------------------------------------------------------------*/
// FIFOに書き込まれたら呼び出される( Glib::signal_io() )
//
// 引数 1: Glib::IOCondition
//
// 戻り値: true
/*-------------------------------------------------------------------*/
bool IOMonitor::slot_ioin( Glib::IOCondition io_condition )
{
    if( ( io_condition & ( Glib::IO_IN | Glib::IO_PRI ) ) == 0 )
    {
        std::cerr << "IOMonitor::slot_ioin(): Invalid fifo response." << std::endl;
    }
    else
    {
        Glib::ustring buffer;

#ifdef GLIBMM_EXCEPTIONS_ENABLED
        Glib::IOStatus io_status;

        // 最大で COMMAND_MAX_LENGTH まで読み出す
        io_status = m_iochannel->read( buffer, COMMAND_MAX_LENGTH );

        if( io_status == Glib::IO_STATUS_ERROR )
        {
            std::cerr << "IOMonitor::slot_ioin(): read error." << std::endl;
        }
#else
        std::auto_ptr< Glib::Error > ex;
        m_iochannel->read( buffer, COMMAND_MAX_LENGTH, ex );

        if( ex.get() )
        {
            std::cerr << "IOMonitor::slot_ioin(): read_error. " << ex->what() << std::endl;
        }
#endif // GLIBMM_EXCEPTIONS_ENABLED

        std::cout << "入力文字: " << buffer << std::endl;

        if( buffer == "Q" ) Gtk::Main::quit();
        //core_set_command( "open_article", buffer, true, "auto" );
    }

    return true;
}

