//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Text;
using System.Diagnostics;

namespace ircam.fts.client
{
    internal class BinaryProtocolDecoder
    {
        private const int Q_INITIAL = 1;

        private const int Q_INT0 = 10;
        private const int Q_INT1 = 11;
        private const int Q_INT2 = 12;
        private const int Q_INT3 = 13;

        private const int Q_FLOAT0 = 20;
        private const int Q_FLOAT1 = 21;
        private const int Q_FLOAT2 = 22;
        private const int Q_FLOAT3 = 23;
        private const int Q_FLOAT4 = 24;
        private const int Q_FLOAT5 = 25;
        private const int Q_FLOAT6 = 26;
        private const int Q_FLOAT7 = 27;

        private const int Q_STRING = 30;

        private const int Q_OBJECT0 = 40;
        private const int Q_OBJECT1 = 41;
        private const int Q_OBJECT2 = 42;
        private const int Q_OBJECT3 = 43;

        private const int Q_SYMBOL_INDEX0 = 50;
        private const int Q_SYMBOL_INDEX1 = 51;
        private const int Q_SYMBOL_INDEX2 = 52;
        private const int Q_SYMBOL_INDEX3 = 53;

        private const int Q_SYMBOL_CACHE0 = 60;
        private const int Q_SYMBOL_CACHE1 = 61;
        private const int Q_SYMBOL_CACHE2 = 62;
        private const int Q_SYMBOL_CACHE3 = 63;
        private const int Q_SYMBOL_CACHE4 = 64;

        private const int Q_RAW_STRING = 70;

        private void ClearAction(int input)
        {
            lval = 0;
        }

        private void ShiftAction(int input)
        {
            lval = lval << 8 | ((uint)input);
        }

        private void ShiftLongAction(int input)
        {
            lval = lval << 8 | ((uint)input);
        }

        private void BufferClearAction(int input)
        {
            buffer.Length = 0;
        }

        private void ClearAllAction(int input)
        {
            lval = 0;
            buffer.Length = 0;
        }

        private void BufferShiftAction(int input)
        {
            buffer.Append((char)input);
        }

        private void EndIntAction(int input)
        {
            lval = lval << 8 | ((uint)input);

            if (argsCount >= 2)
                args.AddInt((int)lval);
            argsCount++;
        }

        private void EndFloatAction(int input)
        {
            lval = lval << 8 | ((uint)input);

            if (argsCount >= 2)
            {
                args.AddDouble(union.LongBitsToDouble(lval));
            }

            argsCount++;
        }

        private void EndSymbolIndexAction(int input)
        {
            lval = lval << 8 | ((uint)input);

            FtsSymbol s = symbolCache[(int)lval];

            if (argsCount == 1)
                selector = s;
            else
                args.AddSymbol(s);

            argsCount++;
        }

        private void EndSymbolCacheAction(int input)
        {
            FtsSymbol s = FtsSymbol.Get(buffer.ToString());

            symbolCache[(int)lval] = s;

            if (argsCount == 1)
                selector = s;
            else
                args.AddSymbol(s);

            argsCount++;
        }

        private void EndStringAction(int input)
        {
            if (argsCount >= 2)
                args.AddString(buffer.ToString());
            argsCount++;
        }

        private void EndObjectAction(int input)
        {
            lval = lval << 8 | ((uint)input);

            FtsObject obj = server[(int)lval];

            if (argsCount == 0)
                target = obj;
            else
                args.AddObject(obj);

            argsCount++;
        }

        private void EndMessageAction(int input)
        {
            Debug.Listeners.Add(new TextWriterTraceListener(Console.Error));
            Debug.AutoFlush = true;
            Debug.WriteLine("[client] received message:" +
                            " target=" + ((target == null) ? "null" : target.ToString()) +
                            " selector=" + ((selector == null) ? "null" : selector.ToString()) +
                            " args=[" + args + "]");

            FtsObject.InvokeMessageHandler(target, selector, args);

            target = null;
            selector = null;
            args.Clear();
            argsCount = 0;
        }

        private void NextState(int input)
        {
            switch (currentState)
            {
                case 0:
                    /* try to skip till end of message */
                    if (input == BinaryProtocol.END_OF_MESSAGE)
                        currentState = Q_INITIAL;
                    break;

                case Q_INITIAL:
                    if (input == BinaryProtocol.INT)
                    {
                        currentState = Q_INT0;
                        ClearAction(input);
                    }
                    else if (input == BinaryProtocol.FLOAT)
                    {
                        currentState = Q_FLOAT0;
                        ClearAction(input);
                    }
                    else if (input == BinaryProtocol.SYMBOL_INDEX)
                    {
                        currentState = Q_SYMBOL_INDEX0;
                        ClearAction(input);
                    }
                    else if (input == BinaryProtocol.SYMBOL_CACHE)
                    {
                        currentState = Q_SYMBOL_CACHE0;
                        ClearAllAction(input);
                    }
                    else if (input == BinaryProtocol.STRING)
                    {
                        currentState = Q_STRING;
                        BufferClearAction(input);
                    }
                    else if (input == BinaryProtocol.OBJECT)
                    {
                        currentState = Q_OBJECT0;
                        ClearAction(input);
                    }
                    else if (input == BinaryProtocol.END_OF_MESSAGE)
                        EndMessageAction(input);
                    else
                        currentState = 0;
                    break;

                case Q_INT0:
                    currentState = Q_INT1;
                    ShiftAction(input);
                    break;

                case Q_INT1:
                    currentState = Q_INT2;
                    ShiftAction(input);
                    break;

                case Q_INT2:
                    currentState = Q_INT3;
                    ShiftAction(input);
                    break;

                case Q_INT3:
                    currentState = Q_INITIAL;
                    EndIntAction(input);
                    break;

                case Q_FLOAT0:
                    currentState = Q_FLOAT1;
                    ShiftAction(input);
                    break;

                case Q_FLOAT1:
                    currentState = Q_FLOAT2;
                    ShiftAction(input);
                    break;

                case Q_FLOAT2:
                    currentState = Q_FLOAT3;
                    ShiftAction(input);
                    break;

                case Q_FLOAT3:
                    currentState = Q_FLOAT4;
                    ShiftAction(input);
                    break;

                case Q_FLOAT4:
                    currentState = Q_FLOAT5;
                    ShiftAction(input);
                    break;

                case Q_FLOAT5:
                    currentState = Q_FLOAT6;
                    ShiftAction(input);
                    break;

                case Q_FLOAT6:
                    currentState = Q_FLOAT7;
                    ShiftAction(input);
                    break;

                case Q_FLOAT7:
                    currentState = Q_INITIAL;
                    EndFloatAction(input);
                    break;

                case Q_SYMBOL_INDEX0:
                    currentState = Q_SYMBOL_INDEX1;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_INDEX1:
                    currentState = Q_SYMBOL_INDEX2;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_INDEX2:
                    currentState = Q_SYMBOL_INDEX3;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_INDEX3:
                    currentState = Q_INITIAL;
                    EndSymbolIndexAction(input);
                    break;

                case Q_SYMBOL_CACHE0:
                    currentState = Q_SYMBOL_CACHE1;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_CACHE1:
                    currentState = Q_SYMBOL_CACHE2;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_CACHE2:
                    currentState = Q_SYMBOL_CACHE3;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_CACHE3:
                    currentState = Q_SYMBOL_CACHE4;
                    ShiftAction(input);
                    break;

                case Q_SYMBOL_CACHE4:
                    if (input == 0)
                    {
                        currentState = Q_INITIAL;
                        EndSymbolCacheAction(input);
                    }
                    else
                        BufferShiftAction(input);
                    break;

                case Q_STRING:
                    if (input == 0)
                    {
                        currentState = Q_INITIAL;
                        EndStringAction(input);
                    }
                    else
                        BufferShiftAction(input);
                    break;

                case Q_OBJECT0:
                    currentState = Q_OBJECT1;
                    ShiftAction(input);
                    break;

                case Q_OBJECT1:
                    currentState = Q_OBJECT2;
                    ShiftAction(input);
                    break;

                case Q_OBJECT2:
                    currentState = Q_OBJECT3;
                    ShiftAction(input);
                    break;

                case Q_OBJECT3:
                    currentState = Q_INITIAL;
                    EndObjectAction(input);
                    break;
            }
        }

        internal BinaryProtocolDecoder(FtsServer server)
        {
            this.server = server;

            buffer = new StringBuilder();
            args = new FtsArgs();
            argsCount = 0;

            symbolCache = new SymbolCache();

            currentState = Q_INITIAL;
        }

        internal void Decode(byte[] data, int offset, int length)
        {
            for (int i = offset; i < length; i++)
            {
                int c = data[i];

                if (c < 0)
                    c += 256;

                NextState(c);

                if (currentState == 0)
                    throw new FtsClientException("Invalid data in protocol : state=" +
                                                 currentState + " input=" + c);
            }
        }

        private FtsServer server;

        private long lval;
        private StringBuilder buffer;

        private FtsObject target;
        private FtsSymbol selector;
        private FtsArgs args;
        private int argsCount;

        private int currentState;

        private SymbolCache symbolCache;

        private Union union = new Union();
    }
}