//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;

namespace ircam.fts.client
{
    internal class BinaryProtocolEncoder
    {
        private class OutputBuffer
        {
            internal OutputBuffer()
            {
                length = 256;
                buffer = new byte[length];
                current = 0;
            }

            internal void Clear()
            {
                current = 0;
            }

            internal void Append(byte b)
            {
                if (current + 1 >= length)
                {
                    length *= 2;
                    byte[] newBuffer = new byte[length];
                    Array.Copy(buffer, 0, newBuffer, 0, current);
                    buffer = newBuffer;
                }

                buffer[current++] = b;
            }

            internal byte[] Bytes
            {
                get
                {
                    return buffer;
                }
            }

            internal int Length
            {
                get
                {
                    return current;
                }
            }

            private byte[] buffer;
            private int current;
            private int length;
        }

        internal BinaryProtocolEncoder(FtsServerConnection connection)
        {
            this.connection = connection;
            outputBuffer = new OutputBuffer();
            symbolCache = new SymbolCache();
        }

        private void Write(int v)
        {
            outputBuffer.Append((byte)((v >> 24) & 0xff));
            outputBuffer.Append((byte)((v >> 16) & 0xff));
            outputBuffer.Append((byte)((v >> 8) & 0xff));
            outputBuffer.Append((byte)((v >> 0) & 0xff));
        }

        private void Write(long v)
        {
            outputBuffer.Append((byte)((v >> 56) & 0xff));
            outputBuffer.Append((byte)((v >> 48) & 0xff));
            outputBuffer.Append((byte)((v >> 40) & 0xff));
            outputBuffer.Append((byte)((v >> 32) & 0xff));
            outputBuffer.Append((byte)((v >> 24) & 0xff));
            outputBuffer.Append((byte)((v >> 16) & 0xff));
            outputBuffer.Append((byte)((v >> 8) & 0xff));
            outputBuffer.Append((byte)((v >> 0) & 0xff));
        }

        private void Write(string v)
        {
            for (int i = 0; i < v.Length; i++)
                outputBuffer.Append((byte)v[i]);

            outputBuffer.Append((byte)0);
        }

        internal void WriteInt(int v)
        {
            outputBuffer.Append(BinaryProtocol.INT);
            Write(v);
        }

        internal void WriteDouble(double v)
        {
            outputBuffer.Append(BinaryProtocol.FLOAT);
            Write(union.DoubleToRawLongBits(v));
        }

        internal void WriteSymbol(FtsSymbol v)
        {
            int index = symbolCache.Index(v);

            if (symbolCache[index] == v)
            {
                outputBuffer.Append(BinaryProtocol.SYMBOL_INDEX);
                Write(index);
            }
            else
            {
                symbolCache[index] = v;

                outputBuffer.Append(BinaryProtocol.SYMBOL_CACHE);
                Write(index);
                Write(v.ToString());
            }
        }

        internal void WriteString(string v)
        {
            outputBuffer.Append(BinaryProtocol.STRING);
            Write(v);
        }

        internal void WriteRawString(string v)
        {
            outputBuffer.Append(BinaryProtocol.RAW_STRING);
            Write(v);
        }

        internal void WriteObject(FtsObject v)
        {
            outputBuffer.Append(BinaryProtocol.OBJECT);

            if (v != null)
                Write(v.ID);
            else
                Write(0);
        }

        /* This version is used for object that have predefined IDs */
        internal void WriteObject(int id)
        {
            outputBuffer.Append(BinaryProtocol.OBJECT);
            Write(id);
        }

        internal void WriteAtoms(FtsAtom[] atoms, int offset, int length)
        {
            for (int i = offset; i < length; i++)
            {
                if (atoms[i].IsInt())
                    WriteInt(atoms[i].intValue);
                else if (atoms[i].IsDouble())
                    WriteDouble(atoms[i].doubleValue);
                else if (atoms[i].IsSymbol())
                    WriteSymbol(atoms[i].symbolValue);
                else if (atoms[i].IsString())
                    WriteString(atoms[i].stringValue);
                else if (atoms[i].IsRawString())
                    WriteRawString(atoms[i].stringValue);
                else if (atoms[i].IsObject())
                    WriteObject(atoms[i].objectValue);
            }
        }

        internal void WriteArgs(FtsArgs v)
        {
            WriteAtoms(v.Atoms, 0, v.Length);
        }

        internal void Flush()
        {
            outputBuffer.Append(BinaryProtocol.END_OF_MESSAGE);

            connection.Write(outputBuffer.Bytes, 0, outputBuffer.Length);

            outputBuffer.Clear();
        }

        private OutputBuffer outputBuffer;
        private SymbolCache symbolCache;
        private FtsServerConnection connection;

        private Union union = new Union();
    }
}