//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Collections;

namespace ircam.fts.client
{
    public delegate void FtsMessageHandler(FtsObject obj, FtsArgs args);

    internal class MessageHandlerEntry
    {
        internal MessageHandlerEntry(Type cl, FtsSymbol selector)
        {
            this.cl = cl;
            this.selector = selector;
        }

        public override int GetHashCode()
        {
            return cl.GetHashCode() + selector.GetHashCode();
        }

        public override bool Equals(object obj)
        {
            if (!(obj is MessageHandlerEntry))
                return false;

            return ((MessageHandlerEntry)obj).cl.Equals(cl) && ((MessageHandlerEntry)obj).selector == selector;
        }

        internal Type cl;
        internal FtsSymbol selector;
    }

    [Serializable]
    public class FtsObject
    {
        public const int NO_ID = -1;

        public FtsObject(FtsServer server, FtsObject parent, FtsSymbol ftsClassName)
        {
            this.server = server;
            this.parent = parent;
            encoder = server.Encoder;

            id = server.NewObjectID;
            server[id] = this;

            encoder.WriteObject(FtsServer.CLIENT_OBJECT_ID);
            encoder.WriteSymbol(sNewObject);
            encoder.WriteObject(parent);
            encoder.WriteInt(id);
            encoder.WriteSymbol(ftsClassName);
            encoder.Flush();
        }

        public FtsObject(FtsServer server, FtsObject parent, FtsSymbol ftsClassName, FtsArgs args)
        {
            this.server = server;
            this.parent = parent;
            encoder = server.Encoder;

            id = server.NewObjectID;
            server[id] = this;

            encoder.WriteObject(FtsServer.CLIENT_OBJECT_ID);
            encoder.WriteSymbol(sNewObject);
            encoder.WriteObject(parent);
            encoder.WriteInt(id);
            encoder.WriteSymbol(ftsClassName);
            encoder.WriteArgs(args);
            encoder.Flush();
        }

        public FtsObject(FtsServer server, FtsObject parent, int id)
        {
            this.server = server;
            this.parent = parent;

            encoder = server.Encoder;

            this.id = id;
            if (id != NO_ID)
                server[id] = this;
        }

        public void Send(FtsSymbol selector, FtsArgs args)
        {
            encoder.WriteObject(this);
            encoder.WriteSymbol(selector);
            encoder.WriteArgs(args);
            encoder.Flush();
        }

        public void Send(FtsSymbol selector)
        {
            encoder.WriteObject(this);
            encoder.WriteSymbol(selector);
            encoder.Flush();
        }

        public void SendProperty(FtsArgs args)
        {
            encoder.WriteObject(FtsServer.CLIENT_OBJECT_ID);
            encoder.WriteSymbol(FtsSymbol.Get("set_object_property"));
            encoder.WriteObject(this);
            encoder.WriteArgs(args);
            encoder.Flush();
        }

        /// <summary> Unmap the object in the client object table </summary>
        public virtual void Dispose()
        {
            if (id != NO_ID)
                server.RemoveObject(id);
            server = null;
            parent = null;
            encoder = null;
        }

        public static void RegisterMessageHandler(Type cl, FtsSymbol selector, FtsMessageHandler messageHandler)
        {
            if (selector == null)
                throw new NullReferenceException();

            messageHandlersTable.Add(new MessageHandlerEntry(cl, selector), messageHandler);
        }

        internal static void InvokeMessageHandler(FtsObject obj, FtsSymbol selector, FtsArgs args)
        {
            if (selector == obj.selectorCache)
            {
                obj.messageHandlerCache(obj, args);
                return;
            }

            Type cl = obj.GetType();

            lookupEntry.selector = selector;

            do
            {
                lookupEntry.cl = cl;

                FtsMessageHandler messageHandler = (FtsMessageHandler)messageHandlersTable[lookupEntry];

                if (messageHandler != null)
                {
                    obj.selectorCache = selector;
                    obj.messageHandlerCache = messageHandler;
                    obj.messageHandlerCache(obj, args);
                    return;
                }

                cl = cl.BaseType;
            } while (cl != null);
        }

        public FtsObject Parent
        {
            get
            {
                return parent;
            }
        }

        public FtsServer Server
        {
            get
            {
                return server;
            }
        }

        internal int ID
        {
            get
            {
                return id;
            }
            set
            {
                this.id = value;
            }
        }

        private int id;

        [NonSerialized()]
        private FtsServer server;
        [NonSerialized()]
        private BinaryProtocolEncoder encoder;

        [NonSerialized()]
        private FtsObject parent;

        [NonSerialized()]
        private FtsSymbol selectorCache;
        [NonSerialized()]
        private FtsMessageHandler messageHandlerCache;

        [NonSerialized()]
        private static Hashtable messageHandlersTable = new Hashtable();
        [NonSerialized()]
        private static MessageHandlerEntry lookupEntry = new MessageHandlerEntry(null, null);

        [NonSerialized()]
        private static FtsSymbol sNewObject = FtsSymbol.Get("new_object");
        [NonSerialized()]
        private static FtsSymbol sDelObject = FtsSymbol.Get("delete_object");
    }
}