//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Collections.Generic;
using System.Windows.Forms;

namespace ircam.jmax
{
    /// <summary>This class provide a basic service for registering windows.</summary>
    /// <summary>It export a ListModel with all the Windows on it (may be later
    /// should become a tree Model ?).
    /// Only a registration function is provided, the unregistration
    /// is made by means of a WindowListener.
    /// The Windows can be any Frame.
    /// It provide three window managing/layout functions (tile/tile
    /// vertical/stack/??), derived from the original Enzo Maggi code.
    /// This class is used by the MaxWindowMenu in ircam.jmax.toolkit.
    /// </summary>

    public class MaxWindowManager
    {
        internal int windowOperationCount = 0;

        internal static readonly int SCREENVERT = Screen.PrimaryScreen.Bounds.Height;
        internal static readonly int SCREENHOR = Screen.PrimaryScreen.Bounds.Width;

        private static MaxWindowManager theWindowManager = new MaxWindowManager();

        private List<Form> windows = new List<Form>();

        private static Form topForm = null;

        static public MaxWindowManager WindowManager
        {
            get
            {
                return theWindowManager;
            }
        }

        MaxWindowManager() { }

        /// <summary>Call this method to add a window to 
        /// from  the window manager window list
        /// </summary>

        public void AddWindow(Form window)
        {
            windowOperationCount++;
            windows.Add(window);
        }

        /// <summary>Call this method to remove a window to
        /// from  the window manager window list
        /// </summary>

        public void RemoveWindow(Form window)
        {
            windowOperationCount++;
            windows.Remove(window);
        }

        /// <summary>call this method to tell the window 
        /// manager that something about the window
        /// changed (i.e. the title, for now).
        /// Actually, this will be obsolete the day
        /// we use a swing list as a menu
        /// </summary>

        public void WindowChanged(Form window)
        {
            // Provoke a null change in the vector,
            // to get an event for the listeners

            windowOperationCount++;

            if (windows.Contains(window))
                windows[windows.IndexOf(window)] = window;
        }

        public List<Form> WindowList
        {
            get
            {
                return windows;
            }
        }

        public Form[] WindowListArrayCopy
        {
            get
            {
                Form[] array = new Form[windows.Count];
                windows.CopyTo(array);
                return array;
            }
        }

        public int WindowOperationCount
        {
            get
            {
                return windowOperationCount;
            }
        }

        /// <summary> Return a Frame considered top level, for dialogs that have no 
        /// "natural" parent frame; it is usually set to the console,
        /// for the current console centered interaction.
        /// </summary>

        public static Form TopForm
        {
            get
            {
                return topForm;
            }
            set
            {
                topForm = value;
            }
        }

        private static Form currentForm = null;
        public static Form CurrentForm
        {
            get
            {
                if (currentForm != null)
                    return currentForm;
                else
                    return topForm;
            }
        }

        /// <summary>Produce a unique window title for the Name 
        /// the technique is the following (naive, but usefull)
        /// it look all the window titles, and count those
        /// which start exactly with theName; finally,
        /// add this count to theName string and return it 
        /// </summary>

        public string MakeUniqueWindowTitle(string theName)
        {
            int count = 0;

            for (int i = 0; i < windows.Count; i++)
            {
                Form w = (Form)windows[i];

                if (w.Text == theName)
                    count++;
            }

            if (count > 0)
                return theName + " (" + count + ")";
            else
                return theName;
        }

        // Event handler for WindowListener

        public void WindowClosing(object sender, System.EventArgs e) { }

        public void WindowOpened(object sender, System.EventArgs e) { }

        public void WindowClosed(object sender, System.EventArgs e)
        {
            if (currentForm == (Form)sender)
                currentForm = null;
            RemoveWindow((Form)sender);
        }

        public void WindowIconified(object sender, System.EventArgs e)
        {
            if (currentForm == (Form)sender)
                currentForm = null;
        }

        public void WindowDeiconified(object sender, System.EventArgs e) { }

        public void WindowActivated(object sender, System.EventArgs e)
        {
            currentForm = (Form)sender;
        }

        public void WindowDeactivated(object sender, System.EventArgs e)
        {
            if (currentForm == (Form)sender)
                currentForm = null;
        }

        public void TileWindows()
        {
            int width;
            int height;
            int initX, x;
            int initY, y;

            Form window;
            bool changHor = false;
            int z, j;

            int nWindows = windows.Count;

            if (nWindows != 0)
            {
                z = 1;

                if (nWindows == 1)
                    j = 1;
                else
                    j = 2;

                while (z * j < nWindows)
                {
                    z++;

                    if (z * j < nWindows)
                        j++;
                }

                height = (int)System.Math.Floor((double)SCREENVERT / j) - 35;
                width = (int)System.Math.Floor((double)SCREENHOR / z) - 10;
                initX = 7;
                initY = 20;
                x = initX;
                y = initY;

                for (int k = 0; k < nWindows; k++)
                {
                    int res;

                    window = (Form)windows[k];

                    res = (int)System.Math.Floor((double)k / j);

                    if ((res * j) == k)
                        changHor = false;

                    if ((res != 0) && (!changHor))
                    {
                        x = initX + (width + 7) * res;
                        y = initY;
                        changHor = true;
                    }
                    else if (k > 0)
                        y += height + 25;

                    window.Location = new Point(x, y);
                }
            }
        }

        public void StackWindows()
        {
            Form window;
            int x, y;
            int height, width;

            if (windows.Count == 0)
                return;

            x = 50;
            y = 50;

            for (int i = 0; i < windows.Count; i++)
            {
                window = (Form)windows[i];

                window.Location = new Point(x, y);
                window.BringToFront();

                x += 20;
                y += 20;
            }
        }
    }
}