//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Collections;
using System.Collections.Generic;

using ircam.fts.client;
using ircam.jmax;
using ircam.jmax.fts;
using ircam.jmax.editors.patcher.objects;

namespace ircam.jmax.editors.patcher
{
    /// <summary>A delegate for actions intended to be mapped on Connections sets.
    /// Work on selections and on the whole displayList 
    /// </summary>
    public delegate void ProcessConnection(GraphicConnection obj);
    public delegate void ProcessObject(GraphicObject obj);    

    /// <summary>This class represent the display list of the patcher editor.
    /// It keep the data base of graphic objects, handle the paiting,
    /// and handle all the geometric requests, like which object this point
    /// belong to ?
    /// It also handle the selection.
    /// </summary>

    public class DisplayList
    {
        internal ErmesSketchPad sketch;

        internal const int NO_DRAG = 0;
        internal const int DRAG_RECTANGLE = 1;
        internal const int DRAG_LINE = 2;

        internal int dragMode;
        internal Rectangle dragRectangle = new Rectangle();
        internal Point lineStart = new Point();
        internal Point lineEnd = new Point();

        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   Constructors                                             //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////

        internal DisplayList(ErmesSketchPad sketch)
        {
            this.sketch = sketch;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   DISPLAY OBJECT DATABASE HANDLING                         //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////

        private MaxVector displayObjects = new MaxVector();

        /* Objects */

        internal void Add(GraphicObject obj)
        {
            displayObjects.AddElement(obj);
        }

        private void AddToBeginning(GraphicObject obj)
        {
            displayObjects.InsertElementAt(obj, 0);
        }

        public void Remove(GraphicObject obj)
        {
            displayObjects.RemoveElement(obj);
        }

        public void Remove(FtsGraphicObject obj)
        {
            Remove(GetGraphicObjectFor(obj));
        }

        internal GraphicObject GetGraphicObjectFor(FtsGraphicObject obj)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicObject)
                {
                    GraphicObject graphicObject = (GraphicObject)values[i];

                    if (graphicObject.FtsObject == obj)
                        return graphicObject;
                }

            return null;
        }

        internal void DisposeAllObjects()
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicObject)
                    ((GraphicObject)values[i]).Dispose();
        }

        // Generic operation on objects in the display List

        public void ApplyToObjects(ProcessObject action)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicObject)
                    action((GraphicObject)values[i]);
        }

        public void ObjectToFront(GraphicObject obj)
        {
            Remove(obj);
            Add(obj);
            ReassignLayers();
            SortDisplayList();
        }

        public void ObjectToBack(GraphicObject obj)
        {
            Remove(obj);
            AddToBeginning(obj);
            ReassignLayers();
            SortDisplayList();
        }

        public void ObjectsToFront(object[] objects, int size)
        {
            GraphicObject obj;

            SortVector(objects, size);
            for (int i = 0; i < size; i++)
            {
                obj = (GraphicObject)objects[i];
                Remove(obj);
                Add(obj);
            }
            ReassignLayers();
            SortDisplayList();
            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //sketch.repaint();
        }

        public void objectsToBack(object[] objects, int size)
        {
            GraphicObject obj;

            SortVector(objects, size);
            for (int i = size - 1; i >= 0; i--)
            {
                obj = (GraphicObject)objects[i];
                Remove(obj);
                AddToBeginning(obj);
            }
            ReassignLayers();
            SortDisplayList();
            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //sketch.Repaint();
        }

        /* Connections */

        public void Add(GraphicConnection connection)
        {
            displayObjects.AddElement(connection);
        }

        public void Remove(GraphicConnection connection)
        {
            displayObjects.RemoveElement(connection);
        }

        public void Remove(FtsConnection connection)
        {
            Remove(GetGraphicConnectionFor(connection));
            ReassignLayers();
        }

        internal GraphicConnection GetGraphicConnectionFor(FtsConnection c)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if (connection.FtsConnection == c)
                        return connection;
                }

            return null;
        }

        public bool ThisConnectionExist(GraphicObject src, int outlet, GraphicObject dst, int inlet)
        {
            GraphicConnection connection;
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == src) && (connection.Outlet == outlet) && (connection.DestObject == dst) && (connection.Inlet == inlet))
                        return true;
                }
            return false;
        }

        public void RedrawConnectionsFor(GraphicObject obj)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == obj) || (connection.DestObject == obj))
                        connection.redraw();
                }
        }

        public List<GraphicConnection> GetConnectionsFor(GraphicObject obj)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;
            List<GraphicConnection> connections = new List<GraphicConnection>();

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == obj) || (connection.DestObject == obj))
                        connections.Add(connection);
                }
            return connections;
        }

        public void UpdateConnectionsFor(GraphicObject obj)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == obj) || (connection.DestObject == obj))
                        connection.updateDimensions();
                }
        }

        public void DeleteConnectionsForObject(GraphicObject obj)
        {
            MaxVector toDelete = new MaxVector();
            object[] values;
            int size;

            values = displayObjects.ObjectArray;
            size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == obj) || (connection.DestObject == obj))
                        toDelete.AddElement(connection);
                }

            values = toDelete.ObjectArray;
            size = toDelete.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    connection.redraw();
                    connection.Delete();
                }
        }

        public void ReleaseConnectionsForObject(GraphicObject obj)
        {
            MaxVector toDelete = new MaxVector();
            object[] values;
            int size;

            values = displayObjects.ObjectArray;
            size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == obj) || (connection.DestObject == obj))
                        toDelete.AddElement(connection);
                }

            values = toDelete.ObjectArray;
            size = toDelete.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    connection.redraw();
                    connection.release();
                }
        }

        public void DeleteConnectionsForOutlet(GraphicObject obj, int pos)
        {
            MaxVector toDelete = new MaxVector();
            object[] values;
            int size;

            values = displayObjects.ObjectArray;
            size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.SourceObject == obj) && (connection.Outlet == pos))
                        toDelete.AddElement(connection);
                }

            values = toDelete.ObjectArray;
            size = toDelete.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    connection.redraw();
                    connection.Delete();
                }
        }

        public void DeleteConnectionsForInlet(GraphicObject obj, int pos)
        {
            MaxVector toDelete = new MaxVector();
            object[] values;
            int size;

            values = displayObjects.ObjectArray;
            size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    if ((connection.DestObject == obj) && (connection.Inlet == pos))
                        toDelete.AddElement(connection);
                }

            values = toDelete.ObjectArray;
            size = toDelete.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                {
                    GraphicConnection connection = (GraphicConnection)values[i];

                    connection.redraw();
                    connection.Delete();
                }
        }

        // Generic operation on objects in the display List

        public void ApplyToConnections(ProcessConnection action)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicConnection)
                    action((GraphicConnection)values[i]);
        }


        // Magic sorting for objects and connections
        // essentially, keep objects are they are
        // and move the connections
        // Must be called by hand in the proper case, beacause calling
        // it automatically will call it just too many time in situations
        // like creating a patch and pasting ...

        public void ReassignLayers()
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;
            int layer = 0;

            for (int i = 0; i < size; i++)
                if (values[i] is GraphicObject)
                    ((GraphicObject)values[i]).Layer = layer++;
        }

        internal void SortVector(object[] values, int size)
        {
            for (int i = 0; i < size; i++)
                for (int j = 0; j < i; j++)
                    if (IsAfter(values[j], values[i]))
                    {
                        object v;

                        v = values[j];
                        values[j] = values[i];
                        values[i] = v;
                    }
        }

        public void SortDisplayList()
        {
            SortVector(displayObjects.ObjectArray, displayObjects.Size);
        }

        private bool IsAfter(object do1, object do2)
        {
            if (do1 is GraphicObject)
            {
                GraphicObject object1 = (GraphicObject)do1;

                if (do2 is GraphicObject)
                {
                    GraphicObject object2 = (GraphicObject)do2;

                    return (object1.Layer > object2.Layer);
                }
                else if (do2 is GraphicConnection)
                {
                    GraphicConnection connection2 = (GraphicConnection)do2;

                    return (object1.Layer >= connection2.SourceObject.Layer || object1.Layer >= connection2.DestObject.Layer);
                }
                else
                    return false;
            }
            else if (do1 is GraphicConnection)
            {
                GraphicConnection connection1 = (GraphicConnection)do1;

                if (do2 is GraphicObject)
                {
                    GraphicObject object2 = (GraphicObject)do2;

                    return (object2.Layer < connection1.SourceObject.Layer && object2.Layer < connection1.DestObject.Layer);
                }
                else if (do2 is GraphicConnection)
                {
                    GraphicConnection connection2 = (GraphicConnection)do2;

                    return ((connection1.SourceObject.Layer > connection2.DestObject.Layer || connection1.SourceObject.Layer > connection2.SourceObject.Layer) && (connection1.DestObject.Layer > connection2.DestObject.Layer || connection1.DestObject.Layer > connection2.SourceObject.Layer));
                }
                else
                    return false;
            }
            else
                return false;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   DISPLAY OBJECT BASIC GEOMETRICAL QUERIS                  //
        //                    AND DISPLAY LIST MANIPULATIONS                          //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////


        public SensibilityArea GetSensibilityAreaAt(int x, int y)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;
            SensibilityArea candidateArea = null;
            /* Look in the reverse order from outside to inside,
            to be consistent with the graphic look */

            for (int i = (size - 1); i >= 0; i--)
            {
                IDisplayObject obj = (IDisplayObject)values[i];
                SensibilityArea area = obj.GetSensibilityAreaAt(x, y);

                if (area != null)
                {
                    if (area.isTransparent)
                    {
                        if (candidateArea == null)
                            candidateArea = area;
                        else
                        {
                            if (candidateArea.Cost > area.Cost)
                            {
                                candidateArea.Dispose();
                                candidateArea = area;
                            }
                        }
                    }
                    else
                        return area;
                }
            }

            // If nothing returned until now, return the candidate area if any

            return candidateArea;
        }

        //colled during connection and in runMode
        public virtual SensibilityArea GetObjectSensibilityAreaAt(int x, int y)
        {
            object[] values = displayObjects.ObjectArray;
            int size = displayObjects.Size;
            SensibilityArea candidateArea = null;
            /* Look in the reverse order from outside to inside,
            to be consistent with the graphic look */

            for (int i = (size - 1); i >= 0; i--)
            {
                IDisplayObject obj = (IDisplayObject)values[i];

                if (obj is GraphicObject)
                {

                    SensibilityArea area = obj.GetSensibilityAreaAt(x, y);

                    if (area != null)
                    {
                        if (area.isTransparent)
                        {
                            if (candidateArea == null)
                                candidateArea = area;
                            else
                            {
                                if (candidateArea.Cost > area.Cost)
                                {
                                    candidateArea.Dispose();
                                    candidateArea = area;
                                }
                            }
                        }
                        else
                        {
                            return area;
                        }
                    }
                }
            }
            // If nothing returned until now, return the candidate area if any
            return candidateArea;
        }

        ////////////////////////////////////////////////////////////
        //
        //                  UPDATING 
        //
        ///////////////////////////////////////////////////////////
        
        internal MaxVector updateObjects = new MaxVector();

        public void AddToUpdate(GraphicObject obj)
        {
            updateObjects.AddElement(obj);
        }

        public void ResetUpdateObjects()
        {
            updateObjects.RemoveAllElements();
        }

        public IEnumerator GetUpdateObjects()
        {
            return updateObjects.Elements();
        }

        internal void UpdatePaint(Graphics g)
        {
            object[] values = updateObjects.ObjectArray;
            for (int i = 0; i < updateObjects.Size; i++)
                ((IDisplayObject)values[i]).UpdatePaint(g);
        }

        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   PAINTING                                                 //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////

        internal bool doneOnce = false;

        public void Paint(Graphics g)
        {
            object[] values = displayObjects.ObjectArray;

            // Very First, if this is the first paint for the window,
            // we do a fake paint with the textRenderer (doesn't work
            // before its first paint) and then we update the dimension
            // of all the editable objects; 
            // and, of course, update for the connections.

            if (!doneOnce)
            {
                doneOnce = true;

                for (int i = 0; i < displayObjects.Size; i++)
                {
                    IDisplayObject obj = (IDisplayObject)values[i];

                    if (obj is Editable)
                    {
                        ((Editable)obj).DrawContent(g);
                        ((Editable)obj).UpdateDimensionsNoConnections(); // HACK ? Yes !
                    }
                }

                for (int i = 0; i < displayObjects.Size; i++)
                {
                    IDisplayObject obj = (IDisplayObject)values[i];

                    if (obj is GraphicConnection)
                        ((GraphicConnection)obj).updateDimensions();
                }
            }

            // First, paint the background

            Rectangle clip = Rectangle.Truncate(g.ClipBounds);

            Color setColor = (sketch.IsLocked ?
                Settings.sharedInstance.LockBackgroundColor :
                Settings.sharedInstance.EditBackgroundColor);

            g.FillRectangle(new SolidBrush(setColor), clip.X, clip.Y, clip.Width, clip.Height);

            // Second, paint the display objects in the right order

            for (int i = 0; i < displayObjects.Size; i++)
            {
                IDisplayObject obj = (IDisplayObject)values[i];

                if (obj.Intersects(clip))
                    obj.Paint(g);
            }

            // Finally, draw the Drag/effermeral thingies
            // Leave the clipping do the clip work; if the drag thingies are
            // active are probabily always repaint.

            switch (dragMode)
            {
                case DRAG_RECTANGLE:
                    g.DrawRectangle(new Pen(Color.Black), dragRectangle.X, dragRectangle.Y, dragRectangle.Width, dragRectangle.Height);
                    break;

                case DRAG_LINE:
                    g.DrawLine(new Pen(Color.Black), lineStart.X, lineStart.Y, lineEnd.X, lineEnd.Y);
                    break;

                case NO_DRAG:
                    break;
            }
        }

        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   SELECTION HANDLING                                       //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////

        /* The following code use a apply style of programming with static
        objects, instantiated once, to avoid:
        1- depending on the display list data structure, that is going to change
        fast.
        2- Avoid allocating new objects for all the operations, including enumeration
        or object/connection Actions.
        */

        ProcessConnection connectionSelecter = delegate(GraphicConnection connection)
        {
            if ((connection.SourceObject.isSelected) && (connection.DestObject.isSelected))
                ErmesSelection.patcherSelection.select(connection);
            else
                ErmesSelection.patcherSelection.deselect(connection);
        };

        private bool toggle;
        public void SelectExactly(Rectangle rect)
        {
            toggle = false;
            ApplyToObjects(
                delegate(GraphicObject obj)
                {
                    if (obj.coreIntersects(rect))
                    {
                        if (obj.isSelected)
                        {
                            if (toggle)
                            {
                                ErmesSelection.patcherSelection.deselect(obj);
                                obj.Redraw();
                            }
                        }
                        else
                        {
                            ErmesSelection.patcherSelection.select(obj);
                            obj.Redraw();
                        }
                    }
                    else
                    {
                        if (obj.isSelected)
                        {
                            if (!toggle)
                            {
                                ErmesSelection.patcherSelection.deselect(obj);
                                obj.Redraw();
                            }
                        }
                    } 
                });
            ApplyToConnections(connectionSelecter);
        }

        public void ToggleSelect(Rectangle rect)
        {
            toggle = true;
            ApplyToObjects(
                delegate(GraphicObject obj)
                {
                    if (obj.coreIntersects(rect))
                    {
                        if (obj.isSelected)
                        {
                            if (toggle)
                            {
                                ErmesSelection.patcherSelection.deselect(obj);
                                obj.Redraw();
                            }
                        }
                        else
                        {
                            ErmesSelection.patcherSelection.select(obj);
                            obj.Redraw();
                        }
                    }
                    else
                    {
                        if (obj.isSelected)
                        {
                            if (!toggle)
                            {
                                ErmesSelection.patcherSelection.deselect(obj);
                                obj.Redraw();
                            }
                        }
                    }
                });
            ApplyToConnections(connectionSelecter);
        }
        
        ProcessObject allObjectSelecter = delegate(GraphicObject obj)
        {
            ErmesSelection.patcherSelection.select(obj);
        };

        ProcessConnection allConnectionSelecter = delegate(GraphicConnection connection)
        {
            ErmesSelection.patcherSelection.select(connection);
        };

        // This one don't do redraws ... should it ?

        public void SelectAll()
        {
            ApplyToObjects(allObjectSelecter);
            ApplyToConnections(allConnectionSelecter);
        }


        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   GEOMETRICAL COMPUTATIONS                                 //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////


        /// <summary>Compute the rectangle including all the objects, on a rectagle passed as object </summary>

        public Rectangle getBounds(Rectangle r)
        {
            // Note that the bounds always include the 0.0 corner
            // this is done on purpose, it is not a bug.

            r.X = 0;
            r.Y = 0;
            r.Height = 0;
            r.Width = 0;

            ApplyToObjects(
                delegate(GraphicObject obj)
                {
                    obj.rectangleUnion(r);
                });

            return r;
        }

        /// <summary>Move all the objects to positive coordinates, and put in the rectangle
        /// passed as argument, a dimension that is suitable for a container for the patch.
        /// Return true if a coordinate change has been done, false otherwise.
        /// </summary>

        internal void MoveAllBy(int dx, int dy)
        {
            ApplyToObjects(
                delegate(GraphicObject obj)
                {
                    obj.moveBy(dx, dy);
                });
        }

        ////////////////////////////////////////////////////////////////////////////////
        //                                                                            //
        //                   TRANSIENT OBJECT DISPLAY                                 //
        //                                                                            //
        ////////////////////////////////////////////////////////////////////////////////

        // The Display List also handle a additional DRAG rectangle,
        // on top of the other stuff; the idea is to use it for select and resize
        // applications; 

        /// <summary>Set the display list drag rectagle; accept a non normalized rectangle,
        /// i.e. one with negative width or height, and flip it correctly
        /// </summary>

        public void SetDragRectangle(int x, int y, int width, int height)
        {
            if (height < 0)
            {
                height = (-1) * height;
                y = y - height;
            }

            if (width < 0)
            {
                width = (-1) * width;
                x = x - width;
            }

            dragRectangle.X = x;
            dragRectangle.Y = y;
            dragRectangle.Width = width;
            dragRectangle.Height = height;
        }

        public void SetDragLine(int x, int y, int x2, int y2)
        {
            // Set the drag rectangle to the rectangle corresponding to
            // the line, in order to be able to use intersect with the
            // clipping area during repaint.

            if (x < x2)
            {
                dragRectangle.X = x;
                dragRectangle.Width = x2 - x;
            }
            else
            {
                dragRectangle.X = x2;
                dragRectangle.Width = x - x2;
            }

            if (y < y2)
            {
                dragRectangle.Y = y;
                dragRectangle.Height = y2 - y;
            }
            else
            {
                dragRectangle.Y = y2;
                dragRectangle.Height = y - y2;
            }

            // Store the two ending points 

            lineStart.X = x;
            lineStart.Y = y;

            lineEnd.X = x2;
            lineEnd.Y = y2;
        }

        public Rectangle GetDragRectangle()
        {
            return dragRectangle;
        }

        public void DragRectangle()
        {
            dragMode = DRAG_RECTANGLE;
        }

        public void DragLine()
        {
            dragMode = DRAG_LINE;
        }

        public bool IsDragLine
        {
            get
            {
                return (dragMode == DRAG_LINE);
            }
        }

        public void NoDrag()
        {
            dragMode = NO_DRAG;
        }

        public void RedrawDragRectangle()
        {
            /* If the rectangle is 'big', we issue four repaint,
            corresponding to the four side of the rectangle
            */

            if ((dragRectangle.Width > 50) && (dragRectangle.Height > 50))
            {
                //HACK: 'javax.swing.JComponent.repaint()' was not converted.
                //sketch.repaint(dragRectangle.X, dragRectangle.Y, 1, dragRectangle.Height + 1);
                //sketch.repaint(dragRectangle.x, dragRectangle.y, dragRectangle.width + 1, 1);
                //sketch.repaint(dragRectangle.x, dragRectangle.y + dragRectangle.height, dragRectangle.width + 1, 1);
                //sketch.repaint(dragRectangle.x + dragRectangle.width, dragRectangle.y, 1, dragRectangle.height + 1);
            }
            else
            {
                //HACK: 'javax.swing.JComponent.repaint()' was not converted.
                //sketch.repaint(dragRectangle.x, dragRectangle.y, dragRectangle.width + 1, dragRectangle.height + 1);
            }
        }

        public void RedrawDragLine()
        {
            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //sketch.repaint(dragRectangle.x, dragRectangle.y, dragRectangle.width + 1, dragRectangle.height + 1);
        }
    }
}