//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;
using System.Drawing;
using System.Collections;
using System.Collections.Generic;

using ircam.jmax;
using ircam.jmax.dialogs;
using ircam.jmax.fts;
using ircam.jmax.editors.patcher.objects;
using ircam.jmax.editors.patcher.interactions;
using ircam.fts.client;

using ircam.jmax.toolkit;

namespace ircam.jmax.editors.patcher
{
    /// <summary>The graphic workbench for the patcher editor.
    /// It handles the interaction of the user with the objects,
    /// propagates the mouse events to the objects themselves,
    /// allow the selection, moving, erasing, resizing of objects.
    /// It keeps track of the toolBar state, it handles the 
    /// offscreen and much, much more...
    /// </summary>

    public class ErmesSketchPad : IEditor, IFtsUpdateGroupListener
    {
        public static FtsUpdateGroup updateGroup;

        private void updateGroupCreate()
        {
            try
            {
                updateGroup = new FtsUpdateGroup();
                updateGroup.start();
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("[ErmesSketchPad]: Error in FtsUpdateGroup creation!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        private bool somethingToUpdate = false;
        private Rectangle invalid = new Rectangle();

        public void paintAtUpdateEnd(GraphicObject obj, int x, int y, int w, int h)
        {
            displayList.AddToUpdate(obj);

            if (!somethingToUpdate)
            {
                somethingToUpdate = true;

                invalid = new Rectangle(x, y, w, h);
            }
            else
            {
                invalid = Rectangle.Union(new Rectangle(x, y, w, h), invalid);
            }
        }

        public void resetUpdate()
        {
            displayList.ResetUpdateObjects();
            somethingToUpdate = false;
        }

        public void UpdateGroupStart()
        {
            resetUpdate();
        }

        public void UpdateGroupEnd()
        {
            Rectangle rect = IEditorContainer.ViewRectangle;
            if (rect.IsEmpty) return;
            Graphics gr;

            if (IsLocked)
            {
                //HACK: 'java.awt.Component.getGraphics()' was not converted.
                //                gr = getGraphics();

                invalid = Rectangle.Union(new Rectangle(rect.X, rect.Y, rect.Width, rect.Height), invalid);

                //HACK: 'java.awt.Graphics.setClip()' was not converted.
                //                gr.setClip(invalid);
                //                displayList.updatePaint(gr);
            }
            else
            {
                //HACK: 'javax.swing.JComponent.repaint()' was not converted.
                //repaint(invalid);
            }
        }

        // ------------------------------------------------

        private DisplayList displayList;

        public DisplayList DisplayList
        {
            get
            {
                return displayList;
            }
        }

        private KeyMap keyMap;

        internal KeyMap KeyMap
        {
            get
            {
                return keyMap;
            }
        }

        private IEditorContainer itsEditorContainer;

        public IEditorContainer IEditorContainer
        {
            get
            {
                return itsEditorContainer;
            }
        }

        private FtsPatcherObject itsPatcher;

        public FtsPatcherObject FtsPatcher
        {
            get
            {
                return itsPatcher;
            }
        }

        public bool isARootPatcher()
        {
            return FtsPatcher.Parent == JMaxLibrary.RootPatcher;
        }

        public bool isASubPatcher
        {
            get
            {
                return !itsPatcher.IsARootPatcher;
            }
        }

        public bool isATemplate
        {
            get
            {
                return (itsPatcher is FtsTemplateObject);
            }
        }

        // ---------------------------------------------------------------------
        // font handling
        // --------------------------------------------------------------------

        private string defaultFontName;
        private int defaultFontStyle;
        private float defaultFontSize;

        public string DefaultFontName
        {
            get
            {
                return defaultFontName;
            }
            set
            {
                defaultFontName = value;
            }
        }

        public float DefaultFontSize
        {
            get
            {
                return defaultFontSize;
            }
            set
            {
                defaultFontSize = value;
            }

        }

        public int DefaultFontStyle
        {
            get
            {
                return defaultFontStyle;
            }
            set
            {
                defaultFontStyle = value;
            }
        }

        public void changeDefaultFontStyle(string style, bool selected)
        {
            if (style.Equals("Bold"))
            {
                if ((defaultFontStyle == (int)FontStyle.Bold || defaultFontStyle == (int)FontStyle.Bold + (int)FontStyle.Italic) && (!selected))
                    defaultFontStyle -= (int)FontStyle.Bold;
                else if ((defaultFontStyle != (int)FontStyle.Bold && defaultFontStyle != (int)FontStyle.Bold + (int)FontStyle.Italic) && (selected))
                    defaultFontStyle += (int)FontStyle.Bold;
            }
            else if (style.Equals("Italic"))
            {
                if ((defaultFontStyle == (int)FontStyle.Italic || defaultFontStyle == (int)FontStyle.Bold + (int)FontStyle.Italic) && (!selected))
                    defaultFontStyle -= (int)FontStyle.Italic;
                else if ((defaultFontStyle != (int)FontStyle.Italic && defaultFontStyle != (int)FontStyle.Bold + (int)FontStyle.Italic) && (selected))
                    defaultFontStyle += (int)FontStyle.Italic;
            }
        }

        private bool automaticFitToText = false;

        public bool isAutomaticFitToText
        {
            get
            {
                return automaticFitToText;
            }
            set
            {
                automaticFitToText = value;
            }
        }

        // ---------------------------------------------------------
        // cut/copy/paste variables and methods
        // ---------------------------------------------------------
        private int incrementalPasteOffsetX;
        private int incrementalPasteOffsetY;
        private int numberOfPaste = 0;
        private GraphicObject anOldPastedObject = null;
        private int startPasteX = -1;
        private int startPasteY = -1;
        private int lastCopyCount;
        private MaxVector pastedObjects = new MaxVector();
        private MaxVector pastedConnections = new MaxVector();

        internal int PasteNumber
        {
            get
            {
                return ++numberOfPaste;
            }
            set
            {
                numberOfPaste = value;
            }
        }

        internal GraphicObject OldPastedObject
        {
            get
            {
                return anOldPastedObject;
            }

            set
            {
                anOldPastedObject = value;
            }
        }

        internal void setStartPasteXY(int x, int y)
        {
            startPasteX = x;
            startPasteY = y;
        }

        internal int StartPasteX
        {
            get
            {
                return startPasteX;
            }
        }

        internal int StartPasteY
        {
            get
            {
                return startPasteY;
            }
        }

        public void AddPastedObject(GraphicObject obj)
        {
            pastedObjects.AddElement(obj);
            if (undoing)
                addUndoRedoObject(obj);
        }

        public void AddPastedConnection(GraphicConnection c)
        {
            pastedConnections.AddElement(c);
        }

        internal void setIncrementalPasteOffsets(int offsetX, int offsetY)
        {
            incrementalPasteOffsetX = offsetX;
            incrementalPasteOffsetY = offsetY;
        }

        internal int PasteOffsetX
        {
            get
            {
                return incrementalPasteOffsetX;
            }
        }

        internal int PasteOffsetY
        {
            get
            {
                return incrementalPasteOffsetY;
            }
        }

        internal int PasteDX
        {
            get
            {
                return numberOfPaste * incrementalPasteOffsetX;
            }
        }

        internal int PasteDY
        {
            get
            {
                return numberOfPaste * incrementalPasteOffsetY;
            }
        }

        internal int LastCopyCount
        {
            get
            {
                return lastCopyCount;
            }
            set
            {
                lastCopyCount = value;
            }
        }

        private bool pasting = false;
        public void startPaste()
        {
            pasting = true;
        }

        public void EndPaste()
        {
            pasting = false;

            ErmesSelection.patcherSelection.DeselectAll();
            for (IEnumerator e = pastedObjects.Elements(); e.MoveNext(); )
            {
                ErmesSelection.patcherSelection.select((GraphicObject)e.Current);
            }
            for (IEnumerator e = pastedConnections.Elements(); e.MoveNext(); )
            {
                ErmesSelection.patcherSelection.select((GraphicConnection)e.Current);
            }

            OldPastedObject = ErmesSelection.patcherSelection.getSingleton();
            if ((startPasteX == -1) || (startPasteY == -1))
                setStartPasteXY(anOldPastedObject.X, anOldPastedObject.Y);

            pastedObjects.RemoveAllElements();
            pastedConnections.RemoveAllElements();

            fixSize();

            // (fd) if only one object pasted, then edit it with all text selected
            if (ErmesSelection.patcherSelection.isSingleton())
            {
                GraphicObject obj = (GraphicObject)ErmesSelection.patcherSelection.getSingleton();

                if ((obj is GraphicObject) && !undoing)
                {
                    ErmesSelection.patcherSelection.DeselectAll();

                    //textEditObject((Editable)obj, Empty);
                    //UNDONE: Method 'javax.swing.text.JTextComponent.selectAll' was not converted.
                    //                    EditField.selectAll();
                }
            }

            undoing = false;

            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //repaint(invalid);
        }

        //--------------------------------------------------------
        //    CONSTRUCTOR
        //--------------------------------------------------------
        internal ErmesSketchPad(ErmesSketchWindow container, FtsPatcherObject thePatcher)
        {
            updateGroupCreate();

            string s;

            // Setting the local variables
            //itsEditorContainer = container;

            itsPatcher = thePatcher;

            // Get the defaultFontName and Size
            defaultFontName = (string)JMaxLibrary.EnvironmentVariables["maxDefaultFont"];

            if ((object)defaultFontName == null)
                defaultFontName = null;

            s = (string)JMaxLibrary.EnvironmentVariables["maxDefaultFontSize"];

            if ((object)s == null)
                defaultFontSize = 0;
            else
                defaultFontSize = Int32.Parse(s);


            defaultFontStyle = 0;

            // Install the display List object

            displayList = new DisplayList(this);

            engine = new InteractionEngine(this);
            keyMap = new KeyMap();

            // Next two temporary (mdc)

            //UNDONE: Method 'javax.swing.JComponent.setOpaque' was not converted. 
            //            setOpaque(true);
            //UNDONE: Method 'java.awt.Container.setLayout' was not converted. 
            //            setLayout(null);

            //itsEditField = new EditField(this);
            //UNDONE: Method 'java.awt.Container.add' was converted to 'System.Windows.Forms.ContainerControl.Controls.Add' which has a different behavior.
            //            Controls.Add(itsEditField);

            fixSize();

            //            KeyEventsManager.addProducer(this);

            //UNDONE: Method 'javax.swing.JComponent.requestDefaultFocus' was not converted. 
            //            equestDefaultFocus();

            updateGroup.Add(this);

            initUndoStuff();
        }

        private float sx, sy;
        public void scale(float scaleX, float scaleY)
        {
            sx = scaleX;
            sy = scaleY;
            DisplayList.ApplyToObjects(
                delegate(GraphicObject obj)
                {
                    obj.Redraw();
                    obj.redrawConnections();
                    obj.scale(sx, sy);
                    obj.Redraw();
                    obj.redrawConnections();
                });
        }


        /* To be called to fix the sketchpad size after some changes (move
        and paste); it also assure the selection is in the scrolled area. */

        internal static Rectangle totalBounds = new Rectangle();
        internal static Size preferredSize = new Size();

        public void fixSize()
        {
            bool redraw = false;

            totalBounds.X = 0;
            totalBounds.Y = 0;
            totalBounds.Width = 0;
            totalBounds.Height = 0;

            displayList.getBounds(totalBounds);

            if ((totalBounds.X < 0) || (totalBounds.Y < 0))
            {
                int dx, dy;

                if (totalBounds.X < 0)
                    dx = (-1) * totalBounds.X;
                else
                    dx = 0;

                if (totalBounds.Y < 0)
                    dy = (-1) * totalBounds.Y;
                else
                    dy = 0;

                displayList.MoveAllBy(dx, dy);

                redraw = true;
            }

            // If the objects bounds are bigger than the current sketch bounds
            // resize the sketch to be 5 points bigger than this size

            //UNDONE: Method 'javax.swing.JComponent.getWidth' was not converted.
            //UNDONE: Method 'javax.swing.JComponent.getHeight' was not converted.
            //            if ((totalBounds.Width > getWidth()) || (totalBounds.Height > getHeight()))
            //            {
            //                preferredSize.Height = totalBounds.Height + 5;
            //                preferredSize.Width = totalBounds.Width + 5;
            //                
            //                // Test to avoid windows too small 
            //                
            //                if (preferredSize.Height < 20)
            //                    preferredSize.Height = 20;
            //                
            //                if (preferredSize.Width < 20)
            //                    preferredSize.Width = 20;
            //                
            //UNDONE: Method 'javax.swing.JComponent.setPreferredSize' was not converted.
            //                setPreferredSize(preferredSize);
            //            }

            if (redraw)
                this.Redraw();

            //UNDONE: Method 'javax.swing.JComponent.revalidate' was not converted.
            //            revalidate(); // ????
        }

        // ----------------------------------------------------------------
        // make the title for the container (window)
        // ----------------------------------------------------------------

        internal string Title
        {
            get
            {
                string name;

                if (itsPatcher.IsARootPatcher)
                {
                    name = itsPatcher.Name;
                    if ((object)name == null)
                        name = "untitled";
                }
                else if (itsPatcher is FtsTemplateObject)
                    name = "template " + itsPatcher.Description;
                else if (itsPatcher is FtsPatcherObject)
                    name = "patcher " + itsPatcher.Description;
                else
                    name = "template " + itsPatcher.Description;
                return name;
            }
        }

        // ----------------------------------------------------------------
        // scrolling support
        // ----------------------------------------------------------------

        public bool pointIsVisible(Point point, int margin)
        {
            Rectangle r = itsEditorContainer.ViewRectangle;

            return ((point.X > r.X + margin) && (point.X < r.X + r.Width - margin)
                && (point.Y > r.Y + margin) && (point.Y < r.Y + r.Height - margin));
        }

        public void scrollBy(int dx, int dy)
        {
            Rectangle r = itsEditorContainer.ViewRectangle;

            r.X += dx;
            r.Y += dy;

            //UNDONE: Method 'javax.swing.JComponent.scrollRectToVisible' was not converted.
            //            scrollRectToVisible(r);

            //UNDONE: Method 'javax.swing.JComponent.revalidate' was not converted.
            //            revalidate();
            Redraw();
        }

        public Point whereItIs(Point point, Point direction, int margin)
        {
            Rectangle r = itsEditorContainer.ViewRectangle;

            direction.X = 0;
            direction.Y = 0;

            // Vertical dimension

            if (point.X <= r.X + margin)
                direction.X = -1;
            else if (point.X >= r.X + r.Width - margin)
                direction.X = 1;

            if (point.Y <= r.Y + margin)
                direction.Y = -1;
            else if (point.Y >= r.Y + r.Height - margin)
                direction.Y = 1;

            return direction;
        }

        // if the selection is outside the current visible 
        // area, rescroll to show it
        internal void showSelection()
        {
            Rectangle selectionBounds = ErmesSelection.patcherSelection.Bounds;
            Rectangle visibleRectangle = itsEditorContainer.ViewRectangle;

            if (selectionBounds.IsEmpty)
            {
                //UNDONE: Method 'javax.swing.SwingUtilities.isRectangleContainingRectangle' was not converted.
                //                if (!SwingUtilities.isRectangleContainingRectangle(visibleRectangle, selectionBounds))
                //                {
                //UNDONE: Method 'javax.swing.JComponent.scrollRectToVisible' was not converted.
                //                    scrollRectToVisible(selectionBounds);
                //UNDONE: Method 'javax.swing.JComponent.revalidate' was not converted.
                //                    revalidate(); // ???
                //                }
            }
        }

        private static readonly Size minSize = new Size(30, 20);
        public static readonly Point Empty;

        public Size MinimumSize
        {
            get
            {
                return minSize;
            }
        }

        // ---------------------------------------------------------------
        // create a graphic Object
        // ---------------------------------------------------------------

        public void AddNewObject(GraphicObject obj, bool doEdit)
        {
            displayList.Add(obj);
            if ((obj.Layer == -1) || (pasting))
                displayList.ReassignLayers();

            obj.updateInOutlets();

            if (!multiAdd)
                ErmesSelection.patcherSelection.DeselectAll();

            if (fromToolbar)
            {
                ErmesSelection.patcherSelection.select(obj);
                fromToolbar = false;
            }
            Redraw();

            //if (doEdit && newObjectEdit && (!multiAdd) && (obj is Editable) && !undoing)
            //{
            //    // The EditField is not really ready until the control
            //    // is returned back to the event loop; this is why we invoke textEditObject 
            //    // with an invoke later command.

            //    textEditObject((Editable)obj, Empty);
            //}

            multiAdd = false;
        }

        public void objectRedefined(FtsGraphicObject fObj)
        {
            //GraphicObject obj = displayList.GetGraphicObjectFor(fObj);
            //if ((obj is Editable) && (TextEditedObject == obj))
            //    stopTextEditing();

            //obj.redefined();

            ErmesSelection.patcherSelection.DeselectAll();
            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //repaint();
        }

        internal bool connFromClient = false;
        public void addingConnection()
        {
            connFromClient = true;
        }

        public GraphicConnection AddNewConnection(FtsConnection fc)
        {
            GraphicConnection connection = new GraphicConnection(this, displayList.GetGraphicObjectFor(fc.From), fc.FromOutlet,
                                                                    displayList.GetGraphicObjectFor(fc.To), fc.ToInlet, fc.Type, fc);
            displayList.Add(connection);
            connection.updateDimensions();
            if (connFromClient)
            {
                ErmesSelection.patcherSelection.select(connection);
                connFromClient = false;
            }
            connection.redraw();

            return connection;
        }

        /***************************************************************/
        /***************************************************************/

        //debug utility 
        private void printObjectsDescription()
        {
            object[] objects = itsPatcher.Objects.ObjectArray;
            int osize = itsPatcher.Objects.Size;

            for (int i = 0; i < osize; i++)
                Console.Error.WriteLine("obj {0} {1}", ((FtsGraphicObject)objects[i]), ((FtsGraphicObject)objects[i]).Description);
        }

        // -----------------------------------------------------------------------
        // Handling of the object text editing
        // -----------------------------------------------------------------------
        private Editable editedObject = null;
        //private EditField itsEditField;

        //public EditField EditField
        //{
        //    get
        //    {
        //        return itsEditField;
        //    }
        //}

        public bool isTextEditingObject
        {
            get
            {
                return editedObject != null;
            }

        }

        public Editable TextEditedObject
        {
            get
            {
                return editedObject;
            }
        }

        //        public void texteditobject(editable obj)
        //        {
        //            texteditobject(obj, null);
        //        }

        public void textEditObject(Editable obj, Point p)
        {
            if (editedObject != null)
                stopTextEditing();

            editedObject = obj;
            //itsEditField.doEdit(obj, p);
        }

        public void stopTextEditing()
        {
            if (editedObject != null)
            {
                //itsEditField.endEdit();
                resetUndoRedo();
            }
            editedObject = null;
        }

        public void abortTextEditing()
        {
            //if (editedObject != null)
            //    itsEditField.abortEdit();

            editedObject = null;
        }

        //////////////////////////////////////////////////
        public void ShowObject(object obj)
        {
            // Should select or highlight obj if it is an FtsObject

            if (obj is FtsGraphicObject)
            {
                if (IsLocked)
                    IsLocked = false;

                GraphicObject graphicObject = displayList.GetGraphicObjectFor((FtsGraphicObject)obj);

                if (graphicObject != null)
                {
                    ErmesSelection.patcherSelection.Owner = this;

                    if (ErmesSelection.patcherSelection.isEmpty)
                    {
                        ErmesSelection.patcherSelection.select(graphicObject);
                        graphicObject.Redraw();
                    }
                    else
                    {
                        ErmesSelection.patcherSelection.redraw();
                        ErmesSelection.patcherSelection.DeselectAll();
                        ErmesSelection.patcherSelection.select(graphicObject);
                        graphicObject.Redraw();
                    }

                    showSelection();
                }
            }
        }

        public void paintComponent(Graphics g)
        {
            displayList.Paint(g);
        }

        //UNDONE: Class 'java.awt.print.PageFormat' was not converted.
        //        public int print(Graphics g, PageFormat pf, int pi)
        //        {
        //            Paper paper = pf.getPaper();
        //            Point sketchPos = SwingUtilities.convertPoint(this, Location, (ErmesSketchWindow) itsEditorContainer);
        //            double onsetX = pf.getImageableX() + sketchPos.X;
        //            double onsetY = pf.getImageableY() + sketchPos.Y;
        //            double width = pf.getImageableWidth();
        //            double height = pf.getImageableHeight();
        //            double scaleX = width / (Size.Width);
        //            double scaleY = height / (Size.Height + 31);
        //            double scale = (scaleX < scaleY) ? scaleX : scaleY;
        //            
        //            if (scale < 1.0)
        //            {
        //                    ((Graphics2D) g).scale(scale, scale);
        //            }
        //            
        //            ((Graphics2D) g).translate(onsetX, onsetY);
        //            
        //            if (pi >= 1)
        //            {
        //                return java.awt.print.Printable_Fields.NO_SUCH_PAGE;
        //            }
        //            displayList.paint((Graphics2D) g);
        //            return java.awt.print.Printable_Fields.PAGE_EXISTS;
        //        }

        public void Redraw()
        {
            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //repaint();
        }

        // ------------------------------------------------------------------------
        // toolbar support
        // ------------------------------------------------------------------------
        private ToolBar toolBar;

        public ToolBar AToolBar
        {
            get
            {
                return toolBar;
            }
            set
            {
                this.toolBar = value;
            }
        }

        //--------------------------------------------------------
        //    cleanAll
        //    Make the cleanup needed before closing the 
        //    sketchpad
        //--------------------------------------------------------

        internal void cleanAll()
        {
            itsPatcher.ResetPatcherListener();

            engine.Dispose();

            if (ErmesSelection.patcherSelection.ownedBy(this))
                ErmesSelection.patcherSelection.DeselectAll();

            displayList.DisposeAllObjects();

            //UNDONE: Method 'java.awt.Component.remove' was converted.
            //            remove(itsEditField);

            // Clean up to help the gc, and found the bugs.
            itsEditorContainer = null; // should not be needed, here to get the grabber !!

            itsPatcher = null;
            anOldPastedObject = null;
            toolBar = null;
            itsEditorContainer = null;
            displayList = null;

            pastedObjects.RemoveAllElements();
            pastedConnections.RemoveAllElements();

            updateGroup.Remove(this);

            ftsUndoClipboard.Dispose();
            ftsUndoClipboard = null;
            undoObjects.Clear();

            removeSelection.ftsSelection.Dispose();
            removeSelection = null;
            itsMessageDisplayer = null;

            if (keyEventClient != null)
            {
                //UNDONE: Method 'java.awt.Component.removeKeyListener' was converted
                //                removeKeyListener( keyEventClient);
                keyEventClient = null;
            }
            highlightedInletObject = null;
            highlightedOutletObject = null;
            connectingObject = null;

            keyMap = null;
        }

        //--------------------------------------------------------
        //    Close 
        //--------------------------------------------------------

        public void Close(bool doCancel)
        {
            if (itsPatcher.IsARootPatcher)
            {
                //UNDONE: Class PatcherPrintManager Not Yet Implemented.
                //                if (PatcherSaveManager.saveClosing(IEditorContainer, doCancel))
                //                {
                //                    itsPatcher.stopUpdates();
                //                    itsPatcher.requestDestroyEditor();
                //                    ((FtsPatcherObject) itsPatcher.Parent).requestDeleteObject(itsPatcher);
                //UNDONE: Method ErmesSketchWindow.Destroy Not Yet Implemented.
                //                    ((ErmesSketchWindow) itsEditorContainer).Destroy();
                //                }
                //            }
                //            else
                //            {
                //                itsPatcher.stopUpdates();
                //                itsPatcher.requestDestroyEditor();
                //                itsPatcher.resetPatcherListener();
                //UNDONE: Method ErmesSketchWindow.Destroy Not Yet Implemented.
                //                ((ErmesSketchWindow) itsEditorContainer).Destroy();
            }
        }

        public void Save()
        {
            //UNDONE: Class PatcherPrintManager Not Yet Implemented.
            //            PatcherSaveManager.save(itsEditorContainer);
        }

        public void SaveAs()
        {
            //UNDONE: Class PatcherPrintManager Not Yet Implemented.
            //            PatcherSaveManager.saveAs(itsEditorContainer);
        }

        public void Print()
        {
            //UNDONE: Class PatcherPrintManager Not Yet Implemented.
            //            PatcherPrintManager.Print(itsEditorContainer);
        }

        // -----------------------------------------------------------------------
        // The waiting/stopWaiting service
        // -----------------------------------------------------------------------

        private int itsWaiting = 0;

        public void waiting()
        {
            if (itsWaiting >= 0)
            {
                //UNDONE: Member 'java.awt.Cursor.getDefaultCursor' was converted to 'System.Windows.Forms.Cursors.Default' which cannot be assigned to an int.
                //                Cursor = System.Windows.Forms.Cursors.WaitCursor;
            }

            itsWaiting++;
        }

        public void StopWaiting()
        {
            itsWaiting--;

            if (itsWaiting <= 0)
            {
                //UNDONE: Member 'java.awt.Cursor.getDefaultCursor' was converted to 'System.Windows.Forms.Cursors.Default' which cannot be assigned to an int.
                //                Cursor = System.Windows.Forms.Cursors.Default;
            }
        }

        // ------------------------------------------------------------------------
        // Mode handling
        // ------------------------------------------------------------------------
        private bool locked = false;

        // To set the initial state: set to edit mode only if the
        // initialMode property of a patcher is set and it is set
        // to something different than "run" (usually, "edit" :)
        internal void InitLock()
        {
            IsLocked = false;
        }

        public bool IsLocked
        {
            get
            {
                return locked;
            }
            set
            {
                //toolBar.isLocked = value;

                this.locked = value;
                // Store the mode in a non persistent, property of 
                // the patch, so that subpatcher can use it as their initial mode
                if (locked)
                {
                    itsPatcher.EditMode = FtsPatcherObject.RUN_MODE;
                    setRunModeInteraction();

                    //if (isTextEditingObject)
                    //    stopTextEditing();

                    if (ErmesSelection.patcherSelection.ownedBy(this))
                        ErmesSelection.patcherSelection.DeselectAll();
                }
                else
                {
                    setEditModeInteraction();
                    itsPatcher.EditMode = FtsPatcherObject.EDIT_MODE;
                }

                KeyEventClient = null; //when changing mode, always remove key listeners
                //UNDONE: Method 'javax.swing.JComponent.requestFocus' was not converted.
                //                requestFocus();
            }
        }

        // ------------------------------------------------------------------------
        //Temporary support for text cut/pasting; will work only between objects,
        //not in the general case; the general case need to use the real JDK clipboard
        //support, and will be done when the patcher editor will be based on the toolkit.
        // ------------------------------------------------------------------------

        internal void pasteText()
        {
            //UNDONE: Method 'javax.swing.text.JTextComponent.paste' was not converted.
            //            itsEditField.paste();
            //itsEditField.resizeIfNeeded();
        }

        internal void copyText()
        {
            //UNDONE: Method 'javax.swing.text.JTextComponent.copy' was not converted.
            //            itsEditField.copy();
        }

        internal void cutText()
        {
            //UNDONE: Method 'javax.swing.text.JTextComponent.cut' was not converted.
            //            itsEditField.cut();
            //itsEditField.resizeIfNeeded();
        }

        //internal bool canCopyText()
        //{
        //    return isTextEditingObject;
        //}

        //internal bool canPasteText()
        //{
        //    return isTextEditingObject;
        //}

        public string SelectedText
        {
            get
            {
                //if (isTextEditingObject)
                //{
                //    //UNDONE: Method 'javax.swing.text.JTextComponent.getSelectedText' was not converted.
                //    //                    return itsEditField.getSelectedText();
                //    return null;
                //}
                //else
                    return null;
            }
        }

        internal void deleteSelectedText()
        {
            //itsEditField.deleteSelectedText();
        }

        // ------------------------------------------------------------------------
        // Annotations; should be done better ..
        // ------------------------------------------------------------------------
        private bool annotating = false;

        public bool isAnnotating
        {
            get
            {
                return annotating;
            }
            set
            {
                annotating = value;
            }
        }

        public void cleanAnnotations()
        {
            if (annotating)
            {
                annotating = false;
                Redraw();
            }
        }

        // ------------------------------------------------------------------------
        // Support for keeping a unique supplementary keylistener
        // in the sketchpad; used currently by the number box, the other
        // are Swing components and use the focus.
        // ------------------------------------------------------------------------

        private IKeyEventClient keyEventClient;

        public IKeyEventClient KeyEventClient
        {
            set
            {
                if (value == keyEventClient)
                    return;

                if (keyEventClient != null)
                {
                    //UNDONE: Method 'java.awt.Component.removeKeyListener' was converted
                    //                    KeyDown -= new System.Windows.Forms.KeyEventHandler(keyEventClient.keyPressed);
                    //                    KeyUp -= new System.Windows.Forms.KeyEventHandler(keyEventClient.keyReleased);
                    //                    KeyPress -= new System.Windows.Forms.KeyPressEventHandler(keyEventClient.keyTyped);
                    //                    this.keyEventClient.keyInputLost();
                }

                keyEventClient = value;

                if (keyEventClient != null)
                {
                    //UNDONE: Method 'java.awt.Component.KaddKeyListener' was converted
                    //                    KeyDown += new System.Windows.Forms.KeyEventHandler(keyEventClient.keyPressed);
                    //                    KeyUp += new System.Windows.Forms.KeyEventHandler(keyEventClient.keyReleased);
                    //                    KeyPress += new System.Windows.Forms.KeyPressEventHandler(keyEventClient.keyTyped);
                    //                    keyEventClient.keyInputGained();
                }
            }

        }

        // ------------------------------------------------------------------------
        // popoup
        // ------------------------------------------------------------------------
        public void showAddPopUp(Point p)
        {
            //UNDONE: Class AddPopUp Not Yet Implemented.    
            //            AddPopUp.popup(this, p.X, p.Y);
        }

        // ------------------------------------------------------------------------
        // Support for the new Interaction Model
        // ------------------------------------------------------------------------
        private InteractionEngine engine;

        public void setRunModeInteraction()
        {
            //            Cursor = System.Windows.Forms.Cursors.Default;
            //stopTextEditing();
            //toolBar.reset();
            resetHighlightedInlet();
            resetHighlightedOutlet();
            engine.TopInteraction = Interactions.runModeInteraction;
        }

        public void setEditModeInteraction()
        {
            //toolBar.reset();
            engine.TopInteraction = Interactions.editModeInteraction;
        }

        public void resetAdding()
        {
            //            Cursor = System.Windows.Forms.Cursors.Default;
            toolBar.reset();
            resetMessage();
            setEditModeInteraction();
        }

        private string newObjectDescription = null;
        private bool newObjectEdit;

        public void setAddModeInteraction(string description, string message, bool edit)
        {
            newObjectDescription = description;
            newObjectEdit = edit;
            showMessage(message);

            //stopTextEditing();

            engine.TopInteraction = Interactions.addModeInteraction;
        }

        internal bool multiAdd = false;
        internal bool fromToolbar = false;
        public void makeAddModeObject(int x, int y, bool edit)
        {
            resetUndoRedo();

            multiAdd = !edit;
            fromToolbar = true;
            itsPatcher.RequestAddObject(newObjectDescription, x, y, edit);
        }

        public InteractionEngine Engine
        {
            get
            {
                return engine;
            }
        }

        public void endInteraction()
        {
            engine.popInteraction();
        }

        // ------------------------------------------------------------------------
        // Support for the inlet/outlet highligthing;
        // to avoid adding to much memory to every object,
        // we store here the id of the higlighted inlet and outlet
        // ------------------------------------------------------------------------

        private int highlightedInlet;
        private GraphicObject highlightedInletObject = null;

        public int HighlightedInlet
        {
            get
            {
                return highlightedInlet;
            }
        }

        public GraphicObject HighlightedInletObject
        {
            get
            {
                return highlightedInletObject;
            }
        }

        public bool isHighlightedInlet(GraphicObject obj, int inlet)
        {
            return ((highlightedInletObject == obj) && (highlightedInlet == inlet));
        }

        public bool hasHighlightedInlet(GraphicObject obj)
        {
            return (highlightedInletObject == obj);
        }

        public bool hasHighlightedInlet()
        {
            return (highlightedInletObject != null);
        }

        public void setHighlightedInlet(GraphicObject obj, int inlet)
        {
            resetHighlightedInlet();

            highlightedInletObject = obj;
            highlightedInlet = inlet;

            if (highlightedInletObject != null)
            {
                highlightedInletObject.Redraw();
                highlightedInletObject.redrawConnections();
            }
        }

        public void resetHighlightedInlet()
        {
            if (highlightedInletObject != null)
            {
                highlightedInletObject.Redraw();
                highlightedInletObject.redrawConnections();
            }

            highlightedInletObject = null;
        }

        private int highlightedOutlet;
        private GraphicObject highlightedOutletObject = null;

        public int HighlightedOutlet
        {
            get
            {
                return highlightedOutlet;
            }
        }

        public GraphicObject HighlightedOutletObject
        {
            get
            {
                return highlightedOutletObject;
            }
        }

        public bool isHighlightedOutlet(GraphicObject obj, int outlet)
        {
            return ((highlightedOutletObject == obj) && (highlightedOutlet == outlet));
        }

        public bool hasHighlightedOutlet(GraphicObject obj)
        {
            return (highlightedOutletObject == obj);
        }

        public bool hasHighlightedOutlet()
        {
            return (highlightedOutletObject != null);
        }

        public void setHighlightedOutlet(GraphicObject obj, int outlet)
        {
            resetHighlightedOutlet();

            highlightedOutletObject = obj;
            highlightedOutlet = outlet;

            if (highlightedOutletObject != null)
            {
                highlightedOutletObject.Redraw();
                highlightedOutletObject.redrawConnections();
            }
        }

        public void resetHighlightedOutlet()
        {
            if (highlightedOutletObject != null)
            {
                highlightedOutletObject.Redraw();
                highlightedOutletObject.redrawConnections();
            }

            highlightedOutletObject = null;
        }

        // General Highlighting reset function
        public void resetHighlighted()
        {
            resetHighlightedOutlet();
            resetHighlightedInlet();
        }

        // ------------------------------------------------------------------------
        // current object during a connection drawing (used in getSensibleArea in GraphicObject
        // ------------------------------------------------------------------------
        private GraphicObject connectingObject = null;

        public GraphicObject ConnectingObject
        {
            get
            {
                return connectingObject;
            }

            set
            {
                connectingObject = value;
            }
        }

        // -----------------------------------------------------------------
        // Messages 
        // -----------------------------------------------------------------
        private IMessageDisplayer itsMessageDisplayer;

        public IMessageDisplayer MessageDisplayer
        {
            get
            {
                return itsMessageDisplayer;
            }
            set
            {
                itsMessageDisplayer = value;
            }
        }

        public void showMessage(string text)
        {
            itsMessageDisplayer.showMessage(text);
        }

        public void resetMessage()
        {
            itsMessageDisplayer.resetMessage();
        }

        public bool isMessageReset
        {
            get
            {
                return itsMessageDisplayer.MessageReset;
            }
        }

        ////////////////////////////////////
        // called at window resize and move
        ///////////////////////////////////

        public void resizeToWindow(int width, int height)
        {
            if (itsPatcher != null)
            {
                itsPatcher.WindowWidth = (int)System.Math.Round((double)ScaleTransform.Instance.InvScaleX(width));
                itsPatcher.WindowHeight = (int)System.Math.Round((double)ScaleTransform.Instance.InvScaleY(height));
                fixSize();
            }
        }

        public void setCurrentWindowSize(int width, int height)
        {
            if (itsPatcher != null)
            {
                itsPatcher.WW = (int)System.Math.Round((double)ScaleTransform.Instance.InvScaleX(width));
                itsPatcher.WH = (int)System.Math.Round((double)ScaleTransform.Instance.InvScaleY(height));
                fixSize();
            }
        }

        public void relocateToWindow(int x, int y)
        {
            if (itsPatcher != null)
            {
                itsPatcher.WindowX = x;
                itsPatcher.WindowY = y;
            }
        }

        internal void stopUpdates()
        {
            // Do the test because the awt can call this before itsPatcher is ready
            if (itsPatcher != null)
                itsPatcher.StopUpdates();
        }
        internal void startUpdates()
        {
            // Do the test because the awt can call this before itsPatcher is ready
            if (itsPatcher != null)
                itsPatcher.StartUpdates();
        }

        public bool IsDirty
        {
            set
            {
                //AToolBar.IsDirty = value;
            }
        }

        // -----------------------------------------------------------------
        // Undo/Redo
        // -----------------------------------------------------------------
        internal List<GraphicObject> undoObjects;
        internal bool canUndo = false;
        internal bool canRedo = false;
        internal bool undoing = false;
        internal bool undoed = false;
        internal bool isRemoveUndo = false;
        internal bool isUpdateUndo = false;
        internal FtsClipboard ftsUndoClipboard;
        internal ErmesSelection removeSelection;
        internal string undoType = "";

        internal void initUndoStuff()
        {
            removeSelection = new ErmesSelection();
            undoObjects = new List<GraphicObject>();

            try
            {
                ftsUndoClipboard = new FtsClipboard();
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("[ErmesSketchPad]: Error in FtsClipboard creation!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void setUndo(string type, bool remove, bool update)
        {
            isUpdateUndo = (undoType.Equals(type) && update && !undoed);
            isRemoveUndo = remove;
            undoType = type;

            if (!isRemoveUndo)
            {
                if (!isUpdateUndo)
                {
                    undoObjects.Clear();
                    for (IEnumerator e = ErmesSelection.patcherSelection.SelectedObjects; e.MoveNext(); )
                    {
                        addUndoRedoObject((GraphicObject)e.Current);
                    }

                    canUndo = (undoObjects.Count > 0);
                    undoed = false;
                }
            }
            else
            {
                removeSelection.DeselectAll();
                for (IEnumerator e = ErmesSelection.patcherSelection.SelectedObjects; e.MoveNext(); )
                {
                    addUndoRedoObject((GraphicObject)e.Current);
                }

                if (removeSelection.SelectedObjectsSize > 0)
                {
                    ftsUndoClipboard.Copy(removeSelection.ftsSelection);
                    canUndo = true;
                }
                else
                    canUndo = false;
            }
            canRedo = false;
        }

        public void setUndo(string type, GraphicObject obj, bool remove, bool update)
        {
            isUpdateUndo = (undoType.Equals(type) && update && !undoed);
            isRemoveUndo = remove;
            undoType = type;

            if (!remove)
            {
                if (!isUpdateUndo)
                {
                    undoObjects.Clear();
                    addUndoRedoObject(obj);
                    canUndo = true;
                    undoed = false;
                }
            }
            else
            {
                removeSelection.DeselectAll();
                addUndoRedoObject(obj);
                ftsUndoClipboard.Copy(removeSelection.ftsSelection);
                canUndo = true;
            }
            canRedo = false;
        }

        public void setRedo()
        {
            canRedo = true;
            canUndo = false;

            if (!isRemoveUndo)
            {
                undoObjects.ForEach(
                    delegate(GraphicObject graphicObject)
                    {
                        graphicObject.setRedo();
                    });
            }
        }

        internal void addUndoRedoObject(GraphicObject obj)
        {
            if (isRemoveUndo)
            {
                removeSelection.add(obj);
                // select all connections for this obj
                displayList.GetConnectionsFor(obj).ForEach(
                    delegate(GraphicConnection connection)
                    {
                        removeSelection.add(connection);
                    });
            }
            else if (!undoObjects.Contains(obj))
            {
                obj.setUndo();
                undoObjects.Add(obj);
            }
        }

        public void undo()
        {
            setRedo();

            undoing = true;

            if (isRemoveUndo)
                FtsPatcher.RequestPaste(ftsUndoClipboard, 0, 0);
            else
            {
                undoObjects.ForEach(
                    delegate(GraphicObject graphicObject)
                    {
                        graphicObject.undo();
                    });

                Redraw();
            }
            undoed = true;
        }

        public void redo()
        {
            if (isRemoveUndo)
            {
                if (removeSelection.SelectedObjectsSize > 0)
                {
                    FtsPatcher.RequestDeleteObjects(removeSelection.SelectedObjects);
                    for (IEnumerator e = removeSelection.SelectedObjects; e.MoveNext(); )
                    {
                        ((IDisplayObject)e.Current).Delete();
                    }
                    removeSelection.DeselectAll();
                    DisplayList.ReassignLayers();
                }
            }
            else
            {
                undoObjects.ForEach(
                    delegate(GraphicObject graphicObject)
                    {
                        graphicObject.redo();
                    });
            }

            Redraw();

            canRedo = false;
            canUndo = true;
        }

        public void resetUndoRedo()
        {
            removeSelection.DeselectAll();
            List<GraphicObject> temp_arraylist;
            temp_arraylist = undoObjects;
            temp_arraylist.RemoveRange(0, temp_arraylist.Count);
            canRedo = false;
            canUndo = false;
            undoing = false;
            undoType = "";
        }

        public bool isCanUndo
        {
            get
            {
                return canUndo;
            }
        }

        public bool isCanRedo
        {
            get
            {
                return canRedo;
            }
        }

        public string UndoType
        {
            get
            {
                return undoType;
            }
        }

        /* ClipboardOwner interface */
        public void lostOwnership(System.Windows.Forms.Clipboard c, System.Windows.Forms.IDataObject t) { }
    }
}