//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Windows.Forms;

using ircam.jmax;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.patcher.objects;

namespace ircam.jmax.editors.patcher.interactions
{
    /// <summary>This class implement the interaction engine of the patcher editor.
    /// The interaction engine is a simplified version of the jMax toolkit
    /// interaction engine; we should move to the toolkit ASAP.
    /// The engine is composed by:
    /// </summary>
    /// <summary><ul>
    /// <li> An input filter, essentially the lexical parser, that translate mouse
    /// events in higher level lexical events; lexical events are implemented as
    /// simple method call to the interaction; the input filter can change configuration,
    /// depending on the interaction; the current interaction can access the filter
    /// thru this class; see the InputFilter and Interaction class for more information about events.
    /// <li> Interactions; a set of class, one for each specific interaction (so that new
    /// interaction can be added by adding the class and branching it in the correct
    /// place in the interaction graph); the interactions are instantiated by need,
    /// and not reused, and have access to the Engine and thru this to the SketchPad/displayList
    /// to do semantic actions.
    /// </ul>
    /// The input filter; translate input mouse events in lexical events
    /// for the Interaction engine; it also provide a couple of input tracking
    /// services, like keeping the position of the last mouse event, and the
    /// previous mouse event position; they are passed to the event.
    /// The lexical analysis made is not modal (state less).
    /// There are events, modifiers and locations;
    /// the event is a combination of bit patterns, defined in the Squeack
    /// class; the squeack is passed to the interaction thru a single method call
    /// together with the destination object, if any, and with the current and previous
    /// mouse point.
    /// Then, there are two properties that control filtering: followingMoves
    /// and followingLocations; if followingMoves is false, no move or drag event are
    /// generated; if followingLocation is false, no squeack with locations is
    /// generated; the two flags should be installed by the interactions
    /// when set in the interaction engine.
    /// </summary>

    sealed public class InteractionEngine
    {
        // The input filter can be set in different status;
        // they regard the different granularity input events
        // are analized.

        internal ErmesSketchPad sketch;
        internal DisplayList displayList;
        public static int SHORTCUT;

        public InteractionEngine(ErmesSketchPad sketch)
        {
            this.sketch = sketch;

            displayList = sketch.DisplayList;
            initAutoScroll();
        }

        public void Dispose()
        {
            //            if (followingMoves)
            //UNDONE: Method 'java.awt.Component.removeMouseMotionListener' was not converted.
            //                sketch.removeMouseMotionListener(this); 
            //
            //            sketch.removeMouseListener(this);
        }

        // Control properties

        private bool followingLocations = false;
        private bool followingInOutletLocations = false;
        private bool followingMoves = false;
        private bool followingInletLocations = false;
        private bool followingOutletLocations = false;
        private bool keyListening = false;
        private bool followingAlsoConnectionLocations = false;

        internal bool isFollowingMoves
        {
            get
            {
                return followingMoves;
            }
            set
            {
                if (followingMoves)
                {
                    if (!value)
                    {
                        //UNDONE: Method 'java.awt.Component.removeMouseMotionListener' was not converted.
                        //                        sketch.removeMouseMotionListener(this);
                    }
                    else
                    {
                        if (value)
                        {
                            //UNDONE: Method 'java.awt.Component.addMouseMotionListener' was not converted.
                            //                            sketch.addMouseMotionListener(this);
                        }
                    }
                }

                followingMoves = value;
            }
        }

        internal bool isKeyListening
        {
            get
            {
                return keyListening;
            }
            set
            {
                if (keyListening)
                {
                    if (!value)
                    {
                        //UNDONE: Method 'java.awt.Component.addKeyListener' was not converted.
                        //                        sketch.removeKeyListener(this);
                    }
                    else
                    {
                        if (value)
                        {
                            //UNDONE: Method 'java.awt.Component.addKeyListener' was not converted.
                            //                            sketch.addKeyListener(this);
                        }
                    }
                }

                keyListening = value;
            }
        }

        internal bool isFollowingAlsoConnectionLocations
        {
            get
            {
                return followingAlsoConnectionLocations;
            }
            set
            {
                followingAlsoConnectionLocations = value;
            }
        }

        internal bool isFollowingLocations
        {
            get
            {
                return followingLocations;
            }
            set
            {
                followingLocations = value;
                GraphicObject.isFollowingLocations = value; // Hack ? Should go thru displayList ?
            }
        }

        internal bool isFollowingInOutletLocations
        {
            get
            {
                return followingInOutletLocations;
            }
            set
            {
                followingInOutletLocations = value;
                GraphicObject.isFollowingInOutletLocations = value; // Hack ? Should go thru displayList ?
            }
        }

        internal bool isFollowingInletLocations
        {
            get
            {
                return followingInletLocations;
            }
            set
            {
                followingInletLocations = value;
                GraphicObject.isFollowingInletLocations = value; // Hack ? Should go thru displayList ?
            }
        }


        internal bool isFollowingOutletLocations
        {
            get
            {
                return followingOutletLocations;
            }

            set
            {
                followingOutletLocations = value;
                GraphicObject.isFollowingOutletLocations = value; // Hack ? Should go thru displayList ?
            }
        }

        // Utilities

        private Point mouse = new Point();
        private Point oldMouse = new Point();

        private void updateMouseHistory(MouseEventArgs e)
        {
            oldMouse = mouse;
            mouse = new Point(e.X, e.Y);
        }

        private int getModifiersBits(MouseEventArgs e)
        {
            int ret = 0;

            //UNDONE: Method 'java.awt.event.InputEvent.isShiftDown' was not converted.            
            //            if (e.isShiftDown())
            ret |= Squeack.SHIFT;

            //jdk bug fix: in macosx platform isMiddleMouseButton() function return true with
            //both CTRL and ALT keys pressed. 
            //UNDONE: Method 'java.awt.event.InputEvent.isControlDown' was not converted.
            //            if (SwingUtilities.isMiddleMouseButton(e) && !e.isControlDown())
            ret |= Squeack.MIDDLE_BUTTON;
            //UNDONE: Method 'java.lang.Class.getModifiers' was not converted.
            //            if ((e.getModifiers() & SHORTCUT) != 0)
            ret |= Squeack.SHORTCUT;

            return ret;
        }


        private int getLocationBits(SensibilityArea area)
        {
            if (area == null)
                return Squeack.BACKGROUND;
            else
                return area.ASqueack;
        }


        private void processEvent(int squeack, MouseEventArgs e)
        {
            //UNDONE: Method 'java.awt.event.InputEvent.getSource' was not converted.    
            //            ErmesSketchPad editor = (ErmesSketchPad) e.getSource;
            SensibilityArea area = null;

            //            editor.cleanAnnotations();

            updateMouseHistory(e);

            squeack |= getModifiersBits(e);

            if (followingLocations || followingInOutletLocations)
            {
                if (!isFollowingAlsoConnectionLocations || isCtrlEditInteraction(squeack))
                {
                    area = displayList.GetObjectSensibilityAreaAt(mouse.X, mouse.Y);
                }
                else
                {
                    /* when shift is pressed don't look for inoutlet's sensibility areas */
                    //UNDONE: Method 'java.awt.event.InputEvent.isShiftDown' was not converted.    
                    //                    GraphicObject.FollowingInOutletLocations = !e.isShiftDown();
                    /***************************/
                    area = displayList.GetSensibilityAreaAt(mouse.X, mouse.Y);
                }

                squeack |= getLocationBits(area);
            }

            //            autoScrollIfNeeded(editor, squeack, mouse, oldMouse);


            //UNDONE: Method 'javax.swing.plaf.basic.BasicScrollBarUI.scrollTimer' was not converted.    
            //            if (!scrollTimer.isRunning())
            //                sendSqueack(editor, squeack, area, mouse, oldMouse);

            if (area != null)
                area.Dispose();
        }

        private void processKeyEvent(int squeack, KeyEventArgs e)
        {
            //UNDONE: Method 'java.awt.event.InputEvent.getSource' was not converted.
            //            ErmesSketchPad editor = (ErmesSketchPad) e.getSource;
            //            editor.cleanAnnotations();
            //UNDONE: Method 'javax.swing.plaf.basic.BasicScrollBarUI.scrollTimer' was not converted.                
            //            if (!scrollTimer.isRunning())
            //                sendSqueack(editor, squeack, null, null, null);
        }

        private bool isCtrlEditInteraction(int squeack)
        {
            //return ( Squeack.isDown(squeack) && Squeack.isCtrl(squeack));
            return (Squeack.isDown(squeack) && Squeack.isShortcut(squeack));
        }

        // Scroll handling
        // The input filter handle automatic smooth scrolling
        // the autoresize is handled by the sketch and by the
        // semantic action directly.

        internal Timer scrollTimer;
        internal ScrollDragAction scroller;
        internal bool autoScroll = false;
        private const int scrollMargin = 5;

        private void initAutoScroll()
        {
            //UNDONE: Method 'javax.swing.plaf.basic.BasicScrollBarUI.scrollTimer' was not converted.
            //            scroller = new ScrollDragAction(this);
            //            scrollTimer = new Timer(8, scroller);
            //            scrollTimer.setCoalesce(true);
            //            scrollTimer.setRepeats(true);
        }

        internal bool isAutoScrolling
        {
            get
            {
                return autoScroll;
            }

            set
            {
                autoScroll = value;
            }
        }

        internal class ScrollDragAction
        {
            internal ErmesSketchPad editor;
            internal int squeack;
            internal Point timerMouse = new Point();
            internal Point oldTimerMouse = new Point();
            internal Point direction = new Point();

            public void actionPerformed(EventArgs evt)
            {
                // Three case: inside the margin, going outside
                // outside the window, inside the margin
                // The third is not handled because this method
                // is never called in this case.

                //                if (!sketch.pointIsVisible(timerMouse, scrollMargin))
                //                {
                //                    // The point is in the margin (visible in the window,
                //                    // not visible if we take out the margin
                //                    
                //                    sendSqueack(editor, squeack, null, timerMouse, oldTimerMouse);
                //                    
                //                    sketch.whereItIs(timerMouse, direction, scrollMargin);
                //                    
                //                    sketch.scrollBy(direction.X * 3, direction.Y * 3);
                //                    
                //                    addPoint(timerMouse);
                //                    timerMouse.X += direction.X * 3;
                //                    timerMouse.Y += direction.Y * 3;
                //                }
            }

            internal int Squeack
            {
                set
                {
                    this.squeack = value;
                }
            }

            internal ErmesSketchPad Editor
            {
                set
                {
                    this.editor = value;
                }
            }

            internal void addPoint(Point mouse)
            {
                oldTimerMouse = timerMouse;
                timerMouse = mouse;
            }

            internal void addPoint(int x, int y)
            {
                oldTimerMouse = timerMouse;
                timerMouse = new Point(x, y);
            }
        }

        internal void autoScrollIfNeeded(ErmesSketchPad editor, int squeack, Point mouse, Point oldMouse)
        {
            // Handle the auto scrolling and autoresizing

            //UNDONE: Method 'javax.swing.plaf.basic.BasicScrollBarUI.scrollTimer' was not converted.            
            //            if (isAutoScrolling && Squeack.isDrag(squeack) && (!sketch.pointIsVisible(mouse, scrollMargin)))
            //            {
            //                if (scrollTimer.isRunning())
            //                {
            //                    // Ignore
            //                }
            //                else
            //                {
            //                    scroller.Editor = editor;
            //                    scroller.Squeack = squeack;
            //                    scroller.addPoint(oldMouse);
            //                    scroller.addPoint(mouse);
            //                    scrollTimer.start();
            //                }
            //            }
            //            else
            //            {
            //                if (scrollTimer.isRunning())
            //                {
            //                    scrollTimer.stop();
            //                }
            //            }
        }

        // Handle the interaction stack

        internal const int INTERACTION_STACK_DEPTH = 8;
        internal Interaction[] interactionStack = new Interaction[INTERACTION_STACK_DEPTH];
        private int tos = -1; //point to the last used element of the stack

        public void pushInteraction(Interaction interaction)
        {
            isFollowingMoves = false;
            isFollowingLocations = false;
            isFollowingInOutletLocations = false;
            isFollowingInletLocations = false;
            isFollowingOutletLocations = false;
            isFollowingAlsoConnectionLocations = false;
            isKeyListening = false;
            isAutoScrolling = false;

            interaction.configureInputFilter(this);
            interaction.reset();

            tos++;
            interactionStack[tos] = interaction;
        }

        // Substitute the top of the stack with a new interaction 

        public Interaction AInteraction
        {
            get
            {
                if (tos < 0)
                    return null;
                else
                    return interactionStack[tos];
            }
            set
            {
                isFollowingMoves = false;
                isFollowingLocations = false;
                isFollowingInOutletLocations = false;
                isFollowingInletLocations = false;
                isFollowingOutletLocations = false;
                isFollowingAlsoConnectionLocations = false;
                isKeyListening = false;
                isAutoScrolling = false;

                value.configureInputFilter(this);
                value.reset();
 
                interactionStack[tos] = value;
            }
        }

        public void popInteraction()
        {
            tos--;
            interactionStack[tos].configureInputFilter(this);
            interactionStack[tos].reset();
        }

        // Reset the stack and set a top level interaction
        public Interaction TopInteraction
        {
            set
            {
                tos = -1;
                pushInteraction(value);
            }
        }

        // Send a squeack to an interaction, but giving the chance to push an interaction before

        internal void sendSqueack(ErmesSketchPad editor, int squeack, SensibilityArea area, Point mouse, Point oldMouse)
        {
            Interaction del;

            del = AInteraction.delegateSqueack(editor, squeack, area, mouse, oldMouse);

            if (del != null)
                pushInteraction(del);

            AInteraction.gotSqueack(editor, squeack, area, mouse, oldMouse);
        }

        // Methods following the mouse

        public void mouseMoved(MouseEventArgs e)
        {
            processEvent(Squeack.MOVE, e);
        }

        public void mouseDragged(MouseEventArgs e)
        {
            processEvent(Squeack.DRAG, e);
        }

        public void mousePressed(MouseEventArgs e)
        {
            //            if (!e.isShiftDown())
            //sketch.stopTextEditing();

            //jdk bug fix: in macosx platform isPopupTrigger() function return true with
            //both CTRL and ALT keys pressed. We want that CTRL is popupTrigger (so right button) 
            //and ALT is middleButton. More: e.isALtdown return true in both CTRL and ALT case  
            bool isPopup;

            //            isPopup = e.isPopupTrigger();
            isPopup = true;

            if (isPopup)
                processEvent(Squeack.POP_UP, e);
            else if (e.Clicks > 1)
                processEvent(Squeack.DOUBLE_CLICK, e);
            else
                processEvent(Squeack.DOWN, e);

            sketch.KeyEventClient = null;
            //            sketch.requestFocus();
            //            sketch.getToolBar().transferFocus();
        }

        public void mouseReleased(MouseEventArgs e)
        {
            processEvent(Squeack.UP, e);

            //            if (e.isPopupTrigger())
            //            {
            //                processEvent(Squeack.POP_UP, e);
            //            }
        }

        // The following methods are not used by the interaction engine.

        public void mouseClicked(EventArgs e) { }

        public void mouseEntered(EventArgs e)
        {
            // sketch.requestFocus();
        }

        public void mouseExited(EventArgs e)
        {
            // sketch.stopTextEditing();
        }

        public void keyTyped(KeyPressEventArgs e) { }
        public void keyPressed(KeyEventArgs e) { }
        public void keyReleased(KeyEventArgs e)
        {
            if (e.KeyValue == (int)Keys.Shift)
            {
                processKeyEvent(Squeack.SHIFT_UP, e);
            }
            else
            {
                if (e.KeyValue == (int)Keys.Escape)
                {
                    processKeyEvent(Squeack.ESCAPE, e);
                }
                else
                {
                    if ((e.KeyValue == (int)Keys.Back) || (e.KeyValue == (int)Keys.Delete))
                    {
                        processKeyEvent(Squeack.DELETE, e);
                    }
                }
            }
        }
    }
}