//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;

using ircam.jmax.fts;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.patcher.interactions;

namespace ircam.jmax.editors.patcher.objects
{
    //
    // The graphic connection. It handles the paint, 
    // the loading/saving of connections. It has user interface
    // methods (mouseclick, move, doubleclick...).
    //

    public delegate void FtsConnectionListener(object sender, FtsConnectionListenerArgs e);

    public class FtsConnectionListenerArgs : EventArgs
    {
        public int type;

        public FtsConnectionListenerArgs(int theType)
        {
            this.type = theType;
        }
    }

    public class GraphicConnection : IDisplayObject
    {
        private GraphicObject from;
        private int outlet;

        private GraphicObject to;
        private int inlet;

        private int type;

        [NonSerialized()]
        private ErmesSketchPad sketch;

        private FtsConnection ftsConnection;
        private bool selected;

        private Point start = new Point();
        private Point end = new Point();

        private bool down;
        private bool right;

        // For info in the near function, see http://www.exaflop.org/docs/cgafaq/

        private float length;

        public GraphicConnection(ErmesSketchPad theSketchPad, GraphicObject fromObj, int theOutlet,
                                GraphicObject toObj, int theInlet, int theType, FtsConnection theFtsConnection)
        {
            ftsConnection = theFtsConnection;
            from = fromObj;
            to = toObj;
            sketch = theSketchPad;
            inlet = theInlet;
            outlet = theOutlet;
            type = theType;
            selected = false;

            ftsConnection.ConnectionListener = this;
        }

        public void updateDimensions()
        {
            start.X = from.getOutletAnchorX(outlet);
            start.Y = from.getOutletAnchorY(outlet);
            end.X = to.getInletAnchorX(inlet);
            end.Y = to.getInletAnchorY(inlet);

            down = (start.X <= end.X);
            right = (start.Y <= end.Y);

            length = (float)Math.Sqrt((start.X - end.X) * (start.X - end.X) + (start.Y - end.Y) * (start.Y - end.Y));
        }

        public void updateType()
        {
            type = ftsConnection.Type;
        }

        public GraphicObject SourceObject
        {
            get
            {
                return from;
            }
        }

        public GraphicObject DestObject
        {
            get
            {
                return to;
            }
        }

        public int Inlet
        {
            get
            {
                return inlet;
            }
        }

        public int Outlet
        {
            get
            {
                return outlet;
            }
        }

        public FtsConnection FtsConnection
        {
            get
            {
                return ftsConnection;
            }
        }

        public ErmesSketchPad SketchPad
        {
            get
            {
                return sketch;
            }
        }

        // Destructor

        public void Delete()
        {
            ftsConnection.Delete();
            sketch.DisplayList.Remove(this);
        }

        public void release()
        {
            ftsConnection.Release();
            sketch.DisplayList.Remove(this);
        }

        public void TypeChanged(object sender, FtsConnectionListenerArgs e)
        {
            type = e.type;
            redraw();
        }

        //--------------------------------------------------------
        // Select
        // select a connection
        //--------------------------------------------------------

        public bool Selected
        {
            get
            {
                return selected;
            }
            set
            {
                selected = value;
            }
        }

        private bool isNear(int x, int y)
        {
            // First, answer false for all the points outside a bounding rectangle for
            // the connection

            if (down)
            {
                if ((x < (start.X - 4)) || (x > (end.X + 4)))
                    return false;
            }
            else
            {
                if ((x < (end.X - 4)) || (x > (start.X + 4)))
                    return false;
            }

            if (right)
            {
                if ((y < start.Y - 4) || (y > end.Y + 4))
                    return false;
            }
            else
            {
                if ((y < end.Y - 4) || (y > start.Y + 4))
                    return false;
            }

            float z = (float)((start.Y - y) * (end.X - start.X) - (start.X - x) * (end.Y - start.Y));

            if (z > 0.0)
                return ((z / length) < 4.0);
            else
                return ((z / length) > -4.0);
        }

        public SensibilityArea GetSensibilityAreaAt(int mouseX, int mouseY)
        {
            if (isNear(mouseX, mouseY))
            {
                SensibilityArea area = SensibilityArea.getSensibilityArea(this, Squeack.CONNECTION);

                if (!((Math.Abs(start.X - end.X) < 2 * GraphicObject.ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA) &&
                      (Math.Abs(start.Y - end.Y) < 2 * GraphicObject.ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA)))
                {
                    SensibilityArea sourceArea = from.GetSensibilityAreaAt(mouseX, mouseY);
                    SensibilityArea destArea = to.GetSensibilityAreaAt(mouseX, mouseY);
                    if ((sourceArea != null && Squeack.onOutlet(sourceArea.ASqueack)) ||
                        (destArea != null && Squeack.onInlet(destArea.ASqueack)))
                    {
                        area.isTransparent = true;
                        area.Cost = 100;
                    }
                }

                return area;
            }
            else
                return null;
        }

        [NonSerialized()]
        internal const float unity = 2.0f;
        [NonSerialized()]
        internal static readonly float[] dash1 = new float[] { unity, 3 * unity };

        [NonSerialized()]
        internal static readonly float[] dash2 = new float[] { unity, unity };

        [NonSerialized()]
        internal static readonly Pen dashed1 = new Pen(Color.Black);

        [NonSerialized()]
        internal static readonly Pen dashed2 = new Pen(Color.Black);

        [NonSerialized()]
        internal static readonly Pen dashed3 = new Pen(Color.Black);

        [NonSerialized()]
        internal static readonly Pen normal = new Pen(Color.Black);

        public void Paint(Graphics g)
        {
            Color setColor;

            if (type == FtsConnection.fts_connection_audio_active)
            {
                setColor = Color.Black;

                double angle = Math.Abs(Math.Atan2((float)(start.Y - end.Y), (float)(end.X - start.X)));
                if ((angle > Math.PI / 4) && (angle < Math.PI * 3 / 4))
                {
                    if (selected)
                    {
                        g.DrawLine(dashed1, start.X + 1, start.Y, end.X + 1, end.Y);
                        g.DrawLine(dashed1, start.X + 2, start.Y, end.X + 2, end.Y);

                        g.DrawLine(dashed2, start.X, start.Y, end.X, end.Y);
                        g.DrawLine(dashed2, start.X + 1, start.Y, end.X + 1, end.Y);

                        g.DrawLine(dashed3, start.X - 1, start.Y, end.X - 1, end.Y);
                        g.DrawLine(dashed3, start.X, start.Y, end.X, end.Y);
                    }
                    else
                    {
                        g.DrawLine(dashed1, start.X + 1, start.Y, end.X + 1, end.Y);

                        g.DrawLine(dashed2, start.X, start.Y, end.X, end.Y);

                        g.DrawLine(dashed3, start.X - 1, start.Y, end.X - 1, end.Y);
                    }
                }
                else
                {
                    if (selected)
                    {
                        g.DrawLine(dashed1, start.X, start.Y - 1, end.X, end.Y - 1);
                        g.DrawLine(dashed1, start.X, start.Y, end.X, end.Y);

                        g.DrawLine(dashed2, start.X, start.Y, end.X, end.Y);
                        g.DrawLine(dashed2, start.X, start.Y + 1, end.X, end.Y + 1);

                        g.DrawLine(dashed3, start.X, start.Y + 1, end.X, end.Y + 1);
                        g.DrawLine(dashed3, start.X, start.Y + 2, end.X, end.Y + 2);
                    }
                    else
                    {
                        g.DrawLine(dashed1, start.X, start.Y - 1, end.X, end.Y - 1);

                        g.DrawLine(dashed2, start.X, start.Y, end.X, end.Y);

                        g.DrawLine(dashed3, start.X, start.Y + 1, end.X, end.Y + 1);
                    }
                }
            }
            else
            {
                if (type == FtsConnection.fts_connection_invalid)
                    setColor = Color.Gray;
                else
                    setColor = Color.Black;


                g.DrawLine(new Pen(Color.Black), start.X, start.Y, end.X, end.Y);
                if (selected)
                {
                    if (Math.Abs(start.X - end.X) > Math.Abs(start.Y - end.Y))
                        g.DrawLine(new Pen(Color.Black), start.X, start.Y + 1, end.X, end.Y + 1);
                    else
                        g.DrawLine(new Pen(Color.Black), start.X + 1, start.Y, end.X + 1, end.Y);
                }
            }
        }

        public void UpdatePaint(Graphics g)
        {
        }

        // Connections should store their bounds !!!

        public void redraw()
        {
            int x, y;
            int height, width;

            if (down)
            {
                x = start.X - 2;
                width = end.X - start.X + 4;
            }
            else
            {
                x = end.X - 2;
                width = start.X - end.X + 4;
            }

            if (right)
            {
                y = start.Y - 2;
                height = end.Y - start.Y + 4;
            }
            else
            {
                y = end.Y - 2;
                height = start.Y - end.Y + 4;
            }

            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //sketch.repaint(x, y, width, height);
        }

        public bool Intersects(Rectangle r)
        {
            bool ret;

            if (down)
            {
                if (right)
                    ret = !((r.X + r.Width < start.X) || (r.Y + r.Height < start.Y) ||
                            (r.X > end.X) || (r.Y > end.Y));
                else
                    ret = !((r.X + r.Width < start.X) || (r.Y + r.Height < end.Y) ||
                            (r.X > end.X) || (r.Y > start.Y));
            }
            else
            {
                if (right)
                    ret = !((r.X + r.Width < end.X) || (r.Y + r.Height < start.Y) ||
                            (r.X > start.X) || (r.Y > end.Y));
                else
                    ret = !((r.X + r.Width < end.X) || (r.Y + r.Height < end.Y) ||
                            (r.X > start.X) || (r.Y > start.Y));
            }

            return ret;
        }


        // Print function

        public override string ToString()
        {
            return ("GraphicConnection<" + from + "." + outlet + "-" + to + "." + inlet +
                    " (" + start.X + "." + start.Y + ")-" + length + "-(" + end.X + "." + end.Y + ")>");
        }
    }
}