//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

using ircam.jmax;
using ircam.jmax.fts;
using ircam.jmax.editors.patcher;

namespace ircam.jmax.editors.patcher.objects
{
    /// <summary> The "number box" graphic object base class</summary>

    abstract public class NumberBox : Editable
    {
        private int nDecimals = 0;
        private string filter;
        public const int DEFAULT_WIDTH = 40;
        public const int DEFAULT_HEIGHT = 15;
        private const int DEFAULT_VISIBLE_DIGIT = 3;
        private static Color floatColor = Color.FromArgb(239, 239, 239);

        public const int NOEDIT_MODE = 0;
        public const int DRAG_MODE = 1;
        public const int TEXT_MODE = 2;
        public const int INCR_MODE = 3;
        private int editMode = NOEDIT_MODE;

        //UNDONE: Class 'java.awt.image.BufferedImage' was not converted.
        //        private BufferedImage buff;
        private Graphics buffG;

        public NumberBox(FtsGraphicObject theFtsObject, string filter)
            : base(theFtsObject)
        {
            this.filter = filter;

            updateOffScreenBuffer();
        }

        internal void updateOffScreenBuffer()
        {
            int h = Height - 2;
            if (h <= 0)
                h = DEFAULT_HEIGHT - 2;

            int w = Width - h / 2 - 3;
            if (w <= 0)
                w = DEFAULT_WIDTH - h / 2 - 3;

            //UNDONE: Constructor 'java.awt.image.BufferedImage.BufferedImage' was not converted.
            //UNDONE: Field 'java.awt.image.BufferedImage.TYPE_INT_RGB' was not converted.
            //            buff = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
            //UNDONE: Method 'java.awt.image.BufferedImage.createGraphics' was not converted.
            //            buffG = buff.createGraphics();
            //UNDONE: Method 'java.awt.Graphics2D.setRenderingHint' was not converted.
            //UNDONE: Field 'java.awt.RenderingHints.KEY_TEXT_ANTIALIASING' was not converted.
            //UNDONE: Field 'java.awt.RenderingHints.VALUE_TEXT_ANTIALIAS_ON' was not converted.
            //            buffG.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        }

        public override void setDefaults()
        {
            base.Width = MinWidth;
            base.Height = MinHeight;
        }

        public override string Args
        {
            get
            {
                return ValueAsText;
            }
        }

        private int MinWidth
        {
            get
            {
                //UNDONE: Method 'java.awt.FontMetrics.stringWidth' was not converted.
                //                return FontMetrics.stringWidth("0") * DEFAULT_VISIBLE_DIGIT + FontMetrics.stringWidth("..") + 17;
                return 1 * DEFAULT_VISIBLE_DIGIT + 1 + 17;
            }
        }

        private int FullTextWidth
        {
            get
            {
                //UNDONE: Method 'java.awt.FontMetrics.stringWidth' was not converted.
                //                return (FontMetrics.stringWidth("..") + FontMetrics.stringWidth(ValueAsText) + 17);
                return (1 + 1 + 17);
            }
        }

        private int MinHeight
        {
            get
            {
                //HACK: FontStyle.Regular ?
                return (int)FontMetrics.GetEmHeight(System.Drawing.FontStyle.Regular) + 4;
            }
        }

        // redefined from base class
        public void setWidth(int theWidth)
        {
            int minWidth = MinWidth;
            if (theWidth < minWidth)
                theWidth = minWidth;

            base.Width = theWidth;

            updateOffScreenBuffer();
        }

        // redefined from base class
        // This way, changing the height from outside is forbidden
        public void setHeight(int theHeight)
        {
            updateOffScreenBuffer();
        }

        // redefined from base class
        public void setFont(Font theFont)
        {
            base.itsFont = theFont;

            int minWidth = MinWidth;
            if (Width < minWidth)
                base.Width = minWidth;

            base.Height = MinHeight;

            updateOffScreenBuffer();
        }

        public override void setCurrentFont(Font theFont)
        {
            base.setCurrentFont(theFont);
            updateOffScreenBuffer();
        }

        public override void fitToText()
        {
            int full = FullTextWidth;
            int min = MinWidth;

            if (full < min)
                base.Width = min;
            else
                base.Width = full;

            base.Height = MinHeight;

            updateOffScreenBuffer();
        }

        public override void redefine(string text)
        {
            ValueAsText = text;
            EditMode = NumberBox.NOEDIT_MODE;
        }

        public override void redefined()
        {
            fitToText();
            Redraw();
        }

        // ----------------------------------------
        // Text area offset
        // ----------------------------------------

        private const int TEXT_X_OFFSET = 3;
        private const int TEXT_Y_OFFSET = 3;

        public override int TextXOffset
        {
            get
            {
                return TextHeightOffset + TriangleWidth;
            }
        }

        public override int TextYOffset
        {
            get
            {
                return TEXT_Y_OFFSET;
            }
        }

        public override int TextWidthOffset
        {
            get
            {
                return TextHeightOffset + TriangleWidth + 4;
            }
        }

        public override int TextHeightOffset
        {
            get
            {
                return 5;
            }
        }

        public override Color TextForeground
        {
            get
            {
                return Color.Black;
            }
        }

        public override Color TextBackground
        {
            get
            {
                return Color.White;
            }
        }

        public override bool isMultiline
        {
            get
            {
                return false;
            }
        }

        // ----------------------------------------
        // ValueAsText property
        // ----------------------------------------
        abstract public string ValueAsText { get; set; }

        public override void Paint(Graphics g)
        {
            int x = X;
            int xp1 = x + 1;
            int y = Y;
            int w = Width;
            int wm1 = w - 1;
            int h = Height;
            int hd2 = h / 2;
            int hm1 = h - 1;
            string aString;

            Color color;

            // Background
            if (!isSelected)
                color = Color.White;
            else
            {
                color = ControlPaint.Dark(Settings.sharedInstance.UIColor);
            }

            g.FillRectangle(new SolidBrush(color), x + 1, y + 1, w - 2, h - 2);

            // Float area
            if ((editMode != TEXT_MODE) && !isSelected && (IntZoneWidth > 0))
            {
                color = floatColor;
                g.FillRectangle(new SolidBrush(color), x + IntZoneWidth + 1, y + 1, w - IntZoneWidth - 2, h - 2);
            }

            // Triangle
            color = Color.Black;
            if (editMode == TEXT_MODE || editMode == INCR_MODE)
            {
                Point[] pt = 
                {
                    new Point(xp1, y),
                    new Point(xp1 + hd2, y + hd2),
                    new Point(xp1, y + hm1)
                };

                g.FillPolygon(new SolidBrush(color), pt);
            }
            else
            {
                g.DrawLine(new Pen(Color.Black), xp1, y, xp1 + hd2, y + hd2);
                g.DrawLine(new Pen(Color.Black), xp1 + hd2, y + hd2, xp1, y + hm1);
            }

            // Text
            aString = getVisibleString(ValueAsText);

            if (editMode == INCR_MODE)
                color = Color.Gray;
            else
                color = Color.Black;

            //HACK: Method 'java.awt.Graphics.getFont()' was not converted.            
            System.Drawing.Font setFont = new System.Drawing.Font("Arial", 16);
            //HACK: Method 'java.awt.FontMetrics.GetAscent()' was not converted.
            g.DrawString(aString, setFont, new SolidBrush(Color.Black), x + hd2 + 5, y + setFont.Height + (h - setFont.Height) / 2 + 1);

            base.Paint(g);
        }

        public override void UpdatePaint(Graphics g)
        {
            //            g.DrawImage(buff, X + Height / 2 + 2, Y + 1, itsSketchPad);
        }

        public void drawContent(Graphics g, int x, int y, int w, int h)
        {
            int hd2 = h / 2;
            string aString;

            Color setColor;

            if (editMode != TEXT_MODE)
            {
                // Fill the background
                setColor = Color.White;

                if (IntZoneWidth > 0)
                {
                    int iw = IntZoneWidth - (hd2 + 2);
                    g.FillRectangle(new SolidBrush(setColor), 0, 0, iw, h);
                    setColor = floatColor;
                    g.FillRectangle(new SolidBrush(setColor), iw, 0, w - iw, h);
                }
                else
                    g.FillRectangle(new SolidBrush(setColor), 0, 0, w, h);

                // Get the value
                aString = getVisibleString(ValueAsText);
            }
            else
            {
                setColor = Color.White;
                g.FillRectangle(new SolidBrush(setColor), 0, 0, w, h);

                aString = getVisibleString(ValueAsText);
            }

            // Draw the value
            if (editMode == INCR_MODE)
                setColor = Color.Gray;
            else
                setColor = Color.Black;
            //HACK: Method 'java.awt.Graphics.getFont()' was not converted.            
            System.Drawing.Font setFont = new System.Drawing.Font("Arial", 16);
            //HACK: Method 'java.awt.FontMetrics.GetAscent()' was not converted.
            g.DrawString(aString, setFont, new SolidBrush(Color.Black), 3, setFont.Height + (h + 2 - setFont.Height / 2));
        }

        public override void updateRedraw()
        {
            int h = Height;
            drawContent(buffG, 0, 0, Width - h / 2 - 3, h - 2);
            base.updateRedraw();
        }

        internal string getVisibleString(string theString)
        {
            string aString = theString;
            string aString2 = "..";

            int aStringLength = theString.Length;

            int aCurrentSpace = Width - (Height / 2 + 5) - 5;
            //UNDONE: Method 'java.awt.FontMetrics.stringWidth' was not converted.
            //            int aStringWidth = FontMetrics.stringWidth(aString);
            int aStringWidth = 0;

            if (aStringWidth < aCurrentSpace)
                return aString;

            while ((aCurrentSpace <= aStringWidth) && (aString.Length > 0))
            {
                aString = aString.Substring(0, (aString.Length - 1) - (0));
                //UNDONE: Method 'java.awt.FontMetrics.stringWidth' was not converted.
                //                aStringWidth = FontMetrics.stringWidth(aString);
            }

            //UNDONE: Method 'java.awt.FontMetrics.stringWidth' was not converted.
            //            if ((aStringWidth + FontMetrics.stringWidth("..") >= aCurrentSpace) && (aString.Length > 0))
            //                aString = aString.Substring(0, (aString.Length - 1) - (0));

            aString = aString + aString2;
            return aString;
        }

        public int TriangleWidth
        {
            get
            {
                return Height / 2;
            }
        }

        abstract public int IntZoneWidth { get; }

        abstract public void increment(bool up, bool shift);

        public void keyPressed(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            //UNDONE: Method 'java.awt.event.InputEvent.isControlDown' was not converted.
            //UNDONE: Method 'java.awt.event.InputEvent.isMetaDown' was not converted.
            //            if (!e.isControlDown() && !e.isMetaDown())
            //            {
            //                if (e.KeyValue == 109)
            //                    return ; //??????
            //                // This stuff should be thrown away, and we should use
            //                // a real text area for the number boxes !!!
            //                
            //                switch (e.KeyValue)
            //                {
            //                    
            //                    case (int) System.Windows.Forms.Keys.Enter: 
            //                        ValueAsText = currentText.ToString();
            //                        currentText.Length = 0;
            //                        EditMode = NOEDIT_MODE;
            //                        itsSketchPad.KeyEventClient = null;
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.Up: 
            //                        if (EditMode == INCR_MODE)
            //                        {
            //UNDONE: Method 'java.awt.event.InputEvent.isShiftDown' was not converted.
            //                            increment(true, e.isShiftDown());
            //                        }
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.Down: 
            //                        if (EditMode == INCR_MODE)
            //                        {
            //UNDONE: Method 'java.awt.event.InputEvent.isShiftDown' was not converted.
            //                            increment(false, e.isShiftDown());
            //                        }
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.Delete: 
            //                    case (int) System.Windows.Forms.Keys.Back: 
            //                    {
            //                        int l = currentText.Length;
            //                        l = (l > 0) ? l - 1 : 0;
            //                        currentText.Length = l;
            //                    }
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad0: 
            //                        currentText.Append("0");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad1: 
            //                        currentText.Append("1");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad2: 
            //                        currentText.Append("2");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad3: 
            //                        currentText.Append("3");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad4: 
            //                        currentText.Append("4");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad5: 
            //                        currentText.Append("5");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad6: 
            //                        currentText.Append("6");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad7: 
            //                        currentText.Append("7");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad8: 
            //                        currentText.Append("8");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.NumPad9: 
            //                        currentText.Append("9");
            //                        break;
            //                    
            //                    case (int) System.Windows.Forms.Keys.Decimal: 
            //                        currentText.Append(".");
            //                        break;
            //                    
            //                    default: 
            //UNDONE: Method 'java.awt.event.KeyEvent.getKeyCode' was converted to 'System.Windows.Forms.KeyEventArgs.KeyValue' which has a different behavior.
            //                        currentText.Append((char) e.KeyValue);
            //                        break;
            //                    
            //                }
            //                
            //UNDONE: Method 'java.awt.event.InputEvent.consume' was not converted.
            //                e.consume();
            //                redraw();
            //            }
        }

        public void keyReleased(object sender, System.Windows.Forms.KeyEventArgs e)
        {
        }

        public void keyTyped(object sender, System.Windows.Forms.KeyPressEventArgs e)
        {
        }

        public void keyInputGained()
        {
            Redraw();
        }

        public void keyInputLost()
        {
            EditMode = NOEDIT_MODE;
            Redraw();
        }

        public int EditMode
        {
            get
            {
                return editMode;
            }

            set
            {
                editMode = value;
                Redraw();
            }
        }

        public bool isOnArrow(Point p)
        {
            int x = X;
            int y = Y;
            int h = Height;

            Point[] pt = 
                {
                    new Point(x, y),
                    new Point(x + h / 2, y + h / 2),
                    new Point(x, y + h)
                };

            //HACK: Constructor 'java.awt.Polygon.Polygon' was converted to 'SupportClass.CreateGraphicsPath' which has a different behavior.
            return (new Rectangle(x, y, h, h)).Contains(p);
        }
    }
}